/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#ifndef COMMONHELPER_H
#define COMMONHELPER_H

#include <string>
#include "CyPhyML.h"
#include "AssemblyInterface.h"
#include "UdmConsole.h"
#include "UdmGme.h"

/** \file
    \brief Contains helper functions.
*/


/** \brief Creates an expression and defines variables in muParser
    \param [in] feature Reference to a Feature whose baseType ID you want to find
    \return long ID of a Feature's baseType object
*/
long FindFeatureArchetypeID(CyPhyML::SolidModelingFeature&);

/** \brief Creates and adds an Assembly::Constraint to an Assembly::CADComponent from the passed in StructuralInterfaceRole ports
    \param [in] component A reference to the CADComponent to add the constraint to
	\param [in] sir_a Reference to StructuralInterfaceRole port whose attribute will form part of the constraint
	\param [in] sir_b Reference to StructuralInterfaceRole port whose attribute will form part of the constraint
    \return void
*/
void AddConstraint2CADComponent(AssemblyInterface::CADComponent& component, const CyPhyML::StructuralInterface& sir_a, const CyPhyML::StructuralInterface& sir_b);

void AddCADModelFeatureDetails(CyPhyML::StructuralInterfaceFeature&, AssemblyInterface::ConstraintFeature&);	//void AddCADModelFeatureDetails(CyPhyML::SolidModelingFeature&, AssemblyInterface::ConstraintFeature&);

/** \brief Creates a constraint that anchors a CADComponent to its parent CADComponent
		   An Assembly::Constraint is made up for a pair of ConstraintFeatures.
		   It calls CreateConstraintFeature.
    \param [in] component Reference to a CADComponent that will have a constraint that anchors it to its parent CADComponent
	\param [in] assemblyParentID ID of the parent CADComponent
    \return void
*/
void AddRootConstraint2CADComponent(AssemblyInterface::CADComponent& component, std::string assemblyParentID);

/** \brief Creates a constraint feature for a constraint
    \param [in] pair The parent for a constraint feature
	\param [in] id A CADComponent's ID
	\param [in] name Name of the feature
	\param [in] orientation Orientation of a feature (NONE, SIDE_A, SIDE_B)
    \return void
*/
void CreateConstraintFeature(AssemblyInterface::Pair& pair, std::string id, std::string name, std::string orientation);


/** \brief Prints StructuralIfaceMap
    \return void
*/
//void PrintStructuralIfaceMap();

/** \brief Checks that an AssemblyInterface::CADComponent has at least 1 constraint
    \param [in] component A map of parameters with parameter name as the key
    \return void
*/
void Check4ZeroConstraints(const AssemblyInterface::CADComponent& component);

/** \brief Calls Check4ZeroConstraints on its child AssemblyInterface::CADComponents
    \param [in] assembly A map of parameters with parameter name as the key
    \return void
*/
void Check4ZeroConstraints(const AssemblyInterface::Assembly& assembly);

// pair(x, y)
pair<int, int> GetXYPosition(const CyPhyML::MgaObject& obj);

struct CompareYPos {
	bool operator()(const CyPhyML::MgaObject& a, const CyPhyML::MgaObject& b);
};


template< class T>
class OrdinalPositionCompareFunctor : public std::binary_function< T, T, bool>
{
public:
	bool operator()( const T& t1, const T& t2) const {
			__int64 pos1= t1.PolygonOrdinalPosition();
			__int64 pos2 = t2.PolygonOrdinalPosition();
			return pos1 < pos2;
	}
};

/** \brief Cleans a string to be Pro/E acceptable
	\param [in] original The String to be cleaned up
	\return cleaned up string
*/
inline std::string CleanString(std::string original, 
							   unsigned int size = 31, 
							   std::string specialchars = "")
{
	std::string newStr;
	std::string Special_Char_String = ",.=()!@#$%&~`'+ ^*[]{}/?:;<>|";		// 11-28-2012: removed "-" from list

	if (specialchars != "")
		Special_Char_String += specialchars;
                                    
	for (unsigned int i = 0; i < original.length(); i++)
	{
		if (Special_Char_String.find(original[i]) == std::string::npos)
			newStr += original[i];
		else
		{
			if (original[i] == '<')
				newStr += "<";
			else if (original[i] == '>')
				newStr += ">";
			else
				newStr += "_";
		}
	}

	if (newStr.length() > size)
	{
		newStr.resize(size);
	//	GMEConsole::Console::writeLine("[" + original + "] is too long for ProE, resized to 31 chars.", MSG_WARNING);
	}

	return newStr;
}

inline string GetRegistryValue(Udm::Object& udmObj, string NodeName)
{
	CComQIPtr<IMgaFCO> comObj =  UdmGme::Udm2Gme(udmObj);
	if (comObj)
	{
		_bstr_t cvalue;
		HRESULT result = comObj->get_RegistryValue(CComBSTR(NodeName.c_str()), cvalue.GetAddress());
		if (result == S_OK)
		{
			if (cvalue.length() > 0)
				return string((char *)cvalue);
		}
	}
	return "";
}


string GetGMEGUID(const Udm::Object &);


int GetDegreesOfFreedom(const CyPhyML::StructuralInterface &si_in);

#endif