/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "CommonTraverse.h"
#include "UdmConsole.h"
#include "UdmGme.h"
  
/** \brief Visits a Join Structure.
		   During the process, it finds the end points of the join structure.
		   If an endpoint is a StructuralInterface port, then it is stored.
		   If an endpoint is a StructuralInterfaceForwarder port, then another visit function is called on the port.
    \param [in] parameters A map of parameters with parameter name as the key
    \return void
*/
void StructuralInterfacePortVisitor::Visit_JoinStructures(const CyPhyML::JoinStructures & joinStructure)
{
	//GMEConsole::Console::writeLine("StructuralInterfacePortVisitor::Visit_JoinStructures()", MSG_INFO);
	std::string gid = UdmGme::UdmId2GmeId(joinStructure.uniqueId());
	if (m_JoinStructuresVisited.find(joinStructure) == m_JoinStructuresVisited.end())   // new
	{
		//GMEConsole::Console::writeLine(" + New JS:" + UdmGme::UdmId2GmeId(joinStructure.uniqueId()), MSG_INFO);
		CyPhyML::StructuralPortType src_spt = joinStructure.srcJoinStructures_end();
		CyPhyML::StructuralPortType dst_spt = joinStructure.dstJoinStructures_end();

		this->m_JoinStructuresVisited.insert(joinStructure);	

		if (src_spt.type() == CyPhyML::StructuralInterface::meta)
			CyPhyML::StructuralInterface::Cast(src_spt).Accept(*this);
		else
			CyPhyML::StructuralInterfaceForwarder::Cast(src_spt).Accept(*this);

		if (dst_spt.type() == CyPhyML::StructuralInterface::meta)
			CyPhyML::StructuralInterface::Cast(dst_spt).Accept(*this);
		else
			CyPhyML::StructuralInterfaceForwarder::Cast(dst_spt).Accept(*this);

		//this->m_JoinStructuresVisited.insert(joinStructure);
	}
}


/** \brief Visits a StructuralInterface port.
		   During the process, it finds the connected Join Structures.
		   If a Join Structure has not been visited then visit is called on it.
    \param [in] sirPort Reference to the StructuralInterface port to be visited.
    \return void
*/
void StructuralInterfacePortVisitor::Visit_StructuralInterface(const CyPhyML::StructuralInterface &sirPort)
{
	std::string gid = UdmGme::UdmId2GmeId(sirPort.uniqueId());
	//GMEConsole::Console::writeLine("StructuralInterfacePortVisitor::Visit_StructuralInterfaceRole()", MSG_INFO);
	if (m_StructuralInterfaceRoleIDs.find(sirPort.uniqueId()) == m_StructuralInterfaceRoleIDs.end())   // new
	{
		//GMEConsole::Console::writeLine("found and adding Port:" + UdmGme::UdmId2GmeId(sirPort.uniqueId()), MSG_INFO);
		this->m_StructuralInterfaceRolePairs.push_back(sirPort);
		this->m_StructuralInterfaceRoleIDs.insert(sirPort.uniqueId());

		set<CyPhyML::JoinStructures> src_JoinSet = sirPort.srcJoinStructures();
		set<CyPhyML::JoinStructures> dst_JoinSet = sirPort.dstJoinStructures();

		for (std::set<CyPhyML::JoinStructures>::iterator i = src_JoinSet.begin(); i != src_JoinSet.end(); i++)
		{
			CyPhyML::JoinStructures join(*i);
			join.Accept(*this);
		}
	
		for (std::set<CyPhyML::JoinStructures>::iterator i = dst_JoinSet.begin(); i != dst_JoinSet.end(); i++)
		{
			CyPhyML::JoinStructures join(*i);
			join.Accept(*this);
		}
	}
}


/** \brief Visits a StructuralInterfaceForwarder port.
		   Gets connected Join Structures and visit those if they have not been visited.
    \param [in] sirForwarder Reference to the StructuralInterfaceForwarder port to be visited.
    \return void
*/
void StructuralInterfacePortVisitor::Visit_StructuralInterfaceForwarder(const CyPhyML::StructuralInterfaceForwarder &sirForwarder)
{
	std::string gid = UdmGme::UdmId2GmeId(sirForwarder.uniqueId());
	//GMEConsole::Console::writeLine("StructuralInterfacePortVisitor::Visit_StructuralInterfaceForwarde()", MSG_INFO);
	std::set<CyPhyML::JoinStructures> dst_JoinSet = sirForwarder.dstJoinStructures();
	std::set<CyPhyML::JoinStructures> src_JoinSet = sirForwarder.srcJoinStructures();
	
	for (std::set<CyPhyML::JoinStructures>::iterator i = src_JoinSet.begin(); i != src_JoinSet.end(); i++)
	{
		CyPhyML::JoinStructures join(*i);
		join.Accept(*this);
	}
	
	for (std::set<CyPhyML::JoinStructures>::iterator i = dst_JoinSet.begin(); i != dst_JoinSet.end(); i++)
	{
		CyPhyML::JoinStructures join(*i);
		join.Accept(*this);
	}
}
 
/** \brief Prints the StructuralInterface ports that are found.
    \return void
*/
void StructuralInterfacePortVisitor::PrintStructuralInterfaceRolePairs()
{
	GMEConsole::Console::writeLine("StructuralInterfacePortVisitor::PrintStructuralInterfacePairs()", MSG_INFO);

	for (unsigned int i = 0; i < m_StructuralInterfaceRolePairs.size(); i++)
	{
		GMEConsole::Console::writeLine("portName: " + std::string(m_StructuralInterfaceRolePairs[i].name()) + " portID:" + UdmGme::UdmId2GmeId(m_StructuralInterfaceRolePairs[i].uniqueId()), MSG_INFO);
	}
}


CyPhyML::StructuralInterface StructuralInterfacePortVisitor::FindConnectedSIRPort(const CyPhyML::StructuralInterface& beginPort)
{
	CyPhyML::StructuralInterface endPort;
	Visit_StructuralInterface(beginPort);

	for (unsigned int i = 0; i < m_StructuralInterfaceRolePairs.size(); i++)
	{
		if (m_StructuralInterfaceRolePairs[i] != beginPort)
		{
			endPort = m_StructuralInterfaceRolePairs[i];
			break;
		}
	}

	return endPort;
}

void StructuralInterfacePortVisitor::FindConnectedSIRPorts_Set(const CyPhyML::StructuralInterface& beginPort, 
																set<CyPhyML::StructuralInterface>& connectedPorts)
{
	Visit_StructuralInterface(beginPort);

	for (unsigned int i = 0; i < m_StructuralInterfaceRolePairs.size(); i++)
	{
		if (m_StructuralInterfaceRolePairs[i] != beginPort)
			connectedPorts.insert(m_StructuralInterfaceRolePairs[i]);
	}
}


////// AnalysisPointVisitor
void AnalysisPointVisitor::FindAnalysisPointEnds(CyPhyML::AnalysisPoint &start_in)
{
	set<CyPhyML::Point> found;
	start_in.Accept(*this);
}

void AnalysisPointVisitor::Visit_AnalysisPointMap(const CyPhyML::AnalysisPointMap &ap_map_in)
{
	if (visitedAPMap.find(ap_map_in) == visitedAPMap.end())
	{
		visitedAPMap.insert(ap_map_in);

		{
			CyPhyML::AnalysisFeatures src_feature = ap_map_in.srcAnalysisPointMap_end();
			if (src_feature.type() == CyPhyML::PointGeometry::meta)
			{
				pointsFound.insert(CyPhyML::PointGeometry::Cast(src_feature));
			}
			else
				CyPhyML::AnalysisPoint::Cast(src_feature).Accept(*this);
		}

		{
			CyPhyML::AnalysisFeatures dst_feature = ap_map_in.dstAnalysisPointMap_end();
			if (dst_feature.type() == CyPhyML::PointGeometry::meta)
			{
				pointsFound.insert(CyPhyML::PointGeometry::Cast(dst_feature));
			}
			else
				CyPhyML::AnalysisPoint::Cast(dst_feature).Accept(*this);
		}
	}
}


void AnalysisPointVisitor::Visit_AnalysisPoint(const CyPhyML::AnalysisPoint &analysis_pt_in)
{
	set<CyPhyML::AnalysisPointMap> src_conn_set = analysis_pt_in.srcAnalysisPointMap();
	for (set<CyPhyML::AnalysisPointMap>::const_iterator ci = src_conn_set.begin(); ci != src_conn_set.end(); ci++)
	{
		CyPhyML::AnalysisPointMap apConn = *ci;
		apConn.Accept(*this);
	}

	set<CyPhyML::AnalysisPointMap> dst_conn_set = analysis_pt_in.dstAnalysisPointMap();
	for (set<CyPhyML::AnalysisPointMap>::const_iterator ci = dst_conn_set.begin(); ci != dst_conn_set.end(); ci++)
	{
		CyPhyML::AnalysisPointMap apConn = *ci;
		apConn.Accept(*this);
	}
}

CyPhyML::PointGeometry AnalysisPointVisitor::GetFoundPoint()
{
	CyPhyML::PointGeometry point;

	if (!pointsFound.empty())
		point = *pointsFound.begin();

	return point;
}

////// end AnalysisPointVisitor


//set<CyPhyML::RefCoordSystem2RefCoordSystem> visited_Connection;
//set<CyPhyML::ReferenceCoordinateSystem> visited_System;
void RefCoordinateSystemVisitor::Visit_ReferenceCoordinateSystem(const CyPhyML::ReferenceCoordinateSystem &refCoord)
{
	if (visited_CoordSystems.find(refCoord) == visited_CoordSystems.end())
	{
		CyPhyML::DesignEntity deParent = CyPhyML::DesignEntity::Cast(refCoord.parent());
		if (deParent.type() == CyPhyML::Component::meta)
		{
			FoundRefCoordComponents.insert(CyPhyML::Component::Cast(deParent));
			FoundRefCoordIDs.insert(deParent.uniqueId());
		}

		visited_CoordSystems.insert(refCoord);

		set<CyPhyML::RefCoordSystem2RefCoordSystem> src_Set = refCoord.srcRefCoordSystem2RefCoordSystem();
		set<CyPhyML::RefCoordSystem2RefCoordSystem> dst_Set = refCoord.dstRefCoordSystem2RefCoordSystem();

		for (std::set<CyPhyML::RefCoordSystem2RefCoordSystem>::iterator i = src_Set.begin(); i != src_Set.end(); i++)
		{
			CyPhyML::RefCoordSystem2RefCoordSystem conn(*i);
			conn.Accept(*this);
		}
	
		for (std::set<CyPhyML::RefCoordSystem2RefCoordSystem>::iterator i = dst_Set.begin(); i != dst_Set.end(); i++)
		{
			CyPhyML::RefCoordSystem2RefCoordSystem conn(*i);
			conn.Accept(*this);
		}
	}
}

void RefCoordinateSystemVisitor::Visit_RefCoordSystem2RefCoordSystem(const CyPhyML::RefCoordSystem2RefCoordSystem &conn)
{
	visited_Connection.insert(conn);

	CyPhyML::ReferenceCoordinateSystem src = conn.srcRefCoordSystem2RefCoordSystem_end();
	CyPhyML::ReferenceCoordinateSystem dst = conn.dstRefCoordSystem2RefCoordSystem_end();

	src.Accept(*this);
	dst.Accept(*this);
}