/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "Convert2DAG.h"
#include "string_utils.h"
#include "UdmGme.h"
#include "Logger.h"

#include "CommonData.h"


void Convert2DAG::ToDAG(list<CAssembly*> &cassemblyParent_in, 
						set<long> &RefCoordSystemComponentIDs_in)
{
	for (list<CAssembly*>::iterator i = cassemblyParent_in.begin(); i != cassemblyParent_in.end(); i++)
	{
		ToDAG(*i, RefCoordSystemComponentIDs_in);
	}
}

void Convert2DAG::ToDAG(CAssembly* cassemblyParent_in,
						set<long> &RefCoordSystemComponentIDs_in)
{
	//LOGMSG("AddConstraints rootCompID:" + UdmGme::UdmId2GmeId(rootComponentID), MSG_INFO);
	vector<CEdge*> edges2Process;
	deque<long> components2Process;
	DisableEdges(cassemblyParent_in, edges2Process);
	
	//int size = RefCoordSystemComponentIDs.size();
	long rootComponentID = cassemblyParent_in->child_Component[0]->GetID();
	for (vector<CComponent*>::const_iterator ci = cassemblyParent_in->child_Component.begin(); ci != cassemblyParent_in->child_Component.end(); ci++)
	{
		long tmp = (*ci)->GetID();
		if (RefCoordSystemComponentIDs_in.find(tmp) != RefCoordSystemComponentIDs_in.end())
		{
			rootComponentID = tmp;
			break;		
		}
	}
	
	cassemblyParent_in->SetRootComponentID(rootComponentID);

	components2Process.push_back(rootComponentID);
	AssignEdgeConstraints(components2Process, edges2Process);		// (long myComponentID, long rootComponentID, vector<CEdge*>& edge_vector, map<long, pair<int,int>>& tally)

	// add constraints to subAssemblies
	for (unsigned int i = 0; i < cassemblyParent_in->child_Assembly.size(); i++)
	{
		CAssembly* childAssembly = cassemblyParent_in->child_Assembly[i];
		if (childAssembly->enabled)
			this->ToDAG(childAssembly, RefCoordSystemComponentIDs_in);
	}
}


void Convert2DAG::DisableEdges(CAssembly* assemblyParent, vector<CEdge*>& edges2Process)
{
	vector<CEdge*> edges_vector = assemblyParent->child_Edges;
	vector<CForwarder*> child_Forwarders = assemblyParent->child_Forwarders;
	for (unsigned int i = 0; i < edges_vector.size(); i++)
	{
		CEdge* edge = edges_vector[i];

		CADData* dst_cdata = edge->destination;
		CADData* src_cdata = edge->source;
		
		if (src_cdata->GetType() == "Forwarder")
		{
			if (assemblyParent->IsMyForwarder(src_cdata->GetID()))
			{
				edge->processed = 1;
				continue;
			}
		}

		if (dst_cdata->GetType() == "Forwarder")
		{
			if (assemblyParent->IsMyForwarder(dst_cdata->GetID()))
			{
				edge->processed = 1;
				continue;
			}
		}

		if (!edge->processed)
			edges2Process.push_back(edge);
	}

}


/** \brief Recursive function to assign constraints
	\param [in] cassemblyParent Top level CAssembly parent
    \return void
*/

// A \
// |  \ 
// C--B
// 
// A (R) --> B
// A (R) --> C
// B --> C or C --> B
void Convert2DAG::AssignEdgeConstraints(deque<long>& components2Process, vector<CEdge*>& edges2Process)
{	
	vector<CEdge*> newEdges;

	long cnt_Src_ID, cnt_Dst_ID;
	//if (components2Process.empty())
	//	return;

	
	cnt_Src_ID = components2Process.front();
	components2Process.pop_front();

	for (unsigned int i = 0; i < edges2Process.size(); i++)
	{
		CEdge* curEdge =  edges2Process[i];

		if (curEdge->processed)
			continue;

		Udm::Object src_udmObj;
		Udm::Object dst_udmObj;	
		
		if (curEdge->source->GetType() == "Forwarder")
			src_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(curEdge->source->GetMyUDMObj()).parent();
		else
			src_udmObj = curEdge->source->GetMyUDMObj();

		if (curEdge->destination->GetType() == "Forwarder")
			dst_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(curEdge->destination->GetMyUDMObj()).parent();
		else
			dst_udmObj = curEdge->destination->GetMyUDMObj();

		bool myEdge = 0;
		if (cnt_Src_ID == src_udmObj.uniqueId())
		{
			cnt_Dst_ID = dst_udmObj.uniqueId();
			myEdge = 1;
		}
		else if (cnt_Src_ID == dst_udmObj.uniqueId())
		{
			// Reverse Src and Dst in CEdge
			CADData* tmpCADData = curEdge->source;
			long tmpPort = curEdge->sourcePort;			
			curEdge->source = curEdge->destination;			
			curEdge->sourcePort = curEdge->destinationPort;
			curEdge->destination = tmpCADData;
			curEdge->destinationPort = tmpPort;
			// end

			cnt_Dst_ID = src_udmObj.uniqueId();
			myEdge = 1;
		}

		if (myEdge == 1)
		{
			AddComponent2Process(components2Process, cnt_Dst_ID);
			curEdge->processed = 1;
		}
		else
			newEdges.push_back(curEdge);				// for next go around
	}
	
	// Process more edges
	if (!components2Process.empty())
		AssignEdgeConstraints(components2Process, newEdges);
}


void Convert2DAG::AddComponent2Process(deque<long>& components2Process, long newID)		// helper
{
	for (unsigned int i = 0; i < components2Process.size(); i++)
	{
		if (components2Process[i] == newID)
			return;
	}

	components2Process.push_back(newID);
}


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////							Convert2DAG_DegreesFreedom 							///////////////////////////
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
Convert2DAG_DOF::Convert2DAG_DOF()
{
	//log_file_.open(Globals::Instance()->CadXmlOutputDirectory + "\\debug_log\\" + "Convert2DAG.log");
	log_file_.open(Globals::Instance()->DebugLogDirectory + "\\" + "CyPhy2CAD_Convert2DAG.log");
}

Convert2DAG_DOF::~Convert2DAG_DOF()
{
	log_file_.close();
}

void Convert2DAG_DOF::ToDAG(list<CAssembly*> &cassemblyParent_in, 
									set<long> &RefCoordSystemComponentIDs_in)
{
	for (list<CAssembly*>::iterator i = cassemblyParent_in.begin(); i != cassemblyParent_in.end(); i++)
	{
		ToDAG(*i, RefCoordSystemComponentIDs_in);
	}
}

void Convert2DAG_DOF::ToDAG(CAssembly* cassemblyParent_in,
				set<long> &RefCoordSystemComponentIDs_in)
{
	long start_component_id = -1;
	set<long> openComponents,  closedComponents; 

	// [1] Add to lookup table
	for (unsigned int i = 0; i < cassemblyParent_in->child_Component.size(); i++)
	{
		display_id_lookup_[cassemblyParent_in->child_Component[i]->GetID()] = cassemblyParent_in->child_Component[i]->GetAssemblyInterface_ID();
	}

	// [2] Create openEdges
	vector<CEdge*> openEdges = cassemblyParent_in->child_Edges;

	// [3] Find starting component and create openComponents set
	start_component_id = FindStartingComponent(cassemblyParent_in->child_Component, RefCoordSystemComponentIDs_in, openEdges, openComponents);
	closedComponents.insert(start_component_id);	
	log_file_<< "***** Root Component ID: " << display_id_lookup_[start_component_id] << " *****\n";
	cassemblyParent_in->SetRootComponentID(start_component_id);

	int degree = 6, openSize = openComponents.size();
	while(!openComponents.empty())
	{
		ConstrainComponents(closedComponents, openEdges, degree, openComponents);
		int tmp_open_size = openComponents.size();

		if (tmp_open_size == openSize)			// was not able to find components with edges of dof 6
			degree--;
		else
			degree = 6;

		openSize = tmp_open_size;
	}

	if (openEdges.size() > 0)
	{
		log_file_<< "+++++ Left Over Edges +++++" << "\n";
		for (unsigned int i = 0; i < openEdges.size(); i++)
			log_file_<< "		Source: " << display_id_lookup_[openEdges[i]->source->GetID()] << "Dst: " << display_id_lookup_[openEdges[i]->destination->GetID()] << "\n";
	}
}

void Convert2DAG_DOF::ConstrainComponents(set<long> &closedComponents_in,
					vector<CEdge*> &openEdges_in,
					int degree_in,
					set<long> &openComponents_inout)
{
	bool search = true;
	set<long> newOpenComponents;
	for (set<long>::const_iterator ci = openComponents_inout.begin(); ci != openComponents_inout.end(); ci++)
	{
		long my_component_id = *ci;
		if (search)
		{
			if ( !FindEdges(my_component_id, degree_in, closedComponents_in, openEdges_in))
				newOpenComponents.insert(my_component_id);
			else
			{
				if (degree_in < 6)
					search = false;
			}
		}
		else
			newOpenComponents.insert(my_component_id);
	}

	openComponents_inout = newOpenComponents;
}


bool Convert2DAG_DOF::FindEdges(long my_component_id_in, 
								int degree_in,
								set<long> &closedComponents_inout, 
								vector<CEdge*> &openEdges_inout)
{	
	bool status;
	vector<CEdge*> newOpenEdges;
	vector<CEdge*> found_edges_set;

	int totalDegrees = 0;
	for (unsigned int i = 0; i < openEdges_inout.size(); i++)
	{
		CEdge* cur_edge = openEdges_inout[i];
		
		Udm::Object src_udmObj;
		Udm::Object dst_udmObj;	

		if (cur_edge->source->GetType() == "Forwarder")
			src_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(cur_edge->source->GetMyUDMObj()).parent();
		else
			src_udmObj = cur_edge->source->GetMyUDMObj();

		if (cur_edge->destination->GetType() == "Forwarder")
			dst_udmObj = CyPhyML::StructuralInterfaceForwarder::Cast(cur_edge->destination->GetMyUDMObj()).parent();
		else
			dst_udmObj = cur_edge->destination->GetMyUDMObj();


		long srcID = src_udmObj.uniqueId(), dstID = dst_udmObj.uniqueId();
		if (my_component_id_in == srcID && closedComponents_inout.find(dstID) != closedComponents_inout.end())
		{
			// Reverse Src and Dst in CEdge
			CADData* tmpCADData = cur_edge->source;
			long tmpPort = cur_edge->sourcePort;			
			cur_edge->source = cur_edge->destination;			
			cur_edge->sourcePort = cur_edge->destinationPort;
			cur_edge->destination = tmpCADData;
			cur_edge->destinationPort = tmpPort;
			// end

			totalDegrees += cur_edge->GetDegreeOfFreedom();
			found_edges_set.push_back(cur_edge);	
		}
		else if (my_component_id_in == dstID && closedComponents_inout.find(srcID) != closedComponents_inout.end())
		{
			totalDegrees += cur_edge->GetDegreeOfFreedom();
			found_edges_set.push_back(cur_edge);
		}
		else
			newOpenEdges.push_back(cur_edge);
	}

	if (totalDegrees < degree_in)
	{
		for (unsigned int j = 0; j < found_edges_set.size(); j++)
			newOpenEdges.push_back(found_edges_set[j]);		// insert edges back into newOpenEdges
		status = false;
	}
	else
	{
		log_file_<< display_id_lookup_[my_component_id_in] << " - DOF(" << totalDegrees << ")" << "\n";
		for (unsigned int j = 0; j < found_edges_set.size(); j++)
			log_file_<< "		Source: " << display_id_lookup_[found_edges_set[j]->source->GetID()] << " Dst: " << display_id_lookup_[found_edges_set[j]->destination->GetID()] << "\n";

		closedComponents_inout.insert(my_component_id_in);
		status = true;
	}
		
	openEdges_inout = newOpenEdges;

	return status;
}


// Find root component ID
// Populate openComponents set
long Convert2DAG_DOF::FindStartingComponent(vector<CComponent*> &components_in,
											set<long> &RefCoordSystemComponentIDs_in,
											vector<CEdge*> &openEdges_in,
											set<long> &openComponents_inout)
{
	long start_id = -1;

	for (vector<CComponent*>::const_iterator ci = components_in.begin(); ci != components_in.end(); ci++)
	{
		long tmp = (*ci)->GetID();
		if ( RefCoordSystemComponentIDs_in.find(tmp) != RefCoordSystemComponentIDs_in.end() )
		{
			start_id = tmp;
			break;
		}
	}

	int degrees = 6;
	while (start_id == -1 && degrees > 2)		// if not in RefCoordSystemComponentIDs_in
	{
		map<long, int> id_degree_table;
		for (unsigned int i = 0; i < openEdges_in.size(); i++)
		{
			if (openEdges_in[i]->GetDegreeOfFreedom() == degrees)
			{
				long src_id = openEdges_in[i]->source->GetID(), dst_id = openEdges_in[i]->destination->GetID();
				if (id_degree_table[src_id] > 0)
					id_degree_table[src_id] = id_degree_table[src_id] + 1;
				else
					id_degree_table[src_id] = 1;

				if (id_degree_table[dst_id] > 0)
					id_degree_table[dst_id] = id_degree_table[dst_id] + 1;
				else
					id_degree_table[dst_id] = 1;
			}
		}

		int max_dof = 0;
		long max_dof_component_id = -1;

		for (map<long, int>::const_iterator di = id_degree_table.begin(); di != id_degree_table.end(); di++)
		{
			if (di->second > max_dof)
			{
				max_dof = di->second;
				max_dof_component_id = di->first;
			}
		}

		if (max_dof_component_id > -1)
			start_id = max_dof_component_id;
		else
			degrees--;
	}

	if (start_id == -1)
		start_id = (*components_in.begin())->GetID();

	for (vector<CComponent*>::const_iterator ci = components_in.begin(); ci != components_in.end(); ci++)
	{
		long tmp = (*ci)->GetID();
		if (start_id != tmp)
			openComponents_inout.insert(tmp);
	}

	return start_id;
}