/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "CyPhy2CADHelper.h"
#include "UdmConsole.h"
#include "CommonHelper.h"
#include "CADDataCreator.h"
#include "UdmFormatter.h"
#include "Logger.h"
#include "CheckConstraint.h"
#include "AssemblyOutputWriter.h"
#include "Convert2DAG.h"
#include "CommonData.h"
#include "CEdgeAnalyzer.h"
#include "string_utils.h"

#include <algorithm>


void CyPhy2CADHelper::GenerateCADFile(CyPhyML::ComponentAssembly &assembly, long startingElement)
{
	// [1] Check constraints in the model
	// [2] CreateIntermediateData + Flatten
	// [3] WriteGraph
	// [4] FindIslands
	// [5] AssignConstraints
	// [6] WriteAssemblyInterfaceFile
	// [7] WriteGraph

	
	set<CyPhyML::Component> AllCyPhyComponents;
	set<CyPhyML::Component> AllCyPhySize2Fit;
	set<long> RefCoordComponentIDs;

	CADDataCreator_Flatten dataCreator;
	Checker checkConstraint;
	int errors = 0, warnings = 0;
	if(checkConstraint.CheckConstraintStart(assembly, AllCyPhyComponents, AllCyPhySize2Fit, RefCoordComponentIDs, errors, warnings))
	{

		CAssembly *topAssemblyCADData = dataCreator.CreateINTCADStructure(assembly, AllCyPhyComponents, AllCyPhySize2Fit);


		AssemblyOutputWriter writer(Globals::Instance()->CadXmlOutputDirectory, Globals::Instance()->CadArtifactDirectory, assembly.name(), std::string(assembly.ConfigurationUniqueID()));				// string out, string cad, string name, int configID

#ifdef _DEBUG		
		//writer.WriteCADData(topAssemblyCADData);
		//writer.WriteGraph(topAssemblyCADData);
#endif

		list<CAssembly*> islands;
		CEdgeAnalyzer analyzer;
		analyzer.AnalyzeEdges(topAssemblyCADData, islands);	

		writer.WriteCADData(islands);
		writer.WriteGraph(islands, analyzer.GetOrphanComponents(), true);


		if (!islands.empty())
		{
			Convert2DAG_DOF converter;				//Convert2DAG converter;
			converter.ToDAG(islands, RefCoordComponentIDs);

			writer.WriteAssemblyInterfaceFile(islands, analyzer.GetOrphanComponents());
		}
		else
		{
			warnings++;
			LOGMSG("No connected Components exist!", MSG_WARNING);
		}

		CommonData::CleanData();
	}

	ShowStatus(errors, warnings);

	return;

}

string LongToString(long input)
{		
	std::stringstream strstream;
	strstream << input;
	return strstream.str();
}

void CyPhy2CADHelper::GenerateCADFile(CyPhyML::CADTestBench& testBench)
{
	// [1] Check constraints in the model
	// [2] CreateIntermediateData, already flattened
	// [3] WriteGraph
	// [4] FindIslands
	// if (island)
	//		exit
	// [5] AssignConstraints
	// [6] WriteAssemblyInterfaceFile
	// [7] WriteGraph

	
	set<CyPhyML::Component> AllCyPhyComponents;
	set<CyPhyML::Component> AllCyPhySize2Fit;
	set<long> RefCoordComponentIDs;


	Checker checkConstraint;
	int errors = 0, warnings = 0;
	if(checkConstraint.CheckConstraintStart(testBench, AllCyPhyComponents, AllCyPhySize2Fit, errors, warnings))
	{
		CADDataCreator_Flatten dataCreator;
		CAssembly* topAssemblyCADData = dataCreator.CreateINTCADStructure(testBench, AllCyPhyComponents, AllCyPhySize2Fit);	

		AssemblyOutputWriter writer(Globals::Instance()->CadXmlOutputDirectory, Globals::Instance()->CadArtifactDirectory, testBench.name(), LongToString((long)testBench.uniqueId()));
#ifdef _DEBUG		
		writer.WriteGraph(topAssemblyCADData);	
		writer.WriteCADData(topAssemblyCADData);
#endif

		list<CAssembly*> islands;
		CEdgeAnalyzer analyzer;
		analyzer.AnalyzeEdges(topAssemblyCADData, islands);	

		writer.WriteCADData(islands);
		writer.WriteGraph(islands, analyzer.GetOrphanComponents(), true);

		if (islands.size() == 1)
		{
			Convert2DAG_DOF converter;
			converter.ToDAG(islands, RefCoordComponentIDs);

			writer.WriteCADData(islands);
			writer.WriteGraph(islands, analyzer.GetOrphanComponents(), true);

			writer.WriteAssemblyInterfaceFile(*islands.begin(), testBench);			//writer.WriteAssemblyInterfaceFile(islands, analyzer.GetOrphanComponents());
		}
		else if (islands.size() < 1)
		{
			errors++;
			LOGMSG("Structural FEA Test Bench has no Test Injection Points or valid ones that references Components! Can not perform FEA!", MSG_ERROR);
		}
		else
		{
			errors++;
			LOGMSG("Structural FEA Test Bench has islands of connected components! Can not perform FEA on multiple islands!", MSG_ERROR);
		}
		
		CommonData::CleanData();
	}

	ShowStatus(errors, warnings);

	return;
}


void CyPhy2CADHelper::GenerateCADFile(std::set<CyPhyML::ComponentAssembly> &componentAssembly_set)
{
	for (std::set<CyPhyML::ComponentAssembly>::const_iterator ci = componentAssembly_set.begin(); ci != componentAssembly_set.end(); ci++)
	{
		CyPhyML::ComponentAssembly assembly = *ci;
		GenerateCADFile(assembly);
	}
}

void CyPhy2CADHelper::GenerateCADFile(CyPhyML::TestBench& testBench)
{
	if (testBench != Udm::null)
	{	
		int errors = 0, warnings = 0;

		Checker checkConstraint;
		checkConstraint.CheckConstraintStart(testBench);
		set<CyPhyML::ComponentAssembly> assemblySet = testBench.ComponentAssembly_kind_children();
		if (assemblySet.size() > 0)
		{
			CyPhyML::ComponentAssembly assembly(*assemblySet.begin());

			set<CyPhyML::Component> AllCyPhyComponents;
			set<CyPhyML::Component> AllCyPhySize2Fit;
			set<long> RefCoordComponentIDs;

			CADDataCreator_Flatten dataCreator;
			if(checkConstraint.CheckConstraintStart(assembly, AllCyPhyComponents, AllCyPhySize2Fit, RefCoordComponentIDs, errors, warnings))
			{
				CAssembly *topAssemblyCADData = dataCreator.CreateINTCADStructure(assembly, AllCyPhyComponents, AllCyPhySize2Fit);

				AssemblyOutputWriter writer(Globals::Instance()->CadXmlOutputDirectory, Globals::Instance()->CadArtifactDirectory, assembly.name(), std::string(assembly.ConfigurationUniqueID()));				// string out, string cad, string name, int configID

#ifdef _DEBUG
				writer.WriteCADData(topAssemblyCADData);
				writer.WriteGraph(topAssemblyCADData);
#endif

				list<CAssembly*> islands;
				CEdgeAnalyzer analyzer;
				analyzer.AnalyzeEdges(topAssemblyCADData, islands);	

				writer.WriteCADData(islands);
				writer.WriteGraph(islands, analyzer.GetOrphanComponents(), true);


#if 0
				if (!islands.empty())
				{
					if (islands.size() > 1 || !analyzer.GetOrphanComponents().empty())
					{
						LOGMSG("Test Bench has islands of connected components and/or orphan components! BoundingBox, CG, and PointCoordiantes metrics would not be calculated!", MSG_ERROR);
					}

					Convert2DAG_DOF converter;
					converter.ToDAG(islands, RefCoordComponentIDs);

					writer.WriteAssemblyInterfaceFile(islands, analyzer.GetOrphanComponents(), testBench);
				}
				else
				{
					warnings++;
					LOGMSG("TestBench's does not have connected components!", MSG_WARNING);
				}
#endif

				set<CyPhyML::CADComputation2Metric> cadcomp_set = testBench.CADComputation2Metric_kind_children();
				set<CyPhyML::PointCoordinates2Metric> ptcoord_set = testBench.PointCoordinates2Metric_kind_children();
				bool has_cadcomputation = (cadcomp_set.size() + ptcoord_set.size()) > 0 ? true : false;

				if (islands.empty() && analyzer.GetOrphanComponents().empty())
				{
					warnings++;
					LOGMSG("TestBench has no connected components and no components with valid cadmodel links. Nothing will be generated.", MSG_WARNING);
				}
				else
				{
					if (has_cadcomputation && (islands.size() > 1 || !analyzer.GetOrphanComponents().empty()))
					{
						errors++;
						LOGMSG("Test Bench has islands of connected components and/or orphan components, metrics can not be computed. Please remove the metrics or orphan components from test bench and try again.", MSG_ERROR);
					}
					else
					{
						Convert2DAG_DOF converter;
						converter.ToDAG(islands, RefCoordComponentIDs);

						writer.WriteAssemblyInterfaceFile(islands, analyzer.GetOrphanComponents(), testBench);
					}
				}

				CommonData::CleanData();
			}
		}
		else
		{
			set<CyPhyML::TopLevelSystemUnderTest> tlsut_set = testBench.TopLevelSystemUnderTest_kind_children();
			if (!tlsut_set.empty())
				LOGMSG("TestBench has a TopLevelSystemUnderTest that has not been elaborated. Nothing will be generated.", MSG_WARNING);
			else
				LOGMSG("TestBench has a TopLevelSystemUnderTest that does not refer to a ComponentAssembly. Nothing will be generated.", MSG_WARNING);

			warnings++;
		}

		ShowStatus(errors, warnings);
	}
}


void CyPhy2CADHelper::ShowStatus(int & errors, 
								int &warnings)
{
	string tmp, message;
	LOGMSG("==========================================================================", MSG_WARNING);
	to_string(tmp, errors);
	message = tmp + "   errors in model, ";
	to_string(tmp, warnings);
	message += "           " + tmp + " warnings in model";
	LOGMSG(message, MSG_WARNING);

	if (errors > 0)
	{
		LOGMSG("There are errors! Interpreter can not proceed until errors have been corrected. Nothing is generated.", MSG_ERROR);
		string logfolder = "<a href=\"file:///" + Globals::Instance()->DebugLogDirectory + "\" target=\"_blank\">" + Globals::Instance()->DebugLogDirectory +"</a>";
		LOGMSG("See CyPhy2CAD.log for details: " + logfolder, MSG_WARNING);
	}
		
	LOGMSG("==========================================================================", MSG_WARNING);
}