/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "CyPhy2CADVisitor.h"
#include "string_utils.h"
//#include "UdmConsole.h"
#include "UdmFormatter.h"

///////////////////////////////////////////////////// CLASS CyPhy2CADVisitor /////////////////////////////////////////////////////
void CyPhy2CADVisitor::TraverseAssembly(CyPhyML::ComponentAssembly &assembly, long startingElement)
{
//	GMEConsole::Console::writeLine("TraverseAssembly() - AN:" + std::string(assembly.name()), MSG_INFO);

	AssemblyInterface::CADComponent currenttop;
	std::map<long, AssemblyInterface::CADComponent>::iterator i = m_CadComponentMap.find(assembly.uniqueId());
	if (i == m_CadComponentMap.end())
	{	
		currenttop = AssemblyInterface::CADComponent::Create(this->m_CadComponentParent);   // new
		FillInAttributes(assembly, currenttop);
		m_CadComponentMap[assembly.uniqueId()] = currenttop;
		this->m_CadComponentParent = currenttop;
	}
	else
		currenttop = i->second;
		
	set<CyPhyML::DesignElement> DesignElementChildren_Set = assembly.DesignElement_kind_children();
	if (DesignElementChildren_Set.size() == 0)
		GMEConsole::Console::writeLine("CyPhy2CADVisitor::TraverseAssembly() Error: Assembly [" + std::string(assembly.name()) + "] contains no design elements", MSG_INFO);
	else
	{
		CyPhyML::DesignElement RootDE;
		
		if (startingElement < 0)
		{
			RootDE = this->FindRootDesignElement(assembly);
		}
		else
		{
			bool found = 0;
			set<CyPhyML::DesignElement>::const_iterator cit = DesignElementChildren_Set.begin();
			for (; cit != DesignElementChildren_Set.end(); cit++)
			{
				if (cit->uniqueId() == startingElement && !IsSize2Fit(*cit))       // not size2fit
				{
					RootDE = *cit;
					found = 1;
					break;
				}
			}
			if (!found)
				RootDE = this->FindRootDesignElement(assembly);
		}

		if (RootDE != Udm::null)
		{
			this->m_CurrentRootComponent = RootDE.uniqueId();
		//	GMEConsole::Console::writeLine("Root Component --> " + std::string(RootDE.name()), MSG_INFO);
			AssemblyInterface::CADComponent cadRoot = AssemblyInterface::CADComponent::Create(currenttop);
			this->FillInAttributes(RootDE, cadRoot);
			this->m_CadComponentMap[RootDE.uniqueId()] = cadRoot;
			this->CreateRootComponentConstraints(cadRoot, UdmGme::UdmId2GmeId(assembly.uniqueId()));

			TraverseDesignElement(RootDE);	
		}
	}

	std::map<long, std::set<CyPhyML::ComponentAssembly> >::iterator aIt = this->m_assemblyToBeVisited.find(assembly.uniqueId());
	if (aIt != this->m_assemblyToBeVisited.end())
	{
		set<CyPhyML::ComponentAssembly> AssemblyChildren_Set = aIt->second;
		for (set<CyPhyML::ComponentAssembly>::iterator i = AssemblyChildren_Set.begin(); i != AssemblyChildren_Set.end(); i++)
		{
			this->m_CadComponentParent = currenttop;

			AssemblyInterface::CADComponent cadComponent;
		
			CyPhyML::ComponentAssembly childAssembly(*i);		
			std::map<long, AssemblyInterface::CADComponent>::iterator j = this->m_CadComponentMap.find(childAssembly.uniqueId());
			if (j == m_CadComponentMap.end())
				cadComponent = AssemblyInterface::CADComponent::Create(this->m_CadComponentParent);// new CADComponent
			else
				cadComponent = j->second;

			this->m_CadComponentParent = cadComponent;		
			TraverseAssembly(childAssembly);
		}
	}
	

	this->m_CadComponentParent = currenttop;
//	GMEConsole::Console::writeLine("TraverseAssembly() END - AN:" + std::string(assembly.name()), MSG_INFO);
}


void CyPhy2CADVisitor::TraverseDesignElement(CyPhyML::DesignElement &designElement)
{
//	GMEConsole::Console::writeLine("===== TraverseDesignElement() - [" + std::string(designElement.name()) + "," + UdmGme::UdmId2GmeId(designElement.uniqueId()) + "]", MSG_INFO);
	std::string name = designElement.name(), id = UdmGme::UdmId2GmeId(designElement.uniqueId());

	Udm::Object currenttop = this->m_CadComponentParent;
	bool iamSize2Fit = IsSize2Fit(designElement), iamCADBypass = this->IsCADByPass(designElement);

	if (!iamCADBypass)
	{
		if (designElement.type() == CyPhyML::ComponentAssembly::meta)                                                       
		{
			Udm::Object parentObj = designElement.parent();
			this->m_assemblyToBeVisited[parentObj.uniqueId()].insert(CyPhyML::ComponentAssembly::Cast(designElement));
		}

		set<CyPhyML::StructuralPortType> structuralPort_Set = designElement.StructuralPortType_kind_children();
		for (set<CyPhyML::StructuralPortType>::iterator i = structuralPort_Set.begin(); i != structuralPort_Set.end(); i++)
		{
			CyPhyML::StructuralPortType StructPort(*i);
			std::vector<JoinStructureAndEndPair> SelectedJoinStructures_Set;		
	#if 0		
			this->PrintJoinStructuresVisited();
	#endif
			this->FindConnectedJoinStructuresAndEnd(StructPort, SelectedJoinStructures_Set);			//  std::pair<CyPhyML::DesignElement, CyPhyML::JoinStructures>

	#if 0
			GMEConsole::Console::writeLine("Printing join structures found for DE:" + UdmGme::UdmId2GmeId(designElement.uniqueId()), MSG_INFO);
			this->PrintFindConnectedJoinStructuresAndEnd(SelectedJoinStructures_Set);
			GMEConsole::Console::writeLine("End of printing", MSG_INFO);
	#endif

			for (unsigned int i = 0; i < SelectedJoinStructures_Set.size(); i++)
			{
				// Have we visited this join structure?
				if (!IsNewJoinStructure(SelectedJoinStructures_Set[i].second.uniqueId()))
				{
					//GMEConsole::Console::writeLine("Have already visited this join from:" + UdmGme::UdmId2GmeId(designElement.uniqueId()) + " to:" + UdmGme::UdmId2GmeId(SelectedJoinStructures_Set[i].first.uniqueId()) + " via JS:" + UdmGme::UdmId2GmeId(SelectedJoinStructures_Set[i].second.uniqueId()), MSG_INFO);
					continue;
				}

				// Find connected StructuralInterfaceRole ports given a JoinStructure
				StructuralInterfacePortVisitor sirVisitor;
				sirVisitor.Visit_JoinStructures(SelectedJoinStructures_Set[i].second);

				AssemblyInterface::CADComponent cadComponent;

				if (sirVisitor.m_StructuralInterfaceRolePairs.size() < 2)
				{
					GMEConsole::Console::writeLine("         *     ERROR: <2 StructuralInterfaceRolePorts for starting Join Structure", MSG_ERROR);				
					GMEConsole::Console::writeLine("         *     " + GMEConsole::Formatter::MakeObjectHyperlink("Starting Join Structures", SelectedJoinStructures_Set[i].second) + "[" + GMEConsole::Formatter::MakeObjectHyperlink("Design Element 1", SelectedJoinStructures_Set[i].first) + "," + GMEConsole::Formatter::MakeObjectHyperlink("Design Element 2", designElement) +"]", MSG_ERROR);
				}
				else if (sirVisitor.m_StructuralInterfaceRolePairs.size() > 2)
				{
					GMEConsole::Console::writeLine("         *     ERROR: >2 StructuralInterfaceRole Ports Found for starting Join Structure", MSG_ERROR);
					GMEConsole::Console::writeLine("         *     " + GMEConsole::Formatter::MakeObjectHyperlink("Starting Join Structures", SelectedJoinStructures_Set[i].second) + "[" + GMEConsole::Formatter::MakeObjectHyperlink("Design Element 1", SelectedJoinStructures_Set[i].first) + "," + GMEConsole::Formatter::MakeObjectHyperlink("Design Element 2", designElement) +"]", MSG_ERROR);
				}
				else
				{
					/*
					std::map<long, CyPhyML::StructuralInterfaceRole>::iterator j = sirVisitor.m_StructuralInterfaceRolePairs.begin();
					CyPhyML::StructuralInterfaceRole a = j->second;
					j++;
					CyPhyML::StructuralInterfaceRole b = j->second;
					*/
					CyPhyML::StructuralInterfaceRole a = sirVisitor.m_StructuralInterfaceRolePairs[0];
					CyPhyML::StructuralInterfaceRole b = sirVisitor.m_StructuralInterfaceRolePairs[1];
					CyPhyML::DesignElement a_parent = a.DesignElement_parent(), b_parent = b.DesignElement_parent();				// design elements that contains the SIRs
					CyPhyML::DesignElement sirDst_DE = SelectedJoinStructures_Set[i].first;                                         // design element with same parent as me that could take this constraint if i don't take it
					bool isA_S2F = IsSize2Fit(a_parent), isB_S2F = IsSize2Fit(b_parent);                                            // size2fit
					bool isA_ByPass = IsCADByPass(a_parent), isB_ByPass = IsCADByPass(b_parent); 
					bool isSirDst_DE_ByPass = IsCADByPass(sirDst_DE);
					bool priorConstraintToDE = PriorConstraintFromDE2DE(designElement, sirDst_DE);

					if (!isSirDst_DE_ByPass)
					{
						if (!iamSize2Fit)              // designElement is NOT Size2Fit
						{
							if (!isA_S2F && !isB_S2F)  
							{
								std::map<long, AssemblyInterface::CADComponent>::iterator k = this->m_CadComponentMap.find(designElement.uniqueId());
								if (k != this->m_CadComponentMap.end())
									cadComponent = k->second;						
								else
								{
									GMEConsole::Console::writeLine("CyPhy2CADVisitor::Visit_Component() Error: Component doesn't exist in m_CadComponentMap when it should!" , MSG_INFO);
									continue;
								}

								set<AssemblyInterface::Constraint> Constraint_Set = cadComponent.Constraint_kind_children();
								if (SelectedJoinStructures_Set[i].first.uniqueId() == this->m_CurrentRootComponent || Constraint_Set.size() == 0 || priorConstraintToDE)      // JS points back to root || i have 0 constraints || i already have a constraint to this guy from earlier, so add constraint to myself
								{
									//GMEConsole::Console::writeLine("Join Structure points back to root component", MSG_INFO);  

									if (!isA_ByPass && !isB_ByPass)
									{
										AddConstraint2CADComponent(cadComponent, a, b);  
										this->AddDE2DEConstraintPairs(designElement, sirDst_DE);
									}

									for (set<long>::iterator k = sirVisitor.m_JoinStructuresVisited.begin(); k != sirVisitor.m_JoinStructuresVisited.end(); k++)
										this->m_JoinStructuresVisited.insert(*k);
								}
								else                                                                                     // Add to connected Design Element
								{
				//					GMEConsole::Console::writeLine("DE ID:" + UdmGme::UdmId2GmeId(sirDst_DE.uniqueId()) + " Name:" + std::string(sirDst_DE.name()), MSG_INFO);

									if (sirDst_DE.type() != CyPhyML::Component::meta && sirDst_DE.type() != CyPhyML::ComponentAssembly::meta)
									{
										GMEConsole::Console::writeLine("Error: Design Element is neither Component or ComponentAssembly!", MSG_ERROR);
									}

									std::map<long, AssemblyInterface::CADComponent>::iterator i = this->m_CadComponentMap.find(sirDst_DE.uniqueId());
									if (i == this->m_CadComponentMap.end())          // New CADComponent, add constraint info from StructuralEnd
									{
										cadComponent = AssemblyInterface::CADComponent::Create(currenttop);
										this->m_CadComponentMap[sirDst_DE.uniqueId()] = cadComponent;
										//this->InitializeCadComponent(sirDst_DE, cadComponent);	
										this->FillInAttributes(sirDst_DE, cadComponent);		
									}
									else									        // Existing CADComponent constraint info from StructuralEnd	
										cadComponent = i->second;
	
				#if 0
									sirVisitor.PrintStructuralInterfaceRolePairs();
				#endif
									AddConstraint2CADComponent(cadComponent, a, b);  
									this->AddDE2DEConstraintPairs(sirDst_DE, designElement);
					
									for (set<long>::iterator k = sirVisitor.m_JoinStructuresVisited.begin(); k != sirVisitor.m_JoinStructuresVisited.end(); k++)
											this->m_JoinStructuresVisited.insert(*k);
					
									//GMEConsole::Console::writeLine(" * " + std::string(designElement.name()) + ":" + std::string(sirDst_DE.name()), MSG_INFO);
									this->TraverseDesignElement(sirDst_DE);		
								}
							}
							else   // connected to a Size2Fit, don't worry about it
							{
								std::map<long, AssemblyInterface::CADComponent>::iterator i = this->m_CadComponentMap.find(sirDst_DE.uniqueId());
								if (i == this->m_CadComponentMap.end())          // New CADComponent, add constraint info from StructuralEnd
								{
									cadComponent = AssemblyInterface::CADComponent::Create(currenttop);
									this->m_CadComponentMap[sirDst_DE.uniqueId()] = cadComponent;
									this->FillInAttributes(sirDst_DE, cadComponent);
								}

								this->TraverseDesignElement(sirDst_DE);
							}
						}
						else         // designElement is Size2Fit
						{
							/*
							DE that contains SIR, and DE that is immediated connected to me
							[1] Is DE on the other end a S2F DE?
							[2] If yes, flag error
							[3] If not: 
									add constaint to myself
									if immediate DE's parent == myParent
											traverse(DE)
							*/
						
							if (isA_S2F && isB_S2F)
							{
								GMEConsole::Console::writeLine("2 Size2Fit parts connected together! [" + UdmGme::UdmId2GmeId(a_parent.uniqueId()) + "," + UdmGme::UdmId2GmeId(b_parent.uniqueId()) + "]", MSG_ERROR);
							}
							else
							{
								//GMEConsole::Console::writeLine("TraverseDesignElement(): Size2Fit [" + (std::string)designElement.name() + "," + UdmGme::UdmId2GmeId(designElement.uniqueId()) + "]", MSG_INFO);
								std::map<long, AssemblyInterface::CADComponent>::iterator i = this->m_CadComponentMap.find(designElement.uniqueId());
								if (i != this->m_CadComponentMap.end())
								{
									cadComponent = i->second;						// Add constraint to myself
									AddConstraint2CADComponent(cadComponent, a, b);  
									this->AddDE2DEConstraintPairs(designElement, sirDst_DE);
									for (set<long>::iterator k = sirVisitor.m_JoinStructuresVisited.begin(); k != sirVisitor.m_JoinStructuresVisited.end(); k++)
										this->m_JoinStructuresVisited.insert(*k);

									if (Udm::Object::Cast(sirDst_DE.parent()) == Udm::Object::Cast(designElement.parent()))
									{
									//	GMEConsole::Console::writeLine("From Size2Fit calling traverse on [" + (std::string)sirDst_DE.name() + "," + UdmGme::UdmId2GmeId(sirDst_DE.uniqueId()) + "]", MSG_INFO);

										std::map<long, AssemblyInterface::CADComponent>::iterator i = this->m_CadComponentMap.find(sirDst_DE.uniqueId());
										if (i == this->m_CadComponentMap.end())          // New CADComponent, add constraint info from StructuralEnd
										{
											AssemblyInterface::CADComponent sirDst_cadComponent = AssemblyInterface::CADComponent::Create(currenttop);
											this->m_CadComponentMap[sirDst_DE.uniqueId()] = sirDst_cadComponent;
											this->FillInAttributes(sirDst_DE, sirDst_cadComponent);
										}
										//this->TraverseDesignElement(sirDst_DE);
									}
								}
								else
									GMEConsole::Console::writeLine("CADComponent does not exist for [" + UdmGme::UdmId2GmeId(designElement.uniqueId()) + "]", MSG_WARNING);
							}
						}
					}
				}
			}
		} // end for
	}

	this->m_CadComponentParent = currenttop;
//	GMEConsole::Console::writeLine("TraverseDesignElement() END - DE Name:" + std::string(designElement.name()), MSG_INFO);
}


///////////////////////////////////// HELPER FUNCTIONS /////////////////////////////////////
// Debug Print functions
#if 0
void CyPhy2CADVisitor::PrintStructuralIfaceMap()
{
	GMEConsole::Console::writeLine("CyPhy2CADVisitor::PrintStructuralIfaceMap()", MSG_INFO);
	for (std::map<long, long>::iterator it = m_StructuralIfaceMap.begin(); it != m_StructuralIfaceMap.end(); it++)
	{
		std::string fir, snd;
		to_string(fir, it->first);
		to_string(snd, it->second);
		GMEConsole::Console::writeLine(fir + ":" + snd, MSG_INFO);
	}
}
#endif

void CyPhy2CADVisitor::PrintFindConnectedJoinStructuresAndEnd(std::vector<JoinStructureAndEndPair> &JoinStructureAndEnd)
{
	// std::pair<CyPhyML::DesignElement, CyPhyML::JoinStructures> JoinStructureAndEndPair;
	GMEConsole::Console::writeLine("PrintFindConnectedJoinStructuresAndEnd()", MSG_INFO);
	for (std::vector<JoinStructureAndEndPair>::const_iterator ci = JoinStructureAndEnd.begin(); ci != JoinStructureAndEnd.end(); ci++)
	{
		GMEConsole::Console::writeLine("/t " + std::string(ci->first.name()) + " DE ID:" + UdmGme::UdmId2GmeId(ci->first.uniqueId()), MSG_INFO);
		GMEConsole::Console::writeLine("/t JoinStructure ID:" + UdmGme::UdmId2GmeId(ci->second.uniqueId()), MSG_INFO);
	}
}

#if 0
long CyPhy2CADVisitor::FindFeatureArchetypeID(CyPhyML::Feature &feature)
{
//	GMEConsole::Console::writeLine("Feature Instance Name " + std::string(feature.name()), MSG_INFO);
	
	CyPhyML::Feature archetype = feature.Archetype();
	while (archetype.isInstance() || archetype.isSubtype())
	{
		archetype = archetype.Archetype();
	}

//	GMEConsole::Console::writeLine("Found feature's Type " + std::string(archetype.name()) + " at " + archetype.getPath(), MSG_INFO);
	return archetype.uniqueId();
}
#endif


bool CyPhy2CADVisitor::IsNewJoinStructure(long id)
{
	return (m_JoinStructuresVisited.find(id) == m_JoinStructuresVisited.end());
}


// typedef std::pair<long, CyPhyML::StructuralInterfaceRole> JoinStructureEnds;
void CyPhy2CADVisitor::FindStructuralInterfaceRoleEnds(CyPhyML::StructuralInterfaceRole& origin, std::vector<JoinStructureEndsItem>& JoinEnds)
{
	std::set<CyPhyML::JoinStructures> src_joinStructs = origin.srcJoinStructures();
	for (std::set<CyPhyML::JoinStructures>::iterator i = src_joinStructs.begin(); i != src_joinStructs.end(); i++)
	{
		long id = i->uniqueId();
		if (this->IsNewJoinStructure(id))
		{
			CyPhyML::StructuralInterfaceRole sir = CyPhyML::StructuralInterfaceRole::Cast(i->srcJoinStructures_end());
			JoinStructureEndsItem item;
			item.first = id;
			item.second = sir;
			JoinEnds.push_back(item);
		}
	}

	std::set<CyPhyML::JoinStructures> dst_joinStructs = origin.dstJoinStructures();
	for (std::set<CyPhyML::JoinStructures>::iterator i = dst_joinStructs.begin(); i != dst_joinStructs.end(); i++)
	{
		long id = i->uniqueId();
		if (IsNewJoinStructure(id))
		{
			CyPhyML::StructuralInterfaceRole sir = CyPhyML::StructuralInterfaceRole::Cast(i->dstJoinStructures_end());
			JoinStructureEndsItem item;
			item.first = id;
			item.second = sir;
			JoinEnds.push_back(item);
		}
	}
}

#if 0
void CyPhy2CADVisitor::AddConstraint2CADComponent(AssemblyInterface::CADComponent& cadComponent, CyPhyML::StructuralInterfaceRole& a, CyPhyML::StructuralInterfaceRole& b)
{
//	GMEConsole::Console::writeLine("AddConstraint2CADComponent() cadComponentID:" + std::string(cadComponent.ComponentID()) + " aPortID:" + UdmGme::UdmId2GmeId(a.uniqueId()) + " bPortID:" + UdmGme::UdmId2GmeId(b.uniqueId()), MSG_INFO);
	CyPhyML::MgaObject aMga = a.parent();
	CyPhyML::MgaObject bMga = b.parent();
//	long aId = aMga.uniqueId();
//	long bId = bMga.uniqueId();
	std::string aId = UdmGme::UdmId2GmeId(aMga.uniqueId());
	std::string bId = UdmGme::UdmId2GmeId(bMga.uniqueId());

	AssemblyInterface::Constraint constraint = AssemblyInterface::Constraint::Create (cadComponent);
	set<CyPhyML::Feature> b_set = b.Feature_kind_children();
	set<CyPhyML::Feature> a_set = a.Feature_kind_children();
	int count = 0;
	for (set<CyPhyML::Feature>::iterator it = a_set.begin(); it != a_set.end(); it++)
	{
		CyPhyML::Feature a_feature(*it);
		long myTypeID = this->FindFeatureArchetypeID(a_feature);

		// iterate through Structural Interface Role Dest_End's feature ports and see if any of them is connected to source feature port
		for (set<CyPhyML::Feature>::iterator dit = b_set.begin(); dit != b_set.end(); dit++)
		{
			CyPhyML::Feature b_feature(*dit);   				
			bool axis = 0;
			bool surface = 0;
			bool point = 0;				

			if (a_feature.type() == CyPhyML::Axis::meta)
				axis = 1;
			else if (a_feature.type() == CyPhyML::Surface::meta)
				surface = 1;
			else if (a_feature.type() == CyPhyML::Point::meta)
				point = 1;


			if (this->m_StructuralIfaceMap[myTypeID] == this->FindFeatureArchetypeID(b_feature))
			{
				if (b_feature.type() != a_feature.type())   // Axis -2- Axis, Surface -2- Surface, Point -2- Point, no mix-match
				{
//					GMEConsole::Console::writeLine("Matching feature ports are not the same type!", MSG_ERROR);
					break;
				}

//				GMEConsole::Console::writeLine("======= Found matching features [" + ta + "," + tb + "]", MSG_INFO);
				// Create a pair
				AssemblyInterface::Pair pair = AssemblyInterface::Pair::Create(constraint);
				pair.FeatureInterfaceType() = "CAD_DATUM";
				if (axis)
					pair.FeatureGeometryType() = "AXIS";
				else if (surface)
					pair.FeatureGeometryType() = "SURFACE";
				else
					pair.FeatureGeometryType() = "POINT";
				pair.FeatureAlignmentType() = "ALIGN";

				std::string tmp;
				AssemblyInterface::ConstraintFeature a_constraint_feature = AssemblyInterface::ConstraintFeature::Create(pair);
				a_constraint_feature.FeatureName() = CleanString(a_feature.Datum());
				//to_string(tmp, aId);
				a_constraint_feature.ComponentID() = tmp;
				a_constraint_feature.ComponentID() = aId;
				a_constraint_feature.FeatureOrientationType() = (surface) ? std::string(CyPhyML::Surface::Cast(a_feature).Orientation()): "NONE";

				AssemblyInterface::ConstraintFeature b_constraint_feature = AssemblyInterface::ConstraintFeature::Create(pair);
				b_constraint_feature.FeatureName() = CleanString(b_feature.Datum());
				//to_string(tmp, bId);
				//b_constraint_feature.ComponentID() = tmp;
				b_constraint_feature.ComponentID() = bId;
				b_constraint_feature.FeatureOrientationType() = (surface) ? std::string(CyPhyML::Surface::Cast(b_feature).Orientation()): "NONE";

				break;
			}
			else
				count ++;	
		}
	}	

	//if (count > 0)
	//	GMEConsole::Console::writeLine("Not all features of StructuralInterfaceRole ports are connected [" + MakeHyperLink(UdmGme::UdmId2GmeId(a.uniqueId()), UdmGme::UdmId2GmeId(a.uniqueId())) + "," + MakeHyperLink(UdmGme::UdmId2GmeId(b.uniqueId()), UdmGme::UdmId2GmeId(b.uniqueId())) + "]", MSG_ERROR);
}
#endif

void CyPhy2CADVisitor::InitializeCadComponent(CyPhyML::MgaObject& mgaObj, AssemblyInterface::CADComponent& cadComp)
{
	bool isComponent = (mgaObj.type() == CyPhyML::Component::meta);
	bool isComponentAssembly = (mgaObj.type() == CyPhyML::ComponentAssembly::meta);

	if (isComponent || isComponentAssembly)
	{
		std::string id;
		to_string(id, mgaObj.uniqueId());
		cadComp.ComponentID() = id;
		std::string type;
		cadComp.Name() = this->GetCadName(CyPhyML::DesignElement::Cast(mgaObj), type);

		cadComp.Type() = type;

		if (isComponent)
		{
			CyPhyML::Component comp = CyPhyML::Component::Cast(mgaObj);
			std::set<CyPhyML::CADParameter> CADParam_Set = comp.CADParameter_kind_children();

			if (CADParam_Set.size() > 0)
			{
				AssemblyInterface::ParametricParameters parametric = AssemblyInterface::ParametricParameters::Create(cadComp);
				std::set<CyPhyML::CADParameter>::iterator i = CADParam_Set.begin();
				for (; i != CADParam_Set.end(); i++)
				{
					AssemblyInterface::CADParameter cad_param = AssemblyInterface::CADParameter::Create(parametric);
					cad_param.Name() = i->ParameterName();
					cad_param.Type() = i->CADParameterType();
					cad_param.Value() = i->Value();		// or from valueFlow

					// cad_param.Value = (i->Value() == "") : i->DefaultValue() ? i->Value();

					//cad_param.Units();			
				}
			}
		}
	}
	else
		GMEConsole::Console::writeLine("CyPhy2CADVisitor::InitializeCadComponent() ERROR: CyPhyML MgaObject is not of type Component or ComponentAssembly!" , MSG_ERROR);
}

bool CyPhy2CADVisitor::IsNewCADComponent(long id)
{
	std::map<long, AssemblyInterface::CADComponent>::iterator i = m_CadComponentMap.find(id);
	return (i == m_CadComponentMap.end());
}

//void CyPhy2CADVisitor::CreateRootComponentConstraints(AssemblyInterface::CADComponent& component, long assemblyParentID, bool rootIsComponentAssembly)
void CyPhy2CADVisitor::CreateRootComponentConstraints(AssemblyInterface::CADComponent& component, std::string assemblyParentID)
{
	std::string name = component.Name();
//	long id; 
//	convert_to(id, std::string(component.ComponentID()));

	std::string id = component.ComponentID();
	bool rootIsComponentAssembly = component.Type() == "ASSEMBLY";

	AssemblyInterface::Constraint constraint = AssemblyInterface::Constraint::Create(component);

	AssemblyInterface::Pair front_pair = AssemblyInterface::Pair::Create(constraint);
	front_pair.FeatureGeometryType() = "SURFACE";
	front_pair.FeatureAlignmentType() = "ALIGN";
	front_pair.FeatureInterfaceType() = "CAD_DATUM";
	
	if (rootIsComponentAssembly)
		this->CreateConstraintFeature(front_pair, id,  "ASM_FRONT", "SIDE_A");
	else
		this->CreateConstraintFeature(front_pair, id,  "FRONT", "SIDE_A");
	this->CreateConstraintFeature(front_pair, assemblyParentID, "ASM_FRONT","SIDE_A");
	
	AssemblyInterface::Pair top_pair = AssemblyInterface::Pair::Create(constraint);
	top_pair.FeatureGeometryType() = "SURFACE";
	top_pair.FeatureAlignmentType() = "ALIGN";
	top_pair.FeatureInterfaceType() = "CAD_DATUM";
	
	if (rootIsComponentAssembly)
		this->CreateConstraintFeature(top_pair, id,  "ASM_TOP", "SIDE_A");
	else
		this->CreateConstraintFeature(top_pair, id,  "TOP", "SIDE_A");
	this->CreateConstraintFeature(top_pair, assemblyParentID, "ASM_TOP","SIDE_A");

	AssemblyInterface::Pair right_pair = AssemblyInterface::Pair::Create(constraint);
	right_pair.FeatureGeometryType() = "SURFACE";
	right_pair.FeatureAlignmentType() = "ALIGN";
	right_pair.FeatureInterfaceType() = "CAD_DATUM";
	
	if (rootIsComponentAssembly)
		this->CreateConstraintFeature(right_pair, id,  "ASM_RIGHT", "SIDE_A");
	else
		this->CreateConstraintFeature(right_pair, id,  "RIGHT", "SIDE_A");
	this->CreateConstraintFeature(right_pair, assemblyParentID, "ASM_RIGHT","SIDE_A");
}

//void CyPhy2CADVisitor::CreateConstraintFeature(AssemblyInterface::Pair& pair, long id, std::string name, std::string orientation)
void CyPhy2CADVisitor::CreateConstraintFeature(AssemblyInterface::Pair& pair, std::string id, std::string name, std::string orientation)
{
//	std::string idStr;
//	to_string(idStr, id);
	AssemblyInterface::ConstraintFeature constraintFeature = AssemblyInterface::ConstraintFeature::Create(pair);
//	constraintFeature.ComponentID() = idStr;
	constraintFeature.ComponentID() = id;
	constraintFeature.FeatureName() = name;
	constraintFeature.FeatureOrientationType() = orientation;
}

void CyPhy2CADVisitor::FindConnectedJoinStructuresAndEnd(CyPhyML::StructuralPortType &structPort, std::vector<JoinStructureAndEndPair> &selectedJoinStructures)
{
	CyPhyML::MgaObject structPort_Parent = structPort.parent();
	Udm::Object structPort_ParentParent = structPort_Parent.parent();

//	GMEConsole::Console::writeLine("FindConnectedJoinStructuresAndEnd() starting port:" + UdmGme::UdmId2GmeId(structPort.uniqueId()), MSG_INFO);

	set<CyPhyML::JoinStructures> srcJS_Set = structPort.srcJoinStructures();
	for (set<CyPhyML::JoinStructures>::iterator i = srcJS_Set.begin(); i != srcJS_Set.end(); i++)
	{
		CyPhyML::JoinStructures join(*i);
		CyPhyML::StructuralPortType sirPort = join.srcJoinStructures_end();

//		GMEConsole::Console::writeLine("== srcJoin:" + UdmGme::UdmId2GmeId(join.uniqueId()) + " connectedPort:" + UdmGme::UdmId2GmeId(sirPort.uniqueId()), MSG_INFO);

		CyPhyML::MgaObject portParent = sirPort.parent();
		Udm::Object portParentParent = portParent.parent();

		if (structPort_ParentParent.uniqueId() == portParentParent.uniqueId() && this->IsNewJoinStructure(join.uniqueId()))       
		{
			JoinStructureAndEndPair jsEndPair;
			if (portParent.type() == CyPhyML::Component::meta || portParent.type() == CyPhyML::ComponentAssembly::meta)
			{
				jsEndPair.first = CyPhyML::DesignElement::Cast(portParent);
				jsEndPair.second = join;
				selectedJoinStructures.push_back(jsEndPair);
			}
		}
	}

	set<CyPhyML::JoinStructures> dstJS_Set = structPort.dstJoinStructures();
	for (set<CyPhyML::JoinStructures>::iterator i = dstJS_Set.begin(); i != dstJS_Set.end(); i++)
	{
		CyPhyML::JoinStructures join(*i);
		CyPhyML::StructuralPortType sirPort = join.dstJoinStructures_end();
		
//		GMEConsole::Console::writeLine("== dstJoin:" + UdmGme::UdmId2GmeId(join.uniqueId()) + " connectedPort:" + UdmGme::UdmId2GmeId(sirPort.uniqueId()), MSG_INFO);

		CyPhyML::MgaObject portParent = sirPort.parent();
		Udm::Object portParentParent = portParent.parent();

	//	GMEConsole::Console::writeLine("==" + UdmGme::UdmId2GmeId(structPort_ParentParent.uniqueId()) + "," + UdmGme::UdmId2GmeId(portParentParent.uniqueId()) , MSG_INFO);
		if (structPort_ParentParent.uniqueId() == portParentParent.uniqueId() && this->IsNewJoinStructure(join.uniqueId()))       
		{
			JoinStructureAndEndPair jsEndPair;
			if (portParent.type() == CyPhyML::Component::meta || portParent.type() == CyPhyML::ComponentAssembly::meta)
			{
				jsEndPair.first = CyPhyML::DesignElement::Cast(portParent);
				jsEndPair.second = join;
				selectedJoinStructures.push_back(jsEndPair);
			}
		}
	}
}

std::string CyPhy2CADVisitor::GetCadName(const CyPhyML::DesignElement &DesignElement, std::string &type)
{
	if (!m_IsThermal)
	{
		std::set<CyPhyML::CADFileLink> CADFileLink_Set = DesignElement.CADFileLink_kind_children();
		if (CADFileLink_Set.size() > 0)
		{
			CyPhyML::CADFileLink link(*(CADFileLink_Set.begin()));
			type = link.FileType();
			toupper(type);
			return CleanString((link.FileName()));
		}

		if (DesignElement.type() == CyPhyML::ComponentAssembly::meta)
			type = ASSEMBLY_DEF;
		else 
			type = PART_DEF;
		return CleanString(DesignElement.name());
	}
	else
	{
		std::set<CyPhyML::ThermalCADModelLink> ThermalCADFileLink_Set = DesignElement.ThermalCADModelLink_kind_children();
		if (ThermalCADFileLink_Set.size() > 0)
		{
			CyPhyML::ThermalCADModelLink link(*(ThermalCADFileLink_Set.begin()));
			type = link.FileType();
			toupper(type);
			return CleanString((link.FileName()));
		}
		
		if (DesignElement.type() == CyPhyML::ComponentAssembly::meta)
			type = ASSEMBLY_DEF;
		else
			type = PART_DEF;
		return (CleanString(DesignElement.name()) + "_Thermal");
	}
}

void CyPhy2CADVisitor::PrintJoinStructuresVisited()
{
//	GMEConsole::Console::writeLine("PrintJoinStructuresVisited()", MSG_INFO);
	for (std::set<long>::const_iterator ci = m_JoinStructuresVisited.begin(); ci != m_JoinStructuresVisited.begin(); ci++)
	{
		GMEConsole::Console::writeLine("Js:" + UdmGme::UdmId2GmeId(*ci), MSG_INFO);
	}
}

bool CyPhy2CADVisitor::IsCADAssembly(CyPhyML::DesignElement &element)
{
	if (element.type() == CyPhyML::ComponentAssembly::meta)
		return 1;
	else if (element.type() == CyPhyML::Component::meta)
	{
		set<CyPhyML::CADFileLink> CADFile_Set = element.CADFileLink_kind_children();
		if (CADFile_Set.size() == 0)
			return 0;
		else
		{
			CyPhyML::CADFileLink link(*CADFile_Set.begin());
			if (link.FileType() == "Assembly")
				return 1;
			else
				return 0;
		}
	}
}

#if 0
//////////////////////////////////////////////////////////////// NEW Stuff ////////////////////////////////////////////////////////////////
void CyPhy2CADVisitor::CreateAssemblyCADComponent(const CyPhyML::ComponentAssembly& compAssembly, Udm::Object &CadComponentParent)
{
	AssemblyInterface::CADComponent cadCompAssembly = AssemblyInterface::CADComponent::Create(CadComponentParent);
	this->FillInAttributes(compAssembly, cadCompAssembly);
	m_CadComponentMap[compAssembly.uniqueId()] = cadCompAssembly;

	set<CyPhyML::Component> Component_Children = compAssembly.Component_kind_children();
	for (set<CyPhyML::Component>::iterator i = Component_Children.begin(); i != Component_Children.end(); i++)
	{
		CyPhyML::Component comp(*i);
		AssemblyInterface::CADComponent cadComp = AssemblyInterface::CADComponent::Create(cadCompAssembly);
		FillInAttributes(comp, cadComp);
		m_CadComponentMap[comp.uniqueId()] = cadComp;		
	}

	set<CyPhyML::ComponentAssembly> ComponentAssembly_Children = compAssembly.ComponentAssembly_kind_children();
	for (set<CyPhyML::ComponentAssembly>::iterator i = ComponentAssembly_Children.begin(); i != ComponentAssembly_Children.end(); i++)
	{
		CyPhyML::ComponentAssembly assembly(*i);
		this->CreateAssemblyCADComponent(assembly, cadCompAssembly);
	}
}



void CyPhy2CADVisitor::TraverseComponentAssembly4Constraints(const CyPhyML::ComponentAssembly& assembly, long startingElement)
{
	set<CyPhyML::DesignElement> DesignElementChildren_Set = assembly.DesignElement_kind_children();
	if (DesignElementChildren_Set.size() == 0)
		GMEConsole::Console::writeLine("CyPhy2CADVisitor::TraverseAssembly() Error: Assembly [" + std::string(assembly.name()) + "] contains no design elements", MSG_WARNING);
	else
	{
		CyPhyML::DesignElement RootDE;
		
		if (startingElement < 0)
		{
			RootDE = this->FindRootDesignElement(assembly);
		}
		else
		{
			bool found = 0;
			set<CyPhyML::DesignElement>::const_iterator cit = DesignElementChildren_Set.begin();
			for (; cit != DesignElementChildren_Set.end(); cit++)
			{
				if (cit->uniqueId() == startingElement && m_Size2FitComponents.find(cit->uniqueId()) == m_Size2FitComponents.end())       // not size2fit
				{
					RootDE = *cit;
					found = 1;
					break;
				}
			}
			if (!found)
				RootDE = this->FindRootDesignElement(assembly);
		}
		
		//bool rootDEIsComponentAssembly = (RootDE.type() == CyPhyML::ComponentAssembly::meta);

		if (RootDE != Udm::null)
		{
			this->m_CurrentRootComponent = RootDE.uniqueId();
		//	GMEConsole::Console::writeLine("Root Component --> " + std::string(RootDE.name()), MSG_INFO);
			AssemblyInterface::CADComponent cadRoot = this->m_CadComponentMap.find(RootDE.uniqueId());
			if (cadRoot != Udm::null)
			{
				this->CreateRootComponentConstraints(cadRoot, assembly.uniqueId(), cadRoot.Type() == "ASSEMBLY");
				TraverseDesignElement(RootDE);	
			}
			else
				GMEConsole::Console::writeLine("Something's not right, CADComponent has not been created for RootDE!", MSG_ERROR);
		}
	}

	std::set<CyPhyML::ComponentAssembly> CompAssembly_Children = assembly.ComponentAssembly_kind_children();
	for (std::set<CyPhyML::ComponentAssembly>::iterator i = CompAssembly_Children.begin(); i != CompAssembly_Children.end(); i++)
	{
		CyPhyML::ComponentAssembly childAssembly(*i);
		this->TraverseComponentAssembly4Constraints(childAssembly);
	}

//	GMEConsole::Console::writeLine("TraverseAssembly() END - AN:" + std::string(assembly.name()), MSG_INFO);
}
#endif 

CyPhyML::DesignElement CyPhy2CADVisitor::FindRootDesignElement(const CyPhyML::ComponentAssembly& assembly)
{
	std::set<CyPhyML::JoinStructures> JS_Set = assembly.JoinStructures_kind_children();
	for (std::set<CyPhyML::JoinStructures>::iterator jIt = JS_Set.begin(); jIt != JS_Set.end(); jIt++)
	{
		CyPhyML::JoinStructures js(*jIt);
		CyPhyML::StructuralPortType src_SIR = js.srcJoinStructures_end();
		CyPhyML::StructuralPortType dst_SIR = js.dstJoinStructures_end();
				
	//	CyPhyML::DesignElement src_parent = CyPhyML::DesignElement::Cast(src_SIR.DesignElementImplementationBase_parent());
		CyPhyML::DesignElement src_parent = src_SIR.DesignElement_parent();
		Udm::Object src_grandParent = src_parent.parent();

	//	CyPhyML::DesignElement dst_parent = CyPhyML::DesignElement::Cast(dst_SIR.DesignElementImplementationBase_parent());
		CyPhyML::DesignElement dst_parent = dst_SIR.DesignElement_parent();
		Udm::Object dst_grandParent = dst_parent.parent();

		if (src_grandParent.uniqueId() == assembly.uniqueId())
		{
			if (!IsSize2Fit(src_parent) && !IsCADByPass(src_parent))      // not size2fit
				return src_parent;
			
		}
		else if (dst_grandParent.uniqueId() == assembly.uniqueId())
		{
			if (!IsSize2Fit(dst_parent) && !IsCADByPass(dst_parent))      // not size2fit
				return dst_parent;		
		}
	}

	if (JS_Set.size() == 0)
	{
		GMEConsole::Console::writeLine("Assembly [" + std::string(assembly.name()) + "] does not have any Design Elements that are connected!", MSG_WARNING);
		set<CyPhyML::DesignElement> DesignElementChildren_Set = assembly.DesignElement_kind_children();
		return *DesignElementChildren_Set.begin();
	}
}


void CyPhy2CADVisitor::FillInAttributes(const CyPhyML::DesignElement& de, AssemblyInterface::CADComponent& cadComp)
{
	std::string type, tmp;
	cadComp.Name() = this->GetCadName(de, type);

	to_string(tmp, de.uniqueId());
	cadComp.ComponentID() = UdmGme::UdmId2GmeId(de.uniqueId());
	//cadComp.ComponentID() = tmp;
	cadComp.Type() = type; 

	set<CyPhyML::SizedToFit> Size2Fit_Set = de.SizedToFit_kind_children();
	if (Size2Fit_Set.size() > 0 && de.type() == CyPhyML::Component::meta)
	{
		cadComp.SpecialInstruction() = "SIZE_TO_FIT";
	//	this->m_Size2FitComponents[de.uniqueId()] = CyPhyML::Component::Cast(de);
	}


	if (de.type() == CyPhyML::Component::meta)
	{
		CyPhyML::Component comp = CyPhyML::Component::Cast(de);
		std::set<CyPhyML::CADParameter> CADParam_Set = comp.CADParameter_kind_children();

		if (CADParam_Set.size() > 0)
		{
			AssemblyInterface::ParametricParameters parametric = AssemblyInterface::ParametricParameters::Create(cadComp);
			std::set<CyPhyML::CADParameter>::iterator i = CADParam_Set.begin();
			for (; i != CADParam_Set.end(); i++)
			{
				AssemblyInterface::CADParameter cad_param = AssemblyInterface::CADParameter::Create(parametric);
				cad_param.Name() = i->ParameterName();
				cad_param.Type() = i->CADParameterType();
				cad_param.Value() = (i->Value() == "") ? "0" : (std::string)i->Value();		// or from valueFlow
				// cad_param.Value = (i->Value() == "") : i->DefaultValue() ? i->Value();
			}
		}
	}
}

#if 0
void CyPhy2CADVisitor::Check4ZeroConstraints(const AssemblyInterface::CADComponent& start)
{
	//GMEConsole::Console::writeLine("Check4ZeroConstraits() for CADComponent [" + (std::string)start.ComponentID() + "," + (std::string)start.Name() + "]", MSG_INFO);
	std::string id = start.ComponentID(), name = start.Name();

	set<AssemblyInterface::Constraint> constraint_Set = start.Constraint_kind_children();
	if (constraint_Set.size() == 0 && AssemblyInterface::Assembly::Cast(start.Assembly_parent()) == Udm::null)
		GMEConsole::Console::writeLine("CADComponent [" + (std::string)start.ComponentID() + " " + (std::string)start.Name() + " ] has 0 constraints!", MSG_ERROR);

	set<AssemblyInterface::CADComponent> comp_Set = start.CADComponent_kind_children();
	for (set<AssemblyInterface::CADComponent>::iterator i = comp_Set.begin(); i != comp_Set.end(); i++)
	{
		AssemblyInterface::CADComponent comp(*i);
		this->Check4ZeroConstraints(comp);
	}
}

void CyPhy2CADVisitor::Check4ZeroConstraints(const AssemblyInterface::Assembly& start)
{
	set<AssemblyInterface::CADComponent> comp_Set = start.CADComponent_kind_children();
	for (set<AssemblyInterface::CADComponent>::iterator i = comp_Set.begin(); i != comp_Set.end(); i++)
	{
		AssemblyInterface::CADComponent comp(*i);
		Check4ZeroConstraints(comp);
	}
}
#endif

bool CyPhy2CADVisitor::IsSize2Fit(const CyPhyML::DesignElement& designElement)
{
	if (designElement.type() == CyPhyML::Component::meta)
	{
		set<CyPhyML::SizedToFit> s2f_Set = designElement.SizedToFit_kind_children();
		return (s2f_Set.size() > 0);
	}
	else
		return 0;
}

bool CyPhy2CADVisitor::IsCADByPass(const CyPhyML::DesignElement& designElement)
{
	if (designElement.type() == CyPhyML::Component::meta)
	{
		set<CyPhyML::CADBypass> s2f_Set = designElement.CADBypass_kind_children();
			return (s2f_Set.size() > 0);
	}
	else
		return 0;
}

bool CyPhy2CADVisitor::PriorConstraintFromDE2DE(CyPhyML::DesignElement &owningDE, CyPhyML::DesignElement &owned)
{
	std::string key, tmp1, tmp2;
	to_string(tmp1, owningDE.uniqueId());
	to_string(tmp2, owned.uniqueId());
	key = tmp1+":"+tmp2;
	return m_DE2DEConstraintPairs.find(key) != m_DE2DEConstraintPairs.end();
}

void CyPhy2CADVisitor::AddDE2DEConstraintPairs(CyPhyML::DesignElement &owningDE, CyPhyML::DesignElement &owned)
{
	std::string key, tmp1, tmp2;
	to_string(tmp1, owningDE.uniqueId());
	to_string(tmp2, owned.uniqueId());
	key = tmp1+":"+tmp2;
	m_DE2DEConstraintPairs.insert(key);
}
