/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "Json_Helper.h"
#include <boost\filesystem.hpp>
#include <boost\property_tree\ptree.hpp>
#include <boost\property_tree\json_parser.hpp>
#include <boost\foreach.hpp>
#include <boost\algorithm\string.hpp>
#include "..\CADAssembler\CADCommonFunctions\RelativePath.h"



namespace Json_Helper
{
	void ParseJsonFile(const string& projectDir, 
					   const string& outputDir,
					   vector<ComponentManifest>& manifestData,
					   string& sharedLib,
					   vector<string>& errMsg)
	{
		using boost::property_tree::ptree;
		boost::filesystem::path p(projectDir + "\\manifest.project.json");
		try
		{
			if (exists(p))
			{
				if (boost::filesystem::is_regular_file(p))
				{
					string relativePrefix = "";
					bool found = false;
					found = isis_CADCommon::FindRelativePath_From_B_to_A(projectDir, outputDir, relativePrefix);
					if (!found)
						relativePrefix = projectDir;
					relativePrefix = boost::trim_right_copy_if(relativePrefix, boost::is_any_of("/\\"));
					{
						boost::filesystem::path sp(projectDir + "\\shared_libraries");
						if (boost::filesystem::exists(sp))
							sharedLib = relativePrefix + "\\shared_libraries";
					}

					ptree pt;
					boost::property_tree::read_json(p.make_preferred().string(), pt);
					BOOST_FOREACH(const ptree::value_type& child, pt.get_child("Project.Components")) 
					{
						string avmid = child.second.get<std::string>("avmid");
						string revision = child.second.get<std::string>("revision");
						string version = child.second.get<std::string>("version");
						string modelpath = child.second.get<std::string>("modelpath");
						modelpath = boost::trim_left_copy_if(modelpath, boost::is_any_of("/\\"));
						modelpath = boost::trim_right_copy_if(modelpath, boost::is_any_of("/\\"));

						boost::filesystem::path mp(modelpath);
						modelpath = mp.make_preferred().parent_path().string();

						if (modelpath != "")
						{			
							UpdateComponentManifest(avmid, revision, version, modelpath, relativePrefix, manifestData);
						}
						else
							errMsg.push_back("manifest.project.json parsing error: avmid [" + avmid + "] has an empty modelpath");
					}
				}
				else
					errMsg.push_back("manifest.project.json not found in " + projectDir);
			}
		}
		catch(const boost::property_tree::ptree_error &exc)
		{
			errMsg.push_back("Boost Property Tree Exception during processing - " + string(exc.what()));
		}
	}

	void UpdateComponentManifest(const string &avm, 
								 const string &rev, 
								 const string &ver, 
								 const string &componentfolder,
								 const string &relativeprefix,
								 vector<ComponentManifest>& manifestData)
	{
		for (int i = 0; i < manifestData.size(); i++)
		{
			if (manifestData[i].AvmID == avm && manifestData[i].Revision == rev && manifestData[i].Version == ver)
			{
				manifestData[i].RelativePrefix = relativeprefix;
				manifestData[i].ComponentCadFolder = componentfolder;

				break;
			}
		}
	}	

	void SerializeManifestData(const string path,
							   const vector<ComponentManifest>& manifestData)
	{
		set<string> serializedSet;

		boost::filesystem::path p(path + "\\cad.manifest.json");
		using boost::property_tree::ptree;
		ptree pt;
		ptree arrayChild;
		ptree arrayElement;

		for (unsigned int i = 0; i < manifestData.size(); i++)
		{
			string key = manifestData[i].AvmID + "|" + manifestData[i].Revision;
			if (serializedSet.find(key) == serializedSet.end())
			{
				arrayElement.put("avmid", manifestData[i].AvmID);
				arrayElement.put("revision", manifestData[i].Revision);
				arrayElement.put("version", manifestData[i].Version);
				arrayChild.push_back(std::make_pair("", arrayElement));
				serializedSet.insert(key);
			}
		}

		pt.put_child("CADAssemblyManifestList", arrayChild);
		boost::property_tree::write_json(p.make_preferred().string(), pt);
	}
}