/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "ScriptWriter.h"
#include <fstream>
#include "..\CADAssembler\CADCommonFunctions\RelativePath.h"
#include <boost\filesystem.hpp>
#include "Logger.h"

namespace ScriptWriter
{
	void GeneratePostProcessScript(const string& outputDirectoryName,
								   const string& projectDir,
								   const vector<string>& parameters)
	{
		using namespace std;
		{
			ofstream outputFile;
			outputFile.open(outputDirectoryName + "//" + "TestBench_PostProcess.cmd");

			outputFile << ":: CAD TestBench post-processing script" << endl;
			outputFile << "echo off" << endl;
			outputFile << "FOR /F \"skip=2 tokens=2,*\" %%A IN ('%SystemRoot%\\SysWoW64\\REG.exe query \"HKLM\\software\\META\" /v \"META_PATH\"') DO \"%%B\\bin\\Python27\\Scripts\\Python.exe\"" << " main_post_process.py" << endl;
			//outputFile << "python main_post_process.py" << endl;
			outputFile.close();
		}

		{
			ofstream outputFile;
			outputFile.open(outputDirectoryName + "//" + "main_post_process.py");

			outputFile << "# resource file:" << endl; // C:\\MetaRepos\\c2m2l\\Fang\\python\\Range.py" << endl;
			outputFile << "import os" << endl;
			outputFile << "import sys" << endl << endl;

			outputFile << "def import_path(fullpath):"  << endl;
			outputFile << "    \"\"\""  << endl;
			outputFile << "    Import a file with full path specification. Allows one to"  << endl;
			outputFile << "    import from anywhere, something __import__ does not do."  << endl;
			outputFile << "    \"\"\""  << endl;
			outputFile << "    try:"  << endl;
			outputFile << "        path, filename = os.path.split(fullpath)"  << endl;
			outputFile << "        filename, ext = os.path.splitext(filename)"  << endl;
			outputFile << "        sys.path.append(path)"  << endl;
			outputFile << "        module = __import__(filename)"  << endl;
			outputFile << "        reload(module) # Might be out of date"  << endl;
			outputFile << "        del sys.path[-1]"  << endl;
			outputFile << "        module.main()"  << endl;
			outputFile << "    except ImportError:"  << endl;
			outputFile << "        exc_type, exc_value, tb_root = sys.exc_info()"  << endl;
			outputFile << "        print exc_value"  << endl << endl;       
 
			outputFile << "if __name__ == '__main__':"  << endl;
			outputFile << "    if not os.path.exists('summary.testresults.json'):" << endl;
			outputFile << "        print 'File does not exist: summary.testresults.json'" << endl;
			outputFile << "        os.exit(0)" << endl << endl;        
			outputFile << "    if (len(sys.argv) > 1):" << endl;
			outputFile << "        if not os.path.exists(sys.argv[1]):" << endl;
			outputFile << "            print 'Given result file does not exist: {0}'.format(sys.argv[1])" << endl;
			outputFile << "            os.exit(0)" << endl << endl;

			for (unsigned int i = 0; i < parameters.size(); i++)
			{
				string scriptPath = projectDir + "//" + parameters[i];
				boost::filesystem::path p(scriptPath);

				// 1-24-2013: Copying post process script to output directory so need to extract filename instead of using relative path
				//outputFile << "    import_path(r'" << parameters[i] << "')" << endl;
				outputFile << "    import_path(r'" << p.make_preferred().filename().string() << "')" << endl;

				string copycmd = "\"xcopy \"" + p.make_preferred().string() + "\" \"" + outputDirectoryName + "\" /Y /I\"";
				int failed = system(copycmd.c_str());
				if (failed != 0)
					LOGMSG("Failed to copy post process script - " + p.make_preferred().string(), MSG_WARNING);
			}

			outputFile.close();
			
			if (!parameters.empty())
			{
				boost::filesystem::path src(projectDir + "//Post_Processing/common");
				boost::filesystem::path dst(outputDirectoryName + "//common");

				if (boost::filesystem::exists(src))
				{
					string copycmd = "\"xcopy \"" + src.make_preferred().string() + "\" \"" + dst.make_preferred().string() + "\" /Y /I\"";
					int failed = system(copycmd.c_str());
					if (failed != 0)
						LOGMSG("Post process common directory does not exist - " + src.make_preferred().string(), MSG_WARNING);
				}
			}
		}
	}


	void GenerateCopyScript(const string& outputDirectoryName,
							const string& cadDirectoryName,
							const string& sharedLib,
							vector<Json_Helper::ComponentManifest>& manifestData,
							bool genManifest)
	{

#if 0
		if (genManifest)
		{
			for (unsigned int i = 0; i < manifestData.size(); i++)
			{
				if (manifestData[i].RelativeModelPath != "")
				{
					outputFile << "FOR /f \"delims=\" %%a IN ('DIR \"" << manifestData[i].RelativeModelPath << "\\*.prt.*\" /s /b') DO xcopy \"%%a\" Cad_Components_Directory\\ /Y /I" << "\n";
					outputFile << "FOR /f \"delims=\" %%a IN ('DIR \"" << manifestData[i].RelativeModelPath << "\\*.asm.*\" /s /b') DO xcopy \"%%a\" Cad_Components_Directory\\ /Y /I" << "\n";
				}
			}

			if (sharedLib != "")
			{
				outputFile << "FOR /f \"delims=\" %%a IN ('DIR \"" << sharedLib << "\\*.prt.*\" /s /b') DO xcopy \"%%a\" Cad_Components_Directory\\ /Y /I" << "\n";
				outputFile << "FOR /f \"delims=\" %%a IN ('DIR \"" << sharedLib << "\\*.asm.*\" /s /b') DO xcopy \"%%a\" Cad_Components_Directory\\ /Y /I" << "\n";
			}
		}
#endif

		if (cadDirectoryName != "")
		{
			ofstream outputFile;
			outputFile.open(outputDirectoryName + "//Copy_Parts.bat");

			outputFile << "REM Pass %1 in if not running from the directory containing this bat file\n";
			outputFile << "REM Example invocation from the directory containing this bat file:" << "\n";
			outputFile << "REM 	Copy_Parts.bat\n";
			outputFile << "REM Example invocation from a directory other than the directory containing this bat file:" << "\n";
			outputFile << "REM       \"C:\\Meta\\MassSpringDamper\\results\\m2e4b1rk\\Copy_Parts.bat\" \"C:\\Meta\\MassSpringDamper\\results\\m2e4b1rk\"" << "\n\n";
			outputFile << "cd %1\n";
			outputFile << "echo ON\n";
			outputFile << "FOR /f \"delims=\" %%a IN ('DIR \"" << cadDirectoryName << "\\*.prt.*\" /s /b') DO xcopy \"%%a\" Cad_Auxiliary_Directory\\ /Y /I" << "\n";
			outputFile << "FOR /f \"delims=\" %%a IN ('DIR \"" << cadDirectoryName << "\\*.asm.*\" /s /b') DO xcopy \"%%a\" Cad_Auxiliary_Directory\\ /Y /I" << "\n";
			outputFile.close();
		}
	}

	void GenerateCreoSearchPath(const string& outputDirectoryName,
								const string& cadDirectoryName,
								const string& sharedLib,
							    vector<Json_Helper::ComponentManifest>& manifestData,
								bool genManifest)
	{
		ofstream outputFile;
		outputFile.open(outputDirectoryName + "//search_META.pro");

		// cad directory first
		if (cadDirectoryName != "")
		{
			outputFile << "\"" << cadDirectoryName<<"\"\n";
		}

		if (genManifest)
		{
			for (unsigned int i = 0; i < manifestData.size(); i++)
			{
				if (manifestData[i].ComponentCadFolder != "")
				{
					string path = manifestData[i].RelativePrefix + "\\" + manifestData[i].ComponentCadFolder + "\\" + manifestData[i].ModelURI;
					outputFile << "\"" << path << "\"\n";
				}
			}
			
			if (sharedLib != "")
				outputFile << "\"" << sharedLib << "\"\n";
		}

		outputFile.close();
	}


	void GenerateZipScript(const string& outputDirectoryName,
						   const vector<Json_Helper::ComponentManifest>& manifestData)
	{
		ofstream outputFile;
		outputFile.open(outputDirectoryName + "//zip.py");

		outputFile<< "# ---------------------------------------------------" << endl;
		outputFile<< "# Auto generated by CyPhy2CAD" << endl;
		outputFile<< "# ---------------------------------------------------" << endl;
		outputFile<< "import zipfile" << endl;
		outputFile<< "import os" << endl;
		outputFile<< "import shutil" << endl;
		outputFile<< "import subprocess" << endl << endl;

		outputFile<< "components_folder = [ \\" << endl;
		for (unsigned int i = 0; i < manifestData.size(); i++)
		{
			if (manifestData[i].ComponentCadFolder != "")
				outputFile << "    r'" << manifestData[i].ComponentCadFolder<< "\\" << manifestData[i].ModelURI << "', \\" << endl;
		}
		outputFile<< "    ]" << endl << endl;		

		outputFile<< "# call Copy_Parts.bat" << endl;
		outputFile<< "copy_bat = 'Copy_Parts.bat'" << endl;
		outputFile<< "if os.path.exists(copy_bat):" << endl;
		outputFile<< "    try:" << endl;
		outputFile<< "        result, error = subprocess.Popen(copy_bat, stdout = subprocess.PIPE, stderr= subprocess.PIPE).communicate()" << endl;
		outputFile<< "        with open('zip_log.txt', 'w') as zip_log:" << endl;
		outputFile<< "            zip_log.write(result)" << endl;
		outputFile<< "            zip_log.write(error)" << endl;
		outputFile<< "    except Exception as msg:" << endl;
		outputFile<< "        with open('_FAILED.txt', 'w') as f_out:" << endl;
		outputFile<< "            f_out.write(str(msg))" << endl;
		outputFile<< "            f_out.write('\\nNot able to copy cad files.')" << endl;
		outputFile<< "        if os.name == 'nt':" << endl;
		outputFile<< "            os._exit(3)" << endl;
		outputFile<< "        elif os.name == 'posix':" << endl;
		outputFile<< "            os._exit(os.EX_OSFILE)" << endl << endl;

		outputFile<< "search_path = 'search_META.pro'" << endl;     
		outputFile<< "if os.path.exists(search_path):" << endl;
		outputFile<< "    shutil.copyfile(search_path, search_path + '.local')" << endl << endl;

		outputFile<< "with open (search_path, 'w') as search_path_file:" << endl;
		outputFile<< "    if os.path.exists('Cad_Auxiliary_Directory'):" << endl;
		outputFile<< "        search_path_file.write('\".\\Cad_Auxiliary_Directory\"\\n')" << endl << endl;
		outputFile<< "    for folder in components_folder:" << endl;
		outputFile<< "        search_path_file.write('\".\\\\' + folder + '\"\\n')" << endl << endl;


		outputFile<< "# zip" << endl;
		outputFile<< "output_filename = 'source_data.zip'" << endl << endl;

		outputFile<< "if os.path.exists(output_filename):" << endl;
		outputFile<< "    os.remove(output_filename)" << endl << endl;

		outputFile<< "with zipfile.ZipFile(output_filename, 'w') as z:" << endl;
		outputFile<< "    parent_dir_name = os.path.basename(os.getcwd())" << endl;
		outputFile<< "    os.chdir('..\\\\')" << endl;
		outputFile<< "    for dirpath,dirs,files in os.walk(parent_dir_name):" << endl;
		outputFile<< "      for f in files:" << endl;
		outputFile<< "        if output_filename == f:" << endl;
		outputFile<< "            continue" << endl;
		outputFile<< "        fn = os.path.join(dirpath, f)" << endl;
		outputFile<< "        z.write(fn, compress_type=zipfile.ZIP_DEFLATED)" << endl;


		outputFile.close();
	}


} // end namespace


