/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Xml;
using System.Xml.Serialization;

namespace CyPhy2CAD_UILib
{
    public partial class MainForm : Form
    {
        public const string ConfigFilename = "CyPhy2CAD_config.xml";
        public CyPhy2CAD_UILib.CADOptions CadOptionsHolder { get; set; }
        public string ProjectPath { get; set; }
        public bool Automated { get; set; }
        public bool PrepareIFab { get; set; }

        /*
        private Dictionary<string, int> dd = new Dictionary<string, int> { 
                                                                             { "AP203_Single_File", 0 }, 
                                                                             { "AP203_E2_Single_File", 1 }, 
                                                                             { "AP203_E2_Seperate_Part_Files", 2 } , 
                                                                             { "AP214_Single_File", 3 },
                                                                             { "AP214_E2_Seperate_Part_Files", 4 }
                                                                         };
        */

        public MainForm(string projectpath,
                        string outputdir, 
                        bool automated = false,
                        bool prepareifab = false)
        {
            InitializeComponent();

            CadOptionsHolder = new CyPhy2CAD_UILib.CADOptions();
            ProjectPath = projectpath;
            Automated = automated;
            PrepareIFab = prepareifab;

            // Parse config file or use default settings.
            GetOptionsFromFile();

            checkBox_ProjectManifest.Checked = CadOptionsHolder.UseProjectManifest;
            // Enable/Disable controls
            // [1] automated --> outputdir textbox readonly, outputdir button disabled
            // [2] prepareifab --> step formats readonly
            if (Automated)
            {
                textBox_OutputDir.Text = outputdir;
                textBox_OutputDir.ReadOnly = true;
                button_OutputDir.Enabled = false;

                CadOptionsHolder.OutputDirectory = outputdir;
            }
            else
            {
                textBox_OutputDir.Text = CadOptionsHolder.OutputDirectory;
            }

            textBox_AuxiliaryDir.Text = CadOptionsHolder.AuxiliaryDirectory;

            // PrepareIFab: 1 step formats selected
            // Other: config file
            if (PrepareIFab)
            {
                SetCheckListBoxItem("AP203_E2_Seperate_Part_Files");
                clb_Step.Enabled = false;
            }
            else
            {
                for (int i = 0; i < CadOptionsHolder.StepFormats.Count; i++)
                    SetCheckListBoxItem(CadOptionsHolder.StepFormats[i]);
            }
        }

        private void button_OutputDir_Click(object sender, EventArgs e)
        {
            // show folder list
            string outputDir = RunFolderDlg();
            textBox_OutputDir.Text = outputDir;
        }

        private void button_AuxiliaryDir_Click(object sender, EventArgs e)
        {
            // show folder list
            string auxDir = RunFolderDlg();
            textBox_AuxiliaryDir.Text = auxDir;
        }

        private void button_Ok_Click(object sender, EventArgs e)
        {
            // save settings to config file
            SaveOptionsToFile();
            DialogResult = System.Windows.Forms.DialogResult.OK;
            Close();
        }

        private void button_Cancel_Click(object sender, EventArgs e)
        {
            DialogResult = System.Windows.Forms.DialogResult.Cancel;
            Close();
        }

        private void SaveOptionsToFile()
        {
            if (Directory.Exists(ProjectPath))
            {
                CadOptionsHolder.AuxiliaryDirectory = textBox_AuxiliaryDir.Text;
                CadOptionsHolder.OutputDirectory = textBox_OutputDir.Text;
                CadOptionsHolder.UseProjectManifest = checkBox_ProjectManifest.Checked;

                if (clb_Step.CheckedItems.Count > 0)
                {
                    CadOptionsHolder.StepFormats = clb_Step.CheckedItems.Cast<string>().ToList();
                }
                else
                {
                    CadOptionsHolder.StepFormats.Clear();
                }

                var settingsFilename = Path.Combine(ProjectPath, ConfigFilename);
                XmlSerializer xs = new XmlSerializer(typeof(CyPhy2CAD_UILib.CADOptions));
                using (XmlWriter writer = XmlWriter.Create(settingsFilename))
                {
                    xs.Serialize(writer, CadOptionsHolder);
                }
            }
        }

        private void GetOptionsFromFile()
        {
            string filename = Path.Combine(ProjectPath, ConfigFilename);
            if (File.Exists(filename))
            {
                XmlSerializer xs = new XmlSerializer(typeof(CyPhy2CAD_UILib.CADOptions));
                using (XmlReader reader = XmlReader.Create(filename))
                {
                    CadOptionsHolder = xs.Deserialize(reader) as CyPhy2CAD_UILib.CADOptions;
                }
            }
            else       // use default
            {
                CadOptionsHolder.AuxiliaryDirectory = "";
                CadOptionsHolder.OutputDirectory = "";
                CadOptionsHolder.UseProjectManifest = true;
                CadOptionsHolder.StepFormats.Add("AP203_E2_Seperate_Part_Files");
                CadOptionsHolder.StepFormats.Add("AP203_E2_Single_File");
            }
        }

        private void SetCheckListBoxItem(string tokens)
        {
            int index = clb_Step.Items.IndexOf(tokens);
            clb_Step.SetItemChecked(index, true);
        }

        private string RunFolderDlg()
        {
            string selectedDir = "";
            string currentDir = Environment.CurrentDirectory;
            var folderDlg = new META.FolderBrowserDialog();
            folderDlg.SelectedPath = currentDir;
            var result = folderDlg.ShowDialog();

            if (result == System.Windows.Forms.DialogResult.OK)
            {
                selectedDir = Path.GetFullPath(folderDlg.SelectedPath);
            }


            return selectedDir;
        }

    }  // end class

}  // end namespace
