/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.IO;
using System.Reflection;
using GME.CSharp;
using GME.MGA;
using GME.MGA.Meta;
using GME.MGA.Core;
using GME.MGA.Parser;
using ISIS.GME.Common.Interfaces;
using AVM;


// using domain specific interfaces
using CyPhyML = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyMLClasses = ISIS.GME.Dsml.CyPhyML.Classes;

namespace AVM2CyPhyML {

    using TypePair = KeyValuePair<Type, Type>;

    public class TypePairCompare : IComparer<TypePair> {

        public static bool isPairSubclassOf(TypePair typePair1, TypePair typePair2) {
            bool f1 = typePair2.Key.UnderlyingSystemType.IsAssignableFrom(typePair2.Key.UnderlyingSystemType);
            bool f2 = typePair2.Value.UnderlyingSystemType.IsAssignableFrom(typePair2.Value.UnderlyingSystemType);
            return typePair2.Key.UnderlyingSystemType.IsAssignableFrom(typePair1.Key.UnderlyingSystemType) && typePair2.Value.UnderlyingSystemType.IsAssignableFrom(typePair1.Value.UnderlyingSystemType);
        }

        public static string getTypeName( Type type ) {
            return type.Namespace + "." + type.Name;
        }

        public static int compareTypeNames(Type type1, Type type2) {
            return getTypeName(type1.UnderlyingSystemType).CompareTo(getTypeName(type2.UnderlyingSystemType));
        }

        public int Compare(TypePair typePair1, TypePair typePair2) {
            int keyCompare = compareTypeNames(typePair1.Key, typePair2.Key);
            if ( keyCompare != 0 ) return keyCompare;

            return compareTypeNames(typePair1.Value, typePair2.Value);
        }
    }

    public class CyPhyMLComponentBuilder {

        public enum MessageType { OUT, ERROR, WARNING, INFO };

        private object _messageConsole = null;

        public object messageConsole {
            get {
                return _messageConsole;
            }
            set {
                _messageConsole = value;
            }
        }

        public void writeMessage( string message, MessageType messageType = MessageType.OUT ) {

            if ( messageConsole == null ) return;

            TextWriter textWriter = null;

            if ( messageConsole is GMEConsole ) {
                GMEConsole gmeConsole = messageConsole as GMEConsole;
                switch( messageType ) {
                    case MessageType.INFO:
                        textWriter = gmeConsole.Info;
                        break;
                    case MessageType.WARNING:
                        textWriter = gmeConsole.Warning;
                        break;
                    case MessageType.ERROR:
                        textWriter = gmeConsole.Error;
                        break;
                    default:
                    case MessageType.OUT:
                        textWriter = gmeConsole.Out;
                        break;
                }
            } else if ( messageConsole is Console ) {
                switch( messageType ) {
                    case MessageType.INFO:
                    case MessageType.WARNING:
                    case MessageType.ERROR:
                        textWriter = Console.Error;
                        break;
                    default:
                    case MessageType.OUT:
                        textWriter = Console.Out;
                        break;
                }
                
            }

            if ( textWriter != null ) textWriter.WriteLine( message );
        }


        private CyPhyML.Component _cyPhyMLComponent;
        Dictionary<string, CyPhyML.Component> _avmidComponentMap;
        private CyPhyML.RootFolder _cyPhyMLRootFolder;
        private CyPhyML.Units _cyPhyMLUnits;

        private Dictionary<AVM.Associable, Object> _avmAssociableCyPhyMLMap = new Dictionary<AVM.Associable, Object>();

        private class CreateMethodProxyAux<Class> where Class : class {

            private MethodInfo _createMethodInfo;
            private Object _roleStrDefaultValue;

            public CreateMethodProxyAux( Type parentClass ) {

                Type classType = typeof( Class );

                MethodInfo[] methodInfoArray = classType.GetMethods( BindingFlags.Static | BindingFlags.Public );

                bool termLoop = false;
                foreach( MethodInfo methodInfo in methodInfoArray ) {
                    if( methodInfo.Name == "Create" ) {
                        ParameterInfo[] parameterInfoArray = methodInfo.GetParameters();
                        foreach( ParameterInfo parameterInfo in parameterInfoArray ) {
                            if( parameterInfo.ParameterType.Equals( parentClass ) ) {
                                _createMethodInfo = methodInfo;
                                termLoop = true;
                                break;
                            }
                        }
                        if( termLoop ) break;
                    }
                }

                foreach( ParameterInfo parameterInfo in _createMethodInfo.GetParameters() ) {
                    if( parameterInfo.Name == "roleStr" ) {
                        _roleStrDefaultValue = parameterInfo.DefaultValue;
                        break;
                    }
                }

            }

            public void call<BaseClass, ParentClass>( ParentClass parent, out BaseClass baseClassObject ) where BaseClass : class {
                baseClassObject = _createMethodInfo.Invoke( null, new object[] { parent, _roleStrDefaultValue } ) as BaseClass;
            }
        }

        private interface CreateMethodProxyBase {
            void call<BaseClass, ParentClass>( ParentClass parent, out BaseClass baseClassObject ) where BaseClass : class;
            Type getClass();
        }

        private class CreateMethodProxy<Class> : CreateMethodProxyBase where Class : class {

            private Dictionary< string, CreateMethodProxyAux< Class > > _parentClassCMPADictionary = new Dictionary< string, CreateMethodProxyAux< Class > >();

            protected CreateMethodProxy() { }

            private static CreateMethodProxy<Class> _singleton = new CreateMethodProxy<Class>();

            public static CreateMethodProxy<Class> get_singleton() {
                return _singleton;
            }

            public Type getClass() {
                return typeof( Class );
            }

            private CreateMethodProxyAux<Class> getCreateMethodProxyAux( Type parentClassType ) {
                string parentClassName = parentClassType.Name;
                if( _parentClassCMPADictionary.ContainsKey( parentClassName ) ) {
                    return _parentClassCMPADictionary[ parentClassName ];
                }

                CreateMethodProxyAux<Class> createMethodProxyAux = new CreateMethodProxyAux<Class>( parentClassType );
                _parentClassCMPADictionary.Add( parentClassType.Name, createMethodProxyAux );
                return createMethodProxyAux;
            }

            public void call<BaseClass, ParentClass>( ParentClass parent, out BaseClass baseClassObject ) where BaseClass : class {
                getCreateMethodProxyAux( typeof( ParentClass ) ).call( parent, out baseClassObject );
            }
            
        }

        private class ConnectProxy {
            private MethodInfo _connectMethodInfo;
            private object[] _argArray;

            public ConnectProxy(MethodInfo connectMethodInfo, object[] argArray) {
                _connectMethodInfo = connectMethodInfo;
                _argArray = argArray;

            }

            public object connect(Object srcObject, Object dstObject) {
                _argArray[0] = srcObject;
                _argArray[1] = dstObject;
                return _connectMethodInfo.Invoke(null, _argArray);
            }

            public Type declaringType {
                get { return _connectMethodInfo.DeclaringType; }
            }
        }

        private void buildCyPhyMLNameConnectionMap() {

            if (_cyPhyMLConnectionMap.Count > 0) return;

            foreach (MgaMetaFCO connection in _cyPhyMLComponent.Impl.Project.RootMeta.RootFolder.DefinedFCOs.Cast<GME.MGA.Meta.MgaMetaFCO>().Where(x => x.ObjType == GME.MGA.Meta.objtype_enum.OBJTYPE_CONNECTION).ToList()) {

                string[] separators = new string[1] { "::" };
                string[] connectionNameComponents = connection.Name.Split( separators, StringSplitOptions.None );

                string assemblyQualifiedName =
                    "ISIS.GME.Dsml.CyPhyML." 
                    + String.Join( ".", connectionNameComponents, 0, connectionNameComponents.Length - 1 ) 
                    + ( connectionNameComponents.Length == 1 ? "" : "." ) 
                    + "Classes." 
                    + connectionNameComponents[ connectionNameComponents.Length - 1 ];

                //Type classType = Type.GetType( assemblyQualifiedName );
                Type classType =
                    typeof(ISIS.GME.Dsml.CyPhyML.Classes.Component).Assembly.GetType(assemblyQualifiedName);
                
                if (classType == null) {
                    //Console.Out.WriteLine("WARNING: Unable to acquire class \"" + assemblyQualifiedName + "\"");
                    continue;
                }

                MethodInfo[] methodInfoArray = classType.GetMethods(BindingFlags.Static | BindingFlags.Public);
                List< MethodInfo > connectMethodInfoArray = new List< MethodInfo >();

                foreach (MethodInfo methodInfo in methodInfoArray) {
                    if (methodInfo.Name != "Connect") continue;
                    
                    ParameterInfo[] parameterInfoArray = methodInfo.GetParameters();
                    if ( parameterInfoArray.Count() >= 5 && !parameterInfoArray[4].ParameterType.IsAssignableFrom( typeof( CyPhyML.Component ) ) ) continue;

                    connectMethodInfoArray.Add(methodInfo);
                }

                foreach (MethodInfo methodInfo in connectMethodInfoArray) {

                    MethodInfo connectMethodInfo = methodInfo;
                    object[] argArray = new Object[connectMethodInfo.GetParameters().Count()];
                    Type[] typeArray = new Type[2];

                    foreach (ParameterInfo parameterInfo in connectMethodInfo.GetParameters()) {
                        if (parameterInfo.IsOptional) argArray[parameterInfo.Position] = parameterInfo.DefaultValue;
                        else typeArray[parameterInfo.Position] = parameterInfo.ParameterType;
                    }

                    if (connectMethodInfoArray.Count > 1 && typeArray[0].Equals(typeArray[1])) continue;

                    TypePair typePair = new TypePair(typeArray[0], typeArray[1]);
                    if (_cyPhyMLConnectionMap.ContainsKey(typePair) ) {
                        if (!_cyPhyMLConnectionMap[typePair].declaringType.Equals(classType)) {
                            //Console.Out.WriteLine("WARNING:  More than one connection type for types \"" + TypePairCompare.getTypeName(typePair.Key) + "\" and \"" + TypePairCompare.getTypeName(typePair.Value) + "\"");
                            //Console.Out.WriteLine("\t\"" + TypePairCompare.getTypeName(_cyPhyMLConnectionMap[typePair].declaringType) + "\" connection type being used rather than \"" + TypePairCompare.getTypeName(classType) + "\" connection type.");
                        }
                    } else {
                        _cyPhyMLConnectionMap.Add(typePair, new ConnectProxy(connectMethodInfo, argArray));
                    }
                }
            }
        }

        private Dictionary<String, CreateMethodProxyBase> _cyPhyMLNameCreateMethodMap = new Dictionary<String, CreateMethodProxyBase>() {
            { typeof( AVM.META.Electrical ).ToString(),                CreateMethodProxy<CyPhyMLClasses.ElectricalPowerPort>.get_singleton() },
            { typeof( AVM.META.Fluid ).ToString(),                     CreateMethodProxy<CyPhyMLClasses.HydraulicPowerPort>.get_singleton() },
            { typeof( AVM.META.MultiBody ).ToString(),                 CreateMethodProxy<CyPhyMLClasses.MultibodyFramePowerPort>.get_singleton() },
            { typeof( AVM.META.Rotational ).ToString(),                CreateMethodProxy<CyPhyMLClasses.RotationalPowerPort>.get_singleton() },
            { typeof( AVM.META.Heat ).ToString(),                      CreateMethodProxy<CyPhyMLClasses.ThermalPowerPort>.get_singleton() },
            { typeof( AVM.META.Translational ).ToString(),             CreateMethodProxy<CyPhyMLClasses.TranslationalPowerPort>.get_singleton() },
            { typeof( CyPhyMLClasses.SoftwareCommunicationPort ).Name, CreateMethodProxy<CyPhyMLClasses.SoftwareCommunicationPort>.get_singleton() },
            { typeof( AVM.META.BusPort ).ToString(),                   CreateMethodProxy<CyPhyMLClasses.BusPort>.get_singleton() },
            { typeof( AVM.StructuralInterface ).ToString(),            CreateMethodProxy<CyPhyMLClasses.StructuralInterface>.get_singleton() },
            { typeof( AVM.META.AggregatePort ).ToString(),             CreateMethodProxy<CyPhyMLClasses.AggregatePort>.get_singleton() },
            { typeof( AVM.META.AggregateInterface ).ToString(),        CreateMethodProxy<CyPhyMLClasses.ModelicaAggregateInterface>.get_singleton() },
            { typeof( AVM.META.CADAnalysisPoint ).ToString(),          CreateMethodProxy<CyPhyMLClasses.AnalysisPoint>.get_singleton() },
            { typeof( CyPhyMLClasses.Axis ).Name,                      CreateMethodProxy<CyPhyMLClasses.Axis>.get_singleton() },
            { typeof( CyPhyMLClasses.Surface ).Name,                   CreateMethodProxy<CyPhyMLClasses.Surface>.get_singleton() },
            { typeof( CyPhyMLClasses.Point ).Name,                     CreateMethodProxy<CyPhyMLClasses.Point>.get_singleton() },
            { typeof( CyPhyMLClasses.CoordinateSystem ).Name,          CreateMethodProxy<CyPhyMLClasses.CoordinateSystem>.get_singleton() },
            { typeof( CyPhyMLClasses.AxisGeometry ).Name,              CreateMethodProxy<CyPhyMLClasses.AxisGeometry>.get_singleton() },
            { typeof( CyPhyMLClasses.SurfaceGeometry ).Name,           CreateMethodProxy<CyPhyMLClasses.SurfaceGeometry>.get_singleton() },
            { typeof( CyPhyMLClasses.PointGeometry ).Name,             CreateMethodProxy<CyPhyMLClasses.PointGeometry>.get_singleton() },
            { typeof( CyPhyMLClasses.CoordinateSystemGeometry ).Name,  CreateMethodProxy<CyPhyMLClasses.CoordinateSystemGeometry>.get_singleton() },
            { typeof( CyPhyMLClasses.ElectricalPin ).Name,             CreateMethodProxy<CyPhyMLClasses.ElectricalPin>.get_singleton() },
            { typeof( CyPhyMLClasses.RotationalFlange ).Name,          CreateMethodProxy<CyPhyMLClasses.RotationalFlange>.get_singleton() },
            { typeof( CyPhyMLClasses.FluidPort ).Name,                 CreateMethodProxy<CyPhyMLClasses.FluidPort>.get_singleton() },
            { typeof( CyPhyMLClasses.MultibodyFrame ).Name,            CreateMethodProxy<CyPhyMLClasses.MultibodyFrame>.get_singleton() },
            { typeof( CyPhyMLClasses.HeatPort ).Name,                  CreateMethodProxy<CyPhyMLClasses.HeatPort>.get_singleton() },
            { typeof( CyPhyMLClasses.TranslationalFlange ).Name,       CreateMethodProxy<CyPhyMLClasses.TranslationalFlange>.get_singleton() },
            { typeof( CyPhyMLClasses.FlowPort ).Name,                  CreateMethodProxy<CyPhyMLClasses.FlowPort>.get_singleton() },
            { typeof( CyPhyMLClasses.AggregatePort ).Name,             CreateMethodProxy<CyPhyMLClasses.AggregatePort>.get_singleton() },
            { typeof( CyPhyMLClasses.RealInput ).Name,                 CreateMethodProxy<CyPhyMLClasses.RealInput>.get_singleton() },
            { typeof( CyPhyMLClasses.IntegerInput ).Name,              CreateMethodProxy<CyPhyMLClasses.IntegerInput>.get_singleton() },
            { typeof( CyPhyMLClasses.BooleanInput ).Name,              CreateMethodProxy<CyPhyMLClasses.BooleanInput>.get_singleton() },
            { typeof( CyPhyMLClasses.RealOutput ).Name,                CreateMethodProxy<CyPhyMLClasses.RealOutput>.get_singleton() },
            { typeof( CyPhyMLClasses.IntegerOutput ).Name,             CreateMethodProxy<CyPhyMLClasses.IntegerOutput>.get_singleton() },
            { typeof( CyPhyMLClasses.BooleanOutput ).Name,             CreateMethodProxy<CyPhyMLClasses.BooleanOutput>.get_singleton() },
            { typeof( CyPhyMLClasses.BusPort ).Name,                   CreateMethodProxy<CyPhyMLClasses.BusPort>.get_singleton() },
            { typeof( CyPhyMLClasses.ManufacturingModel ).Name,        CreateMethodProxy<CyPhyMLClasses.ManufacturingModel>.get_singleton() },
            { typeof(CyPhyMLClasses.ManufacturingModelParameter ).Name,CreateMethodProxy<CyPhyMLClasses.ManufacturingModelParameter>.get_singleton() }
        };

        private SortedDictionary<TypePair, ConnectProxy> _cyPhyMLConnectionMap = new SortedDictionary<TypePair, ConnectProxy>( new TypePairCompare() );

        private Dictionary<AVM.DatumTypeEnum, string> _datumTypeEnumNameMap = new Dictionary<AVM.DatumTypeEnum, string>() {
            { AVM.DatumTypeEnum.Axis,             typeof( CyPhyMLClasses.Axis ).Name },
            { AVM.DatumTypeEnum.CoordinateSystem, typeof( CyPhyMLClasses.CoordinateSystem ).Name },
            { AVM.DatumTypeEnum.Point,            typeof( CyPhyMLClasses.Point ).Name },
            { AVM.DatumTypeEnum.Surface,          typeof( CyPhyMLClasses.Surface ).Name }
        };

        private Dictionary<AVM.DatumTypeEnum, string> _datumTypeEnumGeometryNameMap = new Dictionary<AVM.DatumTypeEnum, string>() {
            { AVM.DatumTypeEnum.Axis,             typeof( CyPhyMLClasses.AxisGeometry ).Name },
            { AVM.DatumTypeEnum.CoordinateSystem, typeof( CyPhyMLClasses.CoordinateSystemGeometry ).Name },
            { AVM.DatumTypeEnum.Point,            typeof( CyPhyMLClasses.PointGeometry ).Name },
            { AVM.DatumTypeEnum.Surface,          typeof( CyPhyMLClasses.SurfaceGeometry ).Name }
        };

        private Dictionary<AVM.META.CADModelType, CyPhyMLClasses.CADModel.AttributesClass.FileType_enum> _cadFileTypeEnumMap = new Dictionary<AVM.META.CADModelType, CyPhyMLClasses.CADModel.AttributesClass.FileType_enum> {
            { AVM.META.CADModelType.Assembly, CyPhyMLClasses.CADModel.AttributesClass.FileType_enum.Assembly },
            { AVM.META.CADModelType.Part, CyPhyMLClasses.CADModel.AttributesClass.FileType_enum.Part }
        };

        private Dictionary<AVM.DataTypeEnum, CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum> _cadParameterTypeEnumMap = new Dictionary<AVM.DataTypeEnum, CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum>() {
            { AVM.DataTypeEnum.Boolean, CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum.Boolean },
            { AVM.DataTypeEnum.Float,   CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum.Float },
            { AVM.DataTypeEnum.Integer, CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum.Integer },
            { AVM.DataTypeEnum.String,  CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum.String }
        };

        /*private Dictionary< AVM.META.DirectionEnum, CyPhyMLClasses.ModelicaSignalPort.AttributesClass.Directionality_enum > _modelicaSignalPortDirectionalityEnumMap = new Dictionary< AVM.META.DirectionEnum, CyPhyMLClasses.ModelicaSignalPort.AttributesClass.Directionality_enum >() {
            { AVM.META.DirectionEnum.Input, CyPhyMLClasses.ModelicaSignalPort.AttributesClass.Directionality_enum.Input },
            { AVM.META.DirectionEnum.Output, CyPhyMLClasses.ModelicaSignalPort.AttributesClass.Directionality_enum.Output }
        };*/

        private Dictionary<AVM.AlignmentEnum, CyPhyMLClasses.Surface.AttributesClass.Alignment_enum> _surfaceAlignmentEnumMap = new Dictionary<AVM.AlignmentEnum, CyPhyMLClasses.Surface.AttributesClass.Alignment_enum>() {
            { AVM.AlignmentEnum.Align, CyPhyMLClasses.Surface.AttributesClass.Alignment_enum.ALIGN },
            { AVM.AlignmentEnum.Mate, CyPhyMLClasses.Surface.AttributesClass.Alignment_enum.MATE }
        };

        private Dictionary<AVM.OrientationEnum, CyPhyMLClasses.SurfaceGeometry.AttributesClass.Orientation_enum> _surfaceOrientationEnumMap = new Dictionary<AVM.OrientationEnum, CyPhyMLClasses.SurfaceGeometry.AttributesClass.Orientation_enum>() {
            { AVM.OrientationEnum.SideA, CyPhyMLClasses.SurfaceGeometry.AttributesClass.Orientation_enum.SIDE_A },
            { AVM.OrientationEnum.SideB, CyPhyMLClasses.SurfaceGeometry.AttributesClass.Orientation_enum.SIDE_B }
        };

        private Dictionary<AVM.META.BehaviorModelPowerPortTypeEnum, string> _powerPortTypeEnumNameMap = new Dictionary<AVM.META.BehaviorModelPowerPortTypeEnum, string>() {
            { AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaElectrical,        typeof( CyPhyMLClasses.ElectricalPin ).Name },
            { AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaFluid,             typeof( CyPhyMLClasses.FluidPort ).Name },
            { AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaFlow,              typeof( CyPhyMLClasses.FlowPort ).Name },
            { AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaFrame,             typeof( CyPhyMLClasses.MultibodyFrame ).Name },
            { AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaHeat,              typeof( CyPhyMLClasses.HeatPort ).Name },
            { AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaRotational,        typeof( CyPhyMLClasses.RotationalFlange ).Name },
            { AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaTranslational,     typeof( CyPhyMLClasses.TranslationalFlange ).Name }
            //{ AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaFlangeWithBearing, typeof( CyPhyMLClasses.ModelicaPowerPort ).Name }
        };

        private Dictionary<AVM.META.LimitType, CyPhyMLClasses.LimitCheck.AttributesClass.LimitType_enum> _limitTypeEnumMap = new Dictionary<AVM.META.LimitType, CyPhyMLClasses.LimitCheck.AttributesClass.LimitType_enum>()
        {
            { AVM.META.LimitType.AbsMax, CyPhyMLClasses.LimitCheck.AttributesClass.LimitType_enum.max__absolute_value_ },
            { AVM.META.LimitType.Max, CyPhyMLClasses.LimitCheck.AttributesClass.LimitType_enum.max },
            { AVM.META.LimitType.Min, CyPhyMLClasses.LimitCheck.AttributesClass.LimitType_enum.min }
        };

        private Dictionary<String, CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum> _cadFileFormatEnumNameMap = new Dictionary<string, CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum>() {
            { "Creo", CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum.Creo },
            { "AP 203", CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum.AP_203 },
            { "AP 214", CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum.AP_214 },
            { "Solidworks", CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum.Solidworks },
            { "NX", CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum.NX }
        };

        private Dictionary<AVM.META.BehaviorModelSignalPortTypeEnum, string> _signalPortTypeEnumNameMap = new Dictionary<AVM.META.BehaviorModelSignalPortTypeEnum, string>() {
            { AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaBooleanInput,  typeof( CyPhyMLClasses.BooleanInput ).Name },
            { AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaIntegerInput,  typeof( CyPhyMLClasses.IntegerInput ).Name },
            { AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaRealInput,     typeof( CyPhyMLClasses.RealInput ).Name },
            { AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaBooleanOutput, typeof( CyPhyMLClasses.BooleanOutput ).Name },
            { AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaIntegerOutput, typeof( CyPhyMLClasses.IntegerOutput ).Name },
            { AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaRealOutput,    typeof( CyPhyMLClasses.RealOutput ).Name }
        };
        
        private Dictionary<AVM.DataTypeEnum, CyPhyMLClasses.Property.AttributesClass.DataType_enum> _propDataTypeEnumMap = new Dictionary<DataTypeEnum, CyPhyMLClasses.Property.AttributesClass.DataType_enum>()
        {
            { AVM.DataTypeEnum.Boolean, CyPhyMLClasses.Property.AttributesClass.DataType_enum.Boolean },
            { AVM.DataTypeEnum.Float, CyPhyMLClasses.Property.AttributesClass.DataType_enum.Float },
            { AVM.DataTypeEnum.Integer, CyPhyMLClasses.Property.AttributesClass.DataType_enum.Integer },
            { AVM.DataTypeEnum.String, CyPhyMLClasses.Property.AttributesClass.DataType_enum.String }
        };

        private Dictionary<AVM.DataTypeEnum, CyPhyMLClasses.Parameter.AttributesClass.DataType_enum> _paramDataTypeEnumMap = new Dictionary<DataTypeEnum, CyPhyMLClasses.Parameter.AttributesClass.DataType_enum>()
        {
            { AVM.DataTypeEnum.Boolean, CyPhyMLClasses.Parameter.AttributesClass.DataType_enum.Boolean },
            { AVM.DataTypeEnum.Float, CyPhyMLClasses.Parameter.AttributesClass.DataType_enum.Float },
            { AVM.DataTypeEnum.Integer, CyPhyMLClasses.Parameter.AttributesClass.DataType_enum.Integer },
            { AVM.DataTypeEnum.String, CyPhyMLClasses.Parameter.AttributesClass.DataType_enum.String }
        };
        
        private static Dictionary<Assembly.torqueUnit, CyPhyMLClasses.MechanicalJoin.AttributesClass.TorqueUnit_enum> _mechanicalJoinTorqueUnitEnumMap =
         new Dictionary<Assembly.torqueUnit, CyPhyMLClasses.MechanicalJoin.AttributesClass.TorqueUnit_enum>() {
            { Assembly.torqueUnit.Nm, CyPhyMLClasses.MechanicalJoin.AttributesClass.TorqueUnit_enum.N_m }
        };


        private Dictionary<String, CyPhyML.unit> _unitSymbolCyPhyMLUnitMap = new Dictionary<string, CyPhyML.unit>();
        public void setUnitMap(Dictionary<String, CyPhyML.unit> unitSymbolCyPhyMLUnitMap)
        {
            _unitSymbolCyPhyMLUnitMap = unitSymbolCyPhyMLUnitMap;
        }
        public Dictionary<String, CyPhyML.unit> getUnitMap()
        {
            return _unitSymbolCyPhyMLUnitMap;
        }

        private bool avmCyPhyMLInteritance(string portType, Type ancestorClass) {
            if (!_cyPhyMLNameCreateMethodMap.ContainsKey(portType)) return false;
            Type cyPhyMLClass = _cyPhyMLNameCreateMethodMap[portType].getClass();
            Type cyPhyMLInterface = cyPhyMLClass.GetInterface(cyPhyMLClass.Name);
            return cyPhyMLInterface.GetInterfaces().Contains(ancestorClass);
        }

        private void getCyPhyMLRootFolder() {
            _cyPhyMLRootFolder = CyPhyMLClasses.RootFolder.GetRootFolder( _cyPhyMLComponent.Impl.Project );
        }

        private void getCyPhyMLUnits( CyPhyML.RootFolder rootFolder ) {
            foreach( CyPhyML.TypeSpecifications typeSpecifications in rootFolder.Children.TypeSpecificationsCollection ) {
                foreach( CyPhyML.Units units in typeSpecifications.Children.UnitsCollection ) {
                    _cyPhyMLUnits = units;
                    break;
                }
                if( _cyPhyMLUnits != null ) break;
            }
        }

        private void getCyPhyMLUnits() {
            _cyPhyMLUnits = null;

            List< CyPhyML.RootFolder > cyPhyMLRootFolderList = _cyPhyMLRootFolder.LibraryCollection.ToList< CyPhyML.RootFolder >();
            if( cyPhyMLRootFolderList.Count > 0 ) {
                foreach( CyPhyML.RootFolder libraryRootFolder in _cyPhyMLRootFolder.LibraryCollection ) {
                    getCyPhyMLUnits( libraryRootFolder );
                    if( _cyPhyMLUnits != null ) break;
                }
            }

            if ( _cyPhyMLUnits == null ) {
                getCyPhyMLUnits( _cyPhyMLRootFolder );
            }
        }

        private static bool isUnitless( CyPhyML.unit cyPhyMLUnit ) {

            if( cyPhyMLUnit is CyPhyML.si_unit ) {
                CyPhyML.si_unit si_unit_var = cyPhyMLUnit as CyPhyML.si_unit;
                return
                 si_unit_var.Attributes.amount_of_substance_exponent == 0 &&
                 si_unit_var.Attributes.electric_current_exponent == 0 &&
                 si_unit_var.Attributes.length_exponent == 0 &&
                 si_unit_var.Attributes.luminous_intensity_exponent == 0 &&
                 si_unit_var.Attributes.mass_exponent == 0 &&
                 si_unit_var.Attributes.thermodynamic_temperature_exponent == 0 &&
                 si_unit_var.Attributes.time_exponent == 0;
            }

            if( cyPhyMLUnit is CyPhyML.derived_unit ) {
                CyPhyML.derived_unit derived_unit_var = cyPhyMLUnit as CyPhyML.derived_unit;
                List< CyPhyML.derived_unit_element > derived_unit_elementList = derived_unit_var.Children.derived_unit_elementCollection.ToList< CyPhyML.derived_unit_element >();
                return derived_unit_elementList.Count == 1 ? isUnitless( derived_unit_elementList.ElementAt( 0 ).Referred.named_unit ) : false;
            }

            CyPhyML.conversion_based_unit cyPhyMLconversion_based_unit = cyPhyMLUnit as CyPhyML.conversion_based_unit;
            List<CyPhyML.reference_unit> reference_unitList = cyPhyMLconversion_based_unit.Children.reference_unitCollection.ToList<CyPhyML.reference_unit>();
            if( reference_unitList.Count() == 0 ) return false;

            CyPhyML.reference_unit cyPhyMLreference_unit = reference_unitList.ElementAt( 0 );
            return reference_unitList.Count == 1 ? isUnitless( cyPhyMLreference_unit.Referred.unit ) : false;
        }

        private void getCyPhyMLNamedUnits( bool resetUnitLibrary = false ) {
            if( _cyPhyMLUnits == null ) return;
            
            // If the caller has passed in this map already
            if (resetUnitLibrary) _unitSymbolCyPhyMLUnitMap.Clear();
            if (_unitSymbolCyPhyMLUnitMap.Count > 0) return;

            foreach( CyPhyML.unit cyPhyMLUnit in _cyPhyMLUnits.Children.unitCollection ) {
                // Angle-type measures are an exception to this rule.
                /*
                if (cyPhyMLUnit.Attributes.Abbreviation != "rad" &&
                    cyPhyMLUnit.Attributes.Abbreviation != "deg" &&
                    isUnitless(cyPhyMLUnit))
                {
                    continue;
                }*/

                if( !_unitSymbolCyPhyMLUnitMap.ContainsKey( cyPhyMLUnit.Attributes.Abbreviation ) ) {
                    _unitSymbolCyPhyMLUnitMap.Add( cyPhyMLUnit.Attributes.Abbreviation, cyPhyMLUnit );
                }
                if (  !_unitSymbolCyPhyMLUnitMap.ContainsKey( cyPhyMLUnit.Attributes.Symbol )  ) {
                    _unitSymbolCyPhyMLUnitMap.Add( cyPhyMLUnit.Attributes.Symbol, cyPhyMLUnit );
                }
                if (  !_unitSymbolCyPhyMLUnitMap.ContainsKey( cyPhyMLUnit.Attributes.FullName )  ) {
                    _unitSymbolCyPhyMLUnitMap.Add( cyPhyMLUnit.Attributes.FullName, cyPhyMLUnit );
                }
            }
        }

        private void init( bool resetUnitLibrary = false ) {
            getCyPhyMLRootFolder();
            getCyPhyMLUnits();
            getCyPhyMLNamedUnits( resetUnitLibrary );
        }

        private CyPhyMLComponentBuilder(CyPhyML.Components cyPhyMLComponentParent, Dictionary<string, CyPhyML.Component> avmidComponentMap, bool resetUnitLibrary = false, object messageConsoleParameter = null) {
            messageConsole = messageConsoleParameter;
            _cyPhyMLComponent = CyPhyMLClasses.Component.Create( cyPhyMLComponentParent );
            _avmidComponentMap = avmidComponentMap;
            init( resetUnitLibrary );
        }

        private CyPhyMLComponentBuilder(CyPhyML.ComponentAssembly cyPhyMLComponentParent, Dictionary<string, CyPhyML.Component> avmidComponentMap, bool resetUnitLibrary = false, object messageConsoleParameter = null) {
            messageConsole = messageConsoleParameter;
            _cyPhyMLComponent = CyPhyMLClasses.Component.Create( cyPhyMLComponentParent );
            _avmidComponentMap = avmidComponentMap;
            init(resetUnitLibrary);
        }

        private CyPhyML.Component getComponent() {
            return _cyPhyMLComponent;
        }

        private void setComponentName( string componentName ) {
            _cyPhyMLComponent.Name = componentName;
        }

        private void setComponentID( string id, string revision = null, string version = null ) {
            _cyPhyMLComponent.Attributes.AVMID = id;
            if (revision != null)
                _cyPhyMLComponent.Attributes.Revision = revision;
            if (version != null)
                _cyPhyMLComponent.Attributes.Version = version;
        }

        private void setClassifications(IEnumerable<string> categories) {
            foreach (string cat in categories)
            {
                if (_cyPhyMLComponent.Attributes.Classifications != "")
                    _cyPhyMLComponent.Attributes.Classifications += "\n";
                _cyPhyMLComponent.Attributes.Classifications += cat;
            }
        }

        private void process( AVM.META.NamedValue avmNamedValue ) {
//            if (!valueIsDouble(avmNamedValue.Value))
//                return;

            CyPhyML.ValueFlowTarget cyPhyMLValueFlowTarget;

            CyPhyML.unit cyPhyMLUnit = null;

            if (avmNamedValue.Unit != null && avmNamedValue.Unit != "")
            {
                if (_unitSymbolCyPhyMLUnitMap.ContainsKey(avmNamedValue.Unit))
                {
                    cyPhyMLUnit = _unitSymbolCyPhyMLUnitMap[avmNamedValue.Unit];
                }
                else
                {
                    writeMessage( String.Format("WARNING: No unit lib match found for: {0}", avmNamedValue.Unit), MessageType.WARNING);
                }
            }

            if( avmNamedValue.IsParameter.HasValue && avmNamedValue.IsParameter.Value == true ) {
                CyPhyML.Parameter cyPhyMLParameter = CyPhyMLClasses.Parameter.Create( _cyPhyMLComponent );
                cyPhyMLParameter.Attributes.Description = avmNamedValue.Description;
                if (avmNamedValue.Dimension != null)
                    cyPhyMLParameter.Attributes.Dimension = avmNamedValue.Dimension;
                cyPhyMLParameter.Attributes.ID = avmNamedValue.id;
                cyPhyMLValueFlowTarget = cyPhyMLParameter;
                if (avmNamedValue.Value != null)
                    cyPhyMLParameter.Attributes.Value = avmNamedValue.Value.ToString();
                cyPhyMLParameter.Attributes.Range = avmNamedValue.ValidRange;
 //               cyPhyMLParameter.Attributes.Type = avmNamedValue.Type;
                cyPhyMLParameter.Referred.unit = cyPhyMLUnit;
                cyPhyMLParameter.Attributes.DataType = _paramDataTypeEnumMap[ avmNamedValue.DataType ];

                // Check for a Default Value
                if (avmNamedValue.DefaultValue != null)
                    cyPhyMLParameter.Attributes.DefaultValue = avmNamedValue.DefaultValue;

            } else {
                CyPhyML.Property cyPhyMLProperty = CyPhyMLClasses.Property.Create( _cyPhyMLComponent );
                cyPhyMLProperty.Attributes.Description = avmNamedValue.Description;
                cyPhyMLProperty.Attributes.ID = avmNamedValue.id;
                if (avmNamedValue.Dimension != null)
                    cyPhyMLProperty.Attributes.Dimension = avmNamedValue.Dimension;
                cyPhyMLValueFlowTarget = cyPhyMLProperty;
                if (avmNamedValue.Value != null)
                    cyPhyMLProperty.Attributes.Value = avmNamedValue.Value.ToString();
//                cyPhyMLProperty.Attributes.Type = avmNamedValue.Type;
                cyPhyMLProperty.Referred.unit = cyPhyMLUnit;
                if (avmNamedValue.IsProminent.HasValue)
                    cyPhyMLProperty.Attributes.IsProminent = avmNamedValue.IsProminent.Value;
                cyPhyMLProperty.Attributes.DataType = _propDataTypeEnumMap[avmNamedValue.DataType];
            }

            cyPhyMLValueFlowTarget.Name = avmNamedValue.Name;

            _avmAssociableCyPhyMLMap.Add( avmNamedValue, cyPhyMLValueFlowTarget );
        }

        private void process( AVM.META.Calculation avmCalculation ) {

            CyPhyML.CustomFormula cyPhyMLCustomFormula = CyPhyMLClasses.CustomFormula.Create( _cyPhyMLComponent );
            cyPhyMLCustomFormula.Attributes.Expression = avmCalculation.Expression;
            
            // MEGA HACK FOR 1.1.6 C2M2L COMPONENTS. WORST HACK EVER.
            if (String.IsNullOrEmpty(avmCalculation.Expression) && !String.IsNullOrEmpty(avmCalculation.Description))
                cyPhyMLCustomFormula.Attributes.Expression = avmCalculation.Description;
            //////

            cyPhyMLCustomFormula.Name = avmCalculation.Name;
            cyPhyMLCustomFormula.Attributes.ID = avmCalculation.id;            

            _avmAssociableCyPhyMLMap.Add( avmCalculation, cyPhyMLCustomFormula );

            if (avmCalculation.Inputs == null) return;
            foreach (AVM.META.CalculationInput avmCalculationInput in avmCalculation.Inputs)
            {
                _avmAssociableCyPhyMLMap.Add(avmCalculationInput, cyPhyMLCustomFormula);
            }
        }

        private void process( AVM.ExternalPort avmExternalPort, CyPhyML.Port cyPhyMLPort ) {

            cyPhyMLPort.Attributes.ID = avmExternalPort.id;

            cyPhyMLPort.Name = avmExternalPort.Name;

            _avmAssociableCyPhyMLMap.Add( avmExternalPort, cyPhyMLPort );

            if( avmExternalPort is AVM.StructuralInterface ) {

                AVM.StructuralInterface avmStructuralInterface = avmExternalPort as AVM.StructuralInterface;
                CyPhyML.StructuralInterface cyPhyMLStructuralInterface = cyPhyMLPort as CyPhyML.StructuralInterface;

                cyPhyMLStructuralInterface.Attributes.Role = avmStructuralInterface.Role;
                cyPhyMLStructuralInterface.Attributes.Type = avmStructuralInterface.Definition;
                if (avmStructuralInterface.Style == StructuralInterfaceStyle.PhysicalJoin)
                    cyPhyMLStructuralInterface.Attributes.Style = CyPhyMLClasses.StructuralInterface.AttributesClass.Style_enum.Physical_Join;
                else if (avmStructuralInterface.Style == StructuralInterfaceStyle.Positional)
                    cyPhyMLStructuralInterface.Attributes.Style = CyPhyMLClasses.StructuralInterface.AttributesClass.Style_enum.Positional;
                
                foreach( AVM.StructuralInterfaceDatum avmStructuralInterfaceDatum in avmStructuralInterface.Datums ) {

                    string datumTypeName = _datumTypeEnumNameMap[ avmStructuralInterfaceDatum.DatumType ];
                    if( !_cyPhyMLNameCreateMethodMap.ContainsKey( datumTypeName ) ) {
                        writeMessage( "WARNING:  No way to create \"" + datumTypeName + "\" datum.", MessageType.WARNING );
                        continue;
                    }

                    CyPhyML.StructuralInterfaceFeature cyPhyMLStructuralInterfaceFeature = null;
                    _cyPhyMLNameCreateMethodMap[ datumTypeName ].call( cyPhyMLStructuralInterface, out cyPhyMLStructuralInterfaceFeature );

                    _avmAssociableCyPhyMLMap.Add( avmStructuralInterfaceDatum, cyPhyMLStructuralInterfaceFeature );

                    cyPhyMLStructuralInterfaceFeature.Name = avmStructuralInterfaceDatum.Name;
                    cyPhyMLStructuralInterfaceFeature.Attributes.ID = avmStructuralInterfaceDatum.id;

                    if( datumTypeName == "Surface" ) {
                        CyPhyML.Surface cyPhyMLSurface = cyPhyMLStructuralInterfaceFeature as CyPhyML.Surface;
                        if( avmStructuralInterfaceDatum.Alignment.HasValue ) {
                            cyPhyMLSurface.Attributes.Alignment = _surfaceAlignmentEnumMap[ avmStructuralInterfaceDatum.Alignment.Value ];
                        }
                    }
                }

                foreach (AVM.iFAB.Join avmJoin in avmStructuralInterface.DefaultJoins) {
                    process(avmJoin, cyPhyMLStructuralInterface);
                }

            } else if( avmExternalPort is AVM.META.BusPort ) {

                AVM.META.BusPort avmBusPort = avmExternalPort as AVM.META.BusPort;
                CyPhyML.BusPort cyPhyMLBusPort = cyPhyMLPort as CyPhyML.BusPort;

                cyPhyMLBusPort.Attributes.Type = avmBusPort.Type;

            } else if( avmExternalPort is AVM.META.AggregatePort ) {

                AVM.META.AggregatePort avmAggregatePort = avmExternalPort as AVM.META.AggregatePort;
                CyPhyML.AggregatePort cyPhyMLAggregatePort = cyPhyMLPort as CyPhyML.AggregatePort;
                cyPhyMLPort.Attributes.ID = avmAggregatePort.id;
                cyPhyMLAggregatePort.Attributes.Type = avmAggregatePort.Type;
                                
                // Awful hack, delete this now!
                if (avmAggregatePort.Type == "Modelica.Mechanics.MultiBody.Interfaces.FlangeWithBearing")
                {
                    CyPhyML.ModelicaParameter p_Hack = CyPhyMLClasses.ModelicaParameter.Create(cyPhyMLAggregatePort);
                    p_Hack.Name = "includeBearingConnector";
                    p_Hack.Attributes.Value = "world.driveTrainMechanics3D";
                }
                //////////////////////////// Seriously!

                foreach( AVM.ExternalPort avmExternalPortChild in avmAggregatePort.AggregatedPorts.Where( feature => feature is AVM.ExternalPort ) ) {

                    string createMethodKey = avmExternalPortChild.GetType().ToString();
                    if( !_cyPhyMLNameCreateMethodMap.ContainsKey( createMethodKey ) ) {
                        writeMessage( "WARNING:  No way to create \"" + createMethodKey + "\" object.", MessageType.WARNING );
                        continue;
                    }

                    CyPhyML.Port cyPhyMLPortChild = null;
                    _cyPhyMLNameCreateMethodMap[ createMethodKey ].call( cyPhyMLAggregatePort, out cyPhyMLPortChild );

                    process( avmExternalPortChild, cyPhyMLPortChild );
                }

            }
        }

        private void process(AVM.iFAB.Join avmJoin, CyPhyML.StructuralInterface cyPhyMLStructuralInterface) {

            Assembly.assemblyDetails assemblyDetails = Assembly.assemblyDetails.DeserializeFromString(avmJoin.Data);

            CyPhyML.JoinData cyPhyMLJoinData = null;

            if (assemblyDetails.Item is Assembly.welded) {

                Assembly.welded welded = assemblyDetails.Item as Assembly.welded;

                CyPhyML.WeldedJoin cyPhyMLWeldedJoin = CyPhyMLClasses.WeldedJoin.Create(cyPhyMLStructuralInterface);
                cyPhyMLWeldedJoin.Name = "WeldedJoinDefaultJoinData";

                cyPhyMLWeldedJoin.Attributes.InspectionRequirement =
                 (CyPhyMLClasses.WeldedJoin.AttributesClass.InspectionRequirement_enum)Enum.Parse(typeof(CyPhyMLClasses.WeldedJoin.AttributesClass.InspectionRequirement_enum), welded.inspectionRequirement.ToString());

                cyPhyMLWeldedJoin.Attributes.JointType = (CyPhyMLClasses.WeldedJoin.AttributesClass.JointType_enum)Enum.Parse(typeof(CyPhyMLClasses.WeldedJoin.AttributesClass.JointType_enum), welded.jointType.ToString());
                cyPhyMLWeldedJoin.Attributes.Length = Double.Parse(welded.length.Value.ToString());
                cyPhyMLWeldedJoin.Attributes.LengthUnit = (CyPhyMLClasses.WeldedJoin.AttributesClass.LengthUnit_enum)Enum.Parse(typeof(CyPhyMLClasses.WeldedJoin.AttributesClass.LengthUnit_enum), welded.length.unit.ToString());

                cyPhyMLWeldedJoin.Attributes.WeldPenetration =
                 (CyPhyMLClasses.WeldedJoin.AttributesClass.WeldPenetration_enum)Enum.Parse(typeof(CyPhyMLClasses.WeldedJoin.AttributesClass.WeldPenetration_enum), welded.weldPenetration.ToString());

                cyPhyMLWeldedJoin.Attributes.WeldType = (CyPhyMLClasses.WeldedJoin.AttributesClass.WeldType_enum)Enum.Parse(typeof(CyPhyMLClasses.WeldedJoin.AttributesClass.WeldType_enum), welded.weldType.ToString());
                cyPhyMLWeldedJoin.Attributes.TwoSided = welded.twoSided;

                cyPhyMLJoinData = cyPhyMLWeldedJoin;

            } else if (assemblyDetails.Item is Assembly.brazed) {

                Assembly.brazed brazed = assemblyDetails.Item as Assembly.brazed;

                CyPhyML.BrazedJoin cyPhyMLBrazedJoin = CyPhyMLClasses.BrazedJoin.Create(cyPhyMLStructuralInterface);
                cyPhyMLBrazedJoin.Name = "BrazedJoinDefaultJoinData";

                cyPhyMLBrazedJoin.Attributes.FillerMaterial = brazed.fillerMaterial.Value;
                cyPhyMLBrazedJoin.Attributes.FluxMaterial = brazed.fluxMaterial.Value;
                cyPhyMLBrazedJoin.Attributes.Length = Double.Parse(brazed.length.Value.ToString());
                cyPhyMLBrazedJoin.Attributes.LengthUnit = (CyPhyMLClasses.BrazedJoin.AttributesClass.LengthUnit_enum)Enum.Parse(typeof(CyPhyMLClasses.BrazedJoin.AttributesClass.LengthUnit_enum), brazed.length.unit.ToString());

                cyPhyMLJoinData = cyPhyMLBrazedJoin;

            } else if (assemblyDetails.Item is Assembly.soldered) {

                Assembly.soldered soldered = assemblyDetails.Item as Assembly.soldered;

                CyPhyML.SolderedJoin cyPhyMLSolderedJoin = CyPhyMLClasses.SolderedJoin.Create(cyPhyMLStructuralInterface);
                cyPhyMLSolderedJoin.Name = "SolderedJoinDefaultJoinData";

                cyPhyMLSolderedJoin.Attributes.FillerMaterial = soldered.fillerMaterial.Value;
                cyPhyMLSolderedJoin.Attributes.FluxMaterial = soldered.fluxMaterial.Value;
                cyPhyMLSolderedJoin.Attributes.Length = Double.Parse(soldered.length.Value.ToString());
                cyPhyMLSolderedJoin.Attributes.LengthUnit = (CyPhyMLClasses.SolderedJoin.AttributesClass.LengthUnit_enum)Enum.Parse(typeof(CyPhyMLClasses.SolderedJoin.AttributesClass.LengthUnit_enum), soldered.length.unit.ToString());

                cyPhyMLJoinData = cyPhyMLSolderedJoin;

            } else if (assemblyDetails.Item is Assembly.glued) {

                Assembly.glued glued = assemblyDetails.Item as Assembly.glued;

                CyPhyML.GluedJoin cyPhyMLGluedJoin = CyPhyMLClasses.GluedJoin.Create(cyPhyMLStructuralInterface);
                cyPhyMLGluedJoin.Name = "GluedJoinDefaultJoinData";

                cyPhyMLGluedJoin.Attributes.Length = Double.Parse(glued.length.Value.ToString());
                cyPhyMLGluedJoin.Attributes.LengthUnit = (CyPhyMLClasses.GluedJoin.AttributesClass.LengthUnit_enum)Enum.Parse(typeof(CyPhyMLClasses.GluedJoin.AttributesClass.LengthUnit_enum), glued.length.unit.ToString());

                cyPhyMLGluedJoin.Attributes.Material = glued.material.Value;

                cyPhyMLGluedJoin.Attributes.Volume = Double.Parse(glued.volume.Value.ToString());
                cyPhyMLGluedJoin.Attributes.VolumeUnit = (CyPhyMLClasses.GluedJoin.AttributesClass.VolumeUnit_enum)Enum.Parse(typeof(CyPhyMLClasses.GluedJoin.AttributesClass.VolumeUnit_enum), glued.volume.unit.ToString());

                cyPhyMLJoinData = cyPhyMLGluedJoin;

            } else if (assemblyDetails.Item is Assembly.mechanical) {

                Assembly.mechanical mechanical = assemblyDetails.Item as Assembly.mechanical;

                CyPhyML.MechanicalJoin cyPhyMLMechanicalJoin = CyPhyMLClasses.MechanicalJoin.Create(cyPhyMLStructuralInterface);
                cyPhyMLMechanicalJoin.Name = "MechanicalJoinDefaultJoinData";

                cyPhyMLMechanicalJoin.Attributes.FasteningMethod =
                 (CyPhyMLClasses.MechanicalJoin.AttributesClass.FasteningMethod_enum)Enum.Parse(typeof(CyPhyMLClasses.MechanicalJoin.AttributesClass.FasteningMethod_enum), mechanical.fasteningMethod.ToString());

                cyPhyMLMechanicalJoin.Attributes.FasteningQuantity = int.Parse(mechanical.fasteningQuantity);
                cyPhyMLMechanicalJoin.Attributes.Torque = Double.Parse(mechanical.torque.Value.ToString());
                cyPhyMLMechanicalJoin.Attributes.TorqueUnit = _mechanicalJoinTorqueUnitEnumMap[mechanical.torque.unit];

                foreach (Assembly.component component in mechanical.components) {

                    CyPhyML.Fastener cyPhyMLFastener = CyPhyMLClasses.Fastener.Create(cyPhyMLMechanicalJoin);
                    cyPhyMLFastener.Attributes.Quantity = int.Parse(component.quantity);
                    
                    if (component.id != null)
                    {
                        cyPhyMLFastener.Attributes.AVMID = component.id;
                        cyPhyMLFastener.Name = component.id;
                    }
                    if (component.description != null)
                        cyPhyMLFastener.Attributes.Description = component.description;
                    
                    CyPhyML.Component cyPhyMLComponent = null;
                    // If we have this component in the model, set the reference to point to it.
                    if (_avmidComponentMap.TryGetValue(component.id, out cyPhyMLComponent)) 
                    {
                        cyPhyMLFastener.GenericReferred = cyPhyMLComponent;
                        cyPhyMLFastener.Name = cyPhyMLComponent.Name;
                    }                    
                }

                cyPhyMLJoinData = cyPhyMLMechanicalJoin;
            }

            cyPhyMLJoinData.Attributes.Description = assemblyDetails.Item.description;
            cyPhyMLJoinData.Attributes.OverrideSemantics = (CyPhyMLClasses.JoinData.AttributesClass.OverrideSemantics_enum)Enum.Parse(typeof(CyPhyMLClasses.JoinData.AttributesClass.OverrideSemantics_enum), avmJoin.OverrideSemantics.ToString());

        }


        private void process( AVM.META.CADModel avmCADModel ) {

            CyPhyML.CADModel cyPhyMLCADModel = CyPhyMLClasses.CADModel.Create( _cyPhyMLComponent );

            _avmAssociableCyPhyMLMap.Add( avmCADModel, cyPhyMLCADModel );

            cyPhyMLCADModel.Name = avmCADModel.Name;
            cyPhyMLCADModel.Attributes.URI = avmCADModel.Location;

            if (avmCADModel.Author != null)             cyPhyMLCADModel.Attributes.Author = avmCADModel.Author;
            if (avmCADModel.AuthorOrganization != null) cyPhyMLCADModel.Attributes.AuthorOrganization = avmCADModel.AuthorOrganization;
            if (avmCADModel.LanguageVersion != null)    cyPhyMLCADModel.Attributes.LanguageVersion = avmCADModel.LanguageVersion;
            if (avmCADModel.ModelingTool != null)       cyPhyMLCADModel.Attributes.ModelingTool = avmCADModel.ModelingTool;
            if (avmCADModel.ToolVersion != null)        cyPhyMLCADModel.Attributes.ToolVersion = avmCADModel.ToolVersion;
            if (avmCADModel.FileFormat != null) {
                if ( _cadFileFormatEnumNameMap.ContainsKey( avmCADModel.FileFormat ) ) {
                    cyPhyMLCADModel.Attributes.FileFormat = _cadFileFormatEnumNameMap[avmCADModel.FileFormat];
                } else {
                    writeMessage( "WARNING:  Unsupported file format \"" + avmCADModel.FileFormat + "\" in CADModel \"" + avmCADModel.Name + "\"", MessageType.WARNING );
                }
            }
            if (avmCADModel.ModelType != null)
                cyPhyMLCADModel.Attributes.FileType = _cadFileTypeEnumMap[avmCADModel.ModelType];

            if( avmCADModel.CADParameters != null ) {
                foreach( AVM.META.CADParameter avmCADParameter in avmCADModel.CADParameters ) {
                    CyPhyML.CADParameter cyPhyMLCADParameter = CyPhyMLClasses.CADParameter.Create( cyPhyMLCADModel );

                    _avmAssociableCyPhyMLMap.Add( avmCADParameter, cyPhyMLCADParameter );

                    cyPhyMLCADParameter.Attributes.CADParameterType = _cadParameterTypeEnumMap[ avmCADParameter.ParameterType ];
                    cyPhyMLCADParameter.Name = avmCADParameter.Name;
                    cyPhyMLCADParameter.Attributes.ParameterName = avmCADParameter.Name;
                    cyPhyMLCADParameter.Attributes.ID = avmCADParameter.id;
                    if (avmCADParameter.ValidRange != null)
                        cyPhyMLCADParameter.Attributes.Range = avmCADParameter.ValidRange.ToString();
                    if (avmCADParameter.Unit != null)
                        cyPhyMLCADParameter.Attributes.Unit = avmCADParameter.Unit;
                    if (avmCADParameter.DefaultValue != null)
                        cyPhyMLCADParameter.Attributes.DefaultValue = avmCADParameter.DefaultValue;
                    cyPhyMLCADParameter.Attributes.Value = avmCADParameter.Value.ToString();
                    cyPhyMLCADParameter.Attributes.DefaultValue = "0";
                }
            }

            if( avmCADModel.Metrics != null ) {
                foreach( AVM.META.Metric avmMetric in avmCADModel.Metrics ) {
                    CyPhyML.CADMetric cyPhyMLCADMetric = CyPhyMLClasses.CADMetric.Create( cyPhyMLCADModel );

                    _avmAssociableCyPhyMLMap.Add( avmMetric, cyPhyMLCADMetric );

                    cyPhyMLCADMetric.Name = avmMetric.Name;
                    cyPhyMLCADMetric.Attributes.Value = avmMetric.Value.ToString();
                    cyPhyMLCADMetric.Attributes.UnitOfMeasurement = avmMetric.Unit.ToString();
                }
            }

            if( avmCADModel.Datums != null ) {
                foreach( AVM.META.Datum avmDatum in avmCADModel.Datums ) {

                    String avmDatumTypeName = _datumTypeEnumGeometryNameMap[ avmDatum.DatumType ];
                    if( !_cyPhyMLNameCreateMethodMap.ContainsKey( avmDatumTypeName ) ) {
                        writeMessage( "WARNING:  No way to create \"" + avmDatumTypeName + "\" AVM datum.", MessageType.WARNING );
                        continue;
                    }

                    CyPhyML.GeometryFeature cyPhyMLGeometryFeature = null;
                    _cyPhyMLNameCreateMethodMap[ avmDatumTypeName ].call( cyPhyMLCADModel, out cyPhyMLGeometryFeature );

                    _avmAssociableCyPhyMLMap.Add( avmDatum, cyPhyMLGeometryFeature );

                    cyPhyMLGeometryFeature.Name = avmDatum.Name;
                    cyPhyMLGeometryFeature.Attributes.Datum = avmDatum.DatumName;
                    cyPhyMLGeometryFeature.Attributes.ID = avmDatum.id;

                    if( avmDatumTypeName == "SurfaceGeometry" ) {
                        CyPhyML.SurfaceGeometry cyPhyMLSurfaceGeometry = cyPhyMLGeometryFeature as CyPhyML.SurfaceGeometry;
                        cyPhyMLSurfaceGeometry.Attributes.Orientation = avmDatum.Orientation.HasValue ? _surfaceOrientationEnumMap[ avmDatum.Orientation.Value ] : CyPhyMLClasses.SurfaceGeometry.AttributesClass.Orientation_enum.NONE;
                    }

                }
            }

            CyPhy2ComponentModel.CyPhyComponentAutoLayout.LayoutChildrenByName(cyPhyMLCADModel);
        }

        private void processAggregatePortParent< CyPhyMLAggregatePortParentClass, AVMAggregateInterfaceClass >( CyPhyMLAggregatePortParentClass cyPhyMLAggregatePortParent, AVMAggregateInterfaceClass avmAggregateInterfaceObject ) where CyPhyMLAggregatePortParentClass : class {

            IEnumerable<AVM.META.Interface> avmPowerPortInterfaceList, avmSignalPortInterfaceList, avmBehaviorParameterList, avmBusPortInterfaceList, avmAggregateInterfaceList;

            if ( avmAggregateInterfaceObject is AVM.META.BehaviorModel ) {
                AVM.META.BehaviorModel avmBehaviorModel = avmAggregateInterfaceObject as AVM.META.BehaviorModel;
                avmPowerPortInterfaceList = avmBehaviorModel.Interfaces.Where( interfaceMember => interfaceMember is AVM.META.PowerPortInterface );
                avmSignalPortInterfaceList = avmBehaviorModel.Interfaces.Where( interfaceMember => interfaceMember is AVM.META.SignalPortInterface );
                avmBehaviorParameterList = avmBehaviorModel.Interfaces.Where( interfaceMember => interfaceMember is AVM.META.BehaviorParameter );
                avmBusPortInterfaceList = avmBehaviorModel.Interfaces.Where( interfaceMember => interfaceMember is AVM.META.BusPortInterface );
                avmAggregateInterfaceList = avmBehaviorModel.Interfaces.Where( interfaceMember => interfaceMember is AVM.META.AggregateInterface );
            } else {
                AVM.META.AggregateInterface avmAggregateInterface = avmAggregateInterfaceObject as AVM.META.AggregateInterface;
                avmPowerPortInterfaceList = avmAggregateInterface.AggregatedInterfaces.Where( interfaceMember => interfaceMember is AVM.META.PowerPortInterface );
                avmSignalPortInterfaceList = avmAggregateInterface.AggregatedInterfaces.Where( interfaceMember => interfaceMember is AVM.META.SignalPortInterface );
                avmBehaviorParameterList = avmAggregateInterface.AggregatedInterfaces.Where( interfaceMember => interfaceMember is AVM.META.BehaviorParameter );
                avmBusPortInterfaceList = avmAggregateInterface.AggregatedInterfaces.Where( interfaceMember => interfaceMember is AVM.META.BusPortInterface );
                avmAggregateInterfaceList = avmAggregateInterface.AggregatedInterfaces.Where( interfaceMember => interfaceMember is AVM.META.AggregateInterface );
            }


            foreach( AVM.META.PowerPortInterface avmPowerPortInterface in avmPowerPortInterfaceList ) {

                string powerPortTypeName = _powerPortTypeEnumNameMap[ avmPowerPortInterface.PowerPortType ];
                if( !_cyPhyMLNameCreateMethodMap.ContainsKey( powerPortTypeName ) ) {
                    writeMessage( "WARNING:  No way to create \"" + powerPortTypeName + "\" power port type.", MessageType.WARNING );
                    continue;
                }

                CyPhyML.ModelicaPowerPortBase cyPhyMLModelicaPowerPortBase = null;
                _cyPhyMLNameCreateMethodMap[ powerPortTypeName ].call( cyPhyMLAggregatePortParent, out cyPhyMLModelicaPowerPortBase );

                _avmAssociableCyPhyMLMap.Add( avmPowerPortInterface, cyPhyMLModelicaPowerPortBase );

                cyPhyMLModelicaPowerPortBase.Name = avmPowerPortInterface.Name;
                cyPhyMLModelicaPowerPortBase.Attributes.ID = avmPowerPortInterface.id;
            }

            foreach( AVM.META.SignalPortInterface avmSignalPortInterface in avmSignalPortInterfaceList ) {

                string signalPortTypeName = _signalPortTypeEnumNameMap[ avmSignalPortInterface.SignalPortType ];
                if( !_cyPhyMLNameCreateMethodMap.ContainsKey( signalPortTypeName ) ) {
                    writeMessage( "WARNING:  No way to create \"" + signalPortTypeName + "\" signal port type.", MessageType.WARNING );
                    continue;
                }

                CyPhyML.ModelicaSignalPort cyPhyMLModelicaSignalPort;
                _cyPhyMLNameCreateMethodMap[ signalPortTypeName ].call( cyPhyMLAggregatePortParent, out cyPhyMLModelicaSignalPort );

                _avmAssociableCyPhyMLMap.Add( avmSignalPortInterface, cyPhyMLModelicaSignalPort );

                cyPhyMLModelicaSignalPort.Name = avmSignalPortInterface.Name;
                //cyPhyMLModelicaSignalPort.Attributes.Directionality = _modelicaSignalPortDirectionalityEnumMap[ avmSignalPortInterface.Directionality ];
                cyPhyMLModelicaSignalPort.Attributes.ID = avmSignalPortInterface.id;
            }

            foreach( AVM.META.BehaviorParameter avmBehaviorParameter in avmBehaviorParameterList ) {

                CyPhyML.ModelicaParameter cyPhyMLModelicaParameter;
                CreateMethodProxy<CyPhyMLClasses.ModelicaParameter>.get_singleton().call( cyPhyMLAggregatePortParent, out cyPhyMLModelicaParameter );

                _avmAssociableCyPhyMLMap.Add( avmBehaviorParameter, cyPhyMLModelicaParameter );

                cyPhyMLModelicaParameter.Name = avmBehaviorParameter.Name;
                if (avmBehaviorParameter.Value != null)
                    cyPhyMLModelicaParameter.Attributes.Value = avmBehaviorParameter.Value;
                if (avmBehaviorParameter.DefaultValue != null)
                    cyPhyMLModelicaParameter.Attributes.DefaultValue = avmBehaviorParameter.DefaultValue;
                cyPhyMLModelicaParameter.Attributes.ID = avmBehaviorParameter.id;
            }

            foreach (AVM.META.BusPortInterface avmBusPortInterface in avmBusPortInterfaceList) {

                CyPhyML.ModelicaBusPortInterface cyPhyMLModelicaBusPortInterface;
                CreateMethodProxy<CyPhyMLClasses.ModelicaBusPortInterface>.get_singleton().call(cyPhyMLAggregatePortParent, out cyPhyMLModelicaBusPortInterface);

                _avmAssociableCyPhyMLMap.Add(avmBusPortInterface, cyPhyMLModelicaBusPortInterface);

                cyPhyMLModelicaBusPortInterface.Name = avmBusPortInterface.Name;
                cyPhyMLModelicaBusPortInterface.Attributes.Type = avmBusPortInterface.InterfaceType;
                cyPhyMLModelicaBusPortInterface.Attributes.ID = avmBusPortInterface.id;
            }

            foreach (AVM.META.AggregateInterface avmAggregateInterface in avmAggregateInterfaceList) {

                CyPhyML.ModelicaAggregateInterface cyPhyMLModelicaAggregateInterface;
                CreateMethodProxy<CyPhyMLClasses.ModelicaAggregateInterface>.get_singleton().call( cyPhyMLAggregatePortParent, out cyPhyMLModelicaAggregateInterface );

                cyPhyMLModelicaAggregateInterface.Name = avmAggregateInterface.Name;
                cyPhyMLModelicaAggregateInterface.Attributes.Type = avmAggregateInterface.InterfaceType;
                cyPhyMLModelicaAggregateInterface.Attributes.ID = avmAggregateInterface.id;

                _avmAssociableCyPhyMLMap.Add( avmAggregateInterface, cyPhyMLModelicaAggregateInterface );

                processAggregatePortParent( cyPhyMLModelicaAggregateInterface, avmAggregateInterface );

            }

        }

        private void process(AVM.iFAB.ManufacturingModel avmManufacturingModel)
        {
            CyPhyML.ManufacturingModel cyPhyMLManufacturingModel = CyPhyMLClasses.ManufacturingModel.Create(_cyPhyMLComponent);

            cyPhyMLManufacturingModel.Name = "Manufacturing Model";
            cyPhyMLManufacturingModel.Attributes.ID = avmManufacturingModel.id;
            cyPhyMLManufacturingModel.Attributes.Location = avmManufacturingModel.Location;
            _avmAssociableCyPhyMLMap.Add(avmManufacturingModel, cyPhyMLManufacturingModel);
            
            // Handle ports
            if (avmManufacturingModel.ManufacturingModelParameter == null) avmManufacturingModel.ManufacturingModelParameter = new List<AVM.iFAB.ManufacturingModelParameter>();
            foreach (AVM.iFAB.ManufacturingModelParameter avmManufParameter in avmManufacturingModel.ManufacturingModelParameter)
            {
                CyPhyML.ManufacturingModelParameter cyPhyMLManufParam = CyPhyMLClasses.ManufacturingModelParameter.Create(cyPhyMLManufacturingModel);
                cyPhyMLManufParam.Name = avmManufParameter.Name;
                cyPhyMLManufParam.Attributes.ID = avmManufParameter.id;
                cyPhyMLManufParam.Attributes.Description = avmManufParameter.Description;
                cyPhyMLManufParam.Attributes.DefaultValue = avmManufParameter.DefaultValue;
                cyPhyMLManufParam.Attributes.Value = avmManufParameter.Value;

                CyPhyML.unit cyPhyMLUnit = null;

                if (avmManufParameter.Unit != null && avmManufParameter.Unit != "") {
                    if (_unitSymbolCyPhyMLUnitMap.ContainsKey(avmManufParameter.Unit)) {
                        cyPhyMLUnit = _unitSymbolCyPhyMLUnitMap[avmManufParameter.Unit];
                    } else {
                        Console.ForegroundColor = ConsoleColor.DarkRed;
                        Console.WriteLine(String.Format("WARNING: No unit lib match found for: {0}", avmManufParameter.Unit));
                        Console.ResetColor();
                    }
                }

                cyPhyMLManufParam.Referred.unit = cyPhyMLUnit;

                _avmAssociableCyPhyMLMap.Add(avmManufParameter, cyPhyMLManufParam);
            }
        }

        private void process( AVM.META.BehaviorModel avmBehaviorModel ) {

            CyPhyML.ModelicaModel cyPhyMLModelicaModel = CyPhyMLClasses.ModelicaModel.Create( _cyPhyMLComponent );

            cyPhyMLModelicaModel.Name = avmBehaviorModel.Name;
            cyPhyMLModelicaModel.Attributes.URI = avmBehaviorModel.Location;

            _avmAssociableCyPhyMLMap.Add( avmBehaviorModel, cyPhyMLModelicaModel );

            processAggregatePortParent( cyPhyMLModelicaModel, avmBehaviorModel );
            
            foreach (AVM.META.LimitCheck avmLimitCheck in avmBehaviorModel.LimitChecks)
            {
                CyPhyML.LimitCheck cyPhyMLLimitCheck = CyPhyMLClasses.LimitCheck.Create(cyPhyMLModelicaModel);

                cyPhyMLLimitCheck.Attributes.LimitType = _limitTypeEnumMap[avmLimitCheck.LimitType];
                cyPhyMLLimitCheck.Attributes.Value = avmLimitCheck.Value;
                cyPhyMLLimitCheck.Attributes.VariableName = avmLimitCheck.VariableName;
            }

            foreach( AVM.META.MaterialSpec avmMaterialSpec in avmBehaviorModel.MaterialSpecs ) {
                CyPhyML.ModelicaMaterialSpec cyPhyMLModelicaMaterialSpec = CyPhyMLClasses.ModelicaMaterialSpec.Create( cyPhyMLModelicaModel );

                string compatibleMaterials = "";
                bool first = true;
                foreach( string compatibleMaterial in avmMaterialSpec.CompatibleMaterials ) {
                    if( first ) first = false;
                    else compatibleMaterials += "\n";
                    compatibleMaterials += compatibleMaterial;
                }

                cyPhyMLModelicaMaterialSpec.Attributes.CompatibleMaterials = compatibleMaterials;
                cyPhyMLModelicaMaterialSpec.Name = avmMaterialSpec.RedeclareParameterName;
                if (avmMaterialSpec.RedeclareType.ToLower() == "package")
                    cyPhyMLModelicaMaterialSpec.Attributes.RedeclareType = CyPhyMLClasses.ModelicaMaterialSpec.AttributesClass.RedeclareType_enum.package;
                else if (avmMaterialSpec.RedeclareType.ToLower() == "record")
                    cyPhyMLModelicaMaterialSpec.Attributes.RedeclareType = CyPhyMLClasses.ModelicaMaterialSpec.AttributesClass.RedeclareType_enum.record;
                else if (avmMaterialSpec.RedeclareType.ToLower() == "model")
                    cyPhyMLModelicaMaterialSpec.Attributes.RedeclareType = CyPhyMLClasses.ModelicaMaterialSpec.AttributesClass.RedeclareType_enum.model;
                else
                    cyPhyMLModelicaMaterialSpec.Attributes.RedeclareType = CyPhyMLClasses.ModelicaMaterialSpec.AttributesClass.RedeclareType_enum.package;

                foreach( AVM.META.PowerPortInterface avmPowerPortInterface in avmMaterialSpec.PowerPorts ) {
                    CyPhyML.FluidPort cyPhyMLFluidPort = _avmAssociableCyPhyMLMap[ avmPowerPortInterface ] as CyPhyML.FluidPort;
                    CyPhyMLClasses.MaterialSpecConnection.Connect( cyPhyMLModelicaMaterialSpec, cyPhyMLFluidPort );
                }
            }

            CyPhy2ComponentModel.CyPhyComponentAutoLayout.LayoutChildrenByName(cyPhyMLModelicaModel);
        }

        static private void swap<Class>( ref Class classObject1, ref Class classObject2 ) {
            Class tempClassObject = classObject1;
            classObject1 = classObject2;
            classObject2 = tempClassObject;
        }

        private Boolean valueIsDouble(String str)
        {
            Double result;
            return Double.TryParse(str, out result);
        }

        public void processGenericComplex(object object_var, CyPhyML.GenericComplex cyPhyMLGenericComplex) {

            if (object_var is AVM.Associable) {
                _avmAssociableCyPhyMLMap[(AVM.Associable)object_var] = cyPhyMLGenericComplex;
            }

            cyPhyMLGenericComplex.Attributes.Type = object_var.GetType().Namespace + "." + object_var.GetType().Name;

            PropertyInfo[] propertyInfoArray = object_var.GetType().GetProperties();

            
            foreach (PropertyInfo propertyInfo in propertyInfoArray) {

                Type propertyType = propertyInfo.PropertyType;


                if (propertyInfo.Name.ToLower() == "id") {
                    cyPhyMLGenericComplex.Attributes.ID = (string)propertyInfo.GetValue(object_var, null);
                    continue;
                }
                if (propertyInfo.Name.ToLower() == "name") {
                    cyPhyMLGenericComplex.Name = (string)propertyInfo.GetValue(object_var, null);
                    continue;
                }
                if (propertyInfo.Name.ToLower() == "type") {
                //    cyPhyMLGenericComplex.Attributes.Type = object_var.GetType().Namespace + "." + object_var.GetType().Name;
                //    cyPhyMLGenericComplex.Attributes.Type = (string)propertyInfo.GetValue(object_var, null);
                    continue;
                }

                if (propertyType.IsGenericType && propertyType.GetGenericTypeDefinition() == typeof(List<>)) {
                    Type listType = propertyType.GetGenericArguments()[0];
                    IEnumerable list = (IEnumerable)propertyInfo.GetValue(object_var, null);
                    if (list != null)
                    {
                        bool hasMembers = false;
                        foreach (object object_var2 in list)
                        {
                            hasMembers = true;
                            break;
                        }
                        if (!hasMembers) continue;
                        if (listType.Namespace.Substring(0, "AVM".Length) == "AVM")
                        {
                            CyPhyML.GenericArray cyPhyMLGenericArray = cyPhyMLGenericComplex is CyPhyML.GenericFeature ?
                             CyPhyMLClasses.GenericArray.Create((CyPhyML.GenericFeature)cyPhyMLGenericComplex) : cyPhyMLGenericArray = CyPhyMLClasses.GenericArray.Create((CyPhyML.GenericObject)cyPhyMLGenericComplex);
                            cyPhyMLGenericArray.Name = propertyInfo.Name;
                            foreach (object object_var2 in list)
                            {
                                CyPhyML.GenericObject cyPhyMLGenericObject = CyPhyMLClasses.GenericObject.Create(cyPhyMLGenericArray);
                                processGenericComplex(object_var2, cyPhyMLGenericObject);
                            }
                        }
                        else
                        {
                            CyPhyML.GenericAttribute cyPhyMLGenericAttribute = cyPhyMLGenericComplex is CyPhyML.GenericFeature ?
                             CyPhyMLClasses.GenericAttribute.Create((CyPhyML.GenericFeature)cyPhyMLGenericComplex) : CyPhyMLClasses.GenericAttribute.Create((CyPhyML.GenericObject)cyPhyMLGenericComplex);
                            foreach (object object_var2 in list)
                            {
                                cyPhyMLGenericAttribute.Name = propertyInfo.Name;
                                cyPhyMLGenericAttribute.Attributes.Value = object_var2.ToString();
                            }
                        }
                    }
                } else {
                    CyPhyML.GenericAttribute cyPhyMLGenericAttribute =
                     cyPhyMLGenericComplex is CyPhyML.GenericFeature ?
                      CyPhyMLClasses.GenericAttribute.Create((CyPhyML.GenericFeature)cyPhyMLGenericComplex) : CyPhyMLClasses.GenericAttribute.Create((CyPhyML.GenericObject)cyPhyMLGenericComplex);
                    cyPhyMLGenericAttribute.Name = propertyInfo.Name;
                    object propertyValue = propertyInfo.GetValue(object_var, null);
                    if (propertyValue != null)
                    {
                        cyPhyMLGenericAttribute.Attributes.Value = propertyValue.ToString();
                    }
                    else
                    {
                        cyPhyMLGenericAttribute.Delete();
                    }
                }
            }
        }

        private void createGenericAssociation( string srcId, string dstId ) {
            CyPhyML.GenericAssociations cyPhyMLGenericAssociations = null;
            IEnumerable<CyPhyML.GenericAssociations> cyPhyMLGenericAssociationsList = _cyPhyMLComponent.Children.GenericAssociationsCollection;
            
            cyPhyMLGenericAssociations = cyPhyMLGenericAssociationsList.FirstOrDefault();
            // if empty:
            if (cyPhyMLGenericAssociations == null)
            {
                cyPhyMLGenericAssociations = CyPhyMLClasses.GenericAssociations.Create(_cyPhyMLComponent);
                cyPhyMLGenericAssociations.Name = "GenericAssociations";
            }

            CyPhyML.GenericAssociation cyPhyMLGenericAssociation = CyPhyMLClasses.GenericAssociation.Create(cyPhyMLGenericAssociations);
            cyPhyMLGenericAssociation.Name = "GenericAssociation";
            cyPhyMLGenericAssociation.Attributes.SrcID = srcId;
            cyPhyMLGenericAssociation.Attributes.DstID = dstId;
            return;
        }

        private void process(AVM.Association avmAssociation) {

            AVM.Associable avmAssociableSrc = avmAssociation.SrcAssociable;
            AVM.Associable avmAssociableDst = avmAssociation.DstAssociable;

            if ( avmAssociableSrc == null ) {
                if ( avmAssociableDst == null ) {
                    writeMessage( "WARNING:  avmAssociation has null source and destination", MessageType.WARNING );
                    return;
                } else {
                    writeMessage( "WARNING:  avmAssociation with destination id \"" + avmAssociableDst.id + "\" has null source" );
                    return;
                }
            } else if ( avmAssociableDst == null ) {
                writeMessage( "WARNING:  avmAssociation with source id \"" + avmAssociableSrc.id + "\" has null destination" );
                return;
            }

            // Sometimes, one of the objects in the association will be filtered out, and the assoc will fail. That's okay.
            Object cyPhyMLObjectSrc = null;
            if (  !_avmAssociableCyPhyMLMap.TryGetValue(avmAssociableSrc, out cyPhyMLObjectSrc)  ) return;
            Object cyPhyMLObjectDst = null;
            if (  !_avmAssociableCyPhyMLMap.TryGetValue(avmAssociableDst, out cyPhyMLObjectDst)  ) return;


            if ( _avmAssociableCyPhyMLMap[ avmAssociableSrc ] is CyPhyML.GenericComplex || _avmAssociableCyPhyMLMap[ avmAssociableDst ] is CyPhyML.GenericComplex ) {
                createGenericAssociation(avmAssociableSrc.id, avmAssociableDst.id);
                return;
            }
            
            string cyPhyMLObjectSrcTypeName = cyPhyMLObjectSrc.GetType().UnderlyingSystemType.AssemblyQualifiedName.Replace(".Classes.", ".Interfaces.");
            string cyPhyMLObjectDstTypeName = cyPhyMLObjectDst.GetType().UnderlyingSystemType.AssemblyQualifiedName.Replace(".Classes.", ".Interfaces.");

            Type cyPhyMLObjectSrcType = Type.GetType(cyPhyMLObjectSrcTypeName);
            Type cyPhyMLObjectDstType = Type.GetType(cyPhyMLObjectDstTypeName);

            if ( _cyPhyMLConnectionMap.Count == 0 ) buildCyPhyMLNameConnectionMap();
            TypePair typePair = new TypePair(cyPhyMLObjectSrcType, cyPhyMLObjectDstType);
            TypePair reverseTypePair = new TypePair(cyPhyMLObjectDstType, cyPhyMLObjectSrcType);

            object connection = null;
            if (_cyPhyMLConnectionMap.ContainsKey(typePair)) {

                ConnectProxy connectProxy = _cyPhyMLConnectionMap[typePair];
                connection = connectProxy.connect(cyPhyMLObjectSrc, cyPhyMLObjectDst);

            } else if (_cyPhyMLConnectionMap.ContainsKey(reverseTypePair)) {

                ConnectProxy connectProxy = _cyPhyMLConnectionMap[reverseTypePair];

                object temp = cyPhyMLObjectSrc;
                cyPhyMLObjectSrc = cyPhyMLObjectDst;
                cyPhyMLObjectDst = temp;

                temp = avmAssociableSrc;
                avmAssociableDst = avmAssociableSrc;
                avmAssociableSrc = temp as AVM.Associable;

                connection = connectProxy.connect(cyPhyMLObjectSrc, cyPhyMLObjectDst);

            } else {
                TypePair baseTypePair = new TypePair(typeof(object), typeof(object));
                TypePair reverseBaseTypePair = new TypePair(typeof(object), typeof(object));
                foreach (TypePair compareTypePair in _cyPhyMLConnectionMap.Keys) {
                    if (TypePairCompare.isPairSubclassOf(typePair, compareTypePair) && TypePairCompare.isPairSubclassOf(compareTypePair, baseTypePair)) baseTypePair = compareTypePair;
                    if (TypePairCompare.isPairSubclassOf(reverseTypePair, compareTypePair) && TypePairCompare.isPairSubclassOf(compareTypePair, reverseBaseTypePair)) reverseBaseTypePair = compareTypePair;
                }

                if ( baseTypePair.Key == typeof(object) ) {
                    if ( reverseBaseTypePair.Key == typeof(object) ) {
                        String sOutput = String.Format("{0}: WARNING: cannot create specific connection from \"{1}\" to \"{2}\".  CREATING GENERIC ASSOCIATION INSTEAD.",
                         this.getComponent().Name, TypePairCompare.getTypeName( cyPhyMLObjectSrc.GetType() ) , TypePairCompare.getTypeName( cyPhyMLObjectDst.GetType() )  );
                        //Console.Out.WriteLine(sOutput);
                        createGenericAssociation(avmAssociableSrc.id, avmAssociableDst.id);
                    } else {
                        ConnectProxy connectProxy = _cyPhyMLConnectionMap[ reverseTypePair ] = _cyPhyMLConnectionMap[ reverseBaseTypePair ];
                        connection = connectProxy.connect(cyPhyMLObjectDst, cyPhyMLObjectSrc);
                    }
                } else {
                    ConnectProxy connectProxy = _cyPhyMLConnectionMap[ typePair ] = _cyPhyMLConnectionMap[ baseTypePair ];
                    connection = connectProxy.connect(cyPhyMLObjectSrc, cyPhyMLObjectDst);
                    if (reverseBaseTypePair.Key != typeof(object)) {
                        _cyPhyMLConnectionMap[reverseTypePair] = _cyPhyMLConnectionMap[reverseBaseTypePair];
                    }
                }
            }

            if (
             connection != null &&
             connection.GetType().UnderlyingSystemType.Equals(typeof(CyPhyMLClasses.ValueFlow)) &&
             avmAssociableDst.GetType().UnderlyingSystemType.Equals(typeof(AVM.META.CalculationInput))
            ) {
                CyPhyML.ValueFlow vf_Connection = connection as CyPhyML.ValueFlow;
                vf_Connection.Attributes.FormulaVariableName = (avmAssociableDst as AVM.META.CalculationInput).Name;
            }

            string value = "";
            if (cyPhyMLObjectSrc is CyPhyML.Parameter)     value = (cyPhyMLObjectSrc as CyPhyML.Parameter).Attributes.Value;
            else if (cyPhyMLObjectSrc is CyPhyML.Property) value = (cyPhyMLObjectSrc as CyPhyML.Property).Attributes.Value;

            if (value != "" && cyPhyMLObjectDst is CyPhyML.ModelicaParameter) (cyPhyMLObjectDst as CyPhyML.ModelicaParameter).Attributes.Value = value;
        }

        public CyPhyML.Component AVM2CyPhyMLNonStatic( AVM.Component avmComponent ) {

            setComponentName( avmComponent.Name );
            setComponentID( avmComponent.AVMID, avmComponent.Revision, avmComponent.Version );
            setClassifications( avmComponent.Category );

            int iNamedValues = 0;
            foreach (AVM.META.NamedValue avmNamedValue in avmComponent.Features.Where(feature => feature is AVM.META.NamedValue)) {
                process(avmNamedValue);
                iNamedValues++;
            }

            foreach( AVM.META.Calculation avmCalculation in avmComponent.Features.Where( feature => feature is AVM.META.Calculation ) ) {
                process( avmCalculation );
            }

            foreach( AVM.ExternalPort avmExternalPort in avmComponent.Features.Where( feature => feature is AVM.ExternalPort ) ) {

                CyPhyML.Port cyPhyMLPort = null;
                if (avmExternalPort is AVM.META.SignalPort) {
                    bool isInputSignalPort = true;
                    foreach (AVM.Association avmAssociation in avmComponent.Associations) {
                        AVM.Associable avmAssociable = null;
                        if ( avmAssociation.SrcAssociable == avmExternalPort )    avmAssociable = avmAssociation.DstAssociable;
                        else if (avmAssociation.DstAssociable == avmExternalPort) avmAssociable = avmAssociation.SrcAssociable;
                        if (avmAssociable != null && avmAssociable is AVM.META.SignalPortInterface) {
                            AVM.META.SignalPortInterface avmSignalPortInterface = avmAssociable as AVM.META.SignalPortInterface;
                            if (avmCyPhyMLInteritance(_signalPortTypeEnumNameMap[avmSignalPortInterface.SignalPortType], typeof(CyPhyML.ModelicaOutputSignalPort))) {
                                isInputSignalPort = false;
                            }
                            break;
                        }
                    }
                    cyPhyMLPort = isInputSignalPort ? CyPhyMLClasses.InputSignalPort.Create(_cyPhyMLComponent) as CyPhyML.SignalPort : CyPhyMLClasses.OutputSignalPort.Create(_cyPhyMLComponent);
                } else {

                    string createMethodKey = avmExternalPort.GetType().ToString();
                    if (!_cyPhyMLNameCreateMethodMap.ContainsKey(createMethodKey)) {
                        writeMessage( "WARNING:  No way to create \"" + createMethodKey + "\" object.", MessageType.WARNING);
                        continue;
                    }

                    _cyPhyMLNameCreateMethodMap[createMethodKey].call(_cyPhyMLComponent, out cyPhyMLPort);
                }

                process( avmExternalPort, cyPhyMLPort );
            }

            foreach (AVM.META.CADModel avmCADModel in avmComponent.Features.Where(feature => feature is AVM.META.CADModel)) {
                process(avmCADModel);
            }

            foreach( AVM.META.BehaviorModel avmBehaviorModel in avmComponent.Features.Where( feature => feature is AVM.META.BehaviorModel ) ) {
                process( avmBehaviorModel );
            }

            foreach (AVM.iFAB.ManufacturingModel avmManufacturingModel in avmComponent.Features.Where(feature => feature is AVM.iFAB.ManufacturingModel))
            {
                process(avmManufacturingModel);
            }

            foreach (AVM.Feature avmFeature in avmComponent.Features) {
                if (_avmAssociableCyPhyMLMap.ContainsKey(avmFeature)) continue;
                CyPhyML.GenericFeature cyPhyMLGenericFeature = CyPhyMLClasses.GenericFeature.Create( _cyPhyMLComponent );
                processGenericComplex(avmFeature, cyPhyMLGenericFeature);
            }

            foreach (AVM.Association avmAssociation in avmComponent.Associations)
            {
                process(avmAssociation);
            }

            foreach (AVM.File af in avmComponent.Files)
            {
                CyPhyML.File cf = CyPhyMLClasses.File.Create(_cyPhyMLComponent);
                cf.Attributes.Description = af.Description;
                cf.Attributes.Hash = af.Hash;
                cf.Attributes.Location = af.Location;

                cf.Name = Path.GetFileName(af.Location);
            }

            return _cyPhyMLComponent;
        }

        public static CyPhyML.Component AVM2CyPhymL(CyPhyML.ComponentAssembly cyPhyMLComponentParent, AVM.Component avmComponent, Dictionary< string, CyPhyML.Component > avmidComponentMap, bool resetUnitLib = true, object messageConsole = null)
        {
            CyPhyMLComponentBuilder cyPhyMLComponentBuilder = new CyPhyMLComponentBuilder(cyPhyMLComponentParent, avmidComponentMap, resetUnitLib, messageConsole);
            return cyPhyMLComponentBuilder.AVM2CyPhyMLNonStatic( avmComponent );
        }

        public static CyPhyML.Component AVM2CyPhymL(CyPhyML.Components cyPhyMLComponentParent, AVM.Component avmComponent, Dictionary<string, CyPhyML.Component> avmidComponentMap, bool resetUnitLib = true, object messageConsole = null)
        {
            CyPhyMLComponentBuilder cyPhyMLComponentBuilder = new CyPhyMLComponentBuilder(cyPhyMLComponentParent, avmidComponentMap, resetUnitLib, messageConsole);
            return cyPhyMLComponentBuilder.AVM2CyPhyMLNonStatic( avmComponent );
        }
    }
}
