/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.IO;
using System.Text.RegularExpressions;
using System.Reflection;
using GME.MGA;
using GME.MGA.Core;
using GME.MGA.Parser;


// using domain specific interfaces
using CyPhyML = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyMLClasses = ISIS.GME.Dsml.CyPhyML.Classes;
using AVM;

namespace CyPhyML2AVM {

    public class AVMComponentBuilder {

        static private Dictionary<String, AVM.DatumTypeEnum> CyPhyFeature2DatumType = new Dictionary<String, AVM.DatumTypeEnum>
        {
            { "Axis", AVM.DatumTypeEnum.Axis },
            { "Surface", AVM.DatumTypeEnum.Surface },
            { "Point", AVM.DatumTypeEnum.Point },
            { "CoordinateSystem", AVM.DatumTypeEnum.CoordinateSystem },
            { "AxisGeometry", AVM.DatumTypeEnum.Axis },
            { "SurfaceGeometry", AVM.DatumTypeEnum.Surface },
            { "PointGeometry", AVM.DatumTypeEnum.Point },
            { "CoordinateSystemGeometry", AVM.DatumTypeEnum.CoordinateSystem }
        };

        static private Dictionary<CyPhyMLClasses.Surface.AttributesClass.Alignment_enum, AVM.AlignmentEnum> CyPhyAlignment2Alignment = new Dictionary<CyPhyMLClasses.Surface.AttributesClass.Alignment_enum, AVM.AlignmentEnum>
        {
            { CyPhyMLClasses.Surface.AttributesClass.Alignment_enum.ALIGN, AVM.AlignmentEnum.Align },
            { CyPhyMLClasses.Surface.AttributesClass.Alignment_enum.MATE, AVM.AlignmentEnum.Mate }
        };

        static private Dictionary<CyPhyMLClasses.SurfaceGeometry.AttributesClass.Orientation_enum, AVM.OrientationEnum> CyPhyOrientation2Orientation = new Dictionary<CyPhyMLClasses.SurfaceGeometry.AttributesClass.Orientation_enum, AVM.OrientationEnum>
        {
            { CyPhyMLClasses.SurfaceGeometry.AttributesClass.Orientation_enum.SIDE_A, AVM.OrientationEnum.SideA },
            { CyPhyMLClasses.SurfaceGeometry.AttributesClass.Orientation_enum.SIDE_B, AVM.OrientationEnum.SideB }
        };

        static private Dictionary<CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum, String> CyPhyCADModelType2String = new Dictionary<CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum, String>()
        {
            { CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum.Creo,"Creo" },
            { CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum.AP_203,"AP 203" },
            { CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum.AP_214,"AP 214" },
            { CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum.Solidworks,"Solidworks" },
            { CyPhyMLClasses.CADModel.AttributesClass.FileFormat_enum.NX ,"NX" }
        };

        static private Dictionary<CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum, AVM.DataTypeEnum> CyPhyCADParam2CADParam = new Dictionary<CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum, AVM.DataTypeEnum>
        {
            { CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum.Boolean, AVM.DataTypeEnum.Boolean},
            { CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum.Float, AVM.DataTypeEnum.Float},
            { CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum.Integer, AVM.DataTypeEnum.Integer},
            { CyPhyMLClasses.CADParameter.AttributesClass.CADParameterType_enum.String, AVM.DataTypeEnum.String}
        };

        static private Dictionary<CyPhyMLClasses.CADModel.AttributesClass.FileType_enum, AVM.META.CADModelType> CyPhyCADModelType2AVMCADModelType = new Dictionary<CyPhyMLClasses.CADModel.AttributesClass.FileType_enum, AVM.META.CADModelType>()
        {
            { CyPhyMLClasses.CADModel.AttributesClass.FileType_enum.Assembly, AVM.META.CADModelType.Assembly},
            { CyPhyMLClasses.CADModel.AttributesClass.FileType_enum.Part, AVM.META.CADModelType.Part}
        };

        static private Dictionary<String, AVM.META.BehaviorModelPowerPortTypeEnum> CyPhyModelicaPP2BehaviorMdlPPT = new Dictionary<string, AVM.META.BehaviorModelPowerPortTypeEnum>
        {
            { "ElectricalPin", AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaElectrical},
            { "HeatPort", AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaHeat},
            { "RotationalFlange", AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaRotational},
            { "TranslationalFlange", AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaTranslational},
            { "FluidPort", AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaFluid},
            { "FlowPort", AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaFlow},
            { "MultibodyFrame", AVM.META.BehaviorModelPowerPortTypeEnum.ModelicaFrame}
        };

        static private Dictionary<CyPhyMLClasses.LimitCheck.AttributesClass.LimitType_enum, AVM.META.LimitType> CyPhyLimitType2AVMLimitType = new Dictionary<CyPhyMLClasses.LimitCheck.AttributesClass.LimitType_enum, AVM.META.LimitType>()
        {
            { CyPhyMLClasses.LimitCheck.AttributesClass.LimitType_enum.max__absolute_value_, AVM.META.LimitType.AbsMax },
            { CyPhyMLClasses.LimitCheck.AttributesClass.LimitType_enum.max, AVM.META.LimitType.Max },
            { CyPhyMLClasses.LimitCheck.AttributesClass.LimitType_enum.min, AVM.META.LimitType.Min }
        };

        static private Dictionary<String, AVM.META.BehaviorModelSignalPortTypeEnum> CyPhyModelicaSigPort2SigPort = new Dictionary<string, AVM.META.BehaviorModelSignalPortTypeEnum>
        {
            { "RealInput", AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaRealInput},
            { "RealOutput", AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaRealOutput},
            { "IntegerInput", AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaIntegerInput},
            { "IntegerOutput", AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaIntegerOutput},
            { "BooleanInput", AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaBooleanInput},
            { "BooleanOutput", AVM.META.BehaviorModelSignalPortTypeEnum.ModelicaBooleanOutput}
        };

        static private Dictionary<CyPhyMLClasses.Property.AttributesClass.DataType_enum, AVM.DataTypeEnum> CyPhyPropDataTypeMap = new Dictionary<CyPhyMLClasses.Property.AttributesClass.DataType_enum, DataTypeEnum>()
        {
            { CyPhyMLClasses.Property.AttributesClass.DataType_enum.Boolean, AVM.DataTypeEnum.Boolean },
            { CyPhyMLClasses.Property.AttributesClass.DataType_enum.Float, AVM.DataTypeEnum.Float },
            { CyPhyMLClasses.Property.AttributesClass.DataType_enum.Integer, AVM.DataTypeEnum.Integer },
            { CyPhyMLClasses.Property.AttributesClass.DataType_enum.String, AVM.DataTypeEnum.String }
        };

        static private Dictionary<CyPhyMLClasses.Parameter.AttributesClass.DataType_enum, AVM.DataTypeEnum> CyPhyParamDataTypeMap = new Dictionary<CyPhyMLClasses.Parameter.AttributesClass.DataType_enum, DataTypeEnum>()
        {
            { CyPhyMLClasses.Parameter.AttributesClass.DataType_enum.Boolean, AVM.DataTypeEnum.Boolean },
            { CyPhyMLClasses.Parameter.AttributesClass.DataType_enum.Float, AVM.DataTypeEnum.Float },
            { CyPhyMLClasses.Parameter.AttributesClass.DataType_enum.Integer, AVM.DataTypeEnum.Integer },
            { CyPhyMLClasses.Parameter.AttributesClass.DataType_enum.String, AVM.DataTypeEnum.String }
        };

        static private Dictionary<String, Type> _externalPortClassMap = new Dictionary<String, Type>() {
            { typeof( CyPhyML.ElectricalPowerPort ).Name,       typeof( AVM.META.Electrical ) },
            { typeof( CyPhyML.HydraulicPowerPort ).Name,        typeof( AVM.META.Fluid ) },
            { typeof( CyPhyML.MultibodyFramePowerPort ).Name,   typeof( AVM.META.MultiBody ) },
            { typeof( CyPhyML.RotationalPowerPort ).Name,       typeof( AVM.META.Rotational ) },
            { typeof( CyPhyML.ThermalPowerPort ).Name,          typeof( AVM.META.Heat ) },
            { typeof( CyPhyML.TranslationalPowerPort ).Name,    typeof( AVM.META.Translational ) },
            { typeof( CyPhyML.SoftwareCommunicationPort ).Name, typeof( AVM.META.CommunicationPort ) },
            { typeof( CyPhyML.InputSignalPort ).Name,           typeof( AVM.META.SignalPort ) },
            { typeof( CyPhyML.OutputSignalPort ).Name,          typeof( AVM.META.SignalPort ) },
            { typeof( CyPhyML.BusPort ).Name,                   typeof( AVM.META.BusPort ) }
        };

        private static Dictionary<CyPhyMLClasses.WeldedJoin.AttributesClass.InspectionRequirement_enum, Assembly.inspectionRequirement> CyPhyWeldedJoinInspectionRequirementMap =
         new Dictionary<CyPhyMLClasses.WeldedJoin.AttributesClass.InspectionRequirement_enum, Assembly.inspectionRequirement>() {
            { CyPhyMLClasses.WeldedJoin.AttributesClass.InspectionRequirement_enum.Visual, Assembly.inspectionRequirement.Visual },
            { CyPhyMLClasses.WeldedJoin.AttributesClass.InspectionRequirement_enum.X_dash_Ray, Assembly.inspectionRequirement.XRay },
            //{ CyPhyMLClasses.WeldedJoin.AttributesClass.InspectionRequirement_enum.Ultrasonic, Assembly.inspectionRequirement.Ultrasonic }
        };

        private static Dictionary<CyPhyMLClasses.MechanicalJoin.AttributesClass.TorqueUnit_enum, Assembly.torqueUnit> CyPhyMechanicalJoinTorqueUnitMap =
         new Dictionary<CyPhyMLClasses.MechanicalJoin.AttributesClass.TorqueUnit_enum, Assembly.torqueUnit>() {
            { CyPhyMLClasses.MechanicalJoin.AttributesClass.TorqueUnit_enum.N_m, Assembly.torqueUnit.Nm }
        };

        private static Dictionary<CyPhyMLClasses.MechanicalJoin.AttributesClass.FasteningMethod_enum, Assembly.fasteningMethod> CyPhyMechanicalJoinFasteningMethodMap =
         new Dictionary<CyPhyMLClasses.MechanicalJoin.AttributesClass.FasteningMethod_enum, Assembly.fasteningMethod>() {
            { CyPhyMLClasses.MechanicalJoin.AttributesClass.FasteningMethod_enum.Bolted, Assembly.fasteningMethod.Bolted }
            //{ CyPhyClasses.MechanicalJoin.AttributesClass.FasteningMethod_enum.Bolted__Blind_Hole_, Assembly.fasteningMethod.BoltedBlindHole},
            //{ CyPhyClasses.MechanicalJoin.AttributesClass.FasteningMethod_enum.Bolted__Through_Hole_, Assembly.fasteningMethod.BoltedThroughHole},
            //{ CyPhyClasses.MechanicalJoin.AttributesClass.FasteningMethod_enum.Machine_Screwed, Assembly.fasteningMethod.MachineScrewed},
            //{ CyPhyClasses.MechanicalJoin.AttributesClass.FasteningMethod_enum.Press_Fitted, Assembly.fasteningMethod.PressFitted},
            //{ CyPhyClasses.MechanicalJoin.AttributesClass.FasteningMethod_enum.Snap_Fitted, Assembly.fasteningMethod.SnapFitted},
        };

        
        static private Regex _doubleRegex = new Regex( @"(?:[0-9]+(?:\.[0-9]*)?|\.[0-9]+)(?:[Ee][+-][0-9]{1,3})?" );
        static private string extractNumber( string input ) {
            Match m = _doubleRegex.Match( input );
            string retval = m.Value;
            if( retval == "" ) retval = "0";
            return retval;
        }

        private Dictionary<Object, Object> _cyPhyMLAVMAssociableMap = new Dictionary< Object, Object >();
        private Dictionary<string, object> _idCyPhyMLObjectMap = new Dictionary<string, object>();
        private HashSet<string> _newIDs = new HashSet<string>();

        Dictionary< AVM.META.Calculation, Dictionary< string, AVM.META.CalculationInput > > _avmCalculationNameInputMap = new Dictionary<AVM.META.Calculation,Dictionary<string,AVM.META.CalculationInput>>();

        private HashSet<AVM.META.CADModel> _avmCADModelSet = new HashSet<AVM.META.CADModel>();

        private AVM.Component _avmComponent = new AVM.Component();

        private string getUnit(CyPhyML.unit cyPhyMLUnit) {

            if (cyPhyMLUnit == null) return "";

            string unitName = cyPhyMLUnit.Attributes.Symbol;
            if (unitName == "") {
                unitName = cyPhyMLUnit.Attributes.Abbreviation;
                if (unitName == "") {
                    unitName = cyPhyMLUnit.Attributes.FullName;
                }
            }

            return unitName;
        }

        private void setComponentName( string componentName ) {
            _avmComponent.Name = componentName;
        }

        private AVM.Component getComponent() {
            return _avmComponent;
        }

        private int _idNo = 0;

        private String getUniqueId() {
            string newID = "ID" + _idNo++.ToString();
            while(  _idCyPhyMLObjectMap.ContainsKey( newID )  ) {
                newID = "ID" + _idNo++.ToString();
            }
            _newIDs.Add( newID );
            return newID;
        }

        private AVM.Associable getAVMAssociableProxy(AVM.Associable avmAssociable, string varName, string uniqueId ) {

            AVM.Associable proxyAVMAssociable = avmAssociable;

            if (avmAssociable.GetType().UnderlyingSystemType == typeof(AVM.META.Calculation)) {

                AVM.META.Calculation avmCalculation = avmAssociable as AVM.META.Calculation;

                if (!_avmCalculationNameInputMap.ContainsKey(avmCalculation)) {
                    _avmCalculationNameInputMap.Add(avmCalculation, new Dictionary<string, AVM.META.CalculationInput>());
                }
                Dictionary<string, AVM.META.CalculationInput> nameAVMCalculationInputMap = _avmCalculationNameInputMap[avmCalculation];

                if (!nameAVMCalculationInputMap.ContainsKey(varName)) {

                    AVM.META.CalculationInput avmCalculationInput = new AVM.META.CalculationInput();
                    avmCalculationInput.Name = varName;
                    avmCalculationInput.id = uniqueId;

                    if (avmCalculation.Inputs == null) avmCalculation.Inputs = new List<AVM.META.CalculationInput>();
                    avmCalculation.Inputs.Add(avmCalculationInput);

                    nameAVMCalculationInputMap.Add(varName, avmCalculationInput);
                }
                proxyAVMAssociable = nameAVMCalculationInputMap[varName];
            }

            return proxyAVMAssociable;
        }

        private AVM.Associable getAVMAssociableProxy(CyPhyML.ValueFlowTarget cyPhyMLValueFlowTarget, string varName, string uniqueId) {
            if (!_cyPhyMLAVMAssociableMap.ContainsKey(cyPhyMLValueFlowTarget)) return null;
            return getAVMAssociableProxy(_cyPhyMLAVMAssociableMap[cyPhyMLValueFlowTarget] as AVM.Associable, varName, uniqueId);
        }

        private void connectAVMAssociable( CyPhyML.ValueFlowTarget cyPhyMLValueFlowTarget ) {

            AVM.Associable avmAssociable = _cyPhyMLAVMAssociableMap[cyPhyMLValueFlowTarget] as AVM.Associable;

            foreach( CyPhyML.ValueFlow cyPhyMLValueFlow in cyPhyMLValueFlowTarget.SrcConnections.ValueFlowCollection ) {
                CyPhyML.ValueFlowTarget upstreamCyPhyMLValueFlowTarget = cyPhyMLValueFlow.SrcEnd as CyPhyML.ValueFlowTarget;
                string varName = cyPhyMLValueFlow.Attributes.FormulaVariableName;
                if (varName == "") varName = upstreamCyPhyMLValueFlowTarget.Name;
                AVM.Associable proxyAVMAssociable = getAVMAssociableProxy(avmAssociable, varName, cyPhyMLValueFlow.Guid.ToString());
                createAVMAssociation(upstreamCyPhyMLValueFlowTarget, proxyAVMAssociable);
            }

            foreach( CyPhyML.ValueFlow cyPhyMLValueFlow in cyPhyMLValueFlowTarget.DstConnections.ValueFlowCollection ) {
                CyPhyML.ValueFlowTarget downstreamCyPhyMLValueFlowTarget = cyPhyMLValueFlow.DstEnd as CyPhyML.ValueFlowTarget;
                string varName = cyPhyMLValueFlow.Attributes.FormulaVariableName;
                if (varName == "") varName = cyPhyMLValueFlowTarget.Name;
                AVM.Associable proxyAVMAssociable = getAVMAssociableProxy(downstreamCyPhyMLValueFlowTarget, varName, cyPhyMLValueFlow.Guid.ToString());
                createAVMAssociation(avmAssociable, proxyAVMAssociable);
            }

        }

        private void createAVMAssociation(AVM.Associable avmSrcAssociable, AVM.Associable avmDstAssociable) {

            if (avmSrcAssociable == null || avmDstAssociable == null) return;

            AVM.Association avmAssociation = new AVM.Association();
            avmAssociation.SrcAssociable = avmSrcAssociable;
            avmAssociation.DstAssociable = avmDstAssociable;
            _avmComponent.Associations.AddItem(avmAssociation);
        }

        private void createAVMAssociation(AVM.Associable avmSrcAssociable, Object cyPhyMLObject) {

            if (cyPhyMLObject != null && _cyPhyMLAVMAssociableMap.ContainsKey(cyPhyMLObject)) {

                AVM.Associable avmDstAssociable = _cyPhyMLAVMAssociableMap[cyPhyMLObject] as AVM.Associable;
                AVM.Association avmAssociation = new AVM.Association();
                avmAssociation.SrcAssociable = avmSrcAssociable;
                avmAssociation.DstAssociable = avmDstAssociable;
                _avmComponent.Associations.AddItem(avmAssociation);
            }
        }

        private void createAVMAssociation(Object cyPhyMLObject, AVM.Associable avmDstAssociable) {

            if (cyPhyMLObject != null && _cyPhyMLAVMAssociableMap.ContainsKey(cyPhyMLObject)) {

                AVM.Associable avmSrcAssociable = _cyPhyMLAVMAssociableMap[cyPhyMLObject] as AVM.Associable;
                AVM.Association avmAssociation = new AVM.Association();
                avmAssociation.SrcAssociable = avmSrcAssociable;
                avmAssociation.DstAssociable = avmDstAssociable;
                _avmComponent.Associations.AddItem(avmAssociation);
            }
        }

        public void createAVMProperty(CyPhyML.Property cyPhyMLProperty) {

            // CREATE PropertyClass OBJECT
            AVM.META.NamedValue avmNamedValue = new AVM.META.NamedValue();
            avmNamedValue.IsParameter = false;
            avmNamedValue.Type = avmNamedValue.GetType().ToString();
            avmNamedValue.DataType = CyPhyPropDataTypeMap[cyPhyMLProperty.Attributes.DataType];

            if ( !String.IsNullOrWhiteSpace(cyPhyMLProperty.Attributes.Dimension ) )
                avmNamedValue.Dimension = cyPhyMLProperty.Attributes.Dimension;
            
            avmNamedValue.Unit = getUnit( cyPhyMLProperty.Referred.unit );
            
            // CREATE MAPPINGf
            _cyPhyMLAVMAssociableMap[ cyPhyMLProperty ] = avmNamedValue;

            // BASIC ATTRIBUTE ASSIGNMENTS
            String name = avmNamedValue.Name = cyPhyMLProperty.Name;
            avmNamedValue.Description = cyPhyMLProperty.Attributes.Description;
            avmNamedValue.IsProminent = cyPhyMLProperty.Attributes.IsProminent;

            String id = cyPhyMLProperty.Attributes.ID;
            if (id == "") {
                avmNamedValue.id = cyPhyMLProperty.Impl.GetGuidDisp();
                cyPhyMLProperty.Attributes.ID = avmNamedValue.id;
            } else {
                avmNamedValue.id = cyPhyMLProperty.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLProperty.Attributes.ID] = cyPhyMLProperty;

            avmNamedValue.Value = cyPhyMLProperty.Attributes.Value;

            connectAVMAssociable( cyPhyMLProperty );

            _avmComponent.Features.AddItem(avmNamedValue);
        }

        public void createAVMParameter( CyPhyML.Parameter cyPhyMLParameter ) {
            AVM.META.NamedValue avmNamedValue = new AVM.META.NamedValue();
            avmNamedValue.IsParameter = true;
            avmNamedValue.Type = avmNamedValue.GetType().ToString();
            avmNamedValue.DataType = CyPhyParamDataTypeMap[cyPhyMLParameter.Attributes.DataType];

            if (cyPhyMLParameter.Attributes.Dimension != "")
                avmNamedValue.Dimension = cyPhyMLParameter.Attributes.Dimension;

            avmNamedValue.Unit = getUnit(cyPhyMLParameter.Referred.unit);

            _cyPhyMLAVMAssociableMap[ cyPhyMLParameter ] = avmNamedValue;
            
            avmNamedValue.Name = cyPhyMLParameter.Name;
            avmNamedValue.Description = cyPhyMLParameter.Attributes.Description;
            avmNamedValue.ValidRange = cyPhyMLParameter.Attributes.Range;
            String id = cyPhyMLParameter.Attributes.ID;
            if (id == "") {
                avmNamedValue.id = cyPhyMLParameter.Impl.GetGuidDisp();
                cyPhyMLParameter.Attributes.ID = avmNamedValue.id;
            } else {
                avmNamedValue.id = cyPhyMLParameter.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLParameter.Attributes.ID] = cyPhyMLParameter;

            if ( cyPhyMLParameter.Attributes.DefaultValue != "")
                avmNamedValue.DefaultValue = cyPhyMLParameter.Attributes.DefaultValue;
            avmNamedValue.Value = cyPhyMLParameter.Attributes.Value;
            avmNamedValue.ValidRange = cyPhyMLParameter.Attributes.Range;
            
            connectAVMAssociable( cyPhyMLParameter );

            _avmComponent.Features.AddItem(avmNamedValue);
        }

        public void createAVMCalculation( CyPhyML.SimpleFormula cyPhyMLSimpleFormula ) {

            AVM.META.Calculation avmCalculation = new AVM.META.Calculation();
            avmCalculation.Type = avmCalculation.GetType().ToString();

            _cyPhyMLAVMAssociableMap[ cyPhyMLSimpleFormula ] = avmCalculation;

            avmCalculation.Name = cyPhyMLSimpleFormula.Name;
            String id = cyPhyMLSimpleFormula.Attributes.ID;
            if (id == "") {
                avmCalculation.id = cyPhyMLSimpleFormula.Impl.GetGuidDisp();
                cyPhyMLSimpleFormula.Attributes.ID = avmCalculation.id;
            } else {
                avmCalculation.id = cyPhyMLSimpleFormula.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLSimpleFormula.Attributes.ID] = cyPhyMLSimpleFormula;

            _avmCalculationNameInputMap.Add(avmCalculation, new Dictionary<string, AVM.META.CalculationInput>());


            String method = cyPhyMLSimpleFormula.Attributes.Method.ToString();

            List<String> nameList = new List<String>();

            foreach( CyPhyML.ValueFlow valueFlow in cyPhyMLSimpleFormula.SrcConnections.ValueFlowCollection ) {
                CyPhyML.ValueFlowTarget valueFlowTarget = valueFlow.SrcEnd as CyPhyML.ValueFlowTarget;
                nameList.Add( valueFlowTarget.Name );
            }

            String op = ", ";
            if( method == "Addition" || method == "ArithmeticMean" ) {
                op = " + ";
            } else if( method == "Multiplication" || method == "GeometricMean" ) {
                op = " * ";
            }

            String expression = String.Join( op, nameList );
            if( method == "Maximum" ) {
                expression = "max( " + expression + " )";
            } else if( method == "Minimum" ) {
                expression = "min( " + expression + " )";
            } else if( method == "ArithmeticMean" ) {
                expression = "( " + expression + " )/" + nameList.Count;
            } else if( method == "GeometricMean" ) {
                expression = "pow( " + expression + ", 1.0/" + nameList.Count + " )";
            }

            avmCalculation.Description = expression;

            connectAVMAssociable( cyPhyMLSimpleFormula );

            _avmComponent.Features.AddItem(avmCalculation);
        }

        public void createAVMCalculation( CyPhyML.CustomFormula cyPhyMLCustomFormula ) {
            AVM.META.Calculation avmCalculation = new AVM.META.Calculation();
            avmCalculation.Type = avmCalculation.GetType().ToString();

            _cyPhyMLAVMAssociableMap[ cyPhyMLCustomFormula ] = avmCalculation;

            avmCalculation.Name = cyPhyMLCustomFormula.Name;
            avmCalculation.Expression = cyPhyMLCustomFormula.Attributes.Expression;
            String id = cyPhyMLCustomFormula.Attributes.ID;
            if (id == "") {
                avmCalculation.id = cyPhyMLCustomFormula.Impl.GetGuidDisp();
                cyPhyMLCustomFormula.Attributes.ID = avmCalculation.id;
            } else {
                avmCalculation.id = cyPhyMLCustomFormula.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLCustomFormula.Attributes.ID] = cyPhyMLCustomFormula;

            _avmCalculationNameInputMap.Add(avmCalculation, new Dictionary<string, AVM.META.CalculationInput>());

            connectAVMAssociable( cyPhyMLCustomFormula );

            _avmComponent.Features.AddItem( avmCalculation );
        }

        public AVM.META.PowerPort createAVMExternalPowerPort( CyPhyML.Port cyPhyMLPort ) {

            Type cyPhyMLPortType = cyPhyMLPort.GetType();
            String cyPhyMLPortTypeName = cyPhyMLPortType.Name;

            if( !_externalPortClassMap.ContainsKey( cyPhyMLPortTypeName ) ) {
                Console.Out.WriteLine( "No AVM Class for CyPhyML Port Class \"" + cyPhyMLPortTypeName + "\": skipping" );
                return null;
            }

            Type avmPowerPortType = _externalPortClassMap[ cyPhyMLPortTypeName ];

            AVM.META.PowerPort avmPowerPort = Activator.CreateInstance( avmPowerPortType ) as AVM.META.PowerPort;
            if( avmPowerPort == null ) {
                Console.Out.WriteLine( cyPhyMLPortTypeName + " isn't a PowerPort in AVM ... " );
                return null;
            }

            _cyPhyMLAVMAssociableMap[ cyPhyMLPort ] = avmPowerPort;

            // PowerPorts are not "Features" -- how do we associate a powerport with a power port on a behavioral model.
            avmPowerPort.Name = cyPhyMLPort.Name;
            avmPowerPort.Type = avmPowerPort.GetType().ToString();
            String id = cyPhyMLPort.Attributes.ID;
            if (id == "") {
                avmPowerPort.id = cyPhyMLPort.Impl.GetGuidDisp();
                cyPhyMLPort.Attributes.ID = avmPowerPort.id;
            } else {
                avmPowerPort.id = cyPhyMLPort.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLPort.Attributes.ID] = cyPhyMLPort;
            
            return avmPowerPort;
        }

        public AVM.META.SignalPortBase createAVMSignalPortBase( CyPhyML.SignalPortType cyPhyMLSignalPortType ) {

            Type cyPhyMLSignalPortTypeType = cyPhyMLSignalPortType.GetType();
            String cyPhyMLSignalPortTypeTypeName = cyPhyMLSignalPortTypeType.Name;
            if( !_externalPortClassMap.ContainsKey( cyPhyMLSignalPortTypeTypeName ) ) {
                Console.Out.WriteLine( "No AVM Class for CyPhyML Signal Port Class \"" + cyPhyMLSignalPortTypeTypeName + "\": skipping" );
                return null;
            }

            Type avmSignalPortBaseType = _externalPortClassMap[ cyPhyMLSignalPortTypeTypeName ];

            AVM.META.SignalPortBase avmSignalPortBase = Activator.CreateInstance( avmSignalPortBaseType ) as AVM.META.SignalPortBase ;

            _cyPhyMLAVMAssociableMap[ cyPhyMLSignalPortType ] = avmSignalPortBase;

            avmSignalPortBase.Name = cyPhyMLSignalPortType.Name;
            if (cyPhyMLSignalPortType is CyPhyML.BusPort) {
                avmSignalPortBase.Type = (cyPhyMLSignalPortType as CyPhyML.BusPort).Attributes.Type;
            } else {
                avmSignalPortBase.Type = avmSignalPortBase.GetType().ToString();
            }

            String id = cyPhyMLSignalPortType.Attributes.ID;
            if (id == "") {
                avmSignalPortBase.id = cyPhyMLSignalPortType.Impl.GetGuidDisp();
                cyPhyMLSignalPortType.Attributes.ID = avmSignalPortBase.id;
            } else {
                avmSignalPortBase.id = cyPhyMLSignalPortType.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLSignalPortType.Attributes.ID] = cyPhyMLSignalPortType;

            return avmSignalPortBase;
        }

        public AVM.StructuralInterface createAVMStructuralInterface( CyPhyML.StructuralInterface cyPhyMLStructuralInterface ) {

            AVM.StructuralInterface avmStructuralInterface = new AVM.StructuralInterface();
            avmStructuralInterface.Datums = new List<AVM.StructuralInterfaceDatum>();
            avmStructuralInterface.DefaultJoins = new List<AVM.iFAB.Join>();

            _cyPhyMLAVMAssociableMap[ cyPhyMLStructuralInterface ] = avmStructuralInterface;

            avmStructuralInterface.Name = cyPhyMLStructuralInterface.Name;
            avmStructuralInterface.Role = cyPhyMLStructuralInterface.Attributes.Role;
            avmStructuralInterface.Definition = cyPhyMLStructuralInterface.Attributes.Type;

            if (cyPhyMLStructuralInterface.Attributes.Style == CyPhyMLClasses.StructuralInterface.AttributesClass.Style_enum.Physical_Join)
                avmStructuralInterface.Style = StructuralInterfaceStyle.PhysicalJoin;
            else if (cyPhyMLStructuralInterface.Attributes.Style == CyPhyMLClasses.StructuralInterface.AttributesClass.Style_enum.Positional)
                avmStructuralInterface.Style = StructuralInterfaceStyle.Positional;

            String id = cyPhyMLStructuralInterface.Attributes.ID;
            if (id == "") {
                avmStructuralInterface.id = cyPhyMLStructuralInterface.Impl.GetGuidDisp();
                cyPhyMLStructuralInterface.Attributes.ID = avmStructuralInterface.id;
            } else {
                avmStructuralInterface.id = cyPhyMLStructuralInterface.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLStructuralInterface.Attributes.ID] = cyPhyMLStructuralInterface;

            foreach( CyPhyML.StructuralInterfaceFeature cyPhyMLStructuralInterfaceFeature in cyPhyMLStructuralInterface.Children.StructuralInterfaceFeatureCollection ) {
                AVM.StructuralInterfaceDatum avmStructuralInterfaceDatum = new AVM.StructuralInterfaceDatum();
                avmStructuralInterfaceDatum.Name = cyPhyMLStructuralInterfaceFeature.Name;
                avmStructuralInterfaceDatum.DatumType = CyPhyFeature2DatumType[ cyPhyMLStructuralInterfaceFeature.GetType().Name ];
                if( cyPhyMLStructuralInterfaceFeature is CyPhyML.Surface ) {
                    CyPhyML.Surface s = cyPhyMLStructuralInterfaceFeature as CyPhyML.Surface;
                    avmStructuralInterfaceDatum.Alignment = CyPhyAlignment2Alignment[ ( cyPhyMLStructuralInterfaceFeature as CyPhyML.Surface ).Attributes.Alignment ];
                }

                String id2 = cyPhyMLStructuralInterfaceFeature.Attributes.ID;
                if (id2 == "") {
                    avmStructuralInterfaceDatum.id = cyPhyMLStructuralInterfaceFeature.Impl.GetGuidDisp();
                    cyPhyMLStructuralInterfaceFeature.Attributes.ID = avmStructuralInterfaceDatum.id;
                } else {
                    avmStructuralInterfaceDatum.id = cyPhyMLStructuralInterfaceFeature.Attributes.ID;
                }
                _idCyPhyMLObjectMap[cyPhyMLStructuralInterfaceFeature.Attributes.ID] = cyPhyMLStructuralInterfaceFeature;

                _cyPhyMLAVMAssociableMap[ cyPhyMLStructuralInterfaceFeature ] = avmStructuralInterfaceDatum;

                avmStructuralInterface.Datums.AddItem( avmStructuralInterfaceDatum );
            }

            foreach (CyPhyML.JoinData cyPhyMLJoinData in cyPhyMLStructuralInterface.Children.JoinDataCollection) {
                AVM.iFAB.Join avmJoin = createAVMJoin(cyPhyMLJoinData);
                avmStructuralInterface.DefaultJoins.Add( avmJoin );
            }

            return avmStructuralInterface;
        }

        public AVM.iFAB.Join createAVMJoin(CyPhyML.JoinData cyPhyMLJoinData) {

            AVM.iFAB.Join avmJoin = new AVM.iFAB.Join();

            if (cyPhyMLJoinData.Attributes.OverrideSemantics == CyPhyMLClasses.JoinData.AttributesClass.OverrideSemantics_enum.Override) {
                avmJoin.OverrideSemantics = AVM.iFAB.JoinOverrideSemantics.Override;
            } else if (cyPhyMLJoinData.Attributes.OverrideSemantics == CyPhyMLClasses.JoinData.AttributesClass.OverrideSemantics_enum.Union) {
                avmJoin.OverrideSemantics = AVM.iFAB.JoinOverrideSemantics.Union;
            }

            Assembly.assemblyDetails assemblyDetails = new Assembly.assemblyDetails();

            if (cyPhyMLJoinData is CyPhyML.WeldedJoin || cyPhyMLJoinData.Kind == "WeldedJoin") {

                CyPhyML.WeldedJoin CyPhyMLWeldedJoin = cyPhyMLJoinData as CyPhyML.WeldedJoin;
                if (CyPhyMLWeldedJoin == null)
                    CyPhyMLWeldedJoin = CyPhyMLClasses.WeldedJoin.Cast(cyPhyMLJoinData.Impl);

                Assembly.welded welded = new Assembly.welded();

                welded.inspectionRequirement = CyPhyWeldedJoinInspectionRequirementMap[CyPhyMLWeldedJoin.Attributes.InspectionRequirement];
                welded.jointType = (Assembly.jointType)Enum.Parse(typeof(Assembly.jointType), CyPhyMLWeldedJoin.Attributes.JointType.ToString());

                welded.length = new Assembly.length();
                welded.length.Value = decimal.Parse(CyPhyMLWeldedJoin.Attributes.Length.ToString());
                welded.length.unit = (Assembly.lengthUnit)Enum.Parse(typeof(Assembly.lengthUnit), CyPhyMLWeldedJoin.Attributes.LengthUnit.ToString());

                welded.weldPenetration = (Assembly.weldPenetration)Enum.Parse(typeof(Assembly.weldPenetration), CyPhyMLWeldedJoin.Attributes.WeldPenetration.ToString());
                welded.weldType = (Assembly.weldType)Enum.Parse(typeof(Assembly.weldType), CyPhyMLWeldedJoin.Attributes.WeldType.ToString());

                welded.twoSided = CyPhyMLWeldedJoin.Attributes.TwoSided;

                assemblyDetails.Item = welded;

            } else if (cyPhyMLJoinData is CyPhyML.BrazedJoin || cyPhyMLJoinData.Kind == "BrazedJoin") {

                CyPhyML.BrazedJoin cyPhyMLBrazedJoin = cyPhyMLJoinData as CyPhyML.BrazedJoin;
                if (cyPhyMLBrazedJoin == null)
                    cyPhyMLBrazedJoin = CyPhyMLClasses.BrazedJoin.Cast(cyPhyMLJoinData.Impl);
                Assembly.brazed brazed = new Assembly.brazed();

                brazed.fillerMaterial = new Assembly.material();
                brazed.fillerMaterial.Value = cyPhyMLBrazedJoin.Attributes.FillerMaterial;
                brazed.fluxMaterial = new Assembly.material();
                brazed.fluxMaterial.Value = cyPhyMLBrazedJoin.Attributes.FluxMaterial;

                brazed.length = new Assembly.length();
                brazed.length.Value = decimal.Parse(cyPhyMLBrazedJoin.Attributes.Length.ToString());
                brazed.length.unit = (Assembly.lengthUnit)Enum.Parse(typeof(Assembly.lengthUnit), cyPhyMLBrazedJoin.Attributes.LengthUnit.ToString());

                assemblyDetails.Item = brazed;

            } else if (cyPhyMLJoinData is CyPhyML.SolderedJoin || cyPhyMLJoinData.Kind == "SolderedJoin") {

                CyPhyML.SolderedJoin cyPhyMLSolderedJoin = cyPhyMLJoinData as CyPhyML.SolderedJoin;
                if (cyPhyMLSolderedJoin == null)
                    cyPhyMLSolderedJoin = CyPhyMLClasses.SolderedJoin.Cast(cyPhyMLJoinData.Impl);
                Assembly.soldered soldered = new Assembly.soldered();

                soldered.fillerMaterial = new Assembly.material();
                soldered.fillerMaterial.Value = cyPhyMLSolderedJoin.Attributes.FillerMaterial;
                soldered.fluxMaterial = new Assembly.material();
                soldered.fluxMaterial.Value = cyPhyMLSolderedJoin.Attributes.FluxMaterial;

                soldered.length = new Assembly.length();
                soldered.length.Value = decimal.Parse(cyPhyMLSolderedJoin.Attributes.Length.ToString());
                soldered.length.unit = (Assembly.lengthUnit)Enum.Parse(typeof(Assembly.lengthUnit), cyPhyMLSolderedJoin.Attributes.LengthUnit.ToString());

                assemblyDetails.Item = soldered;

            } else if (cyPhyMLJoinData is CyPhyML.GluedJoin || cyPhyMLJoinData.Kind == "GluedJoin") {

                CyPhyML.GluedJoin cyPhyMLGluedJoin = cyPhyMLJoinData as CyPhyML.GluedJoin;
                if (cyPhyMLGluedJoin == null)
                    cyPhyMLGluedJoin = CyPhyMLClasses.GluedJoin.Cast(cyPhyMLJoinData.Impl);
                Assembly.glued glued = new Assembly.glued();

                glued.length = new Assembly.length();
                glued.length.Value = decimal.Parse(cyPhyMLGluedJoin.Attributes.Length.ToString());
                glued.length.unit = (Assembly.lengthUnit)Enum.Parse(typeof(Assembly.lengthUnit), cyPhyMLGluedJoin.Attributes.LengthUnit.ToString());

                glued.material = new Assembly.material();
                glued.material.Value = cyPhyMLGluedJoin.Attributes.Material;

                glued.volume = new Assembly.volume();
                glued.volume.Value = decimal.Parse(cyPhyMLGluedJoin.Attributes.Volume.ToString());
                glued.volume.unit = (Assembly.volumeUnit)Enum.Parse(typeof(Assembly.volumeUnit), cyPhyMLGluedJoin.Attributes.VolumeUnit.ToString());

                assemblyDetails.Item = glued;

            } else if (cyPhyMLJoinData is CyPhyML.MechanicalJoin || cyPhyMLJoinData.Kind == "MechanicalJoin") {

                CyPhyML.MechanicalJoin cyPhyMLMechanicalJoin = cyPhyMLJoinData as CyPhyML.MechanicalJoin;
                if (cyPhyMLMechanicalJoin == null)
                    cyPhyMLMechanicalJoin = CyPhyMLClasses.MechanicalJoin.Cast(cyPhyMLJoinData.Impl);
                Assembly.mechanical mechanical = new Assembly.mechanical();

                mechanical.fasteningMethod = CyPhyMechanicalJoinFasteningMethodMap[cyPhyMLMechanicalJoin.Attributes.FasteningMethod];
                mechanical.fasteningQuantity = cyPhyMLMechanicalJoin.Attributes.FasteningQuantity.ToString();

                mechanical.torque = new Assembly.torque();
                mechanical.torque.Value = decimal.Parse(cyPhyMLMechanicalJoin.Attributes.Torque.ToString());
                mechanical.torque.unit = CyPhyMechanicalJoinTorqueUnitMap[cyPhyMLMechanicalJoin.Attributes.TorqueUnit];

                List<Assembly.component> lc_Fasteners = new List<Assembly.component>();
                foreach (CyPhyML.Fastener f in cyPhyMLMechanicalJoin.Children.FastenerCollection) {
                    Assembly.component c = new Assembly.component();

                    if (f.Referred == null || f.Referred.Component == null)
                    {
                        // Null reference; use local attributes
                        c.id = f.Attributes.AVMID;
                        c.description = f.Attributes.Description;
                    }
                    else
                    {
                        // Valid reference; use referent's attributes
                        c.id = f.Referred.Component.Attributes.AVMID;
                        c.description = f.Referred.Component.Attributes.Description;
                    }
                    c.quantity = f.Attributes.Quantity.ToString();

                    lc_Fasteners.Add(c);
                }
                mechanical.components = lc_Fasteners.ToArray();

                assemblyDetails.Item = mechanical;

            }


            assemblyDetails.Item.description = cyPhyMLJoinData.Attributes.Description;

            avmJoin.Data = Serializer.Serialize(assemblyDetails);

            return avmJoin;
        }

        public AVM.META.AggregatePort createAVMAggregatePort(CyPhyML.AggregatePort cyPhyMLAggregatePort) {

            AVM.META.AggregatePort avmAggregatePort = new AVM.META.AggregatePort();

            _cyPhyMLAVMAssociableMap[ cyPhyMLAggregatePort ] = avmAggregatePort;

            avmAggregatePort.Name = cyPhyMLAggregatePort.Name;
            avmAggregatePort.Type = cyPhyMLAggregatePort.Attributes.Type;
            String id = cyPhyMLAggregatePort.Attributes.ID;
            if (id == "") {
                avmAggregatePort.id = cyPhyMLAggregatePort.Impl.GetGuidDisp();
                cyPhyMLAggregatePort.Attributes.ID = avmAggregatePort.id;
            } else {
                avmAggregatePort.id = cyPhyMLAggregatePort.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLAggregatePort.Attributes.ID] = cyPhyMLAggregatePort;

            foreach( CyPhyML.PowerPortType cyPhyMLPowerPortType in cyPhyMLAggregatePort.Children.PowerPortTypeCollection ) {
                AVM.META.PowerPort avmPowerPort = createAVMExternalPowerPort( cyPhyMLPowerPortType );
                if( avmPowerPort != null ) {
                    avmAggregatePort.AggregatedPorts.AddItem( avmPowerPort );
                }
            }
            foreach( CyPhyML.SignalPortType cyPhyMLSignalPortType in cyPhyMLAggregatePort.Children.SignalPortTypeCollection ) {
                AVM.META.SignalPortBase avmSignalPortBase = createAVMSignalPortBase( cyPhyMLSignalPortType );
                if( avmSignalPortBase != null ) {
                    avmAggregatePort.AggregatedPorts.AddItem( avmSignalPortBase );
                }
            }
            foreach( CyPhyML.StructuralInterface cyPhyMLStructuralInterface in cyPhyMLAggregatePort.Children.StructuralInterfaceCollection ) {
                AVM.StructuralInterface avmStructuralInterface = createAVMStructuralInterface( cyPhyMLStructuralInterface );
                avmAggregatePort.AggregatedPorts.AddItem( avmStructuralInterface );
            }
            foreach( CyPhyML.AggregatePort cyPhyMLAggregatePortChild in cyPhyMLAggregatePort.Children.AggregatePortCollection ) {
                AVM.META.AggregatePort avmAggregatePortChild = createAVMAggregatePort( cyPhyMLAggregatePortChild );
                avmAggregatePort.AggregatedPorts.AddItem( avmAggregatePortChild );
            }

            return avmAggregatePort;
        }

        public void createAVMCADModel( CyPhyML.CADModel cyPhyMLCADModel ) {
            AVM.META.CADModel avmCADModel = new AVM.META.CADModel();

            _cyPhyMLAVMAssociableMap[ cyPhyMLCADModel ] = avmCADModel;

            avmCADModel.Name = cyPhyMLCADModel.Name;
            avmCADModel.Type = avmCADModel.GetType().ToString();
            avmCADModel.Location = cyPhyMLCADModel.Attributes.URI;
            avmCADModel.FileFormat = CyPhyCADModelType2String[cyPhyMLCADModel.Attributes.FileFormat];

            avmCADModel.Author = cyPhyMLCADModel.Attributes.Author;
            avmCADModel.AuthorOrganization = cyPhyMLCADModel.Attributes.AuthorOrganization;
            avmCADModel.LanguageVersion = cyPhyMLCADModel.Attributes.LanguageVersion;
            avmCADModel.ModelingTool = cyPhyMLCADModel.Attributes.ModelingTool;
            avmCADModel.ToolVersion = cyPhyMLCADModel.Attributes.ToolVersion;

            avmCADModel.ModelType = CyPhyCADModelType2AVMCADModelType[cyPhyMLCADModel.Attributes.FileType];

            _avmComponent.Features.AddItem(avmCADModel);

            avmCADModel.CADParameters = new List<AVM.META.CADParameter>();
            foreach( CyPhyML.CADParameter cyPhyMLCADParameter in cyPhyMLCADModel.Children.CADParameterCollection ) {
                createAVMCADParameter( avmCADModel, cyPhyMLCADParameter );
            }

            avmCADModel.Datums = new List<AVM.META.Datum>();
            foreach( CyPhyML.GeometryFeature cyPhyMLGeometryFeature in cyPhyMLCADModel.Children.GeometryFeatureCollection ) {
                createAVMDatum( avmCADModel, cyPhyMLGeometryFeature );
            }

            avmCADModel.Metrics = new List<AVM.META.Metric>();
            foreach( CyPhyML.CADMetric cyPhyMLCADMetric in cyPhyMLCADModel.Children.CADMetricCollection ) {
                createAVMMetric( avmCADModel, cyPhyMLCADMetric );
            }

        }

        public void createAVMCADParameter( AVM.META.CADModel avmCADModel, CyPhyML.CADParameter cyPhyMLCADParameter ) {

            AVM.META.CADParameter avmCADParameter = new AVM.META.CADParameter();

            _cyPhyMLAVMAssociableMap[ cyPhyMLCADParameter ] = avmCADParameter;
            
            avmCADParameter.Name = cyPhyMLCADParameter.Attributes.ParameterName;
            avmCADParameter.ParameterType = CyPhyCADParam2CADParam[cyPhyMLCADParameter.Attributes.CADParameterType];
            if (cyPhyMLCADParameter.Attributes.DefaultValue != "")
                avmCADParameter.DefaultValue = cyPhyMLCADParameter.Attributes.DefaultValue;
            avmCADParameter.Unit = cyPhyMLCADParameter.Attributes.Unit;
            avmCADParameter.Value = cyPhyMLCADParameter.Attributes.Value;
            avmCADParameter.ValidRange = cyPhyMLCADParameter.Attributes.Range;
            String id = cyPhyMLCADParameter.Attributes.ID;
            if (id == "") {
                avmCADParameter.id = cyPhyMLCADParameter.Impl.GetGuidDisp();
                cyPhyMLCADParameter.Attributes.ID = avmCADParameter.id;
            } else {
                avmCADParameter.id = cyPhyMLCADParameter.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLCADParameter.Attributes.ID] = cyPhyMLCADParameter;
                
            avmCADModel.CADParameters.AddItem( avmCADParameter );

            foreach( CyPhyML.CADParameterPortMap cyPhyMLCADParameterPortMap in cyPhyMLCADParameter.SrcConnections.CADParameterPortMapCollection ) {
                CyPhyML.ValueFlowTarget cyPhyMLValueFlowTarget = cyPhyMLCADParameterPortMap.SrcEnd as CyPhyML.ValueFlowTarget;
                createAVMAssociation(cyPhyMLValueFlowTarget, avmCADParameter);
            }

        }

        public void createAVMDatum( AVM.META.CADModel avmCADModel, CyPhyML.GeometryFeature cyPhyMLGeometryFeature ) {
            AVM.META.Datum avmDatum = new AVM.META.Datum();

            _cyPhyMLAVMAssociableMap[ cyPhyMLGeometryFeature ] = avmDatum;

            avmDatum.Name = cyPhyMLGeometryFeature.Name;
            avmDatum.DatumName = cyPhyMLGeometryFeature.Attributes.Datum;
            avmDatum.DatumType = CyPhyFeature2DatumType[ cyPhyMLGeometryFeature.GetType().Name ];
            String id = cyPhyMLGeometryFeature.Attributes.ID;
            if (id == "") {
                avmDatum.id = cyPhyMLGeometryFeature.Impl.GetGuidDisp();
                cyPhyMLGeometryFeature.Attributes.ID = avmDatum.id;
            } else {
                avmDatum.id = cyPhyMLGeometryFeature.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLGeometryFeature.Attributes.ID] = cyPhyMLGeometryFeature;

            if( cyPhyMLGeometryFeature is CyPhyML.SurfaceGeometry ) {
                CyPhyML.SurfaceGeometry cyPhyMLSurfaceGeometry = cyPhyMLGeometryFeature as CyPhyML.SurfaceGeometry;
                CyPhyMLClasses.SurfaceGeometry.AttributesClass.Orientation_enum cyPhyMLOrientation = cyPhyMLSurfaceGeometry.Attributes.Orientation;
                if( cyPhyMLOrientation != CyPhyMLClasses.SurfaceGeometry.AttributesClass.Orientation_enum.NONE ) {
                    avmDatum.Orientation = CyPhyOrientation2Orientation[ cyPhyMLOrientation ];
                }
            }

            avmCADModel.Datums.AddItem( avmDatum );

            foreach( CyPhyML.FeatureMap cyPhyMLFeatureMap in cyPhyMLGeometryFeature.SrcConnections.FeatureMapCollection ) {
                CyPhyML.SolidModelingFeature srcCyPhyMLFeature = cyPhyMLFeatureMap.SrcEnd as CyPhyML.SolidModelingFeature;
                createAVMAssociation(srcCyPhyMLFeature, avmDatum);
            }
            foreach( CyPhyML.FeatureMap cyPhyMLFeatureMap in cyPhyMLGeometryFeature.DstConnections.FeatureMapCollection ) {
                CyPhyML.SolidModelingFeature dstCyPhyMLFeature = cyPhyMLFeatureMap.DstEnd as CyPhyML.SolidModelingFeature;
                createAVMAssociation(avmDatum, dstCyPhyMLFeature);
            }

            if (cyPhyMLGeometryFeature is CyPhyML.PointGeometry) {
                CyPhyML.PointGeometry cyPhyMLPointGeometry = cyPhyMLGeometryFeature as CyPhyML.PointGeometry;
                foreach (CyPhyML.AnalysisPointMap cyPhyMLAnalysisPointMap in cyPhyMLPointGeometry.SrcConnections.AnalysisPointMapCollection) {
                    CyPhyML.AnalysisPoint cyPhyMLAnalysisPoint = cyPhyMLAnalysisPointMap.SrcEnd as CyPhyML.AnalysisPoint;
                    createAVMAssociation(cyPhyMLAnalysisPoint, avmDatum);
                }
                foreach (CyPhyML.AnalysisPointMap cyPhyMLAnalysisPointMap in cyPhyMLPointGeometry.DstConnections.AnalysisPointMapCollection) {
                    CyPhyML.AnalysisPoint cyPhyMLAnalysisPoint = cyPhyMLAnalysisPointMap.DstEnd as CyPhyML.AnalysisPoint;
                    createAVMAssociation(avmDatum, cyPhyMLAnalysisPoint);
                }
            }
        }

        public void createAVMMetric( AVM.META.CADModel avmCADModel, CyPhyML.CADMetric cyPhyMLCADMetric ) {

            AVM.META.Metric avmMetric = new AVM.META.Metric();

            avmMetric.Name = cyPhyMLCADMetric.Name;
            avmMetric.Unit = cyPhyMLCADMetric.Attributes.UnitOfMeasurement;
            avmMetric.Value = Double.Parse(extractNumber(cyPhyMLCADMetric.Attributes.Value));

            avmCADModel.Metrics.AddItem( avmMetric );

            _cyPhyMLAVMAssociableMap[ cyPhyMLCADMetric ] = avmMetric;

            foreach( CyPhyML.CADMetricPortMap cyPhyMLCADMetricPortMap in cyPhyMLCADMetric.DstConnections.CADMetricPortMapCollection ) {
                CyPhyML.ValueFlowTarget cyPhyMLValueFlowTarget = cyPhyMLCADMetricPortMap.DstEnd as CyPhyML.ValueFlowTarget;
                createAVMAssociation(avmMetric, cyPhyMLValueFlowTarget);
            }

        }

        public void createAVMBehaviorModel( CyPhyML.ModelicaModel cyPhyMLModelicaModel ) {
            AVM.META.BehaviorModel avmBehaviorModel = new AVM.META.BehaviorModel();

            _cyPhyMLAVMAssociableMap[ cyPhyMLModelicaModel ] = avmBehaviorModel;

            avmBehaviorModel.Name = cyPhyMLModelicaModel.Name;
            avmBehaviorModel.ModelingTool = "Modelica";
            avmBehaviorModel.Type = avmBehaviorModel.GetType().ToString();
            avmBehaviorModel.Location = cyPhyMLModelicaModel.Attributes.URI;
            avmBehaviorModel.MaterialSpecs = new List<AVM.META.MaterialSpec>();

            _avmComponent.Features.AddItem(avmBehaviorModel);

            foreach( CyPhyML.ModelicaPowerPortBase cyPhyMLModelicaPowerPortBase in cyPhyMLModelicaModel.Children.ModelicaPowerPortBaseCollection ) {
                AVM.META.PowerPortInterface avmPowerPortInterface = createAVMBehaviorModelPowerPort(cyPhyMLModelicaPowerPortBase );
                avmBehaviorModel.Interfaces.AddItem( avmPowerPortInterface );
            }

            foreach( CyPhyML.ModelicaSignalPort cyPhyMLModelicaSignalPort in cyPhyMLModelicaModel.Children.ModelicaSignalPortCollection ) {
                AVM.META.SignalPortInterface avmSignalPortInterface = createAVMBehaviorModelSignalPort( cyPhyMLModelicaSignalPort );
                avmBehaviorModel.Interfaces.AddItem( avmSignalPortInterface );
            }

            foreach( CyPhyML.ModelicaParameter cyPhyMLModelicaParameter in cyPhyMLModelicaModel.Children.ModelicaParameterCollection ) {
                AVM.META.BehaviorParameter avmBehaviorParameter = createAVMBehaviorModelParameter( cyPhyMLModelicaParameter );
                avmBehaviorModel.Interfaces.AddItem( avmBehaviorParameter );
            }

            foreach (CyPhyML.ModelicaBusPortInterface cyPhyMLModelicaBusPortInterface in cyPhyMLModelicaModel.Children.ModelicaBusPortInterfaceCollection) {
                AVM.META.BusPortInterface avmBusPortInterface = createAVMBusPortInterface(cyPhyMLModelicaBusPortInterface);
                avmBehaviorModel.Interfaces.AddItem(avmBusPortInterface);
            }

            foreach (CyPhyML.ModelicaAggregateInterface cyPhyMLModelicaAggregateInterface in cyPhyMLModelicaModel.Children.ModelicaAggregateInterfaceCollection) {
                AVM.META.AggregateInterface avmAggregateInterface = createAVMAggregateInterface( cyPhyMLModelicaAggregateInterface );
                avmBehaviorModel.Interfaces.AddItem( avmAggregateInterface );
            }

            foreach (CyPhyML.ModelicaMaterialSpec cyPhyMLModelicaMaterialSpec in cyPhyMLModelicaModel.Children.ModelicaMaterialSpecCollection) {
                AVM.META.MaterialSpec avmMaterialSpec = createAVMMaterialSpec( cyPhyMLModelicaMaterialSpec );
                avmBehaviorModel.MaterialSpecs.AddItem( avmMaterialSpec );
            }

            foreach (CyPhyML.LimitCheck cyPhyMLLimitCheck in cyPhyMLModelicaModel.Children.LimitCheckCollection)
            {
                AVM.META.LimitCheck avmLimitCheck = new AVM.META.LimitCheck();
                avmLimitCheck.LimitType = CyPhyLimitType2AVMLimitType[cyPhyMLLimitCheck.Attributes.LimitType];
                avmLimitCheck.Value = cyPhyMLLimitCheck.Attributes.Value;
                avmLimitCheck.VariableName = cyPhyMLLimitCheck.Attributes.VariableName;
                avmBehaviorModel.LimitChecks.AddItem(avmLimitCheck);
            }

        }

        public AVM.META.PowerPortInterface createAVMBehaviorModelPowerPort( CyPhyML.ModelicaPowerPortBase cyPhyMLModelicaPowerPortBase ) {
            AVM.META.PowerPortInterface avmPowerPortInterface = new AVM.META.PowerPortInterface();

            _cyPhyMLAVMAssociableMap[ cyPhyMLModelicaPowerPortBase ] = avmPowerPortInterface;

            avmPowerPortInterface.Name = cyPhyMLModelicaPowerPortBase.Name;
            avmPowerPortInterface.PowerPortType = CyPhyModelicaPP2BehaviorMdlPPT[cyPhyMLModelicaPowerPortBase.GetType().Name];
            String id = cyPhyMLModelicaPowerPortBase.Attributes.ID;
            if (id == "") {
                avmPowerPortInterface.id = cyPhyMLModelicaPowerPortBase.Impl.GetGuidDisp();
                cyPhyMLModelicaPowerPortBase.Attributes.ID = avmPowerPortInterface.id;
            } else {
                avmPowerPortInterface.id = cyPhyMLModelicaPowerPortBase.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLModelicaPowerPortBase.Attributes.ID] = cyPhyMLModelicaPowerPortBase;
                
            if( cyPhyMLModelicaPowerPortBase.AllDstConnections.Count() != 0 ) {
                CyPhyML.ModelicaPowerPortMapBase cyPhyMLModelicaPowerPortMapBase = cyPhyMLModelicaPowerPortBase.AllDstConnections.First() as CyPhyML.ModelicaPowerPortMapBase;

                CyPhyML.PowerPortType cyPhyMLPowerPortType = cyPhyMLModelicaPowerPortMapBase.DstEnd as CyPhyML.PowerPortType;
                createAVMAssociation(avmPowerPortInterface, cyPhyMLPowerPortType);
            }

            return avmPowerPortInterface;
        }

        public AVM.META.SignalPortInterface createAVMBehaviorModelSignalPort( CyPhyML.ModelicaSignalPort cyPhyMLModelicaSignalPort ) {
            AVM.META.SignalPortInterface avmSignalPortInterface = new AVM.META.SignalPortInterface();

            _cyPhyMLAVMAssociableMap[ cyPhyMLModelicaSignalPort ] = avmSignalPortInterface;

            avmSignalPortInterface.Name = cyPhyMLModelicaSignalPort.Name;
            string cyPhyMLModelicaSignalPortTypeName = cyPhyMLModelicaSignalPort.GetType().Name;
            avmSignalPortInterface.SignalPortType = CyPhyModelicaSigPort2SigPort[cyPhyMLModelicaSignalPortTypeName];
            avmSignalPortInterface.Directionality = cyPhyMLModelicaSignalPortTypeName.EndsWith("Input") ? AVM.META.DirectionEnum.Input : AVM.META.DirectionEnum.Output;
            String id = cyPhyMLModelicaSignalPort.Attributes.ID;
            if (id == "") {
                avmSignalPortInterface.id = cyPhyMLModelicaSignalPort.Impl.GetGuidDisp();
                cyPhyMLModelicaSignalPort.Attributes.ID = avmSignalPortInterface.id;
            } else {
                avmSignalPortInterface.id = cyPhyMLModelicaSignalPort.Attributes.ID;
            }
            _idCyPhyMLObjectMap[ cyPhyMLModelicaSignalPort.Attributes.ID ] = cyPhyMLModelicaSignalPort;

            if( cyPhyMLModelicaSignalPort.AllDstConnections.Count() != 0 ) {

                CyPhyML.SignalPortType cyPhyMLSignalPortType = null;
                CyPhyML.SignalPortMap cyPhyMLSignalPortMap = cyPhyMLModelicaSignalPort.AllDstConnections.First() as CyPhyML.SignalPortMap;
                cyPhyMLSignalPortType = cyPhyMLSignalPortMap.DstEnd as CyPhyML.SignalPortType;
                createAVMAssociation(avmSignalPortInterface, cyPhyMLSignalPortType);
            }

            if (cyPhyMLModelicaSignalPort.AllSrcConnections.Count() != 0) {

                CyPhyML.SignalPortType cyPhyMLSignalPortType = null;
                CyPhyML.SignalPortMap cyPhyMLSignalPortMap = cyPhyMLModelicaSignalPort.AllSrcConnections.First() as CyPhyML.SignalPortMap;
                cyPhyMLSignalPortType = cyPhyMLSignalPortMap.SrcEnd as CyPhyML.SignalPortType;
                createAVMAssociation(cyPhyMLSignalPortType, avmSignalPortInterface);
            }

            return avmSignalPortInterface;
        }

        public AVM.META.BehaviorParameter createAVMBehaviorModelParameter( CyPhyML.ModelicaParameter cyPhyMLModelicaParameter ) {
            AVM.META.BehaviorParameter avmBehaviorParameter = new AVM.META.BehaviorParameter();

            _cyPhyMLAVMAssociableMap[ cyPhyMLModelicaParameter ] = avmBehaviorParameter;

            avmBehaviorParameter.Name = cyPhyMLModelicaParameter.Name;
            String id = cyPhyMLModelicaParameter.Attributes.ID;
            if (id == "") {
                avmBehaviorParameter.id = cyPhyMLModelicaParameter.Impl.GetGuidDisp();
                cyPhyMLModelicaParameter.Attributes.ID = avmBehaviorParameter.id;
            } else {
                avmBehaviorParameter.id = cyPhyMLModelicaParameter.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLModelicaParameter.Attributes.ID] = cyPhyMLModelicaParameter;

            CyPhyML.ValueFlowTarget cyPhyMLValueFlowTarget = null;
            int sourceCount = cyPhyMLModelicaParameter.AllSrcConnections.Count();
            if( sourceCount != 0 ) {
                CyPhyML.ModelicaParameterPortMap cyPhyMLModelicaParameterPortMap = cyPhyMLModelicaParameter.AllSrcConnections.First() as CyPhyML.ModelicaParameterPortMap;
                cyPhyMLValueFlowTarget = cyPhyMLModelicaParameterPortMap.SrcEnd as CyPhyML.ValueFlowTarget;
            }

            if (cyPhyMLModelicaParameter.Attributes.Value != "") {
                avmBehaviorParameter.Value = cyPhyMLModelicaParameter.Attributes.Value;
            }

            if (cyPhyMLModelicaParameter.Attributes.DefaultValue != "")
                avmBehaviorParameter.DefaultValue = cyPhyMLModelicaParameter.Attributes.DefaultValue;

            createAVMAssociation(cyPhyMLValueFlowTarget, avmBehaviorParameter);

            return avmBehaviorParameter;
        }

        public AVM.META.BusPortInterface createAVMBusPortInterface(CyPhyML.ModelicaBusPortInterface cyPhyMLModelicaBusPortInterface) {
            AVM.META.BusPortInterface avmBusPortInterface = new AVM.META.BusPortInterface();

            _cyPhyMLAVMAssociableMap[cyPhyMLModelicaBusPortInterface] = avmBusPortInterface;

            avmBusPortInterface.Name = cyPhyMLModelicaBusPortInterface.Name;
            avmBusPortInterface.InterfaceType = cyPhyMLModelicaBusPortInterface.Attributes.Type;

            String id = cyPhyMLModelicaBusPortInterface.Attributes.ID;
            if (id == "") {
                avmBusPortInterface.id = cyPhyMLModelicaBusPortInterface.Impl.GetGuidDisp();
                cyPhyMLModelicaBusPortInterface.Attributes.ID = avmBusPortInterface.id;
            } else {
                avmBusPortInterface.id = cyPhyMLModelicaBusPortInterface.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLModelicaBusPortInterface.Attributes.ID] = cyPhyMLModelicaBusPortInterface;

            CyPhyML.BusPort cyPhyMLBusPort = null;
            int sourceCount = cyPhyMLModelicaBusPortInterface.AllDstConnections.Count();
            if (sourceCount != 0) {
                CyPhyML.ModelicaBusPortMap cyPhyMLModelicaBusPortMap = cyPhyMLModelicaBusPortInterface.AllDstConnections.First() as CyPhyML.ModelicaBusPortMap;
                cyPhyMLBusPort = cyPhyMLModelicaBusPortMap.DstEnd as CyPhyML.BusPort;
            }

            createAVMAssociation(avmBusPortInterface, cyPhyMLBusPort);

            return avmBusPortInterface;
        }

        public AVM.META.AggregateInterface createAVMAggregateInterface(CyPhyML.ModelicaAggregateInterface cyPhyMLModelicaAggregateInterface) {
            AVM.META.AggregateInterface avmAggregateInterface = new AVM.META.AggregateInterface();

            _cyPhyMLAVMAssociableMap[ cyPhyMLModelicaAggregateInterface ] = avmAggregateInterface;

            avmAggregateInterface.Name = cyPhyMLModelicaAggregateInterface.Name;
            avmAggregateInterface.InterfaceType = cyPhyMLModelicaAggregateInterface.Attributes.Type;
            String id = cyPhyMLModelicaAggregateInterface.Attributes.ID;
            if (id == "") {
                avmAggregateInterface.id = cyPhyMLModelicaAggregateInterface.Impl.GetGuidDisp();
                cyPhyMLModelicaAggregateInterface.Attributes.ID = avmAggregateInterface.id;
            } else {
                avmAggregateInterface.id = cyPhyMLModelicaAggregateInterface.Attributes.ID;
            }
            _idCyPhyMLObjectMap[cyPhyMLModelicaAggregateInterface.Attributes.ID] = cyPhyMLModelicaAggregateInterface;

            foreach( CyPhyML.ModelicaPowerPortBase cyPhyMLModelicaPowerPortBase in cyPhyMLModelicaAggregateInterface.Children.ModelicaPowerPortBaseCollection ) {
                AVM.META.PowerPortInterface avmPowerPortInterface = createAVMBehaviorModelPowerPort( cyPhyMLModelicaPowerPortBase );
                avmAggregateInterface.AggregatedInterfaces.AddItem( avmPowerPortInterface );
            }

            foreach( CyPhyML.ModelicaSignalPort cyPhyMLModelicaSignalPort in cyPhyMLModelicaAggregateInterface.Children.ModelicaSignalPortCollection ) {
                AVM.META.SignalPortInterface avmSignalPortInterface = createAVMBehaviorModelSignalPort( cyPhyMLModelicaSignalPort );
                avmAggregateInterface.AggregatedInterfaces.AddItem( avmSignalPortInterface );
            }

            foreach( CyPhyML.ModelicaParameter cyPhyMLModelicaParameter in cyPhyMLModelicaAggregateInterface.Children.ModelicaParameterCollection ) {
                AVM.META.BehaviorParameter avmBehaviorParameter = createAVMBehaviorModelParameter( cyPhyMLModelicaParameter );
                avmAggregateInterface.AggregatedInterfaces.AddItem( avmBehaviorParameter );
            }

            foreach (CyPhyML.ModelicaBusPortInterface cyPhyMLModelicaBusPortInterface in cyPhyMLModelicaAggregateInterface.Children.ModelicaBusPortInterfaceCollection) {
                AVM.META.BusPortInterface avmBusPortInterface = createAVMBusPortInterface(cyPhyMLModelicaBusPortInterface);
                avmAggregateInterface.AggregatedInterfaces.AddItem(avmBusPortInterface);
            }

            foreach (CyPhyML.ModelicaAggregateInterface cyPhyMLModelicaAggregateInterfaceChild in cyPhyMLModelicaAggregateInterface.Children.ModelicaAggregateInterfaceCollection) {
                AVM.META.AggregateInterface avmAggregateInterfaceChild = createAVMAggregateInterface( cyPhyMLModelicaAggregateInterfaceChild );
                avmAggregateInterface.AggregatedInterfaces.AddItem( avmAggregateInterfaceChild );
            }

            if( cyPhyMLModelicaAggregateInterface.SrcConnections.AggregatePortMapCollection.Count() != 0 ) {
                CyPhyML.AggregatePortMap cyPhyMLAggregatePortMap = cyPhyMLModelicaAggregateInterface.SrcConnections.AggregatePortMapCollection.First();
                CyPhyML.AggregatePort cyPhyMLAggregatePort = cyPhyMLAggregatePortMap.SrcEnd as CyPhyML.AggregatePort;
                createAVMAssociation(cyPhyMLAggregatePort, avmAggregateInterface);
            }

            if (cyPhyMLModelicaAggregateInterface.DstConnections.AggregatePortMapCollection.Count() != 0) {
                CyPhyML.AggregatePortMap cyPhyMLAggregatePortMap = cyPhyMLModelicaAggregateInterface.DstConnections.AggregatePortMapCollection.First();
                CyPhyML.AggregatePort cyPhyMLAggregatePort = cyPhyMLAggregatePortMap.DstEnd as CyPhyML.AggregatePort;
                createAVMAssociation(avmAggregateInterface, cyPhyMLAggregatePort);
            }

            return avmAggregateInterface;
        }

        public AVM.META.MaterialSpec createAVMMaterialSpec( CyPhyML.ModelicaMaterialSpec cyPhyMLModelicaMaterialSpec ) {
            AVM.META.MaterialSpec avmMaterialSpec = new AVM.META.MaterialSpec();
            avmMaterialSpec.CompatibleMaterials = new List<string>();
            avmMaterialSpec.PowerPorts = new List<AVM.META.PowerPortInterface>();
            avmMaterialSpec.RedeclareParameterName = cyPhyMLModelicaMaterialSpec.Name;
            avmMaterialSpec.RedeclareType = cyPhyMLModelicaMaterialSpec.Attributes.RedeclareType.ToString();

            _cyPhyMLAVMAssociableMap[ cyPhyMLModelicaMaterialSpec ] = avmMaterialSpec;

            foreach( string compatibleMaterial in cyPhyMLModelicaMaterialSpec.Attributes.CompatibleMaterials.Split( '\n' ) ) {
                if( compatibleMaterial != "" ) {
                    avmMaterialSpec.CompatibleMaterials.Add( compatibleMaterial );
                }
            }
            foreach( CyPhyML.MaterialSpecConnection cyPhyMLMaterialSpecConnection in cyPhyMLModelicaMaterialSpec.DstConnections.MaterialSpecConnectionCollection ) {
                avmMaterialSpec.PowerPorts.Add( _cyPhyMLAVMAssociableMap[ cyPhyMLMaterialSpecConnection.DstEnd ] as AVM.META.PowerPortInterface );
            }

            return avmMaterialSpec;
        }

        public AVM.iFAB.ManufacturingModel createAVMManufacturingModel(CyPhyML.ManufacturingModel cyPhyMLManufacturingModel) {

            AVM.iFAB.ManufacturingModel avmManufacturingModel = new AVM.iFAB.ManufacturingModel();

            _cyPhyMLAVMAssociableMap[cyPhyMLManufacturingModel] = avmManufacturingModel;

            avmManufacturingModel.Name = cyPhyMLManufacturingModel.Name;
            avmManufacturingModel.id = cyPhyMLManufacturingModel.Attributes.ID;
            avmManufacturingModel.Location = cyPhyMLManufacturingModel.Attributes.Location;

            avmManufacturingModel.ManufacturingModelParameter = new List<AVM.iFAB.ManufacturingModelParameter>();

            foreach (CyPhyML.ManufacturingModelParameter cyPhyMLManufacturingModelParameter in cyPhyMLManufacturingModel.Children.ManufacturingModelParameterCollection) {
                AVM.iFAB.ManufacturingModelParameter avmManufacturingModelParameter = new AVM.iFAB.ManufacturingModelParameter();
                avmManufacturingModelParameter.Name = cyPhyMLManufacturingModelParameter.Name;
                avmManufacturingModelParameter.id = cyPhyMLManufacturingModelParameter.Attributes.ID;
                avmManufacturingModelParameter.Description = cyPhyMLManufacturingModelParameter.Attributes.Description;
                avmManufacturingModelParameter.DefaultValue = cyPhyMLManufacturingModelParameter.Attributes.DefaultValue;
                avmManufacturingModelParameter.Value = cyPhyMLManufacturingModelParameter.Attributes.Value;
                avmManufacturingModelParameter.Unit = getUnit( cyPhyMLManufacturingModelParameter.Referred.unit );

                foreach (CyPhyML.ManufacturingParameterPortMap cyPhyMLManufacturingParameterPortMap in cyPhyMLManufacturingModelParameter.SrcConnections.ManufacturingParameterPortMapCollection) {
                    createAVMAssociation(cyPhyMLManufacturingParameterPortMap.SrcEnd, avmManufacturingModelParameter);
                }
                avmManufacturingModel.ManufacturingModelParameter.Add(avmManufacturingModelParameter);
            }

            _idCyPhyMLObjectMap[cyPhyMLManufacturingModel.Attributes.ID] = cyPhyMLManufacturingModel;

            return avmManufacturingModel;
        }

        public object processCyPhyMLGenericComplex(CyPhyML.GenericComplex cyPhyMLGenericComplex) {

            string fullClassName = cyPhyMLGenericComplex.Attributes.Type;
            Type type = Type.GetType(fullClassName + ",DesignDataPackageLib");
            if (type == null) {
                Console.Out.WriteLine("WARNING:  Could not get type \"" + fullClassName + "\"");
                return null;
            }
            
            object object_var = Activator.CreateInstance(type);
            _idCyPhyMLObjectMap[cyPhyMLGenericComplex.Attributes.ID] = cyPhyMLGenericComplex;
            _cyPhyMLAVMAssociableMap[cyPhyMLGenericComplex] = object_var;

            int noProperties = 2;
            PropertyInfo[] propertyInfoArray = type.GetProperties();
            foreach (PropertyInfo propertyInfo in propertyInfoArray) {
                if (propertyInfo.Name.ToLower() == "name") {
                    propertyInfo.SetValue(object_var, cyPhyMLGenericComplex.Name, null);
                    --noProperties;
                } else if (propertyInfo.Name.ToLower() == "id") {
                    propertyInfo.SetValue(object_var, cyPhyMLGenericComplex.Attributes.ID, null);
                    --noProperties;
                }
                if (noProperties <= 0) break;
            }

            foreach (CyPhyML.GenericAttribute cyPhyMLGenericAttribute in cyPhyMLGenericComplex.Children.GenericAttributeCollection) {

                string propertyName = cyPhyMLGenericAttribute.Name;
                PropertyInfo propertyInfo = type.GetProperty(propertyName);
                if (propertyInfo == null) {
                    Console.Out.WriteLine("WARNING:  Could not get \"" + propertyName + "\" property for class \"" + fullClassName + "\"");
                    continue;
                }

                string propertyValue = cyPhyMLGenericAttribute.Attributes.Value;
                if (propertyValue != "") {

                    object object_var2 = null;
                    Type systemNullableType = null;
                    Type propertyType = propertyInfo.PropertyType;
                    if (propertyType.IsGenericType && propertyType.GetGenericTypeDefinition() == typeof(System.Nullable<>)) {
                        systemNullableType = propertyType;
                        propertyType = propertyType.GetGenericArguments()[0];
                    }

                    if (propertyValue == "") {
                        // DO NOTHING
                    } else if (propertyType.Name.ToLower() == "string") {
                        object_var2 = propertyValue;
                    } else if (propertyType.IsEnum) {
                        object_var2 = Enum.Parse(propertyType, propertyValue);
                    } else {
                        object_var2 = double.Parse(propertyValue);
                    }

                    if (systemNullableType != null) {
                        Type[] typeArray = new Type[1];
                        typeArray[0] = propertyType;
                        ConstructorInfo constructorInfo = systemNullableType.GetConstructor(typeArray);
                        object[] objectArray = new object[1];
                        objectArray[0] = object_var2;
                        object_var2 = constructorInfo.Invoke(objectArray);
                    }

                    propertyInfo.SetValue(object_var, Convert.ChangeType( object_var2, propertyInfo.PropertyType ), null);
                }
            }

            IEnumerable< CyPhyML.GenericObject > cyPhyMLGenericObjectIEnumerable = cyPhyMLGenericComplex is CyPhyML.GenericFeature ?
             (cyPhyMLGenericComplex as CyPhyML.GenericFeature).Children.GenericObjectCollection : (cyPhyMLGenericComplex as CyPhyML.GenericObject).Children.GenericObjectCollection;

            foreach( CyPhyML.GenericObject cyPhyMLGenericObject in cyPhyMLGenericObjectIEnumerable ) {

                string propertyName = cyPhyMLGenericObject.Name;
                PropertyInfo propertyInfo = type.GetProperty(propertyName);
                if (propertyInfo == null) {
                    Console.Out.WriteLine("WARNING:  Could not get \"" + propertyName + "\" object property for class \"" + fullClassName + "\"");
                    continue;
                }
                propertyInfo.SetValue( object_var, processCyPhyMLGenericComplex( cyPhyMLGenericObject ), null );
            }

            IEnumerable<CyPhyML.GenericArray> cyPhyMLGenericArrayIEnumerable = cyPhyMLGenericComplex is CyPhyML.GenericFeature ?
             (cyPhyMLGenericComplex as CyPhyML.GenericFeature).Children.GenericArrayCollection : (cyPhyMLGenericComplex as CyPhyML.GenericObject).Children.GenericArrayCollection;

            foreach (CyPhyML.GenericArray cyPhyMLGenericArray in cyPhyMLGenericArrayIEnumerable) {

                string propertyName = cyPhyMLGenericArray.Name;
                PropertyInfo propertyInfo = type.GetProperty(propertyName);
                if (propertyInfo == null) {
                    Console.Out.WriteLine("WARNING:  Could not get \"" + propertyName + "\" array property for class \"" + fullClassName + "\"");
                    continue;
                }

                Type propertyType = propertyInfo.PropertyType;
                IList propertyValueIList = Activator.CreateInstance(propertyType) as IList;

                propertyInfo.SetValue(object_var, propertyValueIList, null);

                foreach (object object_var2 in cyPhyMLGenericArray.Children.GenericObjectCollection) {
                    object object_var3 = processCyPhyMLGenericComplex(object_var2 as CyPhyML.GenericComplex);
                    propertyValueIList.Add(object_var3);
                }
            }

            return object_var;
        }

        public void processCyPhyMLGenericAssociation(CyPhyML.GenericAssociation cyPhyMLGenericAssociation) {
            string srcAssociableId = cyPhyMLGenericAssociation.Attributes.SrcID;
            if (!_idCyPhyMLObjectMap.ContainsKey(srcAssociableId)) {
                Console.Out.WriteLine("WARNING: Generic Association: CyPhyML object with source ID \"" + srcAssociableId + "\" not found.");
                return;
            }
            string dstAssociableId = cyPhyMLGenericAssociation.Attributes.DstID;
            if (!_idCyPhyMLObjectMap.ContainsKey(dstAssociableId)) {
                Console.Out.WriteLine("WARNING: Generic Association: CyPhyML object with destination ID \"" + dstAssociableId + "\" not found.");
                return;
            }

            object srcCyPhyMLObject = _idCyPhyMLObjectMap[srcAssociableId];
            if (!_cyPhyMLAVMAssociableMap.ContainsKey(srcCyPhyMLObject)) {
                Console.Out.WriteLine("WARNING: Generic Association: source CyPhyML object with ID \"" + srcAssociableId + "\" has no corresponding AVM object.");
                return;
            }
            object dstCyPhyMLObject = _idCyPhyMLObjectMap[dstAssociableId];
            if (!_cyPhyMLAVMAssociableMap.ContainsKey(dstCyPhyMLObject)) {
                Console.Out.WriteLine("WARNING: Generic Association: destination CyPhyML object with ID \"" + dstAssociableId + "\" has no corresponding AVM object.");
                return;
            }

            AVM.Associable srcAssociable = _cyPhyMLAVMAssociableMap[srcCyPhyMLObject] as AVM.Associable;
            AVM.Associable dstAssociable = _cyPhyMLAVMAssociableMap[dstCyPhyMLObject] as AVM.Associable;

            createAVMAssociation(srcAssociable, dstAssociable);
        }

        public AVM.Component CyPhyML2AVMNonStatic( CyPhyML.Component cyPhyMLComponent ) {
            setComponentName( cyPhyMLComponent.Name );

            String avmId = cyPhyMLComponent.Attributes.AVMID;
            if (avmId == "") {
                _avmComponent.AVMID = cyPhyMLComponent.Impl.GetGuidDisp();
                cyPhyMLComponent.Attributes.AVMID = _avmComponent.AVMID;
            } else {
                _avmComponent.AVMID = cyPhyMLComponent.Attributes.AVMID;
            }

            _avmComponent.Revision = cyPhyMLComponent.Attributes.Revision;
            _avmComponent.Version = cyPhyMLComponent.Attributes.Version;

            // Set category
            String s_Classifications = cyPhyMLComponent.Attributes.Classifications;
            List<String> ls_Classifications = s_Classifications.Split('\n').ToList();
            _avmComponent.Category = ls_Classifications;

            foreach( CyPhyML.Property cyPhyMLProperty in cyPhyMLComponent.Children.PropertyCollection ) {
                createAVMProperty( cyPhyMLProperty );
            }
            foreach( CyPhyML.Parameter cyPhyMLParameter in cyPhyMLComponent.Children.ParameterCollection ) {
                createAVMParameter( cyPhyMLParameter );
            }
            foreach( CyPhyML.SimpleFormula cyPhyMLSimpleFormula in cyPhyMLComponent.Children.SimpleFormulaCollection ) {
                createAVMCalculation( cyPhyMLSimpleFormula );
            }
            foreach( CyPhyML.CustomFormula cyPhyMLCustomFormula in cyPhyMLComponent.Children.CustomFormulaCollection ) {
                createAVMCalculation( cyPhyMLCustomFormula );
            }
            foreach( CyPhyML.PowerPortType cyPhyMLPowerPortType in cyPhyMLComponent.Children.PowerPortTypeCollection ) {
                AVM.META.PowerPort avmPowerPort = createAVMExternalPowerPort( cyPhyMLPowerPortType );
                if( avmPowerPort != null ) {
                    _avmComponent.Features.AddItem( avmPowerPort );
                }
            }
            foreach (CyPhyML.AnalysisPoint cyPhyMLAnalysisPoint in cyPhyMLComponent.Children.AnalysisPointCollection) {
                AVM.META.CADAnalysisPoint avmCADAnalysisPoint = new AVM.META.CADAnalysisPoint();
                if (avmCADAnalysisPoint != null) {
                    avmCADAnalysisPoint.Name = cyPhyMLAnalysisPoint.Name;
                    avmCADAnalysisPoint.Type = avmCADAnalysisPoint.GetType().ToString();
                    String id = cyPhyMLAnalysisPoint.Attributes.ID;
                    if (id == "") {
                        avmCADAnalysisPoint.id = cyPhyMLAnalysisPoint.Impl.GetGuidDisp();
                        cyPhyMLAnalysisPoint.Attributes.ID = avmCADAnalysisPoint.id;
                    } else {
                        avmCADAnalysisPoint.id = cyPhyMLAnalysisPoint.Attributes.ID;
                    }
                    _idCyPhyMLObjectMap[cyPhyMLAnalysisPoint.Attributes.ID] = cyPhyMLAnalysisPoint;

                    _cyPhyMLAVMAssociableMap[cyPhyMLAnalysisPoint] = avmCADAnalysisPoint;
                    _avmComponent.Features.AddItem(avmCADAnalysisPoint);
                }
            }
            foreach (CyPhyML.SignalPortType cyPhyMLSignalPortType in cyPhyMLComponent.Children.SignalPortTypeCollection) {
                AVM.META.SignalPortBase avmSignalPortBase = createAVMSignalPortBase(cyPhyMLSignalPortType);
                if (avmSignalPortBase != null) {
                    _avmComponent.Features.AddItem(avmSignalPortBase);
                }
            }
            foreach (CyPhyML.StructuralInterface cyPhyMLStructuralInterface in cyPhyMLComponent.Children.StructuralInterfaceCollection) {
                AVM.StructuralInterface avmStructuralInterface = createAVMStructuralInterface( cyPhyMLStructuralInterface );
                _avmComponent.Features.AddItem( avmStructuralInterface );
            }
            foreach( CyPhyML.AggregatePort cyPhyMLAggregatePort in cyPhyMLComponent.Children.AggregatePortCollection ) {
                AVM.META.AggregatePort avmAggregatePort = createAVMAggregatePort( cyPhyMLAggregatePort );
                _avmComponent.Features.AddItem( avmAggregatePort );                
            }

            foreach( CyPhyML.CADModel cyPhyMLCADModel in cyPhyMLComponent.Children.CADModelCollection ) {
                createAVMCADModel( cyPhyMLCADModel );
            }
            foreach( CyPhyML.ModelicaModel cyPhyMLModelicaModel in cyPhyMLComponent.Children.ModelicaModelCollection ) {
                createAVMBehaviorModel( cyPhyMLModelicaModel );
            }
            foreach (CyPhyML.ManufacturingModel cyPhyMLManufacturingModel in cyPhyMLComponent.Children.ManufacturingModelCollection) {
                AVM.iFAB.ManufacturingModel avmManufacturingModel = createAVMManufacturingModel(cyPhyMLManufacturingModel);
                _avmComponent.Features.AddItem(avmManufacturingModel);
            }

            foreach (CyPhyML.File cyPhyMLFile in cyPhyMLComponent.Children.FileCollection)
            {
                AVM.File af = new AVM.File();
                af.Description = cyPhyMLFile.Attributes.Description;
                af.Hash = cyPhyMLFile.Attributes.Hash;
                af.Location = cyPhyMLFile.Attributes.Location;
                _avmComponent.Files.Add(af);
            }

            foreach (CyPhyML.GenericFeature cyPhyMLGenericFeature in cyPhyMLComponent.Children.GenericFeatureCollection) {
                _avmComponent.Features.Add( processCyPhyMLGenericComplex(cyPhyMLGenericFeature) as AVM.Feature );
            }

            foreach (CyPhyML.GenericAssociations cyPhyMLGenericAssociations in cyPhyMLComponent.Children.GenericAssociationsCollection) {
                foreach (CyPhyML.GenericAssociation cyPhyMLGenericAssociation in cyPhyMLGenericAssociations.Children.GenericAssociationCollection) {
                    processCyPhyMLGenericAssociation(cyPhyMLGenericAssociation);
                }
            }

            return _avmComponent;
        }
 
        public static AVM.Component CyPhyML2AVM( CyPhyML.Component cyPhyMLComponent ) {

            AVMComponentBuilder avmComponentBuilder = new AVMComponentBuilder();
            return avmComponentBuilder.CyPhyML2AVMNonStatic( cyPhyMLComponent );
        }
    }
}
