#!/usr/bin/env python
#
# CyPhy2DMD: auto-generate DMD file from CyPhyML (GME) model
#
# Author: Peter Volgyesi <peter.volgyesi@vanderbilt.edu>
#         Zsolt Lattmann <zsolt.lattmann@vanderbilt.edu>
#
# Copyright (c) 2012, Vanderbilt University
# All rights reserved.
#
# Permission to use, copy, modify, and distribute this software and its
# documentation for any purpose, without fee, and without written agreement is
# hereby granted, provided that the above copyright notice, the following
# two paragraphs and the author appear in all copies of this software.
#
# IN NO EVENT SHALL THE VANDERBILT UNIVERSITY BE LIABLE TO ANY PARTY FOR
# DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
# OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE VANDERBILT
# UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# THE VANDERBILT UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
# AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
# ON AN "AS IS" BASIS, AND THE VANDERBILT UNIVERSITY HAS NO OBLIGATION TO
# PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

from optparse import OptionParser
import win32com.client
import sys
import os.path
from tempfile import NamedTemporaryFile
import logging
import json

#
# TODO catch MGA exceptions and report them through logging
# TODO remove temporary file (created by XME import)
#
logging.basicConfig()
L = logging.getLogger("CyPhy2DMD")

def get_fco_guid(fco):
    return fco.GetGuidDisp()[1:-1]   # removing '{' and '}'
        
# The DMD component is a dictionary object (straightforward JSON support)
def create_component(fco):
    c = {}
    c["id"] = get_fco_guid(fco)
    c["name"] = fco.Name

    if (fco.Meta.Name == "ComponentRef"):
        fco = fco.Referred

    
    vflink = None
    vflinks = fco.GetChildrenOfKind("VFLink")
    if len(vflinks) > 0:
        vflink = vflinks[0]
        if len(vflinks) > 1:
            L.warning("Multiple VFLink objects in node: %s", (fco.AbsPath,))
    
    if vflink:
        c["preview"] = vflink.StrAttrByName("Preview")
        c["exchangeId"] = vflink.StrAttrByName("ComponentGUID")
        catgeroy_str = vflink.StrAttrByName("RevisionGUID")
        if catgeroy_str:
            category = catgeroy_str.split("%")
            if len(category) != 2:
                L.warning("Inproper category string (%s) in VFLink object: %s", (catgeroy_str, vflink.AbsPath))
            else:
                c["category"] = {'id': category[0], 'label': category[1]}
    
    
    if fco.Meta.Name == "DesignContainer":
        c["type"] = fco.StrAttrByName("ContainerType")
    elif (fco.Meta.Name == "Component") or (fco.Meta.Name == "ComponentAssembly"):
        if c.get("exchangeId", None):
            c["type"] = "ExchangeComponent"
        elif fco.StrAttrByName('AVMID'):
            c["type"] = "ExchangeComponent"
        else:
            c["type"] = "LocalComponent"
    else:
        c["type"] = fco.Meta.Name
    
    if fco.Meta.Name == "Component":
        c["AVMID"] = fco.StrAttrByName('AVMID')
        
    # [Zsolt 7/16/2012] store the meta kind, visualizer uses this attribute
    c["metaKind"] = fco.Meta.Name
    c["metadata"] = metadata = {}
    for prop in fco.GetChildrenOfKind("Property"):
        value = prop.StrAttrByName("Value")
        if not value:
            value = "N/A"
        unit = ""
        if prop.Referred:
            unit = prop.Referred.Name
        metadata[prop.Name] = {"value" : value, "unit" : unit}

    # [Zsolt 7/16/2012] include parameters in the metadata
    for prop in fco.GetChildrenOfKind("Parameter"):
        value = prop.StrAttrByName("Value")
        if not value:
            value = "N/A"
        unit = ""
        if prop.Referred:
            unit = prop.Referred.Name
        metadata[prop.Name] = {"value" : value, "unit" : unit}

    c["artifacts"] = artifacts = {}
    for cadfilelink in fco.GetChildrenOfKind("CADFileLink"):
        artifacts[cadfilelink.Name] = cadfilelink.StrAttrByName("FileName")
        
    return c



def process_cmdline():
    usage = "usage: %prog [options] <input-file>"
    parser = OptionParser(usage)
    parser.add_option("-o", "--output", dest="filename",
                      help="write output DMD data to FILENAME")
    parser.add_option("-v", "--verbose",
                      action="store_true", dest="verbose")
    parser.add_option("-q", "--quiet",
                      action="store_false", dest="verbose")
    parser.add_option("-c", "--compact",
                      action="store_true", dest="compact",
                      help="use compact JSON encoding")
                      
    (options, args) = parser.parse_args()
    
    if len(args) != 1:
        parser.error("incorrect number of arguments")
    
    return (options, args)
    

def open_project(ifpath):
    L.info("Initializing MGA Project")
    project = win32com.client.Dispatch("Mga.MgaProject")

    (ifpath_root, ifpath_ext) = os.path.splitext(ifpath)    
    iftype = ifpath_ext.upper()[1:]
    
    if iftype == "XME":
        L.info("Initializing MGA Parser")
        parser = win32com.client.Dispatch("Mga.MgaParser")
        (paradigm, parversion, parguid, xmebasename, xmeversion) = parser.GetXMLInfo(ifpath)
        if paradigm != "CyPhyML":
            L.error("Not a valid CyPhy model: %s (%s)" % (ifpath, paradigm))
            sys.exit(-1)
        
        tmpfile = NamedTemporaryFile(delete=False)
        tmpfile.close()
        
        ifconn = "MGA=" + tmpfile.name
        L.info("Creating model file: %s" % (ifconn,))
        project.Create(ifconn, paradigm);
        
        L.info("Parsing model file: %s -> %s" % (ifpath, ifconn))
        parser.ParseProject(project, ifpath);
        
    elif iftype == "MGA":
        ifconn = "MGA=" + ifpath
        L.info("Loading model file: %s" % (ifconn,))
        project.Open(ifconn)
        
    else:
        L.error("Unable to recognize model file type: %s" % (ifpath,))
        sys.exit(-1)
        
    return project


def collect_top_level(folder):
    top_levels = []
    
    for child_FCO in folder.ChildFCOs:
        if child_FCO.Meta.Name == "DesignContainer":    
            top_levels.append(child_FCO)
            
    for child_folder in folder.ChildFolders:
        top_levels.extend(collect_top_level(child_folder))
    
    return top_levels


def harvest_one(fco, elements, embeds):
    component = create_component(fco)
    elements[component["id"]] = component
    subfcos = []
    
    if (fco.ObjType == 1): #OBJTYPE_MODEL = 1    
        for child_FCO in fco.ChildFCOs:
            if child_FCO.Meta.Name == "ComponentRef":
                if child_FCO.Referred:
                    subfcos.append(child_FCO)
            elif child_FCO.Meta.Name in ["Component", "ComponentAssembly", "DesignContainer"]:
                subfcos.append(child_FCO)
                
    for subfco in subfcos:
        harvest_one(subfco, elements, embeds)
        embeds.append([get_fco_guid(fco), get_fco_guid(subfco)])

def harvest_all(project):
    root = {}
    root["focusedByDefault"] = focusedByDefault = []
    root["elements"] = elements = {}
    root["relations"] = relations = {}
    relations["embeds"] = embeds = []
    
    project.BeginTransaction(None, 1)   
    L.info("Collecting top-level design containers")
    top_levels = collect_top_level(project.RootFolder)
    for top_level in top_levels:
        focusedByDefault.append(get_fco_guid(top_level))
        L.info("Processing top-level design container: %s", (top_level.AbsPath,))
        harvest_one(top_level, elements, embeds)
    project.CommitTransaction()
    
    return root

def main():
    (options, args) = process_cmdline()
    if options.verbose:
        L.setLevel(logging.INFO)
    else:
        L.setLevel(logging.ERROR)
    
    ifpath = os.path.abspath(args[0])    
    if not os.path.isfile(ifpath):
        L.error("File not found: %s" % (ifpath,))
        sys.exit(-1)

    project = open_project(ifpath)
    root = harvest_all(project)
    L.info("Closing model file")
    project.Close()

    L.info("Generating JSON data")
    if options.compact:
        print json.dumps(root, separators=(',',':'))
    else:
        print json.dumps(root, sort_keys=True, indent=4)

if __name__ == '__main__':
    main()
