/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
extern alias run_esmol;

using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;
using System.Linq;
using GME.MGA.Core;
using GME.CSharp;
using GME;
using GME.MGA;
using SmartDataNetwork = Udm.Native.SmartDataNetwork;
using CyPhyMLImpl = Udm.Impl.CyPhyML;
using ESMoLImpl = Udm.Impl.ESMoL;

namespace CyPhy2ESMoL
{
    /// <summary>
    /// This class implements the necessary COM interfaces for a GME interpreter component.
    /// </summary>
    [Guid(ComponentConfig.guid),
    ProgId(ComponentConfig.progID),
    ClassInterface(ClassInterfaceType.AutoDual)]
    [ComVisible(true)]
    public class CyPhy2ESMoLInterpreter : IMgaComponentEx, IGMEVersionInfo
    {
        public CyPhy2ESMoLInterpreter()
        {
            componentParameters = new SortedDictionary<string, object>();
        }

        /// <summary>
        /// Contains information about the GUI event that initiated the invocation.
        /// </summary>
        public enum ComponentStartMode
        {
            GME_MAIN_START = 0, 		// Not used by GME
            GME_BROWSER_START = 1,      // Right click in the GME Tree Browser window
            GME_CONTEXT_START = 2,		// Using the context menu by right clicking a model element in the GME modeling window
            GME_EMBEDDED_START = 3,		// Not used by GME
            GME_MENU_START = 16,		// Clicking on the toolbar icon, or using the main menu
            GME_BGCONTEXT_START = 18,	// Using the context menu by right clicking the background of the GME modeling window
            GME_ICON_START = 32,		// Not used by GME
            GME_SILENT_MODE = 128 		// Not used by GME, available to testers not using GME
        }

        /// <summary>
        /// This function is called for each interpreter invocation before Main.
        /// Don't perform MGA operations here unless you open a tansaction.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        public void Initialize(MgaProject project)
        {
            // TODO: Add your initialization code here...            
        }

        /// <summary>
        /// The main entry point of the interpreter. A transaction is already open,
        /// GMEConsole is avaliable. A general try-catch block catches all the exceptions
        /// coming from this function, you don't need to add it. For more information, see InvokeEx.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        /// <param name="currentobj">The model open in the active tab in GME. Its value is null if no model is open (no GME modeling windows open). </param>
        /// <param name="selectedobjs">
        /// A collection for the selected  model elements. It is never null.
        /// If the interpreter is invoked by the context menu of the GME Tree Browser, then the selected items in the tree browser. Folders
        /// are never passed (they are not FCOs).
        /// If the interpreter is invoked by clicking on the toolbar icon or the context menu of the modeling window, then the selected items 
        /// in the active GME modeling window. If nothing is selected, the collection is empty (contains zero elements).
        /// </param>
        /// <param name="startMode">Contains information about the GUI event that initiated the invocation.</param>
        public void Main(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, ComponentStartMode startMode)
        {
            if (currentobj == null)
            {
                GMEConsole.Error.WriteLine("Must start interpreter from a Component Assembly");
                return;
            }
            string VCP_PATH = Environment.GetEnvironmentVariable("VCP_PATH");
            if (VCP_PATH == null || VCP_PATH == "")
            {
                throw new ApplicationException("Environment variable VCP_PATH doesn't exist. Please install the ESMoL toolchain and restart GME.");
            }
            if (!Directory.Exists(VCP_PATH))
            {
                throw new ApplicationException("VCP_PATH='" + VCP_PATH + "' doesn't exist. Please install the ESMoL toolchain and restart GME.");
            }
            Udm.Native.Udm.Initialize();

            SmartDataNetwork cyPhyMeta = new SmartDataNetwork(Udm.Native.Uml.Uml.diagram);
            cyPhyMeta.OpenExistingFromString(new StreamReader(
                typeof(ESMoL.IComponent).Assembly.GetManifestResourceStream("CyPhyMLUdmCli.Resources.CyPhyML_udm.xml")).ReadToEnd(), "");

            // or meta.OpenExisting(meta_udm_xml);
            Udm.Impl.CyPhyMLInitialize.Initialize(cyPhyMeta);

            Udm.Native.UdmGme.GmeDN_Wrapper wrapper = Udm.Native.UdmGme.GmeDN_Wrapper.Create(Udm.Native.Uml.Diagram.Cast(cyPhyMeta.GetRootObject()), project, true);

            string outdir;
            if (componentParameters.ContainsKey("output_dir"))
            {
                outdir = (string) componentParameters["output_dir"];
                System.IO.Directory.CreateDirectory(outdir);
            }
            else if (project.ProjectConnStr.StartsWith("MGA=", StringComparison.OrdinalIgnoreCase))
            {
                outdir =
                    System.IO.Path.Combine(
                    System.IO.Path.GetDirectoryName(project.ProjectConnStr.Remove(0, 4)),
                    "generated");
                System.IO.Directory.CreateDirectory(outdir);
            }
            else
            {
                outdir = ".";
            }
            string outfile = System.IO.Path.Combine(outdir, currentobj.Name.Replace(" ", "_") + "_ESMoL.mga");
            GMEConsole.Out.WriteLine("Input model is " + CyPhy2ESMoLLogic.GetLink(new CyPhyMLImpl.ComponentAssembly(wrapper.Gme2Udm(currentobj))));
            GMEConsole.Out.WriteLine("ESMoL output file is <a href=\"" + outfile + "\">" + outfile + "</a>");
            SmartDataNetwork esmolMeta = new SmartDataNetwork(Udm.Native.Uml.Uml.diagram);
            esmolMeta.OpenExistingFromString(new StreamReader(
                typeof(ESMoL.IComponent).Assembly.GetManifestResourceStream("CyPhyMLUdmCli.Resources.ESMoL_udm.xml")).ReadToEnd(), "");
            Udm.Impl.ESMoLInitialize.Initialize(esmolMeta);

            SmartDataNetwork esmolOutput = new SmartDataNetwork(esmolMeta);
            Udm.Native.Uml.Class esmolRootFolder = esmolMeta.GetAllInstancesOf(Udm.Native.Uml.Class.meta)
                .Select(Udm.Native.Uml.Class.Cast)
                .First(x => x.name().Get() == "RootFolder");
            //esmolOutput.CreateNew(outfile, @"C:\Users\ksmyth\Documents\META\externals\esmol\ESMoL_M.xsd", esmolRootFolder);
            esmolOutput.CreateNew(outfile, @"ESMoL", esmolRootFolder);
            ESMoL.IRootFolder rf = ESMoLImpl.RootFolder.Cast(esmolOutput.GetRootObject());
            rf.name = project.Name;
            ESMoL.IDesignFolder df = ESMoLImpl.DesignFolder.Create(rf);

            df.name = project.Name;

            /*foreach (CyPhyML.IComponentAssembly assembly in 
                wrapper.dn.GetAllInstancesOf(CyPhyMLImpl.ComponentAssembly.meta)
                .Select(CyPhyMLImpl.ComponentAssembly.Cast)
                // TODO: use currentobj instead
                .Where(x=>x.name=="EDCDeployment"))
            */
            {
                Udm.Native.UdmObject current = wrapper.Gme2Udm(currentobj);
                CyPhyML.IComponentAssembly assembly = new CyPhyMLImpl.ComponentAssembly(current);

                new CyPhy2ESMoLLogic(GMEConsole, wrapper.dn, esmolOutput, esmolMeta).CyPhy2ESMoL(rf, df, assembly, outdir, outfile, componentParameters);
            }



            // CyPhyML.IComponentAssembly => ESMoL.ISystem
            // CyPhyML.IControlFunction => ESMoL.IComponent

            esmolOutput.CloseWithUpdate();

            //CyPhyML.IComponentAssembly ass = CyPhyMLImpl.ComponentAssembly.Cast(wrapper.Gme2Udm(currentobj));
            //GMEConsole.Out.WriteLine(ass.name);
            wrapper.dn.CloseWithUpdate();
            GC.KeepAlive(wrapper);
            cyPhyMeta.CloseNoUpdate();
            GC.KeepAlive(cyPhyMeta);

            IMgaProject esmolProject = new MgaProject();
            bool esmolProjectRO;
            esmolProject.Open("MGA=" + outfile, out esmolProjectRO);
            IMgaComponentEx run_esmol_interpreter = new run_esmol::ESMoL_toolchain.Run_ESMoL_toolchain();
            run_esmol_interpreter.InvokeEx((MgaProject)esmolProject, null, null, 128);
        }


        #region IMgaComponentEx Members

        MgaGateway MgaGateway { get; set; }
        GMEConsole GMEConsole { get; set; }

        public void InvokeEx(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, int param)
        {
            if (!enabled)
            {
                return;
            }

            try
            {
                try
                {
                    GMEConsole = GMEConsole.CreateFromProject(project);
                    MgaGateway = new MgaGateway(project);
                    project.CreateTerritoryWithoutSink(out MgaGateway.territory);

                    object createTransaction;
                    componentParameters.TryGetValue("create_transaction", out createTransaction);
                    if (createTransaction == null)
                        createTransaction = true;
                    if ((bool)createTransaction)
                        MgaGateway.BeginTransaction(transactiontype_enum.TRANSACTION_GENERAL);
                    try
                    {
                        Main(project, currentobj, selectedobjs, Convert(param));
                    }
                    finally
                    {
                        if ((bool)createTransaction)
                            MgaGateway.AbortTransaction();
                    }
                }
                finally
                {
                    if (MgaGateway.territory != null)
                    {
                        MgaGateway.territory.Destroy();
                    }
                    MgaGateway = null;
                    project = null;
                    currentobj = null;
                    selectedobjs = null;
                    GMEConsole = null;
                    GC.Collect();
                    GC.WaitForPendingFinalizers();
                    GC.Collect();
                    GC.WaitForPendingFinalizers();
                }
            }
            catch (Exception e)
            {
                throw new Exception(e.ToString());
            }
        }

        private ComponentStartMode Convert(int param)
        {
            switch (param)
            {
                case (int)ComponentStartMode.GME_BGCONTEXT_START:
                    return ComponentStartMode.GME_BGCONTEXT_START;
                case (int)ComponentStartMode.GME_BROWSER_START:
                    return ComponentStartMode.GME_BROWSER_START;

                case (int)ComponentStartMode.GME_CONTEXT_START:
                    return ComponentStartMode.GME_CONTEXT_START;

                case (int)ComponentStartMode.GME_EMBEDDED_START:
                    return ComponentStartMode.GME_EMBEDDED_START;

                case (int)ComponentStartMode.GME_ICON_START:
                    return ComponentStartMode.GME_ICON_START;

                case (int)ComponentStartMode.GME_MAIN_START:
                    return ComponentStartMode.GME_MAIN_START;

                case (int)ComponentStartMode.GME_MENU_START:
                    return ComponentStartMode.GME_MENU_START;
                case (int)ComponentStartMode.GME_SILENT_MODE:
                    return ComponentStartMode.GME_SILENT_MODE;
            }

            return ComponentStartMode.GME_SILENT_MODE;
        }

        #region Component Information
        public string ComponentName
        {
            get { return GetType().Name; }
        }

        public string ComponentProgID
        {
            get
            {
                return ComponentConfig.progID;
            }
        }

        public componenttype_enum ComponentType
        {
            get { return ComponentConfig.componentType; }
        }
        public string Paradigm
        {
            get { return ComponentConfig.paradigmName; }
        }
        #endregion

        #region Enabling
        bool enabled = true;
        public void Enable(bool newval)
        {
            enabled = newval;
        }
        #endregion

        #region Interactive Mode
        protected bool interactiveMode = true;
        public bool InteractiveMode
        {
            get
            {
                return interactiveMode;
            }
            set
            {
                interactiveMode = value;
            }
        }
        #endregion

        #region Custom Parameters
        SortedDictionary<string, object> componentParameters = null;

        public object get_ComponentParameter(string Name)
        {
            if (Name == "type")
                return "csharp";

            if (Name == "path")
                return GetType().Assembly.Location;

            if (Name == "fullname")
                return GetType().FullName;

            object value;
            if (componentParameters != null && componentParameters.TryGetValue(Name, out value))
            {
                return value;
            }

            return null;
        }

        public void set_ComponentParameter(string Name, object pVal)
        {
            componentParameters[Name] = pVal;
        }
        #endregion

        #region Unused Methods
        // Old interface, it is never called for MgaComponentEx interfaces
        public void Invoke(MgaProject Project, MgaFCOs selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        // Not used by GME
        public void ObjectsInvokeEx(MgaProject Project, MgaObject currentobj, MgaObjects selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        #endregion

        #endregion

        #region IMgaVersionInfo Members

        public GMEInterfaceVersion_enum version
        {
            get { return GMEInterfaceVersion_enum.GMEInterfaceVersion_Current; }
        }

        #endregion

        #region Registration Helpers

        [ComRegisterFunctionAttribute]
        public static void GMERegister(Type t)
        {
            Registrar.RegisterComponentsInGMERegistry();

        }

        [ComUnregisterFunctionAttribute]
        public static void GMEUnRegister(Type t)
        {
            Registrar.UnregisterComponentsInGMERegistry();
        }

        #endregion


    }

    public static class UdmLinq
    {
        public static IEnumerable<TResult> GetUdmInstances<TSource, TResult>(this IEnumerable<TSource> source, Udm.Native.Uml.Class type, Func<Udm.IUdmObject, TResult> icast)
            where TSource : Udm.IUdmObject
        {
            IEnumerable<TSource> s = source.Where<TSource>(x => x.type.Equals(type));
            return System.Linq.Enumerable.Select<TSource, TResult>(s, x => icast(x));
        }

        public static IEnumerable<TResult> GetUdmInstances<TSource, TResult>(this IEnumerable<TSource> source, Udm.ClassFactory<TResult> factory)
            where TSource : Udm.IUdmObject
            where TResult : Udm.IUdmObject
        {
            return source.GetUdmInstances(factory.meta, factory.ICast);
        }

        public static IEnumerable<TResult> GetUdmInstances2<TSource, TResult>(this IEnumerable<TSource> source, Udm.Native.Uml.Class type, Func<Udm.IUdmObject, TResult> icast)
            where TSource : Udm.IUdmObject
        {
            IEnumerable<TSource> s = source.Where<TSource>(x => Udm.Native.Udm.IsDerivedFrom(x.type, type));
            return System.Linq.Enumerable.Select<TSource, TResult>(s, x => icast(x));
        }

        public static IEnumerable<TResult> GetClosure<TResult>(this IEnumerable<TResult> source,
            Func<TResult, IEnumerable<TResult>> f)
        {
            List<TResult> ret = new List<TResult>();
            foreach (TResult t in source)
            {
                IEnumerable<TResult> res = f(t);
                if (res.Count() > 0)
                {
                    ret.AddRange(res.GetClosure(f));
                }
                else
                {
                    ret.Add(t);
                }
            }
            return ret;
        }

        public static IEnumerable<TResult> GetClosureBi<TResult>(this IEnumerable<TResult> source,
           TResult exclude, Func<TResult, IEnumerable<TResult>> f, Func<TResult, IEnumerable<TResult>> f2)
        {
            List<TResult> ret = new List<TResult>();
            foreach (TResult t in source)
            {
                IEnumerable<TResult> res = f(t).Concat(f2(t)).Where(x => !x.Equals(exclude));
                if (res.Count() > 0)
                {
                    ret.AddRange(res.GetClosureBi(t, f, f2));
                }
                else
                {
                    ret.Add(t);
                }
            }
            return ret;
        }
    }
}
