/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using CyPhyMLImpl = Udm.Impl.CyPhyML;
using ESMoLImpl = Udm.Impl.ESMoL;
using Udm;

namespace CyPhy2ESMoL
{
    class CyPhy2ESMoLLogic
    {
        private GME.CSharp.GMEConsole GMEConsole;
        private Dictionary<CyPhyML.IComponent, ESMoL.IComponent> softwareFunctionMap = new Dictionary<CyPhyML.IComponent, ESMoL.IComponent>();
        Dictionary<CyPhyML.IComponent, ESMoL.IIODevice> ioDevices = new Dictionary<CyPhyML.IComponent, ESMoL.IIODevice>();
        Dictionary<CyPhyML.IComponent, ESMoL.INode> nodes = new Dictionary<CyPhyML.IComponent, ESMoL.INode>();
        private Udm.Native.UdmGme.GmeDataNetwork CyPhyDN;
        private Udm.Native.SmartDataNetwork ESMoLDN;
        private Udm.Native.SmartDataNetwork ESMoLMetaDN;

        private ESMoL.IHWElement getIODeviceOrNode(CyPhyML.IComponent cyPhyComponent)
        {
            ESMoL.IIODevice ioDevice;
            if (ioDevices.TryGetValue(cyPhyComponent, out ioDevice))
            {
                return ioDevice;
            }
            return nodes[cyPhyComponent];
        }

        public CyPhy2ESMoLLogic(GME.CSharp.GMEConsole GMEConsole, Udm.Native.UdmGme.GmeDataNetwork CyPhyDN,
            Udm.Native.SmartDataNetwork ESMoLDN,
            Udm.Native.SmartDataNetwork ESMoLMetaDN)
        {
            this.GMEConsole = GMEConsole;
            this.ESMoLDN = ESMoLDN;
            this.ESMoLMetaDN = ESMoLMetaDN;
            this.CyPhyDN = CyPhyDN;
        }

        public static T getOriginalArchetype<T>(T o, Func<T, T> f) where T : Udm.IUdmObject
        {
            while (f(o) != null)
            {
                o = f(o);
            }
            return o;
        }

        public static IEnumerable<Udm.IUdmObject> getDescendents(Udm.IUdmObject o)
        {
            Queue<Udm.IUdmObject> q = new Queue<Udm.IUdmObject>();
            q.Enqueue(o);
            while (q.Count != 0)
            {
                o = q.Dequeue();
                yield return o;
                foreach (Udm.IUdmObject child in o.children)
                {
                    q.Enqueue(child);
                }
            }
        }

        public static IEnumerable<Udm.IUdmObject> getSelfAndAncestors(Udm.IUdmObject o)
        {
            while (o != null && o.id != 0) // KMS: workaround bug in UdmCli
            {
                yield return o;
                o = o.parent;
            }
        }

        public void CyPhy2ESMoL(ESMoL.IRootFolder rf, ESMoL.IDesignFolder df, CyPhyML.IComponentAssembly assembly, string outdir, string outfile, SortedDictionary<string, object> componentParameters)
        {
            // 1. For each software function component type in the CyPhyML model, create an ESMoL Component Type in an ESMoL SystemTypes model.
            // If we could collect one set of original objects, that would be the right set for the SysTypes model in ESMoL.

            ESMoL.ISystemTypes systemTypes = ESMoLImpl.SystemTypes.Create(df);
            systemTypes.name = assembly.name + "_Types";
            // TODO: currentobj ?
            rf.name = assembly.name;
            df.name = assembly.name;

            // TODO TestInjectionPoint
            // TODO TestComponent
            HashSet<CyPhyML.IComponent> softwareFunctionComponents = new HashSet<CyPhyML.IComponent>();
            softwareFunctionComponents.UnionWith(getDescendents(assembly).Where(x => x.type.Equals(CyPhyMLImpl.Component.meta))
                .Select(CyPhyMLImpl.Component.ICast)
                .Where(x => x.ESMoLLink_child != null)
                .Select(x => getOriginalArchetype(x, y => y.archetype)));
            foreach (CyPhyMLImpl.Component softwareFunctionComponent in softwareFunctionComponents)
            {
                ESMoL.IComponent esmolComp = ESMoLImpl.Component.Create(systemTypes);
                esmolComp.name = GetUniqueName(softwareFunctionComponent.name);

                softwareFunctionMap[softwareFunctionComponent] = esmolComp;
            }

            // 2. For each message type in the CyPhyML components, create an ESMoL message type in the SystemTypes model.
            Dictionary<CyPhyML.IMessage, ESMoL.IMessage> messageMap = new Dictionary<CyPhyML.IMessage, ESMoL.IMessage>();
            HashSet<CyPhyML.IMessage> messages = new HashSet<CyPhyML.IMessage>();
            IEnumerable<CyPhyML.IUsesMessage> usesMessages = getDescendents(assembly).GetUdmInstances(CyPhyMLImpl.UsesMessage.meta, CyPhyMLImpl.UsesMessage.ICast);
            foreach (CyPhyML.IUsesMessage nullRef in usesMessages.Where(x => x.@ref == null))
            {
                ThrowError(x => x + " is a null reference", nullRef);
            }
            messages.UnionWith(usesMessages.Select(x => x.@ref));
            foreach (CyPhyML.IMessage message in messages)
            {
                ESMoL.IMessage esmolMessage = ESMoLImpl.Message.Create(systemTypes);
                esmolMessage.name = message.name;
                messageMap[message] = esmolMessage;
            }

            Dictionary<CyPhyML.IUsesMessage, ESMoL.IMessageRef> messageRefMap = new Dictionary<CyPhyML.IUsesMessage, ESMoL.IMessageRef>();
            // 3. For each ESMoL component type, add a reference to the proper ESMoL message type for each input and output message in the CyPhyML function component.
            foreach (KeyValuePair<CyPhyML.IComponent, ESMoL.IComponent> components in softwareFunctionMap)
            {
                foreach (CyPhyML.IUsesMessage usesMessage in getDescendents(components.Key)
                    .GetUdmInstances(CyPhyMLImpl.UsesMessage.meta, CyPhyMLImpl.UsesMessage.ICast))
                {
                    ESMoL.IMessageRef mref = ESMoLImpl.MessageRef.Create(components.Value);
                    mref.name = usesMessage.@ref.name;
                    mref.@ref = messageMap[usesMessage.@ref];
                    messageRefMap[usesMessage] = mref;
                }
            }

            // 4. TODO: How do we get the information regarding the Simulink subsystem block that will implement the ESMoL software component function?


            // 5. Create an ESMoL System model.
            ESMoL.ISystem design = ESMoLImpl.System.Create(df);
            design.name = assembly.name + "_Design";

            // 6. For each CyPhyML deployment, flatten the software dataflows defined by CyPhyML component instances in the assemblies. 
            // In the ESMoL System LogicalArchitecture aspect, create a reference to the proper ESMoL component type for each 
            // corresponding CyPhyML Software component instance in the flattened dataflow.
            Dictionary<CyPhyML.IComponent, ESMoL.IComponentRef> softwareFunctionInstanceMap = new Dictionary<CyPhyML.IComponent, ESMoL.IComponentRef>();
            foreach (CyPhyML.IComponent comp in getDescendents(assembly)
                .GetUdmInstances(CyPhyMLImpl.Component.meta, CyPhyMLImpl.Component.ICast)
                .Where(x => x.ESMoLLink_child != null))
            {
                ESMoL.IComponentRef compRef = ESMoLImpl.ComponentRef.Create(design);
                compRef.name = GetUniqueName(comp.name);
                compRef.@ref = softwareFunctionMap[getOriginalArchetype(comp, y => y.archetype)];
                softwareFunctionInstanceMap[comp] = compRef;
            }

            // 7. Wire together the ESMoL component instance ports based on the connections between the CyPhyML Software Communication Ports.
            foreach (var comps in softwareFunctionInstanceMap)
            {
                foreach (CyPhyML.ISoftwareCommunicationPort srcport in getDescendents(comps.Key).GetUdmInstances(CyPhyMLImpl.SoftwareCommunicationPort.meta, CyPhyMLImpl.SoftwareCommunicationPort.ICast))
                {
                    if (srcport.dstInformationFlow.Count() != 0)
                        continue; // we only care about the start of InformationFlow paths
                    CyPhyML.IUsesMessage srcUses = srcport.UsesMessage_child;
                    if (srcUses == null)
                        ThrowError(x => "Software Communication Port " + x + " has no UsesMessage child", srcport);
                    {
                        ESMoL.IComponentRef srcComponent = comps.Value;
                        // FIXME: verify this is correct (seems too simple)
                        ESMoL.IMessageRef srcmRef = messageRefMap[getOriginalArchetype(srcUses, y => y.archetype)];
                        //ESMoL.IMessageRef srcmRef =
                        //ESMoLImpl.Component.Cast(srcComponent.@ref).MessageRef_children
                        //.Where(x => x.Equals(messageRefMap[getOriginalArchetype(srcUses, y => y.archetype)])).First();

                        foreach (CyPhyML.ISoftwareCommunicationPort dstport in GetClosure((CyPhyML.ISignalPortType)srcport, x => x.srcInformationFlow_targets)
                            .GetUdmInstances(CyPhyMLImpl.SoftwareCommunicationPort.meta, CyPhyMLImpl.SoftwareCommunicationPort.ICast))
                        {
                            if (!dstport.parent.type.Equals(CyPhyMLImpl.Component.meta))
                            {
                                GMEConsole.Out.WriteLine("Ignoring " + GetLink(dstport) + " since it is not contained by a Component");
                                continue;
                            }
                            if (srcport.Equals(dstport))
                                ThrowError(x => "Loop detected at " + x, srcport);
                            // FIXME: cardinality == 1 or need a loop?
                            if (dstport.UsesMessage_child == null && srcport.UsesMessage_child == null)
                                ThrowError(delegate(string x, string y) { return "Neither Software Communication Port " + x + " nor " + y + " has a UsesMessage child"; }, dstport, srcport);
                            CyPhyML.IUsesMessage dstUses;
                            if (dstport.UsesMessage_child == null)
                            {
                                dstUses = srcUses;
                                GMEConsole.Warning.WriteLine("Warning: " + GetLink(dstport) + " has no UsesMessage. Using UsesMessage from " + GetLink(srcport));
                            }
                            else
                                dstUses = dstport.UsesMessage_child;
                            ESMoL.IComponentRef dstComponent = softwareFunctionInstanceMap[CyPhyMLImpl.Component.Cast(dstport.parent)];

                            ESMoL.IMessageRef dstmRef = messageRefMap[getOriginalArchetype(dstUses, y => y.archetype)];

                            ESMoL.IDependency dep = ESMoLImpl.Dependency.Create(srcComponent.parent);
                            dep.srcDependency__rp_helper = dstComponent;
                            dep.dstDependency__rp_helper = srcComponent;
                            dep.dstDependency = srcmRef;
                            dep.srcDependency = dstmRef;
                            if (!srcUses.@ref.Equals(dstUses.@ref))
                                ThrowError((x, y) => x + " does not reference the same Message as " + y + ", yet they are connected", srcUses, dstUses);

                            // FIXME: maybe multiple connections makes this fail, or there is a better way to do it?
                            srcmRef.name = srcmRef.name + "In";
                            dstmRef.name = dstmRef.name + "Out";
                        }
                    }
                }
            }

            // 8. Create an ESMoL Platform model.
            ESMoL.IHardwareUnit platform = ESMoLImpl.HardwareUnit.Create(df);
            platform.name = assembly.name + "_Hardware";
            // 9. From the CyPhyML deployment, flatten the hardware assemblies.
            // 10. Copy the hardware topology from the flattened CyPhyML hardware to the ESMoL platform model, creating Nodes, IODevices, and Buses as needed.
            Dictionary<CyPhyML.IBus, ESMoL.ICANBus> buses = new Dictionary<CyPhyML.IBus, ESMoL.ICANBus>();
            foreach (CyPhyML.IBus bus in getDescendents(assembly)
                .GetUdmInstances(CyPhyMLImpl.Bus.meta, CyPhyMLImpl.Bus.ICast))
            {
                ESMoL.ICANBus esmolBus = ESMoLImpl.CANBus.Create(platform);
                esmolBus.name = GetUniqueNonKeywordname(bus.name);
                buses[bus] = esmolBus;
                //esmolBus.DataRate = GetCyberParameter(CyPhyMLImpl.MgaObject.ICast(bus.parent), "DataRate", "1Mb");
            }

            IQueryable<CyPhyML.IComponent> asse = getDescendents(assembly)
               .GetUdmInstances(CyPhyMLImpl.Component.meta, CyPhyMLImpl.Component.ICast).AsQueryable();

            foreach (CyPhyML.IComponent component in getDescendents(assembly)
                .GetUdmInstances(CyPhyMLImpl.Component.meta, CyPhyMLImpl.Component.ICast)
                // FIXME: what gets converted into a Node or IODevice?
                .Where(x => x.children.GetUdmInstances(CyPhyMLImpl.ExecutionContextAssignPort.meta,
                    CyPhyMLImpl.ExecutionContextAssignPort.ICast).Count() == 0)
                    .Where(x => HasESMoLLink(x))
                )
            {
                if (component.children.GetUdmInstances(CyPhyMLImpl.ExecutionContextPort.meta, CyPhyMLImpl.ExecutionContextPort.ICast).Count() > 0)
                {
                    ESMoL.INode node = ESMoLImpl.Node.Create(platform);
                    node.name = GetUniqueName(component.name);
                    node.PlatformType = "i686-pc-linux";
                    nodes[component] = node;

                    ESMoL.IOS os = ESMoLImpl.OS.Create(node);
                    os.name = "OS";
                }
                else
                {
                    ESMoL.IIODevice io = ESMoLImpl.IODevice.Create(platform);
                    io.name = GetUniqueName(component.name);
                    ioDevices[component] = io;
                }
            }

            Dictionary<CyPhyML.ISignalPortType, ESMoL.IConnectable> portMap = new Dictionary<CyPhyML.ISignalPortType, ESMoL.IConnectable>();
            foreach (var bus in buses)
            {
                foreach (CyPhyML.IBusPort busPort in
                    GetClosureBi((CyPhyML.IUsesBus_Members_Base)bus.Key, x => x.srcUsesBus_targets, x => x.dstUsesBus_targets)
                    .GetUdmInstances(CyPhyMLImpl.BusPort.meta, CyPhyMLImpl.BusPort.ICast)
                    // KMS: I contend this is a bug in the meta
                    .SelectMany(x => GetClosureBi((CyPhyML.ISignalPortType)x, y => y.srcInformationFlow_targets, y => y.dstInformationFlow_targets))
                    .GetUdmInstances(CyPhyMLImpl.BusPort.meta, CyPhyMLImpl.BusPort.ICast))
                {
                    if (busPort.parent.type.Equals(CyPhyMLImpl.Component.meta))
                    {
                        CyPhyML.IComponent component = CyPhyMLImpl.Component.Cast(busPort.parent);
                        ESMoL.IConnectable connend;
                        ESMoL.INode node;
                        if (nodes.TryGetValue(component, out node))
                        {
                            ESMoL.IBChan bchan = ESMoLImpl.BChan.Create(node);
                            connend = bchan;
                            bchan.name = GetUniqueName(bus.Key.name, node);
                        }
                        else
                        {
                            IQueryable<string> ds = ioDevices.Select(x => x.Key.name + " " + x.Value.name).AsQueryable();
                            ESMoL.IIODevice ioDevice;
                            if (!ioDevices.TryGetValue(component, out ioDevice))
                            {
                                ThrowError((x, y) => x + " has no ESMoLLink, but is connected to bus " + y, component, bus.Key);
                            }
                            ESMoL.IOChan ochan = ESMoLImpl.OChan.Create(ioDevice);
                            connend = ochan;
                            ochan.name = GetUniqueName(bus.Key.name, ioDevice);
                        }
                        portMap[busPort] = connend;
                        ESMoL.IWire wire = ESMoLImpl.Wire.Create(platform);
                        wire.srcWire = bus.Value;
                        wire.dstWire = connend;
                    }
                    else
                    {
                        //ThrowError((x,y)=>"Unexpected bus connection from " + x + " to " + y, bus.Key, CyPhyMLImpl.MgaObject.Cast(bus.Key.parent));
                        GMEConsole.Out.WriteLine("Warning: bus connection from " + GetLink(bus.Key) + " to " + GetLink(busPort) + ": parent is not a Component");
                    }
                }
            }


            // 11. Create channels on the Nodes and IODevices to make the appropriate connections
            List<KeyValuePair<CyPhyML.IComponent, ESMoL.IHWElement>> ioDevicesAndNodes = new List<KeyValuePair<CyPhyML.IComponent, ESMoL.IHWElement>>();
            ioDevicesAndNodes.AddRange(ioDevices.Select(x => new KeyValuePair<CyPhyML.IComponent, ESMoL.IHWElement>(x.Key, x.Value)));
            ioDevicesAndNodes.AddRange(nodes.Select(x => new KeyValuePair<CyPhyML.IComponent, ESMoL.IHWElement>(x.Key, x.Value)));
            foreach (var comps in ioDevicesAndNodes)
            {
                CyPhyML.IComponent comp = comps.Key;
                ESMoL.IHWElement srcParent = comps.Value;
                foreach (CyPhyML.IElectricalSignalPort srcPort in comp.children
                    .GetUdmInstances(CyPhyMLImpl.ElectricalSignalPort.meta, CyPhyMLImpl.ElectricalSignalPort.ICast))
                {
                    foreach (CyPhyML.ISignalPortType dstPort in srcPort.dstInformationFlow_targets)
                    {
                        if (!dstPort.parent.type.Equals(CyPhyMLImpl.Component.meta))
                        {
                            if (srcPort.srcActuatorDataConversion_targets.Count() == 0)
                                ThrowWarning((x, y) => x + " is connected to " + y + " that does not have a Component parent", srcPort, dstPort);
                            continue;
                        }
                        if (!dstPort.parent.Equals(srcPort.parent))
                        {
                            if (!HasESMoLLink(CyPhyMLImpl.MgaObject.ICast(dstPort.parent)))
                                ThrowError((x, y) => x + " is connected to " + y + " that doesn't have an ESMoLLink parent", srcPort, CyPhyMLImpl.MgaObject.ICast(dstPort.parent));
                            ESMoL.IHWElement dstParent = getIODeviceOrNode(CyPhyMLImpl.Component.Cast(dstPort.parent));
                            ESMoL.IConnectable src = ESMoLImpl.OChan.Create(srcParent);
                            src.name = GetUniqueName(CyPhyMLImpl.MgaObject.Cast(dstPort.parent).name, srcParent);
                            // FIXME: can this throw an exception due to duplicate keys?
                            portMap[srcPort] = src;
                            ESMoL.IIChan dst = ESMoLImpl.IChan.Create(dstParent);
                            dst.name = GetUniqueName(CyPhyMLImpl.MgaObject.Cast(srcPort.parent).name, dstParent);
                            portMap[dstPort] = dst;

                            ESMoL.IWire wire = ESMoLImpl.Wire.Create(srcParent.parent);
                            wire.srcWire = src;
                            wire.dstWire = dst;
                        }
                    }
                }
            }

            foreach (var iodev in ioDevices)
            {
                // TODO: xxx = 
                //GetCyberParameter(iodev.Key, "SamplePeriod", "10ms");
            }

            Dictionary<CyPhyML.IComponent, ESMoL.IComponent> sensorDataHandlerMap = new Dictionary<CyPhyML.IComponent, ESMoL.IComponent>();
            Dictionary<ESMoL.IMessageRef, ESMoL.IIChan> dataHandlerInputMap = new Dictionary<ESMoL.IMessageRef, ESMoL.IIChan>();
            Dictionary<ESMoL.IIODevice, ESMoL.IComponent> dataHandlerMap = new Dictionary<ESMoL.IIODevice, ESMoL.IComponent>();
            Dictionary<ESMoL.IComponent, ESMoL.IComponentRef> dataHandlerComprefs = new Dictionary<ESMoL.IComponent, ESMoL.IComponentRef>();
            foreach (CyPhyML.IComponent component in getDescendents(assembly)
                .GetUdmInstances(CyPhyMLImpl.Component.meta, CyPhyMLImpl.Component.ICast)
                .Where(x => x.children.GetUdmInstances(CyPhyMLImpl.SensorDataConversion.meta, CyPhyMLImpl.SensorDataConversion.ICast).Count() != 0)
                .Where(HasESMoLLink)
                )
            {
                ESMoL.IIODevice ioDevice;
                ioDevices.TryGetValue(component, out ioDevice);
                if (ioDevice == null)
                {
                    //ThrowError((x,y) => x + " is connected to " + y + " that doesn't have a parent with an ESMoLLink", component);
                }
                ESMoL.IComponent dataHandler;
                dataHandler = ESMoLImpl.Component.Create(systemTypes);
                dataHandlerMap[ioDevice] = dataHandler;
                dataHandler.name = ESMoLImpl.MgaObject.Cast(ioDevice).name + "_DataHandler";
                foreach (var dataPortConversion in component.children.GetUdmInstances(CyPhyMLImpl.SensorDataConversion.meta, CyPhyMLImpl.SensorDataConversion.ICast))
                {
                    CyPhyML.IElectricalSignalPort srcPort = dataPortConversion.srcSensorDataConversion;
                    CyPhyML.ISensorDataPort port = dataPortConversion.dstSensorDataConversion;
                    CyPhyML.IUsesMessage um = port.UsesMessage_child;
                    if (um == null)
                    {
                        CyPhyML.ISoftwareCommunicationPort umcontainer2 = GetClosure((CyPhyML.ISensorDataConsumption_Members_Base)port, x => x.dstSensorDataConsumption_targets)
                            .GetUdmInstances2(CyPhyMLImpl.SignalPortType.meta, CyPhyMLImpl.SignalPortType.ICast)
                            .GetClosure(x => x.dstInformationFlow_targets)
                            .GetUdmInstances(CyPhyMLImpl.SoftwareCommunicationPort.meta, CyPhyMLImpl.SoftwareCommunicationPort.ICast)
                            .Where(x => x.UsesMessage_child != null).FirstOrDefault();
                        if (umcontainer2 != null)
                            um = umcontainer2.UsesMessage_child;
                    }
                    if (um == null)
                    {
                        GMEConsole.Warning.WriteLine("Error: sensor data port " + GetLink(port) + " has no UsesMessage");
                        continue;
                    }
                    sensorDataHandlerMap[component] = dataHandler;
                    ESMoL.IMessageRef dataHandlerInput = ESMoLImpl.MessageRef.Create(dataHandler);
                    dataHandlerInput.@ref = messageMap[um.@ref];
                    dataHandlerInput.name = messageMap[um.@ref].name + "_In";

                    ESMoL.IMessageRef dataHandlerOutput = ESMoLImpl.MessageRef.Create(dataHandler);
                    dataHandlerOutput.@ref = messageMap[um.@ref];
                    dataHandlerOutput.name = messageMap[um.@ref].name + "_Out";
                    dataHandlerOutput.position = "(700,10)";

                    foreach (CyPhyML.ISoftwareCommunicationPort consumer in GetClosure((CyPhyML.ISensorDataConsumption_Members_Base)port, x => x.dstSensorDataConsumption_targets)
                        .GetUdmInstances2(CyPhyMLImpl.SignalPortType.meta, CyPhyMLImpl.SignalPortType.ICast)
                        .GetClosure(x => x.dstInformationFlow_targets)
                        .GetUdmInstances(CyPhyMLImpl.SoftwareCommunicationPort.meta, CyPhyMLImpl.SoftwareCommunicationPort.ICast)
                        )
                    {
                        if (consumer.UsesMessage_child != null)
                        {
                            ESMoL.IMessageRef mr;

                            ESMoL.IComponentRef dataHandlerCompref;
                            if (!dataHandlerComprefs.TryGetValue(dataHandler, out dataHandlerCompref))
                            {
                                dataHandlerCompref = ESMoLImpl.ComponentRef.Create(design);
                                dataHandlerCompref.@ref = dataHandler;
                                dataHandlerCompref.name = dataHandler.name;
                                dataHandlerComprefs[dataHandler] = dataHandlerCompref;
                            }

                            if (messageRefMap.TryGetValue(getOriginalArchetype(consumer.UsesMessage_child, y => y.archetype), out mr))
                            {
                                // DataHandler wiring: data handler msg ref port -> Dependency -> actual consuming task msg ref port
                                ESMoL.IDependency dep = ESMoLImpl.Dependency.Create(design);
                                dep.dstDependency__rp_helper = softwareFunctionInstanceMap[CyPhyMLImpl.Component.Cast(consumer.parent)];
                                dep.dstDependency = mr;
                                // dep.srcDependency__rp_helper =  DataHandlers are deployed exactly once (i.e. just above)
                                dep.srcDependency = dataHandlerOutput;

                                break;
                            }
                            else
                            {
                                dataHandlerInput.parent = null;
                                dataHandlerOutput.parent = null;
                                ThrowWarning((x, y) => "Sensor port " + x + " is connected to " + y + " that doesn't have a software component parent", port, consumer);
                            }
                        }
                    }
                    foreach (CyPhyML.IElectricalSignalPort elecPort in GetClosure((CyPhyML.ISignalPortType)srcPort, x => x.dstInformationFlow_targets)
                        .GetUdmInstances(CyPhyMLImpl.ElectricalSignalPort.meta, CyPhyMLImpl.ElectricalSignalPort.ICast))
                    {
                        ESMoL.IConnectable connectable;
                        if (portMap.TryGetValue(elecPort, out connectable) && connectable.type.Equals(ESMoLImpl.IChan.meta))
                        {
                            dataHandlerInputMap[dataHandlerInput] = ESMoLImpl.IChan.ICast(portMap[elecPort]);
                            break;
                        }
                        else
                        {
                            ThrowWarning((x, y) => "Sensor electrical port " + x + " is connected to " + y + " that doesn't have a processor parent", srcPort, elecPort);
                        }
                    }
                }
            }


            Dictionary<CyPhyML.IComponent, ESMoL.INodeRef> nodeDeploymentMap = new Dictionary<CyPhyML.IComponent, ESMoL.INodeRef>();
            foreach (var x in nodes)
            {
                ESMoL.INodeRef node = ESMoLImpl.NodeRef.Create(design);
                ESMoL.INode origNode = x.Value;
                node.name = origNode.name;
                node.@ref = origNode;
                nodeDeploymentMap[x.Key] = node;
            }

            // 12. From the CyPhyML deployment assembly, create ComponentAssignment connections for each CyPhyML software function instance that has an execution context connection to a particular processor.
            foreach (var software in softwareFunctionInstanceMap)
            {
                foreach (CyPhyML.IExecutionContextAssignPort contextAssign in software.Key.children
                    .GetUdmInstances(CyPhyMLImpl.ExecutionContextAssignPort.meta, CyPhyMLImpl.ExecutionContextAssignPort.ICast)
                    .Where(HasESMoLLink))
                {
                    foreach (CyPhyML.IContextAssignmentPort end in GetClosureBi((CyPhyML.IExecutionAssignment_Members_Base)contextAssign, x => x.dstExecutionAssignment_targets, x => x.srcExecutionAssignment_targets)
                        .GetUdmInstances2(CyPhyMLImpl.ContextAssignmentPort.meta, CyPhyMLImpl.ContextAssignmentPort.ICast))
                    {
                        if (!end.parent.type.Equals(CyPhyMLImpl.Component.meta))
                        {
                            ThrowError(x => "Context Assignment Port end " + x + " does not have a Component parent.", end);
                        }
                        CyPhyML.IComponent processor = CyPhyMLImpl.Component.Cast(end.parent);
                        if (!HasESMoLLink(processor))
                            ThrowError((x, y) => "Context Assignment Port end " + x + " connects to " + y + " that doesn't have an ESMoLLink", end, processor);
                        ESMoL.INodeRef node = nodeDeploymentMap[processor];
                        ESMoL.IComponentAssignment compAssignment = ESMoLImpl.ComponentAssignment.Create(design);
                        compAssignment.srcComponentAssignment = software.Value;
                        compAssignment.dstComponentAssignment = node;
                    }
                }
            }


            // DataHandler wiring: 2. data handler -> ComponentAssignment -> Processor object
            foreach (var nodeEntry in nodeDeploymentMap)
            {
                ESMoL.INodeRef node = nodeEntry.Value;
                foreach (ESMoL.IIChan ichan in node.@ref.Channel_children.GetUdmInstances(ESMoLImpl.IChan.meta, ESMoLImpl.IChan.ICast))
                {
                    foreach (ESMoL.IOChan ochan in ichan.srcWire_targets.GetUdmInstances(ESMoLImpl.OChan.meta, ESMoLImpl.OChan.ICast)
                        .Where(x => x.parent.type.Equals(ESMoLImpl.IODevice.meta)))
                    {
                        ESMoL.IIODevice iodev = ESMoLImpl.IODevice.Cast(ochan.parent);
                        ESMoL.IComponent dataHandler;
                        if (dataHandlerMap.TryGetValue(iodev, out dataHandler))
                        {
                            ESMoL.IComponentRef compref;
                            if (dataHandlerComprefs.TryGetValue(dataHandler, out compref)
                                && !compref.dstComponentAssignment_targets.Contains(node))
                            {
                                ESMoL.IComponentAssignment dhcompAssignment = ESMoLImpl.ComponentAssignment.Create(design);
                                dhcompAssignment.srcComponentAssignment = compref;
                                dhcompAssignment.dstComponentAssignment = node;
                            }
                        }
                    }
                }
            }
            // DataHandler wiring: data handler msg ref port -> CommMapping -> node port
            foreach (var input in dataHandlerInputMap)
            {
                ESMoL.ICommMapping commMapping = ESMoLImpl.CommMapping.Create(design);
                // TODO: __rp_helpers
                commMapping.srcCommMapping = input.Value;
                commMapping.dstCommMapping = input.Key;
            }


            // 13. If an ESMoL component is connected to a processor Node and its corresponding CyPhyML component 
            // has connections to sensors, actuators, or buses then connect the ESMoL message ports accordingly 
            // to the channel objects on the processors. 
            // IMPORTANT: If the ESMoL component is not connected to a processor, do not connect the sensor/actuator/bus connections, since those will have to be handled by an allocation process.
            foreach (var comp in softwareFunctionInstanceMap.Where(x => x.Value.dstComponentAssignment.Count() > 0))
            {
                // FIXME: iterate, or is the cardinality wrong?
                ESMoL.INodeRef node = comp.Value.dstComponentAssignment_targets.First();
                // FIXME: software no longer contains Sensor ports
                foreach (CyPhyML.ISensorDataPort srcPort in comp.Key.children
                    .GetUdmInstances(CyPhyMLImpl.SensorDataPort.meta, CyPhyMLImpl.SensorDataPort.ICast))
                {
                    foreach (CyPhyML.ISensorDataPort dstPort in srcPort.srcSensorDataConsumption_targets
                         .GetClosure(x => x.srcSensorDataConsumption_targets)
                         .Where(x => x.parent.type.Equals(CyPhyMLImpl.Component.meta))
                         .GetUdmInstances(CyPhyMLImpl.SensorDataPort.meta, CyPhyMLImpl.SensorDataPort.ICast))
                    {
                        CyPhyML.IComponent component = CyPhyMLImpl.Component.Cast(dstPort.parent);
                        foreach (var busPort in component.children
                            .GetUdmInstances(CyPhyMLImpl.BusPort.meta, CyPhyMLImpl.BusPort.ICast))
                        {
                            foreach (var bus in busPort.dstUsesBus_targets)
                            {
                                foreach (var nodeBusPort in bus.srcUsesBus_targets.Where(busTarget =>
                                    node.@ref.children.Contains(portMap[CyPhyMLImpl.SignalPortType.ICast(busTarget)])))
                                {
                                    ESMoL.ICommMapping commMapping = ESMoLImpl.CommMapping.Create(design);
                                    commMapping.srcCommMapping__rp_helper = node;
                                    commMapping.dstCommMapping__rp_helper = comp.Value;
                                    commMapping.srcCommMapping = ESMoLImpl.Channel.ICast(portMap[CyPhyMLImpl.SignalPortType.ICast(nodeBusPort)]);
                                    commMapping.dstCommMapping = messageRefMap[getOriginalArchetype(srcPort.UsesMessage_child, y => y.archetype)];
                                }
                            }
                        }
                    }
                }
                foreach (CyPhyML.ISoftwareCommunicationPort srcPort in comp.Key.children
                    .GetUdmInstances(CyPhyMLImpl.SoftwareCommunicationPort.meta, CyPhyMLImpl.SoftwareCommunicationPort.ICast))
                {
                    foreach (CyPhyML.IActuatorControlPort dstPort in
                        GetClosure((CyPhyML.ISignalPortType)srcPort, x => x.dstInformationFlow_targets)
                        .GetUdmInstances2(CyPhyMLImpl.ActuatorControl_Members_Base.meta, CyPhyMLImpl.ActuatorControl_Members_Base.ICast)
                        .GetClosure(x => x.dstActuatorControl_targets)
                        .GetUdmInstances(CyPhyMLImpl.ActuatorControlPort.meta, CyPhyMLImpl.ActuatorControlPort.ICast)
                        .Where(x => x.parent.type.Equals(CyPhyMLImpl.Component.meta)))
                    {
                        IEnumerable<CyPhyML.IElectricalSignalPort> elecPorts = dstPort.dstActuatorDataConversion_targets
                            .SelectMany(conversion => GetClosure((CyPhyML.ISignalPortType)conversion, x => x.srcInformationFlow_targets)
                            .GetUdmInstances(CyPhyMLImpl.ElectricalSignalPort.meta, CyPhyMLImpl.ElectricalSignalPort.ICast));
                        foreach (CyPhyML.IElectricalSignalPort elecPort in elecPorts.Where(elecPort => !portMap.ContainsKey(elecPort)))
                        {
                            ThrowWarning((x, y) => "Actuator port " + x + " is connected to " + y + " that does not have a processor parent", srcPort, elecPort);
                        }
                        elecPorts = elecPorts.Where(elecPort => portMap.ContainsKey(elecPort));
                        foreach (CyPhyML.IElectricalSignalPort elecPort in elecPorts.Where(elecPort => node.@ref.children.Contains(portMap[elecPort])))
                        {
                            ESMoL.ICommMapping commMapping = ESMoLImpl.CommMapping.Create(design);
                            commMapping.dstCommMapping__rp_helper = node;
                            commMapping.srcCommMapping__rp_helper = comp.Value;
                            commMapping.dstCommMapping = ESMoLImpl.Channel.ICast(portMap[elecPort]);
                            commMapping.srcCommMapping = messageRefMap[getOriginalArchetype(srcPort.UsesMessage_child, y => y.archetype)];
                        }
                        if (elecPorts.Count() == 0)
                        {
                            ThrowWarning(x => "Software actuator port " + x + " is not connected to a processor", srcPort);
                        }
                        CyPhyML.IComponent component = CyPhyMLImpl.Component.Cast(dstPort.parent);
                        foreach (var busPort in component.children
                            .GetUdmInstances(CyPhyMLImpl.BusPort.meta, CyPhyMLImpl.BusPort.ICast))
                        {
                            foreach (var bus in busPort.dstUsesBus_targets)
                            {
                                foreach (var nodeBusPort in bus.srcUsesBus_targets.Where(busTarget =>
                                    node.@ref.children.Contains(portMap[CyPhyMLImpl.SignalPortType.ICast(busTarget)])))
                                {
                                    ESMoL.ICommMapping commMapping = ESMoLImpl.CommMapping.Create(design);
                                    commMapping.dstCommMapping__rp_helper = node;
                                    commMapping.srcCommMapping__rp_helper = comp.Value;
                                    commMapping.dstCommMapping = ESMoLImpl.Channel.ICast(portMap[CyPhyMLImpl.SignalPortType.ICast(nodeBusPort)]);
                                    commMapping.srcCommMapping = messageRefMap[getOriginalArchetype(srcPort.UsesMessage_child, y => y.archetype)];
                                }
                            }
                        }
                    }
                }
            }


            //foreach (var comp in softwareFunctionInstanceMap)
            foreach (var esmolComp in design.children.GetUdmInstances(ESMoLImpl.ComponentRef.meta, ESMoLImpl.ComponentRef.ICast))
            {
                //CyPhyML.IComponent cpComp = comp.Key;
                //ESMoL.IComponentRef esmolComp = comp.Value;
                ESMoL.ITTExecInfo componentExecInfo = ESMoLImpl.TTExecInfo.Create(design);
                {
                    componentExecInfo.name = esmolComp.name + "_" + componentExecInfo.id;
                    CyPhyML.IComponent cpComp = softwareFunctionInstanceMap.Where(x => x.Value.Equals(esmolComp)).FirstOrDefault().Key;
                    if (cpComp != null)
                    {
                        //componentExecInfo.ExecPeriod = GetCyberParameter(cpComp, "ExecPeriod", "100ms");
                        //componentExecInfo.WCDuration = GetCyberParameter(cpComp, "WCET", "1ms");
                    }
                    else
                    {
                        cpComp = sensorDataHandlerMap.Where(x => x.Value.Equals(esmolComp.@ref)).FirstOrDefault().Key;
                        if (cpComp != null)
                        {
                            //componentExecInfo.ExecPeriod = GetCyberParameter(cpComp, "SamplePeriod", "100ms");
                            componentExecInfo.WCDuration = "0.02ms";
                        }
                        else
                        {

                        }
                    }
                    ESMoL.IExecutionAssignment execAssignment = ESMoLImpl.ExecutionAssignment.Create(design);
                    execAssignment.srcExecutionAssignment = componentExecInfo;
                    execAssignment.dstExecutionAssignment = esmolComp;
                }
                foreach (ESMoL.IMessageRef messageRef in esmolComp.@ref.children
                    .GetUdmInstances(ESMoLImpl.MessageRef.meta, ESMoLImpl.MessageRef.ICast)
                    .Where(x => x.srcDependency.Count() == 0))
                {
                    ESMoL.ITTExecInfo execInfo = ESMoLImpl.TTExecInfo.Create(design);
                    execInfo.name = messageRef.name + "_" + esmolComp.name + "_" + execInfo.id;
                    execInfo.ExecPeriod = componentExecInfo.ExecPeriod;
                    execInfo.WCDuration = "1ms"; // TODO: this is ignored, no?
                    ESMoL.IExecutionAssignment execAssignment = ESMoLImpl.ExecutionAssignment.Create(design);
                    execAssignment.srcExecutionAssignment = execInfo;
                    execAssignment.dstExecutionAssignment__rp_helper = esmolComp;
                    execAssignment.dstExecutionAssignment = messageRef;
                    foreach (ESMoL.IDependency dstDependency in messageRef.dstDependency)
                    {
                        ESMoL.IExecutionAssignment dstexecAssignment = ESMoLImpl.ExecutionAssignment.Create(design);
                        dstexecAssignment.srcExecutionAssignment = execInfo;
                        dstexecAssignment.dstExecutionAssignment__rp_helper = dstDependency.dstDependency__rp_helper;
                        dstexecAssignment.dstExecutionAssignment = dstDependency.dstDependency;
                    }
                }
            }

            Dictionary<string, Dictionary<string, Udm.IUdmObject>> esmolLinkObjects = new Dictionary<string, Dictionary<string, Udm.IUdmObject>>();
            Dictionary<string, ESMoL.IMessage> importedMessageMap = new Dictionary<string, ESMoL.IMessage>();

            foreach (var link in getDescendents(assembly).GetUdmInstances(CyPhyMLImpl.ESMoLLink.meta, CyPhyMLImpl.ESMoLLink.ICast)
                .Where(link => link.parent.type.Equals(CyPhyMLImpl.Component.meta))
                .Select(x => getOriginalArchetype(x, y => y.archetype))
                .Distinct())
            {
                ESMoL.IComponent esmolComp;
                if (!softwareFunctionMap.TryGetValue(CyPhyMLImpl.Component.ICast(link.parent), out esmolComp))
                    continue;
                if (link.URI == "")
                    continue;

                Uri linkUri = new Uri(link.URI);

                if (!linkUri.Scheme.Equals("esmol", StringComparison.OrdinalIgnoreCase))
                {
                    GMEConsole.Warning.WriteLine("Warning: Ignoring ESMoLLink " + GetLink(link) + ", as it does not use the esmol scheme");
                    continue;
                }
                if (linkUri.Fragment.Length < 4)
                {
                    GMEConsole.Warning.WriteLine("Warning: Ignoring ESMoLLink " + GetLink(link) + " since model path '" + linkUri.Fragment + "' is too short");
                    continue;
                }
                // n.b. the Uris are written wrong...
                string filename = linkUri.Host + linkUri.AbsolutePath;
                string modelpath = linkUri.Fragment.Substring(1); // skip leading #

                IntPtr punkRootFolder = Udm.Native.UdmGme.UdmGme.Udm2Gme(CyPhyDN.GetRootObject());
                // GetObjectForIUnknown: "The reference count will be decremented when the runtime performs garbage collection on the managed object that represents the COM object"
                GME.MGA.IMgaFolder rootFolder = (GME.MGA.IMgaFolder)System.Runtime.InteropServices.Marshal.GetObjectForIUnknown(punkRootFolder);
                System.Runtime.InteropServices.Marshal.Release(punkRootFolder);

                // Skip MGA=
                string mgaDir = System.IO.Path.GetDirectoryName(rootFolder.Project.ProjectConnStr.Substring(4));
                string mgaPath = System.IO.Path.Combine(mgaDir, filename);

                Dictionary<string, Udm.IUdmObject> esmolLinks;
                if (!esmolLinkObjects.TryGetValue(mgaPath, out esmolLinks))
                {
                    esmolLinks = new Dictionary<string, Udm.IUdmObject>();
                    esmolLinkObjects[mgaPath] = esmolLinks;

                    // TODO ensure file exists
                    if (!System.IO.File.Exists(mgaPath))
                    {
                        ThrowWarning(x => "Warning: ESMoLLink " + x + " points to non-existant file " + mgaPath, link);
                        continue;
                    }

                    UpgradeMgaToLatestParadigm(mgaPath, "ESMoL");

                    Udm.Native.SmartDataNetwork linkedSDN = new Udm.Native.SmartDataNetwork(ESMoLMetaDN);
                    linkedSDN.OpenExisting(mgaPath, "ESMoL", Udm.Native.BackendSemantics.CHANGES_LOST_DEFAULT);
                    HashSet<Udm.Native.UdmObject> origObjects = new HashSet<Udm.Native.UdmObject>();
                    origObjects.UnionWith(ESMoLDN.GetRootObject().GetChildObjects());
                    Udm.Native.Udm.CopyObjectHierarchy(linkedSDN.GetRootObject(), ESMoLDN.GetRootObject());
                    HashSet<Udm.Native.UdmObject> newObjects = new HashSet<Udm.Native.UdmObject>();
                    newObjects.UnionWith(ESMoLDN.GetRootObject().GetChildObjects());
                    newObjects.ExceptWith(origObjects);
                    foreach (var x in newObjects)
                    {
                        esmolLinks[ExtractName(x)] = new Udm.Native.UdmCliObject(x);
                        // TODO: could set the name to current name + name of model

                        // Delete copied Systems and HardwareUnits, or the ESMoL toolchain won't know which System to process
                        if (new Udm.Native.UdmCliObject(x).type.Equals(ESMoLImpl.DesignFolder.meta))
                        {
                            ESMoLImpl.DesignFolder designFolder = ESMoLImpl.DesignFolder.Cast(x);
                            foreach (IUdmObject o in designFolder.children
                                .Where(y => y.type.Equals(ESMoLImpl.System.meta) ||
                                            y.type.Equals(ESMoLImpl.HardwareUnit.meta)))
                            {
                                o.parent = null;
                            }
                        }
                    }
                    linkedSDN.CloseNoUpdate();
                }

                modelpath = modelpath.Replace("RootFolder/", "");
                if (modelpath.StartsWith("/"))
                    modelpath = modelpath.Substring(1);

                // n.b. can't cast to ESMoL.MgaObject, since folders aren't MgaObjects
                Udm.IUdmObject uriTarget = esmolLinks.Values.FirstOrDefault(x => ExtractName(x) == modelpath.Split('/')[0]);
                if (uriTarget == null)
                    ThrowError(x => "ESMoLLink " + x + ": couldn't find root element '" + modelpath.Split('/')[0] + "' in " + String.Join(", ", esmolLinks.Values.Select(ExtractName).ToArray())
                        + " in file " + mgaPath, link);
                foreach (string name in modelpath.Split('/').AsEnumerable().Skip(1))
                {
                    uriTarget = uriTarget.children.FirstOrDefault(x => Udm.Native.Udm.ExtractName(((Udm.Native.UdmCliObject)x).backing) == name);
                    if (uriTarget == null)
                        ThrowError(x => "ESMoLLink " + x + ": couldn't find element '" + name + "'", link);
                }
                // TODO: check uriTarget.type
                if (!uriTarget.type.Equals(ESMoLImpl.Component.meta))
                    ThrowError(x => "ESMoLLink " + x + " points to a model element of kind " + uriTarget.type.name().Get() + " instead of expected kind Component", link);

                ESMoL.IComponent referenceComponent = ESMoLImpl.Component.ICast(uriTarget);
                Dictionary<ESMoL.ISubsystemRef, ESMoL.ISubsystemRef> referenceSubsystemToSubsystem = new Dictionary<ESMoL.ISubsystemRef, ESMoL.ISubsystemRef>();
                IEnumerable<ESMoL.ISubsystemRef> referenceSubsystemRefs = referenceComponent.children.GetUdmInstances(ESMoLImpl.SubsystemRef.meta, ESMoLImpl.SubsystemRef.ICast);
                if (referenceSubsystemRefs.Count() == 0)
                    ThrowError(x => "ESMoLLink " + x + " doesn't have  any SubsystemRefs", link);
                foreach (ESMoL.ISubsystemRef referenceSubRef in referenceSubsystemRefs)
                {
                    ESMoL.ISubsystemRef subRef = ESMoLImpl.SubsystemRef.Create(esmolComp);
                    subRef.@ref = referenceSubRef.@ref;
                    subRef.name = referenceSubRef.name;
                    subRef.position = referenceSubRef.position;
                    referenceSubsystemToSubsystem[referenceSubRef] = subRef;
                }

                IEnumerable<ESMoL.IMessage> referenceMessages = referenceComponent.children.GetUdmInstances(ESMoLImpl.MessageRef.meta, ESMoLImpl.MessageRef.ICast)
                    .Select(x => x.@ref).GetUdmInstances(ESMoLImpl.Message.meta, ESMoLImpl.Message.ICast);
                foreach (ESMoL.IMessageRef messageRef in esmolComp.children
                    .GetUdmInstances(ESMoLImpl.MessageRef.meta, ESMoLImpl.MessageRef.ICast))
                {
                    IEnumerable<ESMoL.IMessage> referenceMessageMatches = referenceMessages.Where(x => x.name == ESMoLImpl.Message.ICast(messageRef.@ref).name);
                    if (referenceMessageMatches.Count() == 0)
                        ThrowError(x => "ESMoLLink " + x + " error: cannot find matching message for '" + esmolComp.name + "' MessageRef '" + messageRef.name, link);
                    messageRef.@ref = referenceMessageMatches.First();
                    importedMessageMap[referenceMessageMatches.First().name] = referenceMessageMatches.First();
                }

                // TODO: wiring
                foreach (ESMoL.IConnector referenceConnector in referenceComponent.children.GetUdmInstances(ESMoLImpl.Connector.meta, ESMoLImpl.Connector.ICast))
                {
                    ESMoL.IMessageRef referenceSrc;
                    bool backwards = false;
                    if (referenceConnector.srcConnector.parent.type.Equals(ESMoLImpl.Message.meta))
                    {
                        referenceSrc = ESMoLImpl.MessageRef.Cast(referenceConnector.srcConnector__rp_helper);
                    }
                    else
                    {
                        referenceSrc = ESMoLImpl.MessageRef.Cast(referenceConnector.dstConnector__rp_helper);
                        backwards = true;
                    }
                    IEnumerable<ESMoL.IMessageRef> messageRefs = esmolComp.children
                        .GetUdmInstances(ESMoLImpl.MessageRef.meta, ESMoLImpl.MessageRef.ICast)
                        .Where(x => ExtractName(x) == ExtractName(referenceSrc));
                    if (messageRefs.Count() == 0)
                    {
                        ThrowWarning(x => "ESMoLLink " + x + " error: cannot find matching generated MessageRef for imported MessageRef " + referenceSrc.name, link);
                        continue;
                    }
                    messageRefs.First().position = referenceSrc.position;
                    ESMoL.IConnector connector = ESMoLImpl.Connector.Create(esmolComp);
                    if (!backwards)
                    {
                        connector.dstConnector__rp_helper = referenceSubsystemToSubsystem[ESMoLImpl.SubsystemRef.ICast(referenceConnector.dstConnector__rp_helper)];
                        connector.srcConnector__rp_helper = messageRefs.First();
                    }
                    else
                    {
                        connector.srcConnector__rp_helper = referenceSubsystemToSubsystem[ESMoLImpl.SubsystemRef.ICast(referenceConnector.srcConnector__rp_helper)];
                        connector.dstConnector__rp_helper = messageRefs.First();
                    }
                    connector.dstConnector = referenceConnector.dstConnector;
                    connector.srcConnector = referenceConnector.srcConnector;
                }
            }

            {
                Dictionary<ESMoL.ITypeBase, ESMoL.ISubsystem> passThroughsByType = new Dictionary<ESMoL.ITypeBase, ESMoL.ISubsystem>();
                Udm.Native.UdmDom.str_xsd_storage.StoreXsd("ESMoL.xsd", global::CyPhy2ESMoL.Properties.Resources.ESMoL);
                Udm.Native.SmartDataNetwork DataHandler_Passthrough = new Udm.Native.SmartDataNetwork(ESMoLMetaDN);
                DataHandler_Passthrough.OpenExistingFromString(global::CyPhy2ESMoL.Properties.Resources.DataHandler_Passthrough,
                    "ESMoL.xsd", Udm.Native.BackendSemantics.CHANGES_LOST_DEFAULT);

                foreach (ESMoL.IComponent component in dataHandlerMap.Values)
                {
                    foreach (ESMoL.IMessageRef mr in component.MessageRef_children.Where(x => x.name.EndsWith("Out")))
                    {
                        string baseMessageName = mr.name.Substring(0, mr.name.Length - 4);
                        ESMoL.IMessageRef inmr = component.MessageRef_children.Where(x => x.name == baseMessageName + "_In").FirstOrDefault();
                        ESMoL.IMessage targetMessage;
                        if (!importedMessageMap.TryGetValue(baseMessageName, out targetMessage))
                        {
                            GMEConsole.Warning.WriteLine("Warning: could not find Message " + baseMessageName + " in imported model Messages " + String.Join(", ", importedMessageMap.Keys.ToArray()));
                            continue;
                        }
                        inmr.@ref = targetMessage;
                        mr.@ref = targetMessage;

                        if (mr.dstDependency_targets.Count() > 0)
                            foreach (ESMoL.IMsgPort msgPort in mr.dstDependency_targets.First().@ref.children.GetUdmInstances(ESMoLImpl.MsgPort.meta, ESMoLImpl.MsgPort.ICast))
                            {
                                ESMoL.ITypeBaseRef typeRef = msgPort.dstConnector
                                    .Where(x => x.parent.Equals(mr.dstDependency_targets.First().parent))
                                    .Select(x => x.dstConnector)
                                    .First().children.GetUdmInstances(ESMoLImpl.TypeBaseRef.meta, ESMoLImpl.TypeBaseRef.ICast).FirstOrDefault();
                                ESMoL.ISubsystem passThrough;
                                if (!passThroughsByType.TryGetValue(typeRef.@ref, out passThrough))
                                {
                                    HashSet<Udm.Native.UdmObject> origObjects = new HashSet<Udm.Native.UdmObject>();
                                    origObjects.UnionWith(ESMoLDN.GetRootObject().GetChildObjects());
                                    Udm.Native.Udm.CopyObjectHierarchy(DataHandler_Passthrough.GetRootObject(), ESMoLDN.GetRootObject());
                                    HashSet<Udm.Native.UdmObject> newObjects = new HashSet<Udm.Native.UdmObject>();
                                    newObjects.UnionWith(ESMoLDN.GetRootObject().GetChildObjects());
                                    newObjects.ExceptWith(origObjects);

                                    passThrough = ESMoLImpl.Subsystem.Cast(ESMoLImpl.DesignFolder.Cast(newObjects.First()).children.First().children.First().children.First());
                                    passThrough.name = GetUniqueName(passThrough.name + "_" + typeRef.@ref.name);

                                    ESMoL.IInputPort input = ESMoLImpl.InputPort.Create(passThrough);
                                    input.name = "In";
                                    ESMoL.ITypeBaseRef inputType = ESMoLImpl.TypeBaseRef.Create(input);
                                    inputType.@ref = typeRef.@ref;
                                    inputType.name = typeRef.name;

                                    ESMoL.IOutputPort output = ESMoLImpl.OutputPort.Create(passThrough);
                                    output.name = "Out";
                                    ESMoL.ITypeBaseRef outputType = ESMoLImpl.TypeBaseRef.Create(output);
                                    outputType.@ref = typeRef.@ref;
                                    outputType.name = typeRef.name;

                                    ESMoL.ILine line = ESMoLImpl.Line.Create(passThrough);
                                    line.srcLine = input;
                                    line.dstLine = output;

                                    passThroughsByType.Add(typeRef.@ref, passThrough);
                                }
                                ESMoL.ISubsystemRef subRef = ESMoLImpl.SubsystemRef.Create(component);
                                subRef.@ref = passThrough;
                                subRef.name = GetUniqueName(passThrough.name);

                                ESMoL.IOutputPort slOut = passThrough.children.GetUdmInstances(ESMoLImpl.OutputPort.meta, ESMoLImpl.OutputPort.ICast).FirstOrDefault();
                                ESMoL.IInputPort slIn = passThrough.children.GetUdmInstances(ESMoLImpl.InputPort.meta, ESMoLImpl.InputPort.ICast).FirstOrDefault();
                                ESMoL.IConnector connectorOut = ESMoLImpl.Connector.Create(component);
                                connectorOut.dstConnector__rp_helper = mr;
                                connectorOut.dstConnector = msgPort;
                                connectorOut.srcConnector__rp_helper = subRef;
                                connectorOut.srcConnector = slOut;

                                ESMoL.IConnector connectorIn = ESMoLImpl.Connector.Create(component);
                                connectorIn.srcConnector__rp_helper = inmr;
                                connectorIn.srcConnector = msgPort;
                                connectorIn.dstConnector__rp_helper = subRef;
                                connectorIn.dstConnector = slIn;
                            }
                    }
                }
                DataHandler_Passthrough.CloseNoUpdate();
            }

            // If, in design, two ComponentRefs pass messages, but are deployed to different Nodes, we need to say which Bus the message goes across
            {
                foreach (ESMoL.IDependency dependency in design.Dependency_children)
                {
                    ESMoL.IComponentRef srcComponent = dependency.srcDependency__rp_helper;
                    ESMoL.INodeRef srcNode = srcComponent.dstComponentAssignment_targets.FirstOrDefault();
                    ESMoL.IComponentRef dstComponent = dependency.dstDependency__rp_helper;
                    ESMoL.INodeRef dstNode = dstComponent.dstComponentAssignment_targets.FirstOrDefault();
                    if (srcNode != null && dstNode != null && !srcNode.Equals(dstNode))
                    {
                        foreach (ESMoL.IBChan srcBChan in srcNode.@ref.children.GetUdmInstances(new ESMoLImpl.BChan.Factory()))
                        {
                            ESMoL.IBChan dstBChan = GetClosureBi((ESMoL.IConnectable)srcBChan, x => x.dstWire_targets, x => x.srcWire_targets)
                                .GetUdmInstances(new ESMoLImpl.BChan.Factory())
                                .Where(x => x.parent.Equals(dstNode.@ref)).FirstOrDefault();
                            if (dstBChan != null)
                            {
                                ESMoL.ICommMapping srcMapping = ESMoLImpl.CommMapping.Create(design);
                                srcMapping.srcCommMapping__rp_helper = srcComponent;
                                srcMapping.srcCommMapping = dependency.srcDependency;
                                srcMapping.dstCommMapping__rp_helper = srcNode;
                                srcMapping.dstCommMapping = srcBChan;

                                ESMoL.ICommMapping dstMapping = ESMoLImpl.CommMapping.Create(design);
                                dstMapping.srcCommMapping__rp_helper = dstNode;
                                dstMapping.srcCommMapping = dstBChan;
                                dstMapping.dstCommMapping__rp_helper = dstComponent;
                                dstMapping.dstCommMapping = dependency.dstDependency;

                                // TODO: Warn if this mapping is ambiguous
                            }
                        }
                    }
                }
            }

            Dictionary<CyPhyML.ISignalConnection, int> ichanPortMap = new Dictionary<CyPhyML.ISignalConnection, int>();
            Dictionary<CyPhyML.ISignalConnection, int> ochanPortMap = new Dictionary<CyPhyML.ISignalConnection, int>();
            foreach (var node in nodes)
            {
                foreach (CyPhyML.IElectricalSignalPort port in node.Key.children.GetUdmInstances(new CyPhyMLImpl.ElectricalSignalPort.Factory()))
                {
                    ESMoL.IConnectable esmolChan;
                    if (!portMap.TryGetValue(port, out esmolChan))
                        continue;

                    if (esmolChan.type.Equals(ESMoLImpl.OChan.meta))
                    {
                        int? portnum = null;
                        CyPhyML.IElectricalSignalPort dstport = port.dstInformationFlow_targets.GetUdmInstances(new CyPhyMLImpl.ElectricalSignalPort.Factory()).FirstOrDefault();
                        while (dstport != null)
                        {
                            dstport = dstport.dstInformationFlow_targets.GetUdmInstances(new CyPhyMLImpl.ElectricalSignalPort.Factory()).FirstOrDefault();
                            
                        }
                        if (dstport != null)
                        {
                            foreach (CyPhyML.IElectricalSignalPort srcport in GetClosure((CyPhyML.ISignalPortType)dstport, x => x.srcInformationFlow_targets)
                                .Where(x => !x.Equals(port))
                                .GetUdmInstances(new CyPhyMLImpl.ElectricalSignalPort.Factory()))
                            {
                                foreach (CyPhyML.ISignal endpoint in srcport.srcSignalPort2PhysicalPort_targets
                                    .Cast<CyPhyML.ISignal>()
                                    .GetClosure(x => x.srcSignalConnection_targets))
                                {
                                    if (portnum != null)
                                    { // Warn?
                                    }
                                    else if (endpoint.parent.type.Equals(CyPhyMLImpl.SimulinkModel.meta) && endpoint.type.Equals(CyPhyMLImpl.OutSignal.meta))
                                    {
                                        CyPhyML.IOutSignal simulinkPort = CyPhyMLImpl.OutSignal.Cast(endpoint);
                                        CyPhyML.ISignalConnection signal = simulinkPort.dstSignalConnection.FirstOrDefault();
                                        if (signal != null)
                                        {
                                            // GMEConsole.Out.WriteLine(simulinkPort.name + "PortNum " + signal.OutputPortNumber);
                                            if (signal.OutputPortNumber == "")
                                                ThrowWarning(x => "Warning: " + x + " has no Output Port Number", signal);
                                            portnum = ochanPortMap.Count + 1;
                                            ESMoLImpl.OChan.Cast(esmolChan).ChanNum = (int)portnum;
                                            // FIXME: why is OutputPortNumber a string?
                                            ochanPortMap[signal] = (int)portnum;
                                        }
                                    }
                                }
                            }
                        }
                        if (portnum == null)
                            ThrowWarning(x => "Couldn't find matching Simulink block for " + x, port);
                    }
                    else if (esmolChan.type.Equals(ESMoLImpl.IChan.meta))
                    {
                        // FIXME: this is essentially copy/paste of above, with s/dst/src/ s/src/dst/ s/CompOutSignal/CompInSignal/ s/OutSignal/InSignal/
                        int? portnum = null;
                        CyPhyML.IElectricalSignalPort dstport = port.srcInformationFlow_targets.GetUdmInstances(new CyPhyMLImpl.ElectricalSignalPort.Factory()).FirstOrDefault();
                        while (dstport != null)
                        {
                            dstport = dstport.srcInformationFlow_targets.GetUdmInstances(new CyPhyMLImpl.ElectricalSignalPort.Factory()).FirstOrDefault();
                            
                        }
                        if (dstport != null)
                        {
                            foreach (CyPhyML.IElectricalSignalPort srcport in GetClosure((CyPhyML.ISignalPortType)dstport, x => x.dstInformationFlow_targets)
                                .Where(x => !x.Equals(port))
                                .GetUdmInstances(new CyPhyMLImpl.ElectricalSignalPort.Factory()))
                            {
                                foreach (CyPhyML.ISignal endpoint in srcport.dstPhysicalPort2SignalPort_targets
                                    .Cast<CyPhyML.ISignal>()
                                    .GetClosure(x => x.dstSignalConnection_targets))
                                {
                                    if (portnum != null)
                                    { // Warn?
                                    }
                                    else if (endpoint.parent.type.Equals(CyPhyMLImpl.SimulinkModel.meta) && endpoint.type.Equals(CyPhyMLImpl.InSignal.meta))
                                    {
                                        CyPhyML.IInSignal simulinkPort = CyPhyMLImpl.InSignal.Cast(endpoint);
                                        CyPhyML.ISignalConnection signal = simulinkPort.srcSignalConnection.FirstOrDefault();
                                        if (signal != null)
                                        {
                                            // GMEConsole.Out.WriteLine(simulinkPort.name + "PortNum " + signal.InputPortNumber);
                                            if (signal.InputPortNumber == "")
                                                ThrowWarning(x => "Warning: " + x + " has no Input Port Number", signal);
                                            portnum = ichanPortMap.Count + 1;
                                            ESMoLImpl.IChan.Cast(esmolChan).ChanNum = (int)portnum;
                                            // FIXME: why is InputPortNumber a string?
                                            ichanPortMap[signal] = (int)portnum;
                                        }
                                    }
                                }
                            }
                        }
                        if (portnum == null)
                            ThrowWarning(x => "Couldn't find matching Simulink block for " + x, port);
                    }
                }
            }


            Dictionary<CyPhyML.ISimulinkModel, string> simulinkModelMap = new Dictionary<CyPhyML.ISimulinkModel, string>();
            string buildScriptFile = System.IO.Path.Combine(outdir, assembly.name + "_ESMoL_build.m");
            componentParameters["BuildScript"] = System.IO.Path.GetFileNameWithoutExtension(buildScriptFile);
            using (System.IO.StreamWriter mfile = new System.IO.StreamWriter(buildScriptFile))
            {
                mfile.WriteLine("VCP_DIR = getenv('VCP_PATH');");
                mfile.WriteLine("if strcmp(VCP_DIR, '')");
                mfile.WriteLine("  VCP_DIR = winqueryreg('HKEY_LOCAL_MACHINE', 'Software\\ESMoL', 'VCP_PATH');");
                mfile.WriteLine("end");
                mfile.WriteLine("addpath( [VCP_DIR,'\\bin'] );");
                mfile.WriteLine("init_ttenv");
                mfile.WriteLine();
                mfile.WriteLine("load_system('simulink');");
                mfile.WriteLine("load_system('truetime');");
                string systemPath = assembly.name + "_ESMoL_tt";
                mfile.WriteLine(String.Format("close_system('{0}', 0);", systemPath));
                mfile.WriteLine("%simulink('open');");
                mfile.WriteLine(String.Format("new_system('{0}');", systemPath));

                string esmolWrapper = systemPath + "/ESMoL_Nodes";
                componentParameters["OutputBlock"] = esmolWrapper;
                mfile.WriteLine(String.Format("add_block('built-in/Subsystem', '{0}');", esmolWrapper));
                string esmolNodes = esmolWrapper + "/TTNodes";
                mfile.WriteLine(String.Format("add_block('built-in/Subsystem', '{0}');", esmolNodes));
                mfile.WriteLine(String.Format("% Call ESMoL functions for nodes with path '{0}'", esmolNodes));
                mfile.WriteLine(String.Format("addpath('{0}')", System.IO.Path.GetFileNameWithoutExtension(outfile) + "_ESMoL" + "/" + "target_truetime"));
                foreach (var node in nodes)
                {
                    mfile.WriteLine(String.Format("{0}_gen_{1}('{2}')", System.IO.Path.GetFileNameWithoutExtension(outfile), node.Value.name, esmolNodes));
                    mfile.WriteLine(String.Format("ttmex {0}/target_truetime/{1}_init.cpp", System.IO.Path.GetFileNameWithoutExtension(outfile) + "_ESMoL", node.Value.name));
                }
                mfile.WriteLine();
                mfile.WriteLine(String.Format("% Call ESMoL functions for buses with path '{0}'", esmolNodes));
                foreach (var bus in buses)
                {
                    mfile.WriteLine(String.Format("{0}_gen_{1}('{2}')", System.IO.Path.GetFileNameWithoutExtension(outfile), bus.Value.name, esmolNodes));
                }
                mfile.WriteLine();
                foreach (var portType in new[] { new { PortMap = ichanPortMap, In="In", IChan="IChan", Inport="Inport", From="From",
                                            AddLine="add_line('{0}', '{1}', '{2}', 'autorouting', 'on');"},
                                            new { PortMap = ochanPortMap, In="Out", IChan="OChan", Inport="Outport", From="Goto",
                                            AddLine="add_line('{0}', '{2}', '{1}', 'autorouting', 'on');"} })
                {
                    foreach (var ichan in portType.PortMap)
                    {
                        string portPath = esmolNodes + "/" + portType.In + ichan.Value;
                        mfile.WriteLine(String.Format("%add_block('built-in/{0}', '{1}');", portType.Inport, portPath));
                        mfile.WriteLine(String.Format("%set_param('{0}', 'Port','{1}');", portPath, ichan.Value));

                        string fromPath = esmolWrapper + "/" + portType.IChan + ichan.Value;
                        mfile.WriteLine(String.Format("add_block('simulink/Signal Routing/{0}', '{1}');", portType.From, fromPath));
                        mfile.WriteLine(String.Format("set_param('{0}', 'TagVisibility', 'global', 'GotoTag', '{1}');", fromPath, portType.IChan + ichan.Value));
                        mfile.WriteLine(String.Format(portType.AddLine, esmolWrapper, GetSLPort(fromPath, 1), GetSLPort(esmolNodes, ichan.Value)));
                        mfile.WriteLine();
                    }
                }
                mfile.WriteLine();

                foreach (CyPhyML.ISimulinkModel simulinkModel in ichanPortMap.Keys.Select(x => x.dstSignalConnection.parent).Union(ochanPortMap.Keys.Select(x => x.srcSignalConnection.parent))
                    .GetUdmInstances(new CyPhyMLImpl.SimulinkModel.Factory()))
                {
                    string matlabPath = systemPath + "/" + GetUniqueName(simulinkModel.name);
                    GMEConsole.Out.WriteLine("Redirecting " + simulinkModel.name + " to " + matlabPath);
                    simulinkModel.SimulinkModelName = matlabPath;

                    simulinkModelMap[simulinkModel] = matlabPath;
                    mfile.WriteLine(String.Format("add_block('built-in/Subsystem', '{0}');", matlabPath));
                }
                foreach (var portType in new[] { new { PortMap = ichanPortMap, In="In", IChan="IChan", Inport="Inport", Goto="Goto",
                                            AddLine="add_line('{0}', '{1}', '{2}', 'autorouting', 'on');"},
                                            new { PortMap = ochanPortMap, In="Out", IChan="OChan", Inport="Outport", Goto="From",
                                            AddLine="add_line('{0}', '{2}', '{1}', 'autorouting', 'on');"} })
                {
                    foreach (var ichan in portType.PortMap)
                    {
                        CyPhyML.ISimulinkModel simulinkModel = CyPhyMLImpl.SimulinkModel.ICast((portType.In == "In" ? ichan.Key.dstSignalConnection : ichan.Key.srcSignalConnection).parent);
                        string portNumber = (portType.In == "In" ? ichan.Key.InputPortNumber : ichan.Key.OutputPortNumber);
                        string model = simulinkModelMap[simulinkModel];
                        string portPath = model + "/" + portType.In + portNumber;
                        mfile.WriteLine(String.Format("add_block('built-in/{0}', '{1}');", portType.Inport, portPath));
                        mfile.WriteLine(String.Format("set_param('{0}', 'Port','{1}');", portPath, portNumber));
                        string gotoPath = model + "/" + portType.IChan + ichan.Value;
                        mfile.WriteLine(String.Format("add_block('simulink/Signal Routing/{0}', '{1}');", portType.Goto, gotoPath));
                        mfile.WriteLine(String.Format("set_param('{0}', 'TagVisibility', 'global', 'GotoTag', '{1}');", gotoPath, portType.IChan + ichan.Value));
                        mfile.WriteLine(String.Format(portType.AddLine, model, GetSLPort(portPath, 1), GetSLPort(gotoPath, 1)));
                        mfile.WriteLine();
                    }
                }
                mfile.WriteLine(String.Format("save_system('{0}');", systemPath));
                mfile.WriteLine(String.Format("close_system('{0}');", systemPath));
                mfile.WriteLine(String.Format("% open_system('{0}');", systemPath));
            }
        }
        /*
        private string GetCyberParameter(CyPhyML.IMgaObject cpComp, string name, string def = "")
        {
            Dictionary<string, string> siPrefixes = new Dictionary<string, string>()
            {
                { "peta", "P" },
                { "tera", "T" },
                { "giga", "G" },
                { "mega", "M" },
                { "kilo", "k" },
                { "hecto", "h" },
                { "", "" },
                { "centi", "c" },
                { "milli", "m" },
                { "micro", "μ" },
                { "nano", "n" },
                { "pico", "p" },
            };
            Dictionary<string, string> siUnits = new Dictionary<string, string>()
            {
                { "", "" },
                { "second", "s" },
                { "byte", "B" },
                { "bit", "b" },
            };

            CyPhyML.ICyberParameter ExecPeriod;
            ExecPeriod = cpComp.children.GetUdmInstances(new CyPhyMLImpl.CyberParameter.Factory())
                .Where(x => x.name == name).FirstOrDefault();
            
            if (ExecPeriod == null)
            {
                ThrowWarning(x => "Warning: " + x + " has no " + name + " CyberParameter. Using default " + def, cpComp);
                return def;
            }
            if (ExecPeriod.@ref == null)
            {
                ThrowWarning(x => "Warning: " + x + " is a null reference", ExecPeriod);
                return def;
            }
            string unitname = CyPhyMLImpl.MgaObject.ICast(ExecPeriod.@ref).name;
            string prefix = null;
            foreach (KeyValuePair<string, string> entry in siPrefixes)
            {
                if (unitname.ToLower().StartsWith(entry.Key))
                    prefix = entry.Key;
            }
            if (prefix == null)
            {
                ThrowWarning(x => "Warning: " + x + " has an unknown prefix", ExecPeriod);
                return def;
            }
            string siunit;
            if (!siUnits.TryGetValue(unitname.ToLower().Substring(prefix.Length), out siunit))
            {
                ThrowWarning(x => "Warning: " + x + " has an unknown unit", ExecPeriod);
                return def;
            }
            return ExecPeriod.Value + siPrefixes[prefix] + siunit;
        }*/

        private bool HasESMoLLink(CyPhyML.IMgaObject x)
        {
            // TODO: memoize
            foreach (Udm.IUdmObject o in getSelfAndAncestors(x))
            {
                if (o.children.GetUdmInstances(new CyPhyMLImpl.ESMoLLink.Factory()).Count() != 0)
                    return true;
            }
            return false;
        }

        private string GetUniqueNonKeywordname(string p)
        {
            // TODO: complete this list
            HashSet<string> ESMoL_keywords = new HashSet<string>() { "Bus" };
            while (ESMoL_keywords.Contains(p))
                p = p + "_";
            return GetUniqueName(p);
        }

        private string GetSLPort(string fromPath, int p)
        {
            string[] path = fromPath.Split('/');
            return path.Last() + "/" + p;
        }

        private void UpgradeMgaToLatestParadigm(string mgaPath, string paradigm)
        {
            GME.Util.IMgaRegistrar reg = new GME.Util.MgaRegistrar();
            string metaConnStr;
            object guid = null;
            reg.QueryParadigm(paradigm, out metaConnStr, ref guid, GME.Util.regaccessmode_enum.REGACCESS_BOTH);

            GME.MGA.IMgaProject project = new GME.MGA.MgaProject();
            project.OpenEx("MGA=" + mgaPath, paradigm, guid);
            project.Close();
        }

        private static string ExtractName(Udm.IUdmObject x)
        {
            return Udm.Native.Udm.ExtractName(((Udm.Native.UdmCliObject)x).backing);
        }

        private static string ExtractName(Udm.Native.UdmObject x)
        {
            return Udm.Native.Udm.ExtractName(x);
        }

        public static string GetLink(Udm.IUdmObject o, string linkText = null)
        {
            if (linkText == null)
            {
                linkText = GetPath(o);
            }
            string id = CyPhyMLImpl.MgaObject.Cast(o).GmeId;
            return "<a href=\"mga:" + id + "\">"
                + linkText
                + "</a>";
        }

        public static string GetPath(Udm.IUdmObject o)
        {
            return ((Udm.Native.UdmCliObject)o).backing.getPath2("/");
        }

        public void ThrowError(Func<string, string> f, CyPhyML.IMgaObject a)
        {
            GMEConsole.Error.WriteLine(f(GetLink(a)));
            throw new Exception(f(GetPath(a)));
        }

        public void ThrowWarning(Func<string, string> f, CyPhyML.IMgaObject a)
        {
            GMEConsole.Warning.WriteLine(f(GetLink(a)));
        }

        public void ThrowError(Func<string, string, string> f, CyPhyML.IMgaObject a, CyPhyML.IMgaObject b)
        {
            GMEConsole.Error.WriteLine(f(GetLink(a), GetLink(b)));
            throw new Exception(f(GetPath(a), GetPath(b)));
        }

        public static IEnumerable<TResult> GetClosure<TResult>(TResult source,
            Func<TResult, IEnumerable<TResult>> f)
        {
            return f(source).GetClosure(f);
        }

        public static IEnumerable<TResult> GetClosureBi<TResult>(TResult source,
            Func<TResult, IEnumerable<TResult>> f, Func<TResult, IEnumerable<TResult>> f2)
        {
            return f(source).GetClosureBi(source, f, f2).Concat(f2(source).GetClosureBi(source, f, f2));
        }

        public void ThrowWarning(Func<string, string, string> f, CyPhyML.IMgaObject a, CyPhyML.IMgaObject b)
        {
            GMEConsole.Warning.WriteLine(f(GetLink(a), GetLink(b)));
        }

        private Dictionary<string, int> uniqueNameMap = new Dictionary<string, int>();
        public string GetUniqueName(string orig)
        {
            int n;
            if (uniqueNameMap.TryGetValue(orig, out n))
            {
                uniqueNameMap[orig] = n + 1;
                return orig + "_" + n;
            }
            else
            {
                uniqueNameMap[orig] = 2;
                return orig;
            }
        }

        public string GetUniqueName(string desired, ESMoL.IMgaObject parent, int i = 0)
        {
            string uniq = desired;
            if (i != 0)
                uniq = uniq + "_" + i;
            if (parent.children.Select(ExtractName).FirstOrDefault(x => x == uniq) == null)
            {
                return uniq;
            }
            else
            {
                return GetUniqueName(desired, parent, i + 1);
            }
        }

    }
}
