/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿namespace CyPhy2Modelica.BondGraph
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;

    // using generic common interface
    using Common = ISIS.GME.Common;

    // using domain specific interfaces
    using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
    using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;

    /// <summary>
    /// Represents a power chain in GME.
    /// </summary>
    public class PowerChain
    {
        /// <summary>
        /// Contains all power chain in the model.
        /// </summary>
        private static List<PowerChain> powerChains = new List<PowerChain>();

        /// <summary>
        /// Contains a map of a GME (power) connection and its parent Modelica model object.
        /// </summary>
        private static Dictionary<Common.Interfaces.Connection, Modelica.Model> connectionParents =
            new Dictionary<Common.Interfaces.Connection, Modelica.Model>();

        /// <summary>
        /// Initializes a new instance of the <see cref="PowerChain" /> class.
        /// </summary>
        /// <param name="srcItem">Start element (power port) of the chain.</param>
        private PowerChain(Common.Interfaces.Base srcItem = null)
        {
            this.ChainItems = new List<Common.Interfaces.Base>();
            this.Connections = new List<Common.Interfaces.Connection>();
            if (srcItem != null)
            {
                this.ChainItems.Add(srcItem);
            }
        }

        /// <summary>
        /// Initializes a new instance of the <see cref="PowerChain" /> class.
        /// The new instance is based on an existing instance.
        /// Copies all the properties except the Destination Junction.
        /// </summary>
        /// <param name="currentChain">Chain to copy</param>
        private PowerChain(PowerChain currentChain)
        {
            this.ChainItems = new List<Common.Interfaces.Base>();
            this.Connections = new List<Common.Interfaces.Connection>();
            this.ChainItems.AddRange(currentChain.ChainItems);
            this.Connections.AddRange(currentChain.Connections);
            this.SrcJunction = currentChain.SrcJunction;
        }

        /// <summary>
        /// Gets all power chain in the model.
        /// </summary>
        public static List<PowerChain> PowerChains
        {
            get { return powerChains; }
        }

        /// <summary>
        /// Gets a map of a GME (power) connection and its parent Modelica model object.
        /// </summary>
        public static Dictionary<Common.Interfaces.Connection, Modelica.Model> ConnectionParents
        {
            get { return connectionParents; }
        }

        /// <summary>
        /// Gets or sets a list of power ports, which are in this chain.
        /// </summary>
        public List<Common.Interfaces.Base> ChainItems { get; set; }

        /// <summary>
        /// Gets or sets all connections between the source and the destination
        /// junction among the power chain.
        /// </summary>
        public List<Common.Interfaces.Connection> Connections { get; set; }

        /// <summary>
        /// Gets or sets source Junction of this power chain.
        /// </summary>
        public CyPhy.Junction SrcJunction { get; set; }

        /// <summary>
        /// Gets or sets destination Junction of this power chain.
        /// </summary>
        public CyPhy.Junction DstJunction { get; set; }

        /// <summary>
        /// Creates bond connections in the Modelica model based on the Power Chains and the 
        /// Connection Parents objects.
        /// </summary>
        public static List<GmeMessage> BuildPowerConnections()
        {
            var messages = new List<GmeMessage>();

            foreach (var item in ConnectionParents)
            {
                if (item.Key is CyPhy.PowerFlow ||
                    item.Key is CyPhy.AggregatePortComposition ||
                    item.Key is CyPhy.AggregatePortMap ||
                    item.Key is CyPhy.ModelicaBusPortMap ||
                    item.Key is CyPhy.SignalFlowBusPortMap)
                {
                    // if AggregatePort is in a modelica model it is a specaial case
                    // due modelica model does not appear in item.Value.Models
                    // this block handles the connections among the AggregatePort connectors
                    // within a component (between a component external port and
                    // a modelica model port)
                    ISIS.GME.Common.Interfaces.FCO srcAggregate = null;
                    ISIS.GME.Common.Interfaces.FCO dstAggregate = null;

                    if (item.Key is CyPhy.AggregatePortComposition)
                    {
                        srcAggregate = (item.Key as CyPhy.AggregatePortComposition).SrcEnds.AggregatePort;
                        dstAggregate = (item.Key as CyPhy.AggregatePortComposition).DstEnds.AggregatePort;
                        string srcModelicaUri = (srcAggregate as CyPhy.AggregatePort).Attributes.Type;
                        string dstModelicaUri = (dstAggregate as CyPhy.AggregatePort).Attributes.Type;

                        if (srcModelicaUri != dstModelicaUri)
                        {
                            if ((srcModelicaUri == "Modelica.Blocks.Interfaces.RealOutput" && dstModelicaUri == "Modelica.Blocks.Interfaces.RealInput") ||
                                (srcModelicaUri == "Modelica.Blocks.Interfaces.RealInput" && dstModelicaUri == "Modelica.Blocks.Interfaces.RealOutput"))
                            {
                                messages.Add(new GmeMessage(string.Format("Ports: {0} and {1}, connected through {2}, do not have the same type. You should however rather use In/Output Signals.",
                                    srcAggregate.Name, dstAggregate.Name, GmeConsoleHelper.ToHyperLink(item.Key)), GmeMessage.Severity_enum.Warning));
                            }
                            else
                            {
                                var handled = false;
                                try
                                {
                                    var middleNameSrc = srcModelicaUri.Split('.').ToList();
                                    var middleNameDst = dstModelicaUri.Split('.').ToList();

                                    if (middleNameSrc.Count == middleNameDst.Count &&
                                        middleNameDst.Count > 2)
                                    {
                                        var valid = true;
                                        for (int i = 1; i < middleNameDst.Count - 1; i++)
                                        {
                                            if (middleNameDst[i] != middleNameSrc[i])
                                            {
                                                valid = false;
                                                break;
                                            }
                                        }
                                        if (valid)
                                        {
                                            handled = true;
                                            messages.Add(new GmeMessage(string.Format("Ports: {0} and {1}, connected through {2}, do not have the same (Attributes/)Type.",
                                            srcAggregate.Name, dstAggregate.Name, GmeConsoleHelper.ToHyperLink(item.Key)), GmeMessage.Severity_enum.Warning));
                                        }
                                    }
                                }
                                catch
                                {
                                    // just ignore if any
                                }

                                if (handled == false)
                                {
                                    messages.Add(new GmeMessage(string.Format("Ports: {0} and {1}, connected through {2}, do not have the same (Attributes/)Type.",
                                        srcAggregate.Name, dstAggregate.Name, GmeConsoleHelper.ToHyperLink(item.Key)), GmeMessage.Severity_enum.Error));
                                }
                            }
                        }
                    }
                    else if (item.Key is CyPhy.AggregatePortMap)
                    {
                        srcAggregate = (item.Key as CyPhy.AggregatePortMap).SrcEnd;
                        dstAggregate = (item.Key as CyPhy.AggregatePortMap).DstEnd;
                        string srcModelicaUri;
                        string dstModelicaUri;
                        if (srcAggregate is CyPhy.ModelicaAggregateInterface)
                        {
                            srcModelicaUri = (srcAggregate as CyPhy.ModelicaAggregateInterface).Attributes.Type;
                            dstModelicaUri = (dstAggregate as CyPhy.AggregatePort).Attributes.Type;
                        }
                        else
                        {
                            srcModelicaUri = (srcAggregate as CyPhy.AggregatePort).Attributes.Type;
                            dstModelicaUri = (dstAggregate as CyPhy.ModelicaAggregateInterface).Attributes.Type;
                        }
                        if (srcModelicaUri != dstModelicaUri)
                        {
                            var handled = false;
                            try
                            {
                                var middleNameSrc = srcModelicaUri.Split('.').ToList();
                                var middleNameDst = dstModelicaUri.Split('.').ToList();

                                if (middleNameSrc.Count == middleNameDst.Count &&
                                    middleNameDst.Count > 2)
                                {
                                    var valid = true;
                                    for (int i = 1; i < middleNameDst.Count - 1; i++)
                                    {
                                        if (middleNameDst[i] != middleNameSrc[i])
                                        {
                                            valid = false;
                                            break;
                                        }
                                    }
                                    if (valid)
                                    {
                                        handled = true;
                                        messages.Add(new GmeMessage(string.Format("Ports: {0} and {1}, connected through {2}, do not have the same (Attributes/)Type.",
                                        srcAggregate.Name, dstAggregate.Name, GmeConsoleHelper.ToHyperLink(item.Key)), GmeMessage.Severity_enum.Warning));
                                    }
                                }
                            }
                            catch
                            {
                                // just ignore if any
                            }

                            if (handled == false)
                            {
                                messages.Add(new GmeMessage(string.Format("Ports: {0} and {1}, connected through {2}, do not have the same (Attributes/)Type.",
                                    srcAggregate.Name, dstAggregate.Name, GmeConsoleHelper.ToHyperLink(item.Key)), GmeMessage.Severity_enum.Error));
                            }
                        }
                    }
                    else if (item.Key is CyPhy.ModelicaBusPortMap)
                    {
                        srcAggregate = (item.Key as CyPhy.ModelicaBusPortMap).SrcEnd;
                        dstAggregate = (item.Key as CyPhy.ModelicaBusPortMap).DstEnd;
                        string srcModelicaUri;
                        string dstModelicaUri;
                        if (srcAggregate is CyPhy.BusPort)
                        {
                            srcModelicaUri = (srcAggregate as CyPhy.BusPort).Attributes.Type;
                            dstModelicaUri = (dstAggregate as CyPhy.ModelicaBusPortInterface).Attributes.Type;
                        }
                        else
                        {
                            srcModelicaUri = (srcAggregate as CyPhy.ModelicaBusPortInterface).Attributes.Type;
                            dstModelicaUri = (dstAggregate as CyPhy.BusPort).Attributes.Type;
                        }
                        if (srcModelicaUri != dstModelicaUri)
                        {
                            var handled = false;
                            try
                            {
                                var middleNameSrc = srcModelicaUri.Split('.').ToList();
                                var middleNameDst = dstModelicaUri.Split('.').ToList();

                                if (middleNameSrc.Count == middleNameDst.Count &&
                                    middleNameDst.Count > 2)
                                {
                                    var valid = true;
                                    for (int i = 1; i < middleNameDst.Count - 1; i++)
                                    {
                                        if (middleNameDst[i] != middleNameSrc[i])
                                        {
                                            valid = false;
                                            break;
                                        }
                                    }
                                    if (valid)
                                    {
                                        handled = true;
                                        messages.Add(new GmeMessage(string.Format("Ports: {0} and {1}, connected through {2}, do not have the same (Attributes/)Type.",
                                        srcAggregate.Name, dstAggregate.Name, GmeConsoleHelper.ToHyperLink(item.Key)), GmeMessage.Severity_enum.Warning));
                                    }
                                }
                            }
                            catch
                            {
                                // just ignore if any
                            }

                            if (handled == false)
                            {
                                messages.Add(new GmeMessage(string.Format("Ports: {0} and {1}, connected through {2}, do not have the same (Attributes/)Type.",
                                    srcAggregate.Name, dstAggregate.Name, GmeConsoleHelper.ToHyperLink(item.Key)), GmeMessage.Severity_enum.Error));
                            }
                        }
                    }
                    else if (item.Key is CyPhy.SignalFlowBusPortMap)
                    {
                        srcAggregate = (item.Key as CyPhy.SignalFlowBusPortMap).GenericSrcEnd;
                        dstAggregate = (item.Key as CyPhy.SignalFlowBusPortMap).GenericDstEnd;
                    }

                    var srcModelicaModelParent = srcAggregate != null ? srcAggregate.ParentContainer : null;
                    var dstModelicaModelParent = dstAggregate != null ? dstAggregate.ParentContainer : null;

                    // ASSUMPTION: only one end of the connection is buried
                    // within a modelica model wrapper
                    if (srcModelicaModelParent != null &&
                        (srcModelicaModelParent.Impl.MetaBase.Name == typeof(CyPhy.ModelicaModel).Name ||
                        srcModelicaModelParent.Impl.MetaBase.Name == typeof(CyPhy.Environment).Name ||
                        srcModelicaModelParent.Impl.MetaBase.Name == typeof(CyPhy.SignalFlowModel).Name))
                    {
                        // new connection
                        Modelica.Connection conn = new Modelica.Connection();

                        var srcComp = item.Value.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericSrcEnd.ParentContainer.Name);

                        var srcPort = srcComp.Ports.FirstOrDefault(x => x.Impl.ID == srcAggregate.ID);
                        srcPort.PortType = Modelica.Factory.PortType.CyPhyPower;
                        conn.Src = srcPort;

                        var dstComp = item.Value.Components.FirstOrDefault(x => x.Impl.ID == dstAggregate.ID);
                        if (dstComp != null)
                        {
                            var dstPort = new Modelica.Port(dstComp, dstAggregate);

                            dstPort.PortType = Modelica.Factory.PortType.CyPhyPower;

                            conn.Dst = dstPort;
                        }

                        if (srcPort != null &&
                            dstComp != null)
                        {
                            // add new connection into the Modelica model
                            item.Value.Connections.Add(conn);
                        }
                        else
                        {
                            // if one of the connection endpoint was not found
                            GenerateModelicaModel.Messages.Add(GmeMessage.SkippedConnection(item.Key));
                        }
                    }
                    else if (dstModelicaModelParent != null &&
                            (dstModelicaModelParent.Impl.MetaBase.Name == typeof(CyPhy.ModelicaModel).Name ||
                             dstModelicaModelParent.Impl.MetaBase.Name == typeof(CyPhy.Environment).Name ||
                             dstModelicaModelParent.Impl.MetaBase.Name == typeof(CyPhy.SignalFlowModel).Name))
                    {
                        // new connection
                        Modelica.Connection conn = new Modelica.Connection();

                        var dstComp = item.Value.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericDstEnd.ParentContainer.Name);

                        var dstPort = dstComp.Ports.FirstOrDefault(x => x.Impl.ID == dstAggregate.ID);
                        dstPort.PortType = Modelica.Factory.PortType.CyPhyPower;
                        conn.Dst = dstPort;

                        var srcComp = item.Value.Components.FirstOrDefault(x => x.Impl.ID == srcAggregate.ID);
                        if (srcComp != null)
                        {
                            var srcPort = new Modelica.Port(srcComp, srcAggregate);

                            srcPort.PortType = Modelica.Factory.PortType.CyPhyPower;

                            conn.Src = srcPort;
                        }

                        if (dstPort != null &&
                            srcComp != null)
                        {
                            // add new connection into the Modelica model
                            item.Value.Connections.Add(conn);
                        }
                        else
                        {
                            // if one of the connection endpoint was not found
                            GenerateModelicaModel.Messages.Add(GmeMessage.SkippedConnection(item.Key));
                        }
                    }
                    else
                    {
                        // new connection
                        Modelica.Connection conn = new Modelica.Connection();

                        // power flows are only between components
                        Modelica.Component srcComp = null;
                        if (item.Key.ParentContainer.ID == item.Key.GenericSrcEnd.ParentContainer.ID)
                        {
                            // find src component
                            srcComp = item.Value.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericSrcEnd.Name);

                            if (srcComp != null)
                            {
                                conn.Src = new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                };

                                conn.Src.Name = string.Empty;

                                srcComp.Ports.Add(
                                    new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                                    {
                                        PortType = Modelica.Factory.PortType.CyPhyPower
                                    });
                            }
                        }
                        else
                        {
                            // find src component's parent
                            Modelica.Model srcCompParent = item.Value.Models.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericSrcEnd.ParentContainer.Name);

                            if (srcCompParent != null)
                            {
                                // find src component
                                srcComp = srcCompParent.Components.FirstOrDefault(
                                    x => x.OriginalName == item.Key.GenericSrcEnd.Name);

                                if (srcComp != null)
                                {
                                    conn.Src = new Modelica.Port(srcCompParent, item.Key.GenericSrcEnd)
                                    {
                                        PortType = Modelica.Factory.PortType.CyPhyPower
                                    };

                                    conn.Src.Name = string.Format("{0}", srcComp.Name);

                                    srcComp.Ports.Add(new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                                    {
                                        PortType = Modelica.Factory.PortType.CyPhyPower
                                    });
                                }
                            }
                        }

                        // destination connection
                        Modelica.Component dstComp = null;
                        if (item.Key.ParentContainer.ID == item.Key.GenericDstEnd.ParentContainer.ID)
                        {
                            // find destination component
                            dstComp = item.Value.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericDstEnd.Name);

                            if (dstComp != null)
                            {
                                conn.Dst = new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                };

                                conn.Dst.Name = string.Empty;

                                dstComp.Ports.Add(new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                });
                            }
                        }
                        else
                        {
                            // find destination component's parent
                            Modelica.Model dstCompParent = item.Value.Models.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericDstEnd.ParentContainer.Name);

                            if (dstCompParent != null)
                            {
                                // find destination component
                                dstComp = dstCompParent.Components.FirstOrDefault(
                                    x => x.OriginalName == item.Key.GenericDstEnd.Name);

                                if (dstComp != null)
                                {
                                    conn.Dst = new Modelica.Port(dstCompParent, item.Key.GenericDstEnd)
                                    {
                                        PortType = Modelica.Factory.PortType.CyPhyPower
                                    };

                                    conn.Dst.Name = string.Format(
                                        "{2}",
                                        Modelica.Factory.PowerPort,
                                        dstComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1,
                                        dstComp.Name);

                                    dstComp.Ports.Add(new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                                    {
                                        PortType = Modelica.Factory.PortType.CyPhyPower
                                    });
                                }
                            }
                        }

                        if (srcComp != null &&
                            dstComp != null)
                        {
                            // add new connection into the Modelica model
                            item.Value.Connections.Add(conn);
                        }
                        else
                        {
                            // if one of the connection endpoint was not found
                            GenerateModelicaModel.Messages.Add(GmeMessage.SkippedConnection(item.Key));
                        }
                    }
                }
                else if (item.Key is CyPhy.ModelicaPort2ComponentPortPowerFlow)
                {
                    // ASSUMPTIONS no port on the test bench level
                    var mp2cppf = item.Key as CyPhy.ModelicaPort2ComponentPortPowerFlow;
                    // TODO: DOES NOT WORK WITH SrcEnd and DstEnd (null reference exception) -> look at the generated API
                    //var srcParent = mp2cppf.SrcEnd.ParentContainer;
                    //var dstParent = mp2cppf.DstEnd.ParentContainer;
                    var srcParent = mp2cppf.GenericSrcEnd.ParentContainer;
                    var dstParent = mp2cppf.GenericDstEnd.ParentContainer;

                    Modelica.Connection conn = new Modelica.Connection();

                    if (srcParent.Kind == typeof(CyPhy.Environment).Name)
                    {
                        Modelica.Container srcCompParent = item.Value.Components.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericSrcEnd.ParentContainer.Name);

                        if (srcCompParent != null)
                        {
                            // find src component
                            var srcComp = srcCompParent.Ports.FirstOrDefault(
                                x => x.Impl.Name == item.Key.GenericSrcEnd.Name);

                            if (srcComp != null)
                            {
                                conn.Src = new Modelica.Port(srcCompParent, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                };

                                conn.Src.Name = string.Format("{0}", srcComp.Name);
                            }
                        }
                    }
                    else
                    {
                        Modelica.Model srcCompParent = item.Value.Models.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericSrcEnd.ParentContainer.Name);

                        if (srcCompParent != null)
                        {
                            // find src component
                            var srcComp = srcCompParent.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericSrcEnd.Name);

                            if (srcComp != null)
                            {
                                conn.Src = new Modelica.Port(srcCompParent, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                };

                                conn.Src.Name = string.Format(
                                    "{2}",
                                    Modelica.Factory.PowerPort,
                                    srcComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1,
                                    srcComp.Name);

                                srcComp.Ports.Add(new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                });
                            }
                        }
                    }

                    if (dstParent.Kind == typeof(CyPhy.Environment).Name)
                    {
                        Modelica.Container dstCompParent = item.Value.Components.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericDstEnd.ParentContainer.Name);

                        if (dstCompParent != null)
                        {
                            // find src component
                            var dstComp = dstCompParent.Ports.FirstOrDefault(
                                x => x.Impl.Name == item.Key.GenericDstEnd.Name);

                            if (dstComp != null)
                            {
                                conn.Dst = new Modelica.Port(dstCompParent, item.Key.GenericDstEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                };

                                conn.Dst.Name = string.Format("{0}", dstComp.Name);
                            }
                        }
                    }
                    else
                    {
                        Modelica.Model dstCompParent = item.Value.Models.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericDstEnd.ParentContainer.Name);

                        if (dstCompParent != null)
                        {
                            // find destination component
                            var dstComp = dstCompParent.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericDstEnd.Name);

                            if (dstComp != null)
                            {
                                conn.Dst = new Modelica.Port(dstCompParent, item.Key.GenericDstEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                };

                                conn.Dst.Name = string.Format(
                                    "{2}",
                                    Modelica.Factory.PowerPort,
                                    dstComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1,
                                    dstComp.Name);

                                dstComp.Ports.Add(new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                });
                            }
                        }
                    }

                    if (conn.Dst != null &&
                        conn.Src != null)
                    {
                        // add new connection into the Modelica model
                        item.Value.Connections.Add(conn);
                    }
                    else
                    {
                        // if one of the connection endpoint was not found
                        GenerateModelicaModel.Messages.Add(GmeMessage.SkippedConnection(item.Key));
                    }
                }
                else if (item.Key is CyPhy.PowerPort2PhysicalPort)
                {
                    // new connection
                    Modelica.Connection conn = new Modelica.Connection();

                    // power flows are only between components
                    Modelica.Component srcComp = null;
                    if (item.Key.ParentContainer.ID == item.Key.GenericSrcEnd.ParentContainer.ID)
                    {
                        // find src component
                        srcComp = item.Value.Components.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericSrcEnd.Name);

                        if (srcComp != null)
                        {
                            conn.Src = new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                            {
                                PortType = Modelica.Factory.PortType.CyPhyPower
                            };

                            conn.Src.Name = string.Format(
                                ".{0}",
                                Modelica.Factory.ModelicaPort);

                            srcComp.Ports.Add(
                                new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                });
                        }
                    }
                    else
                    {
                        // find src component's parent
                        Modelica.Model srcCompParent = item.Value.Models.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericSrcEnd.ParentContainer.Name);

                        if (srcCompParent != null)
                        {
                            // find src component
                            srcComp = srcCompParent.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericSrcEnd.Name);

                            if (srcComp != null)
                            {
                                conn.Src = new Modelica.Port(srcCompParent, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                };

                                conn.Src.Name = string.Format(
                                    "{2}.{0}",
                                    Modelica.Factory.ModelicaPort,
                                    srcComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1,
                                    srcComp.Name);

                                srcComp.Ports.Add(new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                });
                            }
                        }
                    }

                    // destination connection
                    Modelica.Component dstComp = null;
                    if (item.Key.ParentContainer.ID == item.Key.GenericDstEnd.ParentContainer.ID)
                    {
                        // find destination component
                        dstComp = item.Value.Components.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericDstEnd.Name);

                        if (dstComp != null)
                        {
                            conn.Dst = new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                            {
                                PortType = Modelica.Factory.PortType.CyPhyPower
                            };

                            conn.Dst.Name = string.Empty;

                            dstComp.Ports.Add(new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                            {
                                PortType = Modelica.Factory.PortType.CyPhyPower
                            });
                        }
                    }
                    else
                    {
                        // find destination component's parent
                        Modelica.Model dstCompParent = item.Value.Models.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericDstEnd.ParentContainer.Name);

                        if (dstCompParent != null)
                        {
                            // find destination component
                            dstComp = dstCompParent.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericDstEnd.Name);

                            if (dstComp != null)
                            {
                                conn.Dst = new Modelica.Port(dstCompParent, item.Key.GenericDstEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                };

                                conn.Dst.Name = string.Format(
                                    "{2}",
                                    Modelica.Factory.PowerPort,
                                    dstComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1,
                                    dstComp.Name);

                                dstComp.Ports.Add(new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                });
                            }
                        }
                    }

                    if (srcComp != null &&
                        dstComp != null)
                    {
                        // add new connection into the Modelica model
                        item.Value.Connections.Add(conn);
                    }
                    else
                    {
                        // if one of the connection endpoint was not found
                        GenerateModelicaModel.Messages.Add(GmeMessage.SkippedConnection(item.Key));
                    }
                }
                else if (item.Key is CyPhy.PhysicalPort2PowerPort)
                {
                    // new connection
                    Modelica.Connection conn = new Modelica.Connection();

                    // power flows are only between components
                    Modelica.Component srcComp = null;
                    if (item.Key.ParentContainer.ID == item.Key.GenericSrcEnd.ParentContainer.ID)
                    {
                        // find src component
                        srcComp = item.Value.Components.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericSrcEnd.Name);

                        if (srcComp != null)
                        {
                            conn.Src = new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                            {
                                PortType = Modelica.Factory.PortType.CyPhyPower
                            };

                            conn.Src.Name = string.Empty;

                            srcComp.Ports.Add(
                                new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                });
                        }
                    }
                    else
                    {
                        // find src component's parent
                        Modelica.Model srcCompParent = item.Value.Models.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericSrcEnd.ParentContainer.Name);

                        if (srcCompParent != null)
                        {
                            // find src component
                            srcComp = srcCompParent.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericSrcEnd.Name);

                            if (srcComp != null)
                            {
                                conn.Src = new Modelica.Port(srcCompParent, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                };

                                conn.Src.Name = string.Format(
                                    "{2}",
                                    Modelica.Factory.PowerPort,
                                    srcComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1,
                                    srcComp.Name);

                                srcComp.Ports.Add(new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                });
                            }
                        }
                    }

                    // destination connection
                    Modelica.Component dstComp = null;
                    if (item.Key.ParentContainer.ID == item.Key.GenericDstEnd.ParentContainer.ID)
                    {
                        // find destination component
                        dstComp = item.Value.Components.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericDstEnd.Name);

                        if (dstComp != null)
                        {
                            conn.Dst = new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                            {
                                PortType = Modelica.Factory.PortType.CyPhyPower
                            };

                            conn.Dst.Name = string.Format(
                                "{0}",
                                Modelica.Factory.ModelicaPort);

                            dstComp.Ports.Add(new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                            {
                                PortType = Modelica.Factory.PortType.CyPhyPower
                            });
                        }
                    }
                    else
                    {
                        // find destination component's parent
                        Modelica.Model dstCompParent = item.Value.Models.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericDstEnd.ParentContainer.Name);

                        if (dstCompParent != null)
                        {
                            // find destination component
                            dstComp = dstCompParent.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericDstEnd.Name);

                            if (dstComp != null)
                            {
                                conn.Dst = new Modelica.Port(dstCompParent, item.Key.GenericDstEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                };

                                conn.Dst.Name = string.Format(
                                    "{2}.{0}",
                                    Modelica.Factory.ModelicaPort,
                                    dstComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1,
                                    dstComp.Name);

                                dstComp.Ports.Add(new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                                {
                                    PortType = Modelica.Factory.PortType.CyPhyPower
                                });
                            }
                        }
                    }

                    if (srcComp != null &&
                        dstComp != null)
                    {
                        // add new connection into the Modelica model
                        item.Value.Connections.Add(conn);
                    }
                    else
                    {
                        // if one of the connection endpoint was not found
                        GenerateModelicaModel.Messages.Add(GmeMessage.SkippedConnection(item.Key));
                    }
                }
                else if (item.Key is CyPhy.Junction2PowerPort ||
                    item.Key is CyPhy.PowerPort2Junction)
                {
                    // new connection
                    Modelica.Connection conn = new Modelica.Connection();

                    // power flows are only between components
                    Modelica.Component srcComp = null;
                    if (item.Key.ParentContainer.ID == item.Key.GenericSrcEnd.ParentContainer.ID)
                    {
                        // find src component
                        srcComp = item.Value.Components.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericSrcEnd.Name);

                        if (srcComp != null)
                        {
                            conn.Src = new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                            {
                                PortType = Modelica.Factory.PortType.Power
                            };

                            conn.Src.Name = string.Format(
                                "Port{1}",
                                Modelica.Factory.PowerPort,
                                srcComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1);

                            srcComp.Ports.Add(
                                new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.Power
                                });
                        }
                    }
                    else
                    {
                        // find src component's parent
                        Modelica.Model srcCompParent = item.Value.Models.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericSrcEnd.ParentContainer.Name);

                        if (srcCompParent != null)
                        {
                            // find src component
                            srcComp = srcCompParent.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericSrcEnd.Name);

                            if (srcComp != null)
                            {
                                conn.Src = new Modelica.Port(srcCompParent, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.Power
                                };

                                conn.Src.Name = string.Format(
                                    "{1}.Port{0}",
                                    srcComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1,
                                    srcComp.Name);

                                srcComp.Ports.Add(new Modelica.Port(srcComp, item.Key.GenericSrcEnd)
                                {
                                    PortType = Modelica.Factory.PortType.Power
                                });
                            }
                        }
                    }

                    string signs = string.Empty; // for junctions
                    if (srcComp != null && item.Key is CyPhy.Junction2PowerPort)
                    {
                        CyPhy.Junction junction = (item.Key as CyPhy.Junction2PowerPort).SrcEnds.Junction;

                        var sign = "-1" /* // "1" */;

                        if (srcComp.Parameters.TryGetValue(Modelica.Factory.JunctionSignsParameter, out signs))
                        {
                            srcComp.Parameters[Modelica.Factory.JunctionSignsParameter] = signs + ", " + sign;
                        }
                        else
                        {
                            srcComp.Parameters.Add(
                                Modelica.Factory.JunctionSignsParameter,
                                sign);
                        }
                    }

                    Modelica.Component dstComp = null;
                    if (item.Key.ParentContainer.ID == item.Key.GenericDstEnd.ParentContainer.ID)
                    {
                        // find destination component
                        dstComp = item.Value.Components.FirstOrDefault(
                            x => x.OriginalName == item.Key.GenericDstEnd.Name);

                        if (dstComp != null)
                        {
                            conn.Dst = new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                            {
                                PortType = Modelica.Factory.PortType.Power
                            };

                            conn.Dst.Name = string.Format(
                                "Port{1}",
                                Modelica.Factory.PowerPort,
                                dstComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1);

                            dstComp.Ports.Add(new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                            {
                                PortType = Modelica.Factory.PortType.Power
                            });
                        }
                    }
                    else
                    {
                        // find destination component's parent
                        Modelica.Model dstCompParent =
                            item.Value.Models.FirstOrDefault(x => x.OriginalName == item.Key.GenericDstEnd.ParentContainer.Name);

                        if (dstCompParent != null)
                        {
                            // find destination component
                            dstComp = dstCompParent.Components.FirstOrDefault(
                                x => x.OriginalName == item.Key.GenericDstEnd.Name);

                            if (dstComp != null)
                            {
                                conn.Dst = new Modelica.Port(dstCompParent, item.Key.GenericDstEnd)
                                {
                                    PortType = Modelica.Factory.PortType.Power
                                };

                                conn.Dst.Name = string.Format(
                                    "{2}.Port{1}",
                                    Modelica.Factory.PowerPort,
                                    dstComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1,
                                    dstComp.Name);

                                dstComp.Ports.Add(new Modelica.Port(dstComp, item.Key.GenericDstEnd)
                                {
                                    PortType = Modelica.Factory.PortType.Power
                                });
                            }
                        }
                    }

                    if (dstComp != null && item.Key is CyPhy.PowerPort2Junction)
                    {
                        CyPhy.Junction junction = (item.Key as CyPhy.PowerPort2Junction).DstEnds.Junction;

                        if (dstComp.Parameters.TryGetValue(Modelica.Factory.JunctionSignsParameter, out signs))
                        {
                            dstComp.Parameters[Modelica.Factory.JunctionSignsParameter] = signs + ", -1"/*", -1"*/;
                        }
                        else
                        {
                            dstComp.Parameters.Add(
                                Modelica.Factory.JunctionSignsParameter,
                                "-1"/* // "-1" */);
                        }
                    }

                    if (srcComp != null &&
                        dstComp != null)
                    {
                        // add new connection into the Modelica model
                        item.Value.Connections.Add(conn);
                    }
                    else
                    {
                        // if one of the connection endpoint was not found
                        GenerateModelicaModel.Messages.Add(GmeMessage.SkippedConnection(item.Key));
                    }
                }
                else if (item.Key is CyPhy.ModelicaPortMap)
                {
                    var modelicaPortMap = item.Key as CyPhy.ModelicaPortMap;

                    // new connection
                    Modelica.Connection conn = new Modelica.Connection();

                    if (modelicaPortMap.SrcEnd.ParentContainer is CyPhy.ModelicaModel ||
                        modelicaPortMap.SrcEnd.ParentContainer is CyPhy.SignalFlowModel)
                    {
                        var srcParent = item.Value.Components.FirstOrDefault(x =>
                            x.Impl.ID == modelicaPortMap.SrcEnd.ParentContainer.ID);

                        if (srcParent != null)
                        {
                            conn.Src = new Modelica.Port(srcParent, modelicaPortMap.SrcEnd);
                            conn.Src.Name = modelicaPortMap.SrcEnd.Name;
                            conn.Src.PortType = Modelica.Factory.PortType.ModelicaConnection;
                        }

                        var dstParent = item.Value.Components.FirstOrDefault(x =>
                            x.Impl.ID == modelicaPortMap.DstEnd.ID);

                        if (dstParent != null)
                        {
                            conn.Dst = new Modelica.Port(dstParent, modelicaPortMap.DstEnd);
                            conn.Dst.PortType = Modelica.Factory.PortType.ModelicaConnection;
                        }

                        if (srcParent != null &&
                            dstParent != null)
                        {
                            item.Value.Connections.Add(conn);
                        }
                    }
                    else
                    {
                        var srcParent = item.Value.Components.FirstOrDefault(x =>
                            x.Impl.ID == modelicaPortMap.SrcEnd.ID);

                        if (srcParent != null)
                        {
                            conn.Src = new Modelica.Port(srcParent, modelicaPortMap.SrcEnd);
                        }

                        var dstParent = item.Value.Components.FirstOrDefault(x =>
                            x.Impl.ID == modelicaPortMap.DstEnd.ParentContainer.ID);

                        if (dstParent != null)
                        {
                            conn.Dst = new Modelica.Port(dstParent, modelicaPortMap.DstEnd);
                            conn.Dst.Name = modelicaPortMap.SrcEnd.Name;
                        }

                        if (srcParent != null &&
                            dstParent != null)
                        {
                            item.Value.Connections.Add(conn);
                        }
                    }
                }
                else
                {
                    string connectionEqu = string.Format("//{0} -> {1}", item.Key.GenericSrcEnd.Path, item.Key.GenericDstEnd.Path);
                    item.Value.Equations.Add(connectionEqu);
                }
            }

            return messages;
        }

        /// <summary>
        /// Builds all power chains starting from this test bench.
        /// </summary>
        /// <param name="model">Test bench model</param>
        public static void BuildPowerChains(CyPhy.TestBench model)
        {
            CheckModelicaNames(model);

            // process all submodels
            foreach (var item in model.Children.DesignElementCollection)
            {
                BuildPowerChains(item);
            }
        }

        /// <summary>
        /// Checks whether all elements have a unique name or not (except the connections)
        /// and if the name is a reserved Modelica keyword.
        /// </summary>
        /// <param name="model">Given container</param>
        public static void CheckModelicaNames(Common.Interfaces.Container model)
        {
            // ignore connections
            // ignore solid modeling
            var modelicaKeyWords = new List<String>()
                {
                    "algorithm",        "discrete", "   false",    "model",         "redeclare", 
                    "and",              "each",         "final",    "not",          "replaceable", 
                    "annotation",       "else",         "flow",     "operator",     "return", 
                    "assert",           "elseif",       "for",      "or",           "stream", 
                    "block",            "elsewhen",     "function", "outer",        "then", 
                    "break",            "encapsulated", "if",       "output",       "true", 
                    "class",            "end",          "import",   "package",      "type", 
                    "connect",          "enumeration",  "in",       "parameter",    "when", 
                    "connector",        "equation",     "initial",  "partial",      "while", 
                    "constant",         "expandable",   "inner",    "protected",    "within", 
                    "constrainedby",    "extends",      "input",    "public", 
                    "der",              "external",     "loop",     "record"
                }; /// from ModelicaLanguageSpecification version 3.2 

            var elements = model.AllChildren
                .Where(x => (x is Common.Interfaces.Connection) == false)
                .Where(x => (x is CyPhy.SolidModelingPortType) == false)
                .Where(x => (x is CyPhy.CADModel) == false)
                .Where(x => (x is CyPhy.ValueFormula) == false) // per http://escher.isis.vanderbilt.edu/JIRA/browse/META-702
                ;


            var elementsWithUniqueNames = new Dictionary<string, List<Common.Interfaces.Base>>();

            foreach (var element in elements)
            {
                List<Common.Interfaces.Base> list = null;
                if (elementsWithUniqueNames.TryGetValue(element.Name, out list))
                {
                    list.Add(element);
                }
                else
                {
                    elementsWithUniqueNames.Add(
                        element.Name,
                        new List<Common.Interfaces.Base>() { element });
                }

                if (modelicaKeyWords.Contains(element.Name))
                {
                    GenerateModelicaModel.Messages.Add(GmeMessage.KeywordName(model, element.Name, element.ToHyperLink()));
                }
            }

            foreach (var kvp in elementsWithUniqueNames)
            {
                if (kvp.Value.Count > 1)
                {
                    StringBuilder sb = new StringBuilder();

                    foreach (var listItem in kvp.Value)
                    {
                        sb.AppendFormat(" {0}", listItem.ToHyperLink());
                    }

                    GenerateModelicaModel.Messages.Add(GmeMessage.SameName(model, sb.ToString()));
                }
            }
        }


        /// <summary>
        /// Builds all power chains starting from this design element.
        /// </summary>
        /// <param name="model">Any design element</param>
        public static void BuildPowerChains(CyPhy.DesignElement model)
        {
            CheckModelicaNames(model);

            // process all submodel
            if (model is CyPhy.Component)
            {
                foreach (var item in (model as CyPhy.Component).Children.BondGraphCollection)
                {
                    BuildPowerChains(item);
                }
            }
            else if (model is CyPhy.ComponentAssembly)
            {
                foreach (var item in (model as CyPhy.ComponentAssembly).Children.ComponentCollection)
                {
                    BuildPowerChains(item);
                }
            }
        }

        /// <summary>
        /// Builds all power chains starting from this physical component.
        /// </summary>
        /// <param name="model">Any bond graph container</param>
        public static void BuildPowerChains(CyPhy.BondGraph model)
        {
            CheckModelicaNames(model);

            // process all submodel
            foreach (var item in model.Children.BondGraphCollection)
            {
                BuildPowerChains(item);
            }

            // get all src power ports
            foreach (var item in model.Children.BGPowerPortCollection)
            {
                if (item is CyPhy.ElectricalPort)
                {
                    var port = item as CyPhy.ElectricalPort;
                    if (port.SrcConnections.ElectricalConnectionCollection.Count() == 0 &&
                        port.SrcConnections.PhysicalPort2PowerPortCollection.Count() == 0)
                    {
                        // build a chain starting with this power port
                        BuildChain(item);
                    }
                }
                else if (item is CyPhy.MechanicalDPort)
                {
                    var port = item as CyPhy.MechanicalDPort;
                    if (port.SrcConnections.MechanicalDConnectionCollection.Count() == 0 &&
                        port.SrcConnections.PhysicalPort2PowerPortCollection.Count() == 0)
                    {
                        // build a chain starting with this power port
                        BuildChain(item);
                    }
                }
                else if (item is CyPhy.MechanicalRPort)
                {
                    var port = item as CyPhy.MechanicalRPort;
                    if (port.SrcConnections.MechanicalRConnectionCollection.Count() == 0 &&
                        port.SrcConnections.PhysicalPort2PowerPortCollection.Count() == 0)
                    {
                        // build a chain starting with this power port
                        BuildChain(item);
                    }
                }
                else if (item is CyPhy.HydraulicPort)
                {
                    var port = item as CyPhy.HydraulicPort;
                    if (port.SrcConnections.HydraulicConnectionCollection.Count() == 0 &&
                        port.SrcConnections.PhysicalPort2PowerPortCollection.Count() == 0)
                    {
                        // build a chain starting with this power port
                        BuildChain(item);
                    }
                }
                else if (item is CyPhy.ThermalPort)
                {
                    var port = item as CyPhy.ThermalPort;
                    if (port.SrcConnections.ThermalConnectionCollection.Count() == 0 &&
                        port.SrcConnections.PhysicalPort2PowerPortCollection.Count() == 0)
                    {
                        // build a chain starting with this power port
                        BuildChain(item);
                    }
                }
            }
        }

        #region Helper Functions

        /// <summary>
        /// Given a power port or junction returns the power chains, which contain
        /// the given element.
        /// </summary>
        /// <param name="element">Given element</param>
        /// <returns>Power chain</returns>
        public static IEnumerable<PowerChain> GetPowerChains(Common.Interfaces.Base element)
        {
            foreach (var item in PowerChains)
            {
                if (item.ChainItems.FirstOrDefault(x => x.ID == element.ID) != null)
                {
                    yield return item;
                }

                if (item.SrcJunction.ID == element.ID)
                {
                    yield return item;
                }

                if (item.DstJunction.ID == element.ID)
                {
                    yield return item;
                }
            }
        }

        /// <summary>
        /// Print the power chains into the Debug console.
        /// </summary>
        public static void DumpDebugInfo()
        {
            int i = 0; // power chain counter
            foreach (var chain in PowerChains)
            {
                i++;
                StringBuilder sb = new StringBuilder();
                sb.AppendFormat("{0:00} ", i);

                sb.AppendFormat("{0}", chain.SrcJunction.Name);

                foreach (var item in chain.ChainItems)
                {
                    sb.AppendFormat(" -> {0}", item.Name);
                }

                if (chain.DstJunction != null)
                {
                    sb.AppendFormat(" -> {0}", chain.DstJunction.Name);
                }
                else
                {
                    sb.AppendFormat("-> null");
                }

                sb.AppendLine();
                System.Diagnostics.Debug.Write(sb.ToString());
            }
        }
        #endregion

        /// <summary>
        /// Builds the entire power chain from a given starting element.
        /// </summary>
        /// <param name="firstElement">First item in the chain.</param>
        private static void BuildChain(Common.Interfaces.Base firstElement)
        {
            if (firstElement is CyPhy.BGPowerPort)
            {
                foreach (var item in (firstElement as CyPhy.BGPowerPort).SrcConnections.Junction2PowerPortCollection)
                {
                    var chain = new PowerChain(firstElement);
                    chain.Connections.Add(item);
                    chain.SrcJunction = item.SrcEnds.Junction;

                    GetNextChainElement(chain, firstElement);
                }

                foreach (var item in (firstElement as CyPhy.BGPowerPort).DstConnections.PowerPort2JunctionCollection)
                {
                    var chain = new PowerChain(firstElement);
                    chain.SrcJunction = item.DstEnds.Junction;
                    chain.Connections.Add(item);

                    GetNextChainElement(chain, firstElement);
                }
            }
        }

        /// <summary>
        /// Build the rest of the power chain recursively.
        /// </summary>
        /// <param name="chain">Actual chain.</param>
        /// <param name="prevElement">Previous power port element.</param>
        private static void GetNextChainElement(
            PowerChain chain,
            Common.Interfaces.Base prevElement)
        {
            var firstItem = chain.ChainItems.FirstOrDefault();

            if (firstItem != null &&
                prevElement.ID != firstItem.ID)
            {
                // if not the first power port is given
                // try to find destination junctions
                if (prevElement is CyPhy.BGPowerPort)
                {
                    foreach (var item in (prevElement as CyPhy.BGPowerPort).DstConnections.PowerPort2JunctionCollection)
                    {
                        var newChain = new PowerChain(chain);
                        newChain.Connections.Add(item);
                        newChain.DstJunction = item.DstEnds.Junction;
                        PowerChains.Add(newChain);
                    }

                    foreach (var item in (prevElement as CyPhy.BGPowerPort).SrcConnections.Junction2PowerPortCollection)
                    {
                        var newChain = new PowerChain(chain);
                        newChain.Connections.Add(item);
                        newChain.DstJunction = item.SrcEnds.Junction;
                        PowerChains.Add(newChain);
                    }
                }
            }

            if (prevElement is CyPhy.PowerPortType)
            {
                foreach (var item in (prevElement as CyPhy.PowerPortType).DstConnections.PowerFlowCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }

                foreach (var item in (prevElement as CyPhy.PowerPortType).DstConnections.PhysicalPort2PowerPortCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }
            }
            else if (prevElement is CyPhy.ElectricalPort)
            {
                var port = prevElement as CyPhy.ElectricalPort;
                foreach (var item in port.DstConnections.ElectricalConnectionCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }

                foreach (var item in port.DstConnections.PowerPort2PhysicalPortCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }
            }
            else if (prevElement is CyPhy.MechanicalDPort)
            {
                var port = prevElement as CyPhy.MechanicalDPort;
                foreach (var item in port.DstConnections.MechanicalDConnectionCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }

                foreach (var item in port.DstConnections.PowerPort2PhysicalPortCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }
            }
            else if (prevElement is CyPhy.MechanicalRPort)
            {
                var port = prevElement as CyPhy.MechanicalRPort;
                foreach (var item in port.DstConnections.MechanicalRConnectionCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }

                foreach (var item in port.DstConnections.PowerPort2PhysicalPortCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }
            }
            else if (prevElement is CyPhy.HydraulicPort)
            {
                var port = prevElement as CyPhy.HydraulicPort;
                foreach (var item in port.DstConnections.HydraulicConnectionCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }

                foreach (var item in port.DstConnections.PowerPort2PhysicalPortCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }
            }
            else if (prevElement is CyPhy.ThermalPort)
            {
                var port = prevElement as CyPhy.ThermalPort;
                foreach (var item in port.DstConnections.ThermalConnectionCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }

                foreach (var item in port.DstConnections.PowerPort2PhysicalPortCollection)
                {
                    var newChain = new PowerChain(chain);
                    newChain.Connections.Add(item);
                    newChain.ChainItems.Add(item.DstEnd);

                    GetNextChainElement(newChain, item.DstEnd);
                }
            }
        }
    }
}
