/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿// -----------------------------------------------------------------------
// <copyright file="CyPhy2Modelica.cs" company="">
// TODO: Update copyright text.
// </copyright>
// -----------------------------------------------------------------------

namespace CyPhy2Modelica
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using System.IO;

    using GME;
    using GME.CSharp;
    using GME.MGA;
    using GME.MGA.Core;

    // using generic common interface
    using Common = ISIS.GME.Common;

    // using domain specific interfaces
    using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
    using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;
    using System.Windows.Forms;
    using System.Diagnostics;

    /// <summary>
    /// TODO: Update summary.
    /// </summary>
    public partial class CyPhy2ModelicaInterpreter
    {
        /// <summary>
        /// Processes the given object and generates modelica code for it.
        /// </summary>
        /// <param name="currentobj">
        /// <list type="number">
        ///  <item><description>[Test bench |</description></item>
        ///  <item><description>Component |</description></item>
        ///  <item><description>Component Assembly]</description></item>
        ///  </list>
        /// </param>
        private void Process(MgaFCO currentobj)
        {
            GMEConsole.Info.WriteLine("Preparing model for translation. Please wait...");
            System.Windows.Forms.Application.DoEvents();

            ApplyFidelitySettings(currentobj);

            CyPhy.TestBench subject = Common.Utils.CreateObject<CyPhyClasses.TestBench>(currentobj as MgaObject);

            if (subject == null)
            {
                throw new Exception(
                    "Model check was not successful or there is an incompatibility between the meta-model and the DSML API.");
            }

            BondGraph.PowerChain.BuildPowerChains(subject);

            // print out all power chain into the debug console
#if DEBUG
            BondGraph.PowerChain.DumpDebugInfo();
#endif
            var generateModelicaModel = true;

            if ((this.componentParameters["automation"] as string) != "true" ||
                (this.componentParameters["do_config"] as string) == "true")
            {
                // show main form
                using (MainForm mf = new MainForm())
                {
                    if (this.Parameters.Automation == "true")
                    {
                        mf.chbRunDymolaSimulation.Visible = false;
                        mf.chbRunOMSimulation.Visible = false;
                    }

                    mf.ShowDialog();
                    generateModelicaModel = mf.DialogResult == DialogResult.OK;
                }
            }

            this.Run = generateModelicaModel;

            var includeLibs = new List<string>();

            try
            {
                includeLibs = CyPhy2Modelica.Properties.Settings.Default
                    .IncludeDirectoryPath
                    .Cast<string>()
                    .ToList();
            }
            catch (Exception ex)
            {
                Trace.TraceWarning(ex.ToString());


            }

            if (generateModelicaModel == false)
            {
                GenerateModelicaModel.Messages.Clear();
                GMEConsole.Warning.WriteLine("Process was cancelled.");
                return;
            }

            // initialize the modelica model generator
            GenerateModelicaModel gmm = new GenerateModelicaModel(subject);

            // build the Modelica assembly based on the GME model
            gmm.BuildModel();

            foreach (var item in gmm.ProcessedObjects)
            {
                // TODO: special characters
                if (item.Name.Contains(' '))
                {
                    GenerateModelicaModel.Messages.Add(GmeMessage.UnsafeVariableName(item));
                }
            }

            gmm.CheckIncludedLibraries(includeLibs);

            this.componentParameters["test_bench_name"] = gmm.asm.Path;

            GMEConsole.Info.WriteLine("Modelica model name: " + subject.Name);

            // generate the modelica model
            gmm.Generate(
                Modelica.Factory.GetModifiedName(currentobj.Name) + ".mo",
                this.OutputDirectory);

            this.componentParameters["lib_package_name"] = includeLibs.Select(x => Path.GetFileName(x)).ToList();
            this.componentParameters["lib_package_folder"] = includeLibs.Select(x => Path.GetDirectoryName(x)).ToList();

            GenerateModelConfig(subject, includeLibs, gmm.asm.Name);

            GeneratePythonScripts(includeLibs);

            //// Writes out .json files 
            CreateFilterFile(gmm);

            CreateLimitFile(gmm);

            GenerateModelicaModel.Messages.Add(GmeMessage.GetSimpleCustomFormulas());

            DumpAllMessageToGMEConsole();

            GMEConsole.Info.WriteLine(
                "Output files are <a href=\"file:///{0}\" target=\"_blank\">{0}</a>.",
                this.OutputDirectory);
        }

        private void CreateLimitFile(GenerateModelicaModel gmm)
        {
            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "limits.json")))
            {
                if (this.simulationType == SimulationType.OpenModelica)
                {
                    gmm.LimitDefinition.PathToMatFile = string.Format("{0}.{1}_res.mat",
                        Path.GetFileName(this.OutputDirectory),
                        gmm.asm.Name);
                }
                else if (this.simulationType == SimulationType.Dymola)
                {
                    gmm.LimitDefinition.PathToMatFile = string.Format("{0}.{1}.mat",
                        Path.GetFileName(this.OutputDirectory),
                        gmm.asm.Name);
                }

                string limits = Newtonsoft.Json.JsonConvert.SerializeObject(
                    gmm.LimitDefinition,
                    Newtonsoft.Json.Formatting.Indented);

                writer.WriteLine(limits);
            }
        }

        private void CreateFilterFile(GenerateModelicaModel gmm)
        {
            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "filter.json")))
            {
                List<string> variables = new List<string>();
                foreach (var item in gmm.asm.Components
                    .Where(x => x.Impl is CyPhy.Parameter || x.Impl is CyPhy.Metric))
                {
                    variables.Add(item.Name);
                }

                var jsonString = Newtonsoft.Json.JsonConvert.SerializeObject(
                    variables,
                    Newtonsoft.Json.Formatting.Indented);

                writer.WriteLine(jsonString);
            }
        }

        private void GeneratePythonScripts(List<string> includeLibs)
        {
            // look up for dymola path
            // have a hard coded path
            string dymolaExe = @"C:\Program Files (x86)\Dymola 2013\bin64\Dymola.exe";

            string dymolaHome = (new FileInfo(dymolaExe)).DirectoryName;

            /*Templates.simscript simscript = new Templates.simscript()
            {
                DymolaHome = dymolaHome,
            };

            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "simscript.py")))
            {
                writer.WriteLine(simscript.TransformText());
            }
            */
            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "simscript.py")))
            {
                writer.WriteLine(CyPhy2Modelica.Properties.Resources.simscript);
            }

            string resultMatFileName = string.Format("{0}.{1}",
                this.Parameters.TestBenchName.TruncateLongString(8),
                this.Parameters.DesignName);

            Templates.tool_simulate om_simulate = new Templates.tool_simulate()
            {
                ToolName = "OpenModelica",
                ResultMatFile = resultMatFileName,
                DymolaHome = "n/a"
            };

            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "simulate.py")))
            {
                writer.WriteLine(CyPhy2Modelica.Properties.Resources.simulate);
            }

            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "version_check.py")))
            {
                writer.WriteLine(CyPhy2Modelica.Properties.Resources.version_check);
            }

            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "om_simulate.py")))
            {
                writer.WriteLine(om_simulate.TransformText());
            }

            Templates.tool_simulate dymola_simulate = new Templates.tool_simulate()
            {
                ToolName = "Dymola",
                ResultMatFile = resultMatFileName,
                DymolaHome = dymolaHome
            };

            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "dymola_simulate.py")))
            {
                writer.WriteLine(dymola_simulate.TransformText());
            }

            Templates.open_package_in_dymola open_in_dymola = new Templates.open_package_in_dymola()
            {
                DymolaExecutable = dymolaExe,
            };

            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "open_package_in_dymola.py")))
            {
                writer.WriteLine(open_in_dymola.TransformText());
            }

            Templates.open_package_in_om open_in_om = new Templates.open_package_in_om();

            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "open_package_in_om.py")))
            {
                writer.WriteLine(open_in_om.TransformText());
            }

            //// source_data.zip is a 'hard-coded' package name JobManager uses it
            //// zip working directory + include modelica package dependency

            Templates.zip zippy = new Templates.zip()
            {
                FullPackagePaths = includeLibs,
                OutputFileName = "source_data.zip",
            };

            using (StreamWriter writer = new StreamWriter(Path.Combine(this.OutputDirectory, "zip.py")))
            {
                writer.WriteLine(zippy.TransformText());
            }
        }

        private void GenerateModelConfig(
            CyPhy.TestBench subject,
            List<string> includeLibs,
            string modelName)
        {
            CyPhyPET.ModelConfig modelConfig = new CyPhyPET.ModelConfig()
            {
                model_name = Path.GetFileName(this.OutputDirectory) + "." + modelName,
                model_file_name = "package.mo",
            };

            var solver = subject.Children.SolverSettingsCollection.FirstOrDefault();
            if (solver != null)
            {
                modelConfig.experiment.StartTime = solver.Attributes.StartTime.ToString();
                modelConfig.experiment.StopTime = solver.Attributes.StopTime.ToString();

                if (solver.Attributes.Tolerance > 0)
                {
                    modelConfig.experiment.Tolerance = solver.Attributes.Tolerance.ToString();
                }

                if (solver.Attributes.IntervalMethod ==
                    CyPhyClasses.SolverSettings.AttributesClass.IntervalMethod_enum.Interval_Length)
                {
                    modelConfig.experiment.IntervalMethod = "Interval";
                    modelConfig.experiment.Interval = solver.Attributes.IntervalLength.ToString();
                }
                else if (solver.Attributes.IntervalMethod ==
                    CyPhyClasses.SolverSettings.AttributesClass.IntervalMethod_enum.Number_of_Intervals)
                {
                    modelConfig.experiment.IntervalMethod = "NumberOfIntervals";
                    if (solver.Attributes.NumberOfIntervals > 0)
                    {
                        modelConfig.experiment.NumberOfIntervals = solver.Attributes.NumberOfIntervals.ToString();
                    }
                }

                var solverString = solver.Attributes.Solver.ToString();
                if (solverString.StartsWith("Dymola__"))
                {
                    var map = new Dictionary<CyPhyClasses.SolverSettings.AttributesClass.Solver_enum, string>()
                    {
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Cerk23__dash__order__3, "cerk23" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Cerk34__dash__order__4, "cerk34" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Cerk45__dash__order__5, "cerk45" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Dopri45__dash__order__5, "dopri45" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Dopri853__dash__order__8, "dopri853" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Esdirk23a__dash__order__3_stiff, "esdirk23a" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Esdirk34a__dash__order__4_stiff, "esdirk34a" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Esdirk45a__dash__order__5_stiff, "esdirk45a" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Lsodar, "lsodar" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Radau_IIa__dash__order__5_stiff, "radau" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Rkfix2, "rkfix2" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Rkfix3, "rkfix3" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Rkfix4, "rkfix4" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Sdirk34hw__dash__order__4_stiff, "sdirk34hw" },
                    };
                    modelConfig.experiment.Algorithm.Dymola = map[solver.Attributes.Solver];
                }
                else if (solverString.StartsWith("OM__"))
                {
                    var map = new Dictionary<CyPhyClasses.SolverSettings.AttributesClass.Solver_enum, string>()
                    {
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.OM__dallsSymJac, "OM__dallsSymJac" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.OM__dasslwort, "OM__dasslwort" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.OM__inline_dash_euler, "OM__inline_dash_euler" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.OM__inline_dash_rungekutta, "OM__inline_dash_rungekutta" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.OM__rungekutta, "OM__rungekutta" },
                    };
                    modelConfig.experiment.Algorithm.OpenModelica = map[solver.Attributes.Solver];
                }
                else
                {
                    // most likely the solver is supported by both tools
                    modelConfig.experiment.Algorithm.Dymola = solverString;
                    modelConfig.experiment.Algorithm.OpenModelica = solverString;
                }

            }

            modelConfig.lib_package_paths.AddRange(
                includeLibs.Select(x => Path.GetDirectoryName(Path.GetFullPath(x))).ToArray());

            modelConfig.lib_package_names.AddRange(
                includeLibs.Select(x => Path.GetFileName(x)).ToArray());

            var model_config_filename = Path.Combine(this.OutputDirectory, "model_config.json");

            using (StreamWriter writer = new StreamWriter(model_config_filename))
            {
                var jsonString = Newtonsoft.Json.JsonConvert.SerializeObject(
                    modelConfig,
                    Newtonsoft.Json.Formatting.Indented);

                writer.WriteLine(jsonString);
            }
        }

        private void DumpAllMessageToGMEConsole()
        {
            // write all collected messages into the GMEConsole
            GenerateModelicaModel.Messages.ForEach(x => x.Write(GMEConsole));

            // write stat out
            int num_errors = GenerateModelicaModel.Messages.Count(x => x.Severity == GmeMessage.Severity_enum.Error);
            int num_warnings = GenerateModelicaModel.Messages.Count(x => x.Severity == GmeMessage.Severity_enum.Warning);
            int num_information = GenerateModelicaModel.Messages.Count(x => x.Severity == GmeMessage.Severity_enum.Information);
            int num_none = GenerateModelicaModel.Messages.Count(x => x.Severity == GmeMessage.Severity_enum.None);

            string msg = string.Format("{0} error(s), {1} warning(s), {2} info(s), {3} message(s).",
                num_errors,
                num_warnings,
                num_information,
                num_none);

            if (num_errors > 0)
            {
                GMEConsole.Error.WriteLine(msg);
            }
            else if (num_warnings > 0)
            {
                GMEConsole.Warning.WriteLine(msg);
            }
            else if (num_information > 0)
            {
                GMEConsole.Info.WriteLine(msg);
            }
            else
            {
                GMEConsole.Out.WriteLine(msg);
            }
        }

        private void ApplyFidelitySettings(MgaFCO currentobj)
        {
            var strSettings = currentobj.RegistryValue["FidelitySettings"];
            var settings = new Dictionary<string, Dictionary<string, string>>();
            if (string.IsNullOrEmpty(strSettings) == false)
            {
                try
                {
                    settings = Newtonsoft.Json.JsonConvert.DeserializeObject<Dictionary<string, Dictionary<string, string>>>(strSettings);
                }
                catch (Exception ex)
                {
                    GenerateModelicaModel.Messages.Add(new GmeMessage(ex.ToString(), GmeMessage.Severity_enum.Warning));
                }
            }

            RemoveUnusedModelicaModels(currentobj, settings);
        }

        private void RemoveUnusedModelicaModels(
            MgaFCO currentobj,
            Dictionary<string, Dictionary<string, string>> settings)
        {
            if (currentobj is MgaModel)
            {
                foreach (var child in currentobj.ChildObjects.OfType<MgaFCO>())
                {
                    RemoveUnusedModelicaModels(child, settings);
                }
            }

            if (currentobj.Meta.Name == typeof(CyPhy.Component).Name)
            {
                if (currentobj.ChildObjects.Cast<IMgaFCO>().Where(x => x.Meta.Name == "ModelicaModel").Count() > 1)
                {
                    if (currentobj.ArcheType != null)
                    {
                        // to be able to delete inside
                        // FIXME: if we cut the inheritance the attributes will be lost...
                        // TODO: should we take care of this here on in GME?
                        //currentobj.DetachFromArcheType();

                        // FIXME: delete items from the base type, we should use the function above...
                        currentobj = currentobj.ArcheType;
                    }

                    Dictionary<string, string> selection = new Dictionary<string, string>();
                    if (settings.TryGetValue(currentobj.StrAttrByName["Classifications"], out selection))
                    {
                        foreach (var kind in selection)
                        {
                            foreach (var item in currentobj.ChildObjects.Cast<IMgaFCO>().Where(x => x.Meta.Name == kind.Key).ToList())
                            {
                                if (item.Name != kind.Value)
                                {
                                    item.DestroyObject();
                                }
                            }
                        }
                    }

                    if (currentobj.ChildObjects.Cast<IMgaFCO>().Where(x => x.Meta.Name == "ModelicaModel").Count() > 1)
                    {
                        GenerateModelicaModel.Messages.Add(
                            new GmeMessage(
                                string.Format("{0} has more than one modelica model.", currentobj.ToMgaHyperLink()),
                                GmeMessage.Severity_enum.Error));
                    }
                }
            }
        }

        /// <summary>
        /// Helper function for copying entire common folder for post processing
        /// </summary>
        /// <param name="source">Source directory</param>
        /// <param name="target">Destination directory</param>
        private static void CopyFilesRecursively(
            DirectoryInfo source,
            DirectoryInfo target)
        {
            foreach (DirectoryInfo dir in source.GetDirectories())
            {
                CopyFilesRecursively(dir, target.CreateSubdirectory(dir.Name));
            }

            foreach (FileInfo file in source.GetFiles())
            {
                file.CopyTo(Path.Combine(target.FullName, file.Name), true);
            }
        }

        /// <summary>
        /// Generates the post processing scripts
        /// </summary>
        /// <param name="outputDir">Output directory</param>
        /// <param name="testBench">Test bench to generate for</param>
        private void GeneratePostProcessingScripts(
            string outputDir,
            CyPhy.TestBench testBench)
        {
            /*
            using (StreamWriter writer = new StreamWriter(Path.Combine(outputDir, "CheckLimits.py")))
            {
                writer.WriteLine(CyPhy2Modelica.Properties.Resources.CheckLimits);
            }
            */

            // TODO: This is a shortcut we need to use a template based approach in the future
            /*string mainFileName = Path.Combine(outputDir, "main_post_process.py");

            using (StreamWriter writer = new StreamWriter(mainFileName))
            {
                foreach (var postProc in testBench.Children.PostProcessingCollection)
                {
                    writer.WriteLine("# resource file: {0}", postProc.Attributes.ScriptPath);

                    foreach (var input in postProc.Children.PPInputCollection)
                    {
                        writer.WriteLine("#  inputs: {0}", input.Name);
                        foreach (var conn in input.AllSrcConnections)
                        {
                            writer.WriteLine("#  input variable names: {0}", conn.GenericSrcEnd.Name);
                        }
                    }

                    foreach (var output in postProc.Children.PPOutputCollection)
                    {
                        writer.WriteLine("#  outputs: {0}", output.Name);
                        foreach (var conn in output.AllDstConnections)
                        {
                            writer.WriteLine("#  output variable names: {0}", conn.GenericDstEnd.Name);
                        }
                    }
                }

                writer.WriteLine("import CheckLimits");
                writer.WriteLine("CheckLimits.main()");

                writer.WriteLine("import os");
                writer.WriteLine("import sys");

                foreach (var postProc in testBench.Children.PostProcessingCollection)
                {
                    writer.Write("sys.path.append(r'");

                    writer.Write(Path.GetDirectoryName(Path.GetFullPath(postProc.Attributes.ScriptPath)));
                    writer.WriteLine("')");
                }

                writer.WriteLine(@"if __name__ == '__main__':
    if not os.path.exists('summary.testresults.json'):
        print 'File does not exist: summary.testresults.json'
        os.exit(0)
        
    if (len(sys.argv) > 1):
        if not os.path.exists(sys.argv[1]):
            print 'Given result file does not exist: {0}'.format(sys.argv[1])
            os.exit(0)
    ");

                StringBuilder sb = new StringBuilder();

                foreach (var postProc in testBench.Children.PostProcessingCollection)
                {
                    if (string.IsNullOrWhiteSpace(postProc.Attributes.ScriptPath) == false)
                    {
                        if (File.Exists(postProc.Attributes.ScriptPath))
                        {
                            string name = Path.GetFileNameWithoutExtension(Path.GetFullPath(postProc.Attributes.ScriptPath));
                            writer.WriteLine("import {0}", name);
                            writer.WriteLine("{0}.main()", name);
                        }
                    }
                }
            }
             */
        }

        /// <summary>
        /// Runs the simulation
        /// </summary>
        /// <param name="outputDir">Output/working directory</param>
        /// <param name="filename">Filename to execute</param>
        /// <param name="currentobj">Currently opened model</param>
        private void RunSimulation(
            string outputDir,
            string filename,
            MgaFCO currentobj)
        {
            System.Diagnostics.ProcessStartInfo psi = new System.Diagnostics.ProcessStartInfo()
            {
                FileName = filename,
                WorkingDirectory = outputDir
            };

            System.Diagnostics.Process p = new System.Diagnostics.Process()
            {
                StartInfo = psi,
            };

            GMEConsole.Info.WriteLine("Running simulation using {0}", filename);
            p.Start();
            p.WaitForExit();
            GMEConsole.Info.WriteLine("Done");

            string reportFileName = Path.Combine(outputDir, "summary.testresults.json");

            if (File.Exists(reportFileName))
            {
                // set default value
                string reportContent = Newtonsoft.Json.JsonConvert.SerializeObject(new AVM.DDP.MetaTBReport());
                using (StreamReader reader = new StreamReader(reportFileName))
                {
                    // overwrite default value from file
                    reportContent = reader.ReadToEnd();
                }

                var reportJson = Newtonsoft.Json.Linq.JObject.Parse(reportContent);

                if (currentobj != null)
                {
                    if (currentobj.MetaBase.Name == "TestBench")
                    {
                        var tb = CyPhyClasses.TestBench.Cast(currentobj);

                        foreach (var item in reportJson["Metrics"])
                        {
                            var cyphyMetric = tb.Children.MetricCollection.FirstOrDefault(x => x.Name == item["Name"].ToString());
                            if (cyphyMetric != null)
                            {
                                cyphyMetric.Attributes.Value = item["Value"].ToString();
                                GMEConsole.Info.WriteLine("Metric: {0} value was updated to: {1}",
                                    cyphyMetric.ToHyperLink(),
                                    cyphyMetric.Attributes.Value);
                            }
                            else
                            {
                                GMEConsole.Warning.WriteLine("Metric was not found {0}", item["Name"].ToString());
                            }
                        }
                    }
                    else
                    {
                        GMEConsole.Warning.WriteLine("Current object is not a test bench [{0}] skipping metric update.",
                            currentobj.MetaBase.Name);
                    }
                }
            }
        }

        /// <summary>
        /// Gets the size of a directory
        /// </summary>
        /// <param name="path">Path to the directory</param>
        /// <returns>Size of the directory</returns>
        public static long GetDirectorySize(string path)
        {
            long folderSize = 0;
            string[] files = Directory.GetFiles(path, "*.*", System.IO.SearchOption.AllDirectories);
            foreach (string file in files)
            {
                FileInfo info = new FileInfo(file);
                folderSize += info.Length;
            }

            return folderSize;
        }

        /// <summary>
        /// Copies folders from one place to another
        /// </summary>
        /// <param name="sourceFolder">Source directory</param>
        /// <param name="destFolder">Destination directory</param>
        /// <param name="ignoredDirectoryNames">Directory names to be skipped e.g. .SVN or .GIT</param>
        public static void CopyFolder(
            string sourceFolder,
            string destFolder,
            IEnumerable<string> ignoredDirectoryNames = null)
        {
            if (ignoredDirectoryNames != null)
            {
                if (ignoredDirectoryNames.Contains(Path.GetFileName(sourceFolder)))
                {
                    GenerateModelicaModel.Messages.Add(new GmeMessage(
                        string.Format("{0} was ignored and not copied.", sourceFolder),
                        GmeMessage.Severity_enum.Information));
                    return;
                }
            }

            if (!Directory.Exists(destFolder))
            {
                Directory.CreateDirectory(destFolder);
            }

            string[] files = Directory.GetFiles(sourceFolder);
            foreach (string file in files)
            {
                string name = Path.GetFileName(file);
                string dest = Path.Combine(destFolder, name);
                File.Copy(file, dest, true);
            }

            string[] folders = Directory.GetDirectories(sourceFolder);
            foreach (string folder in folders)
            {
                string name = Path.GetFileName(folder);
                string dest = Path.Combine(destFolder, name);
                CopyFolder(folder, dest, ignoredDirectoryNames);
            }
        }

    }
}
