/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#define INCLUDE_CYBER

namespace CyPhy2Modelica
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics.Contracts;
    using System.IO;
    using System.Linq;
    using System.Runtime.InteropServices;
    using System.Text;
    using System.Windows.Forms;
    using System.Xml;
    using System.Xml.Serialization;

    using Microsoft.Win32;

    using GME;
    using GME.CSharp;
    using GME.MGA;
    using GME.MGA.Core;

    // using generic common interface
    using Common = ISIS.GME.Common;

    // using domain specific interfaces
    using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
    using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;
    using System.Diagnostics;

    /// <summary>
    /// This class implements the necessary COM interfaces for a GME interpreter component.
    /// </summary>
    [Guid(ComponentConfig.Guid),
    ProgId(ComponentConfig.ProgID),
    ClassInterface(ClassInterfaceType.AutoDual)]
    [ComVisible(true)]
    public partial class CyPhy2ModelicaInterpreter : IMgaComponentEx, IGMEVersionInfo
    {
        /// <summary>
        /// Is this component enabled
        /// </summary>
        private bool enabled = true;

        /// <summary>
        /// Type of the simulation engine
        /// </summary>
        private SimulationType simulationType = SimulationType.OpenModelica;

        private string logFileName;

        public string OutputDirectory { get; set; }

        public ParametersClass Parameters { get; set; }

        public bool Run { get; set; }

        #region IMgaComponentEx Members

        /// <summary>
        /// Gets or sets the gateway object to Multi Graph Architecture.
        /// </summary>
        internal MgaGateway MgaGateway { get; set; }

        /// <summary>
        /// Gets or sets GME console
        /// </summary>
        internal GMEConsole GMEConsole { get; set; }

        public static string Serialize<T>(T value)
        {

            if (value == null)
            {
                return null;
            }

            XmlSerializer serializer = new XmlSerializer(typeof(T));

            XmlWriterSettings settings = new XmlWriterSettings();
            settings.Encoding = new UnicodeEncoding(false, false); // no BOM in a .NET string
            settings.Indent = false;
            settings.OmitXmlDeclaration = false;

            using (StringWriter textWriter = new StringWriter())
            {
                using (XmlWriter xmlWriter = XmlWriter.Create(textWriter, settings))
                {
                    serializer.Serialize(xmlWriter, value);
                }
                return textWriter.ToString();
            }
        }

        public static T Deserialize<T>(string xml)
        {

            if (string.IsNullOrEmpty(xml))
            {
                return default(T);
            }

            XmlSerializer serializer = new XmlSerializer(typeof(T));

            XmlReaderSettings settings = new XmlReaderSettings();
            // No settings need modifying here

            using (StringReader textReader = new StringReader(xml))
            {
                using (XmlReader xmlReader = XmlReader.Create(textReader, settings))
                {
                    return (T)serializer.Deserialize(xmlReader);
                }
            }
        }

        /// <summary>
        /// This is the main entry point of this class.
        /// </summary>
        /// <param name="project">GME project</param>
        /// <param name="currentobj">Currently focused/opened model</param>
        /// <param name="selectedobjs">Selected objects</param>
        /// <param name="param">Type parameter of the call</param>
        public void InvokeEx(
            MgaProject project,
            MgaFCO currentobj,
            MgaFCOs selectedobjs,
            int param)
        {
            if (!this.enabled)
            {
                return;
            }
            string currentWorkDir = System.IO.Directory.GetCurrentDirectory();
            try
            {
                if (string.IsNullOrEmpty(this.Parameters.OriginalProjectFile))
                {
                    this.Parameters.OriginalProjectFile = project.ProjectConnStr.Substring("MGA=".Length);
                }

                System.IO.Directory.SetCurrentDirectory(Path.GetDirectoryName(this.Parameters.OriginalProjectFile));
                this.Run = false;

                // logfile in the project directory
                this.logFileName = Path.Combine(
                    Environment.CurrentDirectory,
                    "log",
                    this.ComponentName + "." + System.Diagnostics.Process.GetCurrentProcess().Id + ".log");

                META.Logger.AddFileListener(this.logFileName, this.ComponentName);

                Trace.TraceInformation("InvokeEx called");

                bool valid = false;

                MgaGateway.PerformInTransaction(() =>
                {
                    Trace.TraceInformation("Model file: {0}", project.ProjectConnStr);
                    Trace.TraceInformation("Paradigm file: {0}", project.ParadigmConnStr);

                    RetrieveSettingsFromRegistry(project);

                    valid = CheckModel(currentobj);
                },
                abort: false);

                if (valid == false)
                {
                    DumpAllMessageToGMEConsole();
                    DumpLog();
                    GMEConsole.Error.WriteLine("Model and/or context is invalid.");
                    Trace.TraceError("Invalid model. CheckModel is false.");
                    return;
                }

                MgaGateway.PerformInTransaction(delegate
                {
                    Prepare(currentobj);
                    CallDependenciesAndRun(project, currentobj, selectedobjs, param);
                },
                abort: true);

                MgaGateway.PerformInTransaction(delegate
                {
                    SaveSettingsInRegistry(project);
                },
                abort: false);

                if (this.Run)
                {
                    MgaGateway.PerformInTransaction(() =>
                    {
                        PostActions(currentobj);
                    },
                    abort: false);
                }

                GMEConsole.Info.WriteLine("Modelica interpreter has finished.");
                Trace.TraceInformation("{0} has finished.", this.ComponentName);
            }
            finally
            {
                if (MgaGateway != null)
                {
                    if (MgaGateway.territory != null)
                    {
                        MgaGateway.territory.Destroy();
                        //Marshal.FinalReleaseComObject(MgaGateway.territory);
                    }
                }

                MgaGateway = null;
                project = null;
                currentobj = null;
                selectedobjs = null;
                GMEConsole = null;

                ElaboratedObjects = null;

                GC.Collect();
                GC.WaitForPendingFinalizers();
                GC.Collect();
                System.IO.Directory.SetCurrentDirectory(currentWorkDir);
            }

            META.Logger.RemoveFileListener(this.ComponentName);
        }

        private void SaveSettingsInRegistry(MgaProject project)
        {
            try
            {
                CyPhy2ModelicaSettings settings = new CyPhy2ModelicaSettings();

                settings.IncludeDirectoryPath = CyPhy2Modelica.Properties.Settings.Default.IncludeDirectoryPath.Cast<string>().ToList();

                settings.NonCheckedIncludeDirPaths = CyPhy2Modelica.Properties.Settings.Default.NonCheckedIncludeDirPaths.Cast<string>().ToList();

                var settingsFilename = Path.Combine(Path.GetDirectoryName(project.ProjectConnStr.Substring("MGA=".Length)), CyPhy2ModelicaSettings.ConfigFilename);
                XmlSerializer xs = new XmlSerializer(typeof(CyPhy2ModelicaSettings));
                using (XmlWriter writer = XmlWriter.Create(settingsFilename))
                {
                    xs.Serialize(writer, settings);
                }
            }
            catch (Exception ex)
            {
                Trace.TraceWarning(ex.ToString());
            }
        }

        private void RetrieveSettingsFromRegistry(MgaProject project)
        {
            try
            {
                if (this.Parameters.Automation == "true")
                {
                    CyPhy2ModelicaSettings settings = new CyPhy2ModelicaSettings();
                    
                    var settingsFilename = Path.Combine(Path.GetDirectoryName(project.ProjectConnStr.Substring("MGA=".Length)), CyPhy2ModelicaSettings.ConfigFilename);

                    XmlSerializer xs = new XmlSerializer(typeof(CyPhy2ModelicaSettings));
                    using (XmlReader reader = XmlReader.Create(settingsFilename))
                    {
                        settings = xs.Deserialize(reader) as CyPhy2ModelicaSettings;
                    }

                    CyPhy2Modelica.Properties.Settings.Default.IncludeDirectoryPath = new System.Collections.Specialized.StringCollection();
                    CyPhy2Modelica.Properties.Settings.Default.IncludeDirectoryPath.AddRange(settings.IncludeDirectoryPath.ToArray());

                    CyPhy2Modelica.Properties.Settings.Default.NonCheckedIncludeDirPaths = new System.Collections.Specialized.StringCollection();
                    CyPhy2Modelica.Properties.Settings.Default.NonCheckedIncludeDirPaths.AddRange(settings.NonCheckedIncludeDirPaths.ToArray());
                }
            }
            catch (Exception ex)
            {
                Trace.TraceWarning(ex.ToString());
            }
        }

        private void PostActions(MgaFCO currentobj)
        {
            if (currentobj.Meta.Name == "TestBench")
            {
                var tb = CyPhyClasses.TestBench.Cast(currentobj);
                tb.Attributes.SpreadsheetLoc = Path.Combine(this.OutputDirectory, "package.mo");
                tb.Attributes.Type = CyPhyClasses.TestBench.AttributesClass.Type_enum.Dynamics_Simulation;
            }

            if (GenerateModelicaModel.Messages.Any(x => x.Severity == GmeMessage.Severity_enum.Error))
            {
                GMEConsole.Error.WriteLine(
                    "Generated files are invalid. They serve only debugging purposes.");
            }
            else
            {
                if (this.Parameters.Automation == "false")
                {
                    if (CyPhy2Modelica.Properties.Settings.Default.RunSimOM)
                    {
                        string filename = "om_simulate.py";
                        MgaGateway.PerformInTransaction(() =>
                        {
                            RunSimulation(this.OutputDirectory, filename, currentobj);
                        },
                        transactiontype_enum.TRANSACTION_GENERAL,
                        abort: false);
                    }

                    if (CyPhy2Modelica.Properties.Settings.Default.RunSimDymola)
                    {
                        string filename = "dymola_simulate.py";
                        MgaGateway.PerformInTransaction(() =>
                        {
                            RunSimulation(this.OutputDirectory, filename, currentobj);
                        },
                        transactiontype_enum.TRANSACTION_GENERAL,
                        abort: false);
                    }
                }
            }
        }

        private void CallDependenciesAndRun(
            MgaProject project,
            MgaFCO currentobj,
            MgaFCOs selectedobjs,
            int param)
        {

#if INCLUDE_CYBER
            // call cyber first the model does not need to be elaborated
            CallCyber(project, currentobj, selectedobjs, param);
#endif

            // elaborate the model for the modelica interpreter.
            CallElaborator(project, currentobj, selectedobjs, param);

            // Assumption: transaction is still open
            this.Main(project, currentobj, selectedobjs, this.Convert(param));
        }

        internal static CyPhyCOMInterfaces.IMgaTraceability ElaboratedObjects { get; set; }
        private void CallElaborator(
            MgaProject project,
            MgaFCO currentobj,
            MgaFCOs selectedobjs,
            int param)
        {
            // call elaborator and expand the references
            Type t = Type.GetTypeFromProgID("MGA.Interpreter.CyPhyElaborate");
            IMgaComponentEx elaborator = Activator.CreateInstance(t) as IMgaComponentEx;
            elaborator.Initialize(project);
            elaborator.ComponentParameter["automated_expand"] = "true";
            elaborator.ComponentParameter["console_messages"] = "off";

            try
            {
                GMEConsole.Info.WriteLine("Elaborating model...");
                System.Windows.Forms.Application.DoEvents();

                elaborator.InvokeEx(project, currentobj, selectedobjs, param);

                CyPhyCOMInterfaces.IMgaTraceability traceability = elaborator.ComponentParameter["traceability"] as CyPhyCOMInterfaces.IMgaTraceability;
                ElaboratedObjects = new MgaTraceability();
                if (traceability != null)
                {
                    traceability.CopyTo(ElaboratedObjects);
                }

                GMEConsole.Info.WriteLine("Elaboration is done.");
                System.Windows.Forms.Application.DoEvents();
            }
            catch (Exception ex)
            {
                GMEConsole.Error.WriteLine(ex);
                return;
            }

            // TODO: get exception message(s) from elaborator
            string msgs = elaborator.ComponentParameter["exception"] as string;
            if (string.IsNullOrWhiteSpace(msgs) == false)
            {
                GMEConsole.Error.WriteLine("Elaborator exception occured: {0}:", msgs);
                System.Windows.Forms.Application.DoEvents();
                throw new Exception(msgs);
            }
        }

        private void CallCyber(
            MgaProject project,
            MgaFCO currentobj,
            MgaFCOs selectedobjs,
            int param)
        {
            try
            {
                // call CyPhy2SLC_CodeGen
                // project is already in transaction
                Type tCyber = Type.GetTypeFromProgID("MGA.Interpreter.CyPhy2SLC_CodeGen");
                if (tCyber == null)
                {
                    GenerateModelicaModel.Messages.Add(GmeMessage.CyberNotInstalled());
                }
                IMgaComponentEx cyberCodeGenerator = Activator.CreateInstance(tCyber) as IMgaComponentEx;

                cyberCodeGenerator.Initialize(project);

                cyberCodeGenerator.ComponentParameter["output_dir"] = this.Parameters.OutputDir;
                cyberCodeGenerator.ComponentParameter["automation"] = this.Parameters.Automation;
                cyberCodeGenerator.ComponentParameter["console_messages"] = "off";
                GMEConsole.Info.WriteLine("Generating code for SignalFlow elements...");
                System.Windows.Forms.Application.DoEvents();
                cyberCodeGenerator.InvokeEx(project, currentobj, selectedobjs, param);
                GMEConsole.Info.WriteLine("SignalFlow code generation is done.");
                System.Windows.Forms.Application.DoEvents();
            }
            catch (Exception ex)
            {
                GenerateModelicaModel.Messages.Add(GmeMessage.CyberFailed(ex));
            }
        }

        private void Prepare(MgaFCO currentobj)
        {
            if (this.Parameters.ConsoleMessages == "on")
            {
                GMEConsole.Clear();
                System.Windows.Forms.Application.DoEvents();
            }

            this.OutputDirectory = this.Parameters.OutputDir;

            if (currentobj != null &&
                string.IsNullOrEmpty(this.OutputDirectory))
            {
                this.OutputDirectory = Path.Combine(
                    Path.GetDirectoryName(currentobj.Project.ProjectConnStr.Substring("MGA=".Length)),
                    "results",
                     Modelica.Factory.GetModifiedName(currentobj.Name));

                this.OutputDirectory = Path.GetFullPath(this.OutputDirectory);
                this.Parameters.OutputDir = this.OutputDirectory;

                this.Parameters.PackageName = Modelica.Factory.GetModifiedName(currentobj.Name);
            }
            else
            {
                this.Parameters.PackageName = Path.GetDirectoryName(this.OutputDirectory);
            }

            if (Directory.Exists(this.OutputDirectory) == false)
            {
                Directory.CreateDirectory(this.OutputDirectory);
            }

            // logfile to the output directory
            // FIXME: breaks the remote execution
            //logFileName = Path.Combine(this.OutputDirectory, "log", this.ComponentName + ".log");
            //META.Logger.AddFileListener(logFileName, this.ComponentName);

            CyPhy.TestBench testBench = CyPhyClasses.TestBench.Cast(currentobj);

            this.Parameters.DesignName = testBench
                .Children
                .TopLevelSystemUnderTestCollection
                .FirstOrDefault()
                .Referred
                .DesignEntity
                .Name;


            string testBenchName = string.IsNullOrEmpty(this.Parameters.TestBench) ?
                testBench.Name :
                this.Parameters.TestBench;

            // generate summary file (Json)
            AVM.DDP.MetaTBReport report = new AVM.DDP.MetaTBReport();

            string reportFile = Path.GetFullPath(
                Path.Combine(this.OutputDirectory, "summary.testresults.json"));

            report.GenerateSummary(
                testBench,
                "summary.testresults.json",
                this.OutputDirectory);


            /* TODO: ????????????
            this.GeneratePostProcessingScripts(
                this.componentParameters["output_dir"] as string,
                testBench);*/

            string outputPostProcDir = Path.Combine(this.OutputDirectory, "PostProcessing");

            foreach (var postProc in testBench.Children.PostProcessingCollection)
            {
                if (Directory.Exists(outputPostProcDir) == false)
                {
                    Directory.CreateDirectory(outputPostProcDir);
                }

                if (string.IsNullOrWhiteSpace(postProc.Attributes.ScriptPath) == false &&
                    File.Exists(postProc.Attributes.ScriptPath))
                {
                    // Copy postScript file
                    FileInfo postScript = new FileInfo(Path.GetFullPath(postProc.Attributes.ScriptPath));
                    string destFile = Path.Combine(outputPostProcDir, postScript.Name);
                    File.Copy(postScript.FullName, destFile, true);

                    // If common folder exist copy entire folder.
                    DirectoryInfo commonSrc = new DirectoryInfo(Path.Combine(postScript.DirectoryName, "common"));
                    DirectoryInfo commonTarget = new DirectoryInfo(Path.Combine(outputPostProcDir, "common"));
                    if (commonSrc.Exists && commonTarget.Exists == false)
                    {
                        commonTarget.Create();
                        CopyFilesRecursively(commonSrc, commonTarget);
                    }
                }
            }
        }

        private bool CheckModel(MgaFCO currentobj)
        {
            bool valid = true;
            // current object exist
            // current object is a test bench
            if (currentobj == null ||
                currentobj.Meta.Name != typeof(CyPhy.TestBench).Name)
            {
                GMEConsole.Error.WriteLine("Test bench must be open.");
                Trace.TraceError("Current object: NULL");
                valid = false;
                return valid;
            }

            Trace.TraceInformation("Current object: {0}", currentobj.AbsPath);

            CyPhy.TestBench testbench = CyPhyClasses.TestBench.Cast(currentobj);

            // Check null end points for connections
            MgaFilter filter = currentobj.Project.CreateFilter();
            filter.ObjType = GME.MGA.Meta.objtype_enum.OBJTYPE_CONNECTION.ToString();

            var invalidConnections = currentobj.Project
                .AllFCOs(filter)
                .Cast<IMgaSimpleConnection>()
                .Where(x => x.Src == null || x.Dst == null);

            if (invalidConnections.Count() > 0)
            {
                GMEConsole.Error.WriteLine("GME model is invalid due connections with null endpoints");
                System.Windows.Forms.Application.DoEvents();
                valid = false;
                throw new Exception("GME model is invalid due connections with null endpoints");
            }

            // no components
            // no component instances
            foreach (var cref in testbench.Children.ComponentCollection)
            {
                GenerateModelicaModel.Messages.Add(GmeMessage.ComponentsAreNotSupportedInTB(cref.ToHyperLink()));
                valid = false;
            }

            // no component assemblies
            foreach (var cref in testbench.Children.ComponentAssemblyCollection)
            {
                GenerateModelicaModel.Messages.Add(GmeMessage.ComponentAssembliesAreNotSupportedInTB(cref.ToHyperLink()));
                valid = false;
            }

            // no component references
            foreach (var cref in testbench.Children.ComponentRefCollection)
            {
                GenerateModelicaModel.Messages.Add(GmeMessage.ComponentRefsAreNotSupportedInTB(cref.ToHyperLink()));
                valid = false;
            }

            // no test injection points
            foreach (var cref in testbench.Children.TestInjectionPointCollection)
            {
                GenerateModelicaModel.Messages.Add(GmeMessage.TIPsAreNotSupportedInTB(cref.ToHyperLink()));
                valid = false;
            }

            // only one top level system under test
            int numTlsuts = testbench.Children.TopLevelSystemUnderTestCollection.Count();
            if (numTlsuts == 0)
            {
                GenerateModelicaModel.Messages.Add(GmeMessage.NoTLSUTInTB());
                valid = false;
            }
            else if (numTlsuts == 1)
            {
                // only one TLSUT
                // tlsut must point to component or CA
                // one componentassembly top level system under test
                // one toplevelsystemundertest
                var tlsut = testbench.Children.TopLevelSystemUnderTestCollection.FirstOrDefault();

                if (tlsut.Impl is MgaReference &&
                    (tlsut.Impl as MgaReference).Referred == null)
                {
                    GenerateModelicaModel.Messages.Add(GmeMessage.TLSUTMustPointToCorCAInTB(tlsut.ToHyperLink()));
                    valid = false;
                }
                else if ((tlsut.Impl as MgaReference).Referred.Meta.Name != typeof(CyPhy.Component).Name &&
                    (tlsut.Impl as MgaReference).Referred.Meta.Name != typeof(CyPhy.ComponentAssembly).Name)
                {
                    GenerateModelicaModel.Messages.Add(GmeMessage.TLSUTMustPointToCorCAInTB(tlsut.ToHyperLink()));
                    valid = false;
                }
            }
            else if (numTlsuts > 1)
            {
                foreach (var item in testbench.Children.TopLevelSystemUnderTestCollection)
                {
                    GenerateModelicaModel.Messages.Add(GmeMessage.MoreThanOneTLSUTInTB(item.ToHyperLink()));
                    valid = false;
                }
            }

            CheckValueFlow(testbench);

            // one workflow ref with one modelica task
            var workflowRef = testbench.Children.WorkflowRefCollection.FirstOrDefault();

            if (workflowRef == null)
            {
                // no workflow
                GenerateModelicaModel.Messages.Add(GmeMessage.OneWorkflowInTB(testbench.ToHyperLink()));
                valid = false;
            }
            else if ((workflowRef.Impl as MgaReference).Referred == null ||
                     testbench.Children.WorkflowRefCollection.Count() != 1)
            {
                // invalid workflow
                foreach (var item in testbench.Children.WorkflowRefCollection)
                {
                    GenerateModelicaModel.Messages.Add(GmeMessage.OneWorkflowInTB(item.ToHyperLink()));
                }
                valid = false;
            }
            else
            {
                var task = workflowRef.Referred.Workflow.Children.TaskCollection.FirstOrDefault();

                if (task != null &&
                    workflowRef.Referred.Workflow.Children.TaskCollection.Count() == 1 &&
                    task.Attributes.COMName == this.ComponentProgID)
                {
                    if (string.IsNullOrWhiteSpace(task.Attributes.Parameters) == false)
                    {
                        try
                        {
                            var parameters = Newtonsoft.Json.Linq.JObject.Parse(task.Attributes.Parameters);
                            if (parameters["simulationTargetTool"] != null)
                            {
                                this.Parameters.SimulationTargetTool = parameters["simulationTargetTool"].ToObject<string>();

                                this.simulationType = SimulationType.OpenModelica;

                                if (this.Parameters.SimulationTargetTool == "OpenModelica")
                                {

                                    this.simulationType = SimulationType.OpenModelica;
                                    this.Parameters.RunCommand = "om_simulate.py";
                                    this.Parameters.Labels = "OpenModelica && py_modelica" + META.VersionInfo.PyModelica;
                                }
                                else if (this.Parameters.SimulationTargetTool == "Dymola")
                                {
                                    this.simulationType = SimulationType.Dymola;
                                    this.Parameters.RunCommand = "dymola_simulate.py";
                                    this.Parameters.Labels = "Dymola && py_modelica" + META.VersionInfo.PyModelica;
                                }
                            }
                        }
                        catch (Newtonsoft.Json.JsonReaderException ex)
                        {
                            System.Diagnostics.Debug.WriteLine(ex);
                        }
                    }

                    GMEConsole.Info.WriteLine(
                        "Using target tool: {0}",
                        this.Parameters.SimulationTargetTool);
                }
                else
                {
                    // invalid workflow with multiple tasks
                    GenerateModelicaModel.Messages.Add(GmeMessage.WorkflowHasMoreThanOneTask(this.ComponentProgID));
                    valid = false;
                }
            }

            // Post processing script existence check
            foreach (var postProc in testbench.Children.PostProcessingCollection)
            {
                if (string.IsNullOrWhiteSpace(postProc.Attributes.ScriptPath))
                {
                    GenerateModelicaModel.Messages.Add(GmeMessage.PostProcScriptPathEmpty(postProc.ToHyperLink()));
                    valid = false;
                }
                else
                {
                    string scriptPath = Path.GetFullPath(postProc.Attributes.ScriptPath);
                    if (File.Exists(scriptPath) == false)
                    {
                        GenerateModelicaModel.Messages.Add(GmeMessage.PostProcScriptDoesNotExist(postProc.ToHyperLink(), scriptPath));
                        valid = false;
                    }
                }
            }


            return valid;
        }

        /// <summary>
        /// Converts the start mode parameter into an enumeration.
        /// </summary>
        /// <param name="param">start parameter</param>
        /// <returns>Start mode</returns>
        private ComponentStartMode Convert(int param)
        {
            switch (param)
            {
                case (int)ComponentStartMode.GME_BGCONTEXT_START:
                    return ComponentStartMode.GME_BGCONTEXT_START;
                case (int)ComponentStartMode.GME_BROWSER_START:
                    return ComponentStartMode.GME_BROWSER_START;

                case (int)ComponentStartMode.GME_CONTEXT_START:
                    return ComponentStartMode.GME_CONTEXT_START;

                case (int)ComponentStartMode.GME_EMBEDDED_START:
                    return ComponentStartMode.GME_EMBEDDED_START;

                case (int)ComponentStartMode.GME_ICON_START:
                    return ComponentStartMode.GME_ICON_START;

                case (int)ComponentStartMode.GME_MAIN_START:
                    return ComponentStartMode.GME_MAIN_START;

                case (int)ComponentStartMode.GME_MENU_START:
                    return ComponentStartMode.GME_MENU_START;
                case (int)ComponentStartMode.GME_SILENT_MODE:
                    return ComponentStartMode.GME_SILENT_MODE;
            }

            return ComponentStartMode.GME_SILENT_MODE;
        }

        #region Component Information
        /// <summary>
        /// Gets the name of the component
        /// </summary>
        public string ComponentName
        {
            get { return GetType().Name; }
        }

        /// <summary>
        /// Gets the Program Identifier
        /// </summary>
        public string ComponentProgID
        {
            get
            {
                return ComponentConfig.ProgID;
            }
        }

        /// <summary>
        /// Gets the type of the component
        /// </summary>
        public componenttype_enum ComponentType
        {
            get { return ComponentConfig.ComponentType; }
        }

        /// <summary>
        /// Gets the associated paradigm
        /// </summary>
        public string Paradigm
        {
            get { return ComponentConfig.ParadigmName; }
        }
        #endregion

        #region Enabling

        /// <summary>
        /// Enables or disables of this component
        /// </summary>
        /// <param name="newval">true if enable</param>
        public void Enable(bool newval)
        {
            this.enabled = newval;
        }

        #endregion

        #region Interactive Mode
        /// <summary>
        /// The call is interactive
        /// </summary>
        private bool interactiveMode = true;

        /// <summary>
        /// Gets or sets a value indicating whether the mode is interactive or not.
        /// </summary>
        public bool InteractiveMode
        {
            get
            {
                return this.interactiveMode;
            }

            set
            {
                this.interactiveMode = value;
            }
        }
        #endregion

        #region Custom Parameters
        /// <summary>
        /// Parameters of this component
        /// </summary>
        private SortedDictionary<string, object> componentParameters = null;

        /// <summary>
        /// Gets a component parameter based on its name
        /// </summary>
        /// <param name="name">name of the parameter</param>
        /// <returns>value of the parameter</returns>
        public object get_ComponentParameter(string name)
        {
            if (name == "type")
            {
                return "csharp";
            }

            if (name == "path")
            {
                return GetType().Assembly.Location;
            }

            if (name == "fullname")
            {
                return GetType().FullName;
            }

            object value;
            if (this.componentParameters != null && this.componentParameters.TryGetValue(name, out value))
            {
                return value;
            }

            return null;
        }

        /// <summary>
        /// Sets a component parameter based on its name and value
        /// </summary>
        /// <param name="name">name of the parameter</param>
        /// <param name="value">new value of the parameter</param>
        public void set_ComponentParameter(string name, object value)
        {
            if (name == "do_config_now")
            {
                MgaProject project = (MgaProject)value;
                // show main form
                DialogResult ok;
                using (MainForm mf = new MainForm())
                {
                    if (this.Parameters.Automation == "true")
                    {
                        mf.chbRunDymolaSimulation.Visible = false;
                        mf.chbRunOMSimulation.Visible = false;
                    }

                    mf.ShowDialog();
                    ok = mf.DialogResult;
                }
                if (ok == DialogResult.OK)
                {
                    SaveSettingsInRegistry(project);
                }
                return;
            }
            if (this.componentParameters == null)
            {
                this.componentParameters = new SortedDictionary<string, object>();
            }

            this.componentParameters[name] = value;
        }
        #endregion

        #region Unused Methods
        /// <summary>
        /// Old interface, it is never called for MGA Component Ex interfaces
        /// </summary>
        /// <param name="project">MGA project</param>
        /// <param name="selectedobjs">selected objects</param>
        /// <param name="param">component start mode parameter</param>
        public void Invoke(MgaProject project, MgaFCOs selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        /// <summary>
        /// Not used by GME
        /// </summary>
        /// <param name="project">MGA project</param>
        /// <param name="currentobj">focused object</param>
        /// <param name="selectedobjs">selected objects</param>
        /// <param name="param">component start mode parameter</param>
        public void ObjectsInvokeEx(MgaProject project, MgaObject currentobj, MgaObjects selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        #endregion

        #endregion

        #region IMgaVersionInfo Members

        /// <summary>
        /// Gets the interface version
        /// </summary>
        public GMEInterfaceVersion_enum version
        {
            get { return GMEInterfaceVersion_enum.GMEInterfaceVersion_Current; }
        }

        #endregion

        #region Registration Helpers

        /// <summary>
        /// Registers the given type
        /// </summary>
        /// <param name="t">Type to register</param>
        [ComRegisterFunctionAttribute]
        public static void GMERegister(Type t)
        {
            Registrar.RegisterComponentsInGMERegistry();
        }

        /// <summary>
        /// Unregisters the given type
        /// </summary>
        /// <param name="t">Type to unregister</param>
        [ComUnregisterFunctionAttribute]
        public static void GMEUnRegister(Type t)
        {
            Registrar.UnregisterComponentsInGMERegistry();
        }

        #endregion

        /// <summary>
        /// This function is called for each interpreter invocation before Main.
        /// Don't perform MGA operations here unless you open a transaction.
        /// </summary>
        /// <param name="project">
        /// The handle of the project opened in GME, for which the interpreter was called.
        /// </param>
        public void Initialize(MgaProject project)
        {
            Contract.Requires(project != null);

            GMEConsole = GMEConsole.CreateFromProject(project);
            MgaGateway = new MgaGateway(project);
            project.CreateTerritoryWithoutSink(out MgaGateway.territory);

            // Clear all static global variables
            BondGraph.PowerChain.PowerChains.Clear();
            BondGraph.PowerChain.ConnectionParents.Clear();
            GenerateModelicaModel.Messages.Clear();
            GmeMessage.SimpleCustomFormulas = new List<string>();
        }

        public CyPhy2ModelicaInterpreter()
        {
            // set up the default component parameters
            this.componentParameters = new SortedDictionary<string, object>();

            this.Parameters = new ParametersClass(this);

            if (CyPhy2Modelica.Properties.Settings.Default.PackageFolders == null)
            {
                CyPhy2Modelica.Properties.Settings.Default.PackageFolders =
                    new System.Collections.Specialized.StringCollection();
            }
            this.componentParameters.Add(
                "lib_package_folder",
                CyPhy2Modelica.Properties.Settings.Default.PackageFolders.Cast<string>().ToList());

            if (CyPhy2Modelica.Properties.Settings.Default.PackageNames == null)
            {
                CyPhy2Modelica.Properties.Settings.Default.PackageNames =
                    new System.Collections.Specialized.StringCollection();
            }
            this.componentParameters.Add(
                "lib_package_name",
                CyPhy2Modelica.Properties.Settings.Default.PackageNames.Cast<string>().ToList());


        }

        /// <summary>
        /// The main entry point of the interpreter. A transaction is already open,
        /// GMEConsole is available. A general try-catch block catches all the 
        /// exceptions coming from this function, you don't need to add it.
        /// For more information, see InvokeEx.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which
        /// the interpreter was called.</param>
        /// <param name="currentobj">The model open in the active tab in GME. 
        /// Its value is null if no model is open (no GME modeling windows open). 
        /// </param>
        /// <param name="selectedobjs">
        /// A collection for the selected  model elements. It is never null.
        /// If the interpreter is invoked by the context menu of the GME Tree Browser, 
        /// then the selected items in the tree browser. Folders
        /// are never passed (they are not FCOs).
        /// If the interpreter is invoked by clicking on the toolbar icon or the 
        /// context menu of the modeling window, then the selected items 
        /// in the active GME modeling window. If nothing is selected, the 
        /// collection is empty (contains zero elements).
        /// </param>
        /// <param name="startMode">Contains information about the GUI event that 
        /// initiated the invocation.</param>
        [ComVisible(false)]
        public void Main(
            MgaProject project,
            MgaFCO currentobj,
            MgaFCOs selectedobjs,
            ComponentStartMode startMode)
        {
            try
            {
                if (this.Parameters.SimulationTargetTool == "OpenModelica")
                {
                    this.simulationType = SimulationType.OpenModelica;
                    this.Parameters.RunCommand = "om_simulate.py";
                    this.Parameters.Labels = "OpenModelica && py_modelica" + META.VersionInfo.PyModelica;
                }
                else if (this.Parameters.SimulationTargetTool == "Dymola")
                {
                    this.simulationType = SimulationType.Dymola;
                    this.Parameters.RunCommand = "dymola_simulate.py";
                    this.Parameters.Labels = "Dymola && py_modelica" + META.VersionInfo.PyModelica;
                }

                this.Process(currentobj);
                this.DumpLog();
            }
            catch (PathTooLongException ex)
            {
                GenerateModelicaModel.Messages.Add(
                    new GmeMessage(ex.Message, GmeMessage.Severity_enum.Error));

                GMEConsole.Error.WriteLine();
                this.DumpAllMessageToGMEConsole();
                this.DumpLog(ex);
            }
            catch (Exception ex)
            {
                GenerateModelicaModel.Messages.Add(new GmeMessage(ex.ToString(), GmeMessage.Severity_enum.Error));
                this.DumpAllMessageToGMEConsole();
                this.DumpLog(ex);
            }
        }

        private void DumpLog(Exception ex = null)
        {
            GenerateModelicaModel.Messages.ForEach((x) =>
            {
                Trace.TraceInformation("{0}: {1}", x.Severity, x.Message);
            });

            if (ex != null)
            {
                Trace.TraceError(ex.ToString());
            }

            GMEConsole.Info.WriteLine(
                "Log file is here: <a href=\"file:///{1}\" target=\"_blank\">{1}</a>. {0}",
                Path.GetFileName(this.logFileName),
                Path.GetDirectoryName(logFileName));
        }

        private void CheckValueFlow(CyPhy.TestBench testBench)
        {
            List<CyPhy.ValueFlowTarget> vfTargets = new List<CyPhy.ValueFlowTarget>();

            foreach (var parameter in testBench.Children.ParameterCollection)
            {
                GetValueFlowDestinations(parameter, vfTargets);
            }

            foreach (var property in testBench.Children.PropertyCollection)
            {
                GetValueFlowDestinations(property, vfTargets);
            }

            var nonParamPropTargets = vfTargets
                .Where(x =>
                    (x.Impl.MetaBase.Name == typeof(CyPhy.Parameter).Name) == false &&
                    (x.Impl.MetaBase.Name == typeof(CyPhy.Property).Name) == false)
                .ToList();

            foreach (var item in nonParamPropTargets)
            {
                // get all effected parameters/properties
                List<CyPhy.ValueFlowTarget> vfTargets2 = new List<CyPhy.ValueFlowTarget>();
                GetValueFlowDestinations(item, vfTargets2);

                // check if the parameters/properties effect any ModelicaModel parameter
                var par2Modelica = vfTargets2
                    .OfType<CyPhy.Parameter>()
                    .Select(x => x.AllDstConnections)
                    .ToList();

                var pro2Modelica = vfTargets2
                    .OfType<CyPhy.Property>()
                    .Select(x => x.AllDstConnections)
                    .ToList();

                if (par2Modelica.Where(x => x.Any(y => y.Impl.MetaBase.Name == "ModelicaParameterPortMap")).Count() +
                    pro2Modelica.Where(x => x.Any(y => y.Impl.MetaBase.Name == "ModelicaParameterPortMap")).Count() > 0)
                {
                    GenerateModelicaModel.Messages.Add(GmeMessage.ValueFlowPropagation(item));
                }
            }
        }

        private void GetValueFlowDestinations(
            CyPhy.ValueFlowTarget vfTarget,
            List<CyPhy.ValueFlowTarget> vfTargets)
        {
            if (vfTargets.FirstOrDefault(x => x.ID == vfTarget.ID) != null)
            {
                return;
            }

            vfTargets.Add(vfTarget);

            foreach (var item in vfTarget.DstConnections.ValueFlowCollection.Select(x => x.DstEnd as CyPhy.ValueFlowTarget))
            {
                if (item != null)
                {
                    GetValueFlowDestinations(item, vfTargets);
                }
            }
        }

        /// <summary>
        /// Gets all components from a specific starting point in the hierarchy.
        /// Note: This function is recursive.
        /// </summary>
        /// <param name="subject">Start model</param>
        /// <returns>Set of components that were found.</returns>
        private IEnumerable<CyPhy.Component> GetComponents(Common.Interfaces.Model subject)
        {
            Contract.Requires(subject != null);

            List<CyPhy.Component> result = new List<CyPhy.Component>();

            if (subject is CyPhy.Component)
            {
                var comp = subject as CyPhy.Component;
                if (comp.Children.ModelicaModelCollection.Count() > 1)
                {
                    result.Add(comp);
                }
            }
            else if (subject is CyPhy.ComponentAssembly)
            {
                foreach (var item in (subject as CyPhy.ComponentAssembly).Children.ComponentCollection)
                {
                    result.AddRange(this.GetComponents(item));
                }
            }
            else if (subject is CyPhy.TestBench)
            {
                foreach (var item in (subject as CyPhy.TestBench).Children.DesignElementCollection)
                {
                    result.AddRange(this.GetComponents(item));
                }
            }

            return result;
        }


        /// <summary>
        /// Contains information about the GUI event that initiated the invocation.
        /// </summary>
        public enum ComponentStartMode
        {
            /// <summary>
            /// Not used by GME
            /// </summary>
            GME_MAIN_START = 0,

            /// <summary>
            /// Right click in the GME Tree Browser window
            /// </summary>
            GME_BROWSER_START = 1,

            /// <summary>
            /// Using the context menu by right clicking a model element in the GME modeling window
            /// </summary>
            GME_CONTEXT_START = 2,

            /// <summary>
            /// Not used by GME
            /// </summary>
            GME_EMBEDDED_START = 3,

            /// <summary>
            /// Clicking on the toolbar icon, or using the main menu
            /// </summary>
            GME_MENU_START = 16,

            /// <summary>
            /// Using the context menu by right clicking the background of the GME modeling window
            /// </summary>
            GME_BGCONTEXT_START = 18,

            /// <summary>
            /// Not used by GME
            /// </summary>
            GME_ICON_START = 32,

            /// <summary>
            /// Not used by GME, available to testers not using GME
            /// </summary>
            GME_SILENT_MODE = 128
        }

        /// <summary>
        /// Type of the simulation engine
        /// </summary>
        [ComVisible(false)]
        public enum SimulationType
        {
            /// <summary>
            /// Open Modelica 
            /// </summary>
            OpenModelica,

            /// <summary>
            /// Dymola 2013
            /// </summary>
            Dymola
        }
    }
}
