/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

// using generic common interface
using Common = ISIS.GME.Common;

// using domain specific interfaces
using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;
using System.IO;

namespace CyPhy2Modelica
{
    public partial class GenerateModelicaModel
    {
        private void BuildConnections(
            Common.Interfaces.Model component,
            Modelica.Model m)
        {
            if (component is CyPhy.BondGraph)
            {
                BuildConnections(component as CyPhy.BondGraph, m);
            }
            else if (component is CyPhy.DesignElement)
            {
                BuildConnections(component as CyPhy.DesignElement, m);
            }
            else if (component is CyPhy.TestBench)
            {
                BuildConnections(component as CyPhy.TestBench, m);
            }
        }

        private void BuildConnections(
            CyPhy.TestBench component,
            Modelica.Model m)
        {
            foreach (var item in component.Children.PowerFlowCollection)
            {
                PowerPortConnection(item, m);
            }

            foreach (var item in component.Children.AggregatePortCompositionCollection)
            {
                PowerPortConnection(item, m);
            }

            foreach (var item in component.Children.ModelicaPort2ComponentPortPowerFlowCollection)
            {
                PowerPortConnection(item, m);
            }

            foreach (var item in component.Children.InformationFlowCollection)
            {
                BuildSignalConnection(item, "", "", m);
            }

            foreach (var item in component.Children.ValueFlowCollection)
            {
                BuildSignalConnection(item, Modelica.Factory.SignalPort, Modelica.Factory.SignalPort, m);
            }

            foreach (var item in component.Children.Signal2MetricCollection)
            {
                BuildSignalConnection(item, "", Modelica.Factory.SignalPort, m);
            }

            foreach (var item in component.Children.SignalConnectionCollection)
            {
                BuildSignalConnection(item, "", "", m);
            }

            foreach (var item in component.Children.MapToVTC_SignalCollection)
            {
                BuildSignalConnection(item, "", "", m);
            }

        }

        private void PowerPortConnection(
            Common.Interfaces.Connection item,
            Modelica.Model m)
        {
            BondGraph.PowerChain.ConnectionParents.Add(item, m);
            return;
        }


        private void BuildSignalConnection(
            Common.Interfaces.Connection item,
            string srcPortName,
            string dstPortName,
            Modelica.Model m)
        {

            if (item is CyPhy.ValueFlow)
            {
                var vf = item as CyPhy.ValueFlow;
                // Skip connections intentionally
                if (vf.SrcEnds.CustomFormula != null ||
                    vf.SrcEnds.SimpleFormula != null ||
                    vf.DstEnds.CustomFormula != null ||
                    vf.DstEnds.SimpleFormula != null)
                {
                    // OK
                    GmeMessage.SaveSimpleCustomFormula(vf.ToHyperLink());
                    return;
                }

                if (vf.SrcEnds.Parameter != null ||
                    vf.SrcEnds.Property != null)
                {
                    if (vf.SrcEnd.ParentContainer.ID == m.Impl.ID &&
                        vf.DstEnd.ParentContainer.ParentContainer.ID == m.Impl.ID)
                    {
                        var comp = m.Components.FirstOrDefault(x => x.Impl.ID == vf.DstEnd.ParentContainer.ID);
                        if (comp != null)
                        {
                            string value = null;
                            if (vf.SrcEnds.Parameter != null)
                            {
                                value = vf.SrcEnds.Parameter.Attributes.Value;
                            }
                            else if (vf.SrcEnds.Property != null)
                            {
                                value = vf.SrcEnds.Property.Attributes.Value;
                            }

                            value = string.IsNullOrEmpty(value) ?
                                 "0.0" :
                                 value;

                            var type = (vf.SrcEnd.Impl as GME.MGA.MgaFCO).RegistryValue["Type"];
                            if (string.IsNullOrEmpty(type))
                            {
                                bool res;
                                if (bool.TryParse(value, out res))
                                {
                                    type = "Boolean";
                                }
                                else
                                {
                                    double res_d;
                                    if (double.TryParse(value, out res_d))
                                    {
                                        type = "Real";
                                    }
                                    else
                                    {
                                        type = "ERROR_UNKNOWN_TYPE";
                                        GenerateModelicaModel.Messages.Add(
                                            GmeMessage.ParameterPropertyTypeNotSupported(item.GenericSrcEnd.Impl.Name));
                                    }
                                }
                            }

                            if (type != "ERROR_UNKNOWN_TYPE")
                            {
                                comp.Parameters[vf.DstEnd.Name] = vf.SrcEnd.Name;
                            }
                        }
                    }
                    else
                    {
                        GmeMessage.SaveSimpleCustomFormula(vf.ToHyperLink());
                    }
                    // value is set do not need to create a connection
                    return;
                }
                // TODO: Metrics
            }


            string srcWithLeadingDot = srcPortName;
            string dstWithLeadingDot = dstPortName;
            if (string.IsNullOrEmpty(srcPortName) == false)
            {
                srcWithLeadingDot = srcPortName.Substring(0, 1) == "." ? srcPortName : "." + srcPortName;
                srcPortName = srcWithLeadingDot.Substring(1);
            }

            if (string.IsNullOrEmpty(dstPortName) == false)
            {
                dstWithLeadingDot = dstPortName.Substring(0, 1) == "." ? dstPortName : "." + dstPortName;
                dstPortName = dstWithLeadingDot.Substring(1);
            }


            Modelica.Connection conn = new Modelica.Connection();
            // power flows are only between components
            Modelica.Component srcComp = null;
            if (item.ParentContainer.ID == item.GenericSrcEnd.ParentContainer.ID)
            {
                srcComp = m.Components.
                    FirstOrDefault(x => x.Impl.ID == item.GenericSrcEnd.ID);

                if (srcComp != null)
                {
                    conn.Src = new Modelica.Port(srcComp, item.GenericSrcEnd);
                    conn.Src.Name = srcPortName;
                    srcComp.Ports.Add(new Modelica.Port(srcComp, item.GenericSrcEnd));
                }
            }
            else
            {
                Modelica.Model srcCompParent = m.Models.
                    FirstOrDefault(x => x.Impl.ID == item.GenericSrcEnd.ParentContainer.ID);

                if (srcCompParent != null)
                {
                    srcComp = srcCompParent.Components.
                    FirstOrDefault(x => x.Impl.ID == item.GenericSrcEnd.ID);

                    if (srcComp != null)
                    {
                        conn.Src = new Modelica.Port(srcCompParent, item.GenericSrcEnd);
                        conn.Src.Name = srcComp.SpecialName + srcWithLeadingDot;
                        srcComp.Ports.Add(new Modelica.Port(srcComp, item.GenericSrcEnd));
                    }
                }
            }

            Modelica.Component dstComp = null;
            if (item.ParentContainer.ID == item.GenericDstEnd.ParentContainer.ID)
            {
                dstComp = m.Components.
                    FirstOrDefault(x => x.Impl.ID == item.GenericDstEnd.ID);

                if (dstComp != null)
                {
                    conn.Dst = new Modelica.Port(dstComp, item.GenericDstEnd);
                    conn.Dst.Name = dstPortName;
                    dstComp.Ports.Add(new Modelica.Port(dstComp, item.GenericDstEnd));
                }
            }
            else
            {
                Modelica.Model dstCompParent = m.Models.
                    FirstOrDefault(x => x.Impl.ID == item.GenericDstEnd.ParentContainer.ID);

                if (dstCompParent != null)
                {
                    dstComp = dstCompParent.Components.
                    FirstOrDefault(x => x.Impl.ID == item.GenericDstEnd.ID);

                    if (dstComp != null)
                    {
                        conn.Dst = new Modelica.Port(dstCompParent, item.GenericDstEnd);
                        conn.Dst.Name = dstComp.SpecialName + dstWithLeadingDot;
                        dstComp.Ports.Add(new Modelica.Port(dstComp, item.GenericDstEnd));
                    }

                }
            }

            if (srcComp != null &&
                dstComp != null)
            {

                if (item is CyPhy.InformationFlow ||
                    item is CyPhy.SignalConnection ||
                    item is CyPhy.MapToVTC_Signal)
                {
                    conn.Src.PortType = Modelica.Factory.PortType.ModelicaConnection;
                    conn.Dst.PortType = Modelica.Factory.PortType.ModelicaConnection;
                }

                m.Connections.Add(conn);
            }
            else if (item is CyPhy.ModulationConnection)
            {
                // handled as an inline function
            }
            else
            {
                m.Equations.Add(String.Format(
                    "// {0}.{1} -> {2}.{3}",
                    item.GenericSrcEnd.ParentContainer.Name,
                    item.GenericSrcEnd.Name,
                    item.GenericDstEnd.ParentContainer.Name,
                    item.GenericDstEnd.Name));

                Messages.Add(GmeMessage.SkippedConnection(item));
            }
        }

        private void BuildConnections(
            CyPhy.DesignElement component,
            Modelica.Model m)
        {
            foreach (var item in component.Children.ValueFlowCollection)
            {
                BuildSignalConnection(item, Modelica.Factory.SignalPort, Modelica.Factory.SignalPort, m);
            }

            foreach (var item in component.Children.SignalConnectionCollection)
            {
                BuildSignalConnection(item, Modelica.Factory.SignalPort, Modelica.Factory.SignalPort, m);
            }

            if (component is CyPhy.ComponentAssembly)
            {
                foreach (var item in (component as CyPhy.ComponentAssembly).Children.PowerFlowCollection)
                {
                    PowerPortConnection(item, m);
                }

                foreach (var item in (component as CyPhy.ComponentAssembly).Children.AggregatePortCompositionCollection)
                {
                    PowerPortConnection(item, m);
                }

                foreach (var item in (component as CyPhy.ComponentAssembly).Children.InformationFlowCollection)
                {
                    BuildSignalConnection(item, "", "", m);
                }
            }



            if (component is CyPhy.ComponentType)
            {
                foreach (var item in (component as CyPhy.ComponentType).Children.PhysicalPort2PowerPortCollection)
                {
                    PowerPortConnection(item, m);
                }

                foreach (var item in (component as CyPhy.ComponentType).Children.PowerPort2PhysicalPortCollection)
                {
                    PowerPortConnection(item, m);
                }

                foreach (var item in (component as CyPhy.ComponentType).Children.ModelicaPowerPortMapBaseCollection)
                {
                    PowerPortConnection(item, m);
                }

                foreach (var item in (component as CyPhy.ComponentType).Children.AggregatePortMapCollection)
                {
                    PowerPortConnection(item, m);
                }

                foreach (var item in (component as CyPhy.ComponentType).Children.ModelicaBusPortMapCollection)
                {
                    PowerPortConnection(item, m);
                }

                foreach (var item in (component as CyPhy.ComponentType).Children.SignalFlowPortMapCollection)
                {
                    PowerPortConnection(item, m);
                }

                foreach (var item in (component as CyPhy.ComponentType).Children.ValueFlow2SignalPortCollection)
                {
                    BuildSignalConnection(item, Modelica.Factory.SignalPort, Modelica.Factory.SignalPort, m);
                }

                foreach (var item in (component as CyPhy.ComponentType).Children.SignalPortMapCollection)
                {
                    BuildModelicaSignalConnection(item, m);
                }

                if (component is CyPhy.TestComponent)
                {
                    foreach (var item in (component as CyPhy.TestComponent).Children.ModelicaSignal2MetricCollection)
                    {
                        BuildModelicaSignalConnection(item, m);
                    }
                }
            }
        }



        private void BuildConnections(
            CyPhy.BondGraph component,
            Modelica.Model m)
        {
            foreach (var item in component.Children.ConnectionBaseCollection)
            {
                Modelica.Connection conn = new Modelica.Connection();

                if (item is CyPhy.BondE2J ||
                    item is CyPhy.BondJ2E ||
                    item is CyPhy.BondJ2J)
                {
                    //Modelica.Component srcComp = m.Components.
                    //  FirstOrDefault(x => x.OriginalName == item.SrcEnd.Name);

                    Modelica.Component srcComp = null;
                    if (item.ParentContainer.ID == item.SrcEnd.ParentContainer.ID)
                    {
                        srcComp = m.Components.
                            FirstOrDefault(x => x.OriginalName == item.SrcEnd.Name);
                    }
                    else
                    {
                        srcComp = m.Components.
                            FirstOrDefault(x => x.OriginalName == item.SrcEnd.ParentContainer.Name);
                    }

                    string signs = "";

                    if (item.SrcEnd is CyPhy.Junction)
                    {
                        CyPhy.Junction junction = item.SrcEnd as CyPhy.Junction;

                        if (srcComp.Parameters.TryGetValue(Modelica.Factory.JunctionSignsParameter, out signs))
                        {
                            srcComp.Parameters[Modelica.Factory.JunctionSignsParameter] = signs + ", 1";
                        }
                        else
                        {
                            srcComp.Parameters.Add(
                                Modelica.Factory.JunctionSignsParameter,
                                "1");

                            //bool initialState = false;
                            //if (junction.Attributes.InitialState == CyPhyClasses.Junction.AttributesClass.InitialState_enum.ON)
                            //{
                            //  initialState = true;
                            //}
                            //else if (junction.Attributes.InitialState == CyPhyClasses.Junction.AttributesClass.InitialState_enum.OFF)
                            //{
                            //  initialState = false;
                            //}

                            //srcComp.Parameters.Add(
                            //  Modelica.Factory.JunctionInitialState,
                            //  initialState.ToString().ToLowerInvariant());

                            //srcComp.Parameters.Add(
                            //  Modelica.Factory.JunctionOnCondition,
                            //  junction.Attributes.OnCondition);

                            //srcComp.Parameters.Add(
                            //  Modelica.Factory.JunctionOffCondition,
                            //  junction.Attributes.OffCondition);
                        }
                    }
                    else if (item.SrcEnd is CyPhy.Source)
                    {
                        if (srcComp.Parameters.ContainsKey(Modelica.Factory.SignParameter))
                        {
                            Messages.Add(new GmeMessage(
                                String.Format(
                                    "{0} does not have unique name.",
                                    item.SrcEnd.ToHyperLink()),
                                GmeMessage.Severity_enum.Error));
                        }
                        else
                        {
                            srcComp.Parameters.Add(
                                Modelica.Factory.SignParameter,
                                "1");
                        }
                    }
                    else if (item.SrcEnd is CyPhy.OnePort)
                    {
                        if (srcComp.Parameters.ContainsKey(Modelica.Factory.SignParameter))
                        {
                            Messages.Add(new GmeMessage(
                                String.Format(
                                    "{0} does not have unique name.",
                                    item.SrcEnd.ToHyperLink()),
                                GmeMessage.Severity_enum.Error));
                        }
                        else
                        {
                            srcComp.Parameters.Add(
                                Modelica.Factory.SignParameter,
                                "-1");
                        }
                    }


                    conn.Src = new Modelica.Port(srcComp, item.GenericSrcEnd) { PortType = Modelica.Factory.PortType.Power };
                    conn.Src.Name = String.Format("Port{1}", Modelica.Factory.PowerPort,
                                    srcComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1);
                    srcComp.Ports.Add(new Modelica.Port(srcComp, item.GenericSrcEnd) { PortType = Modelica.Factory.PortType.Power });

                    //Modelica.Component dstComp = m.Components.
                    //  FirstOrDefault(x => x.OriginalName == item.DstEnd.Name);

                    Modelica.Component dstComp = null;
                    if (item.ParentContainer.ID == item.DstEnd.ParentContainer.ID)
                    {
                        dstComp = m.Components.
                            FirstOrDefault(x => x.OriginalName == item.DstEnd.Name);
                    }
                    else
                    {
                        dstComp = m.Components.
                            FirstOrDefault(x => x.OriginalName == item.DstEnd.ParentContainer.Name);
                    }

                    if (item.DstEnd is CyPhy.Junction)
                    {
                        CyPhy.Junction junction = item.DstEnd as CyPhy.Junction;

                        if (dstComp.Parameters.TryGetValue(Modelica.Factory.JunctionSignsParameter, out signs))
                        {
                            dstComp.Parameters[Modelica.Factory.JunctionSignsParameter] = signs + ", -1";
                        }
                        else
                        {
                            dstComp.Parameters.Add(
                                Modelica.Factory.JunctionSignsParameter,
                                "-1");

                            //bool initialState = false;
                            //if (junction.Attributes.InitialState == CyPhyClasses.Junction.AttributesClass.InitialState_enum.ON)
                            //{
                            //  initialState = true;
                            //}
                            //else if (junction.Attributes.InitialState == CyPhyClasses.Junction.AttributesClass.InitialState_enum.OFF)
                            //{
                            //  initialState = false;
                            //}

                            //dstComp.Parameters.Add(
                            //  Modelica.Factory.JunctionInitialState,
                            //  initialState.ToString().ToLowerInvariant());

                            //dstComp.Parameters.Add(
                            //  Modelica.Factory.JunctionOnCondition,
                            //  junction.Attributes.OnCondition);

                            //dstComp.Parameters.Add(
                            //  Modelica.Factory.JunctionOffCondition,
                            //  junction.Attributes.OffCondition);
                        }
                    }
                    else if (item.DstEnd is CyPhy.Source)
                    {
                        if (dstComp.Parameters.ContainsKey(Modelica.Factory.SignParameter))
                        {
                            Messages.Add(new GmeMessage(
                                String.Format(
                                    "{0} does not have unique name.",
                                    item.DstEnd.ToHyperLink()),
                                GmeMessage.Severity_enum.Error));
                        }
                        else
                        {
                            dstComp.Parameters.Add(
                                Modelica.Factory.SignParameter,
                                "1"/*"-1"*/);
                        }
                    }
                    else if (item.DstEnd is CyPhy.OnePort)
                    {
                        if (dstComp.Parameters.ContainsKey(Modelica.Factory.SignParameter))
                        {
                            Messages.Add(new GmeMessage(
                                String.Format(
                                    "{0} does not have unique name.",
                                    item.DstEnd.ToHyperLink()),
                                GmeMessage.Severity_enum.Error));
                        }
                        else
                        {
                            dstComp.Parameters.Add(
                                Modelica.Factory.SignParameter,
                                "1");
                        }
                    }

                    conn.Dst = new Modelica.Port(dstComp, item.GenericDstEnd) { PortType = Modelica.Factory.PortType.Power };
                    conn.Dst.Name = String.Format("Port{1}", Modelica.Factory.PowerPort,
                                    dstComp.Ports.Where(x => x.PortType != Modelica.Factory.PortType.Signal).Count() + 1);
                    dstComp.Ports.Add(new Modelica.Port(dstComp, item.GenericDstEnd) { PortType = Modelica.Factory.PortType.Power });
                    m.Connections.Add(conn);
                }
                else if (item is CyPhy.PowerPort2Junction)
                {
                    CyPhy.PowerConnection powerConn = item as CyPhy.PowerConnection;

                    BondGraph.PowerChain.ConnectionParents.Add(powerConn, m);

                    // create only ModelicaModel connections
                    var ppConnection = item as CyPhy.PowerPort2Junction;

                    if (ppConnection.SrcEnd.ParentContainer is CyPhy.ModelicaModel)
                    {
                        BuildModelicaPowerConnection(ppConnection, m);
                    }

                }
                else if (item is CyPhy.Junction2PowerPort)
                {
                    CyPhy.PowerConnection powerConn = item as CyPhy.PowerConnection;

                    BondGraph.PowerChain.ConnectionParents.Add(powerConn, m);

                    // create only ModelicaModel connections
                    var ppConnection = item as CyPhy.Junction2PowerPort;

                    if (ppConnection.DstEnd.ParentContainer is CyPhy.ModelicaModel)
                    {
                        BuildModelicaPowerConnection(ppConnection, m);
                    }

                }
                else if (item is CyPhy.PowerConnection)
                {
                    // mechanicalR
                    // mechanicalD
                    // electrical
                    // hydraulic
                    // thermal
                    CyPhy.PowerConnection powerConn = item as CyPhy.PowerConnection;
                    PowerPortConnection(powerConn, m);
                }
                else
                {
                    // assumptions:
                    // - connections exist only between components
                    // - components have unique names

                    //Modelica.Component srcComp = m.Components.
                    //  FirstOrDefault(x => x.OriginalName == item.SrcEnd.ParentContainer.Name);

                    Modelica.Component srcComp = null;
                    if (item.ParentContainer.ID == item.SrcEnd.ParentContainer.ID)
                    {
                        srcComp = m.Components.
                            FirstOrDefault(x => x.OriginalName == item.SrcEnd.Name);
                    }
                    else
                    {
                        srcComp = m.Components.
                            FirstOrDefault(x => x.OriginalName == item.SrcEnd.ParentContainer.Name);
                    }

                    conn.Src = new Modelica.Port(srcComp, item.GenericSrcEnd);
                    if (item is CyPhy.SensingConnection)
                    {
                        if (item is CyPhy.Sensing2Signal)
                        {
                            conn.Src.Name = Modelica.Factory.SignalPort; //"Signal[2].SignalValue";
                        }
                        else if (item is CyPhy.OneJunction2Dq)
                        {
                            conn.Src.Name = CyPhy2Modelica.Modelica.Factory.CommonDisplacement.Substring(1);
                        }
                        else if (item is CyPhy.OneJunction2Df)
                        {
                            conn.Src.Name = CyPhy2Modelica.Modelica.Factory.CommonFlow.Substring(1);
                        }
                        else if (item is CyPhy.ZeroJunction2Dp)
                        {
                            conn.Src.Name = CyPhy2Modelica.Modelica.Factory.CommonMomentum.Substring(1);
                        }
                        else if (item is CyPhy.ZeroJunction2De)
                        {
                            conn.Src.Name = CyPhy2Modelica.Modelica.Factory.CommonEffort.Substring(1);
                        }

                    }
                    else if (item is CyPhy.MonitorConnection)
                    {
                        if (item is CyPhy.Signal2SignalMonitor)
                        {
                            conn.Src.Name = Modelica.Factory.SignalPort;
                        }
                        else if (item is CyPhy.OneJunction2MonitorDisplacement)
                        {
                            conn.Src.Name = CyPhy2Modelica.Modelica.Factory.CommonDisplacement.Substring(1);
                        }
                        else if (item is CyPhy.OneJunction2MonitorFlow)
                        {
                            conn.Src.Name = CyPhy2Modelica.Modelica.Factory.CommonFlow.Substring(1);
                        }
                        else if (item is CyPhy.ZeroJunction2MonitorMomentum)
                        {
                            conn.Src.Name = CyPhy2Modelica.Modelica.Factory.CommonMomentum.Substring(1);
                        }
                        else if (item is CyPhy.ZeroJunction2MonitorEffort)
                        {
                            conn.Src.Name = CyPhy2Modelica.Modelica.Factory.CommonEffort.Substring(1);
                        }
                    }
                    else if (item is CyPhy.ControlFunction2Signal)
                    {
                        conn.Src.Name = item.DstEnd.Name + "." + Modelica.Factory.SignalPort;
                    }
                    else
                    {
                        conn.Src.Name = Modelica.Factory.SignalPort; //item.SrcEnd.Name;
                    }
                    //Modelica.Component dstComp = m.Components.
                    //  FirstOrDefault(x => x.OriginalName == item.DstEnd.ParentContainer.Name);

                    Modelica.Component dstComp = null;
                    if (item.ParentContainer.ID == item.DstEnd.ParentContainer.ID)
                    {
                        dstComp = m.Components.
                            FirstOrDefault(x => x.OriginalName == item.DstEnd.Name);
                    }
                    else
                    {
                        dstComp = m.Components.
                            FirstOrDefault(x => x.OriginalName == item.DstEnd.ParentContainer.Name);
                    }

                    conn.Dst = new Modelica.Port(dstComp, item.GenericDstEnd);
                    if (item is CyPhy.Signal2BGMElement)
                    {
                        conn.Dst.Name = "ParameterSignal.SignalValue";
                    }
                    else if (item is CyPhy.Signal2InitialValue)
                    {
                        conn.Dst.Name = "InitialValueSignal";
                    }
                    else if (item is CyPhy.Signal2ControlFunction)
                    {
                        conn.Dst.Name = item.SrcEnd.Name + "." + Modelica.Factory.SignalPort;
                    }
                    else
                    {
                        conn.Dst.Name = Modelica.Factory.SignalPort; //item.DstEnd.Name;
                    }

                    if (item is Common.Interfaces.Connection &&
                        string.IsNullOrEmpty(conn.Src.Name) == false &&
                        string.IsNullOrEmpty(conn.Dst.Name) == false)
                    {
                        BuildSignalConnection(item, conn.Src.Name, conn.Dst.Name, m);
                        if (item is CyPhy.SignalConnection)
                        {
                            //BuildModelicaSignalConnection(item as CyPhy.SignalConnection, m);
                        }
                        //m.Connections.Add(conn);
                    }
                    else
                    {
                        Messages.Add(new GmeMessage("Connection was skipped: " + item.Kind, GmeMessage.Severity_enum.Error));
                    }
                }
            }
        }

        private void BuildModelicaPowerConnection(
            CyPhy.PowerConnection connection,
            Modelica.Model m)
        {
            // TODO: test this function

            Modelica.Component srcParent = null;
            Modelica.Component dstParent = null;

            foreach (var item in m.Components)
            {
                if (connection.SrcEnd.ParentContainer is CyPhy.ModelicaModel ||
                    connection.SrcEnd.ParentContainer is CyPhy.SignalFlowModel)
                {
                    if (item.Impl.ID == connection.SrcEnd.ParentContainer.ID)
                    {
                        srcParent = item;
                    }
                }
                else
                {
                    if (item.Impl.ID == connection.SrcEnd.ID)
                    {
                        srcParent = item;
                    }
                }

                if (connection.DstEnd.ParentContainer is CyPhy.ModelicaModel ||
                    connection.DstEnd.ParentContainer is CyPhy.SignalFlowModel)
                {
                    if (item.Impl.ID == connection.DstEnd.ParentContainer.ID)
                    {
                        dstParent = item;
                    }
                }
                else
                {
                    if (item.Impl.ID == connection.DstEnd.ID)
                    {
                        dstParent = item;
                    }
                }
            }

            if (srcParent != null || dstParent != null)
            {
                if (srcParent.Impl is CyPhy.ModelicaModel ||
                    srcParent.Impl is CyPhy.SignalFlowModel)
                {
                    var modelicaModel = m.Components.FirstOrDefault(
                        x => x.Impl.ID == connection.SrcEnd.ParentContainer.ID);

                    // power port
                    var pp = modelicaModel.Ports.FirstOrDefault(
                        x => x.Impl.ID == connection.SrcEnd.ID);

                    Modelica.Connection conn = new Modelica.Connection();

                    // modelica model's power port
                    conn.Src = new Modelica.Port(srcParent, pp.Impl);
                    conn.Src.Name = String.Format(
                        "{0}.{1}[1]",
                        pp.Name,
                        Modelica.Factory.PowerPort);
                    conn.Src.PortType = Modelica.Factory.PortType.ModelicaConnection;

                    // junction
                    conn.Dst = new Modelica.Port(dstParent, connection.DstEnd);
                    conn.Dst.Name = String.Format(
                        "Port{1}",
                        Modelica.Factory.PowerPort,
                        dstParent.Ports.Where(
                            x => x.PortType == Modelica.Factory.PortType.Power).Count() + 1);
                    conn.Dst.PortType = Modelica.Factory.PortType.ModelicaConnection;

                    m.Connections.Add(conn);

                    // add sign into the junction's parameters
                    CyPhy.Junction junction = connection.DstEnd as CyPhy.Junction;

                    string signs = "";
                    if (dstParent.Parameters.TryGetValue(Modelica.Factory.JunctionSignsParameter, out signs))
                    {
                        dstParent.Parameters[Modelica.Factory.JunctionSignsParameter] = signs + ", -1"; // HACK: SIGN
                    }
                    else
                    {
                        dstParent.Parameters.Add(
                            Modelica.Factory.JunctionSignsParameter,
                            "-1"); // HACK: SIGN
                    }

                }
                else if (dstParent.Impl is CyPhy.ModelicaModel ||
                    dstParent.Impl is CyPhy.SignalFlowModel)
                {
                    var modelicaModel = m.Components.FirstOrDefault(
                        x => x.Impl.ID == connection.DstEnd.ParentContainer.ID);

                    // power port
                    var pp = modelicaModel.Ports.FirstOrDefault(
                        x => x.Impl.ID == connection.DstEnd.ID);

                    Modelica.Connection conn = new Modelica.Connection();

                    // junction
                    conn.Src = new Modelica.Port(srcParent, connection.SrcEnd);
                    conn.Src.Name = String.Format(
                        "Port{1}",
                        Modelica.Factory.PowerPort,
                        srcParent.Ports.Where(
                            x => x.PortType == Modelica.Factory.PortType.Power).Count() + 1);
                    conn.Src.PortType = Modelica.Factory.PortType.ModelicaConnection;

                    // modelica model's power port
                    conn.Dst = new Modelica.Port(dstParent, pp.Impl);
                    conn.Dst.Name = String.Format(
                        "{0}.{1}[1]",
                        pp.Name,
                        Modelica.Factory.PowerPort);
                    conn.Dst.PortType = Modelica.Factory.PortType.ModelicaConnection;

                    m.Connections.Add(conn);

                    // add sign into the junction's parameters
                    CyPhy.Junction junction = connection.SrcEnd as CyPhy.Junction;

                    string signs = "";
                    if (srcParent.Parameters.TryGetValue(Modelica.Factory.JunctionSignsParameter, out signs))
                    {
                        srcParent.Parameters[Modelica.Factory.JunctionSignsParameter] = signs + ", -1";
                    }
                    else
                    {
                        srcParent.Parameters.Add(
                            Modelica.Factory.JunctionSignsParameter,
                            "-1");
                    }

                }
            }
        }

        private void BuildModelicaSignalConnection(
            CyPhy.SignalPortMap connection,
            Modelica.Model m)
        {
            // TODO: test this function
            Modelica.Component srcParent = null;
            Modelica.Component dstParent = null;

            foreach (var item in m.Components)
            {
                if (connection.GenericSrcEnd.ParentContainer.Kind == typeof(CyPhy.ModelicaModel).Name ||
                    connection.GenericSrcEnd.ParentContainer.Kind == typeof(CyPhy.SignalFlowModel).Name)
                {
                    if (item.Impl.ID == connection.GenericSrcEnd.ParentContainer.ID)
                    {
                        srcParent = item;
                    }
                }
                else
                {
                    if (connection.GenericSrcEnd.ParentContainer.ID == connection.ParentContainer.ID)
                    {
                        if (item.Impl.ID == connection.GenericSrcEnd.ID)
                        {
                            srcParent = item;
                        }
                    }
                    else
                    {
                        if (item.Impl.ID == connection.GenericSrcEnd.ParentContainer.ID)
                        {
                            var model = m.Models.FirstOrDefault(x => x.Impl.ID == connection.GenericSrcEnd.ParentContainer.ID);
                            srcParent = model.Components.FirstOrDefault(x => x.Impl.ID == connection.GenericSrcEnd.ID);
                        }
                    }
                }

                if (connection.GenericDstEnd.ParentContainer.Kind == typeof(CyPhy.ModelicaModel).Name ||
                    connection.GenericDstEnd.ParentContainer.Kind == typeof(CyPhy.SignalFlowModel).Name)
                {
                    if (item.Impl.ID == connection.GenericDstEnd.ParentContainer.ID)
                    {
                        dstParent = item;
                    }
                }
                else
                {
                    if (connection.GenericDstEnd.ParentContainer.ID == connection.ParentContainer.ID)
                    {
                        if (item.Impl.ID == connection.GenericDstEnd.ID)
                        {
                            dstParent = item;
                        }
                    }
                    else
                    {
                        if (item.Impl.ID == connection.GenericDstEnd.ParentContainer.ID)
                        {
                            var model = m.Models.FirstOrDefault(x => x.Impl.ID == connection.GenericDstEnd.ParentContainer.ID);
                            dstParent = model.Components.FirstOrDefault(x => x.Impl.ID == connection.GenericDstEnd.ID);
                        }
                    }
                }
            }

            if (srcParent != null || dstParent != null)
            {
                if (srcParent != null &&
                    (srcParent.Impl is CyPhy.ModelicaModel ||
                    srcParent.Impl is CyPhy.SignalFlowModel))
                {
                    var modelicaModel = m.Components.FirstOrDefault(
                        x => x.Impl.ID == connection.GenericSrcEnd.ParentContainer.ID);

                    // power port
                    var pp = modelicaModel.Ports.FirstOrDefault(
                        x => x.Impl.ID == connection.GenericSrcEnd.ID);

                    Modelica.Connection conn = new Modelica.Connection();

                    // modelica model's port
                    conn.Src = new Modelica.Port(srcParent, pp.Impl);
                    conn.Src.Name = String.Format(
                        "{0}",
                        pp.Name,
                        Modelica.Factory.CommonSignal);

                    conn.Src.PortType = Modelica.Factory.PortType.ModelicaConnection;
                    // 
                    conn.Dst = new Modelica.Port(dstParent, connection.GenericDstEnd);
                    conn.Dst.Name = "";
                    //conn.Dst.Name = String.Format(
                    //  "{0}",
                    //  Modelica.Factory.CommonSignal.Substring(1));

                    conn.Dst.PortType = Modelica.Factory.PortType.ModelicaConnection;

                    m.Connections.Add(conn);

                }
                else if (dstParent != null &&
                    (dstParent.Impl is CyPhy.ModelicaModel ||
                    dstParent.Impl is CyPhy.SignalFlowModel))
                {
                    var modelicaModel = m.Components.FirstOrDefault(
                        x => x.Impl.ID == connection.GenericDstEnd.ParentContainer.ID);

                    // port
                    var pp = modelicaModel.Ports.FirstOrDefault(
                        x => x.Impl.ID == connection.GenericDstEnd.ID);

                    Modelica.Connection conn = new Modelica.Connection();

                    // 
                    conn.Src = new Modelica.Port(srcParent, connection.GenericSrcEnd);
                    conn.Src.Name = "";
                    //conn.Src.Name = String.Format(
                    //   "{0}",
                    //   Modelica.Factory.CommonSignal.Substring(1));
                    conn.Src.PortType = Modelica.Factory.PortType.ModelicaConnection;


                    // modelica model's port
                    conn.Dst = new Modelica.Port(dstParent, pp.Impl);
                    conn.Dst.Name = String.Format(
                        "{0}",
                        pp.Name,
                        Modelica.Factory.CommonSignal);
                    conn.Dst.PortType = Modelica.Factory.PortType.ModelicaConnection;

                    m.Connections.Add(conn);

                }
            }

        }

        private void BuildModelicaSignalConnection(
            CyPhy.ModelicaParameterPortMap connection,
            Modelica.Model m)
        {
            var comp = m.Components.FirstOrDefault(x => x.Impl.ID == connection.DstEnd.ParentContainer.ID);
            if (comp != null)
            {
                string value = null;
                if (connection.SrcEnds.Parameter != null)
                {
                    value = connection.SrcEnds.Parameter.Attributes.Value;
                }
                else if (connection.SrcEnds.Property != null)
                {
                    value = connection.SrcEnds.Property.Attributes.Value;
                }

                value = string.IsNullOrEmpty(value) ?
                     "0.0" :
                     value;

                var type = (connection.SrcEnd.Impl as GME.MGA.MgaFCO).RegistryValue["Type"];
                if (string.IsNullOrEmpty(type))
                {
                    bool res;
                    if (bool.TryParse(value, out res))
                    {
                        type = "Boolean";
                    }
                    else
                    {
                        double res_d;
                        if (double.TryParse(value, out res_d))
                        {
                            type = "Real";
                        }
                        else
                        {
                            type = "ERROR_UNKNOWN_TYPE";
                            GenerateModelicaModel.Messages.Add(
                                GmeMessage.ParameterPropertyTypeNotSupported(connection.SrcEnd.Impl.Name));
                        }
                    }
                }

                if (type != "ERROR_UNKNOWN_TYPE")
                {
                    comp.Parameters[connection.DstEnd.Name] = connection.SrcEnd.Name;
                }
            }
        }

        private void BuildModelicaSignalConnection(
            CyPhy.ModelicaPortMap connection,
            Modelica.Model m)
        {
            // TODO: test this function
            Modelica.Component srcParent = null;
            Modelica.Component dstParent = null;

            foreach (var item in m.Components)
            {
                if (connection.SrcEnd.ParentContainer is CyPhy.ModelicaModel ||
                    connection.SrcEnd.ParentContainer is CyPhy.SignalFlowModel)
                {
                    if (item.Impl.ID == connection.SrcEnd.ParentContainer.ID)
                    {
                        srcParent = item;
                    }
                }
                else
                {
                    if (connection.SrcEnd.ParentContainer.ID == connection.ParentContainer.ID)
                    {
                        if (item.Impl.ID == connection.SrcEnd.ID)
                        {
                            srcParent = item;
                        }
                    }
                    else
                    {
                        if (item.Impl.ID == connection.SrcEnd.ParentContainer.ID)
                        {
                            var model = m.Models.FirstOrDefault(x => x.Impl.ID == connection.SrcEnd.ParentContainer.ID);
                            srcParent = model.Components.FirstOrDefault(x => x.Impl.ID == connection.SrcEnd.ID);
                        }
                    }
                }

                if (connection.DstEnd.ParentContainer is CyPhy.ModelicaModel ||
                    connection.DstEnd.ParentContainer is CyPhy.SignalFlowModel)
                {
                    if (item.Impl.ID == connection.DstEnd.ParentContainer.ID)
                    {
                        dstParent = item;
                    }
                }
                else
                {
                    if (connection.DstEnd.ParentContainer.ID == connection.ParentContainer.ID)
                    {
                        if (item.Impl.ID == connection.DstEnd.ID)
                        {
                            dstParent = item;
                        }
                    }
                    else
                    {
                        if (item.Impl.ID == connection.DstEnd.ParentContainer.ID)
                        {
                            var model = m.Models.FirstOrDefault(x => x.Impl.ID == connection.DstEnd.ParentContainer.ID);
                            dstParent = model.Components.FirstOrDefault(x => x.Impl.ID == connection.DstEnd.ID);
                        }
                    }
                }
            }

            if (srcParent != null || dstParent != null)
            {
                if (srcParent != null &&
                    (srcParent.Impl is CyPhy.ModelicaModel ||
                    srcParent.Impl is CyPhy.SignalFlowModel))
                {
                    var modelicaModel = m.Components.FirstOrDefault(
                        x => x.Impl.ID == connection.SrcEnd.ParentContainer.ID);

                    // power port
                    var pp = modelicaModel.Ports.FirstOrDefault(
                        x => x.Impl.ID == connection.SrcEnd.ID);

                    Modelica.Connection conn = new Modelica.Connection();

                    // modelica model's port
                    conn.Src = new Modelica.Port(srcParent, pp.Impl);
                    conn.Src.Name = String.Format(
                        "{0}",
                        pp.Name,
                        Modelica.Factory.CommonSignal);

                    conn.Src.PortType = Modelica.Factory.PortType.ModelicaConnection;
                    // 
                    conn.Dst = new Modelica.Port(dstParent, connection.DstEnd);
                    conn.Dst.Name = "";
                    //conn.Dst.Name = String.Format(
                    //  "{0}",
                    //  Modelica.Factory.CommonSignal.Substring(1));

                    conn.Dst.PortType = Modelica.Factory.PortType.ModelicaConnection;

                    m.Connections.Add(conn);

                }
                else if (dstParent != null &&
                    (dstParent.Impl is CyPhy.ModelicaModel ||
                    dstParent.Impl is CyPhy.SignalFlowModel))
                {
                    var modelicaModel = m.Components.FirstOrDefault(
                        x => x.Impl.ID == connection.DstEnd.ParentContainer.ID);

                    // port
                    var pp = modelicaModel.Ports.FirstOrDefault(
                        x => x.Impl.ID == connection.DstEnd.ID);

                    Modelica.Connection conn = new Modelica.Connection();

                    // 
                    conn.Src = new Modelica.Port(srcParent, connection.SrcEnd);
                    conn.Src.Name = "";
                    //conn.Src.Name = String.Format(
                    //   "{0}",
                    //   Modelica.Factory.CommonSignal.Substring(1));
                    conn.Src.PortType = Modelica.Factory.PortType.ModelicaConnection;


                    // modelica model's port
                    conn.Dst = new Modelica.Port(dstParent, pp.Impl);
                    conn.Dst.Name = String.Format(
                        "{0}",
                        pp.Name,
                        Modelica.Factory.CommonSignal);
                    conn.Dst.PortType = Modelica.Factory.PortType.ModelicaConnection;

                    m.Connections.Add(conn);

                }
            }

        }


        private void BuildModelicaSignalConnection(
            CyPhy.ModelicaSignal2Metric connection,
            Modelica.Model m)
        {
            // TODO: test this function
            Modelica.Component srcParent = null;
            Modelica.Component dstParent = null;

            foreach (var item in m.Components)
            {
                if (connection.SrcEnd.ParentContainer is CyPhy.ModelicaModel ||
                    connection.SrcEnd.ParentContainer is CyPhy.SignalFlowModel)
                {
                    if (item.Impl.ID == connection.SrcEnd.ParentContainer.ID)
                    {
                        srcParent = item;
                    }
                }
                else
                {
                    if (connection.SrcEnd.ParentContainer.ID == connection.ParentContainer.ID)
                    {
                        if (item.Impl.ID == connection.SrcEnd.ID)
                        {
                            srcParent = item;
                        }
                    }
                    else
                    {
                        if (item.Impl.ID == connection.SrcEnd.ParentContainer.ID)
                        {
                            var model = m.Models.FirstOrDefault(x => x.Impl.ID == connection.SrcEnd.ParentContainer.ID);
                            srcParent = model.Components.FirstOrDefault(x => x.Impl.ID == connection.SrcEnd.ID);
                        }
                    }
                }

                if (connection.DstEnd.ParentContainer is CyPhy.ModelicaModel ||
                    connection.DstEnd.ParentContainer is CyPhy.SignalFlowModel)
                {
                    if (item.Impl.ID == connection.DstEnd.ParentContainer.ID)
                    {
                        dstParent = item;
                    }
                }
                else
                {
                    if (connection.DstEnd.ParentContainer.ID == connection.ParentContainer.ID)
                    {
                        if (item.Impl.ID == connection.DstEnd.ID)
                        {
                            dstParent = item;
                        }
                    }
                    else
                    {
                        if (item.Impl.ID == connection.DstEnd.ParentContainer.ID)
                        {
                            var model = m.Models.FirstOrDefault(x => x.Impl.ID == connection.DstEnd.ParentContainer.ID);
                            dstParent = model.Components.FirstOrDefault(x => x.Impl.ID == connection.DstEnd.ID);
                        }
                    }
                }
            }

            if (srcParent != null || dstParent != null)
            {
                if (srcParent != null &&
                    (srcParent.Impl is CyPhy.ModelicaModel ||
                    srcParent.Impl is CyPhy.SignalFlowModel))
                {
                    var modelicaModel = m.Components.FirstOrDefault(
                        x => x.Impl.ID == connection.SrcEnd.ParentContainer.ID);

                    // power port
                    var pp = modelicaModel.Ports.FirstOrDefault(
                        x => x.Impl.ID == connection.SrcEnd.ID);

                    Modelica.Connection conn = new Modelica.Connection();

                    // modelica model's power port
                    conn.Src = new Modelica.Port(srcParent, pp.Impl);
                    conn.Src.Name = String.Format(
                        "{0}{1}",
                        pp.Name,
                        "");
                    conn.Src.PortType = Modelica.Factory.PortType.Signal;

                    // junction
                    conn.Dst = new Modelica.Port(dstParent, connection.DstEnd);
                    conn.Dst.Name = String.Format(
                        "{0}",
                        Modelica.Factory.CommonSignal.Substring(1));
                    conn.Dst.PortType = Modelica.Factory.PortType.Signal;

                    m.Connections.Add(conn);

                }
                else if (dstParent != null &&
                    (dstParent.Impl is CyPhy.ModelicaModel ||
                    dstParent.Impl is CyPhy.SignalFlowModel))
                {
                    var modelicaModel = m.Components.FirstOrDefault(
                        x => x.Impl.ID == connection.DstEnd.ParentContainer.ID);

                    // power port
                    var pp = modelicaModel.Ports.FirstOrDefault(
                        x => x.Impl.ID == connection.DstEnd.ID);

                    Modelica.Connection conn = new Modelica.Connection();

                    // junction
                    conn.Src = new Modelica.Port(srcParent, connection.SrcEnd);
                    conn.Src.Name = String.Format(
                        "{0}",
                        Modelica.Factory.CommonSignal.Substring(1));
                    conn.Src.PortType = Modelica.Factory.PortType.ModelicaConnection;

                    // modelica model's power port
                    conn.Dst = new Modelica.Port(dstParent, pp.Impl);
                    conn.Dst.Name = String.Format(
                        "{0}{1}",
                        pp.Name,
                        Modelica.Factory.CommonSignal);
                    conn.Dst.PortType = Modelica.Factory.PortType.ModelicaConnection;

                    m.Connections.Add(conn);

                }
            }
        }
    }
}
