/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using CyPhy2Modelica.Properties;
using System.Web.UI;


// using generic common interface
using Common = ISIS.GME.Common;

// using domain specific interfaces
using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;
using System.Diagnostics.Contracts;


namespace CyPhy2Modelica
{
    public partial class GenerateModelicaModel
    {
        public LimitDefinition LimitDefinition { get; set; }

        /// <summary>
        /// Contains different type of messages, which were collected during the
        /// interpretation process.
        /// </summary>
        public static List<GmeMessage> Messages = new List<GmeMessage>();

        public static List<Modelica.Model> ModelicaModelContainers = new List<Modelica.Model>();

        /// <summary>
        /// Gme model, where the interpreter was invoked.
        /// </summary>
        public Common.Interfaces.Model Assembly { get; set; }

        /// <summary>
        /// Modelica main assembly model.
        /// </summary>
        public Modelica.Model asm { get; set; }

        public string OutputDirectory { get; set; }

        /// <summary>
        /// Contains the scopes and the variables that need to be displayed after
        /// the simulation.
        /// </summary>
        public Dictionary<CyPhy.Monitor, string> Scopes { get; set; }


        Dictionary<Modelica.Component, int> oneJunctions = new Dictionary<Modelica.Component, int>();
        Dictionary<Modelica.Component, int> zeroJunctions = new Dictionary<Modelica.Component, int>();

        Dictionary<Modelica.Component, string> hydraulicPorts = new Dictionary<Modelica.Component, string>();
        List<CyPhy.ModelicaMaterialSpec> materialDefs = new List<CyPhy.ModelicaMaterialSpec>();


        public List<ISIS.GME.Common.Interfaces.FCO> ProcessedObjects =
            new List<Common.Interfaces.FCO>();

        /// <summary>
        /// Initialize a new modelica generator instance.
        /// </summary>
        /// <param name="assembly">
        /// TestBench,
        /// ComponentAssembly,
        /// Component,
        /// PhysicalComponent
        /// </param>
        public GenerateModelicaModel(Common.Interfaces.Model assembly)
        {
            Contract.Requires(assembly != null);
            ModelicaModelContainers.Clear();

            Assembly = assembly;
            LimitDefinition = new LimitDefinition();

            asm = new Modelica.Model(assembly, null);

            ModelicaModelContainers.Add(asm);

            ProcessedObjects.Add(assembly);

            Scopes = new Dictionary<CyPhy.Monitor, string>();
            Solver = new SolverSettings();
        }

        /// <summary>
        /// Takes a modelica model identifies all environment blocks in it.
        /// Puts the same environment blocks into each subsystem (model).
        /// Calls the function recursively.
        /// Call it only once. This function does not check for duplicates.
        /// </summary>
        /// <param name="asm">Start model (most top)</param>
        private void PopulateEnvironmentBlocks(Modelica.Model asm)
        {
            foreach (var component in asm.Components)
            {
                if (component.Impl is ISIS.GME.Dsml.CyPhyML.Interfaces.Environment)
                {
                    foreach (var model in asm.Models)
                    {
                        Modelica.Component c = new Modelica.Component(component.Impl);
                        c.IsOuter = true;
                        c.IsInner = false;
                        c.LibraryName = component.LibraryName;
                        c.Parameters = component.Parameters;
                        c.Ports = component.Ports;

                        model.Components.Add(c);
                    }
                }
            }

            foreach (var model in asm.Models)
            {
                PopulateEnvironmentBlocks(model);
            }

        }

        public void BuildModel()
        {
            // build model
            BuildSubModels(Assembly, asm);

            // build PowerChain connections
            Messages.AddRange(BondGraph.PowerChain.BuildPowerConnections());

            asm = asm.Models[0];

            PopulateEnvironmentBlocks(asm);

            foreach (var component in asm.Components)
            {
                if (component.Impl is ISIS.GME.Dsml.CyPhyML.Interfaces.Environment)
                {
                    component.IsInner = true;
                    component.IsOuter = false;
                }
            }

            UpdateFluidPorts();
        }

        public void CheckIncludedLibraries(List<string> pathsToLibraries)
        {
            if (pathsToLibraries == null)
            {
                pathsToLibraries = new List<string>();
            }

            var packageNames = pathsToLibraries
                .Select(x => Path.GetFileName(x))
                .ToList();

            var modelicaModelsURIs = ProcessedObjects
                .OfType<CyPhy.ModelicaModel>()
                .Where(x =>
                    x.Attributes.URI.StartsWith("Modelica.") == false &&
                    x.Attributes.URI.StartsWith("ISIS.") == false)
                .ToList();

            modelicaModelsURIs.ForEach((x) =>
            {
                if (string.IsNullOrWhiteSpace(x.Attributes.URI))
                {
                    Messages.Add(
                        new GmeMessage(
                            string.Format("{0} modelica URI is empty", x.ToHyperLink()),
                            GmeMessage.Severity_enum.Error));
                }

                if (x.Attributes.URI.Contains(".") == false)
                {
                    return;
                }

                var packageName = x.Attributes.URI.Substring(0, x.Attributes.URI.IndexOf('.'));
                if (packageNames.Contains(packageName) == false)
                {
                    Messages.Add(
                        new GmeMessage(
                            string.Format("{0} package name is not in libraries: '{1}'", x.Attributes.URI, string.Join(" ", packageNames)),
                            GmeMessage.Severity_enum.Warning));
                }
            });
        }

        private void UpdateFluidPorts()
        {
            // TODO: Really bad code...

            Dictionary<CyPhy.ModelicaMaterialSpec, List<Modelica.Component>> map =
                new Dictionary<CyPhy.ModelicaMaterialSpec, List<Modelica.Component>>();

            foreach (var item in materialDefs)
            {
                // iterate through the materialDefs
                map.Add(item, GetConnectedFluidPorts(item));
            }

            foreach (var kvp in map)
            {
                var materialSpec = kvp.Key;

                foreach (var comp in kvp.Value)
                {
                    var port = hydraulicPorts.FirstOrDefault(x => x.Key.Impl.Impl == comp.Impl.Impl);
                    var mediumType = materialSpec.Attributes.CompatibleMaterials.Replace('\n', ' ');

                    if (port.Key.Impl is CyPhy.HydraulicPowerPort ||
                        port.Key.Impl is CyPhy.FluidPort)
                    {
                        // fluid port
                        // TODO: what if port is null
                        // key
                        var key = "redeclare final package Medium";
                        if (port.Key.Parameters.ContainsKey(key) == false)
                        {
                            port.Key.Parameters.Add(
                                key,
                                mediumType);
                        }
                        else
                        {
                            if (port.Key.Parameters[key] == "Modelica.Media.Interfaces.PartialMedium" &&
                                mediumType != "Modelica.Media.Interfaces.PartialMedium")
                            {
                                port.Key.Parameters[key] = mediumType;
                            }

                            // TODO: check fluid type consistency
                            if (port.Key.Parameters[key] != mediumType)
                            {
                                var msg = String.Format(
                                    "Fluid type does not match: {0}[{1}] - {2}",
                                    kvp.Key.ToHyperLink(),
                                    mediumType,
                                    port.Key.Parameters[key]);

                                Messages.Add(new GmeMessage(msg, GmeMessage.Severity_enum.Error));
                            }
                        }
                    }
                }


                foreach (var comp in kvp.Value)
                {
                    var port = hydraulicPorts.FirstOrDefault(x => x.Key.Impl.Impl == comp.Impl.Impl);
                    var mediumType = materialSpec.Attributes.CompatibleMaterials.Replace('\n', ' ');

                    if (port.Key.Impl is CyPhy.HydraulicPowerPort ||
                        port.Key.Impl is CyPhy.FluidPort)
                    {

                    }
                    else
                    {
                        string notdefaultmaterial = "Modelica.Media.Interfaces.PartialMedium";
                        //foreach (var components in kvp.Value)
                        //{
                        //    var paramKey = "redeclare final package Medium";
                        //    if (components.Parameters.ContainsKey(paramKey))
                        //    {
                        //        if (components.Parameters[paramKey] != "Modelica.Media.Interfaces.PartialMedium")
                        //        {
                        //            notdefaultmaterial = components.Parameters[paramKey];
                        //        }
                        //    }
                        //}

                        // Get all associated material spec for this modelica wrapper based on 'map'
                        foreach (var kvp2 in map)
                        {
                            var matSpec = kvp2.Key.Attributes.CompatibleMaterials.Replace('\n', ' ');
                            if (map.Values.FirstOrDefault(x => x.Contains(comp)) != null)
                            {
                                if (matSpec != "Modelica.Media.Interfaces.PartialMedium")
                                {
                                    // Get first material spec type which is not default
                                    notdefaultmaterial = matSpec;
                                    break;
                                }
                            }
                        }

                        // modelica model
                        var key = "redeclare package " + materialSpec.Name;
                        if (port.Key.Parameters.ContainsKey(key) &&
                            port.Key.Parameters[key] == "Modelica.Media.Interfaces.PartialMedium" &&
                            notdefaultmaterial != "Modelica.Media.Interfaces.PartialMedium")
                        {
                            port.Key.Parameters[key] = notdefaultmaterial;
                        }
                    }
                }
            }
        }

        private List<Modelica.Component> GetConnectedFluidPorts(CyPhy.ModelicaMaterialSpec materialSpec)
        {
            List<Modelica.Component> result = new List<Modelica.Component>();

            foreach (var item in materialSpec.DstConnections.MaterialSpecConnectionCollection)
            {
                GetConnectedFluidPorts(result, item.DstEnds.FluidPort as CyPhy.FluidPort);
            }

            return result;
        }

        private void GetConnectedFluidPorts(
            List<Modelica.Component> result,
            CyPhy.FluidPort fluidPort)
        {
            var modelicaWrapper = hydraulicPorts.FirstOrDefault(x => x.Key.Impl.Impl == fluidPort.ParentContainer.Impl);
            if (modelicaWrapper.Key != null)
            {
                result.Add(modelicaWrapper.Key);
            }

            foreach (var item in fluidPort.DstConnections.ModelicaFluidPortMapCollection)
            {
                var port = hydraulicPorts.FirstOrDefault(x => x.Key.Impl.Impl == item.DstEnds.HydraulicPowerPort.Impl);
                // TODO: what if port is null?
                if (port.Key != null)
                {
                    result.Add(port.Key);
                    GetConnectedFluidPorts(result, port.Key.Impl as CyPhy.HydraulicPowerPort);
                }
            }
        }

        private void GetConnectedFluidPorts(
            List<Modelica.Component> result,
            CyPhy.HydraulicPowerPort hydraulicPowerPort)
        {
            foreach (var item in hydraulicPowerPort.DstConnections.PowerFlowCollection)
            {
                if (item.DstEnds.HydraulicPowerPort == null)
                {
                    continue;
                }

                var port = hydraulicPorts.FirstOrDefault(x => x.Key.Impl.Impl == item.DstEnds.HydraulicPowerPort.Impl);
                // TODO: what if port is null?
                if (result.Contains(port.Key))
                {
                    // avoid infinite loop
                    continue;
                }
                if (port.Key != null)
                {
                    result.Add(port.Key);
                    GetConnectedFluidPorts(result, port.Key.Impl as CyPhy.HydraulicPowerPort);
                }
            }

            foreach (var item in hydraulicPowerPort.SrcConnections.PowerFlowCollection)
            {
                if (item.SrcEnds.HydraulicPowerPort == null)
                {
                    continue;
                }

                var port = hydraulicPorts.FirstOrDefault(x => x.Key.Impl.Impl == item.SrcEnds.HydraulicPowerPort.Impl);
                // TODO: what if port is null?
                if (result.Contains(port.Key))
                {
                    // avoid infinite loop
                    continue;
                }
                if (port.Key != null)
                {
                    result.Add(port.Key);
                    GetConnectedFluidPorts(result, port.Key.Impl as CyPhy.HydraulicPowerPort);
                }
            }
        }


        public class SolverSettings
        {
            public double StartTime = 0;
            public double StopTime = 1;
            public int NumberOfIntervals = 0;
            public double Tolerance = 1e-4;
        }

        public SolverSettings Solver { get; set; }

        private void BuildSubModels(
            Common.Interfaces.Model model,
            Modelica.Model m)
        {
            // model container
            Modelica.Model newModel = null;
            newModel = new Modelica.Model(model, m);

            ModelicaModelContainers.Add(newModel);

            ProcessedObjects.Add(model);

            if (model is CyPhy.PatternBased_Requirement)
            {
                var pbreq = model as CyPhy.PatternBased_Requirement;
                foreach (var item in pbreq.Children.VTC_SignalCollection)
                {
                    Modelica.Component comp = new Modelica.Component(item)
                    {
                        LibraryName = "Modelica.Blocks.Interfaces.RealInput",
                    };

                    newModel.Components.Add(comp);
                }


                newModel.AdditionalCode.Add(
                    "type Requirement = enumeration(violated, unknown);");

                foreach (var item in pbreq.Children.RequirementParameterCollection)
                {
                    string paramDef = String.Format(
                        "parameter Real {0} = {1};", item.Name, item.Attributes.Value);

                    newModel.AdditionalCode.Add(paramDef);
                }

                foreach (var item in pbreq.Children.PropositionCollection)
                {
                    string propDef = String.Format(
                        "Requirement {0}(start = Requirement.unknown);", item.Name);

                    newModel.AdditionalCode.Add(propDef);

                    StringBuilder sb = new StringBuilder();
                    sb.AppendFormat("if {0} == Requirement.violated or {1} then", item.Name, item.Attributes.Condition);
                    sb.AppendLine();
                    sb.AppendFormat("{0} = Requirement.violated;", item.Name);
                    sb.AppendLine();
                    sb.AppendFormat("else");
                    sb.AppendLine();
                    sb.AppendFormat("{0} = Requirement.unknown;", item.Name);
                    sb.AppendLine();
                    sb.AppendFormat("end if;");
                    sb.AppendLine();

                    newModel.Equations.Add(sb.ToString());
                }
            }
            else if (model is CyPhy.BondGraph)
            {
                var phyComp = model as CyPhy.BondGraph;
                BuildSubModels(phyComp, newModel);
            }
            else if (model is CyPhy.ComponentType)
            {
                var component = (model as CyPhy.ComponentType);

                foreach (var item in component.Children.BondGraphCollection)
                {
                    BuildSubModels(item, newModel);
                }

                foreach (var item in component.Children.ModelicaModelCollection)
                {
                    BuildModelicaModelType(item, newModel);
                }

                foreach (var item in component.Children.SignalFlowModelCollection)
                {
                    BuildSignalFlowModelType(item, newModel);
                }

                Generator.Build(component.Children.MetricCollection, newModel);
                Generator.Build(component.Children.ParameterCollection, newModel);
                Generator.Build(component.Children.PropertyCollection, newModel);
                Generator.Build(component.Children.SignalPortCollection, newModel);

                component.Children.MetricCollection.ToList().ForEach(x => ProcessedObjects.Add(x));
                component.Children.ParameterCollection.ToList().ForEach(x => ProcessedObjects.Add(x));
                component.Children.PropertyCollection.ToList().ForEach(x => ProcessedObjects.Add(x));
                component.Children.SignalPortCollection.ToList().ForEach(x => ProcessedObjects.Add(x));

                foreach (var item in (component as CyPhy.ComponentType).Children.ModelicaParameterPortMapCollection)
                {
                    BuildModelicaSignalConnection(item, newModel);
                }


                // Power Ports
                foreach (var item in component.Children.PowerPortTypeCollection)
                {
                    BuildPowerPort(item, newModel);
                }

                foreach (var item in component.Children.AggregatePortCollection)
                {
                    BuildPowerPort(item, newModel);
                }

                foreach (var item in component.Children.BusPortCollection)
                {
                    BuildPowerPort(item, newModel);
                }

                BuildConnections(model, newModel);

            }
            else if (model is CyPhy.ComponentAssembly)
            {
                var componentAsm = (model as CyPhy.ComponentAssembly);
                foreach (var item in componentAsm.Children.ComponentCollection)
                {
                    BuildSubModels(item, newModel);
                }

                foreach (var item in componentAsm.Children.ComponentAssemblyCollection)
                {
                    BuildSubModels(item, newModel);
                }

                Generator.Build(componentAsm.Children.MetricCollection, newModel);
                Generator.Build(componentAsm.Children.ParameterCollection, newModel);
                Generator.Build(componentAsm.Children.PropertyCollection, newModel);
                Generator.Build(componentAsm.Children.SignalPortCollection, newModel);

                componentAsm.Children.MetricCollection.ToList().ForEach(x => ProcessedObjects.Add(x));
                componentAsm.Children.ParameterCollection.ToList().ForEach(x => ProcessedObjects.Add(x));
                componentAsm.Children.PropertyCollection.ToList().ForEach(x => ProcessedObjects.Add(x));
                componentAsm.Children.SignalPortCollection.ToList().ForEach(x => ProcessedObjects.Add(x));

                // Power Ports
                foreach (var item in componentAsm.Children.PowerPortTypeCollection)
                {
                    BuildPowerPort(item, newModel);
                }

                foreach (var item in componentAsm.Children.AggregatePortCollection)
                {
                    BuildPowerPort(item, newModel);
                }

                foreach (var item in componentAsm.Children.BusPortCollection)
                {
                    BuildPowerPort(item, newModel);
                }

                BuildConnections(model, newModel);
            }
            else if (model is CyPhy.TestBench)
            {
                var testBench = (model as CyPhy.TestBench);

                List<string> solverSettings = new List<string>();

                var settings = testBench.Children.SolverSettingsCollection.FirstOrDefault();

                if (settings != null)
                {

                    if (testBench.Children.SolverSettingsCollection.Count() > 1)
                    {
                        foreach (var item in testBench.Children.SolverSettingsCollection)
                        {
                            Messages.Add(new GmeMessage(
                                "Only one solver settings is supported. More than one solver settings: "
                                  + item.ToHyperLink(),
                                GmeMessage.Severity_enum.Warning));
                        }

                        Messages.Add(new GmeMessage(
                            "Using: "
                              + settings.ToHyperLink(),
                            GmeMessage.Severity_enum.Warning));
                    }


                    Solver.StartTime = settings.Attributes.StartTime;
                    Solver.StopTime = settings.Attributes.StopTime;
                    Solver.NumberOfIntervals = settings.Attributes.NumberOfIntervals;


                    solverSettings.Add("StartTime = " + settings.Attributes.StartTime);
                    solverSettings.Add("StopTime = " + settings.Attributes.StopTime);

                    if (settings.Attributes.Tolerance != 0.0)
                    {
                        solverSettings.Add("Tolerance = " + settings.Attributes.Tolerance);
                        Solver.Tolerance = settings.Attributes.Tolerance;
                    }

                    if (settings.Attributes.IntervalMethod ==
                        CyPhyClasses.SolverSettings.AttributesClass.IntervalMethod_enum.Interval_Length)
                    {
                        if (settings.Attributes.IntervalLength != 0.0)
                        {
                            solverSettings.Add("Interval = " + settings.Attributes.IntervalLength);
                        }
                    }
                    else if (settings.Attributes.IntervalMethod ==
                        CyPhyClasses.SolverSettings.AttributesClass.IntervalMethod_enum.Number_of_Intervals)
                    {
                        if (settings.Attributes.NumberOfIntervals != 0.0)
                        {
                            solverSettings.Add("NumberOfIntervals = " + settings.Attributes.NumberOfIntervals);
                        }
                    }

                    var solverString = settings.Attributes.Solver.ToString();
                    if (solverString.StartsWith("Dymola__"))
                    {
                        var map = new Dictionary<CyPhyClasses.SolverSettings.AttributesClass.Solver_enum, string>()
                    {
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Cerk23__dash__order__3, "cerk23" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Cerk34__dash__order__4, "cerk34" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Cerk45__dash__order__5, "cerk45" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Dopri45__dash__order__5, "dopri45" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Dopri853__dash__order__8, "dopri853" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Esdirk23a__dash__order__3_stiff, "esdirk23a" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Esdirk34a__dash__order__4_stiff, "esdirk34a" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Esdirk45a__dash__order__5_stiff, "esdirk45a" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Lsodar, "lsodar" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Radau_IIa__dash__order__5_stiff, "radau" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Rkfix2, "rkfix2" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Rkfix3, "rkfix3" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Rkfix4, "rkfix4" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.Dymola__Sdirk34hw__dash__order__4_stiff, "sdirk34hw" },
                    };
                        solverSettings.Add("Algorithm = \"" + map[settings.Attributes.Solver] + "\"");
                    }
                    else if (solverString.StartsWith("OM__"))
                    {
                        var map = new Dictionary<CyPhyClasses.SolverSettings.AttributesClass.Solver_enum, string>()
                    {
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.OM__dallsSymJac, "OM__dallsSymJac" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.OM__dasslwort, "OM__dasslwort" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.OM__inline_dash_euler, "OM__inline_dash_euler" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.OM__inline_dash_rungekutta, "OM__inline_dash_rungekutta" },
{CyPhyClasses.SolverSettings.AttributesClass.Solver_enum.OM__rungekutta, "OM__rungekutta" },
                    };
                        solverSettings.Add("Algorithm = \"dassl\"");
                    }
                    else
                    {
                        // most likely the solver is supported by both tools
                        solverSettings.Add("Algorithm = \"" + solverString + "\"");
                    }

                }


                StringBuilder sbExperiment = new StringBuilder();
                sbExperiment.Append("experiment");
                if (solverSettings.Count > 0)
                {
                    sbExperiment.Append("(");
                    sbExperiment.Append(String.Join(", ", solverSettings));
                    sbExperiment.Append(")");
                }

                var toolSpecificAnnotations = "";

                if (settings != null)
                {
                    toolSpecificAnnotations =
                        string.IsNullOrEmpty(settings.Attributes.ToolSpecificAnnotations) ?
                        "" :
                        ", " + settings.Attributes.ToolSpecificAnnotations;
                }

                newModel.Annotation.Values.Add(sbExperiment + toolSpecificAnnotations);

                foreach (var item in testBench.Children.DesignElementCollection)
                {
                    BuildSubModels(item, newModel);
                }

                foreach (var item in testBench.Children.PatternBased_RequirementCollection)
                {
                    BuildSubModels(item, newModel);
                }

                foreach (var item in testBench.Children.EnvironmentCollection)
                {
                    BuildEnvironment(item, newModel);
                }

                Generator.Build(testBench.Children.MetricCollection, newModel);
                Generator.Build(testBench.Children.ParameterCollection, newModel);
                Generator.Build(testBench.Children.PropertyCollection, newModel);

                testBench.Children.MetricCollection.ToList().ForEach(x => ProcessedObjects.Add(x));
                testBench.Children.ParameterCollection.ToList().ForEach(x => ProcessedObjects.Add(x));
                testBench.Children.PropertyCollection.ToList().ForEach(x => ProcessedObjects.Add(x));

                BuildConnections(model, newModel);
            }

            m.Models.Add(newModel);

            Modelica.Component cp = new Modelica.Component(newModel.Impl)
            {
                LibraryName = newModel.Type,
                Name = newModel.Name,
            };

            ProcessedObjects.Add(newModel.Impl as ISIS.GME.Common.Interfaces.FCO);

            m.Components.Add(cp);
        }

        private void BuildModelicaModelType(
            CyPhy.ModelicaModelType item,
            Modelica.Model newModel)
        {
            var modelicaModel = (item as CyPhy.ModelicaModel);
            materialDefs.AddRange(item.Children.ModelicaMaterialSpecCollection);

            if (string.IsNullOrEmpty(modelicaModel.Attributes.URI))
            {
                Messages.Add(new GmeMessage("ModelicaModel has an empty URI: " + GmeConsoleHelper.ToHyperLink(modelicaModel), GmeMessage.Severity_enum.Error));

            }

            Modelica.Component modelicaCp = new Modelica.Component(item);
            ProcessedObjects.Add(item);

            modelicaCp.LibraryName = modelicaModel.Attributes.URI;
            modelicaCp.Name = item.Name;
            foreach (var medium in item.Children.ModelicaMaterialSpecCollection)
            {
                string value = string.Empty;
                string name = "redeclare package " + medium.Name;
                if (modelicaCp.Parameters.TryGetValue(name, out value) == false)
                {
                    modelicaCp.Parameters.Add(
                        name, medium.Attributes.CompatibleMaterials.Replace('\n', ' '));
                }
                else
                {
                    GenerateModelicaModel.Messages.Add(GmeMessage.SameName(item, medium.ToHyperLink()));
                }
            }

            // add this component to the ports
            // this is a modelica model wrapper, but has/might have material spec definition
            hydraulicPorts.Add(modelicaCp, "");

            foreach (var parameter in modelicaModel.Children.ModelicaParameterCollection)
            {
                if (String.IsNullOrWhiteSpace(parameter.Attributes.Value) == false)
                {
                    modelicaCp.Parameters.Add(parameter.Name, parameter.Attributes.Value);
                }
                else
                {
                    Messages.Add(new GmeMessage(String.Format("Parameter : {0} within : {1} does not have a defined value.", parameter.Name, GmeConsoleHelper.ToHyperLink(modelicaModel)), GmeMessage.Severity_enum.Warning));
                }
            }

            foreach (var port in modelicaModel.Children.ModelicaPortCollection)
            {
                modelicaCp.Ports.Add(new Modelica.Port(modelicaCp, port)
                {
                    Name = port.Name,
                });
                ProcessedObjects.Add(port);
            }

            // TODO: review might be in ModelicaPortCollection

            foreach (var port in modelicaModel.Children.ModelicaAggregateInterfaceCollection)
            {
                modelicaCp.Ports.Add(new Modelica.Port(modelicaCp, port)
                {
                    Name = port.Name,
                    PortType = Modelica.Factory.PortType.CyPhyPower,
                });
                ProcessedObjects.Add(port);
            }

            newModel.Components.Add(modelicaCp);

            // LIMITS
            foreach (var limit in modelicaModel.Children.LimitCheckCollection)
            {
                // BUG: AllSrcConnection and AllDstConnection does not contain all connections
                var limitCheck = new LimitCheck()
                {
                    ParentKind = limit.ParentContainer.Kind,
                    LimitFullPath = limit.Path,
                    LimitName = limit.Name,
                    VariableName = limit.Attributes.VariableName,
                    Type = limit.Attributes.LimitType.ToString(),
                    Value = limit.Attributes.Value,
                };

                limitCheck.VariableFullPath = newModel.Path;
                limitCheck.VariableFullPath += ".";
                limitCheck.VariableFullPath += limitCheck.VariableName;

                LimitDefinition.LimitChecks.Add(limitCheck);
            }
        }

        private void BuildSignalFlowModelType(
            CyPhy.SignalFlowModel item,
            Modelica.Model newModel)
        {
            var modelicaModel = (item as CyPhy.SignalFlowModel);

            if (string.IsNullOrEmpty(modelicaModel.Attributes.URI))
            {
                Messages.Add(new GmeMessage("SignalFlowModel has an empty URI: " + GmeConsoleHelper.ToHyperLink(modelicaModel), GmeMessage.Severity_enum.Error));

            }
            Modelica.Component modelicaCp = new Modelica.Component(item);
            ProcessedObjects.Add(item);

            modelicaCp.LibraryName = modelicaModel.Attributes.URI;
            modelicaCp.Name = item.Name;
            // add this component to the ports
            // this is a modelica model wrapper, but has/might have material spec definition

            foreach (var parameter in modelicaModel.Children.SF_ParameterRefCollection)
            {
                string value = string.Empty;

                if (parameter.SrcConnections.SF_ValueFlowDstPortMapCollection.Count() != 1)
                {
                    Messages.Add(new GmeMessage("Signal Flow parameter has no connection: " + GmeConsoleHelper.ToHyperLink(parameter), GmeMessage.Severity_enum.Warning));
                }
                else
                {
                    var connection = parameter.SrcConnections.SF_ValueFlowDstPortMapCollection.FirstOrDefault();
                    if (connection.SrcEnds.Parameter != null)
                    {
                        value = connection.SrcEnds.Parameter.Name;
                    }
                    else if (connection.SrcEnds.Property != null)
                    {
                        value = connection.SrcEnds.Property.Name;
                    }
                    else
                    {
                        Messages.Add(new GmeMessage("Signal Flow parameter must mapped from a Parameter/Property object: " + GmeConsoleHelper.ToHyperLink(connection), GmeMessage.Severity_enum.Error));
                    }
                }

                if (String.IsNullOrEmpty(value) == false)
                {
                    modelicaCp.Parameters.Add(parameter.Name, value);
                }
            }

            foreach (var dataref in modelicaModel.Children.SF_DataRefCollection)
            {
                string value = string.Empty;

                if (dataref.SrcConnections.SF_ValueFlowDstPortMapCollection.Count() != 1)
                {
                    Messages.Add(new GmeMessage("Signal Flow parameter has no connection: " + GmeConsoleHelper.ToHyperLink(dataref), GmeMessage.Severity_enum.Warning));
                }
                else
                {
                    var connection = dataref.SrcConnections.SF_ValueFlowDstPortMapCollection.FirstOrDefault();
                    if (connection.SrcEnds.Parameter != null)
                    {
                        value = connection.SrcEnds.Parameter.Name;
                    }
                    else if (connection.SrcEnds.Property != null)
                    {
                        value = connection.SrcEnds.Property.Name;
                    }
                    else
                    {
                        Messages.Add(new GmeMessage("Signal Flow parameter must mapped from a Parameter/Property object: " + GmeConsoleHelper.ToHyperLink(connection), GmeMessage.Severity_enum.Error));
                    }
                }

                if (String.IsNullOrEmpty(value) == false)
                {
                    modelicaCp.Parameters.Add(dataref.Name, value);
                }
            }

            foreach (var port in modelicaModel.Children.IOSignalCollection)
            {
                modelicaCp.Ports.Add(new Modelica.Port(modelicaCp, port)
                {
                    Name = port.Name,
                });
                ProcessedObjects.Add(port);
            }

            foreach (var port in modelicaModel.Children.SignalFlowBusPortInterfaceCollection)
            {
                modelicaCp.Ports.Add(new Modelica.Port(modelicaCp, port)
                {
                    Name = port.Name,
                    PortType = Modelica.Factory.PortType.CyPhyPower,
                });
                ProcessedObjects.Add(port);
            }

            newModel.Components.Add(modelicaCp);
        }


        private void BuildEnvironment(
            CyPhy.Environment item,
            Modelica.Model newModel)
        {
            var modelicaModel = (item as CyPhy.Environment);
            materialDefs.AddRange(item.Children.ModelicaMaterialSpecCollection);

            if (string.IsNullOrEmpty(modelicaModel.Attributes.URI))
            {
                Messages.Add(new GmeMessage("ModelicaModel has an empty URI: " + GmeConsoleHelper.ToHyperLink(modelicaModel), GmeMessage.Severity_enum.Error));

            }
            else
            {
                //Messages.Add(new GmeMessage("ModelicaModel is in an experimental version.", GmeMessage.Severity_enum.Warning));

                Modelica.Component modelicaCp = new Modelica.Component(item);
                ProcessedObjects.Add(item);

                modelicaCp.LibraryName = modelicaModel.Attributes.URI;
                modelicaCp.Name = item.Name;
                foreach (var medium in item.Children.ModelicaMaterialSpecCollection)
                {
                    modelicaCp.Parameters.Add(
                        "redeclare package " + medium.Name, medium.Attributes.CompatibleMaterials.Replace('\n', ' '));
                }
                foreach (var parameter in modelicaModel.Children.ModelicaParameterCollection)
                {
                    if (String.IsNullOrEmpty(parameter.Attributes.Value) == false)
                    {
                        modelicaCp.Parameters.Add(parameter.Name, parameter.Attributes.Value);
                    }
                }

                foreach (var port in modelicaModel.Children.ModelicaPortCollection)
                {
                    modelicaCp.Ports.Add(new Modelica.Port(modelicaCp, port)
                    {
                        Name = port.Name,
                    });
                    ProcessedObjects.Add(port);
                }

                // TODO: review might be in ModelicaPortCollection

                foreach (var port in modelicaModel.Children.ModelicaAggregateInterfaceCollection)
                {
                    modelicaCp.Ports.Add(new Modelica.Port(modelicaCp, port)
                    {
                        Name = port.Name,
                        PortType = Modelica.Factory.PortType.CyPhyPower,
                    });
                    ProcessedObjects.Add(port);
                }

                newModel.Components.Add(modelicaCp);
            }
        }

        private void BuildPowerPort(
            CyPhy.Port powerport,
            Modelica.Model m)
        {
            string LibName = "";

            Modelica.Component cp = new Modelica.Component(powerport)
            {
                LibraryName = LibName,
                Name = powerport.Name,
            };
            ProcessedObjects.Add(powerport);

            if (powerport is CyPhy.ElectricalPowerPort)
            {
                LibName = Modelica.Factory.PortTypes[Modelica.Factory.PortType.ElectricalPowerPort];
            }
            else if (powerport is CyPhy.TranslationalPowerPort)
            {
                LibName = Modelica.Factory.PortTypes[Modelica.Factory.PortType.TranslationalPowerPort];
            }
            else if (powerport is CyPhy.RotationalPowerPort)
            {
                LibName = Modelica.Factory.PortTypes[Modelica.Factory.PortType.RotationalPowerPort];
            }
            else if (powerport is CyPhy.HydraulicPowerPort)
            {
                LibName = Modelica.Factory.PortTypes[Modelica.Factory.PortType.HydraulicPowerPort];
                hydraulicPorts.Add(cp, "");
            }
            else if (powerport is CyPhy.ThermalPowerPort)
            {
                LibName = Modelica.Factory.PortTypes[Modelica.Factory.PortType.ThermalPowerPort];
            }
            else if (powerport is CyPhy.MultibodyFramePowerPort)
            {
                LibName = Modelica.Factory.PortTypes[Modelica.Factory.PortType.MultiBody];
            }
            else if (powerport is CyPhy.AggregatePort)
            {
                var hierarchicalConn = (powerport as CyPhy.AggregatePort);
                LibName = hierarchicalConn.Attributes.Type;

                foreach (var item in hierarchicalConn.Children.ModelicaParameterCollection)
                {
                    cp.Parameters.Add(item.Name, item.Attributes.Value);
                }
            }
            else if (powerport is CyPhy.BusPort)
            {
                var hierarchicalConn = (powerport as CyPhy.BusPort);
                LibName = hierarchicalConn.Attributes.Type;

                // FIX ME: DOES NOT HAVE PARAMETERS IN THE META MODEL...
                //foreach (var item in hierarchicalConn.Children.ModelicaParameterCollection)
                //{
                //    cp.Parameters.Add(item.Name, item.Attributes.Value);
                //}
            }
            else
            {
                var msg = string.Format("Unrecognized power port type. {0} {1}", powerport.Kind, powerport.ToHyperLink());
                Messages.Add(new GmeMessage(msg, GmeMessage.Severity_enum.Error));
            }

            if (string.IsNullOrEmpty(LibName))
            {
                var msg = string.Format("Unrecognized power port type. {0} {1} LibraryName is empty.", powerport.Kind, powerport.ToHyperLink());
                Messages.Add(new GmeMessage(msg, GmeMessage.Severity_enum.Error));
                LibName = "ERROR_LIBNAME_NOT_FOUND_OR_NOT_DEFINED";
            }

            cp.LibraryName = LibName;

            //cp.Parameters.Add("BondCount",
            //	BondGraph.PowerChain.GetPowerChains(powerport).Count().ToString());

            if (powerport.AllDstConnections.Count() + powerport.AllSrcConnections.Count() < 2)
            {
                string msg = "";
                msg = String.Format("Power port has less than two power connections {0}", powerport.ToHyperLink());
                //Messages.Add(new GmeMessage(msg, GmeMessage.Severity_enum.Warning));
            }

            m.Components.Add(cp);

        }


        private void BuildSubModels(
            CyPhy.BondGraph PhyComponent,
            Modelica.Model m)
        {
            Modelica.Model newModel = new Modelica.Model(PhyComponent, m)
            {
            };

            ModelicaModelContainers.Add(newModel);

            m.Models.Add(newModel);

            ProcessedObjects.Add(PhyComponent);

            Generator.Build(PhyComponent.Children.ExtendedElementsCollection, newModel);
            Generator.Build(PhyComponent.Children.LocalSignalCollection, newModel);

            foreach (var item in PhyComponent.Children.BondGraphCollection)
            {
                BuildSubModels(item, newModel);
            }

            foreach (var item in PhyComponent.Children.SimulinkModelCollection)
            {
                Messages.Add(new GmeMessage("Simulink model is not supported"));
            }

            foreach (var item in PhyComponent.Children.ControlFunctionCollection)
            {
                Modelica.Model controlFunctionModel = new Modelica.Model(item, newModel);

                ModelicaModelContainers.Add(controlFunctionModel);

                ProcessedObjects.Add(item);

                var startindex = item.Attributes.Code.IndexOf('\n');
                if (startindex > 0)
                {
                    var functionHeader = item.Attributes.Code.Substring(0, startindex);
                    // check pattern
                    try
                    {
                        var inputs = functionHeader.Substring(
                            functionHeader.IndexOf('(') + 1,
                            functionHeader.IndexOf(')') - functionHeader.IndexOf('(') - 1).
                            Split(',');

                        foreach (var input in inputs)
                        {
                            controlFunctionModel.AdditionalCode.Add(
                                CyPhy2Modelica.Modelica.Factory.Library[typeof(CyPhyClasses.InSignal)] + " " + input + ";"); // virtual port!
                        }

                        var outputs = functionHeader.Substring(
                            functionHeader.IndexOf('[') + 1,
                            functionHeader.IndexOf(']') - functionHeader.IndexOf('[') - 1).
                            Split(',');

                        foreach (var output in outputs)
                        {
                            controlFunctionModel.AdditionalCode.Add(
                                CyPhy2Modelica.Modelica.Factory.Library[typeof(CyPhyClasses.OutSignal)] + " " + output + ";"); // virtual port!
                        }

                        controlFunctionModel.AdditionalCode.Add("function " + item.Name + Environment.NewLine);
                        foreach (var input in inputs)
                        {
                            controlFunctionModel.AdditionalCode.Add("input Real " + input + ";" + Environment.NewLine);
                        }
                        foreach (var output in outputs)
                        {
                            controlFunctionModel.AdditionalCode.Add("output Real " + output + ";" + Environment.NewLine);
                        }

                        var codeBody = item.Attributes.Code.Substring(
                            item.Attributes.Code.IndexOf("#Modelica") + "#Modelica".Length);

                        controlFunctionModel.AdditionalCode.Add(codeBody);
                        controlFunctionModel.AdditionalCode.Add("end " + item.Name + ";" + Environment.NewLine);

                        StringBuilder functionCall = new StringBuilder();
                        if (outputs.Length > 1)
                        {
                            functionCall.Append("[");
                        }
                        var outs = item.DstConnections.ControlFunction2SignalCollection.Select(x => x.DstEnds.Signal);
                        foreach (var output in outs)
                        {
                            functionCall.Append(output.Name);
                            functionCall.Append(".Signal[1].SignalValue");
                            if (output.ID != outs.LastOrDefault().ID)
                            {
                                functionCall.Append(", ");
                            }
                        }
                        if (outputs.Length > 1)
                        {
                            functionCall.Append("]");
                        }
                        functionCall.Append(" = ");
                        functionCall.Append(item.Name);
                        functionCall.Append("(");
                        var ins = item.SrcConnections.Signal2ControlFunctionCollection.Select(x => x.SrcEnds.Signal);
                        foreach (var input in ins)
                        {
                            functionCall.Append(input.Name);
                            functionCall.Append(".Signal[1].SignalValue");
                            if (input.ID != ins.LastOrDefault().ID)
                            {
                                functionCall.Append(", ");
                            }
                        }
                        functionCall.Append(");");
                        controlFunctionModel.Equations.Add(functionCall.ToString());
                    }
                    catch (Exception ex)
                    {
                        Messages.Add(new GmeMessage(ex.Message, GmeMessage.Severity_enum.Error));
                    }
                }

                newModel.Models.Add(controlFunctionModel);
            }

            foreach (var item in PhyComponent.Children.BGPowerPortCollection)
            {
                Modelica.Component cp = new Modelica.Component(item)
                {
                    LibraryName = "/* Lib name */",
                    Name = item.Name,
                };
                ProcessedObjects.Add(item);

                if (item is CyPhy.ElectricalPort)
                {
                    cp.LibraryName = Modelica.Factory.PortTypes[Modelica.Factory.PortType.Electrical];
                }
                else if (item is CyPhy.MechanicalRPort)
                {
                    cp.LibraryName = Modelica.Factory.PortTypes[Modelica.Factory.PortType.MechanicalRotation];
                }
                else if (item is CyPhy.MechanicalDPort)
                {
                    cp.LibraryName = Modelica.Factory.PortTypes[Modelica.Factory.PortType.MechanicalTranslation];
                }
                else if (item is CyPhy.HydraulicPort)
                {
                    cp.LibraryName = Modelica.Factory.PortTypes[Modelica.Factory.PortType.Hydraulic];
                }
                else if (item is CyPhy.ThermalPort)
                {
                    cp.LibraryName = Modelica.Factory.PortTypes[Modelica.Factory.PortType.Thermal];
                }

                if (item.DstConnections.PowerPort2JunctionCollection.Count() +
                    item.SrcConnections.Junction2PowerPortCollection.Count() > 1)
                {
                    string msg = "";
                    msg = String.Format("Power ports must have one connected junction {0}", item.ToHyperLink());
                    Messages.Add(new GmeMessage(msg, GmeMessage.Severity_enum.Error));
                }

                //cp.Parameters.Add("BondCount", BondGraph.PowerChain.GetPowerChains(item).Count().ToString());
                newModel.Components.Add(cp);
            }

            foreach (var item in PhyComponent.Children.BGNodeCollection)
            {
                Modelica.Component cp = new Modelica.Component(item)
                {
                    LibraryName = "/* Lib name */",
                    Name = item.Name,
                };
                ProcessedObjects.Add(item);

                string LibName;

                if (Modelica.Factory.Library.TryGetValue(item.GetType(), out LibName))
                {
                    // known type
                    cp.LibraryName = LibName;

                    if (item is CyPhy.Junction)
                    {
                        CyPhy.Junction junc = item as CyPhy.Junction;
                        int bonds = 0;

                        bonds += junc.SrcConnections.BondE2JCollection.Count();
                        bonds += junc.SrcConnections.BondJ2JCollection.Count();
                        bonds += junc.SrcConnections.PowerPort2JunctionCollection.Count();

                        bonds += junc.DstConnections.BondJ2ECollection.Count();
                        bonds += junc.DstConnections.BondJ2JCollection.Count();
                        bonds += junc.DstConnections.Junction2PowerPortCollection.Count();

                        //bonds += BondGraph.PowerChain.GetPowerChains(junc).Count();

                        // TODO: add Modelica shell connections

                        foreach (var ppConnection in junc.SrcConnections.PowerPort2JunctionCollection)
                        {
                            if (ppConnection.SrcEnd.ParentContainer is CyPhy.ModelicaModel)
                            {
                                bonds++;
                            }
                        }

                        foreach (var ppConnection in junc.DstConnections.Junction2PowerPortCollection)
                        {
                            if (ppConnection.DstEnd.ParentContainer is CyPhy.ModelicaModel)
                            {
                                bonds++;
                            }
                        }

                        if (bonds < 2)
                        {
                            Messages.Add(new GmeMessage(
                                String.Format("Junctions must have at least two bonds: {0}", junc.ToHyperLink()),
                                GmeMessage.Severity_enum.Error));
                        }
                        if (item is CyPhy.OneJunction)
                        {
                            oneJunctions.Add(cp, bonds);
                        }
                        else if (item is CyPhy.ZeroJunction)
                        {
                            zeroJunctions.Add(cp, bonds);
                        }
                        //cp.Parameters.Add(
                        //  Modelica.Factory.BondCountParameter,
                        //  bonds.ToString());

                        cp.LibraryName = String.Format("{0}{1:00}", cp.LibraryName, bonds);
                    }
                    else
                    {
                        if (item is CyPhy.BGElement)
                        {
                            string value = (item as CyPhy.BGElement).Attributes.ParameterValue;
                            if (string.IsNullOrEmpty(value))
                            {
                                value = "1";
                            }
                            cp.Parameters.Add(
                                "ParameterValue",
                                value);
                        }
                        if (item is CyPhy.Storage)
                        {
                            string value = (item as CyPhy.Storage).Attributes.InitialValue;
                            if (string.IsNullOrEmpty(value))
                            {
                                value = "1";
                            }
                            cp.Parameters.Add(
                                "InitialValue",
                                value);
                        }
                    }
                }
                else
                {
                    // unknown type
                    cp.LibraryName = "/* Unknown type Lib name */";
                }

                newModel.Components.Add(cp);
            }

            BuildConnections(PhyComponent, newModel);

            m.Components.Add(new Modelica.Component(newModel.Impl)
            {
                LibraryName = newModel.Type,
                Name = newModel.Name
            });

            ProcessedObjects.Add(newModel.Impl as ISIS.GME.Common.Interfaces.FCO);

            foreach (var item in PhyComponent.Children.MonitorCollection)
            {
                Scopes.Add(item, GetVarName(item, newModel));
            }

            // modulation
            foreach (var item in PhyComponent.Children.ModulationCollection)
            {
                List<Modelica.Component> sources = GetSources(item, newModel);
                List<Modelica.Component> destination = GetDestinations(item, newModel);

                string expression = GetExpression(item, newModel);

                Modelica.Model modulation = new Modelica.Model(item, newModel);

                ModelicaModelContainers.Add(modulation);

                ProcessedObjects.Add(item);

                foreach (var input in sources)
                {
                    modulation.Components.Add(new Modelica.Component(item)
                    {
                        LibraryName = "Real",
                        Name = input.Name,
                    });
                    var conn = new Modelica.Connection();

                    conn.Src = new Modelica.Port(input, input.Impl as Common.Interfaces.FCO);
                    conn.Src.PortType = Modelica.Factory.PortType.Signal;
                    if (input.Impl is CyPhy.OneJunction)
                    {
                        // HACK: remove '.' from the port name
                        conn.Src.Name = Modelica.Factory.CommonFlow.Substring(1);
                    }
                    else if (input.Impl is CyPhy.ZeroJunction)
                    {
                        // HACK: remove '.' from the port name
                        conn.Src.Name = Modelica.Factory.CommonEffort.Substring(1);
                    }
                    else
                    {
                        conn.Src.Name = "SRC_UNKNOWN";
                    }
                    conn.Dst = new Modelica.Port(modulation, modulation.Impl as Common.Interfaces.FCO);
                    conn.Dst.PortType = Modelica.Factory.PortType.Signal;
                    conn.Dst.Name = input.Name;

                    newModel.Connections.Add(conn);
                }

                foreach (var output in destination)
                {
                    modulation.Components.Add(new Modelica.Component(item)
                    {
                        LibraryName = "Real",
                        Name = output.Name,
                    });

                    modulation.Equations.Add(
                        " " +
                        output.Name +
                        " = " +
                        item.Attributes.Expression +
                        ";");

                    var conn = new Modelica.Connection();

                    conn.Src = new Modelica.Port(modulation, modulation.Impl as Common.Interfaces.FCO);
                    conn.Src.PortType = Modelica.Factory.PortType.Signal;
                    conn.Src.Name = output.Name;

                    conn.Dst = new Modelica.Port(output, output.Impl as Common.Interfaces.FCO);
                    conn.Dst.PortType = Modelica.Factory.PortType.Signal;
                    // HACK: remove '.' from the port name
                    conn.Dst.Name = Modelica.Factory.ModulationSignalPort.Substring(1);

                    newModel.Connections.Add(conn);
                }
                newModel.Models.Add(modulation);

                Modelica.Component cpModulation = new Modelica.Component(item)
                {
                    LibraryName = modulation.Type,
                };

                newModel.Components.Add(cpModulation);
            }

            // TODO: control function

            // TODO: switching

        }

        private List<Modelica.Component> GetSources(
            CyPhy.Modulation modulation,
            Modelica.Model model)
        {
            List<Modelica.Component> result = new List<Modelica.Component>();
            string expression = modulation.Attributes.Expression;
            foreach (var item in modulation.AllSrcConnections)
            {
                Modelica.Component src = null;

                src = model.Components.FirstOrDefault(x => x.Impl.ID == item.GenericSrcEnd.ID);
                if (src != null)
                {
                    result.Add(src);
                }
            }
            return result;
        }

        private string GetExpression(
            CyPhy.Modulation modulation,
            Modelica.Model model)
        {
            string expression = modulation.Attributes.Expression;
            foreach (var item in modulation.AllSrcConnections)
            {
                Modelica.Component src = null;
                src = model.Components.FirstOrDefault(x => x.Impl.ID == item.GenericSrcEnd.ID);
                if (src != null)
                {
                    string portName = "";
                    if (src.Impl is CyPhy.ZeroJunction)
                    {
                        portName = Modelica.Factory.CommonEffort;
                    }
                    else if (src.Impl is CyPhy.OneJunction)
                    {
                        portName = Modelica.Factory.CommonFlow;
                    }
                    expression = expression.Replace(src.Impl.Name, src.Name + portName);
                }
            }
            return expression;
        }

        /// <summary>
        /// Gets destination components for a given modulation function.
        /// </summary>
        /// <param name="modulation"></param>
        /// <param name="model"></param>
        /// <returns></returns>
        private List<Modelica.Component> GetDestinations(
            CyPhy.Modulation modulation,
            Modelica.Model model)
        {
            List<Modelica.Component> result = new List<Modelica.Component>();

            foreach (var item in modulation.AllDstConnections)
            {
                Modelica.Component dst = null;

                dst = model.Components.FirstOrDefault(x => x.Impl.ID == item.GenericDstEnd.ID);
                if (dst != null)
                {
                    result.Add(dst);
                }
            }
            return result;
        }

        /// <summary>
        /// <para>Builds the Modelica model and generates:</para>
        /// <para> - Modelica model</para>
        /// <para> - Modelica package.mo</para>
        /// <para> - Html document</para>
        /// <para> - JModelicaWrapper.py</para>
        /// <para> - Batch files for easy execution</para>
        /// <para> - GraphViz picture</para>
        /// <para> - ISIS Modelica library</para>
        /// <para> - IsisJModelica.py</para>
        /// </summary>
        /// <param name="filename">
        /// Name of the output model file with extension (.mo).
        /// </param>
        /// <param name="outputDirectory"></param>
        public void Generate(
            string filename,
            string outputDirectory)
        {
            Contract.Requires(asm != null);
            OutputDirectory = outputDirectory;

            if (Directory.Exists(OutputDirectory) == false)
            {
                Directory.CreateDirectory(OutputDirectory);
            }

            // build the Modelica assembly based on the GME model
            //BuildModel();
            using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, filename)))
            {
                using (System.CodeDom.Compiler.IndentedTextWriter indentedWriter = new System.CodeDom.Compiler.IndentedTextWriter(writer))
                {
                    //indentedWriter.Write("within {0};", asm.Type);
                    //indentedWriter.Write("within {0};", Path.GetFileName(outputDirectory));
                    // write the entire Modelica model into a file
                    asm.Write(indentedWriter, true, outputDir: OutputDirectory);
                }
            }

            #region Validate Model's name
            List<string> invalidModelNames = new List<string>();
            invalidModelNames.Add("package.mo");
            invalidModelNames.Add(Modelica.Factory.ModelicaLibFileName);
            invalidModelNames.Add(Modelica.Factory.JModelicaLibFileName);
            invalidModelNames.Add(Modelica.Factory.JModelicaWrapperFileName);
            //invalidModelNames.Add(Modelica.Factory.JModelicaSimFileName);

            if (invalidModelNames.Contains(filename))
            {
                Messages.Add(new GmeMessage(
                    "Model name could not be '" + filename + "', please rename it.",
                    GmeMessage.Severity_enum.Error));
                return;
            }
            #endregion

            using (StreamWriter writer = new StreamWriter(Path.Combine(OutputDirectory, "package.mo")))
            {
                writer.WriteLine("package {0}", Path.GetFileName(outputDirectory));
                writer.WriteLine("  extends Modelica.Icons.Package;");
                writer.WriteLine("  annotation(uses(Modelica(version = \"3.2\")), version = \"2\", conversion(noneFromVersion = \"\", noneFromVersion = \"1\"));");
                writer.WriteLine("end {0};", Path.GetFileName(outputDirectory));
            }

            #region ISIS Modelica Library

            string isisDirectory = Path.Combine(OutputDirectory, "ISIS");
            if (Directory.Exists(isisDirectory) == false)
            {
                Directory.CreateDirectory(isisDirectory);
            }

            using (StreamWriter writer = new StreamWriter(
                Path.Combine(isisDirectory, "package.mo")))
            {
                writer.WriteLine("within {0};", Path.GetFileName(outputDirectory));
                writer.WriteLine(Resources.IsisModelicaLib);
            }

            using (StreamWriter writer = new StreamWriter(
                Path.Combine(isisDirectory, "Junctions.mo")))
            {
                writer.WriteLine("within {0}.ISIS;", Path.GetFileName(outputDirectory));
                writer.WriteLine("package Junctions");

                writer.WriteLine("extends Modelica.Icons.Package;");

                for (int i = 2; i < 20; i++)
                {
                    #region OneJunctions
                    if (oneJunctions.Values.Contains(i))
                    {
                        writer.WriteLine("model LibOneJunction{0:00}", i);
                        writer.WriteLine("  parameter Integer Signs[{0}];", i);
                        for (int j = 1; j <= i; j++)
                        {
                            writer.Write("  ISIS.Ports.LibPowerPort Port{0} ", j);
                            writer.WriteLine("annotation(Placement(visible = true, transformation(origin = {-0.731261,36.5631}, extent = {{-12,-12},{12,12}}, rotation = 0), iconTransformation(origin = {-0.731261,36.5631}, extent = {{-12,-12},{12,12}}, rotation = 0)));");
                        }
                        writer.WriteLine("  // ISIS.Ports.LibSignalConnector Flow;");
                        writer.WriteLine("  // ISIS.Ports.LibSignalConnector Displacement;");
                        writer.WriteLine("  annotation(" /*Diagram(), */ + "Icon(graphics = {Text(rotation = 0, lineColor = {0,0,255}, fillColor = {0,0,255}, pattern = LinePattern.Solid, fillPattern = FillPattern.None, lineThickness = 0.25, extent = {{-42.4132,-10.9689},{42.7788,-92.1389}}, textString = \"1\")}));");
                        writer.WriteLine("initial equation");
                        writer.WriteLine("  // Displacement.SignalValue = 0;");
                        writer.WriteLine("equation");

                        writer.Write("  0 = ");
                        for (int j = 1; j <= i; j++)
                        {
                            writer.Write("Signs[{0}] * Port{0}.E", j);
                            if (j < i)
                            {
                                writer.Write(" + ");
                            }
                            else
                            {
                                writer.Write(";");
                            }
                        }
                        writer.WriteLine("");

                        for (int j = 1; j < i; j++)
                        {
                            writer.WriteLine("  Port{0}.F = Port{1}.F;", j, j + 1);
                        }

                        writer.WriteLine("  // Flow.SignalValue = Port1.F;");
                        writer.WriteLine("  // der(Displacement.SignalValue) = Flow.SignalValue;");

                        writer.WriteLine("end LibOneJunction{0:00};", i);
                    }
                    #endregion

                    #region ZeroJunctions
                    if (zeroJunctions.Values.Contains(i))
                    {
                        writer.WriteLine("model LibZeroJunction{0:00}", i);
                        writer.WriteLine("  parameter Integer Signs[{0}];", i);
                        for (int j = 1; j <= i; j++)
                        {
                            writer.Write("  ISIS.Ports.LibPowerPort Port{0} ", j);
                            writer.WriteLine("annotation(Placement(visible = true, transformation(origin = {-0.731261,36.5631}, extent = {{-12,-12},{12,12}}, rotation = 0), iconTransformation(origin = {-0.731261,36.5631}, extent = {{-12,-12},{12,12}}, rotation = 0)));");
                        }
                        writer.WriteLine("  // ISIS.Ports.LibSignalConnector Effort;");
                        writer.WriteLine("  // ISIS.Ports.LibSignalConnector Momentum;");
                        writer.WriteLine("  annotation(" /*Diagram(), */ + "Icon(graphics = {Text(rotation = 0, lineColor = {0,0,255}, fillColor = {0,0,255}, pattern = LinePattern.Solid, fillPattern = FillPattern.None, lineThickness = 0.25, extent = {{-42.4132,-10.9689},{42.7788,-92.1389}}, textString = \"0\")}));");
                        writer.WriteLine("initial equation");
                        writer.WriteLine("  // Momentum.SignalValue = 0;");
                        writer.WriteLine("equation");

                        writer.Write("  0 = ");
                        for (int j = 1; j <= i; j++)
                        {
                            writer.Write("Signs[{0}] * Port{0}.F", j);
                            if (j < i)
                            {
                                writer.Write(" + ");
                            }
                            else
                            {
                                writer.Write(";");
                            }
                        }
                        writer.WriteLine("");

                        for (int j = 1; j < i; j++)
                        {
                            writer.WriteLine("  Port{0}.E = Port{1}.E;", j, j + 1);
                        }

                        writer.WriteLine("  // Effort.SignalValue = Port1.E;");
                        writer.WriteLine("  // der(Momentum.SignalValue) = Effort.SignalValue;");

                        writer.WriteLine("end LibZeroJunction{0:00};", i);
                    }
                    #endregion
                }
                writer.WriteLine("end Junctions;");
            }

            #endregion
        }

        private void WriteGraphsInJson(Modelica.Model model, string path)
        {
            model.WriteGraphInJson(path);
            foreach (var item in model.Models)
            {
                WriteGraphsInJson(item, path);
            }
        }

        private string GetVarName(CyPhy.Monitor item, Modelica.Model model)
        {
            string varName = "";

            if (item is CyPhy.SignalMonitor)
            {
                var monitor = item as CyPhy.SignalMonitor;
                var conn = monitor.SrcConnections.Signal2SignalMonitorCollection.FirstOrDefault();
                if (conn != null)
                {
                    Modelica.Component cp = model.Components.FirstOrDefault(x => x.OriginalName == conn.SrcEnd.Name);
                    if (cp != null)
                    {
                        varName = model.Path + "." + cp.Name + Modelica.Factory.CommonSignal;
                    }
                }
            }
            else if (item is CyPhy.MonitorDisplacement)
            {
                var monitor = item as CyPhy.MonitorDisplacement;
                var conn = monitor.SrcConnections.OneJunction2MonitorDisplacementCollection.FirstOrDefault();
                if (conn != null)
                {
                    Modelica.Component cp = model.Components.FirstOrDefault(x => x.OriginalName == conn.SrcEnd.Name);
                    if (cp != null)
                    {
                        varName = model.Path + "." + cp.Name + Modelica.Factory.CommonDisplacement;
                    }
                }
            }
            else if (item is CyPhy.MonitorMomentum)
            {
                var monitor = item as CyPhy.MonitorMomentum;
                var conn = monitor.SrcConnections.ZeroJunction2MonitorMomentumCollection.FirstOrDefault();
                if (conn != null)
                {
                    Modelica.Component cp = model.Components.FirstOrDefault(x => x.OriginalName == conn.SrcEnd.Name);
                    if (cp != null)
                    {
                        varName = model.Path + "." + cp.Name + Modelica.Factory.CommonMomentum;
                    }
                }
            }
            else if (item is CyPhy.MonitorEffort)
            {
                var monitor = item as CyPhy.MonitorEffort;
                var conn = monitor.SrcConnections.ZeroJunction2MonitorEffortCollection.FirstOrDefault();
                if (conn != null)
                {
                    Modelica.Component cp = model.Components.FirstOrDefault(x => x.OriginalName == conn.SrcEnd.Name);
                    if (cp != null)
                    {
                        varName = model.Path + "." + cp.Name + Modelica.Factory.CommonEffort;
                    }
                }
            }
            else if (item is CyPhy.MonitorFlow)
            {
                var monitor = item as CyPhy.MonitorFlow;
                var conn = monitor.SrcConnections.OneJunction2MonitorFlowCollection.FirstOrDefault();
                if (conn != null)
                {
                    Modelica.Component cp = model.Components.FirstOrDefault(x => x.OriginalName == conn.SrcEnd.Name);
                    if (cp != null)
                    {
                        varName = model.Path + "." + cp.Name + Modelica.Factory.CommonFlow;
                    }
                }
            }

            return varName;
        }

    }
}
