/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace CyPhy2Modelica
{
    public class GmeMessage
    {
        /// <summary>
        /// Contains the message.
        /// </summary>
        public string Message { get; set; }

        public Severity_enum Severity { get; set; }

        public enum Severity_enum
        {
            Information,
            Warning,
            Error,
            None,
        }

        /// <summary>
        /// Creates a new Message.
        /// </summary>
        /// <param name="message"></param>
        /// <param name="severity"></param>
        public GmeMessage(
            string message,
            Severity_enum severity = Severity_enum.Information)
        {
            Message = message;
            Severity = severity;
        }

        /// <summary>
        /// Writes this message into the GMEConsole
        /// </summary>
        /// <param name="console"></param>
        public void Write(GME.CSharp.GMEConsole console)
        {
            if (Severity == Severity_enum.Error)
            {
                console.Error.WriteLine(Message);
            }
            else if (Severity == Severity_enum.Information)
            {
                console.Info.WriteLine(Message);
            }
            else if (Severity == Severity_enum.Warning)
            {
                console.Warning.WriteLine(Message);
            }
            else
            {
                console.Out.WriteLine(Message);
            }
        }

        public static List<string> SimpleCustomFormulas = new List<string>();

        public static void SaveSimpleCustomFormula(string hyperlink)
        {
            SimpleCustomFormulas.Add(hyperlink);
        }

        public static GmeMessage GetSimpleCustomFormulas()
        {
            StringBuilder sb = new StringBuilder();

            sb.AppendFormat("{0} [Simple|CustomFormula] did not get translated to Modelica.",
                SimpleCustomFormulas.Count);

            SimpleCustomFormulas.ForEach(x => sb.AppendFormat(" {0},",x));

            return new GmeMessage(sb.ToString(), Severity_enum.Information);
        }


        # region All messages

        public static GmeMessage SkippedConnection(ISIS.GME.Common.Interfaces.Connection conn)
        {
            return new GmeMessage(string.Format(
                                "Skipped: {0}:{1} -> {2}:{3}",
                                conn.GenericSrcEnd.ParentContainer.Name,
                                conn.GenericSrcEnd.Name,
                                conn.GenericDstEnd.ParentContainer.Name,
                                conn.GenericDstEnd.Name),
                                GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage SameName(ISIS.GME.Common.Interfaces.Container model, string sb)
        {
            return new GmeMessage(string.Format(
                                        "Some elements have the same name, please correct those in {0}: {1}",
                                        model.ToHyperLink(),
                                        sb),
                                    GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage KeywordName(ISIS.GME.Common.Interfaces.Container model, string name, string sb)
        {
            return new GmeMessage(string.Format(
                                        "The name '{0}' is a reserved Modelica keyword, change the name of {1}.",
                                        name,
                                        sb),
                                    GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage CyberNotInstalled()
        {
            return new GmeMessage(
                            string.Format("CyPhy2SLC_CodeGen is not installed on your machine."),
                            GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage CyberFailed(Exception ex)
        {
            return new GmeMessage(
                        string.Format("CyPhy2SLC_CodeGen failed: {0}", ex),
                        GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage ValueFlowPropagation(ISIS.GME.Dsml.CyPhyML.Interfaces.ValueFlowTarget vf)
        {
            string msg = string.Format(
                "Top level parameters/properties cannot feed {0} {1}, where the value flow chain goes into a modelica model parameter.",
                vf.Kind,
                vf.ToHyperLink());

            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage UnsafeVariableName(ISIS.GME.Common.Interfaces.FCO fco)
        {
            return new GmeMessage(string.Format(
                         "Unsafe variable name: {0} Path: {1}",
                         (fco.Impl as GME.MGA.MgaFCO).ToMgaHyperLink(),
                         fco.Path), GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage ParameterPropertyTypeNotSupported(string name)
        {
            return new GmeMessage(
                string.Format("Unknown parameter type {0}. Must be real or boolean.", name),
                GmeMessage.Severity_enum.Warning);
        }

        public static GmeMessage ComponentAssembliesAreNotSupportedInTB(string name)
        {
            var msg = string.Format("ComponentAssemblies are not supported in test benches. {0}", name);
            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage ComponentRefsAreNotSupportedInTB(string name)
        {
            var msg = string.Format("Component references are not supported in test benches. {0}", name);
            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage ComponentsAreNotSupportedInTB(string name)
        {
            var msg = string.Format("Components are not supported in test benches. {0}", name);
            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage TIPsAreNotSupportedInTB(string name)
        {
            var msg = string.Format("TestInjectionPoint are not supported in test benches. {0}", name);
            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage NoTLSUTInTB()
        {
            var msg = string.Format("There is no top level system under test object.");
            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage MoreThanOneTLSUTInTB(string name)
        {
            var msg = string.Format("There is more than one top level system under test object. {0}", name);
            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage WorkflowHasMoreThanOneTask(string comName)
        {
            var msg = string.Format("Workflow is invalid because it has no task or more than one task or one task where the COMName of the task is not {0}.", comName);
            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage TLSUTMustPointToCorCAInTB(string name)
        {
            var msg = string.Format("Top level system under test reference MUST point to a Component or a Component Assembly {0}.", name);
            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage OneWorkflowInTB(string name)
        {
            var msg = "Only one workflow can exist in a test bench.";
            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage PostProcScriptPathEmpty(string name)
        {
            var msg = string.Format("Post processing block must refer to a python script file. {0}.", name);
            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }

        public static GmeMessage PostProcScriptDoesNotExist(string name, string filename)
        {
            var msg = string.Format("Referenced post processing file does not exist {0} {1}.", name, filename);
            return new GmeMessage(msg, GmeMessage.Severity_enum.Error);
        }


        #endregion
    }

    public static class GmeConsoleHelper
    {
        public static string ToHyperLink<T>(this T subject)
            where T : ISIS.GME.Common.Interfaces.Base
        {
            StringBuilder sb = new StringBuilder();

            if (CyPhy2ModelicaInterpreter.ElaboratedObjects != null)
            {
                string id = string.Empty;
                if (CyPhy2ModelicaInterpreter.ElaboratedObjects.TryGetMappedObject(subject.ID, out id))
                {
                    GME.MGA.MgaFCO fco = subject.Impl.Project.GetFCOByID(id);
                    if (fco != null)
                    {
                        // TODO: maybe get it until we reach the original source???
                        sb.AppendFormat("<a href=\"mga:{0}\">{1} (mapped)</a>", fco.ID, fco.Name);
                        return sb.ToString();
                    }
                }
            }

            sb.AppendFormat("<a href=\"mga:{0}\">{1} (not mapped)</a>", subject.ID, subject.Path);
            return sb.ToString();
        }

        public static string ToMgaHyperLink<T>(this T subject)
            where T : GME.MGA.MgaFCO
        {
            StringBuilder sb = new StringBuilder();

            if (CyPhy2ModelicaInterpreter.ElaboratedObjects != null)
            {
                string id = string.Empty;
                if (CyPhy2ModelicaInterpreter.ElaboratedObjects.TryGetMappedObject(subject.ID, out id))
                {
                    GME.MGA.MgaFCO fco = subject.Project.GetFCOByID(id);
                    if (fco != null)
                    {
                        // TODO: maybe get it until we reach the original source???
                        sb.AppendFormat("<a href=\"mga:{0}\">{1} (mapped)</a>", fco.ID, fco.Name);
                        return sb.ToString();
                    }
                }
            }

            sb.AppendFormat("<a href=\"mga:{0}\">{1} (not mapped)</a>", subject.ID, subject.Name);
            return sb.ToString();
        }
    }
}
