/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using CyPhy2Modelica.Properties;
using System.IO;

namespace CyPhy2Modelica
{
    public partial class MainForm : Form
    {
        public MainForm()
        {
            InitializeComponent();

            chbRunOMSimulation.Checked = Settings.Default.RunSimOM;
            chbRunDymolaSimulation.Checked = Settings.Default.RunSimDymola;

            if (Settings.Default.IncludeDirectoryPath == null)
            {
                Settings.Default.IncludeDirectoryPath = new System.Collections.Specialized.StringCollection();
            }
            else
            {
                var existingIncludeDirectoryPaths = Settings.Default.IncludeDirectoryPath.Cast<string>().Where(d => Directory.Exists(d));
                if (existingIncludeDirectoryPaths.Count() > 0)
                {
                    var existing = new System.Collections.Specialized.StringCollection();
                    existing.AddRange(existingIncludeDirectoryPaths.ToArray());
                    Settings.Default.IncludeDirectoryPath = existing;

                    clbIncludeLibs.Items.AddRange(existingIncludeDirectoryPaths.ToArray());
                    for (int i = 0; i < clbIncludeLibs.Items.Count; i++)
                    {
                        clbIncludeLibs.SetItemChecked(i, true);
                    }
                }
                else
                {
                    Settings.Default.IncludeDirectoryPath.Clear();
                }
            }

            if (Settings.Default.NonCheckedIncludeDirPaths == null)
            {
                Settings.Default.NonCheckedIncludeDirPaths = new System.Collections.Specialized.StringCollection();
            }
            else
            {
                var existingNonCheckedIncludeDirPaths = Settings.Default.NonCheckedIncludeDirPaths.Cast<string>().Where(d => Directory.Exists(d));
                if (existingNonCheckedIncludeDirPaths.Count() > 0)
                {
                    var existing = new System.Collections.Specialized.StringCollection();
                    existing.AddRange(existingNonCheckedIncludeDirPaths.ToArray());
                    Settings.Default.NonCheckedIncludeDirPaths = existing;
                    clbIncludeLibs.Items.AddRange(existingNonCheckedIncludeDirPaths.ToArray());
                }
                else
                {
                    Settings.Default.NonCheckedIncludeDirPaths.Clear();
                }
            }

            chbCopyPackage.Checked = Settings.Default.CopyPackage;
        }

        private void btnCancel_Click(object sender, EventArgs e)
        {
            DialogResult = System.Windows.Forms.DialogResult.Cancel;
            Close();
        }

        private void btnOK_Click(object sender, EventArgs e)
        {
            SaveSettings();
            var package_Names = Settings.Default.PackageNames.Cast<string>().ToList();
            bool hasDuplicates = package_Names.Count != package_Names.Distinct().Count();
            if (hasDuplicates)
            {
                MessageBox.Show("Names of Modelica packages must be unique.",
                    "Name collision",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Warning);
                return;
            }
            DialogResult = System.Windows.Forms.DialogResult.OK;
            Close();
        }

        private List<string> dataSourceLibraries = new List<string>();

        private void SaveSettings()
        {
            //dataSourceLibraries = lbIncludeLibs.DataSource as List<string>;
            if (clbIncludeLibs.CheckedItems.Count > 0)
            {
                dataSourceLibraries = clbIncludeLibs.CheckedItems.Cast<string>().ToList();
            }
            Settings.Default.RunSimOM = chbRunOMSimulation.Checked;
            Settings.Default.RunSimDymola = chbRunDymolaSimulation.Checked;
            Settings.Default.IncludeDirectoryPath = new System.Collections.Specialized.StringCollection();
            Settings.Default.IncludeDirectoryPath.AddRange(dataSourceLibraries.ToArray());
            Settings.Default.NonCheckedIncludeDirPaths = new System.Collections.Specialized.StringCollection();
            foreach (var item in clbIncludeLibs.Items)
            {
                if (dataSourceLibraries.Contains(item) == false)
                {
                    Settings.Default.NonCheckedIncludeDirPaths.Add(item as string);
                }
            }

            Settings.Default.CopyPackage = chbCopyPackage.Checked;

            Settings.Default.PackageFolders.Clear();
            Settings.Default.PackageFolders.AddRange(dataSourceLibraries.Select(x => Path.GetDirectoryName(x)).ToArray());

            Settings.Default.PackageNames.Clear();
            Settings.Default.PackageNames.AddRange(dataSourceLibraries.Select(x => Path.GetFileName(x)).ToArray());

            Settings.Default.Save();
        }

        private void btnBrowseIncludeDir_Click(object sender, EventArgs e)
        {
            string startupDir = "";
            var lastItem = clbIncludeLibs.Items.Cast<string>().LastOrDefault();

            if (lastItem != null && Directory.Exists(lastItem))
            {
                startupDir = lastItem;
            }
            else
            {
                startupDir = Environment.CurrentDirectory;
            }

            var fbd = new META.FolderBrowserDialog();
            fbd.SelectedPath = startupDir;
            var dr = fbd.ShowDialog();

            if (dr == System.Windows.Forms.DialogResult.OK)
            {
                var pathToSave = Path.GetFullPath(fbd.SelectedPath);
                // offer relative path if it is available
                Uri uri1 = new Uri(pathToSave, true);
                Uri uri2 = new Uri(Path.Combine(Environment.CurrentDirectory, "project.mga"), true);
                string relativePath = uri2.MakeRelativeUri(uri1).ToString().Replace('/', '\\');

                if (relativePath != pathToSave &&
                    Path.GetFullPath(relativePath) == pathToSave)
                {
                    // relative path is not the same as the selected
                    // relative path is equivalent with the selected one
                    var saveRelative = MessageBox.Show(
                        "Path could be stored as realtive path. Would you like to save it that way?",
                        "Save as relative path",
                        MessageBoxButtons.YesNo,
                        MessageBoxIcon.Information);

                    if (saveRelative == System.Windows.Forms.DialogResult.Yes)
                    {
                        pathToSave = relativePath;
                    }
                }

                clbIncludeLibs.Items.Add(pathToSave);
                clbIncludeLibs.SetItemChecked(clbIncludeLibs.Items.IndexOf(pathToSave), true);
            }
        }

        private void btnSave_Click(object sender, EventArgs e)
        {
            SaveSettings();
        }

        private void btnRemoveLib_Click(object sender, EventArgs e)
        {
            /*var selected = lbIncludeLibs.SelectedItems.Cast<string>().ToList();
            var filtered = (lbIncludeLibs.DataSource as List<string>).ToList();
            foreach (var item in selected.Distinct())
            {
                if (filtered.Contains(item))
                {
                    filtered.Remove(item);
                }
            }
            lbIncludeLibs.DataSource = filtered;
            */
            int index = clbIncludeLibs.SelectedIndex;
            if (index > -1)
            {
                clbIncludeLibs.Items.RemoveAt(index);
            }
        }
    }
}
