/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
// using generic common interface
using Common = ISIS.GME.Common;

namespace CyPhy2Modelica.Modelica
{
    public class Component : Container
    {
        /// <summary>
        /// Full path for the element e.g. Modelica.Electrical.Analog.Basic.Resistor
        /// </summary>
        public string LibraryName { get; set; }

        public bool IsInner { get; set; }
        public bool IsOuter { get; set; }

        /// <summary>
        /// Parameters of this element name, value
        /// </summary>
        public Dictionary<string, string> Parameters { get; set; }

        public Component(Common.Interfaces.Base item)
            : base(item)
        {
            Parameters = new Dictionary<string, string>();
            IsInner = false;
            IsOuter = false;
        }

        // These only applies to ports and deals with the positions in the icon
        public int PortIconSize { get; set; }
        public int PortIconXcoord { get; set; }
        public int PortIconYcoord { get; set; }
        public int DiagramIconSize { get; set; }
        public int DiagramIconXcoord { get; set; }
        public int DiagramIconYcoord { get; set; }


        /// <summary>
        /// Writes this element into a stream
        /// </summary>
        /// <param name="writer"></param>
        public void Write(System.CodeDom.Compiler.IndentedTextWriter writer, Model parent, bool topLevelParent)
        {
            Parameters.OrderBy(x => x.Key);

            string description = "";

            if (Impl is ISIS.GME.Common.Interfaces.Model)
            {
                description = (Impl as ISIS.GME.Common.Interfaces.Model).Preferences.Description;
            }
            else if (Impl is ISIS.GME.Common.Interfaces.Atom)
            {
                description = (Impl as ISIS.GME.Common.Interfaces.Atom).Preferences.Description;
            }
            else if (Impl is ISIS.GME.Common.Interfaces.Reference)
            {
                description = (Impl as ISIS.GME.Common.Interfaces.Reference).Preferences.Description;
            }
            else if (Impl is ISIS.GME.Common.Interfaces.Set)
            {
                description = (Impl as ISIS.GME.Common.Interfaces.Set).Preferences.Description;
            }

            StringBuilder sb = new StringBuilder();

            if (IsInner)
            {
                sb.Append(" inner ");
            }
            if (IsOuter)
            {
                sb.Append(" outer ");
            }


            if (Impl is ISIS.GME.Dsml.CyPhyML.Interfaces.Parameter)
            {
                string value = (Impl as ISIS.GME.Dsml.CyPhyML.Interfaces.Parameter).Attributes.Value;

                value = string.IsNullOrEmpty(value) ?
                    "0.0" :
                    value;

                var type = (Impl.Impl as GME.MGA.MgaFCO).RegistryValue["Type"];
                if (string.IsNullOrEmpty(type))
                {
                    bool res;
                    if (bool.TryParse(value, out res))
                    {
                        type = "Boolean";
                    }
                    else
                    {
                        double res_d;
                        if (double.TryParse(value, out res_d))
                        {
                            type = "Real";
                        }
                        else
                        {
                            type = "ERROR_UNKNOWN_TYPE";
                            GenerateModelicaModel.Messages.Add(
                                GmeMessage.ParameterPropertyTypeNotSupported(Impl.ToHyperLink()));
                        }
                    }
                }

                if (type == "ERROR_UNKNOWN_TYPE")
                {
                    sb.Append("//");
                }

                sb.AppendLine(" " + "parameter " + type + " " + Name + "=" + value + " \"" + description + "\";");

                writer.WriteLine(sb);
                return;
            }
            else if (Impl is ISIS.GME.Dsml.CyPhyML.Interfaces.Property)
            {
                string value = (Impl as ISIS.GME.Dsml.CyPhyML.Interfaces.Property).Attributes.Value;

                value = string.IsNullOrEmpty(value) ?
                     "0.0" :
                     value;

                var type = (Impl.Impl as GME.MGA.MgaFCO).RegistryValue["Type"];
                if (string.IsNullOrEmpty(type))
                {
                    bool res;
                    if (bool.TryParse(value, out res))
                    {
                        type = "Boolean";
                    }
                    else
                    {
                        double res_d;
                        if (double.TryParse(value, out res_d))
                        {
                            type = "Real";
                        }
                        else
                        {
                            type = "ERROR_UNKNOWN_TYPE";
                            GenerateModelicaModel.Messages.Add(
                                GmeMessage.ParameterPropertyTypeNotSupported(Impl.ToHyperLink()));
                        }
                    }
                }

                if (type == "ERROR_UNKNOWN_TYPE")
                {
                    sb.Append("//");
                }

                sb.AppendLine(" " + "parameter " + type + " " + Name + "=" + value + " \"" + description + "\";");

                writer.WriteLine(sb);
                return;
            }
            else if (Impl is ISIS.GME.Dsml.CyPhyML.Interfaces.Metric)
            {
                string value = (Impl as ISIS.GME.Dsml.CyPhyML.Interfaces.Metric).Attributes.Value;

                value = string.IsNullOrEmpty(value) ?
                     "0.0" :
                     value;

                //if ((Impl as ISIS.GME.Dsml.CyPhyML.Interfaces.Metric).SrcConnections.ValueFlowCollection.Count() == 1)
                {
                    sb.AppendLine(" " + "Real" + " " + Name + ";");
                    sb.Append(" " + LibraryName + " " + GetMetricName(Name));
                }
                //else
                //{
                //    return;
                //}
            }
            //else if (Impl is ISIS.GME.Dsml.CyPhyML.Interfaces.Environment)
            //{
            //    // environment is always inner for now
            //    sb.Append(" inner " + LibraryName + " " + Name);
            //}
            else
            {
                sb.Append(" " + LibraryName + " " + Name);
            }

            bool isItemArray = false;

            if (Parameters.Count > 0 &&
                Impl is ISIS.GME.Dsml.CyPhyML.Interfaces.ValueFlowTarget == false)
            {
                sb.Append("(");

                // collect parameters e.g. data.lenght and data.outerDiameter
                var notSupportedParams = Parameters.Where(x => x.Key.Count(y => y == '.') > 1);
                foreach (var item in notSupportedParams)
                {
                    // TODO: log these issues...
                }

                var paramsToCollapse = Parameters.Where(x => x.Key.Count(y => y == '.') == 1);
                Dictionary<string, Dictionary<string, string>> collapsedParams =
                    new Dictionary<string, Dictionary<string, string>>();

                List<string> collapsedParamNames = new List<string>();

                foreach (var item in paramsToCollapse)
                {
                    var keyBase = item.Key.Substring(0, item.Key.IndexOf('.'));
                    var subKey = item.Key.Substring(item.Key.IndexOf('.') + 1);
                    if (collapsedParams.ContainsKey(keyBase) == false)
                    {
                        collapsedParams.Add(keyBase, new Dictionary<string, string>());
                    }
                    collapsedParams[keyBase].Add(subKey, item.Value);
                    collapsedParamNames.Add(item.Key);
                    collapsedParamNames.Add(keyBase);
                }

                foreach (var item in collapsedParams)
                {
                    sb.AppendLine();
                    isItemArray = isArray(item.Key);
                    sb.Append(item.Key);
                    sb.Append(" ( ");

                    foreach (var subparam in item.Value)
                    {
                        isItemArray = isArray(subparam.Key);
                        sb.Append(subparam.Key);
                        sb.Append(" = ");
                        if (string.IsNullOrWhiteSpace(subparam.Value))
                        {
                            sb.Append("\"\" /* was not specified */");
                        }
                        else
                        {
                            if (isItemArray)
                            {
                                sb.Append(" { ");
                                sb.Append(subparam.Value);
                                sb.Append(" } ");
                            }
                            else
                            {
                                sb.Append(subparam.Value);
                            }
                        }
                        if (subparam.Key != item.Value.Last().Key)
                        {
                            sb.Append(", ");
                        }
                    }

                    sb.Append(" ) ");
                    if (Parameters.Keys.Contains(item.Key))
                    {
                        sb.Append("=" + Parameters[item.Key]);
                    }

                    if (item.Key != collapsedParams.Last().Key)
                    {
                        sb.Append(", ");
                    }
                }

                bool firstNotCollapsed = collapsedParams.Count > 0;
                bool didAppendComma = false;
                foreach (var item in Parameters)
                {
                    if (collapsedParamNames.Contains(item.Key))
                    {
                        if (item.Key == Parameters.Last().Key && didAppendComma)
                        {
                            sb.Remove(sb.Length - 2, 2);
                        }
                        continue;
                    }

                    if (firstNotCollapsed)
                    {
                        sb.Append(", ");
                        firstNotCollapsed = false;
                    }

                    sb.AppendLine();
                    isItemArray = isArray(item.Key);
                    sb.Append(item.Key);
                    sb.Append(" = ");
                    if (string.IsNullOrWhiteSpace(item.Value))
                    {
                        sb.Append("\"\" /* was not specified */");
                    }
                    else
                    {
                        if (isItemArray)
                        {
                            sb.Append(" { ");
                            sb.Append(item.Value);
                            sb.Append(" } ");
                        }
                        else
                        {
                            sb.Append(item.Value);
                        }
                    }
                    if (item.Key != Parameters.Last().Key)
                    {
                        sb.Append(", ");
                        didAppendComma = true;
                    }

                }

                sb.Append(")");
            }

            if (string.IsNullOrEmpty(description) == false)
            {
                sb.Append(" \"" + description.Replace("\"", "\\\"") + "\" ");
            }

            // Position business
            DiagramIconSize = 40;
            if ((IsInner || IsOuter) && topLevelParent == false)
            {
                DiagramIconXcoord = -40;
                DiagramIconYcoord = 30 + (70 * parent.EnvironmentCounter++);
                DiagramIconSize = 20;
            }
            else if (this.Impl is CyPhy.Port == false)
            {
                if (parent.Impl.Kind == "Component")
                {
                    DiagramIconSize = 100;
                    DiagramIconXcoord = DiagramIconXcoord < 150 ? 150 : DiagramIconXcoord;
                    DiagramIconXcoord = DiagramIconXcoord > (parent.MaxX - 150) ? (parent.MaxX - 150) : DiagramIconXcoord;

                    DiagramIconYcoord = DiagramIconYcoord < 150 ? 150 : DiagramIconYcoord;
                    DiagramIconYcoord = DiagramIconYcoord > (parent.MaxY - 150) ? (parent.MaxY - 150) : DiagramIconYcoord;

                }
                else if (parent.Impl.Kind == "TestComponent")
                {
                    DiagramIconSize = 80;
                    DiagramIconXcoord = DiagramIconXcoord < 120 ? 120 : DiagramIconXcoord;
                    DiagramIconXcoord = DiagramIconXcoord > parent.MaxX - 120 ? parent.MaxX - 120 : DiagramIconXcoord;

                    DiagramIconYcoord = DiagramIconYcoord < 120 ? 120 : DiagramIconYcoord;
                    DiagramIconYcoord = DiagramIconYcoord > parent.MaxY - 120 ? parent.MaxY - 120 : DiagramIconYcoord;
                }
            }
            

            sb.Append(" annotation(Placement(visible = true, transformation(origin = {" + DiagramIconXcoord + "," + -DiagramIconYcoord + "}" +
                    ", extent = {{" + -DiagramIconSize + "," + -DiagramIconSize + "},{" + DiagramIconSize + "," + DiagramIconSize + "}}, rotation = 0)");

            if (Impl is CyPhy.Port)
            {
                sb.Append(", iconTransformation(origin = {" + PortIconXcoord + "," + PortIconYcoord + "}" +
                    ", extent = {{" + -PortIconSize + "," + -PortIconSize + "},{" + PortIconSize + "," + PortIconSize + "}}, rotation = 0)");
            }
            sb.Append("));");
            writer.WriteLine(sb.ToString());
        }

        internal static string GetMetricName(string Name)
        {
            return "m_" + Name;
        }

        internal static string GetParameterName(string Name)
        {
            return "p_" + Name;
        }

        private bool isArray(string parameterName)
        {
            if (parameterName == Factory.JunctionSignsParameter)
            {
                return true;
            }
            else
            {
                return false;
            }
        }
    }
}
