/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

// using generic common interface
using Common = ISIS.GME.Common;

// using domain specific interfaces
using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;
using GME.MGA;

namespace CyPhy2Modelica.Modelica
{
    public static class Factory
    {
        public const string ModelicaLibFileName = "ISIS.mo";
        public const string ModelicaJunctionsLibFileName = "ISIS_Junctions.mo";
        public const string JModelicaLibFileName = "IsisJModelica.py";
        public const string OModelicaLibFileName = "IsisOModelica.py";
        public const string JModelicaWrapperFileName = "JModelicaWrapper.py";
        public const string OModelicaWrapperFileName = "OModelicaWrapper.py";
        public const string JModelicaSimFileName = "JMSimulation.cmd";
        public const string JModelica15SimFileName = "JMSimulation15.cmd";
        public const string JModelica16SimFileName = "JMSimulation16.cmd";

        public const string StandAloneSubDirName = "standalone";

        public const string IsisJunctionsPackageName = "ISIS.Junctions.";
        public const string IsisPackageName = "ISIS.HBG.";
        public const string IsisIcons = "ISIS.Icons.";
        public const string IsisPortPackage = "ISIS.Ports.";
        public const string IsisExtPackage = "ISIS.Extension.";

        public const string PowerPort = "PowerPorts";
        public const string ModelicaPort = "Port";

        public const string BondCountParameter = "BondCount";
        public const string SignParameter = "Sign";

        public const string JunctionSignsParameter = "Signs";
        public const string JunctionInitialState = "InitialState";
        public const string JunctionOnCondition = "OnCondition";
        public const string JunctionOffCondition = "OffCondition";

        public const string CommonFlow = ".Port1.F";
        public const string CommonEffort = ".Port1.E";
        public const string CommonMomentum = ".ERROR_TODO.Momentum.SignalValue";
        public const string CommonDisplacement = ".ERROR_TODO.Displacement.SignalValue";
        public const string CommonSignal = ".Signal[1].SignalValue";
        public const string ModulationSignalPort = ".Signal[1].SignalValue";

        public const string ModelTypeSuffix = "_type";
        public const string ModelExtension = ".mo";
        public const string UserDefinedPackage = ""; //"ISIS.UserDefined.";
        public const string ModelShellSuffix = "_shell";

        public static Dictionary<Type, string> Library = new Dictionary<Type, string>()
        {
            #region Bond Graph Elements
            {typeof(CyPhyClasses.Se), IsisPackageName + "LibSourceEffort"},
            {typeof(CyPhyClasses.Sf), IsisPackageName + "LibSourceFlow"},
            {typeof(CyPhyClasses.R), IsisPackageName + "LibResistor"},
            {typeof(CyPhyClasses.I), IsisPackageName + "LibInertia"},
            {typeof(CyPhyClasses.C), IsisPackageName + "LibCapacitor"},
            {typeof(CyPhyClasses.GY), IsisPackageName + "LibGyrator"},
            {typeof(CyPhyClasses.TF), IsisPackageName + "LibTransformer"},
            {typeof(CyPhyClasses.OneJunction), IsisJunctionsPackageName + "LibOneJunction"},
            {typeof(CyPhyClasses.ZeroJunction), IsisJunctionsPackageName + "LibZeroJunction"},
            {typeof(CyPhyClasses.MSe), IsisPackageName + "LibModulatedSourceEffort"},
            {typeof(CyPhyClasses.MSf), IsisPackageName + "LibModulatedSourceFlow"},
            {typeof(CyPhyClasses.MR), IsisPackageName + "LibModulatedResistor"},
            {typeof(CyPhyClasses.MI), IsisPackageName + "LibModulatedInertia"},
            {typeof(CyPhyClasses.MC), IsisPackageName + "LibModulatedCapacitor"},
            {typeof(CyPhyClasses.MGY), IsisPackageName + "LibModulatedGyrator"},
            {typeof(CyPhyClasses.MTF), IsisPackageName + "LibModulatedTransformer"},
            #endregion

            #region Extensions
            {typeof(CyPhyClasses.SignalMonitor), IsisExtPackage + "LibScope"},
            {typeof(CyPhyClasses.MonitorDisplacement), IsisExtPackage + "LibScope"},
            {typeof(CyPhyClasses.MonitorMomentum), IsisExtPackage + "LibScope"},
            {typeof(CyPhyClasses.MonitorEffort), IsisExtPackage + "LibScope"},
            {typeof(CyPhyClasses.MonitorFlow), IsisExtPackage + "LibScope"},
            {typeof(CyPhyClasses.LocalSignal), IsisExtPackage + "LibLocalSignal"},
            {typeof(CyPhyClasses.Parameter), IsisExtPackage + "LibParameter"},
            {typeof(CyPhyClasses.Property), IsisExtPackage + "LibProperty"},
            {typeof(CyPhyClasses.Metric), IsisExtPackage + "LibMetric"},
            {typeof(CyPhyClasses.De), IsisExtPackage + "LibDe"},
            {typeof(CyPhyClasses.Df), IsisExtPackage + "LibDf"},
            {typeof(CyPhyClasses.Dq), IsisExtPackage + "LibDq"},
            {typeof(CyPhyClasses.Dp), IsisExtPackage + "LibDp"},
            #endregion

            {typeof(CyPhyClasses.InSignal), "Modelica.Blocks.Interfaces.RealInput"},
            {typeof(CyPhyClasses.OutSignal), "Modelica.Blocks.Interfaces.RealOutput"},
            {typeof(CyPhyClasses.InputSignalPort), "Modelica.Blocks.Interfaces.RealInput"},
            {typeof(CyPhyClasses.OutputSignalPort), "Modelica.Blocks.Interfaces.RealOutput"},
        };

        /// <summary>
        /// Domain specific power port suffix.
        /// </summary>
        public static Dictionary<Type, string> PowerPorts = new Dictionary<Type, string>()
		{
			{typeof(CyPhyClasses.ElectricalPort), ".Electrical[1]"},
			{typeof(CyPhyClasses.MechanicalDPort), ".Translation[1]"},
			{typeof(CyPhyClasses.MechanicalRPort), ".Rotation[1]"},
			{typeof(CyPhyClasses.HydraulicPort), ".Hydraulic[1]"},
			{typeof(CyPhyClasses.ThermalPort), ".Thermal[1]"},
		};

        public const string SignalPort = "Signal[1].SignalValue";

        public static Dictionary<PortType, string> PortTypes = new Dictionary<PortType, string>()
		{
			{PortType.Signal, IsisPortPackage + "Signal"},
			{PortType.Electrical, IsisPortPackage + "LibElectricalPowerPort"},
			{PortType.MechanicalTranslation, IsisPortPackage + "LibMechanicalTranslationPowerPort"},
			{PortType.MechanicalRotation, IsisPortPackage + "LibMechanicalRotationPowerPort"},
			{PortType.Hydraulic, IsisPortPackage + "ERROR"},
			{PortType.Thermal, IsisPortPackage + "Thermal"},
			{PortType.ElectricalPowerPort, "Modelica.Electrical.Analog.Interfaces.Pin"},
			{PortType.TranslationalPowerPort, "Modelica.Mechanics.Translational.Interfaces.Flange_a"},
			{PortType.RotationalPowerPort, "Modelica.Mechanics.Rotational.Interfaces.Flange_a"},
			{PortType.HydraulicPowerPort, "Modelica.Fluid.Interfaces.FluidPort_a"},
			{PortType.ThermalPowerPort, "Modelica.Thermal.HeatTransfer.Interfaces.HeatPort_a"},
            {PortType.MultiBody, "Modelica.Mechanics.MultiBody.Interfaces.Frame_a"},
		};

        public enum PortType
        {
            /// <summary>
            /// Generic signal port
            /// </summary>
            Signal,

            /// <summary>
            /// Generic power port BG
            /// </summary>
            Power,

            /// <summary>
            /// Generic power port CyPhy
            /// </summary>
            CyPhyPower,

            /// <summary>
            /// Generic modelica connection uses the connect statement
            /// </summary>
            ModelicaConnection,

            /// <summary>
            /// BG electrical
            /// </summary>
            Electrical,
            /// <summary>
            /// BG mechanical translation
            /// </summary>
            MechanicalTranslation,
            /// <summary>
            /// BG mechanical rotation
            /// </summary>
            MechanicalRotation,
            /// <summary>
            /// BG hydraulic
            /// </summary>
            Hydraulic,
            /// <summary>
            /// BG thermal
            /// </summary>
            Thermal,
            /// <summary>
            /// CyPhy electrical
            /// </summary>
            ElectricalPowerPort,
            /// <summary>
            /// CyPhy translational
            /// </summary>
            TranslationalPowerPort,
            /// <summary>
            /// CyPhy rotational
            /// </summary>
            RotationalPowerPort,
            /// <summary>
            /// CyPhy hydraulic
            /// </summary>
            HydraulicPowerPort,
            /// <summary>
            /// CyPhy thermal
            /// </summary>
            ThermalPowerPort,

            MultiBody,
        };

        #region Helper Functions

        /// <summary>
        /// Retrives a Modelica compatible variable name.
        /// </summary>
        /// <param name="name"></param>
        /// <returns></returns>
        public static string GetModifiedName(string name)
        {
            string modifiedName = name;
            if (modifiedName.Length == 0)
            {
                throw new ArgumentNullException();
            }
            else
            {
                // name cannot start with a number
                string c = modifiedName.Substring(0, 1);
                int number = 0;
                if (int.TryParse(c, out number))
                {
                    modifiedName = "n" + modifiedName;
                }

                // cannot contain the following characters
                // "@" used for limits do NOT replace it!
                List<char> chars = new List<char>() { ' ', '.', '/', '*', ':', ';', '+', '-', '\\', '\'', ',', '"' };

                char newChar = '_';

                foreach (var item in chars)
                {
                    modifiedName = modifiedName.Replace(item, newChar);
                }

                // TODO: cannot be a key word


                return modifiedName;
            }
        }

        public static string ExGetPath<T>(
            this T subject,
            string delimiter = "./.",
            bool hasRootFolderName = true)
            where T : IMgaObject
        {
            string absPath = subject.AbsPath;
            StringBuilder path = new StringBuilder();

            if (absPath == "")
            {
                return path.ToString();
            }
            while (absPath.Contains("/@"))
            {
                absPath = absPath.Substring(absPath.IndexOf("/@") + 2);
                path.Append(absPath.Substring(0, absPath.IndexOf("|")));
                path.Append(delimiter);
                absPath = absPath.Substring(absPath.IndexOf("|"));
            };
            StringBuilder result = new StringBuilder();
            result.Append(path.ToString().Substring(0, path.ToString().LastIndexOf(delimiter)));
            return subject.Project.RootFolder.Name + delimiter + result.ToString();
        }

        #endregion


    }
}
