/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
// using generic common interface
using Common = ISIS.GME.Common;
using System.Web.UI;

namespace CyPhy2Modelica.Modelica
{
    public class Model : Container
    {
        private const int minSize = 300;

        public int MaxX { get; set; }
        public int MaxY { get; set; }

        public int EnvironmentCounter { get; set; }

        public Tuple<int, int> MaxXY { 
            get 
            {
                int maxX = minSize;
                int maxY = minSize;
                foreach (var item in Components)
                {
                    var Impl = item.Impl;
                    if (Impl is Common.Interfaces.Model)
                    {
                        item.DiagramIconXcoord = (Impl as Common.Interfaces.Model).GenericAspect.X;
                        item.DiagramIconYcoord = (Impl as Common.Interfaces.Model).GenericAspect.Y;
                    }
                    else if (Impl is Common.Interfaces.Atom)
                    {
                        item.DiagramIconXcoord = (Impl as Common.Interfaces.Atom).GenericAspect.X;
                        item.DiagramIconYcoord = (Impl as Common.Interfaces.Atom).GenericAspect.Y;
                    }
                    else if (Impl is Common.Interfaces.Reference)
                    {
                        item.DiagramIconXcoord = (Impl as Common.Interfaces.Reference).GenericAspect.X;
                        item.DiagramIconYcoord = (Impl as Common.Interfaces.Reference).GenericAspect.Y;
                    }
                    else if (Impl is Common.Interfaces.Set)
                    {
                        item.DiagramIconXcoord = (Impl as Common.Interfaces.Set).GenericAspect.X;
                        item.DiagramIconYcoord = (Impl as Common.Interfaces.Set).GenericAspect.Y;
                    }

                    maxX = item.DiagramIconXcoord > maxX ? item.DiagramIconXcoord : maxX;
                    maxY = item.DiagramIconYcoord > maxY ? item.DiagramIconYcoord : maxY;
                }

                return new Tuple<int, int>(maxX, maxY);
            } 
        }

        public Annotation Annotation { get; set; }

        public List<string> AdditionalCode { get; set; }

        public List<Component> Components { get; set; }

        public List<Connection> Connections { get; set; }

        public List<string> Equations { get; set; }

        public List<Model> Models { get; set; }

        public List<Component> Extends { get; set; }

        public bool IsPartial { get; set; }

        public Model Parent { get; set; }

        public string Path
        {
            get
            {
                if (Parent == null)
                {
                    return "";
                }
                StringBuilder sb = new StringBuilder();
                Stack<string> stack = new Stack<string>();
                Model m = Parent;

                while (m != null)
                {
                    stack.Push(m.Name);
                    m = m.Parent;
                }
                stack.Pop();
                while (stack.Count > 0)
                {
                    sb.Append(stack.Pop());
                    sb.Append(".");
                }
                sb.Append(Name);
                return sb.ToString();
            }
        }

        /// <summary>
        /// Name of the type.
        /// </summary>
        public string Type { get; set; }

        public bool HasHydrauilcPort
        {
            get
            {
                bool result = false;
                foreach (var item in Components)
                {
                    if (item.LibraryName == Factory.PortTypes[Factory.PortType.HydraulicPowerPort])
                    {
                        result = true;
                        return result;
                    }
                }

                foreach (var item in Models)
                {
                    result = item.HasHydrauilcPort;
                    if (result)
                    {
                        return result;
                    }
                }
                return result;
            }
        }

        //public Model(string name, Model parent)
        public Model(
            Common.Interfaces.Base item,
            Model parent)
            : base(item)
        {
            Components = new List<Component>();
            Connections = new List<Connection>();
            Models = new List<Model>();
            Extends = new List<Component>();
            Type = Name + Modelica.Factory.ModelTypeSuffix;
            IsPartial = false;
            Parent = parent;
            Annotation = new Annotation();
            Equations = new List<string>();
            AdditionalCode = new List<string>();

        }

        public void Write(
            System.CodeDom.Compiler.IndentedTextWriter writer,
            bool topLevelModel = false,
            string prefix = "",
            string outputDir = "")
        {
            MaxX = MaxXY.Item1;
            MaxY = MaxXY.Item2;

            // spacing
            writer.WriteLine();

            //writer.WriteLine("/* Auto-generated model begins */");

            if (IsPartial)
            {
                writer.Write("partial ");
            }
            
            // define model header
            if (topLevelModel)
            {
                writer.WriteLine("within " + new DirectoryInfo(outputDir).Name + ";");
                writer.WriteLine("model " + Name);
            }
            else
            {
                writer.WriteLine("model " + Type);
            }

            writer.Indent++;

            // icon
            writer.WriteLine("extends " + prefix + Factory.IsisIcons + Impl.Kind + ";");

            // extends
            //foreach (var item in Extends)
            //{
            //    writer.WriteLine("extends " + item.Type + ";");
            //}


            // write submodels
            foreach (var item in Models)
            {
                item.Write(writer, prefix: prefix);
            }


            // deal with IconTransformation for Ports
            CalculateIconTransformationsForPorts();
            EnvironmentCounter = 0;
            foreach (var item in Components)
            {
                if (Extends.FirstOrDefault(x => x.Impl.ID == item.Impl.ID) != null)
                {
                    writer.Write("extends ");
                }

                item.Write(writer, this, topLevelModel);
            }

            foreach (var item in Ports)
            {
                item.Write(writer);
            }

            foreach (var item in AdditionalCode)
            {
                // functions/algorithms etc
                writer.WriteLine(item);
            }

            // write equations
            writer.WriteLine("equation");

            // additional equations
            writer.WriteLine("// Additional equations");
            foreach (var item in Equations)
            {
                writer.WriteLine(item);
            }

            // metrics
            foreach (var item in Components)
            {
                if (item.Impl is ISIS.GME.Dsml.CyPhyML.Interfaces.Metric)
                {
                    var srcConns = (item.Impl as ISIS.GME.Dsml.CyPhyML.Interfaces.Metric).SrcConnections;
                    if (srcConns.ValueFlowCollection.Count() +
                        srcConns.ModelicaSignal2MetricCollection.Count() +
                        srcConns.Signal2MetricCollection.Count() == 0)
                    {
                        // if no recognized input connection
                        writer.WriteLine(" {0}{1} = 0;", item.SpecialName, Factory.CommonSignal);
                    }
                    writer.WriteLine(" {0} = {1}{2};", item.Name, item.SpecialName, Factory.CommonSignal);
                }
            }


            writer.WriteLine("// Connections");
            // write connections
            foreach (var item in Connections)
            {
                item.Write(writer);
            }

            writer.Indent--;

            // write annotations
            Annotation.Values.Add(@"Coordsys(extent=[-120,-120; 120,120], grid=[2,2], scale=0.1)");
            Annotation.Values.Add(@"Diagram(coordinateSystem(preserveAspectRatio=true, extent  =  {{0,"+ -MaxY +"},{" + MaxX + ",0}}))");


            writer.WriteLine("annotation(" + string.Join(", ", Annotation.Values) + ");");

            // write end model
            if (topLevelModel)
            {
                writer.WriteLine("end " + Name + ";");
            }
            else
            {
                writer.WriteLine("end " + Type + ";");
            }

            //writer.WriteLine("/* Auto-generated model ends */");

        }


        public void WriteGraphInJson(string newGraphs)
        {
            using (TextWriter wr = new StreamWriter(System.IO.Path.Combine(newGraphs, Impl.ID + ".js")))
            {
                using (System.CodeDom.Compiler.IndentedTextWriter twr = new System.CodeDom.Compiler.IndentedTextWriter(wr))
                {
                    twr.WriteLine("{");
                    twr.Indent++;
                    twr.WriteLine("\"name\": \"" + Impl.Name + "\",");
                    twr.WriteLine("\"id\": \"" + Impl.ID + "\",");
                    twr.WriteLine("\"nodes\": [");
                    twr.Indent++;
                    foreach (var item in Components)
                    {
                        twr.WriteLine("{");
                        twr.Indent++;
                        twr.WriteLine("\"name\": \"" + item.Impl.Name + "\",");
                        twr.WriteLine("\"id\": \"" + item.Impl.ID + "\",");
                        twr.WriteLine("\"position\": [");
                        twr.Indent++;
                        twr.WriteLine("{");
                        twr.Indent++;
                        var xPos = 0;
                        var yPos = 0;
                        Common.Classes.Aspect aspect = null;
                        if (item.Impl is Common.Interfaces.Model)
                        {
                            aspect = (item.Impl as Common.Interfaces.Model).GenericAspect;
                        }
                        else if (item.Impl is Common.Interfaces.Atom)
                        {
                            aspect = (item.Impl as Common.Interfaces.Atom).GenericAspect;
                        }
                        else if (item.Impl is Common.Interfaces.Reference)
                        {
                            aspect = (item.Impl as Common.Interfaces.Reference).GenericAspect;
                        }
                        else if (item.Impl is Common.Interfaces.Set)
                        {
                            aspect = (item.Impl as Common.Interfaces.Set).GenericAspect;
                        }

                        if (aspect != null)
                        {
                            xPos = aspect.X;
                            yPos = aspect.Y;
                        }

                        twr.WriteLine("\"x\": " + xPos + ",");
                        twr.WriteLine("\"y\": " + yPos + "");
                        twr.Indent--;
                        twr.WriteLine("}");

                        twr.Indent--;
                        twr.WriteLine("],");
                        bool hasScope = false;
                        Model m = Models.FirstOrDefault(x => x.Impl.ID == item.Impl.ID);
                        if (m != null)
                        {
                            hasScope = m.HasScope();
                        }
                        twr.WriteLine("\"hasScope\": \"" + hasScope + "\"");
                        twr.Indent--;
                        if (Components.Last() == item)
                        {
                            twr.WriteLine("}");
                        }
                        else
                        {
                            twr.WriteLine("},");
                        }
                    }
                    twr.Indent--;
                    twr.WriteLine("],");

                    twr.WriteLine("\"links\": [");
                    twr.Indent++;
                    foreach (var item in Connections)
                    {
                        twr.WriteLine("{");
                        twr.Indent++;
                        string color = "#04344E";
                        color = "#0C0C0E";
                        string[] bgNodes = new[] {
							"R",
							"C",
							"OneJunction",
							"ZeroJunction",
							"I",
							"TF",
							"Se",
							"Sf",
							"MSf",
							"MSe",
							"MTF",
							"GY",
							"MGY",
							"MR",
							"MC",
							"MI",};

                        if (new[] {
							"MonitorEffort",
							"MonitorFlow",
							"MonitorDisplacement",
							"MonitorMomentum" }.Contains(item.Dst.Impl.Kind))
                        {
                            color = "#7B5604";
                            color = "#0C0C0E";
                        }
                        else if (bgNodes.Contains(item.Dst.Impl.Kind) &&
                            bgNodes.Contains(item.Src.Impl.Kind))
                        {
                            color = "#7B2B04";
                            color = "#0C0C0E";
                        }
                        twr.WriteLine("\"color\": \"" + color + "\",");

                        twr.WriteLine("\"srcParentId\": \"" + item.Src.Impl.ParentContainer.ID + "\",");
                        twr.WriteLine("\"srcParentName\": \"" + item.Src.Impl.ParentContainer.Name + "\",");

                        twr.WriteLine("\"srcId\": \"" + item.Src.Impl.ID + "\",");
                        twr.WriteLine("\"srcName\": \"" + item.Src.Impl.Name + "\",");

                        twr.WriteLine("\"dstParentId\": \"" + item.Dst.Impl.ParentContainer.ID + "\",");
                        twr.WriteLine("\"dstParentName\": \"" + item.Dst.Impl.ParentContainer.Name + "\",");

                        twr.WriteLine("\"dstId\": \"" + item.Dst.Impl.ID + "\",");
                        twr.WriteLine("\"dstName\": \"" + item.Dst.Impl.Name + "\"");
                        twr.Indent--;
                        if (Connections.Last() == item)
                        {
                            twr.WriteLine("}");
                        }
                        else
                        {
                            twr.WriteLine("},");
                        }
                    }
                    twr.Indent--;
                    twr.WriteLine("]");
                    twr.Indent--;
                    twr.WriteLine("}");
                }
            }
        }


        public bool HasScope()
        {
            foreach (var item in Components)
            {
                if (item.Impl is ISIS.GME.Dsml.CyPhyML.Interfaces.Monitor)
                {
                    return true;
                }
            }

            foreach (var item in Models)
            {
                if (item.HasScope())
                {
                    return true;
                }
            }

            return false;
        }

        internal void CalculateIconTransformationsForPorts()
        {

            List<Component> ports = Components.Where(c => c.Impl is CyPhy.Port).ToList();

            int iconSize = 16;
            int borderMax = 6;
            int posStep = 40;
            int maxNbrOfPorts = 24;
            while (ports.Count > maxNbrOfPorts)
            {
                iconSize = iconSize / 2;
                borderMax = borderMax * 2;
                posStep = posStep / 2;
                maxNbrOfPorts = maxNbrOfPorts * 2;
                // After 192 ports might not be ideal
            }
            
            SortedList<int, Component> north = new SortedList<int ,Component>();
            SortedList<int, Component> east = new SortedList<int, Component>();
            SortedList<int, Component> south = new SortedList<int, Component>();
            SortedList<int, Component> west = new SortedList<int, Component>();
            
            PopulateNorthEastSouthWest(ports, north, east, south, west, borderMax);

            // North
            int pos = -120 + posStep;
            foreach (var kvp in north)
            {
                kvp.Value.PortIconYcoord = 120;
                kvp.Value.PortIconXcoord = pos;
                kvp.Value.PortIconSize = iconSize;
                pos += posStep;
            }
            // East
            pos = 120 - posStep;
            foreach (var kvp in east)
            {
                kvp.Value.PortIconYcoord = pos;
                kvp.Value.PortIconXcoord = 120;
                kvp.Value.PortIconSize = iconSize;
                pos -= posStep;
            }
            // South
            pos = 120 - posStep;
            foreach (var kvp in south)
            {
                kvp.Value.PortIconYcoord = -120;
                kvp.Value.PortIconXcoord = pos;
                kvp.Value.PortIconSize = iconSize;
                pos -= posStep;
            }
            // West
            pos = -120 + posStep;
            foreach(var kvp in west)
            {
                kvp.Value.PortIconYcoord = pos;
                kvp.Value.PortIconXcoord = -120;
                kvp.Value.PortIconSize = iconSize;
                pos += posStep;
            }

        }

        internal void SafeAdd(SortedList<int, Component> sortedList, int key, Component value)
        {
            while (sortedList.Keys.Contains(key))
            {
                key++;
            }
            sortedList.Add(key, value);
        }

        /// <summary>
        /// Populates list of ports
        /// </summary>
        /// <param name="ports"></param>
        /// <param name="north"></param>
        /// <param name="east"></param>
        /// <param name="south"></param>
        /// <param name="west"></param>
        internal void PopulateNorthEastSouthWest(
            List<Component> ports,
            SortedList<int, Component> north,
            SortedList<int, Component> east,
            SortedList<int, Component> south,
            SortedList<int, Component> west,
            int borderMax)
        {

            int midX = MaxX / 2;
            int midY = MaxY / 2;

            foreach (var item in ports)
            {
                int x = item.DiagramIconXcoord;
                int y = item.DiagramIconYcoord;

                float xRelPos = ((float)midX - (float)x) / (float)midX;
                float yRelPos = ((float)midY - (float)y) / (float)midY;

                if (Math.Abs(yRelPos) >= Math.Abs(xRelPos))
                {
                    // Port is closer to North/South Border
                    if (yRelPos > 0)
                    {
                        // North first choice
                        if (north.Count == borderMax)
                        {
                            if (east.Count == borderMax)
                            {
                                if (south.Count == borderMax)
                                {
                                    SafeAdd(west, y, item);
                                }
                                else
                                {
                                    SafeAdd(south, x, item);
                                }
                            }
                            else
                            {
                                SafeAdd(east, y, item);
                            }

                        }
                        else
                        {
                            SafeAdd(north, x, item);
                        }
                    }
                    else
                    {
                        // South first choice
                        if (south.Count == borderMax)
                        {
                            if (west.Count == borderMax)
                            {
                                if (north.Count == borderMax)
                                {
                                    SafeAdd(east, y, item);
                                }
                                else
                                {
                                    SafeAdd(north, x, item);
                                }
                            }
                            else
                            {
                                SafeAdd(west, y, item);
                            }
                        }
                        else
                        {
                            SafeAdd(south, x, item);
                        }
                    }
                }
                else
                {
                    // Port is closer to East/West Border
                    if (xRelPos < 0)
                    {
                        // East first choice
                        if (east.Count == borderMax)
                        {
                            if (south.Count == borderMax)
                            {
                                if (west.Count == borderMax)
                                {
                                    SafeAdd(north, x, item);
                                }
                                else
                                {
                                    SafeAdd(west, y, item);
                                }
                            }
                            else
                            {
                                SafeAdd(south, x, item);
                            }
                        }
                        else
                        {
                            SafeAdd(east, y, item);
                        }
                        
                    }
                    else
                    {
                        // West first choice
                        if (west.Count == borderMax)
                        {
                            if (north.Count == borderMax)
                            {
                                if (east.Count == borderMax)
                                {
                                    SafeAdd(south, x, item);
                                }
                                else
                                {
                                    SafeAdd(east, y, item);
                                }
                            }
                            else
                            {
                                SafeAdd(north, x, item);
                            }
                        }
                        else
                        {
                            SafeAdd(west, y, item);
                        }
                    }
                }
            }
        }

        internal void ModifyLibNames(string prefix)
        {
            foreach (var item in Components)
            {
                if (Models.FirstOrDefault(x => x.Impl.ID == item.Impl.ID) == null)
                {
                    if (item.LibraryName.StartsWith("Modelica.") == false &&
                        item.LibraryName.StartsWith("Real") == false &&
                        item.LibraryName.StartsWith("Boolean") == false &&
                        item.LibraryName.StartsWith("Integer") == false)
                    {
                        item.LibraryName = prefix + "." + item.LibraryName;
                    }
                }
            }

            foreach (var item in Models)
            {
                item.ModifyLibNames(prefix);
            }
        }
    }
}
