 #!/usr/bin/python
import os
import sys
import json
import time
import logging
import subprocess
import py_modelica as pym

# sys.exit is broken under the Windows virtualenv
#  uncaught exceptions are broken too (i.e. %ERRORLEVEL% == 0)
if os.name == 'nt':
    import win32api
    def _exit(code):
        win32api.TerminateProcess(win32api.GetCurrentProcess(), code)
    sys.exit = _exit

def init_and_compile_test_bench(tool_name='Dymola',
                                result_mat_file='',
                                dymola_home=''):
    """
    Instantiates a new tool for simulation,
    compiles and then returns with the tool. 
    
    Can raise ModelicaError
    """
    # Load in model_config.json
    with open('model_config.json') as file_model:
        model_config = json.load(file_model)
    # Read which tool to use from argument
    if tool_name == 'OpenModelica':
        tool = pym.OpenModelica(model_config)
        tool.short_name = result_mat_file
    elif tool_name == 'Dymola':
        tool = pym.Dymola(model_config, dymola_home)
        if result_mat_file:
            tool.result_mat = '{0}.mat'.format(result_mat_file)  
    # Compile model and return with tool
    tool.compile_model()
    
    return tool

def execute_test_bench(tool):
    """
    Reads in parameters from sumary.testresults.json and simulates the test bench. 
    When done, writes metrics to sumary.testresults.json
    
    Can raise ModelicaError
    """
    log = logging.getLogger()
    # Read in parameters        
    parameters = pym.get_parameters_from_report_json()
    log.info('Parameters : {0}'.format(parameters))
    # Try to compile and simulate model
    if parameters:
        tool.change_parameter(parameters)
        
    tool.simulate_model()
    t_stamp = time.time()
        
    pym.update_metrics_in_report_json(tool.result_mat)
        
    pym.check_limits(tool.result_mat)
        
    t_mat = time.time() - t_stamp
    # Run postprocessing scripts (if PostProcessing dir exists)
    try:
        t_post = pym.run_post_scripts(tool.result_mat, log)
    except subprocess.CalledProcessError as err:
        raise pym.ModelicaError("Execution of post-scripts failed.", sp_msg=err.returncode)
    pym.update_analysis_status(new_status="OK")
    
    return t_mat, t_post


if __name__=='__main__':
    if not os.path.isdir('log'):
        os.mkdir('log')
    log = pym.instantiate_logger(os.path.join('log', 'dynamics_debug.log'))
    try:
        # Read which tool to use and mat-filename from argument	
        tool = init_and_compile_test_bench(sys.argv[1], sys.argv[2], sys.argv[3])
        t_mat, t_post = execute_test_bench(tool)
        # write statistics
        pym.write_out_tool_statistics(tool, t_mat, t_post)
    except pym.ModelicaError as err:
        import traceback
        the_trace = traceback.format_exc()
        log.error(err.message)
        log.error("Exception was raised : {0}".format(traceback.format_exc()))
        pym.update_analysis_status("FAILED", err.message)
        traceback.print_exc(file=sys.stderr)
        with open('_FAILED.txt', 'wb') as f_out:
            f_out.writelines(err.message)
            f_out.writelines(traceback.format_exc())
            f_out.writelines(err.sp_msg)
        sys.exit(6)

