 #!/usr/bin/python
import os
import json
import time
import datetime
import multiprocessing
import platform
import subprocess
import sys

LINUX_VIR_ENV = '~/openmdao-0.2.7/'

def _query_registry():
    """
    Queries the Windows registry for META_PATH in order to get the location of
    python virtual environment containing all necessary packages and modules. 
    """
    import _winreg as wr
    try:
        # try to obtain META_PATH from the registry
        key = wr.OpenKey(wr.HKEY_LOCAL_MACHINE, r'software\meta', 0, wr.KEY_READ)
        meta_path = wr.QueryValueEx(key, 'META_PATH')[0]
        py_path = os.path.join(meta_path, r'bin\Python27\Scripts\python')
    except WindowsError:
        sys.stderr.write('Could not find META_PATH in registry, attempting to use default python.')
        py_path = 'python'
    
    return py_path


def _write_out_stat(execution_time, iso_time):
    """
    Writes out statistics to stat.json
    """
    
    dyn_stat = {"total_time": execution_time, "start_time": iso_time}
    try:
        with open('stat.json', 'r') as f_in:
            stat = json.load(f_in)
        if stat['ToolSpecifics'] is None:
            stat['ToolSpecifics'] = {'dynamics': dyn_stat}
        else:
            stat['ToolSpecifics']['dynamics'] = dyn_stat
    except (IOError, KeyError):
        stat = {'dynamics': dyn_stat}

    try:
        with open('tool_stat.json', 'r') as f_in:
            tool_stat = json.load(f_in)
        os.remove('tool_stat.json')
    except IOError:
        tool_stat = {}
        
    dyn_stat['tool_stat'] = tool_stat
    
    dyn_stat['is_pcc'] = False
    dyn_stat['cpu_count'] = multiprocessing.cpu_count()
    dyn_stat['system'] = platform.system()
    dyn_stat['machine'] = platform.machine()
    dyn_stat['processor'] = platform.processor()
    if os.name == 'nt':
        dyn_stat['computer_name'] = os.environ['COMPUTERNAME']
        dyn_stat['win32_ver'] = platform.win32_ver()
    else:
        dyn_stat['COMPUTER_NAME'] = "usr"
        dyn_stat['win32_ver'] = []

    with open('stat.json', 'wb') as f_out:
        json.dump(stat, f_out, indent=4)


def _copy_log_files(prefix, log_folder='log'):
    """
    Gathers all files that ends with either .log or .txt, prepends dyn_ to the file-name
    and copies them into a log-dir

    """
    import glob
    import shutil
    file_list = glob.glob('*cg.log')
    if not os.path.isdir(log_folder):
        os.mkdir(log_folder)
    for f_src in file_list:
        f_dst = os.path.join(log_folder, '{0}_{1}'.format(prefix, f_src))
        shutil.copy2(f_src, f_dst)


def call_simscript(tool='Dymola', result_mat_file='', dymola_home=''):
    """
    Calls simscript.py as a subprocess call using the 
    executable from the python virtual environment.
    """

    t_1 = time.time()
    iso_time = datetime.datetime.now().isoformat()
    if os.name == 'nt':
        py_path = _query_registry()
        command = '"{0}" simscript.py {1} {2} {3}'\
            .format(py_path, tool, result_mat_file, dymola_home)
    elif os.name == 'posix':
        py_exc = os.path.join(LINUX_VIR_ENV, '/bin/python')
        command = '{0} simscript.py {1} {2} {3}'\
            .format(py_exc, tool, result_mat_file, 'dummy_input')
    else:
        sys.stderr.write('ERROR: Only Windows or Linux supported.\n')
        _copy_log_files('dynamics')
        sys.exit(5)

    print 'Calling "{0}" as a subprocess.'.format(command)
    try:
        popen = subprocess.Popen(command, shell=True)
        (stdoutdata, _) = popen.communicate()
        #print stdoutdata
        #popen.wait()
        if popen.returncode != 0:
            raise subprocess.CalledProcessError(popen.returncode, command)
        
        execution_time = time.time() - t_1
        _write_out_stat(execution_time, iso_time)
        _copy_log_files('dynamics')
    except subprocess.CalledProcessError, err:
        sys.stderr.write('Out-print : {0}\n\n{1}\n'.format(err, err.output))
        sys.stderr.write('Failed calling {0}\n'.format(command))
        _copy_log_files('dynamics')
        sys.exit(5)
