/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "ModelicaCodeGen.h"
#include <boost/filesystem.hpp>
#include <boost/lexical_cast.hpp> 

#include "runSL_CodeGen.h"

#include <fstream>
#include <sstream>

#include <ctemplate\template.h>
#include "Rpc.h"
#include "CommandExecutor.h"
#include "ModelicaTemplate.h"

#include "CGLog.h"

#include "utils.h"


MoCodeGen::MoCodeGen(CyPhyML::Component &cyphyComponent, const std::string &outputDirectory, const std::string logFile)
	:_com(cyphyComponent), _directory(outputDirectory), _logFileName(logFile)
{
	_comName = _com.name();
	_wrapperName = _comName + "_wrapper";

	int uid = _com.uniqueId();
}

void MoCodeGen::setPackageName(const std::string &pkg)
{
	_pkgName = pkg;
}

void MoCodeGen::gen()
{
	if(_com == Udm::null)
		return;
	
	SFUtils::obj2VarNameMap.clear();
	set<CyPhyML::SignalFlowModel> sfs = _com.SignalFlowModel_kind_children();
	if(sfs.empty()) return;
	for(auto it_sf = sfs.begin(); it_sf!=sfs.end(); ++it_sf)
	{
		CyPhyML::SignalFlowModel sf = *it_sf;

		if(((std::string)sf.URI()).empty())
		{
			sf.URI() = string("SignalFlow.")+_comName+string("_type");
		}

		set<CyPhyML::SignalFlow::Subsystem> subs  = sf.SignalFlow_Subsystem_kind_children();
		if(subs.empty()) continue;
		CyPhyML::SignalFlow::Subsystem topsub = *(subs.begin());
		_topSubSystems.push_back(topsub);

		set<CyPhyML::SignalFlow::InputPort, InputPortSorter> inports = topsub.InputPort_kind_children_sorted(InputPortSorter());
		for(set<CyPhyML::SignalFlow::InputPort, InputPortSorter>::iterator it_in=inports.begin();it_in!=inports.end();++it_in)
		{
			CyPhyML::SignalFlow::InputPort inport = *it_in;
			_inputs.push_back(inport);
			CyPhyML::SignalFlow::TypeBaseRef inref = inport.TypeBaseRef_child();
			_inputMap[inport] = inref;
		}

		set<CyPhyML::SignalFlow::OutputPort, OutputPortSorter> outports = topsub.OutputPort_kind_children_sorted(OutputPortSorter());
		for(set<CyPhyML::SignalFlow::OutputPort, OutputPortSorter>::iterator it_out=outports.begin();it_out!=outports.end();++it_out)
		{
			CyPhyML::SignalFlow::OutputPort outport = *it_out;
			_outputs.push_back(outport);
			CyPhyML::SignalFlow::TypeBaseRef outref = outport.TypeBaseRef_child();
			_outputMap[outport] = outref;
		}

		set<CyPhyML::SF_ParameterRef> paramRefs = sf.SF_ParameterRef_kind_children();
		for(auto it_ref=paramRefs.begin(); it_ref!=paramRefs.end(); ++it_ref)
		{
			CyPhyML::SF_ParameterRef pref = *it_ref;
			std::string ref_name = pref.name();
			std::string ref_type;
			set<CyPhyML::SF_ValueFlowDstPortMap> inflows = pref.srcSF_ValueFlowDstPortMap();
			if(!inflows.empty())
			{
				CyPhyML::SF_ValueFlowDstPortMap inflow = *(inflows.begin());
				CyPhyML::ValueFlowTarget inend = inflow.srcSF_ValueFlowDstPortMap_end();
				if(Uml::IsDerivedFrom(inend.type(), CyPhyML::Parameter::meta))
				{
					CyPhyML::Parameter inparam = CyPhyML::Parameter::Cast(inend);
					ref_type = inparam.DataType();
				}
			}
			if (ref_type=="Float")
			//	_parameterRefMap[ref_name] = "double";
				_parameterRefMap[pref] = "double";
			else if (ref_type=="Integer")
			//	_parameterRefMap[ref_name] = "long";
				_parameterRefMap[pref] = "long";
			else if (ref_type=="Boolean")
				//_parameterRefMap[ref_name] = "boolean";
				_parameterRefMap[pref] = "boolean";
			else
				//_parameterRefMap[ref_name] = "double";
				_parameterRefMap[pref] = "double";
		}

		set<CyPhyML::SF_DataRef> dataRefs = sf.SF_DataRef_kind_children();
		for(auto it_ref=dataRefs.begin(); it_ref!=dataRefs.end(); ++it_ref)
		{
			CyPhyML::SF_DataRef dref = *it_ref;
			std::string ref_name = dref.name();
			std::string ref_type;
			set<CyPhyML::SF_ValueFlowDstPortMap> inflows = dref.srcSF_ValueFlowDstPortMap();
			if(!inflows.empty())
			{
				CyPhyML::SF_ValueFlowDstPortMap inflow = *(inflows.begin());
				CyPhyML::ValueFlowTarget inend = inflow.srcSF_ValueFlowDstPortMap_end();
				if(Uml::IsDerivedFrom(inend.type(), CyPhyML::Parameter::meta))
				{
					CyPhyML::Parameter inparam = CyPhyML::Parameter::Cast(inend);
					ref_type = inparam.DataType();
				}
			}
			if (ref_type=="Float")
			//	_parameterRefMap[ref_name] = "double";
				_sfdataRefMap[dref] = "double";
			else if (ref_type=="Integer")
			//	_parameterRefMap[ref_name] = "long";
				_sfdataRefMap[dref] = "long";
			else if (ref_type=="Boolean")
			//	_parameterRefMap[ref_name] = "boolean";
				_sfdataRefMap[dref] = "boolean";
			else
			//	_parameterRefMap[ref_name] = "double";
				_sfdataRefMap[dref] = "double";
		}
	}

	ComponentCodeGen cg(_com, _directory, "");
	cg.runCodeGen(_logFileName);
	if(cg.getError())
	{
		printLog(">>>>>>>>>>>>>>>>No modelica files generated due to the SL_CodeGen error.<<<<<<<<<<<<<<<<<<<<<<");
		return;
	}

	_hfiles = cg.getGenerated_H_Files();
	_cfiles = cg.getGenerated_C_Files();

	if(_topSubSystems.empty()) return;

	_topSubSystemName = (std::string)((*(_topSubSystems.begin())).name());
	_contextName = _topSubSystemName + "_context";

	printLog("generated Modelica files: ");
	genWrapper_source();
	genWrapper_proj();
	printLog("generated Modelica Package files: ");
	genModelica();
	genPackageMo();
}

std::string MoCodeGen::getDataVarFullPath(const CyPhyML::SF_DataParameterBase &obj)
{
	std::string ret = obj.name();
	Udm::Object subParent;
	if(Uml::IsDerivedFrom(obj.type(), CyPhyML::SignalFlow::Data::meta))
	{
		CyPhyML::SignalFlow::Data data = CyPhyML::SignalFlow::Data::Cast(obj);
		CyPhyML::SignalFlow::State state = data.State_parent();
		ret = (std::string)state.name()+"_"+ret;
		subParent  = state.GetParent();
	}
	else
	{
		CyPhyML::SignalFlow::SFData sfdata = CyPhyML::SignalFlow::SFData::Cast(obj);
		CyPhyML::SignalFlow::SFState sfstate = sfdata.SFState_parent();
		ret = (std::string)sfstate.name()+"_"+ret;
		subParent  = sfstate.GetParent();
	}

	while(Uml::IsDerivedFrom(subParent.type(), CyPhyML::SignalFlow::Subsystem::meta))
	{
		map<int, std::string>::iterator pos = SFUtils::obj2VarNameMap.find(subParent.uniqueId());
		if(pos!=SFUtils::obj2VarNameMap.end())
			ret = (*pos).second+"."+ret;
		else
		{
			if(subParent.isInstance())
			{
				pos = SFUtils::obj2VarNameMap.find(subParent.archetype().uniqueId());
				if(pos!=SFUtils::obj2VarNameMap.end())
					ret = (*pos).second+"."+ret;
				else
					return ret;
			}
			else
				return ret;
		}
		subParent = subParent.GetParent();
		std::string subname  = subParent.getPath2();
	}

	return ret;
}

std::string MoCodeGen::getParaVarFullPath(const CyPhyML::SignalFlow::SF_Parameter &param)
{
	std::string ret;
	map<int, std::string>::iterator pos = SFUtils::obj2VarNameMap.find(param.uniqueId());
	if(pos!=SFUtils::obj2VarNameMap.end())
		ret = (*pos).second;
	else
	{
		if(param.isInstance())
		{
			pos = SFUtils::obj2VarNameMap.find(param.archetype().uniqueId());
			if(pos!=SFUtils::obj2VarNameMap.end())
				ret = (*pos).second;
			else
				return ret;
		}
		else
			return ret;
	}
		
	Udm::Object block = param.GetParent();		
	Udm::Object subParent  = block.GetParent();

	while(Uml::IsDerivedFrom(subParent.type(), CyPhyML::SignalFlow::Subsystem::meta))
	{
		pos = SFUtils::obj2VarNameMap.find(subParent.uniqueId());
		if(pos!=SFUtils::obj2VarNameMap.end())
			ret = (*pos).second+"."+ret;
		else
		{
			if(subParent.isInstance())
			{
				pos = SFUtils::obj2VarNameMap.find(subParent.archetype().uniqueId());
				if(pos!=SFUtils::obj2VarNameMap.end())
					ret = (*pos).second+"."+ret;
				else
					return ret;
			}
			else
				return ret;
		}
		subParent = subParent.GetParent();
		std::string subname  = subParent.getPath2();
	}

	return ret;
}

void MoCodeGen::genWrapper_source()
{
	ctemplate::TemplateDictionary srcdict( "source" );
	srcdict.SetValue("COMPONENT_NAME", _comName);
	srcdict.SetValue("TOPSUBSYTEM_NAME", _topSubSystemName);
	std::string args, vars;
	for(list<CyPhyML::SignalFlow::InputPort>::iterator it_in=_inputs.begin();it_in!=_inputs.end();++it_in)
	{
		CyPhyML::SignalFlow::InputPort in = *it_in;
		map<CyPhyML::SignalFlow::InputPort, CyPhyML::SignalFlow::TypeBaseRef>::iterator pos = _inputMap.find(in);
		if(pos!=_inputMap.end())
		{
			CyPhyML::SignalFlow::TypeBaseRef tref = (*pos).second;
			ctemplate::TemplateDictionary* inDict = srcdict.AddSectionDictionary("INPUT_SIGNAL");
			inDict->SetValue( "SIGNAL_TYPE", (std::string)tref.name());
			inDict->SetValue( "SIGNAL_NAME", (std::string)in.name());
		}
	}
	for(list<CyPhyML::SignalFlow::OutputPort>::iterator it_out=_outputs.begin();it_out!=_outputs.end();++it_out)
	{
		CyPhyML::SignalFlow::OutputPort out = *it_out;
		map<CyPhyML::SignalFlow::OutputPort, CyPhyML::SignalFlow::TypeBaseRef>::iterator pos = _outputMap.find(out);
		if(pos!=_outputMap.end())
		{
			CyPhyML::SignalFlow::TypeBaseRef tref = (*pos).second;
			ctemplate::TemplateDictionary* inDict = srcdict.AddSectionDictionary("OUTPUT_SIGNAL");
			inDict->SetValue( "SIGNAL_TYPE", (std::string)tref.name());
			inDict->SetValue( "SIGNAL_NAME", (std::string)out.name());
		}
	}

	//{{SINGLE_VAR_TYPE}} {{SINGLE_VAR_NAME}}{{#INPUT_ARG}},{{VAR_TYPE}} {{VAR_NAME}}{{/INPUT_ARG}}
	//{{#VARS}}, {{VAR_NAME}}{{/VARS}}
	if(!_parameterRefMap.empty())
	{
		map<CyPhyML::SF_ParameterRef, std::string >::iterator pos=_parameterRefMap.begin();
		while(pos!=_parameterRefMap.end())
		{
			CyPhyML::SF_ParameterRef sfpref = (*pos).first;
			std::string sftype = (*pos).second;			
			CyPhyML::SignalFlow::SF_Parameter refObj = sfpref.ref();
			if(!refObj)
			{ 
				pos++;
				continue;
			}

			std::string refname = sfpref.name();
			if(pos==_parameterRefMap.begin())
			{
				srcdict.SetValue("SINGLE_VAR_TYPE", sftype);
				srcdict.SetValue("SINGLE_VAR_NAME", refname);
			}
			else
			{				
				ctemplate::TemplateDictionary* argDict = srcdict.AddSectionDictionary("INPUT_ARG");
				argDict->SetValue( "VAR_TYPE", sftype);
				argDict->SetValue( "VAR_NAME", refname);
			}
			
			ctemplate::TemplateDictionary* paramDict = srcdict.AddSectionDictionary("PARAM_INIT");
			paramDict->SetValue("VAR_NAME", refname);
			paramDict->SetValue("PARAM_NAME", getParaVarFullPath(refObj));
			pos++;
		}
	}

	if(!_sfdataRefMap.empty())
	{		
		map<CyPhyML::SF_DataRef, std::string >::iterator pos=_sfdataRefMap.begin();
	
		while(pos!=_sfdataRefMap.end())
		{
			CyPhyML::SF_DataRef sfdref = (*pos).first;
			CyPhyML::SF_DataParameterBase refObj = sfdref.ref();
			if(!refObj)
			{
				pos++;
				continue;
			}

			std::string refname = sfdref.name();

			if(pos==_sfdataRefMap.begin() && _parameterRefMap.empty())
			{
				srcdict.SetValue("SINGLE_VAR_TYPE", (*pos).second);
				srcdict.SetValue("SINGLE_VAR_NAME",  refname);
			}		
			else
			{
				ctemplate::TemplateDictionary* argDict = srcdict.AddSectionDictionary("INPUT_ARG");
				argDict->SetValue("VAR_TYPE", (*pos).second);
				argDict->SetValue("VAR_NAME", refname);
			}

			ctemplate::TemplateDictionary* paramDict = srcdict.AddSectionDictionary("PARAM_INIT");
			paramDict->SetValue("VAR_NAME", refname);
			paramDict->SetValue("PARAM_NAME", getDataVarFullPath(refObj));
			pos++;
		}	
	}

	std::string output;
	//h file
	boost::filesystem::current_path(_directory);
	std::string hFileName = _wrapperName +".h";
	ctemplate::StringToTemplateCache("htpl", Modelica_Template::get_h_tpl(), ctemplate::DO_NOT_STRIP);
	ctemplate::ExpandTemplate("htpl", ctemplate::DO_NOT_STRIP, &srcdict, &output);
	ofstream h_file(hFileName.c_str() );
	h_file << output ;
	h_file.close();
	_hfiles.insert(_directory+"\\"+hFileName);

	printLog("\t\t"+hFileName);
	
	output.clear();

	//c file
	boost::filesystem::current_path(_directory+"\\"+_comName);
	std::string cFileName = _wrapperName +".c";
	ctemplate::StringToTemplateCache("ctpl", Modelica_Template::get_c_tpl(), ctemplate::DO_NOT_STRIP);
	ctemplate::ExpandTemplate("ctpl", ctemplate::DO_NOT_STRIP, &srcdict, &output);
	ofstream c_file(cFileName.c_str() );
	c_file << output;
	c_file.close();
	_cfiles.insert(cFileName);

	printLog("\t\t"+cFileName);
}

void MoCodeGen::genWrapper_proj()
{
	// Initialize the template system
	ctemplate::TemplateDictionary dict( "vcxproj" );
	
	dict.SetValue("COMPONENT_NAME", _comName);
	dict.SetValue("SOLUTION_ID", generateGUID());
	dict.SetValue("PROJECT_ID", generateGUID());
	dict.SetValue("SOURCE_ID", generateGUID());
	dict.SetValue("HEADER_ID", generateGUID());

	std::string include_path="..\\";
	dict.SetValue("INCLUDE_PATH", include_path);

	for(set<std::string>::iterator it_h=_hfiles.begin();it_h!=_hfiles.end();++it_h)
	{
		ctemplate::TemplateDictionary* inDict = dict.AddSectionDictionary("INCLUDE");	
		inDict->SetValue("INCLUDE_H", *it_h);
	}

	for(set<std::string>::iterator it_c=_cfiles.begin();it_c!=_cfiles.end();++it_c)
	{
		ctemplate::TemplateDictionary* cDict = dict.AddSectionDictionary("COMPILE");				
		cDict->SetValue("COMPILE_C", *it_c);
	}

	std::string output;

	boost::filesystem::current_path(_directory+"\\"+_comName);
	//com.vcxproj
	std::string projFileName = _comName +".vcxproj";
	ctemplate::StringToTemplateCache("projtpl", Modelica_Template::get_vcxproj_tpl(), ctemplate::DO_NOT_STRIP);
    ctemplate::ExpandTemplate("projtpl", ctemplate::DO_NOT_STRIP, &dict, &output);
	ofstream projFile(projFileName.c_str() );
	// Write the generated code out to the file and close
	projFile << output;
	projFile.close();
	
	printLog("\t\t"+projFileName);

	output.clear();
	
	//com..vcxproj.filters
	std::string filterFileName = _comName +".vcxproj.filters";
	ctemplate::StringToTemplateCache("filtertpl", Modelica_Template::get_filter_tpl(), ctemplate::DO_NOT_STRIP);
    ctemplate::ExpandTemplate("filtertpl", ctemplate::DO_NOT_STRIP, &dict, &output);
	ofstream filterFile(filterFileName.c_str());
	filterFile << output;
	filterFile.close();

	printLog("\t\t"+filterFileName);

	output.clear();

	//com.sln
	std::string slnFileName = _comName +".sln";	
	ctemplate::StringToTemplateCache("slntpl", Modelica_Template::get_sln_tpl(), ctemplate::DO_NOT_STRIP);
    ctemplate::ExpandTemplate("slntpl", ctemplate::DO_NOT_STRIP, &dict, &output);
	ofstream slnFile(slnFileName.c_str());
	slnFile << output;
	slnFile.close();

	printLog("\t\t"+slnFileName);

	//compile project file
	std::string vs10Tools = std::getenv("VS100COMNTOOLS");
	std::string devexe = "devenv.exe";
	bool vsInstalled = false;
	if(!vs10Tools.empty())
	{
		if(!boost::filesystem::exists( vs10Tools +"\\..\\IDE\\"+devexe ))
		{
			devexe =  "VCExpress.exe";
			if(boost::filesystem::exists( vs10Tools +"\\..\\IDE\\"+devexe ))
				vsInstalled = true;
		}
		else
			vsInstalled = true;
	}

	CommandExecutor cmdExec;
	cmdExec.printMsg("Build "+_comName+" static library:", MSG_INFO);

	printLog("Build "+_comName+" static library");

	std::string cmdline =  devexe +" "+ slnFileName +" /build Release /project "+projFileName+" /projectconfig Release";
	std::string makeFile_name = "make_"+_comName+".bat";
	ofstream makeFile(makeFile_name.c_str());
	makeFile << "call \"%VS100COMNTOOLS%vsvars32.bat\"\n";
	makeFile << "\"%DevEnvDir%"+devexe+"\" "+ slnFileName +" /build Release /project "+projFileName+" /projectconfig Release\n";
	makeFile.close();
	
	if(vsInstalled)
	{
		if(!cmdExec.execute(makeFile_name))
		{
			printLog(_comName+".lib fails.");
			throw udm_exception(cmdline+" failed");
		}
		std::string status = _comName+".lib ---> generated successfully.";
		cmdExec.printMsg(status);
		printLog("generated library: ..\\Release\\"+ _comName+".lib");
	}
	else
		cmdExec.printMsg("Visual Studio 2010 or Visual Studio 2010 Express is not installed."); 
	
//	if(!boost::filesystem::remove("make.bat"))
//		throw exception("boost::filesystem::remove(\"make.bat\") fails");
}

void MoCodeGen::genModelica()
{	
	//bool useBusPort = false;
	//for(list<CyPhyML::SignalFlow::InputPort>::iterator it_in=_inputs.begin();it_in!=_inputs.end();++it_in)
	//{
	//	CyPhyML::SignalFlow::InputPort in = *it_in;
	//	map<CyPhyML::SignalFlow::InputPort, CyPhyML::SignalFlow::TypeBaseRef>::iterator pos = _inputMap.find(in);
	//	if(pos!=_inputMap.end())
	//	{
	//		CyPhyML::SignalFlow::TypeBaseRef tref = (*pos).second;
	//		set<CyPhyML::IOSignal2InPort> inconns = in.srcIOSignal2InPort();
	//		if(!inconns.empty())
	//		{
	//			CyPhyML::IOSignal2InPort inconn = *(inconns.begin());
	//			CyPhyML::InSignal insignal = inconn.srcIOSignal2InPort_end();
	//			Udm::Object signal_parent = insignal.GetParent();

	//			if(Uml::IsDerivedFrom(signal_parent.type(), CyPhyML::SignalFlowBusPortInterface::meta))
	//				useBusPort = true;
	//			
	//			break;
	//		}
	//	}
	//	else
	//		throw udm_exception("unknown InputPort.");
	//}

	//if(!useBusPort)
	//{
	//
	//}

	//if(!useBusPort)
	//	return genModelica_busPort();

	boost::filesystem::current_path(_directory);
	ctemplate::TemplateDictionary modict( "modelica" );
	modict.SetValue("COMPONENT_NAME", _comName);
	
	set<CyPhyML::AggregatePort> aports;
//	set<CyPhyML::SignalFlowBusPortInterface> busports;
	set<CyPhyML::SignalFlowBusPortInterface> busportInterfaceSet;
	for(list<CyPhyML::SignalFlow::InputPort>::iterator it_in=_inputs.begin();it_in!=_inputs.end();++it_in)
	{
		CyPhyML::SignalFlow::InputPort in = *it_in;
		map<CyPhyML::SignalFlow::InputPort, CyPhyML::SignalFlow::TypeBaseRef>::iterator pos = _inputMap.find(in);
		if(pos!=_inputMap.end())
		{
			set<CyPhyML::IOSignal2InPort> iosignal2ins = in.srcIOSignal2InPort();
			if(iosignal2ins.empty())
				continue;
			CyPhyML::IOSignal2InPort iosignal2in = *(iosignal2ins.begin());
			CyPhyML::InSignal insignal = iosignal2in.srcIOSignal2InPort_end();

			CyPhyML::SignalFlow::TypeBaseRef tref = (*pos).second;
			std::string in_name = insignal.name();
			ctemplate::TemplateDictionary* inDict = modict.AddSectionDictionary("INPUT_SIGNAL");	
			inDict->SetValue("DATA_TYPE", convertMoType((std::string)tref.name()));
			inDict->SetValue("DATA_NAME", in_name);

			std::string in_arg_name;
			//check the parent of the insignal
			Udm::Object signal_parent = insignal.GetParent();
			if(Uml::IsDerivedFrom(signal_parent.type(), CyPhyML::SignalFlowBusPortInterface::meta))
			{
				CyPhyML::SignalFlowBusPortInterface sfbpi = CyPhyML::SignalFlowBusPortInterface::Cast(signal_parent);
				CyPhyML::BusPort bp = getBusPort(sfbpi);
				if(!bp)
					continue;
				set<CyPhyML::SignalFlowBusPortInterface>::iterator set_it = busportInterfaceSet.find(sfbpi);
				if(set_it==busportInterfaceSet.end())
				{
					busportInterfaceSet.insert(sfbpi);
					ctemplate::TemplateDictionary* portDefDict = modict.AddSectionDictionary("BUSPORT_DEF");	
					portDefDict->SetValue("PORT_TYPE", (std::string)bp.Type());
					portDefDict->SetValue("PORT_NAME", (std::string)sfbpi.name());
				}
				in_arg_name = (std::string)sfbpi.name()+"."+in_name;
			}
			else
			{
				CyPhyML::SignalFlow::TypeBaseRef tref = (*pos).second;
				std::string in_name = insignal.name();
				ctemplate::TemplateDictionary* indefDict = modict.AddSectionDictionary("INPUT_SIGNAL_DEF");	
				indefDict->SetValue("DATA_TYPE", convertMoType((std::string)tref.name()));
				indefDict->SetValue("DATA_NAME", in_name);
				in_arg_name = in_name;
			}
			ctemplate::TemplateDictionary* inargDict = modict.AddSectionDictionary("INPUT_ARG");	
			inargDict->SetValue("DATA_NAME", in_arg_name);
		}
	}

	if(_outputs.size()>1)
	{
		modict.ShowSection("LEFT");
		modict.ShowSection("RIGHT");
	}

	for(list<CyPhyML::SignalFlow::OutputPort>::iterator it_out=_outputs.begin();it_out!=_outputs.end();++it_out)
	{
		CyPhyML::SignalFlow::OutputPort out = *it_out;
		map<CyPhyML::SignalFlow::OutputPort, CyPhyML::SignalFlow::TypeBaseRef>::iterator pos = _outputMap.find(out);
		if(pos!=_outputMap.end())
		{
			set<CyPhyML::OutPort2IOSignal> out2iosignals = out.dstOutPort2IOSignal();
			if(out2iosignals.empty())
				continue;

			CyPhyML::OutPort2IOSignal out2iosignal = *(out2iosignals.begin());
			CyPhyML::OutSignal outsignal = out2iosignal.dstOutPort2IOSignal_end();

			CyPhyML::SignalFlow::TypeBaseRef tref = (*pos).second;
			//std::string out_name = out.name();
			std::string out_name = outsignal.name();
			ctemplate::TemplateDictionary* outDict = modict.AddSectionDictionary("OUTPUT_SIGNAL");
			outDict->SetValue( "DATA_TYPE", convertMoType((std::string)tref.name()));
			outDict->SetValue( "DATA_NAME", out_name);
			
			std::string out_arg_name;
			Udm::Object signal_parent = outsignal.GetParent();
			if(Uml::IsDerivedFrom(signal_parent.type(), CyPhyML::SignalFlowBusPortInterface::meta))
			{	
				CyPhyML::SignalFlowBusPortInterface sfbpi = CyPhyML::SignalFlowBusPortInterface::Cast(signal_parent);
				CyPhyML::BusPort bp = getBusPort(sfbpi);
				if(!bp)
					continue;
				set<CyPhyML::SignalFlowBusPortInterface>::iterator set_it = busportInterfaceSet.find(sfbpi);
				if(set_it==busportInterfaceSet.end())
				{
					busportInterfaceSet.insert(sfbpi);
					ctemplate::TemplateDictionary* portDefDict = modict.AddSectionDictionary("BUSPORT_DEF");	
					portDefDict->SetValue("PORT_TYPE", (std::string)bp.Type());
					portDefDict->SetValue("PORT_NAME", (std::string)sfbpi.name());
				}
				out_arg_name = (std::string)sfbpi.name()+"."+out_name;
			}
			else
			{
				ctemplate::TemplateDictionary* outdefDict = modict.AddSectionDictionary("OUTPUT_SIGNAL_DEF");
				outdefDict->SetValue( "DATA_TYPE", convertMoType((std::string)tref.name()));
				outdefDict->SetValue( "DATA_NAME", out_name);
				ctemplate::TemplateDictionary* outDict_p = modict.AddSectionDictionary("OUTPUT_SIGNAL_P");
				outDict_p->SetValue( "DATA_TYPE", convertMoType((std::string)tref.name()));
				outDict_p->SetValue( "DATA_NAME", out_name);
				out_arg_name = out_name;
			}
			if(it_out==_outputs.begin())
				modict.SetValue("FIRST_DATA_NAME", out_arg_name);
			else
			{
				ctemplate::TemplateDictionary* outDict_a = modict.AddSectionDictionary("OUTPUT_SIGNAL_A");
				outDict_a->SetValue( "DATA_NAME", out_arg_name);
			}
		}
	}

	//for SF_ParameterRef
	if(!_parameterRefMap.empty())
	{
		map<CyPhyML::SF_ParameterRef, std::string >::iterator pos=_parameterRefMap.begin();
		CyPhyML::SF_ParameterRef sfpref = (*pos).first;
		std::string refname = sfpref.name();
		modict.SetValue("SINGLE_VAR", refname);
	
		while(pos!=_parameterRefMap.end())
		{
			sfpref = (*pos).first;
			refname = sfpref.name();
			ctemplate::TemplateDictionary* argDict = modict.AddSectionDictionary("PARAMETER");
			argDict->SetValue( "PARAMETER_TYPE", convertMoType((*pos).second));
			argDict->SetValue( "PARAMETER_NAME", refname);
			
			if(pos!=_parameterRefMap.begin())
			{
				ctemplate::TemplateDictionary* varDict = modict.AddSectionDictionary("VARS");
				varDict->SetValue("VAR_NAME", refname);	
			}
			pos++;
		}
	}

	if(!_sfdataRefMap.empty())
	{
		map<CyPhyML::SF_DataRef, std::string>::iterator pos=_sfdataRefMap.begin();
		CyPhyML::SF_DataRef sfdref = (*pos).first;
		std::string refname = sfdref.name();
		modict.SetValue("SINGLE_VAR", refname);
					
		while(pos!=_sfdataRefMap.end())
		{
			sfdref = (*pos).first;
			refname = sfdref.name();
			ctemplate::TemplateDictionary* argDict = modict.AddSectionDictionary("PARAMETER");
			argDict->SetValue( "PARAMETER_TYPE", convertMoType((*pos).second));
			argDict->SetValue( "PARAMETER_NAME", refname);
			
			if(pos!=_sfdataRefMap.begin())
			{
				ctemplate::TemplateDictionary* varDict = modict.AddSectionDictionary("VARS");
				varDict->SetValue("VAR_NAME", refname);	
			}
			pos++;
		}
	}

	std::string output;
	std::string moFileName = _comName +".mo";
	ctemplate::StringToTemplateCache("motpl", Modelica_Template::get_mo_tpl(), ctemplate::DO_NOT_STRIP);
    ctemplate::ExpandTemplate("motpl", ctemplate::DO_NOT_STRIP, &modict, &output);
	ofstream moFile(moFileName.c_str() );
	moFile << output;
	moFile.close();
	printLog("\t\t"+moFileName);
}

CyPhyML::BusPort MoCodeGen::getBusPort(const CyPhyML::SignalFlowBusPortInterface &busportInterface)
{
	map<CyPhyML::SignalFlowBusPortInterface, CyPhyML::BusPort>::iterator pos = _busportMap.find(busportInterface);
	if(pos!=_busportMap.end())
		return (*pos).second;

	CyPhyML::BusPort busport;
	CyPhyML::SignalFlowBusPortMap_Members_Base port_base;
	set<CyPhyML::SignalFlowBusPortMap> in_flows = busportInterface.srcSignalFlowBusPortMap();
	if(!in_flows.empty())
	{
		port_base = (*(in_flows.begin())).srcSignalFlowBusPortMap_end();
	}
	else
	{
		set<CyPhyML::SignalFlowBusPortMap> out_flows = busportInterface.dstSignalFlowBusPortMap();
		if(!out_flows.empty())
		{
			port_base = (*(out_flows.begin())).dstSignalFlowBusPortMap_end();
		}
		else
			return busport;
	}
				
	if(Uml::IsDerivedFrom(port_base.type(), CyPhyML::BusPort::meta))
		busport = CyPhyML::BusPort::Cast(port_base);

	_busportMap[busportInterface] = busport;
	return busport;
}
//
//void MoCodeGen::genModelica_busPort()
//{	
//	boost::filesystem::current_path(_directory);
//	ctemplate::TemplateDictionary modict( "modelica" );
//	modict.SetValue("COMPONENT_NAME", _comName);
//	
//	set<CyPhyML::BusPort> busports;
//
//	for(list<CyPhyML::SignalFlow::InputPort>::iterator it_in=_inputs.begin();it_in!=_inputs.end();++it_in)
//	{
//		CyPhyML::SignalFlow::InputPort in = *it_in;
//		map<CyPhyML::SignalFlow::InputPort, CyPhyML::SignalFlow::TypeBaseRef>::iterator pos = _inputMap.find(in);
//		if(pos!=_inputMap.end())
//		{
//			CyPhyML::SignalFlow::TypeBaseRef tref = (*pos).second;
//			ctemplate::TemplateDictionary* inDict = modict.AddSectionDictionary("INPUT_SIGNAL");	
//			inDict->SetValue("DATA_TYPE", convertMoType((std::string)tref.name()));
//			inDict->SetValue("DATA_NAME", (std::string)in.name());
//
//			set<CyPhyML::IOSignal2InPort> inconns = in.srcIOSignal2InPort();
//			if(!inconns.empty())
//			{
//				CyPhyML::IOSignal2InPort inconn = *(inconns.begin());
//				CyPhyML::InSignal insignal = inconn.srcIOSignal2InPort_end();
//				Udm::Object signal_parent = insignal.GetParent();
//
//				if(Uml::IsDerivedFrom(signal_parent.type(), CyPhyML::SignalFlowBusPortInterface::meta))
//				{	
//					CyPhyML::SignalFlowBusPortInterface busportInterface = CyPhyML::SignalFlowBusPortInterface::Cast(signal_parent);
//				
//					CyPhyML::SignalFlowBusPortMap busPortMap;
//					set<CyPhyML::SignalFlowBusPortMap> in_flows = busportInterface.srcSignalFlowBusPortMap();
//					if(!in_flows.empty())
//						busPortMap = *(in_flows.begin());
//					else
//					{
//						set<CyPhyML::SignalFlowBusPortMap> out_flows = busportInterface.dstSignalFlowBusPortMap();
//						if(!out_flows.empty())
//							busPortMap = *(out_flows.begin());
//						else
//							continue;
//					}
//				
//					CyPhyML::BusPort busport;
//					CyPhyML::SignalFlowBusPortMap_Members_Base port_base = busPortMap.srcSignalFlowBusPortMap_end();
//					if(Uml::IsDerivedFrom(port_base.type(), CyPhyML::BusPort::meta))
//						busport = CyPhyML::BusPort::Cast(port_base);
//					else
//					{
//						port_base = busPortMap.dstSignalFlowBusPortMap_end();
//						if(Uml::IsDerivedFrom(port_base.type(), CyPhyML::BusPort::meta))
//							busport = CyPhyML::BusPort::Cast(port_base);
//						else
//							continue;						
//					}
//					if(busports.find(busport)==busports.end())
//					{
//						busports.insert(busport);
//						ctemplate::TemplateDictionary* portDefDict = modict.AddSectionDictionary("PORT_DEF");	
//						portDefDict->SetValue("PORT_TYPE", (std::string)busport.Type());
//						portDefDict->SetValue("PORT_NAME", (std::string)busportInterface.name());
//					}
//
//					ctemplate::TemplateDictionary* inportDict = modict.AddSectionDictionary("INPUT_PORT");	
//					inportDict->SetValue("PORT_NAME", (std::string)busportInterface.name());
//					inportDict->SetValue("IN_PORT", (std::string)insignal.name());
//				}
//			}
//		}
//	}
//
//	if(_outputs.size()>1)
//	{
//		modict.ShowSection("LEFT");
//		modict.ShowSection("RIGHT");
//	}
//
//	for(list<CyPhyML::SignalFlow::OutputPort>::iterator it_out=_outputs.begin();it_out!=_outputs.end();++it_out)
//	{
//		CyPhyML::SignalFlow::OutputPort out = *it_out;
//		map<CyPhyML::SignalFlow::OutputPort, CyPhyML::SignalFlow::TypeBaseRef>::iterator pos = _outputMap.find(out);
//		if(pos!=_outputMap.end())
//		{
//			CyPhyML::SignalFlow::TypeBaseRef tref = (*pos).second;
//			std::string out_name = out.name();
//			ctemplate::TemplateDictionary* outDict = modict.AddSectionDictionary("OUTPUT_SIGNAL");
//			outDict->SetValue( "DATA_TYPE", convertMoType((std::string)tref.name()));
//			outDict->SetValue( "DATA_NAME", out_name);
//
//			set<CyPhyML::OutPort2IOSignal> outconns = out.dstOutPort2IOSignal();
//			if(!outconns.empty())
//			{
//				CyPhyML::OutPort2IOSignal outconn = *(outconns.begin());
//				CyPhyML::OutSignal outsignal = outconn.dstOutPort2IOSignal_end();
//				Udm::Object signal_parent = outsignal.GetParent();
//
//				if(Uml::IsDerivedFrom(signal_parent.type(), CyPhyML::SignalFlowBusPortInterface::meta))
//				{	
//					CyPhyML::SignalFlowBusPortInterface busportInterface = CyPhyML::SignalFlowBusPortInterface::Cast(signal_parent);
//				
//					CyPhyML::SignalFlowBusPortMap busPortMap;
//					set<CyPhyML::SignalFlowBusPortMap> in_flows = busportInterface.srcSignalFlowBusPortMap();
//					if(!in_flows.empty())
//						busPortMap = *(in_flows.begin());
//					else
//					{
//						set<CyPhyML::SignalFlowBusPortMap> out_flows = busportInterface.dstSignalFlowBusPortMap();
//						if(!out_flows.empty())
//							busPortMap = *(out_flows.begin());
//						else
//							continue;
//					}
//				
//					CyPhyML::BusPort busport;
//					CyPhyML::SignalFlowBusPortMap_Members_Base port_base = busPortMap.srcSignalFlowBusPortMap_end();
//					if(Uml::IsDerivedFrom(port_base.type(), CyPhyML::BusPort::meta))
//						busport = CyPhyML::BusPort::Cast(port_base);
//					else
//					{
//						port_base = busPortMap.dstSignalFlowBusPortMap_end();
//						if(Uml::IsDerivedFrom(port_base.type(), CyPhyML::BusPort::meta))
//							busport = CyPhyML::BusPort::Cast(port_base);
//						else
//							continue;						
//					}
//					if(busports.find(busport)==busports.end())
//					{
//						busports.insert(busport);
//						ctemplate::TemplateDictionary* portDefDict = modict.AddSectionDictionary("PORT_DEF");	
//						portDefDict->SetValue("PORT_TYPE", (std::string)busport.Type());
//						portDefDict->SetValue("PORT_NAME", (std::string)busportInterface.name());
//					}
//					if(it_out==_outputs.begin())
//					{
//						modict.SetValue("FIRST_DATA_NAME", (std::string)busportInterface.name());
//						modict.SetValue("FIRST_PORT_NAME", (std::string)outsignal.name());
//					}
//					else
//					{
//						ctemplate::TemplateDictionary* inportDict = modict.AddSectionDictionary("OUTPUT_PORT");	
//						inportDict->SetValue("PORT_NAME", (std::string)busportInterface.name());
//						inportDict->SetValue("OUT_PORT", (std::string)outsignal.name());
//					}
//				}
//			}
//		}
//	}
//
//	std::string output;
//	std::string moFileName = _comName +".mo";
//	ctemplate::StringToTemplateCache("motpl", Modelica_Template::get_mo_tpl_busPort(), ctemplate::DO_NOT_STRIP);
//    ctemplate::ExpandTemplate("motpl", ctemplate::DO_NOT_STRIP, &modict, &output);
//	ofstream moFile(moFileName.c_str() );
//	moFile << output;
//	moFile.close();
//	printLog("\t\t"+moFileName);
//}

void MoCodeGen::genPackageMo()
{
	boost::filesystem::current_path(_directory);
	// Initialize the template system
	ctemplate::TemplateDictionary pkgdict( "pkgmo" );
	pkgdict.SetValue("COMPONENTPKG_NAME", _pkgName);

	std::string output;
	std::string pkgFileName = "package.mo";
	ctemplate::StringToTemplateCache("pkgtpl", Modelica_Template::get_pkg_tpl(), ctemplate::DO_NOT_STRIP);
    ctemplate::ExpandTemplate("pkgtpl", ctemplate::DO_NOT_STRIP, &pkgdict, &output);
	ofstream pkgFile(pkgFileName.c_str() );
	pkgFile << output;
	pkgFile.close();
	printLog("\t\t"+pkgFileName);
}

std::string MoCodeGen::generateGUID()
{
	UUID uuid;     
	UuidCreate ( &uuid );      
	unsigned char * str;     
	UuidToStringA ( &uuid, &str );      
	std::string guid( ( char* ) str );
	guid = "{"+guid+"}";
	RpcStringFreeA ( &str ); 
	return guid;
}

std::string MoCodeGen::convertMoType(const std::string &slType)
{
	if(slType == "double" || slType == "float" || slType == "single")
		return "Real";
	else if(slType == "long" || slType == "int32" || slType == "int16" || slType == "int8" 
			|| slType == "int" || slType == "uint" || slType == "uint32" || slType == "uint16" 
			|| slType== "uint8" )
		return "Integer";
	else if(slType == "boolean")
		return "Boolean";
	else
		return "Real";
}