/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "stdafx.h"
#include "UdmApp.h"
#include "UdmConfig.h"
#include "Uml.h"
#include "UdmUtil.h"

#include <boost/filesystem.hpp>
#include <boost/python.hpp>
#include <queue>

#include "Mga.h"

#include "ModelicaCodeGen.h"
#include "runSL_CodeGen.h"

#include "UdmConsole.h"

//#include "utils.h"

#include "CGLog.h"

#include <algorithm>
#include <UdmUtil.h>

#include <Python.h>

struct PyObject_RAII
{
	PyObject* p;
	PyObject_RAII() : p(NULL) { }
	PyObject_RAII(PyObject* p) : p(p) { }
	operator PyObject*() { return p; }
	~PyObject_RAII() { Py_XDECREF(p); }
};

std::string GetPythonError()
{
	PyObject_RAII type, value, traceback;
	PyErr_Fetch(&type.p, &value.p, &traceback.p);
	PyErr_Clear();
	std::string error;
	if (type)
	{
		PyObject_RAII type_name = PyObject_GetAttrString(type, "__name__");
		if (type_name && PyString_Check(type_name))
		{
			error += PyString_AsString(type_name);
			error += ": ";
		}
	}
	if (value && PyString_Check(value))
	{
		PyObject_RAII str_value = PyObject_Str(value);
		error += PyString_AsString(str_value);
	}
	else
		error += "Unknown exception";
	error += ": ";
	if (traceback)
	{
		PyObject_RAII main = PyImport_ImportModule("__main__");
		PyObject* main_namespace = PyModule_GetDict(main);
		PyObject_RAII dict = PyDict_Copy(main_namespace);
		PyDict_SetItemString(dict, "tb", traceback);
		PyObject_RAII _none = PyRun_StringFlags(
			"import traceback\n"
			"tb = ''.join(traceback.format_tb(tb))\n", Py_file_input, dict, dict, NULL);
		PyObject* formatted_traceback = PyDict_GetItemString(dict, "tb");
		error += PyString_AsString(formatted_traceback);
	}
	else
		error += "Unknown traceback";
	return error;
}


using namespace std;

void showUsage()
{
	CString usage("CyPhy2SLC_CodeGen interpreter cannot be invoked. Please launch the interpreter inside a SignalFlow model.\r\n");
	AfxMessageBox(usage,MB_ICONINFORMATION);				
}

std::string CUdmApp::createWrapperFolder(const std::string &foldername)
{
	boost::filesystem::current_path(CUdmApp::currPath);
	
	//if(boost::filesystem::exists(foldername))
	//{
	//	if(!boost::filesystem::is_empty(foldername))
	//		boost::filesystem::remove_all(foldername);
	//}
	
	if ( !boost::filesystem::exists(foldername) )
	{
		boost::filesystem::create_directory(foldername);
		printLog("create folder: "+foldername);
	}
	boost::filesystem::path work_dir(CUdmApp::currPath+"\\"+foldername);
	boost::filesystem::current_path(work_dir);
	boost::system::error_code err;
	boost::filesystem::create_directory("SignalFlow", err);
	printLog("create folder: SignalFlow");	

	return work_dir.string()+"\\SignalFlow";
}

void CUdmApp::removeWrapperFolder(const std::string &foldername)
{
	boost::filesystem::path work_dir(CUdmApp::currPath);
	boost::filesystem::current_path(work_dir);
	if(!boost::filesystem::remove_all(foldername))
	{
		std::string err = "boost::filesystem::remove(\""+foldername+"\") fails";
		throw exception(err.c_str());
	}
}

//void __declspec(dllimport) runCyPhySF_CodeGen_dllimported(const CyPhyML::SignalFlowModel &sfmodel, std::string cyphy_file, Uml::Diagram &cyphy_meta_diagram, std::string directory="");
void CUdmApp::updateInputPort(CyPhyML::SignalFlow::SFData &sfdata)
{
	set<CyPhyML::SignalFlow::ConnToSFStates> srcConns = sfdata.srcConnToSFStates();
	for(auto it_src=srcConns.begin();it_src!=srcConns.end();++it_src)
	{
		CyPhyML::ConnToSFStates_Members_Base sport = (*it_src).srcConnToSFStates_end();
		if(Uml::IsDerivedFrom(sport.type(), CyPhyML::SignalFlow::InputPort::meta))
		{
			CyPhyML::SignalFlow::InputPort iport = CyPhyML::SignalFlow::InputPort::Cast(sport);
			iport.Number() = sfdata.Port();
			iport.name() = sfdata.name();
		}
	}
}

void CUdmApp::updateOutputPort(CyPhyML::SignalFlow::SFData &sfdata)
{
	set<CyPhyML::SignalFlow::ConnToSFStates> dstConns = sfdata.dstConnToSFStates();
	for(auto it_dst=dstConns.begin();it_dst!=dstConns.end();++it_dst)
	{
		CyPhyML::ConnToSFStates_Members_Base dport = (*it_dst).dstConnToSFStates_end();
		if(Uml::IsDerivedFrom(dport.type(), CyPhyML::SignalFlow::OutputPort::meta))
		{
			CyPhyML::SignalFlow::OutputPort oport = CyPhyML::SignalFlow::OutputPort::Cast(dport);
			oport.Number() = sfdata.Port();
			oport.name() = sfdata.name();
		}
	}
}

void CUdmApp::checkSFState(CyPhyML::SignalFlow::SFState &sfstate)
{
	sfstate.Name() = sfstate.name();
	CyPhyML::SignalFlow::SFState state = sfstate;
	CyPhyML::SignalFlow::Block sub_sfstate = sfstate.Block_parent();
	sub_sfstate.name() = sfstate.name();
	if(sfstate.isInstance())
		state = sfstate.Archetype();
	if(checkedSFStates.find(state)!=checkedSFStates.end())
		return;
	checkedSFStates.insert(state);
	CyPhyML::SignalFlow::Block sub = state.Block_parent();
	sub.name() = state.name();

	int in=1;
	int out=1;
	set<CyPhyML::SignalFlow::SFData> datas = state.SFData_kind_children();
	for(auto it=datas.begin();it!=datas.end();++it)
	{
		CyPhyML::SignalFlow::SFData data = *it;
		std::string scope = data.Scope();
		data.Name() = data.name();
		set<CyPhyML::SignalFlow::SFData> data_insts = data.Instances();
		if(scope=="INPUT_DATA")
		{
			data.Port() = in++;
			updateInputPort(data);
			for(auto it_inst=data_insts.begin();it_inst!=data_insts.end();++it_inst)
			{
				CyPhyML::SignalFlow::SFData data_inst = *it_inst;
				updateInputPort(data_inst);
			}
		}
		else if(scope=="OUTPUT_DATA")
		{
			data.Port() = out++;
			updateOutputPort(data);
			for(auto it_inst=data_insts.begin();it_inst!=data_insts.end();++it_inst)
			{
				CyPhyML::SignalFlow::SFData data_inst = *it_inst;
				updateOutputPort(data_inst);
			}
		}
		
		for(auto it_inst=data_insts.begin();it_inst!=data_insts.end();++it_inst)
		{
			(*it_inst).Port() = data.Port();
		}
	}
}

void CUdmApp::checkPortNumber(const CyPhyML::SignalFlow::Block &block)
{
	set<int> numbers;
	set<CyPhyML::SignalFlow::InputPort> inports = block.InputPort_kind_children();
	for(int i=1; i<=inports.size();++i)
		numbers.insert(i);
	set<CyPhyML::SignalFlow::InputPort> ip_zeros;
	set<int> used_ints;
	for(auto it_in=inports.begin();it_in!=inports.end();++it_in)
	{
		CyPhyML::SignalFlow::InputPort inport = *it_in;
		int n = inport.Number();
		if(n==0) 
			ip_zeros.insert(inport);
		else
		{
			set<int>::iterator in_pos = numbers.find(n);
			if(in_pos!=numbers.end())
				numbers.erase(in_pos);
			else
			{
				in_pos = used_ints.find(n);
				if(in_pos!=used_ints.end())
					ip_zeros.insert(inport);
			}
			used_ints.insert(n);
		}
	}
	for(auto it_z=ip_zeros.begin();it_z!=ip_zeros.end();++it_z)
	{
		CyPhyML::SignalFlow::InputPort ip = *it_z;
		int n = *(numbers.begin());
		numbers.erase(numbers.begin());
		ip.Number() = n;
		if(ip.isInstance())
			CyPhyML::SignalFlow::InputPort::Cast(ip.Archetype()).Number() = n;
	}

	numbers.clear();
	used_ints.clear();
	
	set<CyPhyML::SignalFlow::OutputPort> outports = block.OutputPort_kind_children();
	for(int i=1; i<=outports.size();++i)
		numbers.insert(i);
	set<CyPhyML::SignalFlow::OutputPort> op_zeros;
	for(auto it_o=outports.begin();it_o!=outports.end();++it_o)
	{
		CyPhyML::SignalFlow::OutputPort outport = *it_o;
		int n = outport.Number();
		if(n==0) 
			op_zeros.insert(outport);
		else
		{
			set<int>::iterator out_pos = numbers.find(n);
			if(out_pos!=numbers.end())
				numbers.erase(out_pos);
			else
			{
				out_pos = used_ints.find(n);
				if(out_pos!=used_ints.end())
					op_zeros.insert(outport);
			}
			used_ints.insert(n);
		}
	}

	for(auto it_zo=op_zeros.begin();it_zo!=op_zeros.end();++it_zo)
	{
		CyPhyML::SignalFlow::OutputPort op = *it_zo;
		int n = *(numbers.begin());
		numbers.erase(numbers.begin());
		op.Number() = n;
		if(op.isInstance())
			CyPhyML::SignalFlow::OutputPort::Cast(op.Archetype()).Number() = n;
	}
	
	if(Uml::IsDerivedFrom(block.type(), CyPhyML::SignalFlow::Subsystem::meta))
	{
		CyPhyML::SignalFlow::Subsystem subsys = CyPhyML::SignalFlow::Subsystem::Cast(block);
		set<CyPhyML::SignalFlow::Block> blocks = subsys.Block_kind_children();
		for(auto it_block=blocks.begin();it_block!=blocks.end();++it_block)
			checkPortNumber(*it_block);
	}
}

void CUdmApp::checkComponent(CyPhyML::Component &com)
{
	set<CyPhyML::SignalFlowModel> sfmodels = com.SignalFlowModel_kind_children();
	if(sfmodels.empty()) return;
	for(auto sf_it=sfmodels.begin();sf_it!=sfmodels.end();++sf_it)
	{
		CyPhyML::SignalFlowModel currSF = *sf_it;
		set<CyPhyML::SignalFlowModel>::iterator cpos = checkedSFs.find(currSF);
		if(cpos!=checkedSFs.end())
			continue;
		checkedSFs.insert(currSF);
		set<CyPhyML::SignalFlow::Subsystem> subs = (*sf_it).SignalFlow_Subsystem_kind_children();
		for(auto it_sub=subs.begin();it_sub!=subs.end();++it_sub)
		{
			checkBadNames(*it_sub);
			checkPortNumber(*it_sub);
		}
		
		if ( !badNameObjs.empty() )
		{
			string outmsg( "Found the following invalid names in SignalFlow models: "+std::string(currSF.name())+"\n" );
			GMEConsole::Console::writeLine(outmsg, MSG_ERROR);
			while(!badNameObjs.empty())
			{
				Udm::Object obj = badNameObjs.front();
				badNameObjs.pop_front();

				std::string obj_type = obj.type().name();
				std::string obj_name = UdmUtil::ExtractName(obj);
				std::string err = obj_type +":\t<A HREF=\"mga:"+ UdmGme::UdmId2GmeId(obj.uniqueId()) + "\">" + obj_name + "</A>";
				GMEConsole::Console::writeLine("\t"+err, MSG_ERROR);
				printLog(obj_type+":\t"+obj_name);
			}
		}
	}
}

void CUdmApp::checkBadNames(const CyPhyML::SignalFlow::Block &block)
{
	if(Uml::IsDerivedFrom(block.type(), CyPhyML::SignalFlow::Subsystem::meta))
	{
		const CyPhyML::SignalFlow::Subsystem &subsys = CyPhyML::SignalFlow::Subsystem::Cast(block);
		string subName = subsys.name();
		if ( subName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) 
		{
			badNameObjs.push_back(subsys);
		}
		
		set<CyPhyML::SignalFlow::Block> blocks = subsys.Block_kind_children();
		for(set<CyPhyML::SignalFlow::Block>::iterator bit=blocks.begin();bit!=blocks.end();++bit)
		{
			CyPhyML::SignalFlow::Block currBlock = *bit;
			checkBadNames(currBlock);
		}
	}
	else if(Uml::IsDerivedFrom(block.type(), CyPhyML::SignalFlow::Primitive::meta))
	{
		string primitiveName = block.name();
		if ( primitiveName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) {
			badNameObjs.push_back(block);
		}
	}
	
	set<CyPhyML::SignalFlow::State> states = block.State_kind_children();
	for (set<CyPhyML::SignalFlow::State>::iterator stateIter = states.begin(); stateIter != states.end(); stateIter++ )
	{
		string stateName = (*stateIter).name();
		if ( stateName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) {
			badNameObjs.push_back(*stateIter);
		}

		set<CyPhyML::SignalFlow::Transition> transitions = (*stateIter).Transition_kind_children();
		for ( set<CyPhyML::SignalFlow::Transition>::iterator transIter = transitions.begin();  transIter != transitions.end();  transIter++ )
		{
			string transName = (*transIter).name();
			if ( transName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) {
				badNameObjs.push_back(*transIter);
			}
		}

		set<CyPhyML::SignalFlow::Data> dataelts = (*stateIter).Data_kind_children();
		for (set<CyPhyML::SignalFlow::Data>::iterator dataIter = dataelts.begin(); dataIter != dataelts.end();  dataIter++ )
		{
			string dataName = (*dataIter).name();
			if ( dataName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) {
				badNameObjs.push_back(*dataIter);
			}				
		}

		set<CyPhyML::SignalFlow::Event> events = (*stateIter).Event_kind_children();
		for ( set<CyPhyML::SignalFlow::Event>::iterator eventIter = events.begin(); eventIter != events.end(); eventIter++ )
		{
			string eventName = (*eventIter).name();
			if ( eventName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) {
				badNameObjs.push_back(*eventIter);
			}
		}
	}

	set<CyPhyML::SignalFlow::SFState> sfstates = block.SFState_kind_children();
	for (set<CyPhyML::SignalFlow::SFState>::iterator sfstateIter = sfstates.begin(); sfstateIter != sfstates.end(); sfstateIter++ )
	{
		CyPhyML::SignalFlow::SFState sfState = *sfstateIter;
		string sfstateName = (*sfstateIter).name();
		if ( sfstateName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) {
			badNameObjs.push_back(*sfstateIter);
		}

		set<CyPhyML::SignalFlow::SFTransition> sftransitions = (*sfstateIter).SFTransition_kind_children();
		for ( set<CyPhyML::SignalFlow::SFTransition>::iterator sftransIter = sftransitions.begin();  sftransIter != sftransitions.end();  sftransIter++ )
		{
			string sftransName = (*sftransIter).name();
			if ( sftransName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) {
				badNameObjs.push_back(*sftransIter);
			}
		}

		set<CyPhyML::SignalFlow::SFData> sfdataelts = (*sfstateIter).SFData_kind_children();
		for ( set<CyPhyML::SignalFlow::SFData>::iterator sfdataIter = sfdataelts.begin(); sfdataIter != sfdataelts.end();  sfdataIter++ )
		{
			string sfdataName = (*sfdataIter).name();
			if ( sfdataName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) {
				badNameObjs.push_back(*sfdataIter);
			}
		}

		set<CyPhyML::SignalFlow::SFEvent> sfevents = (*sfstateIter).SFEvent_kind_children();
		for ( set<CyPhyML::SignalFlow::SFEvent>::iterator sfeventIter = sfevents.begin(); sfeventIter != sfevents.end();  sfeventIter++ )
		{
			string sfeventName = (*sfeventIter).name();
			if ( sfeventName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) {
				badNameObjs.push_back(*sfeventIter);
			}				
		}

		//check SFState 
		checkSFState(sfState);
	}

	set<CyPhyML::SignalFlow::SF_Parameter> parameters = block.SF_Parameter_kind_children();
	for(auto it_parameter=parameters.begin();it_parameter!=parameters.end();++it_parameter)
	{
		string parameterName = (*it_parameter).name();
		if ( parameterName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) {
			badNameObjs.push_back(*it_parameter);
		}				
	}

	set<CyPhyML::SignalFlow::SF_Port> ports = block.SF_Port_kind_children();
	for(auto it_port=ports.begin();it_port!=ports.end();++it_port)
	{
		string portName = (*it_port).name();
		if ( portName.find_first_of( " ;.,#:'\"?<>-+=*" ) != std::string::npos ) {
			badNameObjs.push_back(*it_port);
		}				
	}
}	

void CUdmApp::processComponent(CyPhyML::Component &com,  bool genmo, std::string comname)
{
	boost::filesystem::path curr_path = boost::filesystem::current_path();

	if(comname.empty())
		comname = com.name();
	
	set<CyPhyML::SignalFlowModel> sfmodels = com.SignalFlowModel_kind_children();
	if(sfmodels.empty()) return;
	for(auto sf_it=sfmodels.begin();sf_it!=sfmodels.end();++sf_it)
	{
		CyPhyML::SignalFlowModel currSF = *sf_it;
		set<CyPhyML::SignalFlowModel>::iterator cpos = checkedSFs.find(currSF);
		if(cpos!=checkedSFs.end())
			continue;
		checkedSFs.insert(currSF);
		set<CyPhyML::SignalFlow::Subsystem> subs = (*sf_it).SignalFlow_Subsystem_kind_children();
		for(auto it_sub=subs.begin();it_sub!=subs.end();++it_sub)
		{
			checkBadNames(*it_sub);
		}
		
		if ( !badNameObjs.empty() )
		{
			string outmsg( "Found the following invalid names in SignalFlow models: "+std::string(currSF.name())+"\n" );
			GMEConsole::Console::writeLine(outmsg, MSG_ERROR);
			while(!badNameObjs.empty())
			{
				Udm::Object obj = badNameObjs.front();
				badNameObjs.pop_front();

				std::string obj_type = obj.type().name();
				std::string obj_name = UdmUtil::ExtractName(obj);
				std::string err = obj_type +":\t<A HREF=\"mga:"+ UdmGme::UdmId2GmeId(obj.uniqueId()) + "\">" + obj_name + "</A>";
				GMEConsole::Console::writeLine("\t"+err, MSG_ERROR);
				printLog(obj_type+":\t"+obj_name);
			}
			return;
		}
	}

	if(genmo)
	{
		MoCodeGen mocg(com, curr_path.string(), CUdmApp::logFile);
		mocg.setPackageName(comname);
		mocg.gen();
		return;
	}
			
	std::string info ;
	std::string workDir = outputDir;
	if(workDir.empty())
	{
		boost::filesystem::create_directory(comname);
		printLog("create folder: "+comname);	
		workDir = curr_path.string() + "\\"+ comname;	
	}
	boost::filesystem::current_path(workDir);	
	boost::filesystem::create_directory("SignalFlow");
	workDir = workDir + "\\SignalFlow";
	boost::filesystem::current_path(workDir);
	printLog("create folder: SignalFlow");

	for(auto it=sfmodels.begin();it!=sfmodels.end();it++)
	{
		boost::filesystem::current_path(workDir);
		CyPhyML::SignalFlowModel sfmodel = *it;
		if(((std::string)sfmodel.URI()).empty())
		{
			sfmodel.URI() = string("SignalFlow.")+comname+string("_type");
		}
		std::string sfname = sfmodel.name();
		
		std::string sfPath = sfmodel.getPath();
	
		info = "run SF_CodeGen for SignalFlowModel: "+sfname+"...";
	//	CUdmApp::prgDlg.SetProgress(info.c_str());
		GMEConsole::Console::writeLine("run SF_CodeGen for SignalFlowModel: "+sfPath, MSG_INFO);
		
		closeLogFile();
		runCyPhySF_CodeGen_dllimported(sfmodel, (LPCTSTR)CUdmApp::mgaPath, CyPhyML::meta, CUdmApp::logFile, workDir);
	//	if(CUdmApp::prgDlg.m_cancel) return;
		
		openLogFile(CUdmApp::logFile, true);
		info = "run SL_CodeGen for SignalFlowModel: "+sfname+"...";
	//	CUdmApp::prgDlg.SetProgress(info.c_str());
		GMEConsole::Console::writeLine("run SL_CodeGen for SignalFlowModel: "+sfPath, MSG_INFO);
		runCyPhySL_CodeGen(sfmodel, (LPCTSTR)CUdmApp::mgaPath, com.__impl()->__getdn(), workDir);
	//	if(CUdmApp::prgDlg.m_cancel) return;
	}
	logGeneratedFiles(workDir);
}

void getAllComponents_have_SignalFlow(const CyPhyML::ComponentAssembly &ca, set<CyPhyML::Component> &coms)
{
	set<CyPhyML::Component> ca_coms = ca.Component_kind_children();
	for(auto it_com = ca_coms.begin(); it_com != ca_coms.end(); ++it_com)
	{
		CyPhyML::Component com = *it_com;
		set<CyPhyML::SignalFlowModel> sfmodels = com.SignalFlowModel_kind_children();
		if(!sfmodels.empty())
			coms.insert(com);
	}
	set<CyPhyML::ComponentRef> comrefs = ca.ComponentRef_kind_children();
	for(auto it_comref = comrefs.begin(); it_comref != comrefs.end(); ++it_comref)
	{
		CyPhyML::DesignElement de = (*it_comref).ref();
		if(de==Udm::null) continue;
		if(Uml::IsDerivedFrom(de.type(), CyPhyML::Component::meta))
		{
			CyPhyML::Component com = CyPhyML::Component::Cast(de);
			set<CyPhyML::SignalFlowModel> sfmodels = com.SignalFlowModel_kind_children();
			if(!sfmodels.empty())
				coms.insert(com);
		}
		else if(Uml::IsDerivedFrom(de.type(), CyPhyML::ComponentAssembly::meta))
		{
			CyPhyML::ComponentAssembly com_ca = CyPhyML::ComponentAssembly::Cast(de);
			getAllComponents_have_SignalFlow(com_ca, coms);
		}
	}
	set<CyPhyML::ComponentAssembly> cas = ca.ComponentAssembly_kind_children();
	for(auto it_ca = cas.begin(); it_ca != cas.end(); ++it_ca)
	{
		CyPhyML::ComponentAssembly curr_ca = *it_ca;
		getAllComponents_have_SignalFlow(curr_ca, coms);
	}
}

void processDesignEntity(const CyPhyML::DesignEntity &de, set<CyPhyML::Component> &coms)
{
	if(Uml::IsDerivedFrom(de.type(), CyPhyML::Component::meta))
	{
		CyPhyML::Component com = CyPhyML::Component::Cast(de);
		set<CyPhyML::SignalFlowModel> sfmodels = com.SignalFlowModel_kind_children();
		if(!sfmodels.empty())
			coms.insert(com);
	}
	else if(Uml::IsDerivedFrom(de.type(), CyPhyML::ComponentAssembly::meta))
	{
		CyPhyML::ComponentAssembly com_ca = CyPhyML::ComponentAssembly::Cast(de);
		getAllComponents_have_SignalFlow(com_ca, coms);
	}
	else if(Uml::IsDerivedFrom(de.type(), CyPhyML::ComponentRef::meta))
	{
		CyPhyML::ComponentRef comref = CyPhyML::ComponentRef::Cast(de);
		CyPhyML::DesignElement de = comref.ref();

		if(de==Udm::null) return;
		
		if(Uml::IsDerivedFrom(de.type(), CyPhyML::Component::meta))
		{
			CyPhyML::Component com = CyPhyML::Component::Cast(de);
			set<CyPhyML::SignalFlowModel> sfmodels = com.SignalFlowModel_kind_children();
			if(!sfmodels.empty())
				coms.insert(com);
		}
		else if(Uml::IsDerivedFrom(de.type(), CyPhyML::ComponentAssembly::meta))
		{
			CyPhyML::ComponentAssembly com_ca = CyPhyML::ComponentAssembly::Cast(de);
			getAllComponents_have_SignalFlow(com_ca, coms);
		}
	}
}

void getAllComponents_have_SignalFlow(const CyPhyML::TestBench &tb, set<CyPhyML::Component> &coms)
{
	set<CyPhyML::Component> tb_coms = tb.Component_kind_children();
	for(auto it_com = tb_coms.begin(); it_com!=tb_coms.end(); ++it_com)
	{
		CyPhyML::Component com = *it_com;
		set<CyPhyML::SignalFlowModel> sfmodels = com.SignalFlowModel_kind_children();
		if(!sfmodels.empty())
			coms.insert(com);
	}

	set<CyPhyML::ComponentAssembly> tb_cas = tb.ComponentAssembly_kind_children();
	for(auto it_ca = tb_cas.begin(); it_ca!=tb_cas.end(); ++it_ca)
	{
		CyPhyML::ComponentAssembly ca = *it_ca;
		getAllComponents_have_SignalFlow(ca, coms);
	}

	set<CyPhyML::ComponentRef> comrefs = tb.ComponentRef_kind_children();
	for(auto it_cr = comrefs.begin();it_cr!=comrefs.end();++it_cr)
	{
		CyPhyML::DesignElement de = (*it_cr).ref();
		if(de==Udm::null) continue;
		if(Uml::IsDerivedFrom(de.type(), CyPhyML::Component::meta))
		{
			CyPhyML::Component com = CyPhyML::Component::Cast(de);
			set<CyPhyML::SignalFlowModel> sfmodels = com.SignalFlowModel_kind_children();
			if(!sfmodels.empty())
				coms.insert(com);
		}
		else if(Uml::IsDerivedFrom(de.type(), CyPhyML::ComponentAssembly::meta))
		{
			CyPhyML::ComponentAssembly com_ca = CyPhyML::ComponentAssembly::Cast(de);
			getAllComponents_have_SignalFlow(com_ca, coms);
		}
	}

	set<CyPhyML::TestInjectionPoint> inpoints = tb.TestInjectionPoint_kind_children();
	for(auto it_ip = inpoints.begin(); it_ip!=inpoints.end(); ++it_ip)
	{
		CyPhyML::TIPRefBase refbase = (*it_ip).ref();
		if(refbase!=Udm::null && Uml::IsDerivedFrom(refbase.type(), CyPhyML::DesignEntity::meta))
		{
			CyPhyML::DesignEntity de = CyPhyML::DesignEntity::Cast(refbase);
			processDesignEntity(de, coms);	
		}
	}

	int id = tb.uniqueId();
	char tmp[20];
	OutputDebugString(itoa(id, tmp, 10));
	OutputDebugString(tb.getPath("/").c_str());
	set<CyPhyML::TopLevelSystemUnderTest> topsyss = tb.TopLevelSystemUnderTest_kind_children();
	set<Udm::Object> children = tb.GetChildObjects();
	set<string> debug1;
	std::transform(children.begin(), children.end(), std::inserter(debug1, debug1.end()), [&](const Udm::Object& o) { return o.getPath("/") + " " + itoa(o.uniqueId(), tmp, 10) + "\n"; });
	std::for_each(debug1.begin(), debug1.end(), [](const std::string& str) { OutputDebugStringA(str.c_str()); });
	for(auto it_ts = topsyss.begin();it_ts!=topsyss.end();++it_ts)
	{
		CyPhyML::DesignEntity de = (*it_ts).ref();		
		if(de!=Udm::null)
			processDesignEntity(de, coms);
	}
}

void CUdmApp::processComponentAssembly(const CyPhyML::ComponentAssembly &ca, bool genmo, std::string ca_prefix)
{
	set<CyPhyML::Component> coms;
	getAllComponents_have_SignalFlow(ca, coms);
	
	if(coms.empty()) return;
	
	std::string caname = (std::string)ca.name()+"_source";
	
	printLog("==============================================================================");
	printLog("process ComponentAssembly: "+caname+" ("+(std::string)ca.getPath2()+"):");

	boost::filesystem::current_path(CUdmApp::currPath);	
	boost::filesystem::path curr_path = createWrapperFolder(caname); 
	printLog("create folder: "+caname);
	boost::filesystem::current_path(curr_path);
	
	std::string info;

	if(!ca_prefix.empty())
		ca_prefix += "_";
	
	//set<CyPhyML::Component> coms = ca.Component_kind_children();
	for(auto it_com = coms.begin(); it_com != coms.end(); ++it_com)
	{
		boost::filesystem::current_path(curr_path);

		CyPhyML::Component com = *it_com;
		processComponent(com, genmo, ca_prefix+(std::string)com.name());
	//	if(CUdmApp::prgDlg.m_cancel) return;
	}

	if(boost::filesystem::is_empty(curr_path))
	{
		removeWrapperFolder(caname);
	}
}

void CUdmApp::processTestBench(const CyPhyML::TestBench &tb)
{
	set<CyPhyML::Component> coms;
	getAllComponents_have_SignalFlow(tb, coms);
	
	std::string tbname = tb.name();
	if(coms.empty())
	{
		printLog("TestBench: ("+(std::string)tb.getPath2()+"): has no component with SignalFlow.");
		return;
	}
	
	printLog("==============================================================================");
	printLog("process TestBench: "+tbname+" ("+(std::string)tb.getPath2()+"):");

	std::string work_dir = outputDir;
	
	if(work_dir.empty())
	{
		boost::filesystem::current_path(CUdmApp::currPath);	
		work_dir = createWrapperFolder(tbname);
	//	printLog("create folder: "+tbname);
	}
	boost::filesystem::path curr_path(work_dir);
	
	for(auto it_com = coms.begin(); it_com!=coms.end(); ++it_com)
	{
		boost::filesystem::current_path(curr_path);
		CyPhyML::Component com = *it_com;
		checkComponent(com);
		MoCodeGen mocg(com, curr_path.string(), CUdmApp::logFile);
		mocg.setPackageName((std::string)tb.name());
		mocg.gen();
	}

	if(boost::filesystem::is_empty(curr_path))
	{
		removeWrapperFolder(tbname);
	}
}

void CUdmApp::UdmMain(
					 Udm::DataNetwork* p_backend,		// Backend pointer(already open!)
					 Udm::Object focusObject,			// Focus object
					 std::set<Udm::Object> selectedObjects,	// Selected objects
					 long param)						// Parameters
{	
	if(!outputDir.empty())
	{
		if(!boost::filesystem::exists(outputDir))
		{
			std::string err = outputDir+" doesn't exist.";
			AfxMessageBox(err.c_str());
			return;
		}			
		currPath = outputDir;
		boost::filesystem::current_path(currPath);
		//boost::filesystem::create_directory("SignalFlow");
		//currPath = currPath + "\\SignalFlow";
		//outputDir = outputDir + "\\SignalFlow";
		//boost::filesystem::current_path(currPath);
	}

	//prgDlg.Create(IDD_PROGRESS_DIALOG);
	//GetCGProgressDlg(&prgDlg);
	//prgDlg.SetProgress("Preparing for code generator...");

	boost::filesystem::path path((LPCTSTR)mgaPath);
	logFile = path.stem().string()+"_cg.log";
	
	if(currPath.empty())
	{
		currPath = path.remove_filename().string();
		boost::filesystem::current_path(currPath);
		boost::filesystem::create_directory("results");
		currPath = currPath + "\\results";
	}
	boost::filesystem::current_path(currPath);
	boost::filesystem::create_directory("log");
	boost::filesystem::current_path(currPath+"\\log");
	openLogFile(logFile);
	logFile = boost::filesystem::current_path().string()+"\\"+logFile;   //full path of logfile

	boost::filesystem::current_path(currPath);
	if(outputDir.empty())
		printLog("create folder: results");
	else
	{
		boost::filesystem::create_directory("SignalFlow");
		outputDir = outputDir + "\\SignalFlow";
		currPath = currPath + "\\SignalFlow";
		boost::filesystem::current_path(currPath);
		printLog("create folder: SignalFlow");
	}

	try
	{		
		if (!master && (param == GME_SILENT_MODE || ((bool)(GetKeyState(VK_CONTROL) & 0x8000))))
		{
			CyPhyML::RootFolder root = CyPhyML::RootFolder::Cast(p_backend->GetRootObject());
		
			std::queue<CyPhyML::Components> comsQueue;
			set<CyPhyML::Components> comsFdrs = root.Components_kind_children();	
			for (auto it = comsFdrs.begin(); it != comsFdrs.end(); it++)
				comsQueue.push(*it);
		
			std::queue<CyPhyML::ComponentAssemblies> casQueue;
			set<CyPhyML::ComponentAssemblies> casFdrs = root.ComponentAssemblies_kind_children();
			for(auto it_ca=casFdrs.begin();it_ca!=casFdrs.end();it_ca++)
				casQueue.push(*it_ca);

			std::queue<CyPhyML::Testing> testingsQueue;
			set<CyPhyML::Testing> testingsFdrs = root.Testing_kind_children();
			for(auto it_tf=testingsFdrs.begin();it_tf!=testingsFdrs.end();it_tf++)
				testingsQueue.push(*it_tf);

			//iterate ComponentsFolder
			while(!comsQueue.empty())
			{
				CyPhyML::Components comsFdr = comsQueue.front();
				comsQueue.pop();
				set<CyPhyML::Components> comsFdrs_1 = comsFdr.Components_kind_children();
				for (auto it_1 = comsFdrs_1.begin(); it_1 != comsFdrs_1.end(); it_1++)
					comsQueue.push(*it_1);

				set<CyPhyML::ComponentAssemblies> casFdrs_1 = comsFdr.ComponentAssemblies_kind_children();
				for (auto it_2 = casFdrs_1.begin(); it_2 != casFdrs_1.end(); it_2++)
					casQueue.push(*it_2);

				set<CyPhyML::Component> coms = comsFdr.Component_kind_children();
				for (auto it_3 = coms.begin(); it_3 != coms.end(); it_3++)
				{
					boost::filesystem::current_path(currPath);
					CyPhyML::Component com = *it_3;
					set<CyPhyML::SignalFlowModel> sfmodels = com.SignalFlowModel_kind_children();
					if(sfmodels.empty()) continue;
					std::string info = std::string(com.name())+"...";
			//		prgDlg.SetProgress(info.c_str());
					printLog("==============================================================================");
					printLog("process Component: "+(std::string)com.name()+" ("+(std::string)com.getPath2()+"):");
					processComponent(com, false);
					//if(prgDlg.m_cancel)						
					{
						//prgDlg.OnFinished();
						closeLogFile();
						return;
					}
				}
			}


			//iterate ComponentsAssembliesFolder
			while(!casQueue.empty())
			{
				CyPhyML::ComponentAssemblies casFdr = casQueue.front();
				casQueue.pop();
				set<CyPhyML::ComponentAssemblies> casFdrs_1 = casFdr.ComponentAssemblies_kind_children();
				for (auto it_2 = casFdrs_1.begin(); it_2 != casFdrs_1.end(); it_2++)
					casQueue.push(*it_2);

				set<CyPhyML::ComponentAssembly> cas = casFdr.ComponentAssembly_kind_children();
				for(auto it_3 = cas.begin(); it_3!=cas.end(); it_3++)
				{
					boost::filesystem::current_path(currPath);
					CyPhyML::ComponentAssembly ca = *it_3;
					if(std::string(ca.ConfigurationUniqueID())!="") //check whether it is the exported design space configuration
					{
						std::string ca_name = ca.name();
						processComponentAssembly(ca, false);	
					//	if(prgDlg.m_cancel) 
						{
							//prgDlg.OnFinished();
							closeLogFile();
							return;
						}
					}
				}
			}

			//iterate TestBench
			while(!testingsQueue.empty())
			{
				CyPhyML::Testing testing = testingsQueue.front();
				testingsQueue.pop();
				set<CyPhyML::Testing> tigs = testing.Testing_kind_children();
				for(auto it_t1=tigs.begin();it_t1!=tigs.end();it_t1++)
					testingsQueue.push(*it_t1);

				set<CyPhyML::TestBench> tbs = testing.TestBench_kind_children();
				for(auto it_tb=tbs.begin();it_tb!=tbs.end();++it_tb)
				{
					CyPhyML::TestBench tb = *it_tb;
					std::string tb_name = tb.name();

					boost::filesystem::current_path(currPath);
	
					processTestBench(tb);
	
				//	if(prgDlg.m_cancel) 
					{
						//prgDlg.OnFinished();
						closeLogFile();
						return;
					}
				}
			}

			//prgDlg.OnFinished();
			boost::filesystem::current_path(path.remove_filename());
			closeLogFile();
			return;
		}
		
		if(!selectedObjects.empty() && !master)
		{
			std::string info;
			CyPhyML::SignalFlowModel sfmodel;
			for(std::set<Udm::Object>::iterator it=selectedObjects.begin();it!=selectedObjects.end();++it)
			{
				boost::filesystem::current_path(currPath);
				Udm::Object selectObj = *it;
				if(Uml::IsDerivedFrom(selectObj.type(), CyPhyML::SignalFlowModel::meta))
				{
					sfmodel = CyPhyML::SignalFlowModel::Cast(selectObj);
					std::string sfname = sfmodel.name();
					boost::filesystem::create_directory(sfname);
					
					std::string currDir = currPath + "\\" + sfname;
					boost::filesystem::current_path(currDir);
					info = "run SF_CodeGen for SignalFlowModel: "+sfname+"...";
				//	prgDlg.SetProgress(info.c_str());
					
					closeLogFile();
					runCyPhySF_CodeGen_dllimported(sfmodel, (LPCTSTR)mgaPath, CyPhyML::meta, CUdmApp::logFile, currDir);
					
					openLogFile(CUdmApp::logFile, true);
					info = "run SL_CodeGen for SignalFlowModel: "+sfname+"...";
				//	prgDlg.SetProgress(info.c_str());
					runCyPhySL_CodeGen(sfmodel, (LPCTSTR)mgaPath, p_backend, currDir);
				
					std::string log = "SL_Code has been generated for the signalflow model: "+sfname;
					AfxMessageBox(log.c_str(),MB_ICONINFORMATION);
				}
			}
			if(sfmodel==Udm::null)
				showUsage();
		}
		else if(Uml::IsDerivedFrom(focusObject.type(), CyPhyML::SignalFlowModel::meta))
		{
			CyPhyML::SignalFlowModel sfmodel = CyPhyML::SignalFlowModel::Cast(focusObject);
			std::string sfname = sfmodel.name();
			boost::filesystem::create_directory(sfname);
			std::string currDir = currPath + "\\" + sfname;
			boost::filesystem::current_path(currDir);
			
			closeLogFile();
			runCyPhySF_CodeGen_dllimported(sfmodel, (LPCTSTR)mgaPath, CyPhyML::meta, CUdmApp::logFile, currDir);
			
			openLogFile(CUdmApp::logFile, true);
			runCyPhySL_CodeGen(sfmodel, (LPCTSTR)mgaPath, p_backend, currDir);
			
			//AfxMessageBox("SL_Code has been generated for the signalflow model: "+(std::string)sfmodel.name(),MB_ICONINFORMATION);
			std::string log = "SL_Code has been generated for the signalflow model: "+(std::string)sfmodel.name();
			AfxMessageBox(log.c_str(),MB_ICONINFORMATION);
		}
		else if(Uml::IsDerivedFrom(focusObject.type(), CyPhyML::Component::meta))
		{
			CyPhyML::Component com = CyPhyML::Component::Cast(focusObject);
			set<CyPhyML::SignalFlowModel> sfmodels = com.SignalFlowModel_kind_children();
			if(!sfmodels.empty()) 
			{
				printLog("==============================================================================");
				printLog("process Component: "+(std::string)com.name()+" ("+(std::string)com.getPath2()+"):");
				processComponent(com, false);
			}
		//	prgDlg.OnFinished();
		}
		else if(Uml::IsDerivedFrom(focusObject.type(), CyPhyML::ComponentAssembly::meta))
		{
			CyPhyML::ComponentAssembly ca = CyPhyML::ComponentAssembly::Cast(focusObject);
			if(std::string(ca.ConfigurationUniqueID())!="")
				processComponentAssembly(ca, false);
			else
				AfxMessageBox("not allowed");			
		}
		else if(Uml::IsDerivedFrom(focusObject.type(), CyPhyML::TestBench::meta))
		{
			CyPhyML::TestBench tb = CyPhyML::TestBench::Cast(focusObject);
			processTestBench(tb);
		}
		else
		{
		//	prgDlg.OnFinished();
			showUsage();
			return;
		}
	//	prgDlg.OnFinished();
		boost::filesystem::current_path(path.remove_filename());
	}
	catch(udm_exception &exc)
	{
		printLog(exc.what());
		closeLogFile();
	/*	if(&prgDlg != NULL)
			prgDlg.OnFinished();		*/
		AfxMessageBox(exc.what());
		throw exc;
	}
	catch(exception &e)
	{
		printLog(e.what());
		closeLogFile();
	/*	if(&prgDlg != NULL)
			prgDlg.OnFinished();*/
		throw e;
	}
	catch(const boost::python::error_already_set& e)
	{
		std::string error = GetPythonError();
		printLog(error);
		OutputDebugStringA(error.c_str());
		cerr << error << endl;
		GMEConsole::Console::Error::writeLine(error);
	}
	catch(...)
	{
		printLog("Exception throw.");
		closeLogFile();
	/*	if(&prgDlg != NULL)
			prgDlg.OnFinished();*/
		throw;
	}	
	closeLogFile();
}