/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "runSL_CodeGen.h"

#include "SLSF.h"

#include "GenCyPhy2SLC.h"
#include "InputFileRegistry.h"
#include "SFCUdmEngine.hpp"

#include "CPrinter.hpp"
#include "JPrinter.hpp"
#include "utils.h"
//#include "transroot.hpp"
#include "ConfigKeeper.hpp"
#include "CodeGenerator.hpp"
#include "mfile2SFC.hpp"
#include <Uml.h>

//ADDED by AKSHAY AGRAWAL
#include "UdmConsole.h"

#include <boost/filesystem.hpp>

#define NOMINMAX

#include <iostream>
#include <algorithm>
#include <boost/format.hpp>
#include <boost/tokenizer.hpp>
#include <boost/filesystem.hpp>

#include "FMUCodeGen.h"
#include "CGLog.h"
//UDM_USE_DOM // COMMENTED OUT BECAUSE OF INCLUDE OF "SFCUdmEngine.hpp"

#ifdef _WIN32
UDM_USE_MGA
#endif

CyPhyML::SignalFlowModel SFUtils::rootSFModel;

typedef std::set< SLSF::Dataflow > DataflowSet;

// When this function is used as an exported .dll function by some other .dll 
// then we need to initialize the metas according to the following func.
bool __declspec(dllexport) runCyPhySL_CodeGen_dllimported(const CyPhyML::SignalFlowModel &sfmodel, std::string cyphy_file, Uml::Diagram cyphy_meta_diagram, std::string directory)
{
	UdmDom::str_xsd_storage::RemoveXsd("CyPhyML.xsd");

	LINKS::Initialize();
	CyPhyML::Initialize(cyphy_meta_diagram);
	
	// Collect Datanetwork to be passed to runCyPhy2SL_codeGen func
	Udm::DataNetwork * dn = sfmodel.__impl()->__getdn();
	// Generate Code
	return runCyPhySL_CodeGen(sfmodel, cyphy_file, dn, directory);
}

bool runCyPhySL_CodeGen(const CyPhyML::SignalFlowModel &_sfmodel, std::string cyphy_file, Udm::DataNetwork* sdn_cyphy, std::string directory) 
{
	bool ret = true;

	CyPhyML::SignalFlowModel sfmodel = _sfmodel;
	if(_sfmodel.isInstance())
		sfmodel = _sfmodel.Archetype();

	// Set the output directory
	SFUtils::rootSFModel = sfmodel;
	std::string outputDirectory = directory;
	std::string projectName;
	if(!cyphy_file.empty())
	{
		std::string inputFilename = cyphy_file;
		projectName = inputFilename.substr(  0, inputFilename.rfind( "." )  );
		{
			std::string::size_type lastSlashPos = projectName.rfind( "/" );
			std::string::size_type lastBackslashPos = projectName.rfind( "\\" );

			if ( lastSlashPos == std::string::npos ) lastSlashPos = lastBackslashPos;
			if ( lastBackslashPos == std::string::npos ) lastBackslashPos = lastSlashPos;

			std::string::size_type delimPos = std::min( lastSlashPos, lastBackslashPos );
			if ( delimPos != std::string::npos ) projectName = projectName.substr( delimPos + 1 );
		}
	}
	else
	{
		Udm::Object sfparent = sfmodel.parent();
		projectName = UdmUtil::ExtractName(sfparent);
	}
	//try {

		// Open SLSF r
		std::vector< Udm::StaticDataNetworkSpecifier> dnsvec;
		Udm::StaticDataNetworkSpecifier sdns_cyphy("", sdn_cyphy);
		dnsvec.push_back( sdns_cyphy);

		std::string sfName = sfmodel.name();
		std::string slcFilename = outputDirectory + "/" + sfName + "_SLC.xml";

		UdmDom::DomDataNetwork &sdn_sfc = SFCUdmEngine::get_singleton().getDomDataNetwork();
		sdn_sfc.CreateNew(slcFilename, "SFC.xsd", SFC::Project::meta, Udm::CHANGES_PERSIST_ALWAYS);

		Udm::StaticDataNetworkSpecifier sdns_SFC(slcFilename, &sdn_sfc);
		dnsvec.push_back( sdns_SFC);

		Udm::StaticUdmProject prj( dnsvec, CyPhy2SLC::diagram);
		Packets_t projects;
		Packets_t rootDataflows;

		SFC::Project rootSFC = SFC::Project::Cast(sdn_sfc.GetRootObject() );
		rootSFC.name() = projectName;

		projects.push_back( rootSFC);
		rootDataflows.push_back(sfmodel);

		printLog("******Start model transformation(SL_CodeGen) for SignalFlow model: "+(std::string)_sfmodel.name()+"******");
		TL_0 tL_0;
		tL_0(rootDataflows, projects);
		
		if(SFUtils::getErrorReported())
		{
			ret = false;
			printLog("******SL_CodeGen fails******");
		}
		else
		{
			printLog("******End model transformation(SL_CodeGen)******");
			boost::filesystem::current_path( outputDirectory );
			try{
				CPrinter::print( rootSFC );
			}
			catch(udm_exception &e)
			{
				printLog(e.what());
				prj.Close();
				sdn_sfc.CloseWithUpdate();
				return false;
			}
			boost::filesystem::current_path(outputDirectory);
	
			//get the top subsystem name
			set<CyPhyML::SignalFlow::Subsystem> subs = sfmodel.SignalFlow_Subsystem_kind_children();
			if(!subs.empty())
			{
				CyPhyML::SignalFlow::Subsystem topSub = *(subs.begin());
				FMUCodeGen fmu(rootSFC, (std::string)topSub.name(), outputDirectory);
				fmu.gen();
			}
		}
		//get the top class name and args, generate fmi.xml and fmu.c files

		prj.Close();
		// Close SFC w
		sdn_sfc.CloseWithUpdate();

		if ( SFUtils::getPortTypeError() ) 
		{
			if(!boost::filesystem::remove( slcFilename ))
			{
				std::string err = "boost::filesystem::remove(\""+slcFilename+"\") fails";
				throw std::exception(err.c_str());
			}
		}

		return ret;
}

void logGeneratedFiles(std::string workDir)
{
	if(workDir.empty())
		workDir = boost::filesystem::current_path().string();

	list<std::string> fmufiles;	
	if( boost::filesystem::exists(workDir) )
	{
		printLog("generated source files: ");
		boost::filesystem::directory_iterator end;
		for(boost::filesystem::directory_iterator iter(workDir) ; iter != end ; ++iter )
		{
			if ( !boost::filesystem::is_directory( *iter ) )
			{
				std::string fname = iter->path().filename().string();
				size_t pos1 = fname.find("_fmu.c");
				size_t pos2 = fname.find("_Description.xml");
				if(pos1!=std::string::npos || pos2!=std::string::npos)
					fmufiles.push_back(fname);
				else
					printLog("\t\t"+fname);
			}
		}
	}
	if(!fmufiles.empty())
	{
		printLog("generated fmu files: ");
		while(!fmufiles.empty())
		{
			printLog("\t\t"+*(fmufiles.begin()));
			fmufiles.pop_front();
		}
	}
}

ComponentCodeGen::ComponentCodeGen(CyPhyML::Component &cyphyComponent, const std::string &outputDirectory, const std::string &comname)
	:_com(cyphyComponent), _directory(outputDirectory), errorReported(false)
{
	if(comname.empty())
		_comname = _com.name();
}

void ComponentCodeGen::runCodeGen(const std::string &logFile)
{
	set<CyPhyML::SignalFlowModel> sfmodels = _com.SignalFlowModel_kind_children();
	if(sfmodels.empty()) return;

	boost::filesystem::current_path(_directory);
	boost::filesystem::create_directory(_comname);
	printLog("create folder: "+_comname);
	std::string curr_workDir = _directory + "\\"+ _comname;
	boost::filesystem::current_path(curr_workDir);

	//remove existing files
	if( boost::filesystem::exists( curr_workDir ) )
	{
		boost::filesystem::directory_iterator end;
		for(boost::filesystem::directory_iterator iter(curr_workDir) ; iter != end ; ++iter )
		{
			if ( !boost::filesystem::is_directory( *iter ) )
			{
				boost::filesystem::path fpath = iter->path();
				boost::filesystem::remove(fpath);
			}
		}
	}

	for(auto it=sfmodels.begin();it!=sfmodels.end();it++)
	{
		CyPhyML::SignalFlowModel sfmodel = *it;
		std::string sfname = sfmodel.name();

		//printLog("******Start model transformation(SF_CodeGen) for SignalFlow model: "+sfname+"******");
		closeLogFile();
		runCyPhySF_CodeGen_dllimported(sfmodel, "", CyPhyML::meta, logFile, curr_workDir);
		
		openLogFile(logFile, true);
		if(!runCyPhySL_CodeGen(sfmodel,"", sfmodel.__impl()->__getdn(), curr_workDir))
			setError();

		logGeneratedFiles(curr_workDir);

		if( boost::filesystem::exists( curr_workDir ) )
		{
			boost::filesystem::directory_iterator end;
			for(boost::filesystem::directory_iterator iter(curr_workDir) ; iter != end ; ++iter )
			{
				if ( !boost::filesystem::is_directory( *iter ) )
				{
					boost::filesystem::path fpath = iter->path();
					boost::filesystem::path fpath_ext = fpath.extension();
					std::string filepath_str = fpath_ext.string();
					if(filepath_str == ".h")
						_hfiles.insert(fpath.string());
					else if(filepath_str == ".c")
					{
						std::string fname = fpath.filename().string();
						size_t slpos = fname.find("_fmu.c");
						if(slpos==std::string::npos)
							_cfiles.insert(fpath.string());
					}
				}
			}
		}
	}
}

set<std::string> ComponentCodeGen::getGenerated_H_Files()
{
	return _hfiles;
}

set<std::string> ComponentCodeGen::getGenerated_C_Files()
{
	return _cfiles;
}