/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;
using GME.CSharp;
using GME;
using GME.MGA;
using GME.MGA.Core;
using System.Linq;

using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;


namespace CyPhy2Verification
{
    /// <summary>
    /// This class implements the necessary COM interfaces for a GME interpreter component.
    /// </summary>
    [Guid(ComponentConfig.guid),
    ProgId(ComponentConfig.progID),
    ClassInterface(ClassInterfaceType.AutoDual)]
    [ComVisible(true)]
    public class CyPhy2VerificationInterpreter : IMgaComponentEx, IGMEVersionInfo
    {
        /// <summary>
        /// Contains information about the GUI event that initiated the invocation.
        /// </summary>
        public enum ComponentStartMode
        {
            GME_MAIN_START = 0, 		// Not used by GME
            GME_BROWSER_START = 1,      // Right click in the GME Tree Browser window
            GME_CONTEXT_START = 2,		// Using the context menu by right clicking a model element in the GME modeling window
            GME_EMBEDDED_START = 3,		// Not used by GME
            GME_MENU_START = 16,		// Clicking on the toolbar icon, or using the main menu
            GME_BGCONTEXT_START = 18,	// Using the context menu by right clicking the background of the GME modeling window
            GME_ICON_START = 32,		// Not used by GME
            GME_SILENT_MODE = 128 		// Not used by GME, available to testers not using GME
        }

        /// <summary>
        /// This function is called for each interpreter invocation before Main.
        /// Don't perform MGA operations here unless you open a tansaction.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        public void Initialize(MgaProject project)
        {
            // TODO: Add your initialization code here...            
        }

        /// <summary>
        /// The main entry point of the interpreter. A transaction is already open,
        /// GMEConsole is available. A general try-catch block catches all the exceptions
        /// coming from this function, you don't need to add it. For more information, see InvokeEx.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        /// <param name="currentobj">The model open in the active tab in GME. Its value is null if no model is open (no GME modeling windows open). </param>
        /// <param name="selectedobjs">
        /// A collection for the selected model elements. It is never null.
        /// If the interpreter is invoked by the context menu of the GME Tree Browser, then the selected items in the tree browser. Folders
        /// are never passed (they are not FCOs).
        /// If the interpreter is invoked by clicking on the toolbar icon or the context menu of the modeling window, then the selected items 
        /// in the active GME modeling window. If nothing is selected, the collection is empty (contains zero elements).
        /// </param>
        /// <param name="startMode">Contains information about the GUI event that initiated the invocation.</param>
        [ComVisible(false)]
        public void Main(
            MgaProject project,
            MgaFCO currentobj,
            MgaFCOs selectedobjs,
            ComponentStartMode startMode)
        {
            if (CheckTestBenchLimitations(currentobj) == false)
            {
                return;
            }


            GMEConsole.Out.WriteLine("Verification interpreter started.");

            // check context
            if (currentobj == null)
            {
                GMEConsole.Error.WriteLine(
                    "Please open a Testbench before running Verification Interpreter");
                return;
            }

            if (currentobj.Meta.Name != "TestBench")
            {
                GMEConsole.Error.WriteLine(
                    "You must have a Testbench active to run the Verification Interpreter");
                return;
            }

            // context is a test bench
            CyPhy.TestBench tb = ISIS.GME.Common.Utils
                .CreateObject<CyPhyClasses.TestBench>(currentobj as MgaObject);

            // Checks for null values in PatternBasedRequirement attributes
            var pbrc = tb.Children.PatternBased_RequirementCollection;

            foreach (var pbr in pbrc)
            {
                double checkValue;
                var reqPC = pbr.Children.RequirementParameterCollection;
                foreach (var req in reqPC)
                {
                    if (double.TryParse(req.Attributes.Value, out checkValue) == false)
                    {
                        GMEConsole.Warning.WriteLine(
                            "{0} 'Value' field must contain a double.", req.ToHyperLink());
                        GMEConsole.Error.WriteLine("Generated files will be invalid.");
                    }
                }

                var propC = pbr.Children.PropositionCollection;
                foreach (var prop in propC)
                {
                    if (string.IsNullOrWhiteSpace(prop.Attributes.Condition))
                    {
                        GMEConsole.Warning.WriteLine(
                            "{0} 'Condition' field cannot be empty.", prop.ToHyperLink());
                        GMEConsole.Error.WriteLine("Generated files will be invalid.");
                    }
                }
            }

            // call CyPhy2Modelica interpreter
            Type t = Type.GetTypeFromProgID("MGA.Interpreter.CyPhy2Modelica");
            IMgaComponentEx cyPhy2Modelica = Activator.CreateInstance(t) as IMgaComponentEx;

            // dialog box from Modelica interpreter
            cyPhy2Modelica.Initialize(project);
            cyPhy2Modelica.InvokeEx(project, currentobj, selectedobjs, (int)startMode);

            // create a dialog window to choose QR and/or HybridSAL
            // and set Core path
            //string QrmCorePath = "";

            using (VerificationSettings vs = new VerificationSettings())
            {
                var dr1 = vs.ShowDialog();
            }

            //if (Directory.Exists(vs.txtQRMCorePath.Text))
            //{
            //    if (File.Exists(Path.Combine(vs.txtQRMCorePath.Text, "QRM.exe")))
            //    {
            //        QrmCorePath = Path.GetFullPath(vs.txtQRMCorePath.Text);
            //    }
            //    else
            //    {
            //        GMEConsole.Error.WriteLine(
            //        "{0} is invalid. Folder does not contain the appropriate files.",
            //        vs.txtQRMCorePath.Text);
            //        GMEConsole.Error.WriteLine("Generated files are invalid.");
            //    }
            //}
            //else
            //{
            //    GMEConsole.Error.WriteLine(
            //        "Path cannot be found for QRM: \"{0}\"", vs.txtQRMCorePath.Text);
            //}


            string outputDir = cyPhy2Modelica.ComponentParameter["output_dir"] as string;
            outputDir =
                string.IsNullOrEmpty(outputDir) ? Environment.CurrentDirectory : outputDir;

            string lib_package_folder = cyPhy2Modelica.ComponentParameter["lib_package_folder"] as string;
            lib_package_folder =
                string.IsNullOrEmpty(lib_package_folder) ? "" : lib_package_folder;

            string lib_package_name = cyPhy2Modelica.ComponentParameter["lib_package_name"] as string;
            lib_package_name =
                string.IsNullOrEmpty(lib_package_name) ? "" : lib_package_name;

            string driverFilename = "qrm_driver.py";

            // generate script files for QR
            using (StreamWriter writer =
                new StreamWriter(Path.Combine(outputDir, "run_QRM.cmd")))
            {
                writer.WriteLine("call \"{0}\"", META_PATH_PYTHON_ACTIVATE);
                writer.WriteLine("set QRM={0}", META_PATH_QRM_CORE);

                writer.WriteLine(@"set PATH=%PATH%;%QRM%");
                writer.WriteLine("python {0}", driverFilename);
                //writer.WriteLine(@"pause");
            }



            // library files
            Dictionary<string, string> resourceFiles =
                new Dictionary<string, string>()
                {
                    {"d3.v2.js", CyPhy2Verification.Properties.Resources.d3_v2},
                    {"PARC_QRM_Toolkit.py", CyPhy2Verification.Properties.Resources.PARC_QRM_Toolkit},
                    {"template.html", CyPhy2Verification.Properties.Resources.template},
                    {"ui.js", CyPhy2Verification.Properties.Resources.ui},
                };

            foreach (var item in resourceFiles)
            {
                using (StreamWriter writer =
                    new StreamWriter(Path.Combine(outputDir, item.Key)))
                {
                    writer.WriteLine(item.Value);
                }
            }


            // driver file
            using (StreamWriter writer =
                new StreamWriter(Path.Combine(outputDir, driverFilename)))
            {
                CyPhy2Verification.Templates.QRDriver driver = new Templates.QRDriver();
                driver.ModelPath = String.Format(
                    "{0}.{1}",
                    cyPhy2Modelica.ComponentParameter["package_name"],
                    cyPhy2Modelica.ComponentParameter["test_bench_name"]);

                if (string.IsNullOrWhiteSpace(lib_package_name) == false)
                {
                    driver.PackageNames.Add(lib_package_name);
                }

                if (string.IsNullOrWhiteSpace(lib_package_folder) == false)
                {
                    driver.PackagePaths.Add(lib_package_folder);
                }

                writer.WriteLine(driver.TransformText());
            }


            // generate script files for HybridSAL

            if (CyPhy2Verification.Properties.Settings.Default.RunQR)
            {
                var psi = new System.Diagnostics.ProcessStartInfo()
                {
                    FileName = "run_QRM.cmd",
                    WorkingDirectory = outputDir,
                };

                var proc = new System.Diagnostics.Process()
                {
                    StartInfo = psi,
                };

                proc.Start();
                proc.WaitForExit();
                if (proc.ExitCode == 0)
                {
                    GMEConsole.Info.WriteLine("Done.");
                }
                else
                {
                    GMEConsole.Error.WriteLine("Error in run.");
                }
            }

            GMEConsole.Info.WriteLine("Verification interpreter has finished.");
        }

        private bool CheckTestBenchLimitations(MgaFCO currentobj)
        {
            List<string> unsupportedKinds = new List<string>() {
                "PostProcessing"
            };

            if (currentobj.Meta.Name == "TestBench")
            {
                if (currentobj.ChildObjects.Cast<MgaFCO>().Any(x => unsupportedKinds.Contains(x.Meta.Name)))
                {
                    GMEConsole.Error.WriteLine("Unsupported elements are in the test bench such as PostProcessing.");
                    return false;
                }
            }
            return true;
        }

        #region IMgaComponentEx Members

        MgaGateway MgaGateway { get; set; }
        GMEConsole GMEConsole { get; set; }

        public string META_PATH { get; set; }
        public string META_PATH_PYTHON_ACTIVATE
        {
            get
            {
                return Path.Combine(META_PATH, "bin", "Python27", "Scripts", "activate.bat");
            }
        }

        public string META_PATH_QRM_CORE
        {
            get
            {
                return Path.Combine(META_PATH, "bin", "QRM");
            }
        }

        public void InvokeEx(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, int param)
        {
            if (!enabled)
            {
                return;
            }

            try
            {
                string keyName = @"HKEY_LOCAL_MACHINE\Software\META";
                string value = @"META_PATH";

                META_PATH = (string)Microsoft.Win32.Registry.GetValue(
                    keyName,
                    value,
                    "ERROR: " + keyName + value + " does not exist!");

                GMEConsole = GMEConsole.CreateFromProject(project);
                MgaGateway = new MgaGateway(project);
                project.CreateTerritoryWithoutSink(out MgaGateway.territory);

                MgaGateway.PerformInTransaction(delegate
                {
                    Main(project, currentobj, selectedobjs, Convert(param));
                });
            }
            finally
            {
                if (MgaGateway.territory != null)
                {
                    MgaGateway.territory.Destroy();
                }
                MgaGateway = null;
                project = null;
                currentobj = null;
                selectedobjs = null;
                GMEConsole = null;
                GC.Collect();
                GC.WaitForPendingFinalizers();
            }
        }

        private ComponentStartMode Convert(int param)
        {
            switch (param)
            {
                case (int)ComponentStartMode.GME_BGCONTEXT_START:
                    return ComponentStartMode.GME_BGCONTEXT_START;
                case (int)ComponentStartMode.GME_BROWSER_START:
                    return ComponentStartMode.GME_BROWSER_START;

                case (int)ComponentStartMode.GME_CONTEXT_START:
                    return ComponentStartMode.GME_CONTEXT_START;

                case (int)ComponentStartMode.GME_EMBEDDED_START:
                    return ComponentStartMode.GME_EMBEDDED_START;

                case (int)ComponentStartMode.GME_ICON_START:
                    return ComponentStartMode.GME_ICON_START;

                case (int)ComponentStartMode.GME_MAIN_START:
                    return ComponentStartMode.GME_MAIN_START;

                case (int)ComponentStartMode.GME_MENU_START:
                    return ComponentStartMode.GME_MENU_START;
                case (int)ComponentStartMode.GME_SILENT_MODE:
                    return ComponentStartMode.GME_SILENT_MODE;
            }

            return ComponentStartMode.GME_SILENT_MODE;
        }

        #region Component Information
        public string ComponentName
        {
            get { return GetType().Name; }
        }

        public string ComponentProgID
        {
            get
            {
                return ComponentConfig.progID;
            }
        }

        public componenttype_enum ComponentType
        {
            get { return ComponentConfig.componentType; }
        }
        public string Paradigm
        {
            get { return ComponentConfig.paradigmName; }
        }
        #endregion

        #region Enabling
        bool enabled = true;
        public void Enable(bool newval)
        {
            enabled = newval;
        }
        #endregion

        #region Interactive Mode
        protected bool interactiveMode = true;
        public bool InteractiveMode
        {
            get
            {
                return interactiveMode;
            }
            set
            {
                interactiveMode = value;
            }
        }
        #endregion

        #region Custom Parameters
        SortedDictionary<string, object> componentParameters = null;

        public object get_ComponentParameter(string Name)
        {
            if (Name == "type")
                return "csharp";

            if (Name == "path")
                return GetType().Assembly.Location;

            if (Name == "fullname")
                return GetType().FullName;

            object value;
            if (componentParameters != null && componentParameters.TryGetValue(Name, out value))
            {
                return value;
            }

            return null;
        }

        public void set_ComponentParameter(string Name, object pVal)
        {
            if (componentParameters == null)
            {
                componentParameters = new SortedDictionary<string, object>();
            }

            componentParameters[Name] = pVal;
        }
        #endregion

        #region Unused Methods
        // Old interface, it is never called for MgaComponentEx interfaces
        public void Invoke(MgaProject Project, MgaFCOs selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        // Not used by GME
        public void ObjectsInvokeEx(MgaProject Project, MgaObject currentobj, MgaObjects selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        #endregion

        #endregion

        #region IMgaVersionInfo Members

        public GMEInterfaceVersion_enum version
        {
            get { return GMEInterfaceVersion_enum.GMEInterfaceVersion_Current; }
        }

        #endregion

        #region Registration Helpers

        [ComRegisterFunctionAttribute]
        public static void GMERegister(Type t)
        {
            Registrar.RegisterComponentsInGMERegistry();

        }

        [ComUnregisterFunctionAttribute]
        public static void GMEUnRegister(Type t)
        {
            Registrar.UnregisterComponentsInGMERegistry();
        }

        #endregion


    }

    #region Extension Methods
    public static class GmeConsoleHelper
    {
        public static string ToHyperLink<T>(this T subject)
            where T : ISIS.GME.Common.Interfaces.Base
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendFormat("<a href=\"mga:{0}\">{1}</a>", subject.ID, subject.Name);
            return sb.ToString();
        }

        public static string ToMgaHyperLink<T>(this T subject)
            where T : GME.MGA.MgaFCO
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendFormat("<a href=\"mga:{0}\">{1}</a>", subject.ID, subject.Name);
            return sb.ToString();
        }
    }
    #endregion
}
