//
// USAGERIGHTS meta-x
//

// the force layout graph, global so we can look at it in the debugger
var force;

var INITIAL_COLOR = "#77B";
var NORMAL_COLOR = "#A99";
var TERMINAL_COLOR = "cyan";
var FAILURE_COLOR = "red";
var SUCCESS_COLOR = "green";

function display_legend (vis) {
    // put a legend in the upper right corner

    var items = [
        {text: "initial state", marker: {
            type: "svg:rect", width: "20", height: "12", fill: INITIAL_COLOR}},
        {text: "instant", marker: {
            type: "svg:rect", width: "20", height: "12", fill: NORMAL_COLOR}},
        {text: "interval", marker: {
            type: "svg:ellipse", width: "20", height: "12", fill: NORMAL_COLOR,
            rx: "10", ry: "6", cx: "10", cy: "6" }},
        {text: "terminal state", marker: {
            type: "svg:rect", width: "20", height: "12", fill: TERMINAL_COLOR}},
        {text: "failure state", marker: {
            type: "svg:rect", width: "20", height: "12", fill: FAILURE_COLOR}},
        {text: "success state", marker: {
            type: "svg:rect", width: "20", height: "12", fill: SUCCESS_COLOR}},
    ];
        
    vis.legend = vis.append("svg:foreignObject")
        .attr("x", vis.attr("width") - 170)
        .attr("y", 20)
        .attr("width", 150)
        .attr("height", 180);
    // legend.style("background-color", "white");
    body = vis.legend.append("xhtml:body")
        .attr("xmlns", "http://www.w3.org/1999/xhtml");

    // create a table of items
    var tr = body.append("xhtml:table")
        .style("border", "1px solid")
        .style("border-color", NORMAL_COLOR)
        .style("border-collapse", "collapse")
        .selectAll("tr")
        .data(items)
        .enter().append("xhtml:tr")
        .style("padding", "10px");

    var names = tr.append("xhtml:td")
        .style("font-size", "x-small")
        .style("font-family", "sans-serif")
        .style("color", NORMAL_COLOR)
        .each(function(d) {
            this.innerHTML = d.text;
        });
    tr.append("xhtml:td")
        .each(function(d) {
            var marker = d3.select(this).append("svg:svg")
                .attr("width", d.marker.width)
                .attr("height", d.marker.height)
                .append(d.marker.type);
            for (var key in d.marker) {
                if (key != "type") {
                    marker.attr(key, d.marker[key]);
                }
            };
        });
}

function display_graph (div, w, h, envisionment) {

    var nodes = [];
    var labelAnchors = [];
    var labelAnchorLinks = [];
    var links = [];
    var nodePositions = {};
    var conditions = [];

    var labelDistance = 0;

    function starts_with (s, prefix) {
        return (s.length >= prefix.length) && (s.substring(0, prefix.length) == prefix);
    }

    // add an SVG area of the appropriate size
    var vis;
    vis = div.append("svg:svg");
    vis.style("background-color", "black");
    if (div.property("tagName").toUpperCase() != "BODY") {
        vis.attr("width", w).attr("height", h);
    }

    // create an arrowhead for the edges
    vis.append("svg:defs").selectAll("marker")
        .data(["arrowhead"])
        .enter().append("svg:marker")
        .attr("id", String)
        .attr("viewBox", "0 -5 10 10")
        .attr("refX", 15)
        .attr("refY", -1.5)
        .attr("markerWidth", 6)
        .attr("markerHeight", 6)
        .attr("orient", "auto")
        .append("svg:path")
        .attr("d", "M0,-5L10,0L0,5")
        .style("fill", "#AAA")
        .style("stroke", "#AAA");

    // add the label
    var label = vis.append("svg:text")
        .attr("x", 40).attr("y", 30)
        .text(envisionment.model)
        .style("fill", "yellow")
        .style("font-size", "large")
        .style("font-family", "sans-serif");

    // add a legend
    display_legend(vis);

    // add a tooltip which explains the variables
    var title = '';
    for (var key in envisionment.variables) {
        title += key + ': ' + envisionment.variables[key]["type"] + '\n';
    };
    label.append("svg:title").text(title);

    vis.append("svg:text")
        .attr("x", 40).attr("y", 55)
        .text("PARC QRM tool, version "
              + envisionment.envisioner["major_version"] + "."
              + envisionment.envisioner["minor_version"] + "/"
              + envisionment.envisioner["tag"])
        .style("fill", NORMAL_COLOR)
        .style("font-size", "small")
        .style("font-family", "sans-serif");
    vis.append("svg:text")
        .attr("x", 40).attr("y", 80)
        .text(envisionment.model_generator)
        .style("fill", "#655")
        .style("font-size", "small")
        .style("font-family", "sans-serif");

    // find the conditions
    for (var vname in envisionment.variables) {
        if (starts_with(vname, "condition")) {
            conditions.push(vname);
        }
    };

    /*
    // add labels for the conditions
    var ypos = h - 5;
    conditions.forEach(function (cname) {
        vis.append("svg:text")
            .attr("x", 40)
            .attr("y", ypos)
            .text(cname + ": " + envisionment.variables[cname]["type"])
            .style("fill", NORMAL_COLOR)
            .style("font-size", "medium")
            .style("font-family", "monospace");
        ypos -= 40;
    });    
    */

    // add the situations
    envisionment.situations.forEach(function (node) {
        node.weight = 1;
	nodes.push(node);
        // keep track of the vector position of the node
        nodePositions[node.id] = nodes.length-1;
	labelAnchors.push({
            label : "<" + node.id + ">",
	    node : node
	});
	labelAnchors.push({
            label : "<" + node.id + ">",
	    node : node
	});
        // create a varsmap
        node.varsmap = {};
        node.vars.forEach(function (v) {
            node.varsmap[v.name] = v.value;
        });
    });
    
    // add the edges
    envisionment.transitions.forEach(function (edge) {
        var fromNode = nodePositions[edge.from];
        var toNode = nodePositions[edge.to];
        links.push({
            source: fromNode,
            target: toNode,
            weight: Math.random()});
    });

    for(var i = 0; i < nodes.length; i++) {
	labelAnchorLinks.push({
	    source : i * 2,
	    target : i * 2 + 1,
	    weight : 1
	});
    };

    // create the situation graph
    force = d3.layout.force()
        .size([w, h])
        .nodes(nodes)
        .links(links)
        .gravity(1)
        .linkDistance(50)
        .charge(-3000)
        .linkStrength(function(x) {
	    return x.weight * 10
        });

    // animate it
    force.start();

    // create the label overlay graph
    var force2 = d3.layout.force()
        .nodes(labelAnchors)
        .links(labelAnchorLinks)
        .gravity(1)
        .linkDistance(labelDistance)
        .linkStrength(5)
        .charge(-2000)
        .size([w, h]);

    // animate it
    force2.start();

    // add the arrowheads to the edges
    var link = vis.selectAll("line.link")
        .data(links)
        .enter()
        .append("svg:line")
        .attr("class", "link")
        .attr("marker-end", "url(#arrowhead)")
        .style("stroke", NORMAL_COLOR);

    // add the visuals for the situation

    var node = vis.selectAll("g.node")
        .data(force.nodes())
        .enter()
        .append("svg:g")
        .attr("class", "node");
    var instants = node.filter(function (d, i) {
        return (d.type == "INSTANT");
        }).append("svg:rect")
        .attr("width", 20).attr("height", 12)
        .attr("x", -10).attr("y", -6)
        .attr("class", "instant")
        .style("fill", NORMAL_COLOR);
    var intervals = node.filter(function (d, i) {
        return (d.type == "INTERVAL");
        }).append("svg:ellipse")
        .attr("ry", 6).attr("rx", 10)
        .attr("class", "interval")
        .style("fill", NORMAL_COLOR);

    // add bubbles for each condition to each node
    var xpos = -10;
    conditions.forEach(function (cname) {
        node.each(function (d, i) {
            v = eval(d.varsmap[cname]);
            if (v) {
                d3.select(this).append("svg:circle")
                    .attr("r", 3).attr("cx", xpos + 3).attr("cy", 6+3)
                    .style("stroke", NORMAL_COLOR)
                    .style("fill", "yellow");
            } else {
                d3.select(this).append("svg:circle")
                    .attr("r", 3).attr("cx", xpos + 3).attr("cy", 6+3)
                    .attr("stroke", NORMAL_COLOR)
                    .style("fill", "black");
            }
        });
        xpos += 6;
    });

    var node2 = vis.selectAll("[class=interval],[class=instant]");
    node2.filter(function (d, i) {
        return (!d.children);
    }).style("fill", TERMINAL_COLOR);
    node2.filter(function (d, i) {
        return d.initial;
    }).style("fill", INITIAL_COLOR);
    node2.filter(function (d, i) {
        return (d.status == "success");
    }).style("fill", SUCCESS_COLOR);
    node2.filter(function (d, i) {
        return (d.status == "failure");
    }).style("fill", FAILURE_COLOR);

    // add tooltips for the variable values
    node.append('svg:title').text(function (d, i) {
        var title = ''
        d.vars.forEach(function (v) {
            title += v.name + ': ' + v.value + '\n';
        });
        return title;
    });

    // bind the interactive drag behavior to the situation nodes
    // also supports touch events for iOS
    node.call(force.drag);

    // add styles for the links
    var anchorLink = vis.selectAll("line.anchorLink")
        .data(labelAnchorLinks)
        .enter()
        .append("svg:line")
        .attr("class", "anchorLink")
        .style("stroke", "#999")
        .style("stroke-dasharray", "2,2");

    // add the situation labels
    var anchorNode = vis.selectAll("g.anchorNode")
        .data(force2.nodes())
        .enter()
        .append("svg:g")
        .attr("class", "anchorNode");
    anchorNode.append("svg:circle")
        .attr("r", 1)
        .style("fill-opacity", "0.0")
        .style("fill", "#FFF");

    var anchorText = anchorNode.append("svg:text")
        .text(function(d, i) {
	    return i % 2 == 0 ? "" : d.label
        })
        .style("fill", "#966").style("font-family", "monospace");
    anchorText.append('svg:title').text(function (d, i) {
        var title = ''
        d.node.vars.forEach(function (v) {
            title += v.name + ': ' + v.value + '\n';
        });
        return title;
    });

    // define a link update function
    var updateLink = function() {
	this.attr("x1", function(d) {
	    return d.source.x;
	}).attr("y1", function(d) {
	    return d.source.y;
	}).attr("x2", function(d) {
            var xoffset = d.target.x - d.source.x;
            var yoffset = d.target.y - d.source.y;
	    return d.target.x + ((Math.abs(xoffset) > Math.abs(yoffset)) ?
                                 ((xoffset > 0) ? -10 : 10) : 0);
	}).attr("y2", function(d) {
            var xoffset = d.target.x - d.source.x;
            var yoffset = d.target.y - d.source.y;
	    return d.target.y + ((Math.abs(xoffset) <= Math.abs(yoffset)) ?
                                 ((yoffset > 0) ? -6 : 6) : 0);
	});
    }

    // define a node update function
    var updateNode = function() {
	this.attr("transform", function(d) {
            var bbox = this.getBBox();
            var x = Math.max(bbox.width, Math.min(d.x, vis.attr("width") - bbox.width));
            var y = Math.max(bbox.height, Math.min(d.y, vis.attr("height") - bbox.height));
	    return "translate(" + x + "," + y + ")";
	});
    }

    if (div.property("tagName").toUpperCase() == "BODY") {
        // catch resizes and resize SVG as well
        function resize_svg_to_match_window () {
            // this fudge factor of 30 is in there so that
            // scrollbars don't show up
            var w = window.innerWidth - 30;
            var h = window.innerHeight - 30;
            vis.attr("width", w).attr("height", h);
            // re-position the legend
            vis.legend.attr("x", w - 170);
            // re-layout the graph
            force.size([w, h]);
            vis.selectAll("g.node")
                .each(function (d, i) {
                    d.fixed = false;
                    force.start();
                });
            console.log("svg size is now " + vis.attr("width") + "x" + vis.attr("height"));
        }
        d3.select(window).on("resize", function ()
                             {resize_svg_to_match_window(vis);});
        // make the body the same color to remove borders
        div.style("background-color", vis.style("background-color"));
        resize_svg_to_match_window();
    }

    // when the graph settles down, fix the node positions
    force.on("end", function () {
        node.each(function (d, i) {
            d.fixed = true;
        });
    });                 
    // click on the label to re-animate
    label.on("click", function(d, i) {
        node.each(function (d, i) {
            d.fixed = false;
            force.resume();
        });
    });

    // what to do each animation "tick"
    force.on("tick", function() {

	force2.start();

	node.call(updateNode);

	anchorNode.each(function(d, i) {
	    if(i % 2 == 0) {
                // this is the node directly on top of the situation node
		d.x = d.node.x;
		d.y = d.node.y;
	    } else {
                // this is the offset node with the text
		var b = this.childNodes[1].getBBox();

		var diffX = d.x - d.node.x;
		var diffY = d.y - d.node.y;

		var dist = Math.sqrt(diffX * diffX + diffY * diffY);

		var shiftX = b.width * (diffX - dist) / (dist * 2);
		shiftX = Math.max(-b.width, Math.min(0, shiftX));
		var shiftY = 0;
		this.childNodes[1].setAttribute("transform", "translate(" + shiftX + "," + shiftY + ")");
	    }
	});
	anchorNode.call(updateNode);

	link.call(updateLink);
	anchorLink.call(updateLink);

    });

}

function display_envisionment (div, w, h, envisionment) {

    // split the specified div in two
    var graphdiv = d3.select(div).append("div")
        .attr("id", "graph")
        .style("position", "absolute")
        .style("left", 0)
        .style("top", 0)
        .style("width", "75%")
        .style("height", "100%");
    var controlsdiv = d3.select(div).append("div")
        .attr("id", "controls")
        .style("position", "absolute")
        .style("left", "75%")
        .style("top", 0)
        .style("width", "25%")
        .style("height", "100%");

    // save the right-hand 25% for controls
    display_graph (graphdiv, (w * 3)/4, h, envisionment);

    display_controls (controlsdiv, w/4, h, envisionment);
}

function radiobutton (div, d, label, change_callback) {
    var button = d3.select(div).append("svg:svg").style("width", "100%").style("height", "30px").style("cursor", "pointer");
    button.append("svg:circle")
        .attr("r", 8).attr("cx", 10).attr("cy", 10)
        .style("stroke", "gray")
        .style("fill", "yellow")
    var indicator = button.append("svg:circle")
        .attr("class", "indicator")
        .attr("r", 5).attr("cx", 10).attr("cy", 10)
        .style("fill", "black");
    button.append("svg:text")
        .attr("x", 25).attr("y", 17)
        .text(label)
        .style("fill", "black")
        .style("font-size", "medium")
        .style("font-family", "sans-serif");
    button.on(
        "click",
        function(d, i) {
            if (indicator.attr("visibility") == "hidden") {
                indicator.attr("visibility", "visible");
                if (change_callback) {
                    change_callback(button, true, d);
                }
            } else {
                indicator.attr("visibility", "hidden");
                if (change_callback) {
                    change_callback(button, false, d);
                }
            }
        });
    return button;
}

function display_controls (div, w, h, envisionment) {
 
    div.style("background-color", "white");

    var variables = new Array();
    for (var key in envisionment.variables) {
        n = new Array([key, envisionment.variables[key]]);
        variables.push(n);
    }

    // create a table of variables
    var tr = div.append("table").selectAll("tr")
        .data(variables)
        .enter().append("tr");

    var names = tr.selectAll("td")
        .data(function(d) {return d;})
        .enter().append("td")
        .each(function (d) {
            radiobutton(this, d, d[0], function (button, visible, d) {
                console.log(d[0] + (visible ? " is " : " is not ") + "visible");
            });
        })
        .style("font-size", "medium")
        .style("font-family", "monospace")
        .each(function(d) { this.title = d[1]; });

    console.log("tr is " + tr);
}

