/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
///////////////////////////////////////////////////////////////////////////
// RawComponent.cpp, the main RAW COM component implementation file
// This is the file (along with its header RawComponent.h)
// that the component implementor is expected to modify in the first place
//
///////////////////////////////////////////////////////////////////////////
#include "stdafx.h"

#include "ComHelp.h"
#include "GMECOM.h"
#include <ComponentConfig.h>
#include "RawComponent.h"
#include <objbase.h>
#include <climits>

void RawComponent::createStringSetForMeta_id()
{
	std::string types("DesignContainer;Component;ComponentAssembly;CyberComponent;VisualConstraint;Constraint;TestComponent;AlternativeRepresentationContainer;ComponentRef");

	while(!types.empty())
	{
		size_t pos = types.find_first_of(";");
		if(pos == std::string::npos)
		{
			fcoTypes.insert(types.c_str());
			break;
		}
		std::string ctype = types.substr(0, pos);
		fcoTypes.insert(ctype.c_str());
		types = types.substr(pos+1, types.length()-1);
	}
}

CString RawComponent::generateDCEUUID()
{
	GUID guid;
	CoCreateGuid(&guid); 
	LPOLESTR szGUID = new WCHAR [39];
	StringFromGUID2(guid, szGUID, 39);

	std::string uid("");
	for(int i=1;i<37;i++)
	{
		uid+=szGUID[i];
	}

	CString uuid("DCE:");
	uuid.Append(uid.c_str());
	return uuid;
}

CString RawComponent::getFCOMetaName(IMgaFCO *fco)
{
	CBstr bstr;	
	CComPtr<IMgaMetaFCO> meta;
	COMTHROW(fco->get_Meta(&meta));
	COMTHROW(meta->get_Name(bstr));	
	return CString(bstr);
}

bool RawComponent::isDesignElement(IMgaFCO *fco)
{
	CString metaName = getFCOMetaName(fco);
	if(metaName=="DesignContainer")
		return true;
	if(metaName=="Component" || metaName=="CyberComponent" || metaName=="ComponentAssembly" || metaName=="ComponentRef")
	{
		CComPtr<IMgaModel> parent_model;
		HRESULT _hr = fco->get_ParentModel(&parent_model);
		if( FAILED(_hr) ) return false;
		if(!parent_model) return false;
		CComQIPtr<IMgaFCO> parent_fco(parent_model);
		CString parent_metaName = getFCOMetaName(parent_fco);
		if(parent_metaName=="DesignContainer")
			return true;
	}
	else if(metaName=="VisualConstraint" || metaName=="Constraint")
		return true;
	return false;
}

void RawComponent::traverseFolder(IMgaFolder *fdr)
{
	VARIANT_BOOL isLib;
	fdr->get_IsLibObject(&isLib);
	if(isLib) return;
	CComPtr<IMgaFolders> fdrs;
	fdr->get_ChildFolders(&fdrs);
	MGACOLL_ITERATE(IMgaFolder,fdrs) 
	{	
		traverseFolder(MGACOLL_ITER);
	}
	MGACOLL_ITERATE_END;

	CComPtr<IMgaFCOs> fcos;
	COMTHROW(fdr->get_ChildFCOs(&fcos));
	MGACOLL_ITERATE(IMgaFCO,fcos) 
	{	
		CString metaName = getFCOMetaName(MGACOLL_ITER);
		traverseFCO(MGACOLL_ITER, metaName);
	}
	MGACOLL_ITERATE_END;
}

void RawComponent::traverseFCO(IMgaFCO *fco, CString metaName)
{
	if(fcoTypes.find(metaName)==fcoTypes.end()) return;

	long currid = 0;
	COMTHROW(fco->get_IntAttrByName(CBstrIn("ID"), &currid));
	
	if(currid > maxId)
	{
		maxId = currid;
	}
	else if(currid <= 0)
	{
		fcolist.push_back(fco);
	}

	CBstr bstr;
	fco->get_Name(bstr);
	CComPtr<IMgaObjects> objs;
	objtype_enum type;
	COMTHROW(fco->get_ObjType(&type));
	if(type != OBJTYPE_MODEL)
		return;
	COMTHROW(fco->get_ChildObjects(&objs));
	MGACOLL_ITERATE(IMgaObject, objs)
	{
		CComQIPtr<IMgaFCO> child_fco(MGACOLL_ITER);
		CString child_metaName = getFCOMetaName(child_fco);
		traverseFCO(child_fco, child_metaName);
	}
	MGACOLL_ITERATE_END;
}

// this method is called after all the generic initialization is done
// this should be empty, unless application-specific initialization is needed
STDMETHODIMP RawComponent::Initialize(struct IMgaProject *project) {
	
	maxId = 0;
	ids.clear();
	createStringSetForMeta_id();

	//this->project = project;
	//CComPtr<IMgaTerritory> terr;
	//CComBSTR mslProject = 0;
	//CComVariant hostGuid;
	//COMTHROW(project->CreateTerritory(NULL, &terr));
	//COMTHROW(project->BeginTransaction(terr, TRANSACTION_GENERAL));
	//try {
	//	CComPtr<IMgaFolder> rf;
	//	COMTHROW(project->get_RootFolder(&rf));
	//	
	//	//traverseDesignSpace(rf);
	//	traverseFolder(rf);

	//	for (auto fcolistIt = fcolist.begin(); fcolistIt != fcolist.end(); fcolistIt++)
	//	{
	//		CComPtr<IMgaFCO> fco = *fcolistIt;
	//		if(fco)
	//		{
	//			fco->put_IntAttrByName(CBstrIn("ID"), ++maxId);
	//		}
	//	}
	//	
	//	project->CommitTransaction();
	//} catch(HRESULT hr)	{
	//	ASSERT(hr);
	//	project->AbortTransaction();
	//	return hr;
	//}

	return S_OK;
}

// this is the obsolete component interface
// this present implementation either tries to call InvokeEx, or returns an error;
STDMETHODIMP RawComponent::Invoke(IMgaProject* gme, IMgaFCOs *models, long param) {
#ifdef SUPPORT_OLD_INVOKE
	CComPtr<IMgaFCO> focus;
	CComVariant parval = param;
	return InvokeEx(gme, focus, selected, parvar);
#else
	if(interactive) {
		AfxMessageBox("This component does not support the obsolete invoke mechanism");
	}
	return E_MGA_NOT_SUPPORTED;
#endif
}


// This is the main component method for interpereters and plugins. 
// May als be used in case of invokeable addons
STDMETHODIMP RawComponent::InvokeEx( IMgaProject *project,  IMgaFCO *currentobj,  
									IMgaFCOs *selectedobjs,  long param)
{
	COMTRY {
	  if(interactive) {
		CComBSTR projname;
		CComBSTR focusname = "<nothing>";
		CComPtr<IMgaTerritory> terr;
		COMTHROW(project->CreateTerritory(NULL, &terr));
		COMTHROW(project->BeginTransaction(terr));
		try {			
			COMTHROW(project->get_Name(&projname));
			if(currentobj) COMTHROW(currentobj->get_Name(&focusname));
			AfxMessageBox("RAW Com Component --- Plugin!!!! Sample (project: " + CString(projname) +
						", focus: " + CString(focusname));
			COMTHROW(project->CommitTransaction());
		}	catch(...) { project->AbortTransaction(); throw; }
		
	  } 
	} COMCATCH(;);
}

// GME currently does not use this function
// you only need to implement it if other invokation mechanisms are used
STDMETHODIMP RawComponent::ObjectsInvokeEx( IMgaProject *project,  IMgaObject *currentobj,  IMgaObjects *selectedobjs,  long param) {
	if(interactive) {
		AfxMessageBox("Tho ObjectsInvoke method is not implemented");
	}
	return E_MGA_NOT_SUPPORTED;
}


// implement application specific parameter-mechanism in these functions:
STDMETHODIMP RawComponent::get_ComponentParameter(BSTR name, VARIANT *pVal) {
	return S_OK;
}

bool turnedon = true;
bool dontAssignGUIDsOnNextTransaction = false;

STDMETHODIMP RawComponent::put_ComponentParameter(BSTR name, VARIANT newVal) {
	if (name != 0 && newVal.vt == VT_BOOL)
	{
		if (wcscmp(name,L"turnedon") == 0)
			turnedon = newVal.boolVal != VARIANT_FALSE;
		else if (wcscmp(name,L"dontassignguidsonnexttransaction") == 0 
					&& newVal.boolVal == VARIANT_TRUE)
		{
			dontAssignGUIDsOnNextTransaction = true;
		}
	}
	return S_OK;
}


#ifdef GME_ADDON

// these two functions are the main 
STDMETHODIMP RawComponent::GlobalEvent(globalevent_enum event) { 
	// If flag set, clear flag and skip body.
	if (event == GLOBALEVENT_COMMIT_TRANSACTION
		&& dontAssignGUIDsOnNextTransaction == true)
	{
		dontAssignGUIDsOnNextTransaction = false;
	}
	// If abort, clear flag
	if (event == GLOBALEVENT_ABORT_TRANSACTION)
	{
		dontAssignGUIDsOnNextTransaction = false;
	}

	if(event == GLOBALEVENT_UNDO) 
	{	//AfxMessageBox("UNDO!!");
	}
	if( event == APPEVENT_XML_IMPORT_BEGIN)
		turnedon = false;
	else if( event == APPEVENT_XML_IMPORT_END)
		turnedon = true;
	else if( event == APPEVENT_LIB_ATTACH_BEGIN)
		turnedon = false;
	else if( event == APPEVENT_LIB_ATTACH_END)
		turnedon = true;
	return S_OK; 
}

STDMETHODIMP RawComponent::ObjectEvent(IMgaObject * obj, unsigned long eventmask, VARIANT v) 
{
	VARIANT_BOOL isLibObject;
	COMTHROW(obj->get_IsLibObject(&isLibObject));
	COMTRY {
		if(eventmask & OBJEVENT_CREATED && isLibObject == VARIANT_FALSE)
		{
			objtype_enum type;
			COMTHROW(obj->get_ObjType(&type));
			if(type != OBJTYPE_FOLDER)	
			{
				CComQIPtr<IMgaFCO> fco(obj);
				CString fcoType = getFCOMetaName(fco);

				if(fcoTypes.find(fcoType)!=fcoTypes.end())// && isDesignElement(fco)) || fcoType == "ComponentAssembly")
				{	
					if(turnedon)
					{
						COMTHROW(fco->put_IntAttrByName(CBstrIn("ID"), ++maxId));

						if ( dontAssignGUIDsOnNextTransaction == false &&
							 ( fcoType == "Component" || fcoType == "ComponentRef" ) )
						{
							// Populate InstanceGUID field with the object's GUID
							BSTR newGUID;
							fco->GetGuidDisp(&newGUID);
							COMTHROW(fco->put_StrAttrByName(CBstrIn("InstanceGUID"), newGUID ));
						}
					}
				}
				else if(fcoType == "And_operator" || fcoType == "Or_operator" || fcoType == "DesignEntityRef")
				{
					CComQIPtr<IMgaModel> vcon;
					fco->get_ParentModel(&vcon);
					makeDirtyConfigurations_by_constraint(vcon);
				}
			}
		}
		else if((eventmask & OBJEVENT_ATTR) && turnedon && isLibObject == VARIANT_FALSE && v.vt != VT_EMPTY)
		{
			objtype_enum type;
			COMVERIFY(obj->get_ObjType(&type));
			if(type != OBJTYPE_FOLDER)	
			{
				CComQIPtr<IMgaFCO> fco(obj);
				CString metaName = getFCOMetaName(fco);
				if(metaName == "Constraint")
				{
					CComQIPtr<IMgaFCOs> fcos;
					fco->get_ReferencedBy(&fcos);
					MGACOLL_ITERATE(IMgaFCO, fcos)
					{
						CComQIPtr<IMgaModel> cfgs;
						MGACOLL_ITER->get_ParentModel(&cfgs);
						cfgs->put_BoolAttrByName((CBstrIn)"isDirty",true);
					}
					MGACOLL_ITERATE_END;		
				}
				else if(metaName == "DesignContainer" || metaName=="Component" || metaName=="ComponentAssembly"||metaName=="ComponentRef")
				{
					//if(isDesignElement(fco))
					//{
					//	makeDirtyConfigurations_by_component(fco);
					//}
					if(metaName == "DesignContainer" )
					{
						CBstr ctype;
						if ( ! SUCCEEDED( fco->get_StrAttrByName((CBstrIn)"ContainerType", ctype) ))
							return S_OK;
						CString conType(ctype);
						if(conType=="Compound")
							fco->put_RegistryValue((CBstrIn)"icon", (CBstrIn)"mandatory_ds.png");
						else if(conType=="Alternative")
							fco->put_RegistryValue((CBstrIn)"icon", (CBstrIn)"alternative_ds.png");
						else
							fco->put_RegistryValue((CBstrIn)"icon", (CBstrIn)"optional_ds.png");
					}
				}
				else if(metaName=="Configurations")
				{
					VARIANT_BOOL isDirty;
					if ( ! SUCCEEDED( fco->get_BoolAttrByName((CBstrIn)"isDirty", &isDirty) ) )
						return S_OK;
					if(isDirty)
					{
						fco->put_RegistryValue((CBstrIn)"icon", (CBstrIn)"CfgsDirty.png");
					}
					else 
					{
						fco->put_RegistryValue((CBstrIn)"icon", (CBstrIn)"CfgsClean.png");
					}
				
					VARIANT_BOOL isArchived;
					if ( ! SUCCEEDED( fco->get_BoolAttrByName((CBstrIn)"isArchived", &isArchived) ) )
						return S_OK;
					if(isArchived)
					{
						fco->put_RegistryValue((CBstrIn)"icon", (CBstrIn)"CfgsArchived.png");
					}
				}
				else if(metaName=="Property" || metaName=="Parameter")//check the parent type
				{
					CComQIPtr<IMgaModel> parent;
					fco->get_ParentModel(&parent);
					if(parent)
					{
						CString pName = getFCOMetaName(parent);
						if(pName=="Component")
							if(isDesignElement(parent))
								makeDirtyConfigurations_by_component(parent);
					}
				}
				else if(metaName = "StructuralInterface")
				{
					VARIANT_BOOL isIgnored;
					if ( ! SUCCEEDED( fco->get_BoolAttrByName((CBstrIn)"IgnoreInterface", &isIgnored) ) ) // FIXME: this attribute doesn't exist in the meta anymore
						return S_OK;
					if(isIgnored)
						fco->put_RegistryValue((CBstrIn)"icon", (CBstrIn)"struct_intf_role_ignored.png");
					else
						fco->put_RegistryValue((CBstrIn)"icon", (CBstrIn)"struct_intf_role.png");
				}
			}
		}
		else if(eventmask & OBJEVENT_PRE_DESTROYED)
		{
			objtype_enum type;
			COMVERIFY(obj->get_ObjType(&type));
			if(type != OBJTYPE_FOLDER)	
			{
				CComQIPtr<IMgaFCO> fco(obj);
				CString metaName = getFCOMetaName(fco);
				if(metaName == "Constraint" || metaName == "VisualConstraint")
				{
					makeDirtyConfigurations_by_constraint(fco);
				}
				else if(metaName == "Configurations" || metaName == "ComponentAssembly")
				{
					CComQIPtr<IMgaFCOs> fcos;
					fco->get_ReferencedBy(&fcos);
					MGACOLL_ITERATE(IMgaFCO, fcos)
					{
						MGACOLL_ITER->DestroyObject();
					}
					MGACOLL_ITERATE_END;
				}
				else if(isDesignElement(fco))
					makeDirtyConfigurations_by_component(fco);
			}
		}
		else if(((eventmask & OBJEVENT_CONNECTED) || (eventmask & OBJEVENT_DISCONNECTED)) && isLibObject == VARIANT_FALSE)
		{
			CComQIPtr<IMgaFCO> fco(obj);		
			CString fcoType = getFCOMetaName(fco);
			if(fcoType == "And_operator" || fcoType == "Or_operator" || fcoType == "DesignEntityRef")
			{
				CComQIPtr<IMgaModel> vcon;
				fco->get_ParentModel(&vcon);
				makeDirtyConfigurations_by_constraint(vcon);
			}
		}
	} COMCATCH(;)
		
	// If flag set, clear flag.
	if (eventmask & GLOBALEVENT_COMMIT_TRANSACTION
		&& dontAssignGUIDsOnNextTransaction == true)
	{
		dontAssignGUIDsOnNextTransaction = false;
	}
	// If abort, clear flag
	if (eventmask & GLOBALEVENT_ABORT_TRANSACTION)
	{
		dontAssignGUIDsOnNextTransaction = false;
	}

	return S_OK;
}

void RawComponent::makeDirtyConfigurations_by_constraint(IMgaFCO *fco)
{
	CComQIPtr<IMgaFCOs> fcos;
	fco->get_ReferencedBy(&fcos);
	MGACOLL_ITERATE(IMgaFCO, fcos)
	{
		CComQIPtr<IMgaModel> cfgs;
		MGACOLL_ITER->get_ParentModel(&cfgs);
		cfgs->put_BoolAttrByName((CBstrIn)"isDirty",true);
	}
	MGACOLL_ITERATE_END;
}

void RawComponent::makeDirtyConfigurations_by_component(IMgaFCO *fco)
{	
	CComQIPtr<IMgaModel> parent;
	if ( ! SUCCEEDED( fco->get_ParentModel(&parent) ) )
		return;
	if(!parent) return;
	CString metaName = getFCOMetaName(parent);
	if(metaName=="DesignContainer")
	{
		CComPtr<IMgaObjects> objs;
		COMTHROW(parent->get_ChildObjects(&objs));
		MGACOLL_ITERATE(IMgaObject, objs)
		{
			CComQIPtr<IMgaFCO> child_fco(MGACOLL_ITER);
			CString child_metaName = getFCOMetaName(child_fco);
			if(child_metaName=="Configurations")
			{
				child_fco->put_BoolAttrByName((CBstrIn)"isDirty",true);
				child_fco->put_RegistryValue((CBstrIn)"icon", (CBstrIn)"CfgsDirty.png");
			}
		}
		MGACOLL_ITERATE_END;

		makeDirtyConfigurations_by_component(parent);
	}
	else if(metaName=="ComponentAssembly")
	{
		makeDirtyConfigurations_by_component(parent);
	}
}

#endif