/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿namespace CyPhyCheckerLib
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Text;
    using GME.MGA;
    using System.Diagnostics.Contracts;

    using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
    using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;
    using System.Reflection;


    public class CyPhyChecker
    {
        private Dictionary<string, Func<ISIS.GME.Common.Interfaces.Base, List<Message>, bool>> m_functionMap =
            new Dictionary<string, Func<ISIS.GME.Common.Interfaces.Base, List<Message>, bool>>();

        public IMgaProject Project { get; set; }
        public CyPhy.RootFolder RootFolder { get; set; }

        public bool IsValid { get; set; }
        public bool IsChecked { get; set; }

        public Dictionary<ISIS.GME.Common.Interfaces.Base, List<Message>> Messages { get; set; }

        public List<Message> KnownMessages { get; set; }

        public CyPhyChecker(IMgaProject project)
        {
            Contract.Requires(project != null);

            Project = project;
            RootFolder = CyPhyClasses.RootFolder.GetRootFolder(Project as MgaProject);

            IsValid = false;
            IsChecked = false;
            Messages = new Dictionary<ISIS.GME.Common.Interfaces.Base, List<Message>>();
            KnownMessages = new List<Message>();

            // TODO: load messages from file
            #region ErrorMessages
            KnownMessages.Add(new Message()
            {
                Id = KnownMessages.Count,
                Name = "ObjectNameMustBeValid",
                Type = Message.MessageType.Error,
                Description = "Object's name must be defined and cannot contain special characters.",
                FullDescription = "TODO: Full desc",
                MethodName = "CheckObjectNameMustBeValid",
            });

            m_functionMap.Add("ObjectNameMustBeValid", CheckObjectNameMustBeValid);

            KnownMessages.Add(new Message()
            {
                Id = KnownMessages.Count,
                Name = "ValueMustBeRealOrBoolean",
                Type = Message.MessageType.Error,
                Description = "CyPhy Parameters and Properties must either" +
                              " be scalar reals or scalar booleans.",
                FullDescription = "TODO: Desc ValueMustBeRealOrBoolean",
                MethodName = "CheckValueMustBeRealOrBoolean"
            });

            m_functionMap.Add("ValueMustBeRealOrBoolean", CheckValueMustBeRealOrBoolean);

            KnownMessages.Add(new Message()
            {
                Id = KnownMessages.Count,
                Name = "ModelicaModelURIMustBeValid",
                Type = Message.MessageType.Error,
                Description = "",
                FullDescription = "TODO: Desc ValueMustBeModelicaCompatible",
                MethodName = "CheckModelicaModelURIMustBeValid"
            });

            m_functionMap.Add("ModelicaModelURIMustBeValid", CheckModelicaModelURIMustBeValid);

            KnownMessages.Add(new Message()
            {
                Id = KnownMessages.Count,
                Name = "NamesMustBeUnique",
                Type = Message.MessageType.Error,
                Description = "Each object in a model/folder must have a unique name.",
                FullDescription = "TODO: ",
                MethodName = "CheckNamesMustBeUnique"
            });

            m_functionMap.Add("NamesMustBeUnique", CheckNamesMustBeUnique);

            KnownMessages.Add(new Message()
            {
                Id = KnownMessages.Count,
                Name = "PortsMustBeConnected",
                Type = Message.MessageType.Error,
                Description = "Ports must be connected within the (test)component.",
                FullDescription = "TODO: ",
                MethodName = "CheckPortsMustBeConnected"
            });

            m_functionMap.Add("PortsMustBeConnected", CheckPortsMustBeConnected);

            KnownMessages.Add(new Message()
            {
                Id = KnownMessages.Count,
                Name = "ConnectionCannotBeRedundant",
                Type = Message.MessageType.Error,
                Description = "Only one connection should exist between two ports; No self-connections allowed.",
                FullDescription = "TODO: ",
                MethodName = "CheckConnectionCannotBeRedundant"
            });

            m_functionMap.Add("ConnectionCannotBeRedundant", CheckConnectionCannotBeRedundant);

            KnownMessages.Add(new Message()
            {
                Id = KnownMessages.Count,
                Name = "ReferenceCannotBeNull",
                Type = Message.MessageType.Error,
                Description = "Reference-Type objects cannot have a null reference",
                FullDescription = "TODO: ",
                MethodName = "CheckReferenceCannotBeNull"
            });

            m_functionMap.Add("ReferenceCannotBeNull", CheckReferenceCannotBeNull);
            
            #endregion

            #region WarningMessages
            KnownMessages.Add(new Message()
            {
                Id = KnownMessages.Count,
                Name = "NamesShouldBeUnique",
                Type = Message.MessageType.Warning,
                Description = "You should use unique names components/folders at the same level.",
                FullDescription = "TODO: ",
                MethodName = "CheckNamesShouldBeUnique"
            });

            m_functionMap.Add("NamesShouldBeUnique", CheckNamesShouldBeUnique);
            #endregion
        }

        public bool CheckProject()
        {
            bool result = true;

            Messages.Clear();

            // visits all objects in the entire tree
            // starting with the RootFolder CallBack function
            // is called with the domain specific classes
            RootFolder.TraverseDFS(x => x, CallBack);

            //CyPhy.TestBench tb;
            //tb.TraverseDFS(x => x, CallBack);

            IsChecked = true;
            IsValid = result;

            return result;
        }

        public void CallBack(
            ISIS.GME.Common.Interfaces.Base subject,
            int depth)
        {
            foreach (var item in KnownMessages)
            {
                _CheckObject(subject, item.Name);
            }
        }


        #region Handles for each kind to process graph recursively
        private bool _CheckObject(
            ISIS.GME.Common.Interfaces.Base subject,
            string messageName)
        {
            if (subject.Name == "THIS ONE")
            {
                System.Diagnostics.Debug.WriteLine(subject.Name);
            }

            var l = Messages.Select(x => x.Key.Name).ToList();
            l.Sort();

            var kvp = Messages.FirstOrDefault(x => x.Key.Impl == subject.Impl);
            var itemMessageList = new List<Message>();

            if (kvp.Key == null)
            {
                Messages.Add(subject, itemMessageList);
            }
            else
            {
                itemMessageList = kvp.Value;
            }

            m_functionMap[messageName](subject, itemMessageList);

            return itemMessageList.Count == 0;
        }
        #endregion

        #region Helper Functions
        private Message _GetMyMessageType(string methodname)
        {
            var templateMsg = KnownMessages.FirstOrDefault(x => x.MethodName == methodname);
            var myMsg = new Message();

            if (templateMsg == null)
            {
                myMsg = new Message()
                {
                    Id = 9999999,
                };
            }
            else
            {
                // create a Copy function for the templateMsg
                myMsg.Description = templateMsg.Description;
                myMsg.FullDescription = templateMsg.FullDescription;
                myMsg.Hint = templateMsg.Hint;
                myMsg.Id = templateMsg.Id;
                myMsg.MethodName = templateMsg.MethodName;
                myMsg.Name = templateMsg.Name;
                myMsg.Type = templateMsg.Type;
            }

            return myMsg;
        }
        #endregion

        #region Checker Functions
        public bool CheckObjectNameMustBeValid(
            ISIS.GME.Common.Interfaces.Base subject,
            List<Message> currentMessages)
        {
            string name = subject.Name;
            string suggestedName = name;
            bool result = CheckSingleNameForSpecialChars(ref suggestedName);
             
            var myMsg = _GetMyMessageType(MethodInfo.GetCurrentMethod().Name);

            myMsg.Hint = " Suggested name is " + suggestedName;

            if (result == false)
            {
                currentMessages.Add(myMsg);
            }

            return result;
        }

        public bool CheckValueMustBeRealOrBoolean(
            ISIS.GME.Common.Interfaces.Base subject,
            List<Message> currentMessages)
        {
            double d;
            bool b;
            bool result = false;

            if (subject is CyPhy.Parameter)
            {
                var parameter = subject as CyPhy.Parameter;
                if (double.TryParse(parameter.Attributes.Value, out d))
                {
                    result = true;
                }

                if (bool.TryParse(parameter.Attributes.Value, out b))
                {
                    result = true;
                }
            }
            else if (subject is CyPhy.Property)
            {
                var parameter = subject as CyPhy.Property;
                if (double.TryParse(parameter.Attributes.Value, out d))
                {
                    result = true;
                }

                if (bool.TryParse(parameter.Attributes.Value, out b))
                {
                    result = true;
                }
            }
            else
            {
                result = true;
            }
            var myMsg = _GetMyMessageType(MethodInfo.GetCurrentMethod().Name);

            if (result == false)
            {
                currentMessages.Add(myMsg);
            }

            return result;
        }

        public bool CheckModelicaModelURIMustBeValid(
            ISIS.GME.Common.Interfaces.Base subject,
            List<Message> currentMessages)
        {
            bool result = true;
            string suggestedName = "";
            if (subject is CyPhy.ModelicaModelType)
            {
                var modelicaModel = subject as CyPhy.ModelicaModelType;
                var URI = modelicaModel.Attributes.URI;
                if (string.IsNullOrWhiteSpace(URI) == false)
                {
                    string[] namePieces = URI.Split('.');
                    foreach (var namePiece in namePieces)
                    {
                        string suggestedNamePiece = namePiece;
                        bool properName = CheckSingleNameForSpecialChars(
                            ref suggestedNamePiece);
                        suggestedName = suggestedName + "." + suggestedNamePiece;
                        if (properName == false)
                        {
                            result = false;
                        }
                    }
                    suggestedName = suggestedName.TrimStart('.');
                }
                else
                {
                    suggestedName = "myPackage.mySubPackage.myModel";
                    result = false;
                }
            }

            var myMsg = _GetMyMessageType(MethodInfo.GetCurrentMethod().Name);
            
            myMsg.Hint = " Suggested name is " + suggestedName;

            if (result == false)
            {
                currentMessages.Add(myMsg);
            }

            return result;
        }

        public bool CheckNamesMustBeUnique(
            ISIS.GME.Common.Interfaces.Base subject,
            List<Message> currentMessages)
        {
            bool result = true;
            int nameCount = 0;
            int distCount = 0;
            if(subject is CyPhy.ModelicaModelType)
            {
                var modelicaModel = subject as CyPhy.ModelicaModelType;

                List<string> names = new List<string>();
                foreach (var item in modelicaModel.AllChildren
                    .Where(x => x is ISIS.GME.Common.Interfaces.Connection == false))
                {
                    names.Add(item.Name);
                }
                nameCount = names.Count;
                distCount = names.Distinct().Count();
                if (nameCount != distCount)
                {
                    result = false;
                }
            }
            else if (subject is CyPhy.ComponentType)
            {
                var componentType = subject as CyPhy.ComponentType;

                List<string> names = new List<string>();
                foreach (var item in componentType.AllChildren
                    .Where(x => x is ISIS.GME.Common.Interfaces.Connection == false))
                {
                    names.Add(item.Name);
                }
                nameCount = names.Count;
                distCount = names.Distinct().Count();
                if (nameCount != distCount)
                {
                    
                    result = false;
                }
            }

            var myMsg = _GetMyMessageType(MethodInfo.GetCurrentMethod().Name);
            
            myMsg.Hint = " You only have " + distCount.ToString() + " unique names, for " +
                            nameCount.ToString() + " different objects.";

            if (result == false)
            {
                currentMessages.Add(myMsg);
            }

            return result;
        }

        public bool CheckNamesShouldBeUnique(
            ISIS.GME.Common.Interfaces.Base subject,
            List<Message> currentMessages)
        {
            bool result = true;
            int nameCount = 0;
            int distCount = 0;
            if (subject is CyPhy.Components)
            {
                var componentFolder = subject as CyPhy.Components;

                List<string> names = new List<string>();
                foreach (var item in componentFolder.AllChildren)
                {
                    names.Add(item.Name);
                }
                nameCount = names.Count;
                distCount = names.Distinct().Count();
                if (nameCount != distCount)
                {
                    result = false;
                }
            }
            else if (subject is CyPhy.Testing)
            {
                var testingFolder = subject as CyPhy.Testing;

                List<string> names = new List<string>();
                foreach (var item in testingFolder.AllChildren)
                {
                    names.Add(item.Name);
                }
                nameCount = names.Count;
                distCount = names.Distinct().Count();
                if (nameCount != distCount)
                {
                    result = false;
                }
            }
            else if (subject is CyPhy.TestComponents)
            {
                var testComponentsFolder = subject as CyPhy.TestComponents;
               
                List<string> names = new List<string>();
                foreach (var item in testComponentsFolder.AllChildren)
                {
                    names.Add(item.Name);
                }
                nameCount = names.Count;
                distCount = names.Distinct().Count();
                if (nameCount != distCount)
                {
                    result = false;
                }
            }

            var myMsg = _GetMyMessageType(MethodInfo.GetCurrentMethod().Name);

            myMsg.Hint = " You only have " + distCount.ToString() + " unique names, for " +
                            nameCount.ToString() + " different objects.";

            if (result == false)
            {
                currentMessages.Add(myMsg);
            }

            return result;
        }

        public bool CheckPortsMustBeConnected(
            ISIS.GME.Common.Interfaces.Base subject,
            List<Message> currentMessages)
        {
            bool result = true;

            var myMsg = _GetMyMessageType(MethodInfo.GetCurrentMethod().Name);
            myMsg.Hint = "";

            List<CyPhy.Port> DisconnectedPorts = new List<CyPhy.Port>();

            if (subject is CyPhy.ComponentType)
            {
                var component = subject as CyPhy.ComponentType;
                var compPorts = component.Children.PortCollection;

                // This loop verifies that each port in a component has at least one connection within that component
                foreach (var port in compPorts)
                {
                    bool hasOneConnection = false;
                    var parentContainer = port.ParentContainer;
                    var portParName = parentContainer.Name;

                    // TODO: add ModelicaPort2ComponentPortPowerFlow to 
                    // "AllSrcConnections" and "AllDstConnections" ??
                    // OR add code to detect this connection somehow...
                    var connections = port.AllDstConnections.Concat(port.AllSrcConnections);
                    // or alternatively:
                    // var connList = port.AllDstConnections.ToList();
                    // connList.AddRange(port.AllSrcConnections.ToList());

                    if (port is CyPhy.ElectricalPowerPort)
                    {
                        var connectikizoinks = 
                            (port as CyPhy.ElectricalPowerPort).AllDstConnections.Concat(
                            (port as CyPhy.ElectricalPowerPort).AllSrcConnections);
                    }

                    foreach (var conn in connections)
                    {
                        var connParName = conn.ParentContainer.Name;
                        if (connParName == portParName)
                        {
                            hasOneConnection = true;
                            break;
                        }
                    }

                    if (port is CyPhy.ElectricalPowerPort)
                    {
                        var connectikizoinks =
                            (port as CyPhy.ElectricalPowerPort).
                                SrcConnections.ModelicaPort2ComponentPortPowerFlowCollection;

                        var modPort2CompPortPwrFlow_1 = connectikizoinks.FirstOrDefault();

                        foreach (var conn in connectikizoinks)
                        {
                            var connParName = conn.ParentContainer.Name;
                            if (connParName == portParName)
                            {
                                hasOneConnection = true;
                                break;
                            }
                        }
                    }
                    
                    if (hasOneConnection == false)
                    {
                        DisconnectedPorts.Add(port);
                        result = false;
                    }
                }
            }

            else if (subject is CyPhy.ComponentAssembly)
            {
                var assembly = subject as CyPhy.ComponentAssembly;
                var assemblyPorts = assembly.Children.PortCollection;

                // This loop verifies that each port in a component has at least one connection within that component
                foreach (var port in assemblyPorts)
                {
                    bool hasOneConnection = false;
                    var portParName = port.ParentContainer.Name;
                    var connections = port.AllDstConnections.Concat(port.AllSrcConnections);
                    foreach (var conn in connections)
                    {
                        var connParName = conn.ParentContainer.Name;
                        if (connParName == portParName)
                        {
                            hasOneConnection = true;
                            break;
                        }
                    }
                    if (hasOneConnection == false)
                    {
                        DisconnectedPorts.Add(port);
                        result = false;
                    }
                }

            }

            myMsg.Hint = myMsg.Hint + "ProblemPorts: ";
            foreach (var port in DisconnectedPorts)
            {
                myMsg.Hint = myMsg.Hint + " '" + port.Name + "' ";
                myMsg.Hint = myMsg.Hint +
                    " Checker cannot detect 'ModelicaPort2ComponentPortPowerFlow'... This port might be OK. ";

            }

            if (result == false)
            {
                currentMessages.Add(myMsg);
            }

            return result;
        }

        public bool CheckConnectionCannotBeRedundant(
            ISIS.GME.Common.Interfaces.Base subject,
            List<Message> currentMessages)
        {
            bool result = true;

            var myMsg = _GetMyMessageType(MethodInfo.GetCurrentMethod().Name);
            myMsg.Hint = "";

            List<ISIS.GME.Common.Interfaces.Connection> redundantConnections = 
                new List<ISIS.GME.Common.Interfaces.Connection>();
            List<ISIS.GME.Common.Interfaces.Connection> selfConnections =
                new List<ISIS.GME.Common.Interfaces.Connection>();

            List<string> redundantConns = new List<string>(); 
            List<string> selfConns = new List<string>();

            if (subject is CyPhy.Port)
            {
                var port = subject as CyPhy.Port;
                var connections = port.AllDstConnections.Concat(
                    port.AllSrcConnections);
                List<string> otherTerminal = new List<string>();
                int redundantCount = 0;
                int redundantCountNew;
                foreach (var conn in connections)
                {
                    var destPort = conn.GenericDstEnd;
                    var srcPort = conn.GenericSrcEnd;
                    if (destPort.ID == srcPort.ID)
                    {
                        result = false;
                        if (selfConnections.Contains(conn) == false)
                        {
                            selfConnections.Add(conn);
                        }
                    }
                    else if (port.ID == destPort.ID)
                    {
                        otherTerminal.Add(srcPort.ID.ToString());
                    }
                    else if (port.ID == srcPort.ID)
                    {
                        otherTerminal.Add(destPort.ID.ToString());
                    }
                    redundantCountNew = otherTerminal.Count() - otherTerminal.Distinct().Count();
                    if (redundantCountNew != redundantCount)
                    {
                        result = false;
                        redundantCount = redundantCountNew;
                        redundantConnections.Add(conn);
                    }
                }
            }

            if (result == false)
            {
                if (selfConnections.Count() != 0)
                {
                    myMsg.Hint = myMsg.Hint + "Self-connections: ";
                    foreach (var conn in selfConnections)
                    {
                        myMsg.Hint = myMsg.Hint + "'" + conn.ID.ToString() + "' ";
                    }
                    if (redundantConnections.Count() !=0)
                    {
                        myMsg.Hint = myMsg.Hint + "|__| ";
                    }
                }

                if (redundantConnections.Count() != 0)
                {
                    myMsg.Hint = myMsg.Hint + "Redundant connections: ";
                    foreach (var conn in redundantConnections)
                    {
                        myMsg.Hint = myMsg.Hint + "'" + conn.ID.ToString() + "' ";
                    }
                }
                currentMessages.Add(myMsg);
            }

            return result;
        }

        
        public bool CheckReferenceCannotBeNull(
            ISIS.GME.Common.Interfaces.Base subject,
            List<Message> currentMessages)
        {
            bool result = true;

            var myMsg = _GetMyMessageType(MethodInfo.GetCurrentMethod().Name);
            myMsg.Hint = "";

            if (subject is CyPhy.Parameter)
            {
                var param = subject as CyPhy.Parameter;
                if (param.AllReferred == null)
                {
                    myMsg.Hint = myMsg.Hint + "Parameter '" + param.Name + "' must reference an SI unit.";
                    result = false;
                }
            }

            if (subject is CyPhy.Property)
            {
                var prop = subject as CyPhy.Property;
                if (prop.AllReferred == null)
                {
                    myMsg.Hint = myMsg.Hint + "Property '" + prop.Name + "' must reference an SI unit.";
                    result = false;
                }
            }

            if (subject is CyPhy.Metric)
            {
                var metric = subject as CyPhy.Metric;
                if (metric.AllReferred == null)
                {
                    myMsg.Hint = myMsg.Hint + "Metric '" + metric.Name + "' must reference an SI unit.";
                    result = false;
                }
            }

            if (subject is CyPhy.TestBenchRef)
            {
                var tb = subject as CyPhy.TestBenchRef;
                if (tb.Referred.TestBench == null)
                {
                    result = false;
                    myMsg.Hint = myMsg.Hint + "'" + tb.Name + "' must point to an existing testbench.";
                }
            }

            if (result == false)
            {
                currentMessages.Add(myMsg);
            }

            return result;
        }

        #region Helper functions for Checker Functions

        public bool CheckSingleNameForSpecialChars(ref string suggestedName)
        {
            bool result = true;
            List<char> validChars = new List<char>()
            {
                'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
                'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
                'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
                'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
                '_'
            };

            List<char> numbers = new List<char>()
            {
                '0', '1', '2', '3', '4', '5', '6', '7', '8', '9'
            };

            if (string.IsNullOrWhiteSpace(suggestedName) == false)
            {
                char[] nameChars = suggestedName.ToCharArray();
                var wrongChars = nameChars.Where(c =>
                    validChars.Contains(c) == false &&
                    numbers.Contains(c) == false);

                foreach (var item in wrongChars)
                {
                    suggestedName = suggestedName.Replace(item, '_');
                    result = false;
                }

                if (numbers.Contains(nameChars.First()))
                {
                    suggestedName = "_" + suggestedName;
                    result = false;
                }
            }
            else
            {
                suggestedName = "someName";
                result = false;
            }

            return result;
        }
        #endregion

        #endregion




    }
}
