/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;
using GME.CSharp;
using GME;
using GME.MGA;
using GME.MGA.Core;
using System.Windows.Forms;
using CyPhyML = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyMLClasses = ISIS.GME.Dsml.CyPhyML.Classes;
using AVM;
using CyPhy2ComponentModel;

using JsonSchema = Newtonsoft.Json.Schema.JsonSchema;
using Newtonsoft.Json;
using System.Threading.Tasks;
using System.Collections.Concurrent;

namespace CyPhyComponentExporterCL {

    public class CyPhyComponentExporterCL {

        private static MgaProject GetProject( String filename ) {
            MgaProject result = null;

            if( filename != null && filename != "" ) {
                if( Path.GetExtension( filename ) == ".mga" ) {
                    result = new MgaProject();
                    if( System.IO.File.Exists( filename ) ) {
                        Console.Out.Write( "Opening {0} ... ", filename );
                        bool ro_mode = true;
                        result.Open( "MGA=" + filename, out ro_mode );
                        Console.Out.WriteLine( "Done." );
                    } else {
                        Console.Error.WriteLine( "{0} file must be an existing mga project.", filename );
                    }
                } else {
                    Console.Error.WriteLine( "{0} file must be an mga project.", filename );
                }
            } else {
                Console.Error.WriteLine( "Please specify an Mga project." );
            }

            return result;
        }

        private static List<CyPhyML.Component> getCyPhyMLComponentList( CyPhyML.RootFolder cyPhyMLRootFolder ) {

            List<CyPhyML.Component> cyPhyMLComponentList = new List<CyPhyML.Component>();

            foreach( CyPhyML.Components childComponentsFolder in cyPhyMLRootFolder.Children.ComponentsCollection ) {
                cyPhyMLComponentList.AddRange( getCyPhyMLComponentList( childComponentsFolder ) );
            }

            foreach( CyPhyML.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLRootFolder.Children.ComponentAssembliesCollection ) {
                cyPhyMLComponentList.AddRange( getCyPhyMLComponentList( childComponentAssembliesFolder ) );
            }

            return cyPhyMLComponentList;

        }

        private static List<CyPhyML.Component> getCyPhyMLComponentList( CyPhyML.Components cyPhyMLComponents ) {

            List<CyPhyML.Component> cyPhyMLComponentList = new List<CyPhyML.Component>();

            foreach( CyPhyML.Components childComponentsFolder in cyPhyMLComponents.Children.ComponentsCollection ) {
                cyPhyMLComponentList.AddRange( getCyPhyMLComponentList( childComponentsFolder ) );
            }

            foreach( CyPhyML.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLComponents.Children.ComponentAssembliesCollection ) {
                cyPhyMLComponentList.AddRange( getCyPhyMLComponentList( childComponentAssembliesFolder ) );
            }

            foreach( CyPhyML.Component childComponent in cyPhyMLComponents.Children.ComponentCollection ) {
                cyPhyMLComponentList.Add( childComponent );
            }

            return cyPhyMLComponentList;
        }

        private static List<CyPhyML.Component> getCyPhyMLComponentList( CyPhyML.ComponentAssemblies cyPhyMLComponentAssemblies ) {

            List<CyPhyML.Component> cyPhyMLComponentList = new List<CyPhyML.Component>();

            foreach( CyPhyML.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLComponentAssemblies.Children.ComponentAssembliesCollection ) {
                cyPhyMLComponentList.AddRange( getCyPhyMLComponentList( childComponentAssembliesFolder ) );
            }

            foreach( CyPhyML.ComponentAssembly childComponentAssemblyFolder in cyPhyMLComponentAssemblies.Children.ComponentAssemblyCollection ) {
                cyPhyMLComponentList.AddRange( getCyPhyMLComponentList( childComponentAssemblyFolder ) );
            }

            return cyPhyMLComponentList;
        }

        private static List<CyPhyML.Component> getCyPhyMLComponentList( CyPhyML.ComponentAssembly cyPhyMLComponentAssembly ) {

            List<CyPhyML.Component> cyPhyMLComponentList = new List<CyPhyML.Component>();

            foreach( CyPhyML.ComponentAssembly childComponentAssemblyFolder in cyPhyMLComponentAssembly.Children.ComponentAssemblyCollection ) {
                cyPhyMLComponentList.AddRange( getCyPhyMLComponentList( childComponentAssemblyFolder ) );
            }

            foreach( CyPhyML.Component childComponent in cyPhyMLComponentAssembly.Children.ComponentCollection ) {
                cyPhyMLComponentList.Add( childComponent );
            }

            return cyPhyMLComponentList;
        }

        private static List<CyPhyML.Component> getCyPhyMLComponentList( CyPhyML.Component cyPhyMLComponent ) {
            List<CyPhyML.Component> cyPhyMLComponentList = new List<CyPhyML.Component>();
            cyPhyMLComponentList.Add( cyPhyMLComponent );
            return cyPhyMLComponentList;
        }

        private static String Safeify( String s_in ) {
            String rtn = s_in;
            rtn = rtn.Replace( "\\", "_" );
            rtn = rtn.Replace( "/", "_" );
            return rtn;
        }

        public static void Main( String[] args ) {

            if( args.Length < 1 ) {
                Console.Out.WriteLine( "Usage: <program> CyPhyMLFile.mga [-f] [outputDirectory]" );
                Environment.Exit( 1 );
            }

            MgaProject mgaProject = null;

            string outputDirectory = ".";
            bool flat = false;

            String mgaFilePath = null;
            for( int ix = 0; ix < args.Length; ++ix ) {
                if( args[ ix ] == "-f" ) {
                    flat = true;
                } else if ( mgaProject == null ) {
                    mgaProject = GetProject( args[ ix ] );
                    mgaFilePath = args[ix];
                } else {
                    outputDirectory = args[ ix ];
                }
            }

            if ( mgaProject == null ) {
                Console.Out.WriteLine( "Usage: <program> CyPhyMLFile.mga [-f] [outputDirectory]" );
                Environment.Exit( 1 );
            }

            MgaGateway mgaGateway = new MgaGateway( mgaProject );
            mgaProject.CreateTerritoryWithoutSink( out mgaGateway.territory );

            mgaGateway.PerformInTransaction( delegate {
                try {
                    CyPhyML.RootFolder cyPhyMLRootFolder = ISIS.GME.Common.Utils.CreateObject<CyPhyMLClasses.RootFolder>( mgaProject.RootFolder as MgaObject );
                    List<CyPhyML.Component> cyPhyMLComponentList = getCyPhyMLComponentList( cyPhyMLRootFolder );

                    ParallelOptions options = new ParallelOptions();
                    options.MaxDegreeOfParallelism = 4;

                    int i_Counter = 1;
                    Parallel.ForEach(cyPhyMLComponentList, options, cyPhyMLComponent =>
                        {
                            try
                            {
                                //foreach( CyPhyML.Component cyPhyMLComponent in cyPhyMLComponentList ) {
                                AVM.Component avmComponent = CyPhy2ComponentModel.Convert.CyPhyML2AVMComponent(cyPhyMLComponent);
                                string componentPath = null;

                                if (flat)
                                {
                                    componentPath = outputDirectory;
                                }
                                else
                                {
                                    componentPath = cyPhyMLComponent.Path;
                                    componentPath = outputDirectory + "/" + componentPath.Substring(componentPath.IndexOf("/"));
                                }

                                Directory.CreateDirectory(componentPath);

                                if (!flat)
                                {
                                    Directory.CreateDirectory(componentPath + "/images");
                                    //Directory.CreateDirectory( componentPath + "/CyPhyML" );
                                    Directory.CreateDirectory(componentPath + "/doc");
                                    Directory.CreateDirectory(componentPath + "/CAD");
                                }

                                String s_outFilePath = String.Format("{0}/{1}.component.json", componentPath, Safeify(cyPhyMLComponent.Name));
                                avmComponent.SerializeToFile(s_outFilePath);
                                
                                Console.Out.WriteLine(string.Format("({0}/{1}) {2}", i_Counter++, cyPhyMLComponentList.Count, Safeify(cyPhyMLComponent.Name)));
                            } catch( Exception ex ) {
                                Console.Error.WriteLine( "Exception: {0}", ex.Message.ToString() );
                                Console.Error.WriteLine( "Stack: {0}", ex.StackTrace.ToString() );
                            }
                        }
                    );
                } catch( Exception ex ) {
                    Console.Error.WriteLine( "Exception: {0}", ex.Message.ToString() );
                    Console.Error.WriteLine( "Stack: {0}", ex.StackTrace.ToString() );
                }
            } );

            mgaProject.Close();

            //try {
            //    if( mgaGateway.territory != null ) {
            //        mgaGateway.territory.Destroy();
            //    }
            //} catch( Exception ex ) {}

        }

    }
}
