/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;
using System.Diagnostics;
using GME.CSharp;
using GME;
using GME.MGA;
using GME.MGA.Core;
using System.Windows.Forms;
using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyClasses = ISIS.GME.Dsml.CyPhyML.Classes;
using AVM;
using CyPhy2ComponentModel;
using Microsoft.Win32;
using System.Reflection;
using Newtonsoft.Json;

namespace CyPhyComponentImporter
{
    /// <summary>
    /// This class implements the necessary COM interfaces for a GME interpreter component.
    /// </summary>
    [Guid(ComponentConfig.guid),
    ProgId(ComponentConfig.progID),
    ClassInterface(ClassInterfaceType.AutoDual)]
    [ComVisible(true)]
    public class CyPhyComponentImporterInterpreter : IMgaComponentEx, IGMEVersionInfo
    {
        private static string token = "";

        /// <summary>
        /// Contains information about the GUI event that initiated the invocation.
        /// </summary>
        public enum ComponentStartMode
        {
            GME_MAIN_START = 0, 		// Not used by GME
            GME_BROWSER_START = 1,      // Right click in the GME Tree Browser window
            GME_CONTEXT_START = 2,		// Using the context menu by right clicking a model element in the GME modeling window
            GME_EMBEDDED_START = 3,		// Not used by GME
            GME_MENU_START = 16,		// Clicking on the toolbar icon, or using the main menu
            GME_BGCONTEXT_START = 18,	// Using the context menu by right clicking the background of the GME modeling window
            GME_ICON_START = 32,		// Not used by GME
            GME_SILENT_MODE = 128 		// Not used by GME, available to testers not using GME
        }

        private static void WriteLine( Func<string, string, string> f, IMgaFCO a, IMgaFCO b ) {
            //if (GMEConsole != null)
            //{
            //    GMEConsole.Out.WriteLine(f(GetLink(a, a.Name), GetLink(b, b.Name)));
            //}
            //else
            {
                Console.Out.WriteLine(f(a.AbsPath, b.AbsPath));
            }
        }

        /// <summary>
        /// This function is called for each interpreter invocation before Main.
        /// Don't perform MGA operations here unless you open a tansaction.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        public void Initialize(MgaProject project)
        {
            // TODO: Add your initialization code here...            
        }

        private CyPhy.Components GetImportFolder(CyPhy.RootFolder rf) {
            CyPhy.Components rtn = null;
            foreach (CyPhy.Components c in rf.Children.ComponentsCollection)
            {
                if (c.Name == "Imported Components")
                {
                    rtn = c;
                }
            }

            if (rtn == null)
            {
                rtn = CyPhyClasses.Components.Create(rf);
                rtn.Name = "Imported Components";
            }

            return rtn;
        }

        private static Dictionary<string, CyPhy.Component> getCyPhyMLComponentDictionary_ByAVMID( CyPhy.RootFolder cyPhyMLRootFolder ) {

            Dictionary<string, CyPhy.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhy.Component>();

            foreach( CyPhy.Components childComponentsFolder in cyPhyMLRootFolder.Children.ComponentsCollection ) {
                foreach( KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID( childComponentsFolder ) ) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach( CyPhy.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLRootFolder.Children.ComponentAssembliesCollection ) {
                foreach( KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID( childComponentAssembliesFolder ) ) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            return cyPhyMLComponentDictionary;

        }

        private static Dictionary<string, CyPhy.Component> getCyPhyMLComponentDictionary_ByAVMID( CyPhy.Components cyPhyMLComponents ) {

            Dictionary<string, CyPhy.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhy.Component>();

            foreach( CyPhy.Components childComponentsFolder in cyPhyMLComponents.Children.ComponentsCollection ) {
                foreach( KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID( childComponentsFolder ) ) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach( CyPhy.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLComponents.Children.ComponentAssembliesCollection ) {
                foreach( KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID( childComponentAssembliesFolder ) ) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach( CyPhy.Component childComponent in cyPhyMLComponents.Children.ComponentCollection ) {
                cyPhyMLComponentDictionary[childComponent.Attributes.AVMID] = childComponent;
            }

            return cyPhyMLComponentDictionary;
        }

        private static Dictionary<string, CyPhy.Component> getCyPhyMLComponentDictionary_ByAVMID( CyPhy.ComponentAssemblies cyPhyMLComponentAssemblies ) {

            Dictionary<string, CyPhy.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhy.Component>();

            foreach( CyPhy.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLComponentAssemblies.Children.ComponentAssembliesCollection ) {
                foreach( KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID( childComponentAssembliesFolder ) ) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach( CyPhy.ComponentAssembly childComponentAssemblyFolder in cyPhyMLComponentAssemblies.Children.ComponentAssemblyCollection ) {
                foreach( KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID( childComponentAssemblyFolder ) ) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            return cyPhyMLComponentDictionary;
        }

        private static Dictionary<string, CyPhy.Component> getCyPhyMLComponentDictionary_ByAVMID( CyPhy.ComponentAssembly cyPhyMLComponentAssembly ) {

            Dictionary<string, CyPhy.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhy.Component>();

            foreach( CyPhy.ComponentAssembly childComponentAssemblyFolder in cyPhyMLComponentAssembly.Children.ComponentAssemblyCollection ) {
                foreach( KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID( childComponentAssemblyFolder ) ) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach( CyPhy.Component childComponent in cyPhyMLComponentAssembly.Children.ComponentCollection ) {
                cyPhyMLComponentDictionary[childComponent.Attributes.AVMID] = childComponent;
            }

            return cyPhyMLComponentDictionary;
        }


        private static Dictionary<string, CyPhy.Component> getCyPhyMLComponentDictionary_ByName(CyPhy.RootFolder cyPhyMLRootFolder)
        {

            Dictionary<string, CyPhy.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhy.Component>();

            foreach (CyPhy.Components childComponentsFolder in cyPhyMLRootFolder.Children.ComponentsCollection)
            {
                foreach (KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByName(childComponentsFolder))
                {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach (CyPhy.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLRootFolder.Children.ComponentAssembliesCollection)
            {
                foreach (KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByName(childComponentAssembliesFolder))
                {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            return cyPhyMLComponentDictionary;

        }

        private static Dictionary<string, CyPhy.Component> getCyPhyMLComponentDictionary_ByName(CyPhy.Components cyPhyMLComponents)
        {

            Dictionary<string, CyPhy.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhy.Component>();

            foreach (CyPhy.Components childComponentsFolder in cyPhyMLComponents.Children.ComponentsCollection)
            {
                foreach (KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByName(childComponentsFolder))
                {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach (CyPhy.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLComponents.Children.ComponentAssembliesCollection)
            {
                foreach (KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByName(childComponentAssembliesFolder))
                {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach (CyPhy.Component childComponent in cyPhyMLComponents.Children.ComponentCollection)
            {
                cyPhyMLComponentDictionary[childComponent.Name] = childComponent;
            }

            return cyPhyMLComponentDictionary;
        }

        private static Dictionary<string, CyPhy.Component> getCyPhyMLComponentDictionary_ByName(CyPhy.ComponentAssemblies cyPhyMLComponentAssemblies)
        {

            Dictionary<string, CyPhy.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhy.Component>();

            foreach (CyPhy.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLComponentAssemblies.Children.ComponentAssembliesCollection)
            {
                foreach (KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByName(childComponentAssembliesFolder))
                {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach (CyPhy.ComponentAssembly childComponentAssemblyFolder in cyPhyMLComponentAssemblies.Children.ComponentAssemblyCollection)
            {
                foreach (KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByName(childComponentAssemblyFolder))
                {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            return cyPhyMLComponentDictionary;
        }

        private static Dictionary<string, CyPhy.Component> getCyPhyMLComponentDictionary_ByName(CyPhy.ComponentAssembly cyPhyMLComponentAssembly)
        {

            Dictionary<string, CyPhy.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhy.Component>();

            foreach (CyPhy.ComponentAssembly childComponentAssemblyFolder in cyPhyMLComponentAssembly.Children.ComponentAssemblyCollection)
            {
                foreach (KeyValuePair<string, CyPhy.Component> keyValuePair in getCyPhyMLComponentDictionary_ByName(childComponentAssemblyFolder))
                {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach (CyPhy.Component childComponent in cyPhyMLComponentAssembly.Children.ComponentCollection)
            {
                cyPhyMLComponentDictionary[childComponent.Name] = childComponent;
            }

            return cyPhyMLComponentDictionary;
        }

        public static string Meta_Path
        {
            get
            {
                const string keyName = "Software\\META";

                RegistryKey metaKey = RegistryKey.OpenBaseKey(RegistryHive.LocalMachine, RegistryView.Registry32)
                    .OpenSubKey(keyName, false);

                string metaPath = "C:\\Program Files (x86)\\META";
                if (metaKey != null)
                {
                    metaPath = (string)metaKey.GetValue("META_PATH", metaPath);
                }

                return metaPath;
            }
        }

        public static bool getCredentials(out string username, out string password) {

            username = "";
            password = "";

            LoginForm loginForm = new LoginForm();
            DialogResult dialogResult = loginForm.ShowDialog();

            if (dialogResult == DialogResult.Cancel) return false;

            username = loginForm.txtUsername.Text;
            password = loginForm.mtbPassword.Text;
            return true;
        }

        /// <summary>
        /// The main entry point of the interpreter. A transaction is already open,
        /// GMEConsole is available. A general try-catch block catches all the exceptions
        /// coming from this function, you don't need to add it. For more information, see InvokeEx.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for which the interpreter was called.</param>
        /// <param name="currentobj">The model open in the active tab in GME. Its value is null if no model is open (no GME modeling windows open). </param>
        /// <param name="selectedobjs">
        /// A collection for the selected model elements. It is never null.
        /// If the interpreter is invoked by the context menu of the GME Tree Browser, then the selected items in the tree browser. Folders
        /// are never passed (they are not FCOs).
        /// If the interpreter is invoked by clicking on the toolbar icon or the context menu of the modeling window, then the selected items 
        /// in the active GME modeling window. If nothing is selected, the collection is empty (contains zero elements).
        /// </param>
        /// <param name="startMode">Contains information about the GUI event that initiated the invocation.</param>
        [ComVisible(false)]
        public void Main(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, ComponentStartMode startMode)
        {
            string libroot = Path.GetDirectoryName(project.ProjectConnStr.Substring("MGA=".Length));

            // TODO: Add your interpreter code
            GMEConsole.Out.WriteLine("Running Component Importer...");

            string[] FileNames = null;
            DialogResult dr;
            using (OpenFileDialog ofd = new OpenFileDialog())
            {
                ofd.CheckFileExists = true;
                ofd.DefaultExt = "component.json";
                ofd.Multiselect = true;
                ofd.Filter = "component files (*.component.json;*.zip)|*.component.json;*.zip|All files (*.*)|*.*";

                dr = ofd.ShowDialog();
                if (dr == DialogResult.OK)
                {
                    FileNames = ofd.FileNames;
                }
            }
            if (dr == DialogResult.OK)
            {
                MgaGateway.PerformInTransaction(delegate
                {
                    ImportFiles(project, libroot, FileNames);
                });
                return;
            }
            else
            {
                GMEConsole.Warning.WriteLine("Component Importer cancelled");
                return;
            }

            // Get RootFolder
            //IMgaFolder rootFolder = project.RootFolder;
            //GMEConsole.Out.WriteLine(rootFolder.Name);

        }

        private void ImportFiles(MgaProject project, string libroot, string[] FileNames)
        {
            {
                bool replaceAll = false;
                bool doNotReplaceAll = false;
                CyPhy.RootFolder rootFolder = ISIS.GME.Common.Utils.CreateObject<CyPhyClasses.RootFolder>(project.RootFolder as MgaObject);
                Dictionary< string, CyPhy.Component> avmidComponentMap = getCyPhyMLComponentDictionary_ByAVMID( rootFolder );
                Dictionary<string, CyPhy.Component> nameComponentMap = getCyPhyMLComponentDictionary_ByName( rootFolder );
                Boolean b_FirstComponent = true;
                
                foreach (String inputJSONFile in FileNames)
                {
                    GMEConsole.Info.WriteLine("Importing {0}", inputJSONFile);
                    string s_file = "";
                    using (Process process = new Process())
                    {
                        process.StartInfo.FileName = Path.Combine(Meta_Path, "bin", "Python27", "Scripts", "python.exe");
                        process.StartInfo.EnvironmentVariables.Remove("PYTHONHOME");
                        process.StartInfo.UseShellExecute = false;

                        if (!System.IO.File.Exists(process.StartInfo.FileName)) {
                            GMEConsole.Error.WriteLine("ERROR:  Could not find Python executable (current path is \"" + process.StartInfo.FileName + "\")!");
                            return;
                        }

                        string componentLibraryManagerPath = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\componentLibraryManager.py";
                        if (!System.IO.File.Exists(componentLibraryManagerPath)) {
                            GMEConsole.Error.WriteLine("ERROR:  Could not find componentLibraryManager Python script (current path is \"" + componentLibraryManagerPath + "\")!");
                            return;
                        }
                        process.StartInfo.Arguments += String.Format("\"{0}\"", componentLibraryManagerPath);
                        process.StartInfo.Arguments += " --add ";
                        process.StartInfo.Arguments += String.Format("\"{0}\"", inputJSONFile);
                        process.StartInfo.Arguments += " --projroot ";
                        process.StartInfo.Arguments += String.Format("\"{0}\"", libroot);
                        if ( token != "" ) {
                            process.StartInfo.Arguments += " --token ";
                            process.StartInfo.Arguments += String.Format("\"{0}\"", token);
                        }

                        StringBuilder stderr = new StringBuilder();
                        StringBuilder stdout = new StringBuilder();

                        process.StartInfo.UseShellExecute = false;
                        process.StartInfo.RedirectStandardInput = true;
                        process.StartInfo.RedirectStandardOutput = true;
                        process.StartInfo.RedirectStandardError = true;
                        //process.OutputDataReceived += ((sender2, e2) =>
                        //{
                        //    stdout.Append(e2.Data);
                        //});
                        //process.ErrorDataReceived += ((sender2, e2) =>
                        //{
                        //    GMEConsole.Error.WriteLine(e2.Data);
                        //});

                        process.StartInfo.CreateNoWindow = true;
                        process.Start();
//                        process.BeginOutputReadLine();
//                        process.BeginErrorReadLine();
                        while (!process.StandardOutput.EndOfStream) {
                            string line = process.StandardOutput.ReadLine();
                            if ( line == "" ) break;
                            Newtonsoft.Json.Linq.JObject jObject;
                            try {
                                jObject = Newtonsoft.Json.Linq.JObject.Parse(line);
                            } catch (Exception e) {
                                GMEConsole.Out.WriteLine(line);
                                continue;
                            }
                            Newtonsoft.Json.Linq.JToken jToken;
                            if (jObject.TryGetValue("ERROR", out jToken)) {

                                GMEConsole.Error.WriteLine("ERROR: " + jToken.ToString());
                                break;

                            } else if (jObject.TryGetValue("WARNING", out jToken)) {

                                GMEConsole.Warning.WriteLine( "WARNING: " + jToken.ToString());

                            } else if (jObject.TryGetValue("INFO", out jToken)) {

                                GMEConsole.Info.WriteLine(jToken.ToString());

                            } else if (jObject.TryGetValue("AUTH", out jToken)) {

                                string authType = jToken.ToString();
                                string username, password;
                                if ( !getCredentials(out username, out password) ) {
                                    GMEConsole.Info.WriteLine( "Could not complete request:  username/password query cancelled." );
                                    jObject.RemoveAll();
                                    jObject.Add( "CANCEL", "CANCEL" );
                                    process.StandardInput.WriteLine( Newtonsoft.Json.JsonConvert.SerializeObject( jObject ) );
                                    continue;
                                }
                                jObject.RemoveAll();
                                jObject.Add("USERNAME", username);
                                jObject.Add("PASSWORD", password);
                                process.StandardInput.WriteLine( Newtonsoft.Json.JsonConvert.SerializeObject( jObject ) );

                            } else if (jObject.TryGetValue( "TOKEN", out jToken ) ) {

                                token = jToken.ToString();

                            } else if (jObject.TryGetValue("FILE", out jToken)) {

                                s_file = jToken.ToString();

                            }
                        }

                        string errorOutput = process.StandardError.ReadToEnd();

                        process.WaitForExit();
                        if (process.ExitCode != 0)
                        {
                            GMEConsole.Error.WriteLine("Error executing componentLibraryManager (exit code = {0}):", process.ExitCode);

                            byte[] byteArray = Encoding.ASCII.GetBytes( errorOutput );
                            MemoryStream stream = new MemoryStream(byteArray);

                            // convert stream to string
                            StreamReader reader = new StreamReader(stream);

                            while (!reader.EndOfStream) {
                                string line = reader.ReadLine();
                                GMEConsole.Error.WriteLine(line);
                            }

                            continue;
                        }
                    }

                    if (s_file == "") {
                        GMEConsole.Error.WriteLine("Error in componentLibraryManager:  did not render component import file");
                        continue;
                    }
                    
                    try
                    {
                        AVM.Component ac_import = AVM.Component.DeserializeFromFile(s_file);
                        string componentName = ac_import.Name;

                        CyPhy.ComponentAssembly cyPhyMLComponentAssembly = null;
                        CyPhy.Components cyPhyMLComponentsFolder = null;

                        CyPhy.Component cyPhyReplaceComponent = null;

                        if( nameComponentMap.TryGetValue( componentName, out cyPhyReplaceComponent) ) {
                            bool replace = false;
                            if( !doNotReplaceAll && !replaceAll ) {
                                // Present dialog to see if user wants to replace component with AVMID avmid
                                // Maybe have a "do all" checkbox (which sets "replaceAll" to "true") if many items are being imported.
                                // If yes, replace = true;
                                
                                String s_ExistingName = cyPhyReplaceComponent.Name;
                                String s_ExistingAVMID = cyPhyReplaceComponent.Attributes.AVMID;
                                String s_ExistingVersion = cyPhyReplaceComponent.Attributes.Version;
                                String s_ExistingRevision = cyPhyReplaceComponent.Attributes.Revision;

                                String s_ExistingDescriptor = cyPhyReplaceComponent.Name;
                                if (s_ExistingAVMID != "")
                                    s_ExistingDescriptor += "\nAVM ID: " + s_ExistingAVMID;
                                if (s_ExistingVersion != "")
                                    s_ExistingDescriptor += "\nVersion: " + s_ExistingVersion;
                                if (s_ExistingRevision != "")
                                    s_ExistingDescriptor += "\nRevision: " + s_ExistingRevision;

                                String s_NewName = ac_import.Name;
                                String s_NewAVMID = ac_import.AVMID;
                                String s_NewVersion = ac_import.Version;
                                String s_NewRevision = ac_import.Revision;

                                String s_NewDescriptor = ac_import.Name;
                                if (s_NewAVMID != "")
                                    s_NewDescriptor += "\nAVM ID: " + s_NewAVMID;
                                if (s_NewVersion != "")
                                    s_NewDescriptor += "\nVersion: " + s_NewVersion;
                                if (s_NewRevision != "")
                                    s_NewDescriptor += "\nRevision: " + s_NewRevision;
                                
                                String s_MessageBoxPromptTemplate = "Would you like to replace\n\n{0}\n\nwith\n\n{1}";
                                String s_MessageBoxPrompt = String.Format( s_MessageBoxPromptTemplate, s_ExistingDescriptor, s_NewDescriptor );

                                using (UpdatePrompt up = new UpdatePrompt())
                                {
                                    up.DialogText.Text = s_MessageBoxPrompt;
                                    up.ShowDialog();

                                    Dictionary<DialogResult, string> d_TranslateResult = new Dictionary<DialogResult, string>()
                                    {
                                        { DialogResult.OK, "Replace" },
                                        { DialogResult.Abort, "Import as New"},
                                    };

                                    if( d_TranslateResult[ up.DialogResult ] == "Replace" ) {
                                        replace = true;
                                        if( up.applyToAll.Checked ) {
                                            replaceAll = true;
                                        }
                                    } else if( d_TranslateResult[ up.DialogResult ] == "Import as New" ) {
                                        if( up.applyToAll.Checked ) {
                                            doNotReplaceAll = true;
                                        }
                                    }
                                }
                            }

                            if( !replace && !replaceAll || doNotReplaceAll ) {
                                cyPhyReplaceComponent = null;
                            }
                        }

                        if( cyPhyReplaceComponent != null ) {
                            Object replaceComponentParent = cyPhyReplaceComponent.ParentContainer;
                            if( replaceComponentParent is CyPhy.ComponentAssembly ) {
                                cyPhyMLComponentAssembly = replaceComponentParent as CyPhy.ComponentAssembly;
                            } else {
                                cyPhyMLComponentsFolder = replaceComponentParent as CyPhy.Components;
                            }
                        } else {
                            cyPhyMLComponentsFolder = GetImportFolder( rootFolder );
                        }

                        Boolean b_ResetUnitMap;
                        if (b_FirstComponent)
                            b_ResetUnitMap = true;
                        else
                            b_ResetUnitMap = false;

                        CyPhy.Component c = null;
                        if( cyPhyMLComponentAssembly != null ) {
                            c = CyPhy2ComponentModel.Convert.AVMComponent2CyPhyML(cyPhyMLComponentAssembly, ac_import, avmidComponentMap, b_ResetUnitMap, GMEConsole);
                        } else {
                            c = CyPhy2ComponentModel.Convert.AVMComponent2CyPhyML(cyPhyMLComponentsFolder, ac_import, avmidComponentMap, b_ResetUnitMap, GMEConsole);
                        }

                        if( cyPhyReplaceComponent != null ) {
                            foreach( IMgaReference reference in ( cyPhyReplaceComponent.Impl as IMgaFCO ).ReferencedBy ) {
                                ReferenceSwitcher.Switcher.MoveReferenceWithRefportConnections( c.Impl as IMgaFCO, reference, WriteLine );
                            }
                            cyPhyReplaceComponent.Delete();
                        }

                        // If icon available, set it in Preferences.
                        // Relative paths here are from our project root.
                        String componentModelRelativePath = Path.GetDirectoryName(s_file);
                        componentModelRelativePath = componentModelRelativePath.Replace(libroot,"");
                        String iconRelativePath = "";
                        foreach (AVM.File file in ac_import.Files)
                        {
                            if (file.Location.Contains("Icon.png"))
                                iconRelativePath = componentModelRelativePath + "/" + file.Location;
                        }
                        if (iconRelativePath != "")
                        {
                            // Remove leading "\"
                            if (iconRelativePath.IndexOf("\\") == 0)
                                iconRelativePath = iconRelativePath.Remove(0, 1);

                            // Shrink double backslash to single
                            iconRelativePath = iconRelativePath.Replace("\\\\","\\");
                            
                            // Replace / with \
                            iconRelativePath = iconRelativePath.Replace('/', '\\');
                            
                            // If icon exists, set it
                            if (System.IO.File.Exists(iconRelativePath))
                            {
                                // Set "icon" regnode
                                (c.Impl as GME.MGA.IMgaFCO).set_RegistryValue("icon", iconRelativePath);
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        GMEConsole.Error.WriteLine("Exception: {0}", ex.Message.ToString());
                        GMEConsole.Info.WriteLine("Stack: {0}", ex.StackTrace.ToString());
                    }
                    b_FirstComponent = false;
                }



            }

			// Get RootFolder
			//IMgaFolder rootFolder = project.RootFolder;
			//GMEConsole.Out.WriteLine(rootFolder.Name);
			
        }

        #region IMgaComponentEx Members

        MgaGateway MgaGateway { get; set; }
        GMEConsole GMEConsole { get; set; }

        public void InvokeEx(MgaProject project, MgaFCO currentobj, MgaFCOs selectedobjs, int param)
        {
            if (!enabled)
            {
                return;
            }

            try
            {
                GMEConsole = GMEConsole.CreateFromProject(project);
                MgaGateway = new MgaGateway(project);
                project.CreateTerritoryWithoutSink(out MgaGateway.territory);

                Main(project, currentobj, selectedobjs, Convert(param));
            }
            finally
            {
                if (MgaGateway.territory != null)
                {
                    MgaGateway.territory.Destroy();
                }
                MgaGateway = null;
                project = null;
                currentobj = null;
                selectedobjs = null;
                GMEConsole = null;
                GC.Collect();
                GC.WaitForPendingFinalizers();
            }
        }

        private ComponentStartMode Convert(int param)
        {
            switch (param)
            {
                case (int)ComponentStartMode.GME_BGCONTEXT_START:
                    return ComponentStartMode.GME_BGCONTEXT_START;
                case (int)ComponentStartMode.GME_BROWSER_START:
                    return ComponentStartMode.GME_BROWSER_START;

                case (int)ComponentStartMode.GME_CONTEXT_START:
                    return ComponentStartMode.GME_CONTEXT_START;

                case (int)ComponentStartMode.GME_EMBEDDED_START:
                    return ComponentStartMode.GME_EMBEDDED_START;

                case (int)ComponentStartMode.GME_ICON_START:
                    return ComponentStartMode.GME_ICON_START;

                case (int)ComponentStartMode.GME_MAIN_START:
                    return ComponentStartMode.GME_MAIN_START;

                case (int)ComponentStartMode.GME_MENU_START:
                    return ComponentStartMode.GME_MENU_START;
                case (int)ComponentStartMode.GME_SILENT_MODE:
                    return ComponentStartMode.GME_SILENT_MODE;
            }

            return ComponentStartMode.GME_SILENT_MODE;
        }

        #region Component Information
        public string ComponentName
        {
            get { return GetType().Name; }
        }

        public string ComponentProgID
        {
            get
            {
                return ComponentConfig.progID;
            }
        }

        public componenttype_enum ComponentType
        {
            get { return ComponentConfig.componentType; }
        }
        public string Paradigm
        {
            get { return ComponentConfig.paradigmName; }
        }
        #endregion

        #region Enabling
        bool enabled = true;
        public void Enable(bool newval)
        {
            enabled = newval;
        }
        #endregion

        #region Interactive Mode
        protected bool interactiveMode = true;
        public bool InteractiveMode
        {
            get
            {
                return interactiveMode;
            }
            set
            {
                interactiveMode = value;
            }
        }
        #endregion

        #region Custom Parameters
        SortedDictionary<string, object> componentParameters = null;

        public object get_ComponentParameter(string Name)
        {
            if (Name == "type")
                return "csharp";

            if (Name == "path")
                return GetType().Assembly.Location;

            if (Name == "fullname")
                return GetType().FullName;

            object value;
            if (componentParameters != null && componentParameters.TryGetValue(Name, out value))
            {
                return value;
            }

            return null;
        }

        public void set_ComponentParameter(string Name, object pVal)
        {
            if (componentParameters == null)
            {
                componentParameters = new SortedDictionary<string, object>();
            }

            componentParameters[Name] = pVal;
        }
        #endregion

        #region Unused Methods
        // Old interface, it is never called for MgaComponentEx interfaces
        public void Invoke(MgaProject Project, MgaFCOs selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        // Not used by GME
        public void ObjectsInvokeEx(MgaProject Project, MgaObject currentobj, MgaObjects selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        #endregion

        #endregion

        #region IMgaVersionInfo Members

        public GMEInterfaceVersion_enum version
        {
            get { return GMEInterfaceVersion_enum.GMEInterfaceVersion_Current; }
        }

        #endregion

        #region Registration Helpers

        [ComRegisterFunctionAttribute]
        public static void GMERegister(Type t)
        {
            Registrar.RegisterComponentsInGMERegistry();

        }

        [ComUnregisterFunctionAttribute]
        public static void GMEUnRegister(Type t)
        {
            Registrar.UnregisterComponentsInGMERegistry();
        }

        #endregion


    }
}
