/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
using System;
using System.Collections;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;
using System.Linq;
using GME.CSharp;
using GME;
using GME.MGA;
using GME.MGA.Core;
using ISIS.GME.Common;
using System.Windows.Forms;
using CyPhyML = ISIS.GME.Dsml.CyPhyML.Interfaces;
using CyPhyMLClasses = ISIS.GME.Dsml.CyPhyML.Classes;
using AVM;
using CyPhy2ComponentModel;

using JsonSchema = Newtonsoft.Json.Schema.JsonSchema;
using Newtonsoft.Json;
using System.Reflection;
using Microsoft.Win32;

namespace CyPhyComponentImporterCL {

    public class CyPhyComponentImporterCL {

        private static void WriteLine(Func<string, string, string> f, IMgaFCO a, IMgaFCO b) {
            //if (GMEConsole != null)
            //{
            //    GMEConsole.Out.WriteLine(f(GetLink(a, a.Name), GetLink(b, b.Name)));
            //}
            //else
            {
                Console.Out.WriteLine(  f( a.AbsPath, b.AbsPath )  );
            }
        }

        private static Dictionary<string, CyPhyML.Component> getCyPhyMLComponentDictionary_ByAVMID(CyPhyML.RootFolder cyPhyMLRootFolder) {

            Dictionary<string, CyPhyML.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhyML.Component>();

            foreach (CyPhyML.Components childComponentsFolder in cyPhyMLRootFolder.Children.ComponentsCollection) {
                foreach (KeyValuePair<string, CyPhyML.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID(childComponentsFolder)) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach (CyPhyML.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLRootFolder.Children.ComponentAssembliesCollection) {
                foreach (KeyValuePair<string, CyPhyML.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID(childComponentAssembliesFolder)) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            return cyPhyMLComponentDictionary;

        }

        private static Dictionary<string, CyPhyML.Component> getCyPhyMLComponentDictionary_ByAVMID(CyPhyML.Components cyPhyMLComponents) {

            Dictionary<string, CyPhyML.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhyML.Component>();

            foreach (CyPhyML.Components childComponentsFolder in cyPhyMLComponents.Children.ComponentsCollection) {
                foreach (KeyValuePair<string, CyPhyML.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID(childComponentsFolder)) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach (CyPhyML.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLComponents.Children.ComponentAssembliesCollection) {
                foreach (KeyValuePair<string, CyPhyML.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID(childComponentAssembliesFolder)) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach (CyPhyML.Component childComponent in cyPhyMLComponents.Children.ComponentCollection) {
                cyPhyMLComponentDictionary[childComponent.Attributes.AVMID] = childComponent;
            }

            return cyPhyMLComponentDictionary;
        }

        private static Dictionary<string, CyPhyML.Component> getCyPhyMLComponentDictionary_ByAVMID(CyPhyML.ComponentAssemblies cyPhyMLComponentAssemblies) {

            Dictionary<string, CyPhyML.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhyML.Component>();

            foreach (CyPhyML.ComponentAssemblies childComponentAssembliesFolder in cyPhyMLComponentAssemblies.Children.ComponentAssembliesCollection) {
                foreach (KeyValuePair<string, CyPhyML.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID(childComponentAssembliesFolder)) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach (CyPhyML.ComponentAssembly childComponentAssemblyFolder in cyPhyMLComponentAssemblies.Children.ComponentAssemblyCollection) {
                foreach (KeyValuePair<string, CyPhyML.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID(childComponentAssemblyFolder)) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            return cyPhyMLComponentDictionary;
        }

        private static Dictionary<string, CyPhyML.Component> getCyPhyMLComponentDictionary_ByAVMID(CyPhyML.ComponentAssembly cyPhyMLComponentAssembly) {

            Dictionary<string, CyPhyML.Component> cyPhyMLComponentDictionary = new Dictionary<string, CyPhyML.Component>();

            foreach (CyPhyML.ComponentAssembly childComponentAssemblyFolder in cyPhyMLComponentAssembly.Children.ComponentAssemblyCollection) {
                foreach (KeyValuePair<string, CyPhyML.Component> keyValuePair in getCyPhyMLComponentDictionary_ByAVMID(childComponentAssemblyFolder)) {
                    cyPhyMLComponentDictionary[keyValuePair.Key] = keyValuePair.Value;
                }
            }

            foreach (CyPhyML.Component childComponent in cyPhyMLComponentAssembly.Children.ComponentCollection) {
                cyPhyMLComponentDictionary[childComponent.Attributes.AVMID] = childComponent;
            }

            return cyPhyMLComponentDictionary;
        }


        private static CyPhyML.Component getComponent( CyPhyML.RootFolder cyPhyMLParentObject, string componentPath ) {


	        if ( componentPath == null || componentPath == "" ) {
		        Console.Out.WriteLine( "ERROR: Could not find a CyPhyML Component in empty path");
		        return null;
	        }

            string[] componentPathElementsArray = componentPath.Split( '/' );
            IEnumerator cpaItr = componentPathElementsArray.GetEnumerator();
            cpaItr.MoveNext();
	        if (  componentPath.StartsWith( "/" )  ) {
                cpaItr.MoveNext();
            }

            string objectName = cpaItr.Current as string;

            foreach( CyPhyML.Components cyPhyMLComponentsFolder in cyPhyMLParentObject.Children.ComponentsCollection ) {
			    if ( cyPhyMLComponentsFolder.Name == objectName  ) return getComponent( cyPhyMLComponentsFolder, componentPath, cpaItr );
            }
            foreach( CyPhyML.ComponentAssemblies cyPhyMLComponentsAssembliesFolder in cyPhyMLParentObject.Children.ComponentAssembliesCollection ) {
			    if ( cyPhyMLComponentsAssembliesFolder.Name == objectName  ) return getComponent( cyPhyMLComponentsAssembliesFolder, componentPath, cpaItr );
            }

	        Console.Out.WriteLine( "ERROR: Could not find \"" + objectName + "\" in path \"" + componentPath + "\"" );
	        return null;
        }

        private static CyPhyML.Component getComponent( CyPhyML.Components cyPhyMLParentObject, string componentPath, IEnumerator cpaItr ) {

	        if ( !cpaItr.MoveNext() ) {
		        Console.Out.WriteLine( "ERROR: Could not find a CyPhyML Component in path \"" + componentPath + "\"" );
		        return null;
	        }

            string objectName = cpaItr.Current as string;

            foreach( CyPhyML.Components cyPhyMLComponentsFolder in cyPhyMLParentObject.Children.ComponentsCollection ) {
			    if ( cyPhyMLComponentsFolder.Name == objectName  ) return getComponent( cyPhyMLComponentsFolder, componentPath, cpaItr );
            }
            foreach( CyPhyML.ComponentAssemblies cyPhyMLComponentsAssembliesFolder in cyPhyMLParentObject.Children.ComponentAssembliesCollection ) {
			    if ( cyPhyMLComponentsAssembliesFolder.Name == objectName  ) return getComponent( cyPhyMLComponentsAssembliesFolder, componentPath, cpaItr );
            }
            foreach( CyPhyML.Component cyPhyMLComponent in cyPhyMLParentObject.Children.ComponentCollection ) {
			    if ( cyPhyMLComponent.Name == objectName  ) return getComponent( cyPhyMLComponent, componentPath, cpaItr );
            }

	        Console.Out.WriteLine( "ERROR: Could not find \"" + objectName + "\" in path \"" + componentPath + "\"" );
	        return null;
        }

        private static CyPhyML.Component getComponent( CyPhyML.ComponentAssemblies cyPhyMLParentObject, string componentPath, IEnumerator cpaItr ) {

	        if ( !cpaItr.MoveNext() ) {
		        Console.Out.WriteLine( "ERROR: Could not find a CyPhyML Component in path \"" + componentPath + "\"" );
		        return null;
	        }

            string objectName = cpaItr.Current as string;

            foreach( CyPhyML.ComponentAssemblies cyPhyMLComponentsAssembliesFolder in cyPhyMLParentObject.Children.ComponentAssembliesCollection ) {
			    if ( cyPhyMLComponentsAssembliesFolder.Name == objectName  ) return getComponent( cyPhyMLComponentsAssembliesFolder, componentPath, cpaItr );
            }
            foreach( CyPhyML.ComponentAssembly cyPhyMLComponentsAssemblyFolder in cyPhyMLParentObject.Children.ComponentAssemblyCollection ) {
			    if ( cyPhyMLComponentsAssemblyFolder.Name == objectName  ) return getComponent( cyPhyMLComponentsAssemblyFolder, componentPath, cpaItr );
            }

	        Console.Out.WriteLine( "ERROR: Could not find \"" + objectName + "\" in path \"" + componentPath + "\"" );
	        return null;
        }

        private static CyPhyML.Component getComponent( CyPhyML.ComponentAssembly cyPhyMLParentObject, string componentPath, IEnumerator cpaItr ) {

	        if ( !cpaItr.MoveNext() ) {
		        Console.Out.WriteLine( "ERROR: Could not find a CyPhyML Component in path \"" + componentPath + "\"" );
		        return null;
	        }

            string objectName = cpaItr.Current as string;

            foreach( CyPhyML.ComponentAssembly cyPhyMLComponentsAssemblyFolder in cyPhyMLParentObject.Children.ComponentAssemblyCollection ) {
			    if ( cyPhyMLComponentsAssemblyFolder.Name == objectName  ) return getComponent( cyPhyMLComponentsAssemblyFolder, componentPath, cpaItr );
            }
            foreach( CyPhyML.ComponentRef cyPhyMLComponentRef in cyPhyMLParentObject.Children.ComponentRefCollection ) {
			    if ( cyPhyMLComponentRef.Name == objectName  ) return getComponent( cyPhyMLComponentRef, componentPath, cpaItr );
            }
            foreach( CyPhyML.Component cyPhyMLComponent in cyPhyMLParentObject.Children.ComponentCollection ) {
			    if ( cyPhyMLComponent.Name == objectName  ) return getComponent( cyPhyMLComponent, componentPath, cpaItr );
            }

	        Console.Out.WriteLine( "ERROR: Could not find \"" + objectName + "\" in path \"" + componentPath + "\"" );
	        return null;
        }

        private static CyPhyML.Component getComponent( CyPhyML.ComponentRef cyPhyMLComponentRef, string componentPath, IEnumerator cpaItr ) {

            string objectName = cpaItr.Current as string;

            CyPhyML::Component cyPhyMLComponent = cyPhyMLComponentRef.Referred.Component;
	        if ( cyPhyMLComponent == null ) {
                Console.Out.WriteLine( "ERROR: Null Component Reference \"" + objectName + "\" in path \"" + componentPath + "\"" );
		        return null;
	        }

            if( cpaItr.MoveNext() ) {
                Console.Out.WriteLine( "WARNING:  CyPhyML Component found at \"" + objectName + "\" in path \"" + componentPath + "\"" );
            }

            return getComponent( cyPhyMLComponent, componentPath, cpaItr );
        }

        private static CyPhyML.Component getComponent( CyPhyML.Component cyPhyMLComponent, string componentPath, IEnumerator cpaItr ) {

            string objectName = cpaItr.Current as string;

            if( cpaItr.MoveNext() ) {
                Console.Out.WriteLine( "WARNING:  CyPhyML Component found at \"" + objectName + "\" in path \"" + componentPath + "\"" );
            }

            return cyPhyMLComponent;
        }

        public static string Meta_Path
        {
            get
            {
                const string keyName = "Software\\META";

                RegistryKey metaKey = RegistryKey.OpenBaseKey(RegistryHive.LocalMachine, RegistryView.Registry32)
                    .OpenSubKey(keyName, false);

                string metaPath = "C:\\Program Files (x86)\\META";
                if (metaKey != null)
                {
                    metaPath = (string)metaKey.GetValue("META_PATH", metaPath);
                }

                return metaPath;
            }
        }

        private static MgaProject GetProject( String filename ) {
            MgaProject result = null;

            if( filename != null && filename != "" ) {
                if( Path.GetExtension( filename ) == ".mga" ) {
                    result = new MgaProject();
                    if( System.IO.File.Exists( filename ) ) {
                        Console.Out.Write( "Opening {0} ... ", filename );
                        bool ro_mode;
                        result.Open( "MGA=" + filename, out ro_mode );
                    } else {
                        Console.Out.Write( "Creating {0} ... ", filename );
                        result.Create( "MGA=" + filename, "CyPhyML" );
                    }
                    Console.Out.WriteLine( "Done." );
                } else {
                    Console.Error.WriteLine( "{0} file must be an mga project.", filename );
                }
            } else {
                Console.Error.WriteLine( "Please specify an Mga project." );
            }

            return result;
        }

        private static CyPhyML.Components getCyPhyMLImportedComponentsFolder( CyPhyML.RootFolder cyPhyMLRootFolder ) {
            foreach( CyPhyML.Components cyPhyMLComponents in cyPhyMLRootFolder.Children.ComponentsCollection ) {
                if( cyPhyMLComponents.Name == "Imported Components" ) return cyPhyMLComponents;
            }

            CyPhyML.Components cyPhyMLImportedComponentsFolder = CyPhyMLClasses.Components.Create( cyPhyMLRootFolder );
            cyPhyMLImportedComponentsFolder.Name = "Imported Components";
            return cyPhyMLImportedComponentsFolder;
        }

        private static void usage() {
            Console.Out.WriteLine( "Usage: <program> AVMFile [ -p <path-to-component-to-replace> ] CyPhyMLFile.mga" );
            Console.Out.WriteLine( "Usage: <program> -r <path> CyPhyMLFile.mga" );
            Environment.Exit( 1 );
        }

        public static bool getCredentials(out string username, out string password) {

            username = "";
            password = "";

            LoginForm loginForm = new LoginForm();
            DialogResult dialogResult = loginForm.ShowDialog();

            if (dialogResult == DialogResult.Cancel) return false;

            username = loginForm.txtUsername.Text;
            password = loginForm.mtbPassword.Text;
            return true;
        }

        public static void Main( String[] args ) {

            if( args.Length < 2 || args.Length > 4 ) usage();

            MgaProject mgaProject;
            List<String> lp_FilesToImport = new List<string>();
            string avmFilePath = "";
            string mgaProjectPath = "";
            string componentReplacementPath = "";

            bool rOptionUsed = false;
            bool pOptionUsed = false;

            for( int ix = 0 ; ix < args.Length ; ++ix ) {

                if( args[ ix ].ToLower() == "-r" ) {

                    if( pOptionUsed ) usage();
                    rOptionUsed = true;

                    if ( ++ix >= args.Length ) usage();

                    if( avmFilePath != null && avmFilePath != "" ) {
                        if( mgaProjectPath != null && mgaProjectPath != "" ) usage();
                        mgaProjectPath = avmFilePath;
                        avmFilePath = "";
                        lp_FilesToImport.Clear();
                    }

                    String sImportDirectory = args[ ix ];

                    String startingDirectory = Path.GetFullPath( sImportDirectory );
                    string[] jsonFiles = Directory.GetFiles( startingDirectory, "*.component.json", SearchOption.AllDirectories );

                    foreach( String p_JsonFile in jsonFiles ) {
                        lp_FilesToImport.Add( Path.GetFullPath( p_JsonFile ) );
                    }

                } else if( args[ ix ].ToLower() == "-p" ) {

                    if( rOptionUsed ) usage();
                    pOptionUsed = true;

                    if ( ++ix >= args.Length ) usage();
                    componentReplacementPath = args[ ix ];

                } else if ( lp_FilesToImport.Count == 0 && avmFilePath == "" ) {

                    avmFilePath = args[ ix ];
                    try
                    {
                        lp_FilesToImport.Add(Path.GetFullPath(avmFilePath));
                    }
                    catch (System.ArgumentException ex)
                    {
                        Console.Out.WriteLine(ex.Message);
                        Console.Out.WriteLine(avmFilePath);
                        throw ex;
                    }

                } else {

                    if( mgaProjectPath != null && mgaProjectPath != "" ) usage();
                    mgaProjectPath = args[ ix ];
                }

            }


            mgaProject = GetProject( mgaProjectPath );
                    
            if( mgaProject != null ) {
                MgaGateway mgaGateway = new MgaGateway(mgaProject);

                mgaProject.CreateTerritoryWithoutSink(out mgaGateway.territory);
                mgaGateway.PerformInTransaction(delegate
                {

                    string libroot = Path.GetDirectoryName( Path.GetFullPath(mgaProjectPath) );

                    CyPhyML.RootFolder cyPhyMLRootFolder = ISIS.GME.Common.Utils.CreateObject<CyPhyMLClasses.RootFolder>(mgaProject.RootFolder as MgaObject);

                    IMgaFolder oldQudt = mgaProject.RootFolder.ChildFolders.Cast<IMgaFolder>().Where(   x => x.LibraryName != "" && (  x.Name.ToLower().Contains( "qudt" )  )   ).FirstOrDefault();
                    string mgaQudtPath = Meta_Path + "\\meta\\CyPhyMLQudt.mga";

                    bool needAttach = false;
                    if ( oldQudt == null ) {

                        needAttach = true;

                    } else {

                        long loldModTime;
                        DateTime oldModTime = long.TryParse( oldQudt.RegistryValue[ "modtime" ], out loldModTime ) ? DateTime.FromFileTimeUtc(loldModTime) : DateTime.MinValue;
                        needAttach = System.IO.File.GetLastWriteTimeUtc(mgaQudtPath).CompareTo(oldModTime) > 0;
                        if ( !needAttach ) {
                            Console.Error.WriteLine("QUDT is up-to-date: embedded library modified " + oldModTime.ToString() + ", CyPhyMLQudt.mga modified " + System.IO.File.GetLastWriteTimeUtc(mgaQudtPath).ToString());
                        }
                    }

                    if ( needAttach ) {

                        Console.Error.WriteLine("Attaching library " + mgaQudtPath);
                        ISIS.GME.Common.Interfaces.RootFolder newQudt = ISIS.GME.Common.Classes.RootFolder.GetRootFolder(mgaProject).AttachLibrary("MGA=" + mgaQudtPath);
                        DateTime modtime = System.IO.File.GetLastWriteTimeUtc(mgaQudtPath);
                        ((newQudt as ISIS.GME.Common.Classes.RootFolder).Impl as GME.MGA.IMgaFolder).RegistryValue["modtime"] =
                                modtime.ToFileTimeUtc().ToString();

                        if ( oldQudt != null ) {
                            ReferenceSwitcher.Switcher sw = new ReferenceSwitcher.Switcher(oldQudt, newQudt.Impl, null);
                            sw.UpdateSublibrary();
                            oldQudt.DestroyObject();
                        }
                        ((newQudt as ISIS.GME.Common.Classes.RootFolder).Impl as GME.MGA.IMgaFolder).LibraryName = "UnitLibrary QUDT";
                        Console.Error.WriteLine((oldQudt == null ? "Attached " : "Refreshed") + " Qudt library.");
                    }

                    Dictionary<string, CyPhyML.Component> avmidComponentMap = getCyPhyMLComponentDictionary_ByAVMID( cyPhyMLRootFolder );

                    CyPhyML.Component cyPhyMLReplaceComponent = null;
                    if( componentReplacementPath != null && componentReplacementPath != "" ) {
                        cyPhyMLReplaceComponent = getComponent( cyPhyMLRootFolder, componentReplacementPath );
                    }

                    CyPhyML.ComponentAssembly cyPhyMLComponentAssembly = null;
                    CyPhyML.Components cyPhyMLComponentsFolder = null;

                    if( cyPhyMLReplaceComponent != null ) {
                        Object replaceComponentParent = cyPhyMLReplaceComponent.ParentContainer;
                        if( replaceComponentParent is CyPhyML.ComponentAssembly ) {
                            cyPhyMLComponentAssembly = replaceComponentParent as CyPhyML.ComponentAssembly;
                        } else if( replaceComponentParent is CyPhyML.Components ) {
                            cyPhyMLComponentsFolder = replaceComponentParent as CyPhyML.Components;
                        } else {
                            Console.Out.WriteLine( "ERROR:  Invalid component parent, class is \"" + replaceComponentParent.GetType().Name + "\"" );
                            Environment.Exit( 2 );
                        }
                    } else {
                        cyPhyMLComponentsFolder = getCyPhyMLImportedComponentsFolder( cyPhyMLRootFolder );
                    }

                    Boolean b_FirstComponent = true;
                    foreach (String inputJSONFile in lp_FilesToImport)
                    {
                        try
                        {
                            AVM.Component avmComponent = AVM.Component.DeserializeFromFile(inputJSONFile);
                            CyPhyML.Component c = null;
                            if( cyPhyMLComponentAssembly != null )
                            {
                                c = CyPhy2ComponentModel.Convert.AVMComponent2CyPhyML( cyPhyMLComponentAssembly, avmComponent, avmidComponentMap );
                            } 
                            else 
                            {
                                /// Re-use unit map if possible
                                if (b_FirstComponent)
                                {
                                    c = CyPhy2ComponentModel.Convert.AVMComponent2CyPhyML(cyPhyMLComponentsFolder, avmComponent, avmidComponentMap, true);
                                    b_FirstComponent = false;
                                }
                                else
                                {
                                    c = CyPhy2ComponentModel.Convert.AVMComponent2CyPhyML(cyPhyMLComponentsFolder, avmComponent, avmidComponentMap, false);
                                }
                            }

                            if (cyPhyMLReplaceComponent != null)
                            {
                                foreach( IMgaReference reference in ( cyPhyMLReplaceComponent.Impl as IMgaFCO ).ReferencedBy ) {
                                    ReferenceSwitcher.Switcher.MoveReferenceWithRefportConnections( c.Impl as IMgaFCO, reference, WriteLine );
                                }
                                cyPhyMLReplaceComponent.Delete();
                            }
                            
                            // ARN HACK for Ricardo names
                            //c.Name = Path.GetFileNameWithoutExtension(p_JsonFile);
                        }
                        catch (Exception ex)
                        {
                            Console.Error.WriteLine("Exception: {0} ({1})", ex.Message.ToString(), inputJSONFile);
                            //Console.Error.WriteLine("Stack: {0}", ex.StackTrace.ToString());
                        }
                    }
                });
                
                mgaProject.Save();
                
                if (mgaGateway.territory != null)
                {
                    mgaGateway.territory.Destroy();
                }

            }

        }

    }
}
