/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "CyPhyConnectionUtility.h"
#include "Uml.h"
#include "UmlExt.h"
#include "UdmUtil.h"

void cu_TestBench(const TestBench& tb);
void cu_DesignContainer(const DesignContainer& dc);

void CyPhyConnectionUtility_main(const Object& focusObject) {
	// What case?
	if (focusObject.type() == TestBench::meta)
		cu_TestBench( TestBench::Cast(focusObject) );

	if (focusObject.type() == DesignContainer::meta)
		cu_DesignContainer( DesignContainer::Cast(focusObject) );
};

pair<bool,ValueFlowTarget> getTBVF_equivalent(const set<ValueFlowTarget>& tbvfset,const ValueFlowTarget& vft) {
	pair<bool,ValueFlowTarget> rtn;
	rtn.first = false;

	for (set<ValueFlowTarget>::const_iterator i = tbvfset.begin(); i != tbvfset.end(); i++) {
		ValueFlowTarget vfi(*i);
		if ( vfi.name() == vft.name() && vfi.type() == vft.type() ) {
			rtn.first = true;
			rtn.second = vfi;
			return rtn;
		}
	}
	return rtn;
};

/** \brief Connection discovery and establishment for TestBench models.
    \param [in] tb TestBench context to process
    \return void
*/

void cu_TestBench(const TestBench& tb) {
	// Let's get all of the ValueFlowTarget objects in the TestBench.
	set<ValueFlowTarget> tbVftSet = tb.ValueFlowTarget_children();

	// This chunk checks for "top level system under test" as reference
	TopLevelSystemUnderTest tlsut = tb.TopLevelSystemUnderTest_child();
	if (tlsut) {
		// Let's get all of this guy's ValueFlowTargets
		set<ValueFlowTarget> vftSet;
		if ( tlsut.type() == DesignContainer::meta)
			vftSet = DesignContainer::Cast(tlsut.ref()).ValueFlowTarget_kind_children();
		else
			vftSet = DesignElement::Cast(tlsut.ref()).ValueFlowTarget_kind_children();

		for (set<ValueFlowTarget>::const_iterator j = vftSet.begin(); j != vftSet.end(); j++) {
			ValueFlowTarget sut_vft(*j);
			
			Uml::Class sut_vft_type = sut_vft.type();
			if (Uml::IsDerivedFrom(sut_vft_type,Parameter::meta) || Uml::IsDerivedFrom(sut_vft_type,Property::meta)) {
				pair<bool,ValueFlowTarget> result = getTBVF_equivalent(tbVftSet,sut_vft);
				if (result.first) {
					ValueFlowTarget tb_vft = result.second;
					// Okay, so we found something. Let's hook it up.
					if (Uml::IsDerivedFrom(tb_vft.type(),Property::meta)) {
						// Let's go from the SUT-Property to the TB-Property
						ValueFlow newVf = ValueFlow::Create(tb);
						
						newVf.srcValueFlow_refport_parent() = tlsut;
						newVf.srcValueFlow_end() = sut_vft;

						newVf.dstValueFlow_end() = tb_vft;
					} else if (Uml::IsDerivedFrom(tb_vft.type(),Parameter::meta)) {
						// Let's go from the TB-Parameter to the SUT-Parameter
						ValueFlow newVf = ValueFlow::Create(tb);
						
						newVf.srcValueFlow_end() = tb_vft;
						
						newVf.dstValueFlow_refport_parent() = tlsut;
						newVf.dstValueFlow_end() = sut_vft;
					}
				}
			}
		}
	} 
	else // This chunk searches for "top level system under test" as an instance
	{		
		set<DesignElement> deSet = tb.DesignElement_kind_children();
		for (set<DesignElement>::const_iterator i = deSet.begin(); i != deSet.end(); i++) {
			DesignElement de(*i);

			if (Uml::IsDerivedFrom(de.type(),TestComponent::meta))
				continue;
			// Should be left with only TopLevelSystemUnderTest (as instance)		
		
			// Let's get all of this guy's ValueFlowTargets
			set<ValueFlowTarget> vftSet = de.ValueFlowTarget_kind_children();
			for (set<ValueFlowTarget>::const_iterator j = vftSet.begin(); j != vftSet.end(); j++) {
				ValueFlowTarget sut_vft(*j);
			
				if (Uml::IsDerivedFrom(sut_vft.type(),Parameter::meta) || Uml::IsDerivedFrom(sut_vft.type(),Property::meta)) {
					pair<bool,ValueFlowTarget> result = getTBVF_equivalent(tbVftSet,sut_vft);
					if (result.first) {
						ValueFlowTarget tb_vft = result.second;
						// Okay, so we found something. Let's hook it up.
						if (Uml::IsDerivedFrom(tb_vft.type(),Property::meta)) {
							// Let's go from the SUT-Property to the TB-Property
							ValueFlow newVf = ValueFlow::Create(tb);
							newVf.srcValueFlow_end() = sut_vft;
							newVf.dstValueFlow_end() = tb_vft;
						} else if (Uml::IsDerivedFrom(tb_vft.type(),Parameter::meta)) {
							// Let's go from the SUT-Property to the TB-Property
							ValueFlow newVf = ValueFlow::Create(tb);
							newVf.srcValueFlow_end() = tb_vft;
							newVf.dstValueFlow_end() = sut_vft;
						}
					}
				}
			}
		}
	}
};

/** \brief Helper function for getting src and dst objects of a Composition connection.
    \param [in] c Composition connection to analyze
    \return pair<Object,Object> 1st-src 2nd-dst
*/

pair<Object,Object> getCompositionEnds(const Composition& c) {
	pair<Object,Object> rtn;
	Uml::Class cClass = c.type();
	#define idt(x) cClass == x
	
	if (idt(PowerFlow::meta)) {
		PowerFlow pf = PowerFlow::Cast(c);
		rtn.first = pf.srcPowerFlow_end();
		rtn.second = pf.dstPowerFlow_end();
	} else if (idt(InformationFlow::meta)) {
		InformationFlow assoc = InformationFlow::Cast(c);
		rtn.first = assoc.srcInformationFlow_end();
		rtn.second = assoc.dstInformationFlow_end();		
	} else if (idt(ExecutionAssignment::meta)) {
		ExecutionAssignment assoc = ExecutionAssignment::Cast(c);
		rtn.first = assoc.srcExecutionAssignment_end();
		rtn.second = assoc.dstExecutionAssignment_end();		
	} else if (idt(JoinStructures::meta)) {
		JoinStructures assoc = JoinStructures::Cast(c);
		rtn.first = assoc.srcJoinStructures_end();
		rtn.second = assoc.dstJoinStructures_end();	
	}
	return rtn;
};

/** \brief Helper function for creating a composition connection without having to cast the Ports as specific types.
    \param [in] type Class of connection to be created
    \param [in] src Source port
    \param [in] dst Destination port
    \param [in] parent DesignContainer parent for the new connection
    \return void
*/

void createComposition(const Uml::Class type,const Port& src,const Port& dst,const DesignContainer& parent) {
	if (type == PowerFlow::meta) {
		PowerFlow newPF = PowerFlow::Create(parent);
		newPF.srcPowerFlow_end() = PowerPortType::Cast(src);
		newPF.dstPowerFlow_end() = PowerPortType::Cast(dst);
	} else if (type == InformationFlow::meta) {
		InformationFlow assoc = InformationFlow::Create(parent);
		assoc.srcInformationFlow_end() = SignalPortType::Cast(src);
		assoc.dstInformationFlow_end() = SignalPortType::Cast(dst);
	} else if (type == ExecutionAssignment::meta) {
		ExecutionAssignment assoc = ExecutionAssignment::Create(parent);
		assoc.srcExecutionAssignment_end() = ExecutionAssignment_Members_Base::Cast(src);
		assoc.dstExecutionAssignment_end() = ExecutionAssignment_Members_Base::Cast(dst);
	} else if (type == JoinStructures::meta) {
		JoinStructures assoc = JoinStructures::Create(parent);
		assoc.srcJoinStructures_end() = StructuralPortType::Cast(src);
		assoc.dstJoinStructures_end() = StructuralPortType::Cast(dst);
	}
};

/** \brief Connection discovery and establishment for DesignContainer (Design Space) models.
    \param [in] dc DesignContainer (Design Space) context to process
    \return void
*/

void cu_DesignContainer(const DesignContainer& dc) {
	// Okay, we need to get all DesignEntity-type children.
	//   This includes Components, ComponentAssemblies, and DesignContainers.
	set<DesignEntity> deKids;
	{
		set<Component> cKids = dc.Component_children();
		for (set<Component>::const_iterator i = cKids.begin(); i != cKids.end(); i++)
			deKids.insert(*i);

		set<ComponentAssembly> caKids = dc.ComponentAssembly_children();
		for (set<ComponentAssembly>::const_iterator i = caKids.begin(); i != caKids.end(); i++)
			deKids.insert(*i);

		set<DesignContainer> dcKids = dc.DesignContainer_children();
		for (set<DesignContainer>::const_iterator i = dcKids.begin(); i != dcKids.end(); i++)
			deKids.insert(*i);
	}

	set<Property> dePropKids = dc.Property_children();
	set<Parameter> deParamKids = dc.Parameter_children();

	// For each, iterate over its "Property" and "Parameter" objects.
	// Form a list of the names of these objects using the first object we parse.
	// If a later option does not share that entry, remove it.
	// At the end, we should have a list of "Property" objects shared by all
	map<string,set<Property>> properties;
	map<string,set<Parameter>> parameters;
	for (set<DesignEntity>::const_iterator i = deKids.begin(); i != deKids.end(); i++) {
		DesignEntity dei(*i);
		set<Property> deiPropertySet;
		set<Parameter> deiParameterSet;
		if (Uml::IsDerivedFrom(dei.type(),DesignElement::meta)) {
			deiPropertySet = DesignElement::Cast(dei).Property_children();
			deiParameterSet = DesignElement::Cast(dei).Parameter_children();
		} else if (Uml::IsDerivedFrom(dei.type(),DesignContainer::meta)) {
			deiPropertySet = DesignContainer::Cast(dei).Property_children();
			deiParameterSet = DesignContainer::Cast(dei).Parameter_children();
		}
		
		for (set<Property>::const_iterator j = deiPropertySet.begin(); j != deiPropertySet.end(); j++)
			properties[(*j).name()].insert(*j);
		for (set<Parameter>::const_iterator j = deiParameterSet.begin(); j != deiParameterSet.end(); j++)
			parameters[(*j).name()].insert(*j);
	}

	int numDEkids = deKids.size();
	/*for (map<string,set<Property>>::const_iterator i = properties.begin(); i != properties.end(); i++) {
		int size = (*i).second.size();
		if ( size > propHighest )
			propHighest = size;
	}*/
	for (map<string,set<Property>>::const_iterator i = properties.begin(); i != properties.end(); i++) {
		if ( (*i).second.size() < numDEkids )
			properties.erase(i);
	}
	int paramHighest = deKids.size();
	/*for (map<string,set<Parameter>>::const_iterator i = parameters.begin(); i != parameters.end(); i++) {
		int size = (*i).second.size();
		if ( size > paramHighest )
			paramHighest = size;
	}*/
	for (map<string,set<Parameter>>::const_iterator i = parameters.begin(); i != parameters.end(); i++) {
		if ( (*i).second.size() < numDEkids )
			parameters.erase(i);
	}
	
	for (map<string,set<Property>>::const_iterator i = properties.begin(); i != properties.end(); i++) {
		// Okay, this is one of our common properties. Let's see if we have one at the top level already.
		string name = (*i).first;
		ValueFlowTarget target;
		for (set<Property>::const_iterator j = dePropKids.begin(); j != dePropKids.end() && !target; j++) {
			Property pj(*j);
			if (pj.name() == name)
				target = pj;
		}

		// Okay, if it exists, is this a Compound container? 
		//   If so, we actually need to target the SimpleFormula that feeds it.
		if (target && dc.ContainerType() == "Compound") {
			set<ValueFlow> vfSet = target.srcValueFlow();
			for (set<ValueFlow>::const_iterator j = vfSet.begin(); j != vfSet.end(); j++) {
				ValueFlow vfj(*j);
				ValueFlowTarget vft(vfj.srcValueFlow_end());
				if (vft.type() == SimpleFormula::meta)
					target = vft;
				/*vft = vfj.dstValueFlow_end();
				if (Udm::IsDerivedFrom(vft.type(),SimpleFormula::meta))
					target = vft;*/
			}
		}

		// Does it not exist? Let's make one.
		if (!target) {
			Property topLevelProp = Property::Create(dc);
			topLevelProp.name() = name;
			if (dc.ContainerType() == "Compound") {
				// If this is a Compound, we need to create a SimpleFormula of "Addition" and target it instead.
				SimpleFormula sf = SimpleFormula::Create(dc);
				
				sf.Method() = "Addition";
				ValueFlow newVF = ValueFlow::Create(dc);
				newVF.srcValueFlow_end() = sf;
				newVF.dstValueFlow_end() = topLevelProp;
				
				target = sf;
			} else
				target = topLevelProp;
		}

		// Okay, now we have a target. Let's hook upexists all the sources.
		//   Be sure to check that they're not already hooked up.
		set<Property> propSet = (*i).second;
		// Iterate over all instances of this property that are within the child components
		for (set<Property>::const_iterator j = propSet.begin(); j != propSet.end(); j++) {
			Property pj(*j);
			set<ValueFlow> vfSet = pj.dstValueFlow();
			bool exists = false;
			for (set<ValueFlow>::const_iterator k = vfSet.begin(); k != vfSet.end() && !exists; k++) {
				ValueFlow vfk(*k);
				if ( ValueFlowTarget::Cast(vfk.dstValueFlow_end()) == target )
					exists = true;
			}
			if (!exists) {
				ValueFlow newVF = ValueFlow::Create(dc);
				newVF.dstValueFlow_end() = target;
				newVF.srcValueFlow_end() = pj;
			}
		}
	}
	for (map<string,set<Parameter>>::const_iterator i = parameters.begin(); i != parameters.end(); i++) {
		// Okay, this is one of our common parameters. Let's see if we have one at the top level already.
		string name = (*i).first;
		Parameter topLevelParam;
		for (set<Parameter>::const_iterator j = deParamKids.begin(); j != deParamKids.end() && !topLevelParam; j++) {
			Parameter pj(*j);
			if (pj.name() == name)
				topLevelParam = pj;
		}
		// Does it not exist? Let's make one.
		if (!topLevelParam) {
			topLevelParam = Parameter::Create(dc);
			topLevelParam.name() = name;
		}

		// Okay, now we have a target. Let's hook upexists all the sources.
		//   Be sure to check that they're not already hooked up.
		set<Parameter> paramSet = (*i).second;
		for (set<Parameter>::const_iterator j = paramSet.begin(); j != paramSet.end(); j++) {
			Parameter pj(*j);
			set<ValueFlow> vfSet = pj.srcValueFlow();
			bool exists = false;
			for (set<ValueFlow>::const_iterator k = vfSet.begin(); k != vfSet.end() && !exists; k++) {
				ValueFlow vfk(*k);
				if ( ValueFlowTarget::Cast(vfk.srcValueFlow_end()) == topLevelParam )
					exists = true;
			}
			if (!exists) {
				ValueFlow newVF = ValueFlow::Create(dc);
				newVF.srcValueFlow_end() = topLevelParam;
				newVF.dstValueFlow_end() = pj;
			}
		}

	}

	if (dc.ContainerType() == "Alternative") {
		// Let's try to handle Ports now.
		//   First, let's find the common ports.
		//   Then, let's see if any of them are already connected to another port at this level (exemplar).
		//   If so, then take the other ports of this same kind and hook them up the same way.
		map<string,set<Port>> deiPorts;
		for (set<DesignEntity>::const_iterator i = deKids.begin(); i != deKids.end(); i++) {
			DesignEntity dei(*i);
			set<Port> deiPortSet;
			if (Uml::IsDerivedFrom(dei.type(),DesignElement::meta))
				deiPortSet = DesignElement::Cast(dei).Port_kind_children();
			else if (Uml::IsDerivedFrom(dei.type(),DesignContainer::meta))
				deiPortSet = DesignContainer::Cast(dei).Port_kind_children();
		
			for (set<Port>::const_iterator j = deiPortSet.begin(); j != deiPortSet.end(); j++)
				deiPorts[(*j).name()].insert(*j);
		}
		for (map<string,set<Port>>::const_iterator i = deiPorts.begin(); i != deiPorts.end(); i++) {
			if ( (*i).second.size() < numDEkids )
				deiPorts.erase(i);
		}
	
		// Okay, so for each "Common" port, we need to find a port at the DC level that corresponds.
		//   The only way to really do this is to check for a connection to/from one of the "common" ports.
		for (map<string,set<Port>>::const_iterator i = deiPorts.begin(); i != deiPorts.end(); i++) {
			// piSet = Ports that each component has & share a common name + type.
			set<Port> piSet( (*i).second );

			// Now we need to find exemplar connections.
			// An exemplar needs to be a Port from piSet that's connected to a port that's a child of the DesignContainer.
			// Once we find one, we'll note whether it was in- or out-going, as well as its type.
			// Finally, we'll iterate over the exemplars and make sure each other port in the set has a matching connection.
			struct ExemplarConnStats { 
				Uml::Class kind;
				Port dcLevelPort;
				bool dePortAsSrc;  // These two are...
				bool dcPortAsSrc;  // ...mutually exclusive
			};
			set<ExemplarConnStats*> exemplarConns;

			// This part sucks. We have to get every Composition connection in the DesignContainer.
			// Then we'll see if one of our ports is either a SRC or DST.
			set<Composition> dcCompositionKids = dc.Composition_kind_children();
			for (set<Composition>::const_iterator j = dcCompositionKids.begin(); j != dcCompositionKids.end(); j++) {
				Composition cj(*j);
				pair<Object,Object> ends_ = getCompositionEnds(cj);
				pair<Port,Port> ends(Port::Cast(ends_.first),Port::Cast(ends_.second));
				if ( !ends.first || !ends.second )
					continue;

				// Okay, we have the ports for this conn. We need to iterate over "piSet" to see if any of them match.
				for (set<Port>::const_iterator k = piSet.begin(); k != piSet.end(); k++) {
					Port pk(*k);
					bool dePortAsSrc = (ends.first == pk);
					bool dcPortAsSrc = (ends.second == pk);
					if (dePortAsSrc || dcPortAsSrc) {
						ExemplarConnStats* ecs = new ExemplarConnStats;
						ecs->dePortAsSrc = dePortAsSrc;
						ecs->dcPortAsSrc = dcPortAsSrc;
						if (dePortAsSrc)
							ecs->dcLevelPort = ends.second;
						else
							ecs->dcLevelPort = ends.first;
						ecs->kind = cj.type();
						exemplarConns.insert(ecs);
					}
				}
			}

			// Okay, let's iterate over our exemplar examples and make sure that each
			//   port in "piSet" (our common component-level ports) has one.
			for (set<ExemplarConnStats*>::const_iterator j = exemplarConns.begin(); j != exemplarConns.end(); j++) {
				ExemplarConnStats* ecs(*j);
				for (set<Port>::const_iterator k = piSet.begin(); k != piSet.end(); k++) {
					Port src, dst;
					if (ecs->dcPortAsSrc) {
						src = ecs->dcLevelPort;
						dst = *k;
					} else {
						src = *k;
						dst = ecs->dcLevelPort;
					}
					// Check if connection exists.
					set<Object> objSet = src.GetAdjacentUniqueObjects(dst.type(),ecs->kind);
					bool exists = false;
					for (set<Object>::const_iterator m = objSet.begin(); m != objSet.end() && !exists; m++)
						if ( *m == dst )
							exists = true;

					// If it doesn't exist, create it.
					if (!exists)
						createComposition(ecs->kind,src,dst,dc);
				}
			}
		}
	}
};