/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#ifndef CYPHYDSECONVERTER_CPP
#define CYPHYDSECONVERTER_CPP

#include "CyPhyDSEConverter.h"
#include "Uml.h"
#include "UmlExt.h"
#include <queue>
#include <UdmGme.h>


CyPhyDSEConverter::CyPhyDSEConverter(CyPhyML::RootFolder &_root):rootFolder(_root),genCA(false)
{
	//std::map<Uml::Class, Uml::AssociationRole> refMemo;
}

void CyPhyDSEConverter::initCopyMap()
{
	std::queue<Udm::Object> dsChildren;
	dsChildren.push(rootFolder);
	while (!dsChildren.empty())
	{
		Udm::Object child = dsChildren.front();
		dsChildren.pop();
		std::set<Udm::Object> childObjects = child.GetChildObjects();
		for (auto it = childObjects.begin(); it != childObjects.end(); it++)
		{
			dsChildren.push(*it);
		}
		Uml::Class type = child.type();
		if (type == CyPhyML::ComponentRef::meta)
		{
			Udm::Object comp = CyPhyML::ComponentRef::Cast(child).ref();
			if (comp)
			{
				copyMap[comp] = comp;
				std::set<Udm::Object> compChildren = comp.GetChildObjects();
				std::for_each(compChildren.begin(), compChildren.end(), [&](const Udm::Object& o) 
				{ 
					copyMap[o] = o; 
				}
				);
			}
		}
	}
}

void CyPhyDSEConverter::initCopyMap(const Udm::Object &rootObj)
{
	std::queue<Udm::Object> dsChildren;
	dsChildren.push(rootObj);
	while (!dsChildren.empty())
	{
		Udm::Object child = dsChildren.front();
		dsChildren.pop();
		std::set<Udm::Object> childObjects = child.GetChildObjects();
		for (auto it = childObjects.begin(); it != childObjects.end(); it++)
		{
			dsChildren.push(*it);
		}
		Uml::Class type = child.type();
		if (type == CyPhyML::ComponentRef::meta)
		{
			Udm::Object comp = CyPhyML::ComponentRef::Cast(child).ref();
			if (comp)
			{
				copyMap[comp] = comp;
				std::set<Udm::Object> compChildren = comp.GetChildObjects();
				std::for_each(compChildren.begin(), compChildren.end(), [&](const Udm::Object& o) 
				{ 
					copyMap[o] = o; 
				}
				);
			}
		}
	}
}

CyPhyML::DesignContainer CyPhyDSEConverter::createNewDesignContainer(CyPhyML::DesignSpace &dsFolder, const std::string &name)
{
	new_dc = CyPhyML::DesignContainer::Create(dsFolder);
	new_dc.name() = name;
	new_dc.ContainerType() = "Compound";
	return new_dc;
}

CyPhyML::DesignContainer CyPhyDSEConverter::createNewDesignContainer(CyPhyML::DesignContainer &dc, const std::string &name, bool isAlt)
{
	new_dc = CyPhyML::DesignContainer::Create(dc);
	new_dc.name() = name;
	if(isAlt)
		new_dc.ContainerType() = "Alternative";
	else
		new_dc.ContainerType() = "Compound";
	return new_dc;
}

CyPhyML::ComponentAssembly CyPhyDSEConverter::createNewCA(Udm::Object &parentObj, const std::string &name)
{
	if(Uml::IsDerivedFrom(parentObj.type(), CyPhyML::ComponentAssembly::meta))
		new_ca = CyPhyML::ComponentAssembly::Create(CyPhyML::ComponentAssembly::Cast(parentObj));
	else if(Uml::IsDerivedFrom(parentObj.type(), CyPhyML::DesignContainer::meta))
		new_ca = CyPhyML::ComponentAssembly::Create(CyPhyML::DesignContainer::Cast(parentObj));
	else if(Uml::IsDerivedFrom(parentObj.type(), CyPhyML::TestBench::meta))
	{
		CyPhyML::TestBench tb = CyPhyML::TestBench::Cast(parentObj);
		new_ca = CyPhyML::ComponentAssembly::Create(tb);
	}
	else
		return new_ca;

	new_ca.name() = name;
	genCA = true;
	return new_ca;
}

void CyPhyDSEConverter::setNewDesignContainer(CyPhyML::DesignContainer &dc)
{
	new_dc = dc;
	genCA = false;
}

void CyPhyDSEConverter::setNewCA(CyPhyML::ComponentAssembly &ca)
{
	new_ca = ca;
	genCA = true;
}

void CyPhyDSEConverter::createNewObject(const Udm::Object &from_obj, bool delObj)
{
	if(!Uml::IsDerivedFrom(from_obj.type(), CyPhyML::DesignEntity::meta))
		return;
	
	if(delObj)
		tobeDelObjs.insert(from_obj);
	if(!genCA)
		createNewObject(from_obj, new_dc);
	else
		createNewObject(from_obj, new_ca);
}

void CyPhyDSEConverter::createNewObject(const Udm::Object &from_obj, Udm::Object &to_parent)
{
	std::string status = "Create new object for: "+UdmUtil::ExtractName(from_obj); 
	prgDlg.SetProgress(status.c_str());

	Uml::Class objType = from_obj.type();
	if(objType == CyPhyML::StructuralInterface::meta)
	{
		CyPhyML::StructuralInterface si = CyPhyML::StructuralInterface::Cast(from_obj);
		CyPhyML::StructuralInterfaceForwarder sifor = CyPhyML::StructuralInterfaceForwarder::Create(to_parent);
		sifor.name() = si.name();
		sifor.position() = si.position();
		copyMap[si] = sifor;
		return;
	}

	if(objType == CyPhyML::ComponentRef::meta)
	{
		CyPhyML::ComponentRef from_ref = CyPhyML::ComponentRef::Cast(from_obj);
		CyPhyML::ComponentRef new_ref = CyPhyML::ComponentRef::Create(to_parent);
		new_ref.name() = from_ref.name();
		new_ref.ref() = from_ref.ref();
		new_ref.position() = from_ref.position();
		copyMap[from_ref] = new_ref;
	}
	/*else if(objType == CyPhyML::ComponentAssembly::meta)
	{
		createCACopy(CyPhyML::ComponentAssembly::Cast(from_obj), to_parent); 
	}*/
	else
	{
		Udm::Object to_obj;
		map<int, Udm::Object> fromArchId2FromObj;
		int arch_level = 0;	
		if(from_obj.isInstance()|| from_obj.isSubtype())
		{
			Udm::Object arch_com = from_obj.archetype();
			arch_level++;
			while(arch_com.isInstance())
			{
				arch_com = arch_com.archetype();
				arch_level++;
			}
			makeArch2InstMap(from_obj, fromArchId2FromObj, arch_level);
			to_obj = to_parent.CreateObject(objType, arch_com, from_obj.isSubtype());
			copyMap[from_obj] = to_obj;
			updateUdmCopySubtype(from_obj, to_obj);
			copyUdmObjAttributes(from_obj, to_obj, fromArchId2FromObj, true);
		}
		else
		{
			to_obj = to_parent.CreateObject(objType);
			copyMap[from_obj] = to_obj;
			UdmUtil::copy_assoc_map _tmpMap;
			UdmUtil::CopyObjectHierarchy(from_obj.__impl(), to_obj.__impl(), from_obj.__impl()->__getdn(), _tmpMap);
			to_obj.__impl()->CopyAttributesFrom(from_obj.__impl());
			for(auto it=_tmpMap.begin();it!=_tmpMap.end();++it)
			{
				std::string from_name = (*it).first.getPath2();
				std::string to_name = (*it).second.getPath2();
				copyMap[(*it).first] = (*it).second;
			}
		}
	}
}

void CyPhyDSEConverter::makeArch2InstMap(const Udm::Object &obj, map<int, Udm::Object> &fromArchId2FromObj, int arch_level)
{
	int al = arch_level;
	Udm::Object arch_obj = obj.archetype();
	al--;
	while(al>0)
	{
		arch_obj = arch_obj.archetype();
		al--;
	}
	fromArchId2FromObj[arch_obj.uniqueId()]=obj;

	set<Udm::Object> &objs = obj.GetChildObjects();
	for(set<Udm::Object>::iterator i=objs.begin();i!=objs.end();++i)
	{
		makeArch2InstMap(Udm::Object(*i), fromArchId2FromObj, arch_level);
	}
}

void CyPhyDSEConverter::copyUdmObjAttributes(const Udm::Object &from, Udm::Object &to, map<int, Udm::Object> &fromArchId2FromObj, bool makemap)
{
	std::string status = "Copy attributes from: "+UdmUtil::ExtractName(from); 
	prgDlg.SetProgress(status.c_str());

	to.__impl()->CopyAttributesFrom(from.__impl());
	set<Udm::Object> &from_objs = from.GetChildObjects();
	set<Udm::Object> &to_objs = to.GetChildObjects();
//	ASSERT(from_objs.size()==to_objs.size());
	for(set<Udm::Object>::iterator i=to_objs.begin();i!=to_objs.end();++i)
	{
		Udm::Object to_obj = *i;
		Udm::Object to_arch = to_obj.archetype();
		Udm::Object from_obj;
		map<int, Udm::Object>::iterator pos = fromArchId2FromObj.find(to_arch.uniqueId());
		if(pos!=fromArchId2FromObj.end())
		{
			from_obj = (*pos).second;
		}
		if(from_obj!=Udm::null)
		{
			if(makemap)
			{
				const Uml::Class &objType = to_obj.type();
				if(!Uml::IsAssocClass(objType))
					copyMap[from_obj]=to_obj;
			}
			copyUdmObjAttributes(from_obj, to_obj, fromArchId2FromObj);
		}
	}
}

void CyPhyDSEConverter::gatherConnections_tobe_reconstruce(const CyPhyML::ComponentAssembly &cyphy_ca, const Udm::Object &focusObj)
{
	set<Udm::Object> childObjs = focusObj.GetChildObjects();
	for(set<Udm::Object>::iterator i=childObjs.begin();i!=childObjs.end();++i)
	{
		Udm::Object from_obj = *i;
		Uml::Class objType = from_obj.type();
		if(!Uml::IsAssocClass(objType))
			continue;

		std::string status = "Collect to-be-reconstruct association in: "+UdmUtil::ExtractName(focusObj); 
		prgDlg.SetProgress(status.c_str());

		CyPhyUtil::ComponentPortPair endpair = CyPhyUtil::getConnectionEnds(objType, from_obj);
		CyPhyUtil::ComponentPort fromEnd1 = endpair.src;
		Udm::Object end1_parent = fromEnd1.port.GetParent();
		if(end1_parent==cyphy_ca)
		{
			tobeReconnecteLinks.insert(from_obj);
			continue;
		}

		CyPhyUtil::ComponentPort fromEnd2 = endpair.dst;
		Udm::Object end2_parent = fromEnd2.port.GetParent();
		if(end2_parent==cyphy_ca)
		{
			tobeReconnecteLinks.insert(from_obj);
		}
	}
}

void CyPhyDSEConverter::createCACopy(const CyPhyML::DesignElement &cyphy_ca, Udm::Object &to_parent)
{
	std::string status = "Processing ComponentAssembly: "+(std::string)cyphy_ca.name(); 
	prgDlg.SetProgress(status.c_str());
	set<Udm::Object> assocs;
	set<Udm::Object> childObjs = cyphy_ca.GetChildObjects();
	for(set<Udm::Object>::iterator i=childObjs.begin();i!=childObjs.end();++i)
	{
		Udm::Object from_obj = *i;
		Uml::Class objType = from_obj.type();
		if(Uml::IsAssocClass(objType))
			assocs.insert(from_obj);
		else
		{
			if(from_obj.isInstance() || from_obj.isSubtype())
				createNewObject(from_obj, to_parent);
			else
			{
				if(Uml::IsDerivedFrom(objType, CyPhyML::ComponentAssembly::meta))
				{
					Udm::Object to_obj = to_parent.CreateObject(objType);
					copyMap[from_obj] = to_obj;
					to_obj.__impl()->CopyAttributesFrom(from_obj.__impl());
					createCACopy(CyPhyML::DesignElement::Cast(from_obj), to_obj);
				}
				else
					createNewObject(from_obj, to_parent);
			}
		}
	}
	for(set<Udm::Object>::iterator ai=assocs.begin();ai!=assocs.end();++ai)
	{
		Udm::Object fromAssoc = *ai;
		copyAssociation(fromAssoc, to_parent);
	}
}

void CyPhyDSEConverter::reconstructExtractCAAssociations(Udm::Object &focusObj)
{
	for(auto it=tobeReconnecteLinks.begin();it!=tobeReconnecteLinks.end();++it)
	{
		Udm::Object from_obj = *it;
		Uml::Class objType = from_obj.type();

		CyPhyUtil::ComponentPortPair endpair = CyPhyUtil::getConnectionEnds(objType, from_obj);
		CyPhyUtil::ComponentPort fromEnd1 = endpair.src;
		CyPhyUtil::ComponentPort fromEnd2 = endpair.dst;
		
		Udm::Object toEnd1, toEnd2;
		CyPhyML::ComponentRef ref_parent1, ref_parent2;

		Udm::Object end1_parent = fromEnd1.port.GetParent();
		Udm::Object end2_parent = fromEnd2.port.GetParent();
		if(end1_parent==focusObj)
		{
			toEnd1 = fromEnd1.port;
			toEnd2 = getMappingObject(fromEnd2.port);
		}
		else if(end2_parent==focusObj)
		{
			toEnd2 = fromEnd2.port;
			toEnd1 = getMappingObject(fromEnd1.port);
		}
		else
		{
			toEnd1 = getMappingObject(fromEnd1.port);
			if(!toEnd1)
			{
				toEnd1 = fromEnd1.port;
				if(fromEnd1.port_ref_parent)
					ref_parent1 = CyPhyML::ComponentRef::Cast(fromEnd1.port_ref_parent);
			}
			toEnd2 = getMappingObject(fromEnd2.port);
			if(!toEnd2)
			{
				toEnd2 = fromEnd2.port;
				if(fromEnd2.port_ref_parent)
					ref_parent2 = CyPhyML::ComponentRef::Cast(fromEnd2.port_ref_parent);
			}
		}

		Udm::Object new_assoc;
		if(objType==CyPhyML::ModelicaPort2ComponentPortPowerFlow::meta)
			new_assoc = CyPhyUtil::reconstructConnection(CyPhyML::PowerFlow::meta, focusObj, toEnd1, ref_parent1, toEnd2, ref_parent2);
		else
			new_assoc = CyPhyUtil::reconstructConnection(objType, focusObj, toEnd1, ref_parent1, toEnd2, ref_parent2);

	}
}

void CyPhyDSEConverter::createDSFromCA(const CyPhyML::DesignElement &cyphy_ca)
{
	if(cyphy_ca.isInstance() || cyphy_ca.isSubtype() )
	{
		CyPhyML::ComponentAssembly newca = CyPhyML::ComponentAssembly::Create(new_dc);
		newca.name() = (std::string)cyphy_ca.name()+" DesignContainer";
		createNewObject(cyphy_ca, newca);
	}		
	else
		createCACopy(cyphy_ca, new_dc);
}

void CyPhyDSEConverter::copyAssociation(Udm::Object &fromAssoc, Udm::Object &toParent)
{
	Uml::Class assocType = fromAssoc.type();
	std::string status = "Create association in: "+UdmUtil::ExtractName(toParent); 
	prgDlg.SetProgress(status.c_str());

	CyPhyUtil::ComponentPortPair endpair = CyPhyUtil::getConnectionEnds(assocType, fromAssoc);
	CyPhyUtil::ComponentPort fromEnd1 = endpair.src;
	CyPhyUtil::ComponentPort fromEnd2 = endpair.dst;
	CyPhyUtil::ComponentPort toEnd1 = getMappingComponentPort(endpair.src);
	CyPhyUtil::ComponentPort toEnd2 = getMappingComponentPort(endpair.dst);
	CyPhyML::ComponentRef comref1, comref2;
	if(toEnd1.port_ref_parent && Uml::IsDerivedFrom(toEnd1.port_ref_parent.type(), CyPhyML::ComponentRef::meta))
		comref1 = CyPhyML::ComponentRef::Cast(toEnd1.port_ref_parent);
	if(toEnd2.port_ref_parent && Uml::IsDerivedFrom(toEnd2.port_ref_parent.type(), CyPhyML::ComponentRef::meta))
		comref2 = CyPhyML::ComponentRef::Cast(toEnd2.port_ref_parent);
	
	if(toEnd1.port==Udm::null || toEnd2.port==Udm::null)
		return;
	else 
		Udm::Object new_assoc = CyPhyUtil::reconstructConnection(assocType, toParent, toEnd1.port, comref1, toEnd2.port, comref2);
}

void CyPhyDSEConverter::createAssociation(Udm::Object &assocParent, Uml::Class &assocType,
										  std::string &roleName1, Udm::Object &end1,
										  std::string &roleName2, Udm::Object &end2)
{
	Udm::Object newAssoc = assocParent.CreateObject(assocType);
	Udm::Object::AssociationInfo assocInfo(assocType);
	assocInfo.strSrcRoleName = roleName1;
	assocInfo.strDstRoleName = roleName2;
	end1.CreateLink(newAssoc, assocInfo);
	assocInfo.strSrcRoleName = roleName2;
	assocInfo.strDstRoleName = roleName1;
	end2.CreateLink(newAssoc, assocInfo);	
}

void CyPhyDSEConverter::reconstructAssociations(Udm::Object &parentObj)
{
	Udm::Object new_parent;
	if(genCA)
		new_parent = new_ca;
	else
		new_parent = new_dc;

	//set<Udm::Object> childObjs = dc.GetChildObjects();
	set<Udm::Object> childObjs = parentObj.GetChildObjects();
	for(set<Udm::Object>::iterator i=childObjs.begin();i!=childObjs.end();++i)
	{
		Udm::Object from_obj = *i;
		Uml::Class objType = from_obj.type();
		if(!Uml::IsAssocClass(objType))
			continue;

		std::string status = "Create association in: "+UdmUtil::ExtractName(parentObj); 
		prgDlg.SetProgress(status.c_str());

		CyPhyUtil::ComponentPortPair endpair = CyPhyUtil::getConnectionEnds(objType, from_obj);
		CyPhyUtil::ComponentPort fromEnd1 = endpair.src;
		CyPhyUtil::ComponentPort fromEnd2 = endpair.dst;
		CyPhyUtil::ComponentPort toEnd1 = getMappingComponentPort(endpair.src);
		CyPhyUtil::ComponentPort toEnd2 = getMappingComponentPort(endpair.dst);
		CyPhyML::ComponentRef comref1, comref2;
		if(toEnd1.port_ref_parent && Uml::IsDerivedFrom(toEnd1.port_ref_parent.type(), CyPhyML::ComponentRef::meta))
			comref1 = CyPhyML::ComponentRef::Cast(toEnd1.port_ref_parent);
		if(toEnd2.port_ref_parent && Uml::IsDerivedFrom(toEnd2.port_ref_parent.type(), CyPhyML::ComponentRef::meta))
			comref2 = CyPhyML::ComponentRef::Cast(toEnd2.port_ref_parent);
		
		if(toEnd1.port==Udm::null && toEnd2.port==Udm::null)
			continue;
		else if(toEnd1.port!=Udm::null  && toEnd2.port!=Udm::null)
		{
			Udm::Object new_assoc;
			if(objType==CyPhyML::ModelicaPort2ComponentPortPowerFlow::meta)
				new_assoc = CyPhyUtil::reconstructConnection(CyPhyML::PowerFlow::meta, new_parent, toEnd1.port, comref1, toEnd2.port, comref2);
			else
				new_assoc = CyPhyUtil::reconstructConnection(objType, new_parent, toEnd1.port, comref1, toEnd2.port, comref2);
		}
		else if(toEnd1.port!=Udm::null  && toEnd2.port==Udm::null)
		{
			if(toEnd2.port_ref_parent==Udm::null && fromEnd2.port_ref_parent && Uml::IsDerivedFrom(fromEnd2.port_ref_parent.type(), CyPhyML::ComponentRef::meta))
				comref2 = CyPhyML::ComponentRef::Cast(fromEnd2.port_ref_parent);

			Udm::Object toEnd1_parent = toEnd1.port_ref_parent;
			if(toEnd1_parent==Udm::null)
				toEnd1_parent  = toEnd1.port.GetParent();
			
			CyPhyML::ComponentRef port_ref;
			Udm::Object toEnd1_port;
			if(toEnd1_parent != new_parent)
			{
				if(componentPortMap.find(fromEnd1)!=componentPortMap.end())
					toEnd1_port = componentPortMap[fromEnd1];	
				else
				{
					toEnd1_port = createNewPort(fromEnd1, new_parent, UdmUtil::ExtractName(fromEnd2.port)+"_"+UdmUtil::ExtractName(toEnd1_parent));	
					Udm::Object new_assoc1;
					if(objType==CyPhyML::ModelicaPort2ComponentPortPowerFlow::meta)
						new_assoc1 = CyPhyUtil::reconstructConnection(CyPhyML::PowerFlow::meta, new_parent, toEnd1.port, comref1, toEnd1_port, port_ref);
					else
						new_assoc1 = CyPhyUtil::reconstructConnection(objType, new_parent, toEnd1.port, comref1, toEnd1_port, port_ref);
				}
			}
			else
				toEnd1_port = toEnd1.port;
			Udm::Object new_assoc2 = CyPhyUtil::reconstructConnection(objType, parentObj, toEnd1_port, port_ref, fromEnd2.port, comref2);
		}
		else if(toEnd1.port==Udm::null  && toEnd2.port!=Udm::null)
		{
			if(toEnd1.port_ref_parent==Udm::null && fromEnd1.port_ref_parent && Uml::IsDerivedFrom(fromEnd1.port_ref_parent.type(), CyPhyML::ComponentRef::meta))
				comref1 = CyPhyML::ComponentRef::Cast(fromEnd1.port_ref_parent);

			Udm::Object toEnd2_parent = toEnd2.port_ref_parent;
			if(toEnd2_parent==Udm::null)
				toEnd2_parent  = toEnd2.port.GetParent();
			
			CyPhyML::ComponentRef port_ref;
			Udm::Object toEnd2_port;
			if(toEnd2_parent != new_parent)
			{
				if(componentPortMap.find(fromEnd2)!=componentPortMap.end())
					toEnd2_port = componentPortMap[fromEnd2];
				else
				{
					toEnd2_port = createNewPort(fromEnd2, new_parent, UdmUtil::ExtractName(fromEnd2.port)+"_"+UdmUtil::ExtractName(toEnd2_parent));	
					Udm::Object new_assoc1;
					if(objType==CyPhyML::ModelicaPort2ComponentPortPowerFlow::meta)
						new_assoc1 = CyPhyUtil::reconstructConnection(CyPhyML::PowerFlow::meta, new_parent, toEnd2_port, port_ref, toEnd2.port, comref2);
					else
						new_assoc1 = CyPhyUtil::reconstructConnection(objType, new_parent, toEnd2_port, port_ref, toEnd2.port, comref2);
				}
			}
			else
				toEnd2_port = toEnd2.port;
			CyPhyML::ComponentRef fromEnd1_ref;
			if(fromEnd1.port_ref_parent && Uml::IsDerivedFrom(fromEnd1.port_ref_parent.type(), CyPhyML::ComponentRef::meta))
				fromEnd1_ref = CyPhyML::ComponentRef::Cast(fromEnd1.port_ref_parent);
			Udm::Object new_assoc2 = CyPhyUtil::reconstructConnection(objType, parentObj, fromEnd1.port, fromEnd1_ref, toEnd2_port, port_ref);
		}
	}
}

Udm::Object CyPhyDSEConverter::createNewPort(const CyPhyUtil::ComponentPort &from, Udm::Object &to_parent, std::string port_name)
{
	long portId = from.port.uniqueId();
	char buffer [33];
	_itoa(portId, buffer, 10);
	if(port_name.empty())
#if _DEBUG
     // Good for developers
	 port_name = UdmUtil::ExtractName(from.port)+"_"+buffer;
#else
		port_name = UdmUtil::ExtractName(from.port);
#endif
	
	Uml::Class objType = from.port.type();
	if(objType == CyPhyML::StructuralInterface::meta)
	{
		CyPhyML::StructuralInterface si = CyPhyML::StructuralInterface::Cast(from.port);
		CyPhyML::StructuralInterfaceForwarder sifor = CyPhyML::StructuralInterfaceForwarder::Create(to_parent);
		sifor.name() = si.name();
		sifor.position() = si.position();
		componentPortMap[from] = sifor;
		return sifor;
	}

	Udm::Object to;
	map<int, Udm::Object> fromArchId2FromObj;
	int arch_level = 0;	
	if(from.port.isInstance()|| from.port.isSubtype())
	{
		Udm::Object arch_com = from.port.archetype();
		arch_level++;
		while(arch_com.isInstance())
		{
			arch_com = arch_com.archetype();
			arch_level++;
		}
		makeArch2InstMap(from.port, fromArchId2FromObj, arch_level);
		to = to_parent.CreateObject(objType, arch_com, from.port.isSubtype());
		
		updateUdmCopySubtype(from.port, to);
		copyUdmObjAttributes(from.port, to, fromArchId2FromObj, true);
	}
	else
	{
		to = to_parent.CreateObject(objType);
		UdmUtil::copy_assoc_map _tmpcopyMap;
		UdmUtil::CopyObjectHierarchy(from.port.__impl(), to.__impl(), from.port.__impl()->__getdn(), _tmpcopyMap);
		to.__impl()->CopyAttributesFrom(from.port.__impl());
	}

	to.SetStrValue("name", port_name);
	componentPortMap[from] = to;	
	return to;
}

void CyPhyDSEConverter::deleteConvertedObjs()
{
	for(set<Udm::Object>::iterator di=tobeDelObjs.begin();di!=tobeDelObjs.end();++di)
	{
		Udm::Object delObj = *di;
		if(delObj)
			delObj.DeleteObject();
	}
}

void CyPhyDSEConverter::startProgressBar()
{
	prgDlg.Create(IDD_PROGRESS_DIALOG);
	GetStatusDlg(&prgDlg);
	prgDlg.SetRange(100);
}

void CyPhyDSEConverter::endProgressBar()
{
	prgDlg.OnFinished();
}

Udm::Object CyPhyDSEConverter::getMappingObject(const Udm::Object &obj)
{
	Udm::Object ret;
	map<Udm::Object, Udm::Object>::iterator pos = copyMap.find(obj);
	if(pos!=copyMap.end())
	{
		ret = (*pos).second;
	}
	return ret;
}

CyPhyUtil::ComponentPort CyPhyDSEConverter::getMappingComponentPort(CyPhyUtil::ComponentPort &from_port)
{
	CyPhyUtil::ComponentPort ret;
	
	if(from_port.port_ref_parent!=Udm::null)
	{
		ret.port_ref_parent = getMappingObject(from_port.port_ref_parent);
		if(ret.port_ref_parent!=Udm::null)
		{
			if(Uml::IsDerivedFrom(ret.port_ref_parent.type(), CyPhyML::ComponentRef::meta))
				ret.port = from_port.port;
			else   //for example, replace the componentRef with its referred ComponentAssembly
				ret.port = getMappingObject(from_port.port);
		}
	}
	else
	{
		ret.port = getMappingObject(from_port.port);
	}

	return ret;
}

void CyPhyDSEConverter::updateUdmCopySubtype(const Udm::Object &from, Udm::Object &to)
{
	set<Udm::Object> from_objs = from.GetChildObjects();
	set<Udm::Object> to_objs = to.GetChildObjects();
	
	if(from_objs.size()==to_objs.size()) return;

	for(auto i=from_objs.begin();i!=from_objs.end();++i)
	{
		Udm::Object currObj = *i;
		if(currObj.isInstance() || currObj.isSubtype())
			continue;
		createNewObject(currObj, to);
	}
}

void CyPhyDSEConverter::reconstructReference(CyPhyML::DesignContainer &from_dc)
{
	set<CyPhyML::VisualConstraint> vcs = from_dc.VisualConstraint_kind_children();
	for(set<CyPhyML::VisualConstraint>::iterator it_vc = vcs.begin();it_vc!=vcs.end();++it_vc)
	{
		set<CyPhyML::ImpliesEnd> iends = (*it_vc).ImpliesEnd_kind_children();
		set<CyPhyML::ImpliesEnd>::iterator it_im;
		while(!iends.empty())
		{
			it_im = iends.begin();
			CyPhyML::ImpliesEnd iend = *it_im;
			iends.erase(it_im);
			if(Uml::IsDerivedFrom(iend.type(), CyPhyML::And_Or::meta))
			{
				set<CyPhyML::ImpliesEnd> and_or_iends = (CyPhyML::And_Or::Cast(iend)).ImpliesEnd_kind_children();
				iends.insert(and_or_iends.begin(), and_or_iends.end());
			}
			else
			{
				Udm::Object to_ref_obj = getMappingObject(iend);
				if(to_ref_obj==Udm::null) continue;

				if(Uml::IsDerivedFrom(iend.type(), CyPhyML::DesignEntityRef::meta))
				{				
					CyPhyML::DesignEntityRef from_dref = CyPhyML::DesignEntityRef::Cast(iend);
					CyPhyML::DesignEntityRef to_dref = CyPhyML::DesignEntityRef::Cast(to_ref_obj);
					CyPhyML::DesignEntity from_de = from_dref.ref();
					Udm::Object to_de_obj = getMappingObject(from_de);
					if(to_de_obj!=Udm::null)
						to_dref.ref() = CyPhyML::DesignEntity::Cast(to_de_obj);
					else
						to_dref.ref() = from_de;
				}
				else if(Uml::IsDerivedFrom(iend.type(), CyPhyML::NullOptionRef::meta))
				{
					CyPhyML::NullOptionRef from_nref = CyPhyML::NullOptionRef::Cast(iend);
					CyPhyML::NullOptionRef to_nref = CyPhyML::NullOptionRef::Cast(to_ref_obj);
					CyPhyML::DesignContainer from_dc = from_nref.ref();
					Udm::Object to_dc_obj = getMappingObject(from_dc);
					if(to_dc_obj!=Udm::null)
						to_nref.ref() = CyPhyML::DesignContainer::Cast(to_dc_obj);
					else
						to_nref.ref() = from_dc;
				}
				else if(Uml::IsDerivedFrom(iend.type(), CyPhyML::PropertyConstraintRef::meta))
				{
					CyPhyML::PropertyConstraintRef from_pref = CyPhyML::PropertyConstraintRef::Cast(iend);
					CyPhyML::PropertyConstraintRef to_pref = CyPhyML::PropertyConstraintRef::Cast(to_ref_obj);
					CyPhyML::PropertyConstraint from_prop = from_pref.ref();
					Udm::Object to_pc_obj = getMappingObject(from_prop);
					if(to_pc_obj!=Udm::null)
						to_pref.ref() = CyPhyML::PropertyConstraint::Cast(from_prop);
					else
						to_pref.ref() = from_prop;
				}
			}
		}
	}

	set<CyPhyML::DesignContainer> dcs = from_dc.DesignContainer_kind_children();
	for(set<CyPhyML::DesignContainer>::iterator it_dc=dcs.begin();it_dc!=dcs.end();++it_dc)
	{
		CyPhyML::DesignContainer dc = *it_dc;
		reconstructReference(dc);
	}
}

void CyPhyDSEConverter::reconstructReference(CyPhyML::DesignElement &delem)
{
	Udm::Object copyElem = getMappingObject(delem);
	if(!copyElem)
		return;
	if(!Uml::IsDerivedFrom(copyElem.type(), CyPhyML::DesignElement::meta))
		return;
	CyPhyML::DesignElement copy_delem = CyPhyML::DesignElement::Cast(copyElem);
	set<CyPhyML::ComponentRef> comRefs = delem.referedbyComponentRef();
	for(auto it=comRefs.begin();it!=comRefs.end();++it)
	{
		(*it).ref() = copy_delem;
	}
}

void CyPhyDSEConverter::addCopyMapInput(Udm::Object &from, Udm::Object &to)
{
	copyMap[from] = to;
}

#endif