/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "Equations.h"
#include "TBond.h"

#include <sstream>
#include <regex>
#include "EquHelper.h"
#include "UdmGme.h"

#define CAST_JUNCTION(x) (CyPhyML::Junction::Cast(*((x)->Ref)))

extern ScriptFile sfFile; // output script file and datastructure file
extern vector<TBGNode> BGNodeList; // flat bond graph
extern map<long,TBond> AllBonds; // contains all bonds (flat BG graph)

extern string StopTime;

Equations::Equations()
{
	if ((sfFile.szModelName != "") &&
			(!(this->equ_file_.is_open())))
	{
		// szModelName is specified
		string szEquFile;
		string szScriptFile;

		szScriptFile = sfFile.OutputDirectory + sfFile.szModelName;
		szScriptFile += "_equ.mos";

		remove(szScriptFile.c_str());
		this->equScriptFile.open(szScriptFile.c_str());

		equScriptFile << "// Generated script file for simulation" << endl;

		szEquFile = sfFile.OutputDirectory + sfFile.szModelName;
		//szEquFile += ".equ";												// extension
		szEquFile += "_equ.mo";												// extension
		remove(szEquFile.c_str());									// delete file if exists
		this->equ_file_.open(szEquFile.c_str());		// create the new file

		if (BGNodeList.size() != 0)
		{

			for (vector<TBGNode>::iterator itBGNodeList = BGNodeList.begin();
					 itBGNodeList != BGNodeList.end();
					 ++itBGNodeList)
			{
				// separate HybridJunctions, NonHybrisJunctions, and Elements
				if (Udm::IsDerivedFrom(itBGNodeList->Ref->type(),CyPhyML::BGElement::meta))
				{
					// element
					this->vElements.push_back(*itBGNodeList);

					if (Udm::IsDerivedFrom(itBGNodeList->Ref->type(), CyPhyML::BGModulatedElement::meta))
					{
						map<CyPhyML::Modulation, string> mod = EquHelper::GetModulations(CyPhyML::BGModulatedElement::Cast(*(itBGNodeList->Ref)));
						for (map<CyPhyML::Modulation, string>::const_iterator it = mod.begin();
							it != mod.end();
							++it)
						{
							modulation.insert(*it);
						}

						map<CyPhyML::MgaObject, CyPhyML::MgaObject> sig = EquHelper::GetSignalVariables(CyPhyML::BGModulatedElement::Cast(*(itBGNodeList->Ref)));
						for (map<CyPhyML::MgaObject, CyPhyML::MgaObject>::const_iterator it = sig.begin();
							it != sig.end();
							++it)
						{
							signals.insert(*it);
						}
					}
				}
				else if (Udm::IsDerivedFrom(itBGNodeList->Ref->type(),CyPhyML::Junction::meta))
				{
					// junction
					CyPhyML::Junction jJunction;
					jJunction = CyPhyML::Junction::Cast((*(itBGNodeList->Ref)));

					string scope = EquHelper::GetScope(jJunction);
					if (scope != "")
					{
						Scopes.insert(scope);
					}

					map<CyPhyML::Switching, string> sw = EquHelper::GetBooleanParameters(jJunction);
					for (map<CyPhyML::Switching, string>::const_iterator it = sw.begin();
						it != sw.end();
						++it)
					{
						switching.insert(*it);
					}

					map<CyPhyML::MgaObject, CyPhyML::MgaObject> sig = EquHelper::GetSignalVariables(CyPhyML::Junction::Cast(*(itBGNodeList->Ref)));
					for (map<CyPhyML::MgaObject, CyPhyML::MgaObject>::const_iterator it = sig.begin();
						it != sig.end();
						++it)
					{
						signals.insert(*it);
					}

					if ((string(jJunction.OnCondition()) == "") &&
							(string(jJunction.OffCondition()) == ""))
					{
						// non-hybrid
						this->vNonHybridJunctions.push_back(*itBGNodeList);
					}
					else
					{
						// hybrid
						this->vHybridJunctions.push_back(*itBGNodeList);
					}
				}
				else
				{
					// error this is not a BGNode
					ASSERT(0);
				}
			}

			equ_file_ << "model " << sfFile.szModelName << endl;
			equ_file_ << "annotation(experiment(";
			equ_file_ << "StartTime = 0.0, ";
			equ_file_ << "StopTime = " << StopTime << ", ";
			equ_file_ << "Tolerance = 1e-006), ";
			equ_file_ << "Documentation(info = \"" << endl;
			// additional info comes here.
			stringstream ss;

			ss << "<html>" << endl;
			ss << "<head>" << endl;
			ss << "<link href=\"special.css\" rel=\"stylesheet\" type=\"text/css\">" << endl;
			ss << "</head>" << endl;
			ss << "<body>" << endl;
			ss << "<h1><b>Configuration</b></h1>";
			ss << endl;
			
			ss << "<table id=\"configuration\">" << endl;

			ss << "<tr>" << endl;
			ss << "<td>";
			ss << "# of hybrid junctions: ";
			ss << "</td>" << endl;
			ss << "<td>";
			ss << vHybridJunctions.size();
			ss << "</td>" << endl;
			ss << "</tr>" << endl;
			ss << endl;
			ss << "<tr class=\"alt\">" << endl;
			ss << "<td>";
			ss << "# of non-hybrid junctions: ";
			ss << "</td>" << endl;
			ss << "<td>";
			ss << vNonHybridJunctions.size();
			ss << "</td>" << endl;
			ss << "</tr>" << endl;
			ss << endl;
			ss << "<tr>" << endl;
			ss << "<td>";
			ss << "# of elements: ";
			ss << "</td>" << endl;
			ss << "<td>" << endl;
			ss << vElements.size();
			ss << "</td>" << endl;
			ss << "</tr>" << endl;
			ss << endl;
			ss << "<tr class=\"alt\">" << endl;
			ss << "<td>" << endl;
			ss << "# of bonds: ";
			ss << "</td>" << endl;
			ss << "<td>" << endl;
			ss << AllBonds.size();
			ss << "</td>" << endl;
			ss << "</tr>" << endl;
			int nCase;
			nCase = (1 << vHybridJunctions.size());
			ss << "<tr>" << endl;
			ss << "<td>" << endl;
			ss << "# of cases: ";
			ss << "</td>" << endl;
			ss << "<td>" << endl;
			ss << nCase;
			ss << "</td>" << endl;
			ss << "</tr>" << endl;
			ss << "</table>" << endl;

			ss << "<h1><b>Flat Bond Graph of the system</b></h1>" << endl;
			string ret = sfFile.szGraphVizCmdFileName;
			tr1::regex rx("\\\\");
			string fmt = "/";
			ret = string(tr1::regex_replace(ret, rx, fmt));

			ss << "Note: please run the <a href=\"file:///" << ret << "\">command</a> first." << endl;
			ss << "<p align=\"center\">" << endl;

			ret = sfFile.szGraphVizPNGFileName;
			ret = string(tr1::regex_replace(ret, rx, fmt));
			ss << "<img src=\"file:///" << ret << "\">" << endl;

			ss << "</p>" << endl;

			// table
			ss << "<hr />" << endl;
			ss << "<h1><b>Variable map</b></h1>" << endl;
			ss << "<p>The table contains the mapping between the GME model and the equations.</p>" << endl;
			ss << "<p>Note: You can open the html file in the GME console, and the hyperlinks will work.</p>" << endl;
			ss << "<table id=\"variableList\" border=\"1\" align=\"center\">" << endl;
			ss << "<tr>" << endl;

			// CELL DEFINITIONS ===========================
			// ss << "<th>Variable name</th>" << endl;
			// ss << "<th>Gme object name</th>" << endl;
			// ss << "<th>Type</th>" << endl;
			// //ss << "<th>Link</th>" << endl;
			// ss << "<th>Path</th>" << endl;
			// ============================================

			ss << "<th align=\"center\">Variable name</th>" << endl;
			ss << "<th align=\"center\">Name in GME</th>" << endl;
			ss << "<th align=\"center\">Type</th>" << endl;
			//ss << "<th align=\"center\">Link<br/>(copy this link into the gme console)</th>" << endl;
			ss << "<th align=\"center\">Path</th>" << endl;
			ss << "</tr>" << endl;

			int rowId = 0;

			for (vector<TBGNode>::iterator itBGNodeList = BGNodeList.begin();
				 itBGNodeList != BGNodeList.end();
				 ++itBGNodeList)
			{
				ss << GenerateDocumentationLine(*(itBGNodeList->Ref), rowId++);
			}

			for (map<CyPhyML::Switching, string>::const_iterator it = switching.begin();
				it != switching.end();
				++it)
			{
				ss << GenerateDocumentationLine(it->first, rowId++);
			}

			for (map<CyPhyML::Modulation, string>::const_iterator it = modulation.begin();
				it != modulation.end();
				++it)
			{
				ss << GenerateDocumentationLine(it->first, rowId++);
			}

			for (map<CyPhyML::MgaObject, CyPhyML::MgaObject>::const_iterator it = signals.begin();
				it != signals.end();
				++it)
			{
				ss << GenerateDocumentationLine(it->first, rowId++);
				if (it->second != Udm::null)
				{
					if (it->second.type() == CyPhyML::ControlFunction::meta)
					{
						ss << GenerateDocumentationLine(it->second, rowId++);
					}
				}
			}



			ss << "</table>" << endl;

			ss << "</body></html>" << endl;

			// create html document
			string szHtmlFile = sfFile.OutputDirectory + sfFile.szModelName;
			szHtmlFile += "_equ.html";												// extension
			remove(szHtmlFile.c_str());									// delete file if exists
			ofstream shtml;

			shtml.open(szHtmlFile.c_str());		// create the new file
			shtml << ss.str();
			shtml.close();

			// replace " => \"
			ret = ss.str();
			rx = "\"";
			fmt = "\\\"";
			ret = string(tr1::regex_replace(ret, rx, fmt));

			equ_file_ << ret;
			equ_file_ << "\"));" << endl;
			this->GenerateVariableSpec();
			this->GenerateParameters();

			this->equ_file_ << "/* Swithing functions */" << endl;
			for (map<CyPhyML::Switching, string>::const_iterator it = switching.begin();
				it != switching.end();
				++it)
			{
				this->equ_file_ << "Boolean " << EquHelper::getFunctionName(it->first) << ";" << endl;
			}

			this->equ_file_ << "/* Modulation functions */" << endl;
			for (map<CyPhyML::Modulation, string>::const_iterator it = modulation.begin();
				it != modulation.end();
				++it)
			{
				this->equ_file_ << "Real " << EquHelper::getFunctionName(it->first) << ";" << endl;
			}

			this->equ_file_ << "/* Signal values */" << endl;
			for (map<CyPhyML::MgaObject, CyPhyML::MgaObject>::const_iterator it = signals.begin();
				it != signals.end();
				++it)
			{
				if (Udm::IsDerivedFrom(it->first.type(), CyPhyML::ValueFlowTarget::meta))
				{
					if (it->first.type() == CyPhyML::Parameter::meta)
					{
						CyPhyML::Parameter vft = CyPhyML::Parameter::Cast(it->first);
						this->equ_file_ << "Real " << EquHelper::getFunctionName(it->first) << " = " << vft.Value() << ";" << endl;
					}
					else if (it->first.type() == CyPhyML::Property::meta)
					{
						CyPhyML::Property vft = CyPhyML::Property::Cast(it->first);
						this->equ_file_ << "Real " << EquHelper::getFunctionName(it->first) << " = " << vft.Value() << ";" << endl;
					}
					//else if (it->first.type() == CyPhyML::CADProperty::meta)
					//{
					//	CyPhyML::CADProperty vft = CyPhyML::CADProperty::Cast(it->first);
					//	this->equ_file_ << "Real " << EquHelper::getFunctionName(it->first) << " = " << vft.Value() << ";" << endl;
					//}
				}
				else
				{
					this->equ_file_ << "Real " << EquHelper::getFunctionName(it->first) << "(start = 1);" << endl;
				}
			}

			this->GenerateFunctions();

			this->GenerateInitialValues();
			this->GenerateODEs();
			
			this->equ_file_ << "end " << sfFile.szModelName << ";";
		}

		equScriptFile << "//loadModel(Modelica);" << endl;
		equScriptFile << "loadFile(\"" << sfFile.szModelName << "_equ.mo\");" << endl;
		equScriptFile << "//dumpXMLDAE(" << sfFile.szModelName << ", storeInTemp = false, addMathMLCode=true);" << endl;
		equScriptFile << "//exportDAEtoMatlab(" << sfFile.szModelName << ");" << endl;
		// output stored in .mat file by default
		equScriptFile << "simulate(" << sfFile.szModelName << ");" << endl;
		if (Scopes.size() == 0)
		{
			equScriptFile << "plotAll(title=\"All variable of " << sfFile.szModelName << "\");" << endl;
		}
		else
		{
			string variables = "";

			for (set<string>::const_iterator it = Scopes.begin();
				it != Scopes.end();
				++it)
			{
				variables += *it;
				++it;
				if (it != Scopes.end())
				{
					variables += ", ";
				}
				--it;
			}

			equScriptFile << "plot3({" << variables << "}, title=\"" << sfFile.szModelName << "\");" << endl;
		}
		// if we need the csv.
		equScriptFile << "simulate(" << sfFile.szModelName << ", outputFormat=\"csv\");" << endl;
		equScriptFile << "// TODO: remove modelica generated files, which ones are not needed." << endl;

		string runEqus = sfFile.OutputDirectory + "run_" + sfFile.szModelName + "_ModelicaEqu.cmd";
		ofstream runequ;
		remove(runEqus.c_str());
		runequ.open(runEqus.c_str());

		runequ << ":: You need to install open modelica in order to run this script" << endl;
		runequ << "C:\\OpenModelica1.7.0\\bin\\omc.exe \"" << szScriptFile << '"' << endl;

		runequ.close();

	}
}

Equations::~Equations()
{
	if (this->equ_file_.is_open())
	{
		this->equ_file_.close();
	}

	if (this->equScriptFile.is_open())
	{

		this->equScriptFile.close();
	}
}


void Equations::GenerateVariableSpec()
{
	this->equ_file_ << "// Variable spec (BEGIN) " << endl;

	for (map<long, TBond>::iterator itBond = AllBonds.begin();
			 itBond != AllBonds.end();
			 ++itBond)
	{
		this->equ_file_ << "Real e" << itBond->second.bondID << "; ";
		this->equ_file_ << "Real f" << itBond->second.bondID << ";" << endl;
	}

	for (vector<TBGNode>::const_iterator it = vHybridJunctions.begin();
		it != vHybridJunctions.end();
		++it)
	{
		this->equ_file_ << "discrete Boolean " << \
			GetStateName(CyPhyML::Junction::Cast(*(it->Ref))) << \
			"(start = true);" << endl;

		this->equ_file_ << "Boolean " << \
			GetOnConditionName(CyPhyML::Junction::Cast(*(it->Ref))) << \
			"(start = true);" << endl;

		this->equ_file_ << "Boolean " << \
			GetOffConditionName(CyPhyML::Junction::Cast(*(it->Ref))) << \
			"(start = true);" << endl;
	}

	this->equ_file_ << "// Variable spec (END) " << endl;
}

void Equations::GenerateParameters()
{
	this->equ_file_ << "// Parameters (BEGIN) " << endl;

	if (BGNodeList.size() != 0)
	{
		for (vector<TBGNode>::iterator itBGNodeList = BGNodeList.begin();
			itBGNodeList != BGNodeList.end();
			++itBGNodeList)
		{
			if (Udm::IsDerivedFrom(itBGNodeList->Ref->type(), CyPhyML::BGElement::meta))
			{
				string value = CyPhyML::BGElement::Cast(*(itBGNodeList->Ref)).ParameterValue();

				this->equ_file_ << "Real " << GetParameterName(*itBGNodeList) << " = " << value << ";" << endl;
			}
		}
	}

	this->equ_file_ << "// Parameters (END) " << endl;
}

void Equations::GenerateInitialValues()
{
	this->equ_file_ << "// Initial values (BEGIN) " << endl;

	if (BGNodeList.size() != 0)
	{
		this->equ_file_ << "initial equation" << endl;

		for (vector<TBGNode>::iterator itBGNodeList = BGNodeList.begin();
			itBGNodeList != BGNodeList.end();
			++itBGNodeList)
		{
			if (Udm::IsDerivedFrom(itBGNodeList->Ref->type(), CyPhyML::Storage::meta))
			{
				string value = CyPhyML::Storage::Cast(*(itBGNodeList->Ref)).InitialValue();

				if (Udm::IsDerivedFrom(itBGNodeList->Ref->type(), CyPhyML::C::meta) ||
					Udm::IsDerivedFrom(itBGNodeList->Ref->type(), CyPhyML::MC::meta))
				{
					// ASSUMPTION: the elmenet has exactly one bond
					this->equ_file_ << "e" << itBGNodeList->BondMap.begin()->second.first->bondID << " = " << value << ";" << endl;
				}
				else if (Udm::IsDerivedFrom(itBGNodeList->Ref->type(), CyPhyML::I::meta) ||
					Udm::IsDerivedFrom(itBGNodeList->Ref->type(), CyPhyML::MI::meta))
				{
					// ASSUMPTION: the elmenet has exactly one bond
					this->equ_file_ << "f" << itBGNodeList->BondMap.begin()->second.first->bondID << " = " << value << ";" << endl;
				}
			}
		}
	}

	this->equ_file_ << "// Initial values (END) " << endl;
}

void Equations::GenerateFunctions()
{
	this->equ_file_ << "/* ControlFunctions (BEGIN) */" << endl;

	//set<CyPhyML::PhysicalComponent> parents;

	//if (BGNodeList.size() != 0)
	//{
	//	for (vector<TBGNode>::iterator itBGNodeList = BGNodeList.begin();
	//		itBGNodeList != BGNodeList.end();
	//		++itBGNodeList)
	//	{
	//		CyPhyML::PhysicalComponent parent = (*itBGNodeList).Ref->PhysicalComponent_parent();
	//		parents.insert(parent);
	//	}

	//	for (set<CyPhyML::PhysicalComponent>::iterator it = parents.begin();
	//		it != parents.end();
	//		++it)
	//	{
	//		set<CyPhyML::ControlFunction> cf = it->ControlFunction_kind_children();
	//		for (set<CyPhyML::ControlFunction>::iterator itCF = cf.begin();
	//			itCF != cf.end();
	//			++itCF)
	//		{
	for (map<CyPhyML::MgaObject, CyPhyML::MgaObject>::const_iterator it = signals.begin();
		it != signals.end();
		++it)
	{
		if (it->second != Udm::null)
		{
			if (it->second.type() == CyPhyML::ControlFunction::meta)
			{
				CyPhyML::ControlFunction itCF = CyPhyML::ControlFunction::Cast(it->second);
				equ_file_ << "/*=============== Function begins ==============*/" << endl;

				equ_file_ << "/* Original string" << endl;
				equ_file_ << itCF.Code() << endl;
				equ_file_ << "Original string*/" << endl;

				equ_file_ << "function " << EquHelper::getFunctionName(itCF) << endl;

				vector<string> inputs;
				vector<unsigned int> inIds;
				vector<string> outputs;
				vector<unsigned int> outIds;


				GetVariables(outputs, outIds, inputs, inIds, itCF);

				for (vector<string>::const_iterator itt = inputs.begin();
					itt != inputs.end();
					++itt)
				{
					equ_file_ << "  input Real " << *itt << ";" << endl;
				}

				for (vector<string>::const_iterator itt = outputs.begin();
					itt != outputs.end();
					++itt)
				{
					equ_file_ << "  output Real " << *itt << ";" << endl;
				}

				//equ_file_ << "algorithm" << endl;

				string code = itCF.Code();
				string keyword = "#Modelica";
				if (code.find(keyword) != string::npos)
				{
					equ_file_ << code.substr(code.find(keyword) + keyword.size()) << endl;
				}

				equ_file_ << "end " << EquHelper::getFunctionName(itCF) << ";" << endl;
				equ_file_ << "/*===============  Function ends  ==============*/" << endl;
			}
		}
	}

	this->equ_file_ << "/* ControlFunctions (END) */" << endl;
}

void Equations::GenerateODEs()
{
	int nCase;
	nCase = (1 << vHybridJunctions.size()) - 1;

	int width = 0;
	int number;

	// calculate the # of digits
	number = vHybridJunctions.size();
	while (number > 0)
	{
		width++;
		number = number / 10;
	}

	int widthCases = 0;

	// calculate the # of digits
	number = nCase;
	while (number > 0)
	{
		widthCases++;
		number = number / 10;
	}

	this->equ_file_ << "/*# of cases : ";
	this->equ_file_ << (nCase + 1);
	this->equ_file_ << endl;
	this->equ_file_ << endl;

	this->equ_file_ << "*/";
	this->equ_file_ << endl;

	this->equ_file_ << "equation";
	this->equ_file_ << endl;

	//for (; nCase >= 0; --nCase)
	{
		// prepare the current case/configuration
		vOnJunctions.clear();
		vOffJunctions.clear();

		vOnJunctions = vNonHybridJunctions;

		this->equ_file_ << endl;
		this->equ_file_ << "//Case #: ";
		this->equ_file_ << nCase;
		this->equ_file_ << endl;

		for (unsigned int i = 0; i < vHybridJunctions.size(); ++i)
		{
			// check whether the current junction is ON or OFF
			// nCases AND (junction's bit mask)
			if ((nCase & (1 << i)) > 0)
			{
				// this junction is ON
				this->vOnJunctions.push_back(vHybridJunctions[i]);
			}
			else
			{
				// this junction is OFF
				this->vOffJunctions.push_back(vHybridJunctions[i]);
			}
		}
		this->PrintConfiguration();
	}
}


void Equations::PrintConfiguration()
{
	stringstream szsEffortEqu;
	stringstream szsFlowEqu;

	stringstream szsEffortEquOFF;
	stringstream szsFlowEquOFF;

	set<CyPhyML::BGNode> sProcessedTwoPorts;

	this->equ_file_ << "/* Swithing functions */" << endl;
	for (map<CyPhyML::Switching, string>::const_iterator it = switching.begin();
		it != switching.end();
		++it)
	{
		this->equ_file_ << EquHelper::getFunctionName(it->first) << " = " << it->second << ";" << endl;
	}
	
	this->equ_file_ << "/* Modulation functions */" << endl;
	for (map<CyPhyML::Modulation, string>::const_iterator it = modulation.begin();
		it != modulation.end();
		++it)
	{
		equ_file_ << EquHelper::getFunctionName(it->first) << " = " << it->second << ";" << endl;
	}

	this->equ_file_ << "/* Signal values */" << endl;
	for (map<CyPhyML::MgaObject, CyPhyML::MgaObject>::const_iterator it = signals.begin();
		it != signals.end();
		++it)
	{
		if (it->second != Udm::null)
		{
			if (it->second.type() == CyPhyML::ControlFunction::meta)
			{
				string inputParameters = ""; // stores the function call input parameters
				CyPhyML::ControlFunction cf = CyPhyML::ControlFunction::Cast(it->second);

				set<CyPhyML::Signal2ControlFunction> s2cf = cf.srcSignal2ControlFunction();
				for (set<CyPhyML::Signal2ControlFunction>::const_iterator itt = s2cf.begin();
					itt != s2cf.end();
					++itt)
				{
					CyPhyML::Signal sig = itt->srcSignal2ControlFunction_end();

					inputParameters += EquHelper::getFunctionName(sig);
					++itt;
					if (itt != s2cf.end())
					{
						inputParameters += ", ";
					}
					--itt;
				}

				equ_file_ << EquHelper::getFunctionName(it->first) << " = " << EquHelper::getFunctionName(it->second) << "(" << inputParameters << ");" << endl;
			}
			else if (Udm::IsDerivedFrom(it->second.type(), CyPhyML::Junction::meta))
			{
				equ_file_ << EquHelper::getFunctionName(it->first) << " = " << EquHelper::GetVariable(CyPhyML::Junction::Cast(it->second)) << ";" << endl;
			}
			else
			{
				equ_file_ << EquHelper::getFunctionName(it->first) << " = " << EquHelper::getFunctionName(it->second) << ";" << endl;
			}
		}
	}


	this->equ_file_ << "/* System's equations */" << endl;
	for (vector<TBGNode>::iterator itvOn = this->vOnJunctions.begin();
			 itvOn != this->vOnJunctions.end();
			 ++itvOn)
	{
		// clear junction's equs
		szsEffortEqu.str("");
		szsFlowEqu.str("");
		szsEffortEquOFF.str("");
		szsFlowEquOFF.str("");

		for (map<int, pair<TBond*, EndPointType>>::iterator itBond = itvOn->BondMap.begin();
				 itBond != itvOn->BondMap.end();
				 ++itBond)
		{
			CyPhyML::BGNode bgSrc;
			CyPhyML::BGNode bgDst;

			bgSrc = itBond->second.first->src;
			bgDst = itBond->second.first->dst;

			if (itBond->second.second == SOURCE)
			{
				// this junction is the source of the bond

				if ((bgDst.type() == CyPhyML::Se::meta) ||
						(bgDst.type() == CyPhyML::MSe::meta))
				{
					this->equ_file_ << "e";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << " = ";
					if (bgDst.type() == CyPhyML::Se::meta)
					{
						// Se
						this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgDst)); // CyPhyML::Se::Cast(bgDst).ParameterValue();
					}
					else
					{
						// MSe
						// TODO: connected parameter
						//this->equ_file_ << "{ModulatedElement}";
						set<CyPhyML::Signal2BGMElement> s2bgm;
						s2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcSignal2BGMElement();
						set<CyPhyML::Modulation2BGMElement> m2bgm;
						m2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcModulation2BGMElement();

						//this->equ_file_ << "{";
						if (s2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
						}
						else if (m2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
						}
						else
						{
							this->equ_file_ << "Unhandled modulation";
						}
						//this->equ_file_ << "}";
					}
					this->equ_file_ << ";" << endl;
				}
				else if ((bgDst.type() == CyPhyML::Sf::meta) ||
								 (bgDst.type() == CyPhyML::MSf::meta))
				{
					this->equ_file_ << "f";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << " = ";
					if (bgDst.type() == CyPhyML::Sf::meta)
					{
						// Sf
						this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgDst)); // CyPhyML::Sf::Cast(bgDst).ParameterValue();
					}
					else
					{
						// MSf
						// TODO: connected parameter
						//this->equ_file_ << "{ModulatedElement}";
						set<CyPhyML::Signal2BGMElement> s2bgm;
						s2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcSignal2BGMElement();
						set<CyPhyML::Modulation2BGMElement> m2bgm;
						m2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcModulation2BGMElement();

						//this->equ_file_ << "{";
						if (s2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
						}
						else if (m2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
						}
						else
						{
							this->equ_file_ << "Unhandled modulation";
						}
						//this->equ_file_ << "}";
					}
					this->equ_file_ << ";" << endl;
				}
				else if ((bgDst.type() == CyPhyML::R::meta) ||
								 (bgDst.type() == CyPhyML::MR::meta))
				{
					this->equ_file_ << "e";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << " = ";
					this->equ_file_ << "f";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << "*";
					if (bgDst.type() == CyPhyML::R::meta)
					{
						// R
						this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgDst)); // CyPhyML::R::Cast(bgDst).ParameterValue();
					}
					else
					{
						// MR
						// TODO: connected parameter
						//this->equ_file_ << "{ModulatedElement}";
						set<CyPhyML::Signal2BGMElement> s2bgm;
						s2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcSignal2BGMElement();
						set<CyPhyML::Modulation2BGMElement> m2bgm;
						m2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcModulation2BGMElement();

						//this->equ_file_ << "{";
						if (s2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
						}
						else if (m2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
						}
						else
						{
							this->equ_file_ << "Unhandled modulation";
						}
						//this->equ_file_ << "}";
					}
					this->equ_file_ << ";" << endl;
				}
				else if ((bgDst.type() == CyPhyML::C::meta) ||
								 (bgDst.type() == CyPhyML::MC::meta))
				{
					//this->equ_file_ << "//e";
					//this->equ_file_ << itBond->second.first->bondID;
					//this->equ_file_ << "(0) = ";
					//this->equ_file_ << CyPhyML::Storage::Cast(bgDst).InitialValue();
					//this->equ_file_ << ";" << endl;

					this->equ_file_ << "f";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << " = ";
					this->equ_file_ << "der(e";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << ")*";
					if (bgDst.type() == CyPhyML::C::meta)
					{
						// C
						this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgDst)); // CyPhyML::C::Cast(bgDst).ParameterValue();
					}
					else
					{
						// MC
						// TODO: connected parameter
						//this->equ_file_ << "{ModulatedElement}";
						set<CyPhyML::Signal2BGMElement> s2bgm;
						s2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcSignal2BGMElement();
						set<CyPhyML::Modulation2BGMElement> m2bgm;
						m2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcModulation2BGMElement();

						//this->equ_file_ << "{";
						if (s2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
						}
						else if (m2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
						}
						else
						{
							this->equ_file_ << "Unhandled modulation";
						}
						//this->equ_file_ << "}";
					}
					this->equ_file_ << ";" << endl;
				}
				else if ((bgDst.type() == CyPhyML::I::meta) ||
								 (bgDst.type() == CyPhyML::MI::meta))
				{
					//this->equ_file_ << "//f";
					//this->equ_file_ << itBond->second.first->bondID;
					//this->equ_file_ << "(0) = ";
					//this->equ_file_ << CyPhyML::Storage::Cast(bgDst).InitialValue();
					//this->equ_file_ << ";" << endl;

					this->equ_file_ << "e";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << " = ";
					this->equ_file_ << "der(f";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << ")*";
					if (bgDst.type() == CyPhyML::I::meta)
					{
						// I
						this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgDst)); // CyPhyML::I::Cast(bgDst).ParameterValue();
					}
					else
					{
						// MI
						// TODO: connected parameter
						//this->equ_file_ << "{ModulatedElement}";
						set<CyPhyML::Signal2BGMElement> s2bgm;
						s2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcSignal2BGMElement();
						set<CyPhyML::Modulation2BGMElement> m2bgm;
						m2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcModulation2BGMElement();

						//this->equ_file_ << "{";
						if (s2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
						}
						else if (m2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
						}
						else
						{
							this->equ_file_ << "Unhandled modulation";
						}
						//this->equ_file_ << "}";
					}
					this->equ_file_ << ";" << endl;
				}
				else if ((bgDst.type() == CyPhyML::TF::meta) ||
								 (bgDst.type() == CyPhyML::MTF::meta))
				{
					if (sProcessedTwoPorts.end() == sProcessedTwoPorts.find(bgDst))
					{
						sProcessedTwoPorts.insert(bgDst);
						CyPhyML::TwoPort tp;
						tp = CyPhyML::TwoPort::Cast(bgDst);
						set<CyPhyML::BondE2J> sbe2j;
						sbe2j = tp.dstBondE2J();
						if (sbe2j.size() == 1)
						{
							// must have only one connction of this type Element2Junction

							// effort equation
							this->equ_file_ << "e";
							this->equ_file_ << GetBondId(*(sbe2j.begin()));
							this->equ_file_ << "*";
							if (tp.type() == CyPhyML::TF::meta)
							{
								this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgDst)); // tp.ParameterValue();
							}
							else
							{
								// TODO: connected parameter
								//this->equ_file_ << "{ModulatedElement}";
								set<CyPhyML::Signal2BGMElement> s2bgm;
								s2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcSignal2BGMElement();
								set<CyPhyML::Modulation2BGMElement> m2bgm;
								m2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcModulation2BGMElement();
								set<CyPhyML::Actuation2ModulatedTwoPort> mtp;
								mtp = CyPhyML::ModulatedTwoPort::Cast(bgDst).srcActuation2ModulatedTwoPort();

								//this->equ_file_ << "{";
								if (s2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
								}
								else if (m2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
								}
								else if (mtp.size() == 1)
								{
									this->equ_file_ << CyPhyML::Actuation::Cast((*mtp.begin()).srcActuation2ModulatedTwoPort_end()).name();
								}
								else
								{
									this->equ_file_ << "Unhandled modulation";
								}
								//this->equ_file_ << "}";
							}
							this->equ_file_ << " = e";
							this->equ_file_ << itBond->second.first->bondID;
							this->equ_file_ << ";" << endl;

							// flow equation
							this->equ_file_ << "f";
							this->equ_file_ <<  itBond->second.first->bondID;
							this->equ_file_ << "*";
							if (tp.type() == CyPhyML::TF::meta)
							{
								this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgDst)); // tp.ParameterValue();
							}
							else
							{
								// TODO: connected parameter
								//this->equ_file_ << "{ModulatedElement}";
								set<CyPhyML::Signal2BGMElement> s2bgm;
								s2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcSignal2BGMElement();
								set<CyPhyML::Modulation2BGMElement> m2bgm;
								m2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcModulation2BGMElement();
								set<CyPhyML::Actuation2ModulatedTwoPort> mtp;
								mtp = CyPhyML::ModulatedTwoPort::Cast(bgDst).srcActuation2ModulatedTwoPort();

								//this->equ_file_ << "{";
								if (s2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
								}
								else if (m2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
								}
								else if (mtp.size() == 1)
								{
									this->equ_file_ << CyPhyML::Actuation::Cast((*mtp.begin()).srcActuation2ModulatedTwoPort_end()).name();
								}
								else
								{
									this->equ_file_ << "Unhandled modulation";
								}
								//this->equ_file_ << "}";
							}
							this->equ_file_ << " = f";
							this->equ_file_ << GetBondId(*(sbe2j.begin()));
							this->equ_file_ << ";" << endl;

						}
					}
				}
				else if ((bgDst.type() == CyPhyML::GY::meta) ||
								 (bgDst.type() == CyPhyML::MGY::meta))
				{
					if (sProcessedTwoPorts.end() == sProcessedTwoPorts.find(bgDst))
					{
						sProcessedTwoPorts.insert(bgDst);
						CyPhyML::TwoPort tp;
						tp = CyPhyML::TwoPort::Cast(bgDst);
						set<CyPhyML::BondE2J> sbe2j;
						sbe2j = tp.dstBondE2J();
						if (sbe2j.size() == 1)
						{
							// must have only one connction of this type Element2Junction

							// first equation
							this->equ_file_ << "f";
							this->equ_file_ << itBond->second.first->bondID;
							this->equ_file_ << "*";
							if (tp.type() == CyPhyML::GY::meta)
							{
								this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgDst)); // tp.ParameterValue();
							}
							else
							{
								// TODO: connected parameter
								//this->equ_file_ << "{ModulatedElement}";
								set<CyPhyML::Signal2BGMElement> s2bgm;
								s2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcSignal2BGMElement();
								set<CyPhyML::Modulation2BGMElement> m2bgm;
								m2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcModulation2BGMElement();
								set<CyPhyML::Actuation2ModulatedTwoPort> mtp;
								mtp = CyPhyML::ModulatedTwoPort::Cast(bgDst).srcActuation2ModulatedTwoPort();

								//this->equ_file_ << "{";
								if (s2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
								}
								else if (m2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
								}
								else if (mtp.size() == 1)
								{
									this->equ_file_ << CyPhyML::Actuation::Cast((*mtp.begin()).srcActuation2ModulatedTwoPort_end()).name();
								}
								else
								{
									this->equ_file_ << "Unhandled modulation";
								}
								//this->equ_file_ << "}";
							}
							this->equ_file_ << " = e";
							this->equ_file_ << GetBondId(*(sbe2j.begin()));
							this->equ_file_ << ";" << endl;

							// second equation
							this->equ_file_ << "f";
							this->equ_file_ << GetBondId(*(sbe2j.begin()));
							this->equ_file_ << "*";
							if (tp.type() == CyPhyML::GY::meta)
							{
								this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgDst)); // tp.ParameterValue();
							}
							else
							{
									// TODO: connected parameter
								//this->equ_file_ << "{ModulatedElement}";
								set<CyPhyML::Signal2BGMElement> s2bgm;
								s2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcSignal2BGMElement();
								set<CyPhyML::Modulation2BGMElement> m2bgm;
								m2bgm = CyPhyML::BGModulatedElement::Cast(bgDst).srcModulation2BGMElement();
								set<CyPhyML::Actuation2ModulatedTwoPort> mtp;
								mtp = CyPhyML::ModulatedTwoPort::Cast(bgDst).srcActuation2ModulatedTwoPort();

								//this->equ_file_ << "{";
								if (s2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
								}
								else if (m2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
								}
								else if (mtp.size() == 1)
								{
									this->equ_file_ << CyPhyML::Actuation::Cast((*mtp.begin()).srcActuation2ModulatedTwoPort_end()).name();
								}
								else
								{
									this->equ_file_ << "Unhandled modulation";
								}
								//this->equ_file_ << "}";
							}
							this->equ_file_ << " = e";
							this->equ_file_ << itBond->second.first->bondID;
							this->equ_file_ << ";" << endl;

						}
					}
				}

				if (itvOn->Ref->type() == CyPhyML::OneJunction::meta)
				{
					if (isHybridJunction(*itvOn))
					{
						szsFlowEquOFF << "f";
						szsFlowEquOFF << itBond->second.first->bondID;
						szsFlowEquOFF << " = 0;" << endl;
					}

					szsEffortEqu << "-e";
					szsEffortEqu << itBond->second.first->bondID;
					szsEffortEqu << " ";

					if (*itBond != *(itvOn->BondMap.rbegin()))
					{
						// not the last bond
						szsFlowEqu << "f";
						szsFlowEqu << itBond->second.first->bondID;
						szsFlowEqu << " ";
						++itBond;
						szsFlowEqu << "= f" << itBond->second.first->bondID << ";" << endl;
						--itBond;
					}
					else
					{
						// last bond
						szsEffortEqu << "= 0;" << endl;
					}
				}
				else if (itvOn->Ref->type() == CyPhyML::ZeroJunction::meta)
				{
					if (isHybridJunction(*itvOn))
					{
						szsEffortEquOFF << "e";
						szsEffortEquOFF << itBond->second.first->bondID;
						szsEffortEquOFF << " = 0;" << endl;
					}

					szsFlowEqu << "-f";
					szsFlowEqu << itBond->second.first->bondID;
					szsFlowEqu << " ";

					if (*itBond != *(itvOn->BondMap.rbegin()))
					{
						// not the last bond
						szsEffortEqu << "e";
						szsEffortEqu << itBond->second.first->bondID;
						szsEffortEqu << " ";
						++itBond;
						szsEffortEqu << "= e" << itBond->second.first->bondID << ";" << endl;
						--itBond;
					}
					else
					{
						// last bond
						szsFlowEqu << "= 0;" << endl;
					}
				}
			}
			else if (itBond->second.second == DESTINATION)
			{
				// this junction is the destination of the bond

				if ((bgSrc.type() == CyPhyML::Se::meta) ||
						(bgSrc.type() == CyPhyML::MSe::meta))
				{
					this->equ_file_ << "e";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << " = ";
					if (bgSrc.type() == CyPhyML::Se::meta)
					{
						// Se
						this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgSrc)); // CyPhyML::Se::Cast(bgSrc).ParameterValue();
					}
					else
					{
						// MSe
						// TODO: connected parameter
						//this->equ_file_ << "{ModulatedElement}";
						set<CyPhyML::Signal2BGMElement> s2bgm;
						s2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcSignal2BGMElement();
						set<CyPhyML::Modulation2BGMElement> m2bgm;
						m2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcModulation2BGMElement();

						//this->equ_file_ << "{";
						if (s2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
						}
						else if (m2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
						}
						else
						{
							this->equ_file_ << "Unhandled modulation";
						}
								//this->equ_file_ << "}";
					}
					this->equ_file_ << ";" << endl;
				}
				else if ((bgSrc.type() == CyPhyML::Sf::meta) ||
								 (bgSrc.type() == CyPhyML::MSf::meta))
				{
					this->equ_file_ << "f";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << " = ";
					if (bgSrc.type() == CyPhyML::Sf::meta)
					{
						// Sf
						this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgSrc)); // CyPhyML::Sf::Cast(bgSrc).ParameterValue();
					}
					else
					{
						// MSf
						// TODO: connected parameter
						//this->equ_file_ << "{ModulatedElement}";
						set<CyPhyML::Signal2BGMElement> s2bgm;
						s2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcSignal2BGMElement();
						set<CyPhyML::Modulation2BGMElement> m2bgm;
						m2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcModulation2BGMElement();

						//this->equ_file_ << "{";
						if (s2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
						}
						else if (m2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
						}
						else
						{
							this->equ_file_ << "Unhandled modulation";
						}
								//this->equ_file_ << "}";
					}
					this->equ_file_ << ";" << endl;
				}
				else if ((bgSrc.type() == CyPhyML::R::meta) ||
								 (bgSrc.type() == CyPhyML::MR::meta))
				{
					this->equ_file_ << "e";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << " = ";
					this->equ_file_ << "f";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << "*";
					if (bgSrc.type() == CyPhyML::R::meta)
					{
						// R
						this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgSrc)); // CyPhyML::R::Cast(bgSrc).ParameterValue();
					}
					else
					{
						// MR
						// TODO: connected parameter
						//this->equ_file_ << "{ModulatedElement}";
						set<CyPhyML::Signal2BGMElement> s2bgm;
						s2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcSignal2BGMElement();
						set<CyPhyML::Modulation2BGMElement> m2bgm;
						m2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcModulation2BGMElement();

						//this->equ_file_ << "{";
						if (s2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
						}
						else if (m2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
						}
						else
						{
							this->equ_file_ << "Unhandled modulation";
						}
								//this->equ_file_ << "}";
					}
					this->equ_file_ << ";" << endl;
				}
				else if ((bgSrc.type() == CyPhyML::C::meta) ||
								 (bgSrc.type() == CyPhyML::MC::meta))
				{
					//this->equ_file_ << "//e";
					//this->equ_file_ << itBond->second.first->bondID;
					//this->equ_file_ << "(0) = ";
					//this->equ_file_ << CyPhyML::Storage::Cast(bgSrc).InitialValue();
					//this->equ_file_ << ";" << endl;

					this->equ_file_ << "f";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << " = ";
					this->equ_file_ << "der(e";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << ")*";
					if (bgSrc.type() == CyPhyML::C::meta)
					{
						// C
						this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgSrc)); // CyPhyML::C::Cast(bgSrc).ParameterValue();
					}
					else
					{
						// MC
						// TODO: connected parameter
						//this->equ_file_ << "{ModulatedElement}";
						set<CyPhyML::Signal2BGMElement> s2bgm;
						s2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcSignal2BGMElement();
						set<CyPhyML::Modulation2BGMElement> m2bgm;
						m2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcModulation2BGMElement();

						//this->equ_file_ << "{";
						if (s2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
						}
						else if (m2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
						}
						else
						{
							this->equ_file_ << "Unhandled modulation";
						}
								//this->equ_file_ << "}";
					}
					this->equ_file_ << ";" << endl;
				}
				else if ((bgSrc.type() == CyPhyML::I::meta) ||
								 (bgSrc.type() == CyPhyML::MI::meta))
				{
					//this->equ_file_ << "//f";
					//this->equ_file_ << itBond->second.first->bondID;
					//this->equ_file_ << "(0) = ";
					//this->equ_file_ << CyPhyML::Storage::Cast(bgSrc).InitialValue();
					//this->equ_file_ << ";" << endl;

					this->equ_file_ << "e";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << " = ";
					this->equ_file_ << "der(f";
					this->equ_file_ << itBond->second.first->bondID;
					this->equ_file_ << ")*";
					if (bgSrc.type() == CyPhyML::I::meta)
					{
						// I
						this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgSrc)); // CyPhyML::I::Cast(bgSrc).ParameterValue();
					}
					else
					{
						// MI
						// TODO: connected parameter
						//this->equ_file_ << "{ModulatedElement}";
						set<CyPhyML::Signal2BGMElement> s2bgm;
						s2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcSignal2BGMElement();
						set<CyPhyML::Modulation2BGMElement> m2bgm;
						m2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcModulation2BGMElement();

						//this->equ_file_ << "{";
						if (s2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
						}
						else if (m2bgm.size() == 1)
						{
							this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
						}
						else
						{
							this->equ_file_ << "Unhandled modulation";
						}
								//this->equ_file_ << "}";
					}
					this->equ_file_ << ";" << endl;
				}
				else if ((bgSrc.type() == CyPhyML::TF::meta) ||
								 (bgSrc.type() == CyPhyML::MTF::meta))
				{
					if (sProcessedTwoPorts.end() == sProcessedTwoPorts.find(bgSrc))
					{
						sProcessedTwoPorts.insert(bgSrc);
						CyPhyML::TwoPort tp;
						tp = CyPhyML::TwoPort::Cast(bgSrc);
						set<CyPhyML::BondJ2E> sbj2e;
						sbj2e = tp.srcBondJ2E();
						if (sbj2e.size() == 1)
						{
							// must have only one connction of this type Element2Junction

							// effort equation
							this->equ_file_ << "e";
							this->equ_file_ << itBond->second.first->bondID;
							this->equ_file_ << "*";
							if (tp.type() == CyPhyML::TF::meta)
							{
								this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgSrc)); // tp.ParameterValue();
							}
							else
							{
								// TODO: connected parameter
								//this->equ_file_ << "{ModulatedElement}";
								set<CyPhyML::Signal2BGMElement> s2bgm;
								s2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcSignal2BGMElement();
								set<CyPhyML::Modulation2BGMElement> m2bgm;
								m2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcModulation2BGMElement();
								set<CyPhyML::Actuation2ModulatedTwoPort> mtp;
								mtp = CyPhyML::ModulatedTwoPort::Cast(bgSrc).srcActuation2ModulatedTwoPort();

								//this->equ_file_ << "{";
								if (s2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
								}
								else if (m2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
								}
								else if (mtp.size() == 1)
								{
									this->equ_file_ << CyPhyML::Actuation::Cast((*mtp.begin()).srcActuation2ModulatedTwoPort_end()).name();
								}
								else
								{
									this->equ_file_ << "Unhandled modulation";
								}
								//this->equ_file_ << "}";
							}
							this->equ_file_ << " = e";
							this->equ_file_ << GetBondId(*(sbj2e.begin()));
							this->equ_file_ << ";" << endl;

							// flow equation
							this->equ_file_ << "f";
							this->equ_file_ << GetBondId(*(sbj2e.begin()));
							this->equ_file_ << "*";
							if (tp.type() == CyPhyML::TF::meta)
							{
								this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgSrc)); // tp.ParameterValue();
							}
							else
							{
								// TODO: connected parameter
								//this->equ_file_ << "{ModulatedElement}";
								set<CyPhyML::Signal2BGMElement> s2bgm;
								s2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcSignal2BGMElement();
								set<CyPhyML::Modulation2BGMElement> m2bgm;
								m2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcModulation2BGMElement();
								set<CyPhyML::Actuation2ModulatedTwoPort> mtp;
								mtp = CyPhyML::ModulatedTwoPort::Cast(bgSrc).srcActuation2ModulatedTwoPort();

								//this->equ_file_ << "{";
								if (s2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
								}
								else if (m2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
								}
								else if (mtp.size() == 1)
								{
									this->equ_file_ << CyPhyML::Actuation::Cast((*mtp.begin()).srcActuation2ModulatedTwoPort_end()).name();
								}
								else
								{
									this->equ_file_ << "Unhandled modulation";
								}
								//this->equ_file_ << "}";
							}
							this->equ_file_ << " = f";
							this->equ_file_ << itBond->second.first->bondID;
							this->equ_file_ << ";" << endl;

						}
					}
				}
				else if ((bgSrc.type() == CyPhyML::GY::meta) ||
								 (bgSrc.type() == CyPhyML::MGY::meta))
				{
					if (sProcessedTwoPorts.end() == sProcessedTwoPorts.find(bgSrc))
					{
						sProcessedTwoPorts.insert(bgSrc);
						CyPhyML::TwoPort tp;
						tp = CyPhyML::TwoPort::Cast(bgSrc);
						set<CyPhyML::BondJ2E> sbj2e;
						sbj2e = tp.srcBondJ2E();
						if (sbj2e.size() == 1)
						{
							// must have only one connction of this type Element2Junction

							// first equation
							this->equ_file_ << "f";
							this->equ_file_ << itBond->second.first->bondID;
							this->equ_file_ << "*";
							if (tp.type() == CyPhyML::GY::meta)
							{
								this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgSrc)); // tp.ParameterValue();
							}
							else
							{
								// TODO: connected parameter
								//this->equ_file_ << "{ModulatedElement}";
								set<CyPhyML::Signal2BGMElement> s2bgm;
								s2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcSignal2BGMElement();
								set<CyPhyML::Modulation2BGMElement> m2bgm;
								m2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcModulation2BGMElement();
								set<CyPhyML::Actuation2ModulatedTwoPort> mtp;
								mtp = CyPhyML::ModulatedTwoPort::Cast(bgSrc).srcActuation2ModulatedTwoPort();

								//this->equ_file_ << "{";
								if (s2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
								}
								else if (m2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
								}
								else if (mtp.size() == 1)
								{
									this->equ_file_ << CyPhyML::Actuation::Cast((*mtp.begin()).srcActuation2ModulatedTwoPort_end()).name();
								}
								else
								{
									this->equ_file_ << "Unhandled modulation";
								}
								//this->equ_file_ << "}";
							}
							this->equ_file_ << " = e";
							this->equ_file_ << GetBondId(*(sbj2e.begin()));
							this->equ_file_ << ";" << endl;

							// second equation
							this->equ_file_ << "f";
							this->equ_file_ << GetBondId(*(sbj2e.begin()));
							this->equ_file_ << "*";
							if (tp.type() == CyPhyML::GY::meta)
							{
								this->equ_file_ << GetParameterName(CyPhyML::BGElement::Cast(bgSrc)); // tp.ParameterValue();
							}
							else
							{
								// TODO: connected parameter
								//this->equ_file_ << "{ModulatedElement}";
								set<CyPhyML::Signal2BGMElement> s2bgm;
								s2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcSignal2BGMElement();
								set<CyPhyML::Modulation2BGMElement> m2bgm;
								m2bgm = CyPhyML::BGModulatedElement::Cast(bgSrc).srcModulation2BGMElement();
								set<CyPhyML::Actuation2ModulatedTwoPort> mtp;
								mtp = CyPhyML::ModulatedTwoPort::Cast(bgSrc).srcActuation2ModulatedTwoPort();

								//this->equ_file_ << "{";
								if (s2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Signal::Cast((*s2bgm.begin()).srcSignal2BGMElement_end()));
								}
								else if (m2bgm.size() == 1)
								{
									this->equ_file_ << EquHelper::getFunctionName(CyPhyML::Modulation::Cast((*m2bgm.begin()).srcModulation2BGMElement_end()));
								}
								else if (mtp.size() == 1)
								{
									this->equ_file_ << CyPhyML::Actuation::Cast((*mtp.begin()).srcActuation2ModulatedTwoPort_end()).name();
								}
								else
								{
									this->equ_file_ << "Unhandled modulation";
								}
								//this->equ_file_ << "}";
							}
							this->equ_file_ << " = e";
							this->equ_file_ << itBond->second.first->bondID;
							this->equ_file_ << ";" << endl;

						}
					}
				}

				if (itvOn->Ref->type() == CyPhyML::OneJunction::meta)
				{
					if (isHybridJunction(*itvOn))
					{
						szsFlowEquOFF << "f";
						szsFlowEquOFF << itBond->second.first->bondID;
						szsFlowEquOFF << " = 0;" << endl;
					}

					szsEffortEqu << "+e";
					szsEffortEqu << itBond->second.first->bondID;
					szsEffortEqu << " ";

					if (*itBond != *(itvOn->BondMap.rbegin()))
					{
						// not the last bond
						szsFlowEqu << "f";
						szsFlowEqu << itBond->second.first->bondID;
						szsFlowEqu << " ";
						++itBond;
						szsFlowEqu << "= f" << itBond->second.first->bondID << ";" << endl;
						--itBond;
					}
					else
					{
						// last bond
						szsEffortEqu << "= 0;" << endl;
					}
				}
				else if (itvOn->Ref->type() == CyPhyML::ZeroJunction::meta)
				{
					if (isHybridJunction(*itvOn))
					{
						szsEffortEquOFF << "e";
						szsEffortEquOFF << itBond->second.first->bondID;
						szsEffortEquOFF << " = 0;" << endl;
					}

					szsFlowEqu << "+f";
					szsFlowEqu << itBond->second.first->bondID;
					szsFlowEqu << " ";

					if (*itBond != *(itvOn->BondMap.rbegin()))
					{
						// not the last bond
						szsEffortEqu << "e";
						szsEffortEqu << itBond->second.first->bondID;
						szsEffortEqu << " ";
						++itBond;
						szsEffortEqu << "= e" << itBond->second.first->bondID << ";" << endl;
						--itBond;
					}
					else
					{
						// last bond
						szsFlowEqu << "= 0;" << endl;
					}
				}
			}
		}

		// print this junction's equations
		if (isHybridJunction(*itvOn))
		{
			//goOn = time > 20.0 and time < 40.0;
			this->equ_file_ << GetOnConditionName(CAST_JUNCTION(itvOn)) << " = " << \
				GetOnCondition(*itvOn) << ";" << endl;

			//goOff = time > 40.0;
			this->equ_file_ << GetOffConditionName(CAST_JUNCTION(itvOn)) << " = " << \
				GetOffCondition(*itvOn) << ";" << endl;

			//state = goOn and not goOff;
			this->equ_file_ << GetStateName(CAST_JUNCTION(itvOn)) << " = " << \
				GetOnConditionName(CAST_JUNCTION(itvOn)) << " and not " << \
				GetOffConditionName(CAST_JUNCTION(itvOn)) << ";" << endl;

			//if state then
			this->equ_file_ << "if " << GetStateName(CAST_JUNCTION(itvOn)) << " then" << endl;

			//  on equs
			if (itvOn->Ref->type() == CyPhyML::OneJunction::meta)
			{
				this->equ_file_ << szsFlowEqu.str();
				this->equ_file_ << szsEffortEqu.str();
			}
			else
			{
				this->equ_file_ << szsEffortEqu.str();
				this->equ_file_ << szsFlowEqu.str();
			}
			//else
			this->equ_file_ << "else" << endl;
			
			//  off equs
			this->equ_file_ << szsEffortEquOFF.str();
			this->equ_file_ << szsFlowEquOFF.str();

			//end if;
			this->equ_file_ << "end if;" << endl;

		}
		else
		{
			this->equ_file_ << szsEffortEqu.str();
			this->equ_file_ << szsFlowEqu.str();
		}
	}
}

int Equations::GetBondId(const CyPhyML::Bond obond)
{
	for (map<long, TBond>::iterator itBond = AllBonds.begin();
			 itBond != AllBonds.end();
			 ++itBond)
	{
		if (itBond->second.uniqueID == obond.uniqueId())
		{
			return itBond->second.bondID;
		}
	}
	return -1;
}

string Equations::GetParameterName(TBGNode tbgnode)
{
	stringstream ss;
	ss << "Parameter_" << tbgnode.Ref->uniqueId() << "_" << GetModifiedName(tbgnode.Name);
	return ss.str();
}

string Equations::GetParameterName(CyPhyML::BGElement element)
{
	stringstream ss;
	ss << "Parameter_" << element.uniqueId() << "_" << GetModifiedName(element);
	return ss.str();
}

string Equations::GetStateName(CyPhyML::Junction junction)
{
	stringstream ss;
	ss << "State_" << junction.uniqueId() << "_" << GetModifiedName(junction);
	return ss.str();
}

string Equations::GetOnConditionName(CyPhyML::Junction junction)
{
	stringstream ss;
	ss << "OnGuard_" << junction.uniqueId() << "_" << GetModifiedName(junction);
	return ss.str();
}

string Equations::GetOffConditionName(CyPhyML::Junction junction)
{
	stringstream ss;
	ss << "OffGuard_" << junction.uniqueId() << "_" << GetModifiedName(junction);
	return ss.str();
}

bool Equations::isHybridJunction(TBGNode tbgnode)
{
	if (Udm::IsDerivedFrom(tbgnode.Ref->type(), CyPhyML::Junction::meta))
	{
		CyPhyML::Junction junc = CyPhyML::Junction::Cast(*(tbgnode.Ref));
		if ((string(junc.OnCondition()) == "") &&
			(string(junc.OffCondition()) == ""))
		{
			return false;
		}
		else
		{
			return true;
		}
	}
	else
	{
		return false;
	}
}

string Equations::GetOnCondition(TBGNode tbgnode)
{
	if (Udm::IsDerivedFrom(tbgnode.Ref->type(), CyPhyML::Junction::meta))
	{
		CyPhyML::Junction junc = CyPhyML::Junction::Cast(*(tbgnode.Ref));
		return ConvertToModelicaSyntax(EquHelper::GetOnCondition(junc));
	}
	else
	{
		return "";
	}
}

string Equations::GetOffCondition(TBGNode tbgnode)
{
	if (Udm::IsDerivedFrom(tbgnode.Ref->type(), CyPhyML::Junction::meta))
	{
		CyPhyML::Junction junc = CyPhyML::Junction::Cast(*(tbgnode.Ref));
		return ConvertToModelicaSyntax(EquHelper::GetOffCondition(junc));
	}
	else
	{
		return "";
	}
}

string Equations::ConvertToModelicaSyntax(string input)
{
	string ret=input;
	tr1::regex rx("");
	string fmt = "";

	rx = "&&";
	fmt = " and ";
	ret = string(tr1::regex_replace(ret, rx, fmt));


	rx = "!=";
	fmt = "<>";
	ret = string(tr1::regex_replace(ret, rx, fmt));

	rx = "!";
	fmt = " not ";
	ret = string(tr1::regex_replace(ret, rx, fmt));

	return ret;
}


string Equations::GenerateDocumentationLine(
	CyPhyML::MgaObject obj,
	int rowId,
	string variableName)
{
	stringstream ss;
	string varName = "";
	if (variableName.empty())
	{
		varName = EquHelper::getFunctionName(obj);
	}
	else
	{
		varName = variableName;
	}

	// CELL DEFINITIONS ===========================
	// ss << "<th>Variable name</th>" << endl;
	// ss << "<th>Gme object name</th>" << endl;
	// ss << "<th>Type</th>" << endl;
	// //ss << "<th>Link</th>" << endl;
	// ss << "<th>Path</th>" << endl;
	// ============================================

	ss << "<tr";
	if (rowId % 2 == 1)
	{
		ss << " class=\"alt\"";
	}
	ss << ">" << endl;

	ss << "<td align=\"right\">" << varName << "</td>" << endl; // cell 1
	ss << "<td align=\"center\"><a href=\"mga:" << UdmGme::UdmId2GmeId(obj.uniqueId()) << "\">" << obj.name() << "</a></td>" << endl; // cell 2
	ss << "<td align=\"center\">" << obj.type().name() << "</td>" << endl; // cell 3
	///ss << "<td align=\"center\">&lt;a href=\"mga:" << UdmGme::UdmId2GmeId(obj.uniqueId()) << "\"&gt;" << obj.name() << "&lt;/a&gt;</td>" << endl; // cell 4
	ss << "<td>" << obj.getPath2() << "</td>" << endl; // cell 5
	ss << "</tr>" << endl;
	return ss.str();
}