/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "Graph.h"
#include "CyPhyML.h"

#include "ScriptFile.h"
#include "Log.h"
#include "Tools.h"

#include "TBGNode.h"
#include "TBond.h"
#include "TJunction.h"
#include "PowerChain.h"

#include "PortMapping.h"
#include "PowerPortInfo.h"

#include "HbgGenerator.h"
#include "Equations.h"

#include "MDAODialog.h"
//#include "ProgressDlg.h"
#include "MdaoAssembly.h"
#include "MdaoComponent.h"


#include "BlockConnection.h"
#include "Message.h"
#include <queue>

ScriptFile sfFile; // output script file and datastructure file
unsigned int uinTraversal = 0; // # of traversal

long uidbond=0;

bool bMDAO = false;
bool bSilentMode = false;
bool bAlgLoops = false;
bool bProcessAll = true;

// Global variables
priority_queue<Message> pqMessages;
vector<BlockConnection> conns;
vector<DML::MgaObject> vIoPorts;
//extern vector<BlockConnection> SecondRunConnections;
set<Udm::Object> SelectedObj;

//contains uniqueID -> PortNumber pairs (in simulink)
vector<PortMapping> pmSignal;  

// Flat bond-graph
map<long,TBond>			AllBonds;			// contains all bonds (flat BG graph)
vector<TJunction>		AllJunctions;	// contains all junctions (flat BG graph)
vector<TBGNode>			BGNodeList;
vector<PowerChain>	PowerPortChains;
map<int, map<long, PowerPortInfo>> PPInfos;

//map<long,DML::PowerPort> mProcessedPP; // processed PowerPorts

// Maintain the port-structure
map<long,PortNumbers>    ioPortNumbers; // contains the number of input/output ports.

extern map<long, int> mLayout; // long = uid, int = # of object wo position
extern set<long> VisitedBGNodes;

map<string, string> MDAOInput; // MDAO Tunable parameters of the system (current version) [Name, default value]
set<string> MDAOParameter;   // Tunable parameters of the system (old version)
map<string, string> MDAOOutput;      // MDAO Output final output variables after the postprocessing
set<string> MDAODataStorage; // 'Scopes' = monitor points flow, and effort
set<string> SimulinkSystemNames; // Referenced simulink block/system names

set<DML::ValueFlowTarget> ValueFlowTargets; // parameter & property (ranges)

extern string StopTime;

extern set<DML::Metric> MetricIDs;

// object to process
// <Obj, true = full container process>
MapContainer ObjToProcess;

//set<string> sFullProcessElements;

// global variables for automation
extern bool Automation;
extern string OutputDir;
extern string ModelName;
//extern bool SilentMode;
extern bool bParametricExp;

void ClearAllGlobals()
{
	bProcessAll = true;
	//sfFile.Clear();
	uinTraversal = 0;
	SelectedObj.clear();

	uidbond = 0;
	//pqMessages.
	conns.clear();
	vIoPorts.clear();

	pmSignal.clear();
	
	AllBonds.clear();
	AllJunctions.clear();
	BGNodeList.clear();
	PowerPortChains.clear();
	PPInfos.clear();

	ioPortNumbers.clear();

	mLayout.clear();
	VisitedBGNodes.clear();

	sfFile.sScopes.clear();
	MDAOParameter.clear();
	MDAOOutput.clear();
	MDAOInput.clear();
	MDAODataStorage.clear();
	SimulinkSystemNames.clear();

	ObjToProcess.clear();
	//sFullProcessElements.clear();
	ValueFlowTargets.clear();

	MetricIDs.clear();
}

void Graph::CheckBGConstraints()
{
	// iterate through all the BGNodes
	for (vector<TBGNode, allocator<TBGNode>>::iterator it = BGNodeList.begin();
		it != BGNodeList.end();
		++it)
	{
		if (Udm::IsDerivedFrom(it->Ref->type(), DML::Junction::meta))
		{
			// if it is a (One or Zero)junction 
			if (it->NumberOfBonds < 2)
			{
				string err = "";
				err = "Junction has less than 2 bond connections in the composed graph: ";
				err += GMEConsole::Formatter::MakeObjectHyperlink(it->Ref->name(), *(it->Ref));
				pqMessages.push(Message(err, MSG_ERROR));
			}
		}

		if (Udm::IsDerivedFrom(it->Ref->type(), DML::TwoPort::meta))
		{
			// if it is a TwoPort (TF, GY, MTF, MGY)
			if (it->NumberOfBonds != 2)
			{
				string err = "";
				err = "Two port elements (TF,GY) must have exactly 2 bond connections: ";
				err += GMEConsole::Formatter::MakeObjectHyperlink(it->Ref->name(), *(it->Ref));
				pqMessages.push(Message(err, MSG_ERROR));
			}
		}

	}

}

bool IsNeedToBeProcessed(Udm::Object &o)
{
	if (ObjToProcess.empty())
	{
		// We need to process everything
		return true;
	}

	if (o.type() == DML::InformationFlow::meta)
	{
		// this is a connection
		// process it only if both the src and the dst
		//  need to be  processed
		DML::InformationFlow iflow = DML::InformationFlow::Cast(o);

		DML::SignalPortType srcSpt = iflow.srcInformationFlow_end();
		DML::SignalPortType dstSpt = iflow.dstInformationFlow_end();

		if (IsNeedToBeProcessed(Udm::Object::Cast(srcSpt)) &&
				IsNeedToBeProcessed(Udm::Object::Cast(dstSpt)))
		{
			return true;
		}
		else
		{
			return false;
		}
	}

	MapContainer::iterator it;
	it = ObjToProcess.find(o);
	if (it != ObjToProcess.end())
	{
		// We need to process this
		return true;
	}
	else
	{
		// We need to find a parent which is in the ObjToProcess
		Udm::Object parent = o.GetParent();
		while (parent != Udm::null)
		{
			it = ObjToProcess.find(parent);
			if (it != ObjToProcess.end())
			{
				if (it->second == true)
				{
					return true;
				}
				else
				{
					return false;
				}
			}
			parent = parent.GetParent();
		}
		return false;
	}
}

void SetUniqueNameForChildren(Udm_VS10::Object o)
{
	// Recursive function.
	// Iterates through all the child objects
	// and sets a unique name temporary for 
	// all objects except for connections
	// These new unique names won't be stored 
	// in the model, but the user will get 
	// messages, when an object was renamed
 	Udm::Object child;
	set<Udm_VS10::Object> children;
	set<Udm_VS10::Object>::iterator itChild;

	set<string> childNames;
	string name;

	string meta_name;

	string msg;

	char buffer[33];
	int radix = 10;

	long id;
	
	// check all children
	children = o.GetChildObjects();
	for (itChild = children.begin();
		itChild != children.end();
		++itChild)
	{
		child = *itChild;
		id = child.uniqueId();

		if (id > 400000000 &&
			  id < 500000000)
		{
			// skip connections
			continue;
		}
		name = GetModifiedName(Udm_VS10::Object::Cast(*itChild));
		if (childNames.insert(name).second)
		{
			// unique
		}
		else
		{
			// not unique
			// new name needed
			msg = "Object name has changed: FROM ";
			msg += name;

			_ltoa_s(itChild->uniqueId(), buffer, radix);
			name += "_ID";
			name += string(buffer);
			
			child.setStringAttr(Uml::Attribute::meta_name, name);
			childNames.insert(name);

			msg += " TO ";
			msg += GMEConsole::Formatter::MakeObjectHyperlink(name, child);
			pqMessages.push(Message(msg, MSG_INFO));
		}
		SetUniqueNameForChildren(*itChild);
	}
}

void GenerateUniqueNames(Udm_VS10::Object startObj)
{
	MapContainer::const_iterator itMap;

	if (ObjToProcess.empty())
	{
		SetUniqueNameForChildren(startObj);
	}
	else
	{
		for (itMap = ObjToProcess.begin();
			itMap != ObjToProcess.end();
			++itMap)
		{
			if (itMap->second == true)
			{
				SetUniqueNameForChildren(itMap->first);
			}
		}
	}
}

void Graph::Process(Udm::DataNetwork *data_network)
{
	// This function traveses the GME project three times and collects all the 
	// necessary data for building a Simulink model.
	// 
	// (1) Gets the information for the initialization of the output directories
	//     and files.
	// (2) Opens the new generated files
	// (3) Writes the header(s)
	// (4) 1. Graph traversal and creates the hierarchical subsystem components.
	// (5) Writes the datastructure file.
	// (6) Creates the Bond Graph nodes in the subsystems.
	// (7) Creates the input and output ports for the associated power ports.
	// (8) 2. Graph traversal	for IO ports
	// (9) Creates the remaining input and output ports.
	// (10) 3. Graph traversal for connections
	// (11) Resizes the subsystems to make the generated model much readable.
	// (12) Writes all remaining connections.
	// (13) Writes the footer(s).
	// (14) Closes the files (script and datastruct).
	// (15) Creates the value ranges file (this is just for debugging)
	// (16) Creates a Graph Viz file for the flat Bond Graph
	// (17) Prints all the collected messages during the model traversals
	//      into the (GME)Console.

	GenerateUniqueNames(data_network->GetRootObject());

	if (!ObjToProcess.empty())
	{
		bProcessAll = false;
	}

	// 1. create models and elements (except I/O Ports and connecions)
	DML::RootFolder rfDomainModel; // root folder of the current project
	rfDomainModel = DML::RootFolder::Cast(data_network->GetRootObject());

	// init script file
	string szModelName = GetModifiedName(rfDomainModel.name());
	if (ModelName.empty())
	{
		sfFile.szModelName = GetModifiedName(rfDomainModel.name());
	}
	else
	{
		sfFile.szModelName = ModelName;
	}

	sfFile.OutputDirectory = OutputDir;

	Log lLogFile;

	sfFile.Open();
	sfFile.PrintHeader();

	MapContainer::iterator it;

	// First traversal
	uinTraversal = 1;

	set<DML::Components> sComponents;
	sComponents = rfDomainModel.Components_kind_children();

	for (set<DML::Components>::iterator itComponents = sComponents.begin();
			 itComponents != sComponents.end();
			 ++itComponents)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itComponents)))
		{
			this->ProcessComponents(DML::Components::Cast(*itComponents));
		}
	}

	set<DML::Testing> sTesting;
	sTesting = rfDomainModel.Testing_children();

	for (set<DML::Testing>::iterator itTesting = sTesting.begin();
			 itTesting != sTesting.end();
			 ++itTesting)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itTesting)))
		{
			this->ProcessTesting(DML::Testing::Cast(*itTesting));
		}
	}

	CheckPowerPorts2Junctions();

	DeriveBGNodeList(sfFile.szModelName);
	CheckBGConstraints();
	sfFile.PrintDataStructure();

	sfFile.PrintBGElements();
	sfFile.PrintPowerPort();


	// 2. create I/O ports
	// Second traversal
	uinTraversal = 2;
	if (!ObjToProcess.empty())
	{
		bProcessAll = false;
	}
	else
	{
		bProcessAll = true;
	}
	for (set<DML::Components>::iterator itComponents = sComponents.begin();
		 itComponents != sComponents.end();
		 ++itComponents)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itComponents)))
		{
			this->ProcessComponents(DML::Components::Cast(*itComponents));
		}
	}

	for (set<DML::Testing>::iterator itTesting = sTesting.begin();
			 itTesting != sTesting.end();
			 ++itTesting)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itTesting)))
		{
			this->ProcessTesting(DML::Testing::Cast(*itTesting));
		}
	}


	sfFile.PrintIoPort();

	// 3. create all connections
	// Third traversal
	uinTraversal = 3;
	if (!ObjToProcess.empty())
	{
		bProcessAll = false;
	}
	else
	{
		bProcessAll = true;
	}
	for (set<DML::Components>::iterator itComponents = sComponents.begin();
		 itComponents != sComponents.end();
		 ++itComponents)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itComponents)))
		{
			this->ProcessComponents(DML::Components::Cast(*itComponents));
		}
	}

	for (set<DML::Testing>::iterator itTesting = sTesting.begin();
			 itTesting != sTesting.end();
			 ++itTesting)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itTesting)))
		{
			this->ProcessTesting(DML::Testing::Cast(*itTesting));
		}
	}

	if (bMDAO)
	{
		MDAODialog mdaodlg;
		mdaodlg.DoModal();
		/*if (bSilentMode)
		{
			mdaodlg.OnBnClickedBtnDiscover();
			mdaodlg.OnBnClickedOk();
		}
		else
		{
			mdaodlg.DoModal();
		}*/
	}
	
	if (!bParametricExp)
	{
		sfFile.ResizeSubsystems();
		sfFile.PrintConnections();
		sfFile.PrintFooter();
		sfFile.Close();


		sfFile.PrintValueRanges(ValueFlowTargets);
		sfFile.PrintGraphViz();
		Equations equODE; // writes the equations to a file
	}
	// Print message queue
	lLogFile.Flush();

	if (bSilentMode &&
		Automation == false)
	{
		string szCommand = "";
		szCommand = "start matlab -nosplash -nodesktop -r ";
		szCommand += sfFile.szModelName;
		szCommand += "_buildscript";
		int error_code = system(szCommand.c_str());
		if (error_code != 0)
		{
			string szExceptionTxt = "";
			szExceptionTxt += "The following command failed: \r\n";
			szExceptionTxt += szCommand;
			szExceptionTxt += "\r\n\r\nMake sure that MatLab is installed on your machine.";
			szExceptionTxt += " If it is not, please remove the check mark from the Generate Matlab model checkbox.";
			throw udm_exception(szExceptionTxt);
		}
	}
	//prgBar.CloseWindow();
}


void Graph::ProcessDesignSpace(DML::DesignSpace &oDesignSpace)
{
	if (uinTraversal == 1)
	{
		// create this component
		sfFile << oDesignSpace;

		ioPortNumbers.insert( make_pair(oDesignSpace.uniqueId(),
													PortNumbers(oDesignSpace)));
	}

	set<DML::DesignSpace> sDesignSpace;
	sDesignSpace = oDesignSpace.DesignSpace_kind_children();

	// Process sub DesignSpace folders
	for (set<DML::DesignSpace>::iterator itDesignSpace = sDesignSpace.begin();
			 itDesignSpace != sDesignSpace.end();
			 ++itDesignSpace)
	{
		this->ProcessDesignSpace(DML::DesignSpace::Cast(*itDesignSpace));
	}

	set<DML::DesignContainer> sDesignContainer;
	sDesignContainer = oDesignSpace.DesignContainer_kind_children();

	// Process sub DesignContainer models
	for (set<DML::DesignContainer>::iterator itDesignContainer = sDesignContainer.begin();
			 itDesignContainer != sDesignContainer.end();
			 ++itDesignContainer)
	{
		this->ProcessDesignContainer(DML::DesignContainer::Cast(*itDesignContainer));
	}
}

void Graph::ProcessTesting(DML::Testing &oTesting)
{
	if (uinTraversal == 1)
	{
		// create this component
		sfFile << oTesting;

		ioPortNumbers.insert( make_pair(oTesting.uniqueId(),
													PortNumbers(oTesting)));
	}
	if (uinTraversal == 1)
	{
		string err = "";
		err = "TBA Process: ";
		err += GMEConsole::Formatter::MakeObjectHyperlink(oTesting.name(), oTesting);
		//pqMessages.push(Message(err, MSG_WARNING));
	}

	set<DML::Testing> sTesting;
	sTesting = oTesting.Testing_kind_children();

	for (set<DML::Testing>::iterator itTesting = sTesting.begin();
			 itTesting != sTesting.end();
			 ++itTesting)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itTesting)))
		{
			this->ProcessTesting(DML::Testing::Cast(*itTesting));
		}
	}

	set<DML::TestBench> sTestBench;
	sTestBench = oTesting.TestBench_kind_children();

	for (set<DML::TestBench>::iterator itTestBench = sTestBench.begin();
			 itTestBench != sTestBench.end();
			 ++itTestBench)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itTestBench)))
		{
			this->ProcessTestBench(DML::TestBench::Cast(*itTestBench));
		}
	}

	set<DML::ParametricExplorationFolder> sPef;
	sPef = oTesting.ParametricExplorationFolder_kind_children();

	for (set<DML::ParametricExplorationFolder>::iterator itPef = sPef.begin();
			 itPef != sPef.end();
			 ++itPef)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itPef)))
		{
			this->ProcessPEF(DML::ParametricExplorationFolder::Cast(*itPef));
		}
	}

	set<DML::TestComponents> sTestComponents;
	sTestComponents = oTesting.TestComponents_kind_children();

		for (set<DML::TestComponents>::iterator itTestComponents = sTestComponents.begin();
			 itTestComponents != sTestComponents.end();
			 ++itTestComponents)
	{
		if (uinTraversal == 1)
		{
			string err = "";
			err = "TBA Process: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(itTestComponents->name(), *itTestComponents);
			//pqMessages.push(Message(err, MSG_WARNING));
		}
		//this->ProcessTestComponents(DML::TestBench::Cast(*itTestComponents));
	}

}

void Graph::ProcessTestBench(DML::TestBench &oTestBench)
{
	if (uinTraversal == 1)
	{
		// create this component
		sfFile << oTestBench;

		ioPortNumbers.insert( make_pair(oTestBench.uniqueId(),
													PortNumbers(oTestBench)));
	}

	if (uinTraversal == 1)
	{
		string err = "";
		err = "TBA Process: ";
		err += GMEConsole::Formatter::MakeObjectHyperlink(oTestBench.name(), oTestBench);
		//pqMessages.push(Message(err, MSG_WARNING));
	}

	set<DML::TopLevelSystemUnderTest> sTLSUT;
	sTLSUT = oTestBench.TopLevelSystemUnderTest_kind_children();

	for (set<DML::TopLevelSystemUnderTest>::iterator itTLSUT = sTLSUT.begin();
			 itTLSUT != sTLSUT.end();
			 ++itTLSUT)
	{
		if (uinTraversal == 1)
		{
			string err = "";
			err = "TBA Process: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(itTLSUT->name(), *itTLSUT);
			err += " ref obj: ";
			if (itTLSUT->getReferencedObject() != Udm::null)
			{
				err += GMEConsole::Formatter::MakeObjectHyperlink(DML::MgaObject::Cast(itTLSUT->getReferencedObject()).name(), itTLSUT->getReferencedObject());
				string type = "";
				type = DML::MgaObject::Cast(itTLSUT->getReferencedObject()).type().name();

			}
			else
			{
				err += "NULL";
			}
			
			//pqMessages.push(Message(err, MSG_WARNING));
		}

		if (itTLSUT->getReferencedObject() != Udm::null)
		{
			if (Udm::IsDerivedFrom(DML::MgaObject::Cast(itTLSUT->getReferencedObject()).meta, DML::DesignContainer::meta))
			{
				// TODO: How can we resolve this problem???
				//ProcessDesignContainer(DML::DesignContainer::Cast(itTLSUT->getReferencedObject()));
			}
		}
		//this->ProcessTestComponent(DML::TestComponent::Cast(*itTLSUT));
	}


	//oTestBench.TestInjectionPoint_kind_children();

	set<DML::TestComponent> sTestComponent;
	sTestComponent = oTestBench.TestComponent_kind_children();

	// Process sub component models
	for (set<DML::TestComponent>::iterator itTestComponent = sTestComponent.begin();
			 itTestComponent != sTestComponent.end();
			 ++itTestComponent)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itTestComponent)))
		{
			this->ProcessTestComponent(DML::TestComponent::Cast(*itTestComponent));
		}
	}

	set<DML::Component> sComponent;
	sComponent = oTestBench.Component_kind_children();

	set<DML::ComponentAssembly> sAssembly;
	sAssembly = oTestBench.ComponentAssembly_kind_children();

	if (uinTraversal == 1)
	{
		if ((sComponent.size() + sAssembly.size()) != 1)
		{
			string err = "";
			err = "Test Bench must conatin exactly one System Under Test component instance: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(oTestBench.name(), oTestBench);
			//pqMessages.push(Message(err, MSG_WARNING));
		}
	}

	// Process sub component models
	for (set<DML::Component>::iterator itComponent = sComponent.begin();
			 itComponent != sComponent.end();
			 ++itComponent)
	{
		if (uinTraversal == 1)
		{
			if (itComponent->isInstance() == false)
			{
			string err = "";
			err = "Test Bench must have a System Under Test component as an instance: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(itComponent->name(), *itComponent);
			//pqMessages.push(Message(err, MSG_WARNING));
			}
		}
 		if (IsNeedToBeProcessed(Udm::Object::Cast(*itComponent)))
		{
			this->ProcessComponent(DML::Component::Cast(*itComponent));
		}
	}

	// Process sub ComponentAssembly models
	for (set<DML::ComponentAssembly>::iterator itAssembly = sAssembly.begin();
			 itAssembly != sAssembly.end();
			 ++itAssembly)
	{
		if (uinTraversal == 1)
		{
			if (itAssembly->isInstance() == false)
			{
			string err = "";
			err = "Test Bench must have a System Under Test component as an instance: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(itAssembly->name(), *itAssembly);
			//pqMessages.push(Message(err, MSG_WARNING));
			}
		}
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itAssembly)))
		{
			this->ProcessComponentAssembly(DML::ComponentAssembly::Cast(*itAssembly));
		}
	}

	if (uinTraversal == 2)
	{
		set<DML::ValueFlowTarget> vft;
		vft = oTestBench.ValueFlowTarget_kind_children();
		for (set<DML::ValueFlowTarget>::iterator it = vft.begin();
			it != vft.end();
			++it)
		{
			ProcessValueFlowTarget(DML::ValueFlowTarget::Cast(*it));
		}
	}

	if (uinTraversal == 3)
	{
		// TODO: Process connections

		set<DML::SignalConnectionType> sSType;
		sSType = oTestBench.SignalConnectionType_kind_children();

		for (set<DML::SignalConnectionType>::iterator itSType = sSType.begin();
				 itSType != sSType.end();
				 ++itSType)
		{
			if (IsNeedToBeProcessed(Udm::Object::Cast(*itSType)))
			{
				this->ProcessConnection(DML::SignalConnectionType::Cast(*itSType));
			}
		}

		set<DML::InformationFlow> sIFlow;
		sIFlow = oTestBench.InformationFlow_kind_children();

		for (set<DML::InformationFlow>::iterator itIFlow = sIFlow.begin();
				 itIFlow != sIFlow.end();
				 ++itIFlow)
		{
			if (IsNeedToBeProcessed(Udm::Object::Cast(*itIFlow)))
			{
				this->ProcessConnection(DML::InformationFlow::Cast(*itIFlow));
			}
		}

		set<DML::SignalConnection> sCBase;
		sCBase = oTestBench.SignalConnection_kind_children();

		for (set<DML::SignalConnection>::iterator itCBase = sCBase.begin();
			itCBase != sCBase.end();
			++itCBase)
		{
			this->ProcessConnection(DML::ConnectionBase::Cast(*itCBase));
		}

		set<DML::ValueFlow2SignalPort> sVf2Sp;
		sVf2Sp = oTestBench.ValueFlow2SignalPort_kind_children();

		for (set<DML::ValueFlow2SignalPort>::iterator itsVf2Sp = sVf2Sp.begin();
				 itsVf2Sp != sVf2Sp.end();
				 ++itsVf2Sp)
		{
			this->ProcessConnection(DML::ValueFlow2SignalPort::Cast(*itsVf2Sp));
		}

		//set<DML::SignalPort2ValueFlow> sSp2Vf;
		//sSp2Vf = oTestBench.SignalPort2ValueFlow_kind_children();

		//for (set<DML::SignalPort2ValueFlow>::iterator itsSp2Vf = sSp2Vf.begin();
		//		 itsSp2Vf != sSp2Vf.end();
		//		 ++itsSp2Vf)
		//{
		//	this->ProcessConnection(DML::SignalConnectionType::Cast(*itsSp2Vf));
		//}

	}
}

void Graph::ProcessTestComponent(DML::TestComponent &oTestComponent)
{
	if (uinTraversal == 1)
	{
		// create this component
		sfFile << oTestComponent;

		ioPortNumbers.insert( make_pair(oTestComponent.uniqueId(),
																		PortNumbers(oTestComponent)));
	}

	if (uinTraversal == 2)
	{
		this->ProcessIoPorts(oTestComponent);
		
		set<DML::ValueFlowTarget> vft;
		vft = oTestComponent.ValueFlowTarget_kind_children();
		for (set<DML::ValueFlowTarget>::iterator it = vft.begin();
			it != vft.end();
			++it)
		{
			ProcessValueFlowTarget(DML::ValueFlowTarget::Cast(*it));
		}
	}

	set<DML::BondGraph> sPhyComponent;
	sPhyComponent = oTestComponent.BondGraph_kind_children();

	// Process physical component models
	for (set<DML::BondGraph>::iterator itPC = sPhyComponent.begin();
			 itPC != sPhyComponent.end();
			 ++itPC)
	{
		this->ProcessPhyComponent(DML::BondGraph::Cast(*itPC));
	}

	if (uinTraversal == 3)
	{
		// TODO: Process connections
		set<DML::InformationFlow> sIFlow;
		sIFlow = oTestComponent.InformationFlow_kind_children();

		for (set<DML::InformationFlow>::iterator itIFlow = sIFlow.begin();
				 itIFlow != sIFlow.end();
				 ++itIFlow)
		{
			this->ProcessConnection(DML::InformationFlow::Cast(*itIFlow));
		}

		set<DML::SignalConnectionType> sSType;
		sSType = oTestComponent.SignalConnectionType_kind_children();

		for (set<DML::SignalConnectionType>::iterator itSType = sSType.begin();
				 itSType != sSType.end();
				 ++itSType)
		{
			this->ProcessConnection(DML::SignalConnectionType::Cast(*itSType));
		}

		set<DML::ConnectionBase> sCBase;
		sCBase = oTestComponent.ConnectionBase_kind_children();

		for (set<DML::ConnectionBase>::iterator itCBase = sCBase.begin();
				 itCBase != sCBase.end();
				 ++itCBase)
		{
			this->ProcessConnection(DML::ConnectionBase::Cast(*itCBase));
		}

		set<DML::ValueFlow2SignalPort> sVf2Sp;
		sVf2Sp = oTestComponent.ValueFlow2SignalPort_kind_children();

		for (set<DML::ValueFlow2SignalPort>::iterator itsVf2Sp = sVf2Sp.begin();
				 itsVf2Sp != sVf2Sp.end();
				 ++itsVf2Sp)
		{
			this->ProcessConnection(DML::ValueFlow2SignalPort::Cast(*itsVf2Sp));
		}
	}

}

void Graph::ProcessComponents(DML::Components &oComponents)
{
	if (uinTraversal == 1)
	{
		// create this component
		sfFile << oComponents;

		ioPortNumbers.insert( make_pair(oComponents.uniqueId(),
													PortNumbers(oComponents)));
	}

	set<DML::Components> sComponents;
	sComponents = oComponents.Components_kind_children();

	// Process sub components folders
	for (set<DML::Components>::iterator itComponents = sComponents.begin();
			 itComponents != sComponents.end();
			 ++itComponents)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itComponents)))
		{
			this->ProcessComponents(DML::Components::Cast(*itComponents));
		}
	}

	set<DML::ComponentAssemblies> sAssemblies;
	sAssemblies = oComponents.ComponentAssemblies_kind_children();

	// Process sub ComponentAssemblies folders
	for (set<DML::ComponentAssemblies>::iterator itAssemblies = sAssemblies.begin();
			 itAssemblies != sAssemblies.end();
			 ++itAssemblies)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itAssemblies)))
		{
			this->ProcessComponentAssemblies(DML::ComponentAssemblies::Cast(*itAssemblies));
		}
	}

	set<DML::Component> sComponent;
	sComponent = oComponents.Component_kind_children();

	// Process sub component models
	for (set<DML::Component>::iterator itComponent = sComponent.begin();
			 itComponent != sComponent.end();
			 ++itComponent)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itComponent)))
		{
			this->ProcessComponent(DML::Component::Cast(*itComponent));
		}
	}
}

void Graph::ProcessComponentAssemblies(DML::ComponentAssemblies &oCAssemblies)
{
	if (uinTraversal == 1)
	{
		// create this component
		sfFile << oCAssemblies;

		ioPortNumbers.insert( make_pair(oCAssemblies.uniqueId(),
													PortNumbers(oCAssemblies)));
	}

	set<DML::ComponentAssemblies> sAssemblies;
	sAssemblies = oCAssemblies.ComponentAssemblies_kind_children();

	// Process sub ComponentAssemblies folders
	for (set<DML::ComponentAssemblies>::iterator itAssemblies = sAssemblies.begin();
			 itAssemblies != sAssemblies.end();
			 ++itAssemblies)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itAssemblies)))
		{
			this->ProcessComponentAssemblies(DML::ComponentAssemblies::Cast(*itAssemblies));
		}
	}

	set<DML::ComponentAssembly> sAssembly;
	sAssembly = oCAssemblies.ComponentAssembly_kind_children();

	// Process sub ComponentAssembly models
	for (set<DML::ComponentAssembly>::iterator itAssembly = sAssembly.begin();
			 itAssembly != sAssembly.end();
			 ++itAssembly)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itAssembly)))
		{
			this->ProcessComponentAssembly(DML::ComponentAssembly::Cast(*itAssembly));
		}
	}

}

void Graph::ProcessPEF(DML::ParametricExplorationFolder &oPef)
{
	set<DML::ParametricExploration> sPe;
	sPe = oPef.ParametricExploration_kind_children();

	// Process sub ComponentAssembly models
	for (set<DML::ParametricExploration>::iterator itPe = sPe.begin();
			 itPe != sPe.end();
			 ++itPe)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itPe)))
		{
			this->ProcessPE(DML::ParametricExploration::Cast(*itPe));
		}
	}
}

void Graph::ProcessPE(DML::ParametricExploration &oPe)
{
	if (uinTraversal == 1)
	{
		// create mdao package
		MdaoAssembly Assembly(sfFile.szModelName + "_python");
		Assembly.Name = GetModifiedName(oPe.name());

		// TODO: assumption only one CONMIN optimizer!
		set<DML::Optimizer> optimizers = oPe.Optimizer_kind_children();
		if (optimizers.size() == 0)
		{
			pqMessages.push(Message("There is no optimizer found.", MSG_ERROR));
		}
		else if (optimizers.size() > 1)
		{
			throw exception("The Problem definition must contain exactly ONE optimizer.");
		}
		for (set<DML::Optimizer>::iterator it = optimizers.begin();
			it != optimizers.end();
			++it)
		{
			set<DML::DesignVariable> dv = it->DesignVariable_kind_children();
			for (set<DML::DesignVariable>::iterator itDv = dv.begin();
				itDv != dv.end();
				++itDv)
			{
				string Min = "0";
				string Max = "0";
				string Range = string(itDv->Range());
				if (Range != "" && 
					Range.find(',') != string::npos)
				{
					Min = Range.substr(0, Range.find_first_of(','));
					Max = Range.substr(Range.find_first_of(',') + 1);
				}
				Assembly.DesignVariables.insert(
					make_pair<string, pair<string, string>>(
					GetModifiedName(DML::MgaObject::Cast(*itDv)),
					make_pair<string, string>(Min, Max)));
			}
			set<DML::Objective> obj = it->Objective_kind_children();
			for (set<DML::Objective>::iterator itObj = obj.begin();
				itObj != obj.end();
				++itObj)
			{
				Assembly.Objectives.insert(GetModifiedName(DML::MgaObject::Cast(*itObj)));
			}
			set<DML::OptimizerConstraint> constraints = it->OptimizerConstraint_kind_children();
			for (set<DML::OptimizerConstraint>::iterator itCons = constraints.begin();
				itCons != constraints.end();
				++itCons)
			{
				string Min = "0";
				string Max = "0";

				Min = itCons->MinValue();
				Max = itCons->MaxValue();

				// good way
				//string ConstraintName = GetModifiedName(DML::MgaObject::Cast(*itCons));
				// HACK: bad way
				string ConstraintName = "!ERROR!";

				set<DML::ConstraintMapping> srcCMs = itCons->srcConstraintMapping();
				if (srcCMs.size() > 0)
				{
					DML::ConstraintMapping srcCM = *srcCMs.begin();
					ConstraintName = GetModifiedName(DML::TestBenchRef::Cast(srcCM.srcConstraint_refport_parent()));
					ConstraintName += ".";
					ConstraintName += GetModifiedName(DML::MgaObject::Cast(srcCM.srcConstraintMapping_end()));
				}
				else
				{
					throw exception("Optimizer's contraint must have exactly one incoming connection from a metric");
				}
				Assembly.Constraints.insert(
					make_pair<string, pair<string, string>>(
					ConstraintName,
					make_pair<string, string>(Min, Max)));
			}
		}

		// get components
		set<DML::TestBenchRef> sTBRef;
		sTBRef = oPe.TestBenchRef_kind_children();
		for (set<DML::TestBenchRef>::iterator it = sTBRef.begin();
			it != sTBRef.end();
			++it)
		{
			bool isExists = false;
			string newClassName = GetModifiedName(it->getReferencedObject());
			for (set<MdaoComponent*>::iterator itbase = Assembly.Bases.begin();
				itbase != Assembly.Bases.end();
				++itbase)
			{
				if ((*itbase)->Name == newClassName)
				{
					isExists = true;
					break;
				}
			}
			if (isExists)
			{
				continue;
			}
			MdaoComponent* c = new MdaoComponent(sfFile.szModelName + "_python");
			// TODO: switch instance name and name! (modify it in connections as well)
			c->InstanceName = GetModifiedName(it->name());
			c->Name = GetModifiedName(it->getReferencedObject());

			// get input parameters of components
			set<DML::Parameter> params;
			params = DML::TestBench::Cast(it->getReferencedObject()).Parameter_kind_children();
			for (set<DML::Parameter>::iterator p = params.begin();
				p != params.end();
				++p)
			{
				c->Inputs.insert(make_pair<string, string>(GetModifiedName(p->name()),p->Value()));
			}
			// get output parameters of components
			set<DML::Metric> metrics;
			metrics = DML::TestBench::Cast(it->getReferencedObject()).Metric_kind_children();
			for (set<DML::Metric>::iterator p = metrics.begin();
				p != metrics.end();
				++p)
			{
				string value = "0";
				if (string(p->Value()) != "")
				{
					value = p->Value();
				}
				c->Outputs.insert(make_pair<string, string>(GetModifiedName(p->name()), value));
			}

			string name = DML::TestBench::Cast(it->getReferencedObject()).name();
			string type = DML::TestBench::Cast(it->getReferencedObject()).Type();
			//if (name.find("MATLAB") != string::npos)
			if (type == "DynSim")
			{
				c->Type = MdaoComponent::MatLab;
				c->TargetFileName = DML::TestBench::Cast(it->getReferencedObject()).SpreadsheetLoc();
				//c->TargetFileName += sfFile.szModelName;
				//c->TargetFileName += "_Wrapper.exe";
			}
			//else if (name.find("EXCEL") != string::npos)
			else if (type == "Excel")
			{
				c->Type = MdaoComponent::Excel;
				//c->TargetFileName = "";
				//c->TargetFileName += sfFile.szModelName;
				//c->TargetFileName += ".xlsx";
				c->TargetFileName = DML::TestBench::Cast(it->getReferencedObject()).SpreadsheetLoc();
			}
			else if (type == "SimpleCalc")
			{
				// block is the defult if the type is not specified
				c->Type = MdaoComponent::Block;
				set<DML::CustomFormula> cf;
				cf = DML::TestBench::Cast(it->getReferencedObject()).CustomFormula_kind_children();
				set<DML::CustomFormula> sCFSaved;
				for (set<DML::CustomFormula>::const_iterator itCF = cf.begin();
					itCF != cf.end();
					++itCF)
				{
					// TODO: check unique names!
					c->CustomFormulas.insert(
						make_pair<string, string>(
							GetModifiedName(itCF->name()),
							itCF->Expression()));

					set<DML::ValueFlow> connections = itCF->dstValueFlow();
					for (set<DML::ValueFlow>::const_iterator itConn = connections.begin();
						itConn != connections.end();
						++itConn)
					{
						DML::ValueFlowTarget target;
						target = DML::ValueFlowTarget::Cast(itConn->dstValueFlow_end());
						if (target.type() == DML::Metric::meta)
						{
							c->MetricCustomFormula.insert(
								make_pair(
									GetModifiedName(target),
									GetModifiedName(itCF->name())));

							// TODO: create custom formula order
							// FIXME:
							// HACK:
							if (sCFSaved.find(DML::CustomFormula::Cast(*itCF)) == sCFSaved.end())
							{
								sCFSaved.insert(DML::CustomFormula::Cast(*itCF));
								c->CustomFormulaOrder.push_back(GetModifiedName(itCF->name()));
							}
							set<DML::ValueFlow> srcConns = itCF->srcValueFlow();
							for (set<DML::ValueFlow>::const_iterator itSrcConn = srcConns.begin();
								itSrcConn != srcConns.end();
								++itSrcConn)
							{
								DML::ValueFlowTarget srcTarget;
								srcTarget = DML::ValueFlowTarget::Cast(itSrcConn->srcValueFlow_end());
								if (srcTarget.type() == DML::CustomFormula::meta)
								{
									if (sCFSaved.find(DML::CustomFormula::Cast(srcTarget)) == sCFSaved.end())
									{
										sCFSaved.insert(DML::CustomFormula::Cast(srcTarget));
										c->CustomFormulaOrder.push_back(GetModifiedName(srcTarget));
									}
								}
							}
						}
					}
				}
			}

			Assembly.Bases.insert(c);
		}
		
		// connections
		// get assembly input connections
		set<DML::VariableSweep> sVarSweep;
		sVarSweep = oPe.VariableSweep_kind_children();
		for (set<DML::VariableSweep>::iterator it = sVarSweep.begin();
			it != sVarSweep.end();
			++it)
		{
			Udm::Object srcRef;
			Udm::Object dstRef;
			GetReferences(Udm::Object::Cast(*it), srcRef, dstRef);

			MdaoWire *w = new MdaoWire();
			w->Src = GetModifiedName(DML::MgaObject::Cast(it->GetParent()));
			w->SrcPortName = GetModifiedName(DML::MgaObject::Cast(it->srcVariableSweep_end()));
			//w->Dst = GetModifiedName(DML::Parameter::Cast(it->dstVariableSweep_end()).GetParent());
			w->Dst = GetModifiedName(dstRef);
			w->DstPortName = GetModifiedName(DML::Parameter::Cast(it->dstVariableSweep_end()));
			w->Type = MdaoWire::Input;
			Assembly.Wires.insert(w);
		}
		// get assembly internal connections
		set<DML::ResultFlow> sResultFlow;
		sResultFlow = oPe.ResultFlow_kind_children();
		for (set<DML::ResultFlow>::iterator it = sResultFlow.begin();
			it != sResultFlow.end();
			++it)
		{
			Udm::Object srcRef;
			Udm::Object dstRef;
			GetReferences(Udm::Object::Cast(*it), srcRef, dstRef);

			if (srcRef == dstRef)
			{
				string szErrMsg = "";
				szErrMsg = "Self connection detected please remove it: ";
				szErrMsg += GMEConsole::Formatter::MakeObjectHyperlink( it->name(),
																																*it);
				pqMessages.push(Message(szErrMsg, MSG_ERROR));
			}

			MdaoWire *w = new MdaoWire();
			//w->Src = GetModifiedName(DML::Metric::Cast(it->srcResultFlow_end()).GetParent());
			w->Src = GetModifiedName(srcRef);
			w->SrcPortName = GetModifiedName(DML::Metric::Cast(it->srcResultFlow_end()));
			//w->Dst = GetModifiedName(DML::Parameter::Cast(it->dstResultFlow_end()).GetParent());
			w->Dst = GetModifiedName(dstRef);
			w->DstPortName = GetModifiedName(DML::Parameter::Cast(it->dstResultFlow_end()));
			w->Type = MdaoWire::Internal;
			Assembly.Wires.insert(w);
		}
		// get assembly output connections
		set<DML::ObjectiveMapping> sObjMap;
		sObjMap = oPe.ObjectiveMapping_kind_children();
		for (set<DML::ObjectiveMapping>::iterator it = sObjMap.begin();
			it != sObjMap.end();
			++it)
		{
			Udm::Object srcRef;
			Udm::Object dstRef;
			GetReferences(Udm::Object::Cast(*it), srcRef, dstRef);

			MdaoWire *w = new MdaoWire();
			//w->Src = GetModifiedName(DML::Metric::Cast(it->srcObjectiveMapping_end()).GetParent());
			w->Src = GetModifiedName(srcRef);
			w->SrcPortName = GetModifiedName(DML::Metric::Cast(it->srcObjectiveMapping_end()));
			w->Dst = "";
			w->DstPortName = "";
			w->Type = MdaoWire::Output;
			Assembly.Wires.insert(w);
		}
		set<DML::ConstraintMapping> sConstraintMap;
		sConstraintMap = oPe.ConstraintMapping_children();
		for (set<DML::ConstraintMapping>::iterator it = sConstraintMap.begin();
			it != sConstraintMap.end();
			++it)
		{
			Udm::Object srcRef;
			Udm::Object dstRef;
			GetReferences(Udm::Object::Cast(*it), srcRef, dstRef);

			MdaoWire *w = new MdaoWire();
			//w->Src = GetModifiedName(DML::Metric::Cast(it->srcObjectiveMapping_end()).GetParent());
			w->Src = GetModifiedName(srcRef);
			w->SrcPortName = GetModifiedName(DML::Metric::Cast(it->srcConstraintMapping_end()));
			w->Dst = "";
			w->DstPortName = "";
			w->Type = MdaoWire::Output;
			Assembly.Wires.insert(w);
		}

		Assembly.PrintComponents(); // create py scripts for all components
		Assembly.PrintPython(); // create py script for assembly
		Assembly.PrintLoadAll();
	}
}

void Graph::ProcessDesignContainer(DML::DesignContainer &oDesignContainer)
{
	if (uinTraversal == 1)
	{
		// create this component
		sfFile << oDesignContainer;

		ioPortNumbers.insert( make_pair(oDesignContainer.uniqueId(),
													PortNumbers(oDesignContainer)));
	}

	// subsets
	set<DML::Component> sComponent;
	set<DML::ComponentAssembly> sAssembly;
	set<DML::DesignContainer> sDesignContainer;


	// Process sub component models
	for (set<DML::Component>::iterator itComponent = sComponent.begin();
			 itComponent != sComponent.end();
			 ++itComponent)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itComponent)))
		{
			this->ProcessComponent(DML::Component::Cast(*itComponent));
		}
	}

	for (set<DML::ComponentAssembly>::iterator itAssembly = sAssembly.begin();
			 itAssembly != sAssembly.end();
			 ++itAssembly)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itAssembly)))
		{
			this->ProcessComponentAssembly(DML::ComponentAssembly::Cast(*itAssembly));
		}
	}

	for (set<DML::DesignContainer>::iterator itDesignContainer = sDesignContainer.begin();
			 itDesignContainer != sDesignContainer.end();
			 ++itDesignContainer)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itDesignContainer)))
		{
			this->ProcessDesignContainer(DML::DesignContainer::Cast(*itDesignContainer));
		}
	}
}

void Graph::ProcessComponent(DML::Component &oComponent)
{
	//size_t found;
	//found = string(oComponent.name()).find("MDAO");
	//if (found!=string::npos)
	//{
	//	bMDAO = true;
	//}
	//else
	//{
	//	bMDAO = false;
	//}
	//found = string(oComponent.name()).find("CMD");
	//if (found!=string::npos)
	//{
	//	bSilentMode = true;
	//}
	//else
	//{
	//	bSilentMode = false;
	//}

	if (GetModifiedName(oComponent.name()) == "C13_Diesel")
	{
		int zso = 0;
	}

	MapContainer::iterator it;

	if (uinTraversal == 1)
	{
		// create this component
		sfFile << oComponent;

		ioPortNumbers.insert( make_pair(oComponent.uniqueId(),
			PortNumbers(oComponent)));

		set<DML::ModelicaModel> sModelicaModel;
		sModelicaModel = oComponent.ModelicaModel_kind_children();

		for (set<DML::ModelicaModel>::const_iterator it = sModelicaModel.begin();
			it != sModelicaModel.end();
			++it)
		{
			if (uinTraversal == 1)
			{
				string err = "";
				err = "Modelica type models are not supported during simulink model generation: ";
				err += GMEConsole::Formatter::MakeObjectHyperlink(it->name(), *it);
				pqMessages.push(Message(err, MSG_WARNING));
			}
		}

	}

	if (uinTraversal == 2)
	{
		this->ProcessIoPorts(oComponent);

		set<DML::ValueFlowTarget> vft;
		vft = oComponent.ValueFlowTarget_kind_children();
		for (set<DML::ValueFlowTarget>::iterator it = vft.begin();
			it != vft.end();
			++it)
		{
			ProcessValueFlowTarget(DML::ValueFlowTarget::Cast(*it));
		}
	}

	set<DML::BondGraph> sPhyComponent;
	sPhyComponent = oComponent.BondGraph_kind_children();

	// Process physical component models
	for (set<DML::BondGraph>::iterator itPC = sPhyComponent.begin();
			 itPC != sPhyComponent.end();
			 ++itPC)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itPC)))
		{
			this->ProcessPhyComponent(DML::BondGraph::Cast(*itPC));
		}
	}

	if (uinTraversal == 3)
	{
		// TODO: Process connections
		set<DML::InformationFlow> sIFlow;
		sIFlow = oComponent.InformationFlow_kind_children();

		for (set<DML::InformationFlow>::iterator itIFlow = sIFlow.begin();
				 itIFlow != sIFlow.end();
				 ++itIFlow)
		{
			if (IsNeedToBeProcessed(Udm::Object::Cast(*itIFlow)))
			{
				this->ProcessConnection(DML::InformationFlow::Cast(*itIFlow));
			}
		}

		set<DML::SignalConnectionType> sSType;
		sSType = oComponent.SignalConnectionType_kind_children();

		for (set<DML::SignalConnectionType>::iterator itSType = sSType.begin();
				 itSType != sSType.end();
				 ++itSType)
		{
			if (IsNeedToBeProcessed(Udm::Object::Cast(*itSType)))
			{
				this->ProcessConnection(DML::SignalConnectionType::Cast(*itSType));
			}
		}

		set<DML::ConnectionBase> sCBase;
		sCBase = oComponent.ConnectionBase_kind_children();

		for (set<DML::ConnectionBase>::iterator itCBase = sCBase.begin();
				 itCBase != sCBase.end();
				 ++itCBase)
		{
			this->ProcessConnection(DML::ConnectionBase::Cast(*itCBase));
		}

		set<DML::ValueFlow2SignalPort> sVf2Sp;
		sVf2Sp = oComponent.ValueFlow2SignalPort_kind_children();

		for (set<DML::ValueFlow2SignalPort>::iterator itsVf2Sp = sVf2Sp.begin();
				 itsVf2Sp != sVf2Sp.end();
				 ++itsVf2Sp)
		{
			this->ProcessConnection(DML::ValueFlow2SignalPort::Cast(*itsVf2Sp));
		}
	}

}

// HACK: esmol support
set<DML::MgaObject> esmolExclude;

void Graph::ProcessComponentAssembly(DML::ComponentAssembly &oCAssembly)
{
	MapContainer::iterator it;
	
	if (esmolExclude.find(oCAssembly) != esmolExclude.end())
	{
		return;
	}

	if (uinTraversal == 1)
	{
		// create this component
		sfFile << oCAssembly;

		ioPortNumbers.insert( make_pair(oCAssembly.uniqueId(),
																		PortNumbers(oCAssembly)));
	}

	if (uinTraversal == 2)
	{
		this->ProcessIoPorts(oCAssembly);

		set<DML::ValueFlowTarget> vft;
		vft = oCAssembly.ValueFlowTarget_kind_children();
		for (set<DML::ValueFlowTarget>::iterator it = vft.begin();
			it != vft.end();
			++it)
		{
			ProcessValueFlowTarget(DML::ValueFlowTarget::Cast(*it));
		}
	}

	set<DML::Component> sComponent;
	sComponent = oCAssembly.Component_kind_children();

	// Process sub component models
	for (set<DML::Component>::iterator itComponent = sComponent.begin();
			 itComponent != sComponent.end();
			 ++itComponent)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itComponent)))
		{
			this->ProcessComponent(DML::Component::Cast(*itComponent));
		}
	}

	set<DML::ComponentAssembly> sAssembly;
	sAssembly = oCAssembly.ComponentAssembly_kind_children();

	// Process sub ComponentAssembly models
	for (set<DML::ComponentAssembly>::iterator itAssembly = sAssembly.begin();
			 itAssembly != sAssembly.end();
			 ++itAssembly)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itAssembly)))
		{
			this->ProcessComponentAssembly(DML::ComponentAssembly::Cast(*itAssembly));
		}
	}

	if (uinTraversal == 3)
	{
		// TODO: Process connections
		set<DML::SignalConnectionType> sSType;
		sSType = oCAssembly.SignalConnectionType_kind_children();

		for (set<DML::SignalConnectionType>::iterator itSType = sSType.begin();
				 itSType != sSType.end();
				 ++itSType)
		{
			if (IsNeedToBeProcessed(Udm::Object::Cast(*itSType)))
			{
				this->ProcessConnection(DML::SignalConnectionType::Cast(*itSType));
			}
		}
		
		set<DML::InformationFlow> sIFlow;
		sIFlow = oCAssembly.InformationFlow_kind_children();

		for (set<DML::InformationFlow>::iterator itIFlow = sIFlow.begin();
				 itIFlow != sIFlow.end();
				 ++itIFlow)
		{
			if (IsNeedToBeProcessed(Udm::Object::Cast(*itIFlow)))
			{
				this->ProcessConnection(DML::InformationFlow::Cast(*itIFlow));
			}
		}

		set<DML::ConnectionBase> sCBase;
		sCBase = oCAssembly.ConnectionBase_kind_children();

		for (set<DML::ConnectionBase>::iterator itCBase = sCBase.begin();
				 itCBase != sCBase.end();
				 ++itCBase)
		{
			this->ProcessConnection(DML::ConnectionBase::Cast(*itCBase));
		}

		//set<DML::ValueFlow2SignalPort> sVf2Sp;
		//sVf2Sp = oCAssembly.ValueFlow2SignalPort_kind_children();

		//for (set<DML::ValueFlow2SignalPort>::iterator itsVf2Sp = sVf2Sp.begin();
		//		 itsVf2Sp != sVf2Sp.end();
		//		 ++itsVf2Sp)
		//{
		//	this->ProcessConnection(DML::ValueFlow2SignalPort::Cast(*itsVf2Sp));
		//}
	}
}

void Graph::ProcessPhyComponent(DML::BondGraph &oPhyComponent)
{
	string szPath;
	szPath = GetObjectPath(oPhyComponent);

	// TODO: do we need to process the whole PhyComp even if the modeler selected
	// only one element ???

	if (uinTraversal == 1)
	{
		// create this component
		sfFile << oPhyComponent;

		ioPortNumbers.insert( make_pair(oPhyComponent.uniqueId(),
																		PortNumbers(oPhyComponent)));
	}

	set<DML::BondGraph> sPhyComponent;
	sPhyComponent = oPhyComponent.BondGraph_kind_children();

	// Process sub physical component models
	for (set<DML::BondGraph>::iterator itPC = sPhyComponent.begin();
			 itPC != sPhyComponent.end();
			 ++itPC)
	{
		if (IsNeedToBeProcessed(Udm::Object::Cast(*itPC)))
		{
			this->ProcessPhyComponent(DML::BondGraph::Cast(*itPC));
		}
	}

	// Process this physical component
	if (uinTraversal == 1)
	{
		set<DML::ExtendedElements> sExtElements;
		sExtElements = oPhyComponent.ExtendedElements_kind_children();

		// Process extended elements
		for (set<DML::ExtendedElements>::iterator itEE = sExtElements.begin();
				 itEE != sExtElements.end();
				 ++itEE)
		{
			this->ProcessExtendedElements(DML::ExtendedElements::Cast(*itEE));
		}

		set<DML::LocalSignal> sLocalSignal;
		sLocalSignal = oPhyComponent.LocalSignal_kind_children();

		// Process Local Signals
		for (set<DML::LocalSignal>::iterator itLSignal = sLocalSignal.begin();
				 itLSignal != sLocalSignal.end();
				 ++itLSignal)
		{
			this->ProcessLocalSignal(DML::LocalSignal::Cast(*itLSignal));
		}

		set<DML::BGNode> sBGNode;
		sBGNode = oPhyComponent.BGNode_kind_children();

		// Process BGNodes
		for (set<DML::BGNode>::iterator itBGNode = sBGNode.begin();
				 itBGNode != sBGNode.end();
				 ++itBGNode)
		{
			this->ProcessBGNode(DML::BGNode::Cast(*itBGNode));
		}

		set<DML::Bond> sBond;
		sBond = oPhyComponent.Bond_kind_children();

		// Process bonds
		for (set<DML::Bond>::iterator itBond = sBond.begin();
				 itBond != sBond.end();
				 ++itBond)
		{
			this->ProcessBond(DML::Bond::Cast(*itBond));
		}

        set<DML::BGPowerPort> sPowerPort;
		sPowerPort = oPhyComponent.BGPowerPort_kind_children();

		// TODO: Process PowerPorts review
#pragma region POWERPORTS
		// PowerPorts
		for (set<DML::BGPowerPort>::iterator itPowerPort = sPowerPort.begin();
				 itPowerPort != sPowerPort.end();
				 ++itPowerPort)
		{
			string pptype = string(itPowerPort->type().name());
			
			set<Udm::Object> InConnections;
			set<Udm::Object> tempInSet;
			set<Udm::Object> OutConnections;
			set<Udm::Object> tempOutSet;

			if(pptype == "ElectricalPort")
			{
				InConnections = (*itPowerPort).getAssociation(DML::ElectricalPort::meta_srcElectricalConnection);

				OutConnections = (*itPowerPort).getAssociation(DML::ElectricalPort::meta_dstElectricalConnection);
			}
			else if(pptype == "MechanicalDPort")
			{
				InConnections = (*itPowerPort).getAssociation(DML::MechanicalDPort::meta_srcMechanicalDConnection);

				OutConnections = (*itPowerPort).getAssociation(DML::MechanicalDPort::meta_dstMechanicalDConnection);
			}
			else if(pptype == "MechanicalRPort")
			{
				InConnections = (*itPowerPort).getAssociation(DML::MechanicalRPort::meta_srcMechanicalRConnection);

				OutConnections = (*itPowerPort).getAssociation(DML::MechanicalRPort::meta_dstMechanicalRConnection);
			}
			else if(pptype == "ThermalPort")
			{
				InConnections = (*itPowerPort).getAssociation(DML::ThermalPort::meta_srcThermalConnection);

				OutConnections = (*itPowerPort).getAssociation(DML::ThermalPort::meta_dstThermalConnection);
			}
			else if(pptype == "HydraulicPort")
			{
				InConnections = (*itPowerPort).getAssociation(DML::HydraulicPort::meta_srcHydraulicConnection);

				OutConnections = (*itPowerPort).getAssociation(DML::HydraulicPort::meta_dstHydraulicConnection);
			}

#ifdef COMPONENTPOWERPROPAGATION
			// COMPONENT POWER PROPAGATION: first element
			if (Udm::IsDerivedFrom(itPowerPort->type(),DML::PowerPortType::meta))
			{
				tempInSet = (*itPowerPort).getAssociation(DML::BGPowerPort::meta_srcPhysicalPort2PowerPort);
			}
			if (Udm::IsDerivedFrom(itPowerPort->type(),DML::BGPowerPort::meta))
			{
				tempOutSet = (*itPowerPort).getAssociation(DML::BGPowerPort::meta_dstPowerPort2PhysicalPort);
			}
#endif

			for(set<Udm::Object>::iterator tempPPIt = tempInSet.begin(); tempPPIt != tempInSet.end(); ++tempPPIt)
			{
				InConnections.insert(*tempPPIt);
			}
			for(set<Udm::Object>::iterator tempPPIt = tempOutSet.begin(); tempPPIt != tempOutSet.end(); ++tempPPIt)
			{
				OutConnections.insert(*tempPPIt);
			}

			vector<vector<DML::MgaObject>> Vectors;
			
			// If PP is a source OR chain_length > 1
			if((InConnections.size() == 0) && (OutConnections.size() != 0))
			{			
				vector<DML::MgaObject> NewPPVector;
				NewPPVector.push_back(*itPowerPort);
				Vectors.push_back(NewPPVector);
				GetNextPowerPort(DML::MgaObject::Cast(*itPowerPort), Vectors);
				
				for(vector<vector<DML::MgaObject>>::iterator VectorIt = Vectors.begin();
					VectorIt != Vectors.end(); VectorIt++)
				{
					vector<DML::MgaObject> PowerPortVector = *VectorIt;
					PowerChain ThisChain;

					/* DEBUG PURPOSE BLOCK */
					string DebugString = "";
					for(vector<DML::MgaObject>::iterator PPChainIterator = PowerPortVector.begin();
						PPChainIterator != PowerPortVector.end(); ++PPChainIterator)
					{
						DebugString += " ";
						DebugString += string(PPChainIterator->name());
					}
					/* END OF DEBUG PURPOSE BLOCK*/

					ThisChain.PowerPortVector = PowerPortVector;
					ThisChain.DomainType = PowerPortVector.begin()->type().name();
					
					set<Udm::Object> TempSourceJunctionSet = PowerPortVector.begin()->getAssociation(DML::BGPowerPort::meta_srcJunction2PowerPort);
					set<Udm::Object> TempDestinationJunctionSet = (PowerPortVector.end()-1)->getAssociation(DML::BGPowerPort::meta_dstPowerPort2Junction);
					
					string szSourcePath = GetObjectPath(*PowerPortVector.begin());
					string szDestinationPath = GetObjectPath(*(PowerPortVector.end()-1));

					for(set<Udm::Object>::iterator TempSourceJunctionIt = TempSourceJunctionSet.begin();
						TempSourceJunctionIt != TempSourceJunctionSet.end(); ++TempSourceJunctionIt)
					{
						ThisChain.SourceJunctionSet.insert(DML::Junction::Cast(*TempSourceJunctionIt));
					}

					for(set<Udm::Object>::iterator TempDestinationJunctionIt = TempDestinationJunctionSet.begin();
						TempDestinationJunctionIt != TempDestinationJunctionSet.end(); ++TempDestinationJunctionIt)
					{
						ThisChain.DestinationJunctionSet.insert(DML::Junction::Cast(*TempDestinationJunctionIt));
					}

					ThisChain.ChainWidthUpdate();
					PowerPortChains.push_back(ThisChain);

					// Refresh AllJunction's connectioncounts
					for(set<DML::Junction>::iterator srcJunctionIt = ThisChain.SourceJunctionSet.begin();
						srcJunctionIt != ThisChain.SourceJunctionSet.end(); ++srcJunctionIt)
					{
						for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
						{
							if(jit->p.uniqueId() == srcJunctionIt->uniqueId())
							{
								jit->connectioncount += ThisChain.ChainWidth;
								break;
							}
						}
					}
					
					// Destination junction might not be in the AllJunctions vector, because of the recursive traversal
					// Add it, if it this is the case
					for(set<DML::Junction>::iterator dstJunctionIt = ThisChain.DestinationJunctionSet.begin();
						dstJunctionIt != ThisChain.DestinationJunctionSet.end(); ++dstJunctionIt)
					{
						bool contains = false;
						for(vector<TJunction>::iterator checkID = AllJunctions.begin(); checkID != AllJunctions.end(); checkID++)
						{
							if(checkID->p.uniqueId() == dstJunctionIt->uniqueId())
							{
								contains = true;
								break;
							}
						}
						if(!contains)
						{
							AllJunctions.push_back(*dstJunctionIt);
						}

						for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
						{
							if(jit->p.uniqueId() == dstJunctionIt->uniqueId())
							{
								jit->connectioncount += ThisChain.ChainWidth;
								break;
							}
						}
					}

			
					// Place bond in AllBonds structure
					for(set<DML::Junction>::iterator SourceJunctionIt = ThisChain.SourceJunctionSet.begin();
						SourceJunctionIt != ThisChain.SourceJunctionSet.end(); SourceJunctionIt++)
					{
						for(set<DML::Junction>::iterator DestinationJunctionIt = ThisChain.DestinationJunctionSet.begin();
						DestinationJunctionIt != ThisChain.DestinationJunctionSet.end(); DestinationJunctionIt++)
						{
							TBond NewBond(uidbond,
											 DML::BGNode::Cast(*SourceJunctionIt),
											 DML::BGNode::Cast(*DestinationJunctionIt),
											 szPath);
							AllBonds.insert(make_pair(uidbond, NewBond));
							uidbond++;
						}
					}
				}
			}
			else if(InConnections.size() == 0 && OutConnections.size() == 0) // gonna be the source of a one-length chain
			{
				vector<DML::MgaObject> NewPPVector;
				NewPPVector.push_back(*itPowerPort);
				Vectors.push_back(NewPPVector);
				GetNextPowerPort(DML::MgaObject::Cast(*itPowerPort), Vectors);
				
				for(vector<vector<DML::MgaObject>>::iterator VectorIt = Vectors.begin();
					VectorIt != Vectors.end(); VectorIt++)
				{
					vector<DML::MgaObject> PowerPortVector = *VectorIt;
					PowerChain ThisChain;

					ThisChain.PowerPortVector = PowerPortVector;
					ThisChain.DomainType = PowerPortVector.begin()->type().name();
					
					set<Udm::Object> TempSourceJunctionSet = PowerPortVector.begin()->getAssociation(DML::BGPowerPort::meta_srcJunction2PowerPort);
					set<Udm::Object> TempDestinationJunctionSet = (PowerPortVector.begin())->getAssociation(DML::BGPowerPort::meta_dstPowerPort2Junction);
					
					for(set<Udm::Object>::iterator TempSourceJunctionIt = TempSourceJunctionSet.begin();
						TempSourceJunctionIt != TempSourceJunctionSet.end(); ++TempSourceJunctionIt)
					{
						ThisChain.SourceJunctionSet.insert(DML::Junction::Cast(*TempSourceJunctionIt));
					}

					for(set<Udm::Object>::iterator TempDestinationJunctionIt = TempDestinationJunctionSet.begin();
						TempDestinationJunctionIt != TempDestinationJunctionSet.end(); ++TempDestinationJunctionIt)
					{
						ThisChain.DestinationJunctionSet.insert(DML::Junction::Cast(*TempDestinationJunctionIt));
					}

					ThisChain.ChainWidthUpdate();
					PowerPortChains.push_back(ThisChain);
			
					// Source junction might not be in the AllJunctions vector, because of the recursive traversal
					// Add it, if it this is the case
					for(set<DML::Junction>::iterator srcJunctionIt = ThisChain.SourceJunctionSet.begin();
						srcJunctionIt != ThisChain.SourceJunctionSet.end(); ++srcJunctionIt)
					{
						bool contains = false;
						for(vector<TJunction>::iterator checkID = AllJunctions.begin(); checkID != AllJunctions.end(); checkID++)
						{
							if(checkID->p.uniqueId() == srcJunctionIt->uniqueId())
							{
								contains = true;
								break;
							}
						}
						if(!contains)
						{
							AllJunctions.push_back(*srcJunctionIt);
						}

						for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
						{
							if(jit->p.uniqueId() == srcJunctionIt->uniqueId())
							{
								jit->connectioncount += ThisChain.ChainWidth;
								break;
							}
						}
					}

					// Destination junction might not be in the AllJunctions vector, because of the recursive traversal
					// Add it, if it this is the case
					for(set<DML::Junction>::iterator dstJunctionIt = ThisChain.DestinationJunctionSet.begin();
						dstJunctionIt != ThisChain.DestinationJunctionSet.end(); ++dstJunctionIt)
					{
						bool contains = false;
						for(vector<TJunction>::iterator checkID = AllJunctions.begin(); checkID != AllJunctions.end(); checkID++)
						{
							if(checkID->p.uniqueId() == dstJunctionIt->uniqueId())
							{
								contains = true;
								break;
							}
						}
						if(!contains)
						{
							AllJunctions.push_back(*dstJunctionIt);
						}

						for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
						{
							if(jit->p.uniqueId() == dstJunctionIt->uniqueId())
							{
								jit->connectioncount += ThisChain.ChainWidth;
								break;
							}
						}
					}
			
					// Place bond in AllBonds structure
					for(set<DML::Junction>::iterator SourceJunctionIt = ThisChain.SourceJunctionSet.begin();
						SourceJunctionIt != ThisChain.SourceJunctionSet.end(); SourceJunctionIt++)
					{
						for(set<DML::Junction>::iterator DestinationJunctionIt = ThisChain.DestinationJunctionSet.begin();
						DestinationJunctionIt != ThisChain.DestinationJunctionSet.end(); DestinationJunctionIt++)
						{
							TBond NewBond(uidbond,
											 DML::BGNode::Cast(*SourceJunctionIt),
											 DML::BGNode::Cast(*DestinationJunctionIt),
											 szPath);
							AllBonds.insert(make_pair(uidbond, NewBond));
							uidbond++;
						}
					}
				}
			}
		} // END OF PowerPort loop
#pragma endregion POWERPORTS

		// TODO: Checks (Connection: MO, etc.)
		set<DML::SignalConnection> sSignalConnection;
		sSignalConnection = oPhyComponent.SignalConnection_kind_children();
		int nError;
		nError = CheckSignalConstraints(sSignalConnection);
	}

	if (uinTraversal == 2)
	{
		this->ProcessIoPorts(oPhyComponent);

		set<DML::ControlNode> sControlNode;
		sControlNode = oPhyComponent.ControlNode_kind_children();
		for (set<DML::ControlNode>::iterator itCNode = sControlNode.begin();
				 itCNode != sControlNode.end();
				 ++itCNode)
		{
			this->ProcessControlNode(DML::ControlNode::Cast(*itCNode));
		}
	}

	if (uinTraversal == 3)
	{
		set<DML::ConnectionBase> sCBase;
		sCBase = oPhyComponent.ConnectionBase_kind_children();

		for (set<DML::ConnectionBase>::iterator itCBase = sCBase.begin();
				 itCBase != sCBase.end();
				 ++itCBase)
		{
			this->ProcessConnection(DML::ConnectionBase::Cast(*itCBase));
		}
	}
}


void Graph::ProcessExtendedElements(DML::ExtendedElements &oExtElements)
{
	sfFile << oExtElements;
}


void Graph::ProcessBGNode(DML::BGNode &oBGNode)
{
	// TODO: review this function
	string szType = oBGNode.type().name();

	//if ((oBGNode.type() == DML::OneJunction::meta) ||
	//		(oBGNode.type() == DML::ZeroJunction::meta))
	if (Udm::IsDerivedFrom(oBGNode.type(), DML::Junction::meta))
	{
    DML::Junction junc = DML::Junction::Cast(oBGNode);
    set<DML::Signal2Junction> s2j = junc.srcSignal2Junction();
    if (s2j.size() > 0)
    {
      if (junc.OnCondition()  == "")
      {
			  string err = "";
			  err = "The OnCondition of the following junction must be specified: ";
			  err += GMEConsole::Formatter::MakeObjectHyperlink(junc.name(), junc);
        pqMessages.push(Message(err, MSG_ERROR));
      }
      if (junc.OffCondition() == "")
      {
        string err = "";
			  err = "The OffCondition of the following junction must be specified: ";
			  err += GMEConsole::Formatter::MakeObjectHyperlink(junc.name(), junc);
        pqMessages.push(Message(err, MSG_ERROR));
      }
    }
		bool contains = false;
		for (vector<TJunction>::iterator checkID = AllJunctions.begin();
				 checkID != AllJunctions.end();
				 ++checkID)
		{
			if(checkID->p.uniqueId() == oBGNode.uniqueId())
			{
				contains = true;
				break;
			}
		}
		if(!contains)
		{
			AllJunctions.push_back(DML::Junction::Cast(oBGNode));
		}
	}
	else
	{
		string szParameterValue;
		szParameterValue = DML::BGElement::Cast(oBGNode).ParameterValue();
		if (szParameterValue == "")
		{
			string szErrMsg = "";
			szErrMsg = "The parameter value of the following element is empty: ";
			szErrMsg += GMEConsole::Formatter::MakeObjectHyperlink( oBGNode.name(),
																															oBGNode);
			pqMessages.push(Message(szErrMsg, MSG_ERROR));
		}
	}
}


void Graph::ProcessBond(DML::Bond &oBond)
{
	// TODO: review this function

	string szPath;
	szPath = GetParentPath(oBond);

	string _btypename = oBond.type().name();
	if ((_btypename == "BondJ2E") || (_btypename == "BondE2J") || (_btypename == "BondJ2J"))
	{
		// Bond tmpBond(oBond);	
		if (_btypename == "BondJ2E")
		{
			TBond tmpBond(oBond.uniqueId(),DML::BGNode::Cast(DML::BondJ2E::Cast(oBond).srcBondJ2E_end()),DML::BGNode::Cast(DML::BondJ2E::Cast(oBond).dstBondJ2E_end()),szPath);
			AllBonds.insert(make_pair(oBond.uniqueId(),tmpBond));
		}
		else if (_btypename == "BondE2J")
		{
			TBond tmpBond(oBond.uniqueId(),
				DML::BGNode::Cast(DML::BondE2J::Cast(oBond).srcBondE2J_end()),
				DML::BGNode::Cast(DML::BondE2J::Cast(oBond).dstBondE2J_end()),
				szPath);
			AllBonds.insert(make_pair(oBond.uniqueId(),tmpBond));
		}
		else if (_btypename == "BondJ2J")
		{
			TBond tmpBond(oBond.uniqueId(),DML::BGNode::Cast(DML::BondJ2J::Cast(oBond).srcBondJ2J_end()),DML::BGNode::Cast(DML::BondJ2J::Cast(oBond).dstBondJ2J_end()),szPath);
			AllBonds.insert(make_pair(oBond.uniqueId(),tmpBond));
		}
		else
		{
			// unhandled bonds
		}
	}
}




void Graph::ProcessControlNode(DML::ControlNode &oControlNode)
{
	// Inward connection process order: Junction, Constant, Signal
	
	char buffer[33];
	int radix = 10;
	string src, dst;
	string type;
	string path = GetParentPath(oControlNode);
	unsigned int portnumber = 1;

	if(string(oControlNode.type().name()) == "Modulation")
	{
		DML::Modulation omod = DML::Modulation::Cast(oControlNode);
		if(string(omod.Expression()) == "")
		{
			std::string errmsg = "";
			errmsg = "Please provide function specification for Modulation element: ";
			errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oControlNode.name(), oControlNode);
			pqMessages.push(Message(errmsg, MSG_ERROR));
		}

		// Process input connections
		// - from junction
		set<DML::Junction2Modulation> sJ2M = omod.srcJunction2Modulation();
		for (set<DML::Junction2Modulation>::iterator it = sJ2M.begin(); it != sJ2M.end(); ++it)	
		{
			// src connections
			src = DML::Junction(it->srcJunction2Modulation_end()).name();
			src += "/";

			int numbonds;
			for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
			{
				if(jit->p.uniqueId() == DML::Junction(it->srcJunction2Modulation_end()).uniqueId())
				{
					numbonds = jit->connectioncount;
					_ltoa_s(numbonds+1,buffer,radix);
					src += string(buffer);
					break;
				}
			}

			dst = omod.name();
			dst += "/";
			_ltoa_s(portnumber++,buffer,radix);
			dst += string(buffer);
			
			conns.push_back(BlockConnection(path,src,dst));
		}

		// - from constant
		/*set<DML::Constant2Modulation> sC2M = omod.srcConstant2Modulation();
		for (set<DML::Constant2Modulation>::iterator it = sC2M.begin(); it != sC2M.end(); ++it)	
		{
			// src connections
			src = DML::Constant(it->srcConstant2Modulation_end()).name();
			src += "/1";

			// dst connections
			dst = omod.name();
			dst += "/";
			_ltoa_s(portnumber++, buffer, radix);
			dst += string(buffer);
			
			conns.push_back(BlockConnection(path,src,dst));
		}*/


		// Process outward connections
		std::set<DML::Modulation2BGMElement> sBGM = omod.dstModulation2BGMElement();
		for (std::set<DML::Modulation2BGMElement>::iterator it = sBGM.begin(); it != sBGM.end(); ++it)
		{
			type = string(DML::BGModulatedElement(it->dstModulation2BGMElement_end()).type().name());
			
			// Oneports
			//if ((type == "MSe") || (type == "MSf") || (type == "MR") || (type == "MC") || (type == "MI"))
			if (Udm::IsDerivedFrom(DML::BGModulatedElement(it->dstModulation2BGMElement_end()).type(),
														 DML::OnePort::meta))
			{
				src = omod.name();
				src += "/1";
				dst = DML::BGModulatedElement(it->dstModulation2BGMElement_end()).name();

				if (!Udm::IsDerivedFrom(it->type(), DML::Storage::meta))
				{
					dst += "/2";
				}
				else
				{
					DML::Storage s = DML::Storage::Cast(*it);
					set<DML::Signal2InitialValue> s2iv = s.srcSignal2InitialValue();
					if (s2iv.empty())
					{
						// initial value is a constant
						dst += "/2";
					}
					else
					{
						// initial value is a signal, its port number 2
						// the modulation signal's port number must be 3
						dst += "/3";
					}
				}

				//dst += "/2";	
				conns.push_back(BlockConnection(path, src, dst));
			}
			// Twoports
			else if ((type == "MTF") || (type == "MGY"))
			{
				src = omod.name();
				src += "/1";
				dst = DML::BGModulatedElement(it->dstModulation2BGMElement_end()).name();
				dst += "/3";	
				conns.push_back(BlockConnection(path,src,dst));					
			}
		}

		//printScript(outfile, omod, path);
	}

	else if(string(oControlNode.type().name()) == "Switching")
	{
		DML::Switching osw = DML::Switching::Cast(oControlNode);
		if(string(osw.SwExpression()) == "")	
		{
			std::string errmsg = "";
			errmsg = "Please provide function specification for Switching element: ";
			errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oControlNode.name(), oControlNode);
			pqMessages.push(Message(errmsg, MSG_ERROR));
		}

		// Process Switching element's inward connections
		// - from junction
		std::set<DML::Junction2Switching> J2Sw = osw.srcJunction2Switching();
		for (std::set<DML::Junction2Switching>::iterator it = J2Sw.begin(); it != J2Sw.end(); ++it)	
		{
			src = DML::Junction(it->srcJunction2Switching_end()).name();
			src += "/";
			int numbonds;
			for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
			{
				if(jit->p.uniqueId() == DML::Junction(it->srcJunction2Switching_end()).uniqueId())
				{
					numbonds = jit->connectioncount;
					_ltoa_s(numbonds+1,buffer,radix);
					src += string(buffer);
					break;
				}
			}

			dst = osw.name();
			dst += "/";
			_ltoa_s(portnumber,buffer,radix);
			dst += string(buffer);
			
			conns.push_back(BlockConnection(path, src, dst));

			++portnumber;
		}

		// - from constant
		/*std::set<DML::Constant2Switching> C2Sw = osw.srcConstant2Switching();
		for (std::set<DML::Constant2Switching>::iterator it = C2Sw.begin(); it != C2Sw.end(); ++it)	
		{
			src = DML::Constant(it->srcConstant2Switching_end()).name();
			src += "/1";

			dst = osw.name();
			dst += "/";
			_ltoa_s(portnumber,buffer,radix);		
			dst += string(buffer);

			conns.push_back(BlockConnection(path, src, dst));

			++portnumber;
		}*/

		// Process Switching element's outward connections
		std::set<DML::Switching2Junction> Sw2J = osw.dstSwitching2Junction();
		for (std::set<DML::Switching2Junction>::iterator it = Sw2J.begin(); it != Sw2J.end(); ++it)	
		{
			src = osw.name();
			src += "/1";

			dst = DML::Junction(it->dstSwitching2Junction_end()).name();
			dst += "/";

			int numbonds;
			for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
			{
				if(jit->p.uniqueId() == DML::Junction(it->dstSwitching2Junction_end()).uniqueId())
				{
					numbonds = jit->connectioncount;
					_ltoa_s(numbonds + 1,buffer,radix);
					dst += string(buffer);
					break;
				}
			}
			
			conns.push_back(BlockConnection(path, src, dst));
		}
		//printScript(outfile, osw, path);
	}
}

void Graph::ProcessIoPorts(DML::TestComponent &oTestComponent)
{
	int iY;
	int i;

	set<DML::ElectricalSignalPort> sAllSignal;
	set<DML::ElectricalSignalPort>::iterator itAllSignal;
	
	set<DML::ElectricalSignalPort> sInSignalE;
	set<DML::ElectricalSignalPort>::iterator itInSignalE;
	set<DML::ElectricalSignalPort> sOutSignalE;
	set<DML::ElectricalSignalPort>::iterator itOutSignalE;

	// separate in/out signal ports
	sAllSignal = oTestComponent.ElectricalSignalPort_kind_children();

	for (itAllSignal = sAllSignal.begin();
		itAllSignal != sAllSignal.end();
		++itAllSignal)
	{
		InOut io = CheckType(*itAllSignal);

		if (io == IO_UNDEFINED)
		{
			string err = "";
			err = "Unrecognized port";
			err += GMEConsole::Formatter::MakeObjectHyperlink(itAllSignal->name(), *itAllSignal);
			pqMessages.push(Message(err, MSG_WARNING));
		}
		else if (io == IO_INPUT)
		{
			sInSignalE.insert(*itAllSignal);
			string err = "";
			err = "Input port: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(itAllSignal->name(), *itAllSignal);
			//pqMessages.push(Message(err, MSG_INFO));
		}
		else if (io == IO_OUTPUT)
		{
			sOutSignalE.insert(*itAllSignal);
			string err = "";
			err = "Output port: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(itAllSignal->name(), *itAllSignal);
			//pqMessages.push(Message(err, MSG_INFO));
		}
	}

	//sInSignal = oPhyComponent.InSignal_kind_children();
	itInSignalE = sInSignalE.begin();

	iY = 0;
	i = 1;

	while (!sInSignalE.empty())
	{
		itInSignalE = sInSignalE.begin();
		iY = GetObjectYPosition(itInSignalE->position(),"");
		for (set<DML::ElectricalSignalPort>::iterator _itISig = sInSignalE.begin(); _itISig != sInSignalE.end(); ++_itISig)
		{
			if (iY > GetObjectYPosition(_itISig->position(),""))
			{
				iY = GetObjectYPosition(_itISig->position(),"");
				itInSignalE = _itISig;
			}
		}	
		//ioPortNumbers.find(oPhyComponent.uniqueId())->second.signalIN++;
		vIoPorts.push_back(*itInSignalE);
		sInSignalE.erase(itInSignalE);
	}

	//sOutSignal = oPhyComponent.OutSignal_kind_children();
	itOutSignalE = sOutSignalE.begin();

	iY = 0;
	i = 1;

	while (!sOutSignalE.empty())
	{
		itOutSignalE = sOutSignalE.begin();
		iY = GetObjectYPosition(itOutSignalE->position(),"");
		for (set<DML::ElectricalSignalPort>::iterator _itOSig = sOutSignalE.begin(); _itOSig != sOutSignalE.end(); ++_itOSig)
		{
			if (iY > GetObjectYPosition(_itOSig->position(),""))
			{
				iY = GetObjectYPosition(_itOSig->position(),"");
				itOutSignalE = _itOSig;
			}
		}	
		//ioPortNumbers.find(oPhyComponent.uniqueId())->second.signalOUT++;
		vIoPorts.push_back(*itOutSignalE);
		sOutSignalE.erase(itOutSignalE);
	}

	// InSignal and OutSignal
	// int iY;
	// int i;

	set<DML::InSignal> sInSignal;
	set<DML::InSignal>::iterator itInSignal;
	set<DML::OutSignal> sOutSignal;
	set<DML::OutSignal>::iterator itOutSignal;

	sInSignal = oTestComponent.InSignal_kind_children();
	itInSignal = sInSignal.begin();

	iY = 0;
	i = 1;

	while (!sInSignal.empty())
	{
		itInSignal = sInSignal.begin();
		iY = GetObjectYPosition(itInSignal->position(),"");
		for (set<DML::InSignal>::iterator _itISig = sInSignal.begin(); _itISig != sInSignal.end(); ++_itISig)
		{
			if (iY > GetObjectYPosition(_itISig->position(),""))
			{
				iY = GetObjectYPosition(_itISig->position(),"");
				itInSignal = _itISig;
			}
		}	
		//ioPortNumbers.find(oPhyComponent.uniqueId())->second.signalIN++;
		vIoPorts.push_back(*itInSignal);
		sInSignal.erase(itInSignal);
	}

	sOutSignal = oTestComponent.OutSignal_kind_children();
	itOutSignal = sOutSignal.begin();

	iY = 0;
	i = 1;

	while (!sOutSignal.empty())
	{
		itOutSignal = sOutSignal.begin();
		iY = GetObjectYPosition(itOutSignal->position(),"");
		for (set<DML::OutSignal>::iterator _itOSig = sOutSignal.begin(); _itOSig != sOutSignal.end(); ++_itOSig)
		{
			if (iY > GetObjectYPosition(_itOSig->position(),""))
			{
				iY = GetObjectYPosition(_itOSig->position(),"");
				itOutSignal = _itOSig;
			}				
		}	
		//ioPortNumbers.find(oPhyComponent.uniqueId())->second.signalOUT++;
		vIoPorts.push_back(*itOutSignal);
		sOutSignal.erase(itOutSignal);
	}

}


void Graph::ProcessIoPorts(DML::DesignElement &oDesignElement)
{
	int iY;
	int i;

	set<DML::ElectricalSignalPort> sAllSignal;
	set<DML::ElectricalSignalPort>::iterator itAllSignal;
	
	set<DML::ElectricalSignalPort> sInSignalE;
	set<DML::ElectricalSignalPort>::iterator itInSignalE;
	set<DML::ElectricalSignalPort> sOutSignalE;
	set<DML::ElectricalSignalPort>::iterator itOutSignalE;

	// separate in/out signal ports
	sAllSignal = oDesignElement.ElectricalSignalPort_kind_children();

	for (itAllSignal = sAllSignal.begin();
		itAllSignal != sAllSignal.end();
		++itAllSignal)
	{
		InOut io = CheckType(*itAllSignal);

		if (io == IO_UNDEFINED)
		{
			string err = "";
			err = "Unrecognized port";
			err += GMEConsole::Formatter::MakeObjectHyperlink(itAllSignal->name(), *itAllSignal);
			pqMessages.push(Message(err, MSG_WARNING));
		}
		else if (io == IO_INPUT)
		{
			sInSignalE.insert(*itAllSignal);
			string err = "";
			err = "Input port: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(itAllSignal->name(), *itAllSignal);
			//pqMessages.push(Message(err, MSG_INFO));
		}
		else if (io == IO_OUTPUT)
		{
			sOutSignalE.insert(*itAllSignal);
			string err = "";
			err = "Output port: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(itAllSignal->name(), *itAllSignal);
			//pqMessages.push(Message(err, MSG_INFO));
		}
	}

	//sInSignal = oPhyComponent.InSignal_kind_children();
	itInSignalE = sInSignalE.begin();

	iY = 0;
	i = 1;

	while (!sInSignalE.empty())
	{
		itInSignalE = sInSignalE.begin();
		iY = GetObjectYPosition(itInSignalE->position(),"");
		for (set<DML::ElectricalSignalPort>::iterator _itISig = sInSignalE.begin(); _itISig != sInSignalE.end(); ++_itISig)
		{
			if (iY > GetObjectYPosition(_itISig->position(),""))
			{
				iY = GetObjectYPosition(_itISig->position(),"");
				itInSignalE = _itISig;
			}
		}	
		//ioPortNumbers.find(oPhyComponent.uniqueId())->second.signalIN++;
		vIoPorts.push_back(*itInSignalE);
		sInSignalE.erase(itInSignalE);
	}

	//sOutSignal = oPhyComponent.OutSignal_kind_children();
	itOutSignalE = sOutSignalE.begin();

	iY = 0;
	i = 1;

	while (!sOutSignalE.empty())
	{
		itOutSignalE = sOutSignalE.begin();
		iY = GetObjectYPosition(itOutSignalE->position(),"");
		for (set<DML::ElectricalSignalPort>::iterator _itOSig = sOutSignalE.begin(); _itOSig != sOutSignalE.end(); ++_itOSig)
		{
			if (iY > GetObjectYPosition(_itOSig->position(),""))
			{
				iY = GetObjectYPosition(_itOSig->position(),"");
				itOutSignalE = _itOSig;
			}
		}	
		//ioPortNumbers.find(oPhyComponent.uniqueId())->second.signalOUT++;
		vIoPorts.push_back(*itOutSignalE);
		sOutSignalE.erase(itOutSignalE);
	}

	// InSignal and OutSignal
	// int iY;
	// int i;

	set<DML::InSignal> sInSignal;
	set<DML::InSignal>::iterator itInSignal;
	set<DML::OutSignal> sOutSignal;
	set<DML::OutSignal>::iterator itOutSignal;

	sInSignal = oDesignElement.InSignal_kind_children();
	itInSignal = sInSignal.begin();

	iY = 0;
	i = 1;

	while (!sInSignal.empty())
	{
		itInSignal = sInSignal.begin();
		iY = GetObjectYPosition(itInSignal->position(),"");
		for (set<DML::InSignal>::iterator _itISig = sInSignal.begin(); _itISig != sInSignal.end(); ++_itISig)
		{
			if (iY > GetObjectYPosition(_itISig->position(),""))
			{
				iY = GetObjectYPosition(_itISig->position(),"");
				itInSignal = _itISig;
			}
		}	
		//ioPortNumbers.find(oPhyComponent.uniqueId())->second.signalIN++;
		vIoPorts.push_back(*itInSignal);
		sInSignal.erase(itInSignal);
	}

	sOutSignal = oDesignElement.OutSignal_kind_children();
	itOutSignal = sOutSignal.begin();

	iY = 0;
	i = 1;

	while (!sOutSignal.empty())
	{
		itOutSignal = sOutSignal.begin();
		iY = GetObjectYPosition(itOutSignal->position(),"");
		for (set<DML::OutSignal>::iterator _itOSig = sOutSignal.begin(); _itOSig != sOutSignal.end(); ++_itOSig)
		{
			if (iY > GetObjectYPosition(_itOSig->position(),""))
			{
				iY = GetObjectYPosition(_itOSig->position(),"");
				itOutSignal = _itOSig;
			}				
		}	
		//ioPortNumbers.find(oPhyComponent.uniqueId())->second.signalOUT++;
		vIoPorts.push_back(*itOutSignal);
		sOutSignal.erase(itOutSignal);
	}

}

void Graph::ProcessIoPorts(DML::BondGraph &oPhyComponent)
{
	int iY;
	int i;

	set<DML::InSignal> sInSignal;
	set<DML::InSignal>::iterator itInSignal;
	set<DML::OutSignal> sOutSignal;
	set<DML::OutSignal>::iterator itOutSignal;

	sInSignal = oPhyComponent.InSignal_kind_children();
	itInSignal = sInSignal.begin();

	iY = 0;
	i = 1;

	while (!sInSignal.empty())
	{
		itInSignal = sInSignal.begin();
		iY = GetObjectYPosition(itInSignal->position(),"");
		for (set<DML::InSignal>::iterator _itISig = sInSignal.begin(); _itISig != sInSignal.end(); ++_itISig)
		{
			if (iY > GetObjectYPosition(_itISig->position(),""))
			{
				iY = GetObjectYPosition(_itISig->position(),"");
				itInSignal = _itISig;
			}
		}	
		//ioPortNumbers.find(oPhyComponent.uniqueId())->second.signalIN++;
		vIoPorts.push_back(*itInSignal);
		sInSignal.erase(itInSignal);
	}

	sOutSignal = oPhyComponent.OutSignal_kind_children();
	itOutSignal = sOutSignal.begin();

	iY = 0;
	i = 1;

	while (!sOutSignal.empty())
	{
		itOutSignal = sOutSignal.begin();
		iY = GetObjectYPosition(itOutSignal->position(),"");
		for (set<DML::OutSignal>::iterator _itOSig = sOutSignal.begin(); _itOSig != sOutSignal.end(); ++_itOSig)
		{
			if (iY > GetObjectYPosition(_itOSig->position(),""))
			{
				iY = GetObjectYPosition(_itOSig->position(),"");
				itOutSignal = _itOSig;
			}				
		}	
		//ioPortNumbers.find(oPhyComponent.uniqueId())->second.signalOUT++;
		vIoPorts.push_back(*itOutSignal);
		sOutSignal.erase(itOutSignal);
	}

}


void Graph::ProcessLocalSignal(DML::LocalSignal &oLocalSignal)
{
	string szParentPath;
	string szParentName;

	szParentPath = GetParentPath(oLocalSignal.GetParent());
	szParentName = GetModifiedName(oLocalSignal.GetParent());

	PortMapping pmTmp(oLocalSignal.uniqueId(),
										1,
										szParentPath,
										szParentName,
										GetModifiedName(oLocalSignal.name()));
	pmSignal.push_back(pmTmp);

	sfFile.AddBlock(oLocalSignal, sfFile.MatLabConfig[oLocalSignal.type().name()][ScriptFile::M_BLOCKTYPE]);
}


void Graph::ProcessValueFlowTarget(DML::ValueFlowTarget &oValueFlowTarget)
{
	string szParentPath;
	string szParentName;
	bool bStoreInPortMapping = true;

	szParentPath = GetParentPath(oValueFlowTarget.GetParent());
	szParentName = GetModifiedName(oValueFlowTarget.GetParent());

	if (oValueFlowTarget.getReferencedObject() != Udm::null)
	{
		string err = "";
		err += "Referenced object will not be processed: ";
		err += GMEConsole::Formatter::MakeObjectHyperlink(oValueFlowTarget.name(), oValueFlowTarget);
		//pqMessages.push(Message(err, MSG_WARNING));
	}

	if (oValueFlowTarget.GetParent().type() == DML::TestBench::meta)
	{
		if (oValueFlowTarget.type() == DML::Parameter::meta)
		{
			string value = DML::Parameter::Cast(oValueFlowTarget).Value();
			  if (oValueFlowTarget.name() == "StopTime")
			  {
				StopTime = value;
			  }
			if (value != "")
			{
				MDAOInput.insert(make_pair<string, string>(GetModifiedName(oValueFlowTarget), value));
			}
		}
		else if (oValueFlowTarget.type() == DML::Property::meta)
		{
			string value = DML::Property::Cast(oValueFlowTarget).Value();
      if (oValueFlowTarget.name() == "StopTime")
      {
        StopTime = value;
      }
			if (value != "")
			{
				MDAOInput.insert(make_pair<string, string>(GetModifiedName(oValueFlowTarget), value));
			}
		}
		else
		{
			// do not care
		}
	}

	if (oValueFlowTarget.type() == DML::Metric::meta)
	{
		// metric name, unit
		string unit = "";
		DML::MgaObject unitObj = DML::MgaObject::Cast(oValueFlowTarget.getReferencedObject());
		if (unitObj != Udm::null)
		{
			unit = unitObj.name();
		}
		MDAOOutput.insert(make_pair<string, string>(GetModifiedName(oValueFlowTarget), unit));
		MetricIDs.insert(DML::Metric::Cast(oValueFlowTarget));
	}

	set<DML::ValueFlow2SignalPort> sVf2Sp = oValueFlowTarget.dstValueFlow2SignalPort();
	set<DML::ValueFlow2SignalPort>::iterator itsVf2Sp;
	for (itsVf2Sp = sVf2Sp.begin();
		itsVf2Sp != sVf2Sp.end();
		++itsVf2Sp)
	{
		if (itsVf2Sp->GetParent() != oValueFlowTarget.GetParent())
		{
			// ValueFlowTarget could not be a port for a signal connection
			bStoreInPortMapping = false;
			string err = "";
			err += "ValueFlowTarget could not be port: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(oValueFlowTarget.name(), oValueFlowTarget);
			pqMessages.push(Message(err, MSG_WARNING));
		}
	}

	if (bStoreInPortMapping)
	{
		PortMapping pmTmp(oValueFlowTarget.uniqueId(),
											1,
											szParentPath,
											szParentName,
											GetModifiedName(oValueFlowTarget.name()));
		pmSignal.push_back(pmTmp);

		ValueFlowTargets.insert(oValueFlowTarget);
	}

	sfFile << Udm::Object::Cast(oValueFlowTarget);
}

void Graph::ProcessConnection(DML::InformationFlow &oIFlow)
{
	char buffer[33];
	int radix = 10;

#pragma region InformationFlow
	if (DML::InformationFlow::meta == oIFlow.type())
	{	
		string src, dst, path;
		vector<PortMapping>::iterator itsrc, itdst;
		bool hasSrc = false;
		bool hasDst = false;

		string dstName = DML::SignalPortType::Cast(DML::InformationFlow::Cast(oIFlow).dstInformationFlow_end()).name();
		string srcName = DML::SignalPortType::Cast(DML::InformationFlow::Cast(oIFlow).srcInformationFlow_end()).name();
		
		// Get the two ends of the connection
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if (DML::SignalPortType::Cast(DML::InformationFlow::Cast(oIFlow).srcInformationFlow_end()).uniqueId()
				== it->GetUniqueID())
			{
				itsrc = it;
				hasSrc = true;
			}
			if (DML::SignalPortType::Cast(DML::InformationFlow::Cast(oIFlow).dstInformationFlow_end()).uniqueId()
				== it->GetUniqueID())
			{
				itdst = it;
				hasDst = true;
			}
		}

		if (hasSrc && hasDst)
		{
			if (itsrc->GetPath() == itdst->GetPath())
			{
				string tmp1 = itsrc->GetUniqueName();
				tmp1 += "/1";
				string tmp2 = itdst->GetUniqueName();
				tmp2 += "/1";

				conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
			}
			else if (itsrc->GetPath() == itdst->GetParentPath())
			{
				string tmp1 = itsrc->GetUniqueName();
				tmp1 += "/1";
				string tmp2 = itdst->GetParentName();
				tmp2 += "/";
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);

				conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
			}
			else if (itsrc->GetParentPath() == itdst->GetPath())
			{
				string tmp1 = itsrc->GetParentName();
				tmp1 += "/";
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
				string tmp2 = itdst->GetUniqueName();
				tmp2 += "/1";

				conns.push_back(BlockConnection(itdst->GetPath(), tmp1, tmp2));
			}
			else if (itsrc->GetParentPath() == itdst->GetParentPath())
			{
				string tmp1 = itsrc->GetParentName();
				tmp1 += "/";
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
				string tmp2 = itdst->GetParentName();
				tmp2 += "/";
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);

				conns.push_back(BlockConnection(itdst->GetParentPath(), tmp1, tmp2));
			}
		}
		else
		{
			string err = "";
			err += "Connection skipped: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(oIFlow.name(), oIFlow);
			pqMessages.push(Message(err, MSG_WARNING));
		}
	}
#pragma endregion InformationFlow
}

void Graph::ProcessConnection(DML::ValueFlow2SignalPort &oVf2Sp)
{
	char buffer[33];
	int radix = 10;

#pragma region ValueFlow2SignalPort
	if (DML::ValueFlow2SignalPort::meta == oVf2Sp.type())
	{	
		string src, dst, path;
		vector<PortMapping>::iterator itsrc, itdst;
		bool hasSrc = false;
		bool hasDst = false;

		string dstName = DML::IOSignal::Cast(DML::ValueFlow2SignalPort::Cast(oVf2Sp).dstValueFlow2SignalPort_end()).name();
		string srcName = DML::ValueFlowTarget::Cast(DML::ValueFlow2SignalPort::Cast(oVf2Sp).srcValueFlow2SignalPort_end()).name();
		
		// Get the two ends of the connection
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if (DML::ValueFlowTarget::Cast(DML::ValueFlow2SignalPort::Cast(oVf2Sp).srcValueFlow2SignalPort_end()).uniqueId()
				== it->GetUniqueID())
			{
				itsrc = it;
				hasSrc = true;
			}
			if (DML::IOSignal::Cast(DML::ValueFlow2SignalPort::Cast(oVf2Sp).dstValueFlow2SignalPort_end()).uniqueId()
				== it->GetUniqueID())
			{
				itdst = it;
				hasDst = true;
			}
		}

		if (hasSrc && hasDst)
		{
			if (itsrc->GetPath() == itdst->GetPath())
			{
				string tmp1 = itsrc->GetUniqueName();
				tmp1 += "/1";
				string tmp2 = itdst->GetUniqueName();
				tmp2 += "/1";

				conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
			}
			else if (itsrc->GetPath() == itdst->GetParentPath())
			{
				string tmp1 = itsrc->GetUniqueName();
				tmp1 += "/1";
				string tmp2 = itdst->GetParentName();
				tmp2 += "/";
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);

				conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
			}
			else if (itsrc->GetParentPath() == itdst->GetPath())
			{
				string tmp1 = itsrc->GetParentName();
				tmp1 += "/";
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
				string tmp2 = itdst->GetUniqueName();
				tmp2 += "/1";

				conns.push_back(BlockConnection(itdst->GetPath(), tmp1, tmp2));
			}
			else if (itsrc->GetParentPath() == itdst->GetParentPath())
			{
				string tmp1 = itsrc->GetParentName();
				tmp1 += "/";
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
				string tmp2 = itdst->GetParentName();
				tmp2 += "/";
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);

				conns.push_back(BlockConnection(itdst->GetParentPath(), tmp1, tmp2));
			}
		}
		else
		{
			string err = "";
			err += "Connection skipped: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(oVf2Sp.name(), oVf2Sp);
			pqMessages.push(Message(err, MSG_WARNING));
		}
	}
#pragma endregion ValueFlow2SignalPort
}

void Graph::ProcessConnection(DML::SignalConnectionType &oSType)
{
	char buffer[33];
	int radix = 10;

	bool bSkipped = false;

#pragma region SignalPort2PhysicalPort
	if (DML::SignalPort2PhysicalPort::meta == oSType.type())
	{	
		string src, dst, path;
		vector<PortMapping>::iterator itsrc, itdst;
		bool hasSrc = false;
		bool hasDst = false;

		string dstName = DML::SignalPortType::Cast(DML::SignalPort2PhysicalPort::Cast(oSType).dstSignalPort2PhysicalPort_end()).name();
		string srcName = DML::IOSignal::Cast(DML::SignalPort2PhysicalPort::Cast(oSType).srcSignalPort2PhysicalPort_end()).name();
		
		// Get the two ends of the connection
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if (DML::IOSignal::Cast(DML::SignalPort2PhysicalPort::Cast(oSType).srcSignalPort2PhysicalPort_end()).uniqueId()
				== it->GetUniqueID())
			{
				itsrc = it;
				hasSrc = true;
			}
			if (DML::SignalPortType::Cast(DML::SignalPort2PhysicalPort::Cast(oSType).dstSignalPort2PhysicalPort_end()).uniqueId()
				== it->GetUniqueID())
			{
				itdst = it;
				hasDst = true;
			}
		}

		if (hasSrc && hasDst)
		{
			if (itsrc->GetPath() == itdst->GetPath())
			{
				string tmp1 = itsrc->GetUniqueName();
				tmp1 += "/1";
				string tmp2 = itdst->GetUniqueName();
				tmp2 += "/1";
				if (CheckType(DML::ElectricalSignalPort::Cast(DML::SignalPort2PhysicalPort::Cast(oSType).dstSignalPort2PhysicalPort_end())) ==
					IO_OUTPUT)
				{
					conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
				}
				else
				{
					bSkipped = true;
				}
			}
			else if (itsrc->GetPath() == itdst->GetParentPath())
			{
				string tmp1 = itsrc->GetUniqueName();
				tmp1 += "/1";
				string tmp2 = itdst->GetParentName();
				tmp2 += "/";
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);
				if (CheckType(DML::ElectricalSignalPort::Cast(DML::SignalPort2PhysicalPort::Cast(oSType).dstSignalPort2PhysicalPort_end())) ==
					IO_INPUT)
				{
					conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
				}
				else
				{
					bSkipped = true;
				}
			}
			else if (itsrc->GetParentPath() == itdst->GetPath())
			{
				string tmp1 = itsrc->GetParentName();
				tmp1 += "/";
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
				string tmp2 = itdst->GetUniqueName();
				tmp2 += "/1";
				if (CheckType(DML::ElectricalSignalPort::Cast(DML::SignalPort2PhysicalPort::Cast(oSType).dstSignalPort2PhysicalPort_end())) ==
					IO_OUTPUT)
				{
					conns.push_back(BlockConnection(itdst->GetPath(), tmp1, tmp2));
				}
				else
				{
					bSkipped = true;
				}
			}
			else if (itsrc->GetParentPath() == itdst->GetParentPath())
			{
				string tmp1 = itsrc->GetParentName();
				tmp1 += "/";
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
				string tmp2 = itdst->GetParentName();
				tmp2 += "/";
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);
				if (CheckType(DML::ElectricalSignalPort::Cast(DML::SignalPort2PhysicalPort::Cast(oSType).dstSignalPort2PhysicalPort_end())) ==
					IO_INPUT)
				{
					conns.push_back(BlockConnection(itdst->GetParentPath(), tmp1, tmp2));
				}
				else
				{
					bSkipped = true;
				}
			}
		}
		else
		{
			bSkipped = true;
		}
	}
#pragma endregion SignalPort2PhysicalPort

#pragma region PhysicalPort2SignalPort
	if (DML::PhysicalPort2SignalPort::meta == oSType.type())
	{	
		string src, dst, path;
		vector<PortMapping>::iterator itsrc, itdst;
		bool hasSrc = false;
		bool hasDst = false;

		string dstName = DML::IOSignal::Cast(DML::PhysicalPort2SignalPort::Cast(oSType).dstPhysicalPort2SignalPort_end()).name();
		string srcName = DML::SignalPortType::Cast(DML::PhysicalPort2SignalPort::Cast(oSType).srcPhysicalPort2SignalPort_end()).name();
		
		// Get the two ends of the connection
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if (DML::SignalPortType::Cast(DML::PhysicalPort2SignalPort::Cast(oSType).srcPhysicalPort2SignalPort_end()).uniqueId()
				== it->GetUniqueID())
			{
				itsrc = it;
				hasSrc = true;
			}
			if (DML::IOSignal::Cast(DML::PhysicalPort2SignalPort::Cast(oSType).dstPhysicalPort2SignalPort_end()).uniqueId()
				== it->GetUniqueID())
			{
				itdst = it;
				hasDst = true;
			}
		}

		if (hasSrc && hasDst)
		{
			if (itsrc->GetPath() == itdst->GetPath())
			{
				string tmp1 = itsrc->GetUniqueName();
				tmp1 += "/1";
				string tmp2 = itdst->GetUniqueName();
				tmp2 += "/1";
				if (CheckType(DML::ElectricalSignalPort::Cast(DML::PhysicalPort2SignalPort::Cast(oSType).srcPhysicalPort2SignalPort_end())) ==
					IO_INPUT)
				{
					conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
				}
				else
				{
					bSkipped = true;
				}
			}
			else if (itsrc->GetPath() == itdst->GetParentPath())
			{
				string tmp1 = itsrc->GetUniqueName();
				tmp1 += "/1";
				string tmp2 = itdst->GetParentName();
				tmp2 += "/";
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);
				if (CheckType(DML::ElectricalSignalPort::Cast(DML::PhysicalPort2SignalPort::Cast(oSType).srcPhysicalPort2SignalPort_end())) ==
					IO_INPUT)
				{
					conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
				}
				else
				{
					bSkipped = true;
				}
			}
			else if (itsrc->GetParentPath() == itdst->GetPath())
			{
				string tmp1 = itsrc->GetParentName();
				tmp1 += "/";
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
				string tmp2 = itdst->GetUniqueName();
				tmp2 += "/1";
				if (CheckType(DML::ElectricalSignalPort::Cast(DML::PhysicalPort2SignalPort::Cast(oSType).srcPhysicalPort2SignalPort_end())) ==
					IO_OUTPUT)
				{
					conns.push_back(BlockConnection(itdst->GetPath(), tmp1, tmp2));
				}
				else
				{
					bSkipped = true;
				}
			}
			else if (itsrc->GetParentPath() == itdst->GetParentPath())
			{
				string tmp1 = itsrc->GetParentName();
				tmp1 += "/";
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
				string tmp2 = itdst->GetParentName();
				tmp2 += "/";
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);
				if (CheckType(DML::ElectricalSignalPort::Cast(DML::PhysicalPort2SignalPort::Cast(oSType).srcPhysicalPort2SignalPort_end())) ==
					IO_OUTPUT)
				{
					conns.push_back(BlockConnection(itdst->GetParentPath(), tmp1, tmp2));
				}
				else
				{
					bSkipped = true;
				}
			}
		}
		else
		{
			bSkipped = true;
		}
	}
#pragma endregion PhysicalPort2SignalPort

	if (bSkipped)
	{
		string err = "";
		err += "Connection skipped: ";
		err += GMEConsole::Formatter::MakeObjectHyperlink(oSType.name(), oSType);
		pqMessages.push(Message(err, MSG_WARNING));
	}
}

void Graph::ProcessConnection(DML::ConnectionBase &oConnectionBase)
{
	string szType;
	szType = oConnectionBase.type().name();

	char buffer[33];
	int radix = 10;
	
#pragma region Signal2Junction
	if ("Signal2Junction" == szType)
	{
		vector<PortMapping>::iterator SignalPort;
		DML::Junction junction = DML::Junction::Cast((DML::Signal2Junction::Cast(oConnectionBase)).dstSignal2Junction_end());
		string src, dst, path;
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if (DML::Signal::Cast(DML::Signal2Junction::Cast(oConnectionBase).srcSignal2Junction_end()).uniqueId()
				== it->GetUniqueID())
			{
				SignalPort = it;
			}
		}
		//string junctionPath = GetSimulinkPath(GetPathWOName(junction.getPath2("/")));
		string junctionPath = GetSimulinkPath(GetObjectPath(junction.GetParent()));

		if(SignalPort->GetPath() == junctionPath)
		{
			src = SignalPort->GetName();
			src += "/1";
		}
		else // Signal is a level lower
		{
			src = SignalPort->GetParentName();
			src += "/";
			_itoa_s(SignalPort->GetPortNumber(), buffer, 10);
			src += string(buffer);
		}

		dst = junction.name();
		dst += "/";

		int numbonds;
		for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
		{
			if(jit->p.uniqueId() == junction.uniqueId())
			{
				numbonds = jit->connectioncount;
				_ltoa_s(numbonds+1,buffer,radix);
				dst += string(buffer);
				break;
			}
		}
		
		//path = GetPathWOName(junction.getPath2("/"));
		path = GetObjectPath(junction.GetParent());
		conns.push_back(BlockConnection(path, src, dst));
	}
#pragma endregion Signal2Junction
#pragma region Junction2Signal
	else if("Junction2Signal" == szType)
	{
		string errmsg = "Please do not use Junction2Signal connections. They are not supported yet.";
		errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
		pqMessages.push(Message(errmsg, MSG_ERROR));
	}
#pragma endregion Junction2Signal
#pragma region SignalConnection
	else if ("SignalConnection" == szType)
	{	
		string src, dst, path;
		vector<PortMapping>::iterator itsrc, itdst;

		string dstName = DML::Signal::Cast(DML::SignalConnection::Cast(oConnectionBase).dstSignalConnection_end()).name();
		string srcName = DML::Signal::Cast(DML::SignalConnection::Cast(oConnectionBase).srcSignalConnection_end()).name();
		bool ValidSrc = false;
		bool ValidDst = false;
		// Get the two ends of the connection
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if (DML::Signal::Cast(DML::SignalConnection::Cast(oConnectionBase).srcSignalConnection_end()).uniqueId()
				== it->GetUniqueID())
			{
				itsrc = it;
				ValidSrc = true;
			}
			if (DML::Signal::Cast(DML::SignalConnection::Cast(oConnectionBase).dstSignalConnection_end()).uniqueId()
				== it->GetUniqueID())
			{
				itdst = it;
				ValidDst = true;
			}
		}

		if ((ValidSrc && ValidDst) == false)
		{
			string msg = "";
			msg += "Connection was skipped: ";
			msg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(), oConnectionBase);
			pqMessages.push(Message(msg,MSG_ERROR));
			return;
		}

		if (itsrc->GetPath() == itdst->GetPath())
		{
			string tmp1 = itsrc->GetUniqueName();
			tmp1 += "/1";
			string tmp2 = itdst->GetUniqueName();
			tmp2 += "/1";

			conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
		}
		else if (itsrc->GetPath() == itdst->GetParentPath())
		{
			string tmp1 = itsrc->GetUniqueName();
			tmp1 += "/1";
			string tmp2 = itdst->GetParentName();
			tmp2 += "/";
			// TRICK: override the port number if the parent is a simulink model
			if (DML::Signal::Cast(DML::SignalConnection::Cast(oConnectionBase).dstSignalConnection_end()).GetParent().type() ==
				DML::SimulinkModel::meta)
			{
				string dstLabel1 = GetDstLabel1(oConnectionBase);
				string dstLabel2 = GetDstLabel2(oConnectionBase);
				if (dstLabel1 != "<blank>")
				{
					tmp2 += dstLabel1;
				}
				else if (dstLabel2 != "<blank>")
				{
					string PortNum = DML::SignalConnection::Cast(oConnectionBase).InputPortNumber();
					if (!PortNum.empty())
					{
						tmp2 += PortNum;
					}
					else
					{
						if (dstLabel2 == "%InputPortNumber%")
						{
							std::string errmsg = "";
							errmsg = "Destination port mapping (i.e. port number) must be defined for connection: ";
							errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
							pqMessages.push(Message(errmsg, MSG_ERROR));
						}
						tmp2 += dstLabel2;
					}
				}
				else
				{
					std::string errmsg = "";
					errmsg = "Destination port mapping (i.e. port number) must be defined for connection: ";
					errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
					pqMessages.push(Message(errmsg, MSG_ERROR));
				}
			}
			else
			{
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);
			}
			conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
		}
		else if (itsrc->GetParentPath() == itdst->GetPath())
		{
			string tmp1 = itsrc->GetParentName();
			tmp1 += "/";
			// TRICK: override the port number if the parent is a simulink model
			if (DML::Signal::Cast(DML::SignalConnection::Cast(oConnectionBase).srcSignalConnection_end()).GetParent().type() ==
				DML::SimulinkModel::meta)
			{
				string srcLabel1 = GetSrcLabel1(oConnectionBase);
				string srcLabel2 = GetSrcLabel2(oConnectionBase);
				if (srcLabel1 != "<blank>")
				{
					tmp1 += srcLabel1;
				}
				else if (srcLabel2 != "<blank>")
				{
					DML::SignalConnection sig = DML::SignalConnection::Cast(oConnectionBase);
					string PortNum = sig.OutputPortNumber();

					if (!PortNum.empty())
					{
						tmp1 += PortNum;
					}
					else
					{
						if (srcLabel2 == "%OutputPortNumber%")
						{
							std::string errmsg = "";
							errmsg = "Source port mapping (i.e. port number) must be defined for connection: ";
							errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
							pqMessages.push(Message(errmsg, MSG_ERROR));
						}
						tmp1 += srcLabel2;
					}
				}
				else
				{
					std::string errmsg = "";
					errmsg = "Source port mapping (i.e. port number) must be defined for connection: ";
					errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
					pqMessages.push(Message(errmsg, MSG_ERROR));
				}
			}
			else
			{
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
			}
			string tmp2 = itdst->GetUniqueName();
			tmp2 += "/1";

			conns.push_back(BlockConnection(itdst->GetPath(), tmp1, tmp2));
		}
		else if (itsrc->GetParentPath() == itdst->GetParentPath())
		{
			string tmp1 = itsrc->GetParentName();
			tmp1 += "/";
			// TRICK: override the port number if the parent is a simulink model
			if (DML::Signal::Cast(DML::SignalConnection::Cast(oConnectionBase).srcSignalConnection_end()).GetParent().type() ==
				DML::SimulinkModel::meta)
			{
				string srcLabel1 = GetSrcLabel1(oConnectionBase);
				string srcLabel2 = GetSrcLabel2(oConnectionBase);
				if (srcLabel1 != "<blank>")
				{
					tmp1 += srcLabel1;
				}
				else if (srcLabel2 != "<blank>")
				{
					string PortNum = DML::SignalConnection::Cast(oConnectionBase).OutputPortNumber();
					if (!PortNum.empty())
					{
						tmp1 += PortNum;
					}
					else
					{
						if (srcLabel2 == "%OutputPortNumber%")
						{
							std::string errmsg = "";
							errmsg = "Source port mapping (i.e. port number) must be defined for connection: ";
							errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
							pqMessages.push(Message(errmsg, MSG_ERROR));
						}
						tmp1 += srcLabel2;
					}
				}
				else
				{
					std::string errmsg = "";
					errmsg = "Source port mapping (i.e. port number) must be defined for connection: ";
					errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
					pqMessages.push(Message(errmsg, MSG_ERROR));
				}
			}
			else
			{
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
			}

			string tmp2 = itdst->GetParentName();
			tmp2 += "/";
			// TRICK: override the port number if the parent is a simulink model
			if (DML::Signal::Cast(DML::SignalConnection::Cast(oConnectionBase).dstSignalConnection_end()).GetParent().type() ==
				DML::SimulinkModel::meta)
			{
				string dstLabel1 = GetDstLabel1(oConnectionBase);
				string dstLabel2 = GetDstLabel2(oConnectionBase);
				if (dstLabel1 != "<blank>")
				{
					tmp2 += dstLabel1;
				}
				else if (dstLabel2 != "<blank>")
				{
					string PortNum = DML::SignalConnection::Cast(oConnectionBase).InputPortNumber();
					if (!PortNum.empty())
					{
						tmp2 += PortNum;
					}
					else
					{
						if (dstLabel2 == "%InputPortNumber%")
						{
							std::string errmsg = "";
							errmsg = "Destination port mapping (i.e. port number) must be defined for connection: ";
							errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
							pqMessages.push(Message(errmsg, MSG_ERROR));
						}
						tmp2 += dstLabel2;
					}
				}
				else
				{
					std::string errmsg = "";
					errmsg = "Destination port mapping (i.e. port number) must be defined for connection: ";
					errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
					pqMessages.push(Message(errmsg, MSG_ERROR));
				}
			}
			else
			{
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);
			}

			conns.push_back(BlockConnection(itdst->GetParentPath(), tmp1, tmp2));
		}
	}
#pragma endregion SignalConnection
#pragma region ControlFunction2Signal
	else if ("ControlFunction2Signal" == szType)
	{		
		bool _print=false;
		vector<PortMapping>::iterator itsrc,itdst;			
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if (DML::Signal::Cast(DML::ControlFunction2Signal::Cast(oConnectionBase).dstControlFunction2Signal_end()).uniqueId()
				== it->GetUniqueID())
			{
				itdst = it;
			}
		}
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if ((DML::ControlFunction::Cast(DML::ControlFunction2Signal::Cast(oConnectionBase).srcControlFunction2Signal_end()).uniqueId()
				== it->GetUniqueID()) && (it->GetName() == itdst->GetName()))
			{
				itsrc = it;	
				_print=true;
			}
		}
		if (_print)
		{
			if (itsrc->GetParentPath() == itdst->GetPath())
			{
				string tmp1 = itsrc->GetParentName();
				tmp1 += "/";
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);	
				tmp1 += string(buffer);
				string tmp2 = itdst->GetUniqueName();
				tmp2 += "/1";

				conns.push_back(BlockConnection(itsrc->GetParentPath(), tmp1, tmp2));
			}			
			else if (itsrc->GetParentPath() == itdst->GetParentPath())
			{
				string tmp1 = itsrc->GetParentName();
				tmp1 += "/";
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
				string tmp2 = itdst->GetParentName();
				tmp2 += "/";
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);

				conns.push_back(BlockConnection(itsrc->GetParentPath(), tmp1, tmp2));
			}
		}
		else
		{
		#ifdef _DEBUG
			string strWarning;
			strWarning = GMEConsole::Formatter::MakeObjectHyperlink(DML::ControlFunction::Cast(DML::ControlFunction2Signal::Cast(oConnectionBase).srcControlFunction2Signal_end()).name(),DML::ControlFunction::Cast(DML::ControlFunction2Signal::Cast(oConnectionBase).srcControlFunction2Signal_end()));
			strWarning += " has not got parameter ";
			strWarning += GMEConsole::Formatter::MakeObjectHyperlink(DML::Signal::Cast(DML::ControlFunction2Signal::Cast(oConnectionBase).dstControlFunction2Signal_end()).name(),DML::Signal::Cast(DML::ControlFunction2Signal::Cast(oConnectionBase).dstControlFunction2Signal_end()));
			pqMessages.push(Message(strWarning, MSG_WARNING));
		#endif
		}
	}
#pragma endregion ControlFunction2Signal
#pragma region Signal2ControlFunction
	else if ("Signal2ControlFunction" == szType)
	{
		bool _print = false;
		vector<PortMapping>::iterator itsrc,itdst;			
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if (DML::Signal::Cast(DML::Signal2ControlFunction::Cast(oConnectionBase).srcSignal2ControlFunction_end()).uniqueId() == it->GetUniqueID())
			{
				itsrc = it;
			}
		}
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if ((DML::ControlFunction::Cast(DML::Signal2ControlFunction::Cast(oConnectionBase).dstSignal2ControlFunction_end()).uniqueId() == it->GetUniqueID()) && (it->GetName() == itsrc->GetName()))
			{
				itdst = it;	
				_print=true;
			}
		}
		if (_print)
		{
			if (itsrc->GetPath() == itdst->GetParentPath())
			{
				string tmp1 = itsrc->GetUniqueName();
				tmp1 += "/1";				
				string tmp2 = itdst->GetParentName();
				tmp2 += "/";
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);

				conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
			}			
			else if (itsrc->GetParentPath() == itdst->GetParentPath())
			{
				string tmp1 = itsrc->GetParentName();
				tmp1 += "/";
				_itoa_s(itsrc->GetPortNumber(),buffer,radix);
				tmp1 += string(buffer);
				string tmp2 = itdst->GetParentName();
				tmp2 += "/";
				_itoa_s(itdst->GetPortNumber(),buffer,radix);
				tmp2 += string(buffer);

				conns.push_back(BlockConnection(itsrc->GetParentPath(), tmp1, tmp2));
			}
		}
		
	}
#pragma endregion Signal2ControlFunction
#pragma region Sensing2Signal
	else if ("Sensing2Signal" == szType)
	{		
		string src, dst, path;
		vector<PortMapping>::iterator itsrc,itdst;
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if (DML::Sensing::Cast(DML::Sensing2Signal::Cast(oConnectionBase).srcSensing2Signal_end()).uniqueId()
				== it->GetUniqueID())
			{
				itsrc = it;
			}
			if (DML::Signal::Cast(DML::Sensing2Signal::Cast(oConnectionBase).dstSensing2Signal_end()).uniqueId()
				== it->GetUniqueID())
			{
				itdst = it;
			}
		}
		if (itsrc->GetPath() == itdst->GetPath())
		{
			string tmp1 = itsrc->GetUniqueName();
			tmp1 += "/1";
			string tmp2 = itdst->GetUniqueName();
			tmp2 += "/1";

			conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
		}
		else if (itsrc->GetPath() == itdst->GetParentPath())
		{
			string tmp1 = itsrc->GetUniqueName();
			tmp1 += "/1";
			string tmp2 = itdst->GetParentName();
			tmp2 += "/";
			_itoa_s(itdst->GetPortNumber(),buffer,radix);
			tmp2 += string(buffer);

			conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
		}
	}
#pragma endregion Sensing2Signal
#pragma region Signal2Actuation
	else if ("Signal2Actuation" == szType)
	{		
		string src,dst,path;
		vector<PortMapping>::iterator itsrc,itdst;
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if (DML::Signal::Cast(DML::Signal2Actuation::Cast(oConnectionBase).srcSignal2Actuation_end()).uniqueId()
				== it->GetUniqueID())
			{
				itsrc = it;
			}
			if (DML::Actuation::Cast(DML::Signal2Actuation::Cast(oConnectionBase).dstSignal2Actuation_end()).uniqueId() == it->GetUniqueID())
			{
				itdst = it;
			}
		}
		if (itsrc->GetPath() == itdst->GetPath())
		{
			string tmp1 = itsrc->GetUniqueName();
			tmp1 += "/1";
			string tmp2 = itdst->GetUniqueName();
			tmp2 += "/1";

			conns.push_back(BlockConnection(itsrc->GetPath(), tmp1, tmp2));
		}
		else if (itsrc->GetParentPath() == itdst->GetPath())
		{
			string tmp1 = itsrc->GetParentName();
			tmp1 += "/";
			_itoa_s(itsrc->GetPortNumber(),buffer,radix);
			tmp1 += string(buffer);
			string tmp2 = itdst->GetUniqueName();
			tmp2 += "/1";

			conns.push_back(BlockConnection(itdst->GetPath(), tmp1, tmp2));
		}
	}
#pragma endregion Signal2Actuation
#pragma region PowerPort2Junction
	else if ("PowerPort2Junction" == szType)
	{
		/*__int64 dim = DML::PowerPort2Junction::Cast(oConnectionBase).dimension();
		if (dim != 1)
		{
			std::string errmsg = "";
			errmsg = "Dimension should be equal to one (otherwise it is not supported yet): ";
			errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
			pqMessages.push(Message(errmsg, MSG_ERROR));
		}
		*/
	}
#pragma endregion PowerPort2Junction
#pragma region Junction2PowerPort
	else if ("Junction2PowerPort" == szType)
	{
		/*
		__int64 dim = DML::Junction2PowerPort::Cast(oConnectionBase).dimension();
		if (dim != 1)
		{
			std::string errmsg = "";
			errmsg = "Dimension should be equal to one (otherwise it is not supported yet): ";
			errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
			pqMessages.push(Message(errmsg, MSG_ERROR));
		}
		*/
	}
#pragma endregion Junction2PowerPort
#pragma region BondJ2J
	else if ("BondJ2J" == szType)
	{
		/*
		__int64 dim = DML::Bond::Cast(oConnectionBase).dimension();
		if (dim != 1)
		{
			std::string errmsg = "";
			errmsg = "Dimension should be equal to one (otherwise it is not supported yet): ";
			errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
			pqMessages.push(Message(errmsg, MSG_ERROR));
		}
		*/
	}
#pragma endregion BondJ2J
#pragma region BondE2J
	else if ("BondE2J" == szType) 
	{
		/*
		__int64 dim = DML::Bond::Cast(oConnectionBase).dimension();
		if (dim != 1)
		{
			std::string errmsg = "";
			errmsg = "Dimension should be equal to one (otherwise it is not supported yet): ";
			errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
			pqMessages.push(Message(errmsg, MSG_ERROR));
		}
		*/
	}
#pragma endregion BondE2J
#pragma region BondJ2E
	else if ("BondJ2E" == szType)
	{
		/*
		__int64 dim = DML::Bond::Cast(oConnectionBase).dimension();
		if (dim != 1)
		{
			std::string errmsg = "";
			errmsg = "Dimension should be equal to one (otherwise it is not supported yet): ";
			errmsg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(),oConnectionBase);
			pqMessages.push(Message(errmsg, MSG_ERROR));
		}
		*/
	}
#pragma endregion BondJ2E
#pragma region Bond
	else if("Bond" == szType)
	{
		// Processed in the first traversal
	}
#pragma endregion Bond
#pragma region Junction2Switching
	else if("Junction2Switching" == szType)
	{
		// Processed in "ProcessControlNode" function
	}
#pragma endregion Junction2Switching
#pragma region Switching2Junction
	else if("Switching2Junction" == szType)
	{
		// Processed in "ProcessControlNode" function
	}
#pragma endregion Switching2Junction
#pragma region PowerPropagations
	else if("PowerPropagations" == szType)
	{
		// Processed in powerport-traversal
	}
#pragma endregion PowerPropagations
#pragma region Junction2Modulation
	else if("Junction2Modulation" == szType)
	{
		// Processed in "ProcessControlNode" function
	}
#pragma endregion Junction2Modulation
#pragma region Modulation2BGMElement
	else if("Modulation2BGMElement" == szType)
	{
		// Processed in "ProcessControlNode" function
	}
#pragma endregion Modulation2BGMElement
#pragma region Constant2Modulation
	else if("Constant2Modulation" == szType)
	{
		// Processed in "ProcessControlNode" function
	}
#pragma endregion Constant2Modulation
#pragma region Constant2Switching
	else if("Constant2Switching" == szType)
	{
		// Processed in "ProcessControlNode" function
	}
#pragma endregion Constant2Switching
#pragma region Actuation2ModulatedTwoPort
	else if("Actuation2ModulatedTwoPort" == szType)
	{
		string src, dst;
		src = string(DML::Actuation::Cast(DML::Actuation2ModulatedTwoPort::Cast(oConnectionBase).srcActuation2ModulatedTwoPort_end()).name()) + "/1";
		dst = string(DML::ModulatedTwoPort(DML::Actuation2ModulatedTwoPort::Cast(oConnectionBase).dstActuation2ModulatedTwoPort_end()).name()) + "/3";
		//conns.push_back(BlockConnection(oConnectionBase.GetParent().getPath2("/"), src, dst));
		conns.push_back(BlockConnection(GetObjectPath(oConnectionBase.GetParent()), src, dst));
	}
#pragma endregion Actuation2ModulatedTwoPort
#pragma region Signal2BGMElement
	else if("Signal2BGMElement" == szType)
	{
		DML::Signal2BGMElement s2bgme;
		DML::Signal sSrc;
		DML::BGModulatedElement bgmeDst;
		string szDstType;

		string src, dst;
		char buffer[33];
		int radix = 10;

		s2bgme = DML::Signal2BGMElement::Cast(oConnectionBase);
		sSrc = s2bgme.srcSignal2BGMElement_end();
		bgmeDst = s2bgme.dstSignal2BGMElement_end();
		szDstType = bgmeDst.type().name();

		for (vector<PortMapping>::iterator it = pmSignal.begin();
				 it != pmSignal.end();
				 ++it)
		{
			if (it->GetUniqueID() == sSrc.uniqueId())
			{
				if (sSrc.GetParent() == oConnectionBase.GetParent())
				{
					src = it->GetUniqueName();
					src += "/1";
				}
				else 
				{
					src = it->GetParentName();
					src += "/";
					_ltoa_s(it->GetPortNumber(),buffer,radix);
					src += string(buffer); 
				}
			}
		}

		if ((szDstType == "MSe") ||
				(szDstType == "MSf") ||
				(szDstType == "MR") ||
				(szDstType == "MI") ||
				(szDstType == "MC"))
		{
			dst = bgmeDst.name();
			
			if (!Udm::IsDerivedFrom(bgmeDst.type(), DML::Storage::meta))
			{
				dst += "/2";
			}
			else
			{
				DML::Storage s = DML::Storage::Cast(bgmeDst);
				set<DML::Signal2InitialValue> s2iv = s.srcSignal2InitialValue();
				if (s2iv.empty())
				{
					// initial value is a constant
					dst += "/2";
				}
				else
				{
					// initial value is a signal, its port number 2
					// the modulation signal's port number must be 3
					dst += "/3";
				}
			}
			//if (!bMDAO)
			{
				conns.push_back(BlockConnection(GetParentPath(oConnectionBase), src, dst));
			}
			/*else
			{
				string DS_ReadSrc = "DS_";
				DS_ReadSrc += GetModifiedName(bgmeDst);
				DS_ReadSrc += "/1";
				conns.push_back(BlockConnection(GetParentPath(oConnectionBase), DS_ReadSrc, dst));
			}*/
		}
		else if ((szDstType == "MTF") ||
						 (szDstType == "MGY"))
		{
			dst = bgmeDst.name();
			dst += "/3";
			conns.push_back(BlockConnection(GetParentPath(oConnectionBase), src, dst));
		}
		else
		{
			// cannot identify the destination element's type
			ASSERT(0);
		}


	}
#pragma endregion Signal2BGMElement
#pragma region Signal2InitialValue
	else if("Signal2InitialValue" == szType)
	{
		DML::Signal2InitialValue s2bgme;
		DML::Signal sSrc;
		DML::Storage bgmeDst;
		string szDstType;

		string src, dst;
		char buffer[33];
		int radix = 10;

		s2bgme = DML::Signal2InitialValue::Cast(oConnectionBase);
		sSrc = s2bgme.srcSignal2InitialValue_end();
		bgmeDst = s2bgme.dstSignal2InitialValue_end();
		szDstType = bgmeDst.type().name();

		for (vector<PortMapping>::iterator it = pmSignal.begin();
				 it != pmSignal.end();
				 ++it)
		{
			if (it->GetUniqueID() == sSrc.uniqueId())
			{
				if (sSrc.GetParent() == oConnectionBase.GetParent())
				{
					src = it->GetUniqueName();
					src += "/1";
				}
				else
				{
					src = it->GetParentName();
					src += "/";
					_ltoa_s(it->GetPortNumber(),buffer,radix);
					src += string(buffer); 
				}
			}
		}

		if ((szDstType == "I") ||
			(szDstType == "C") ||
			(szDstType == "MI") ||
			(szDstType == "MC"))
		{
			dst = bgmeDst.name();
			dst += "/2";
			conns.push_back(BlockConnection(GetParentPath(oConnectionBase), src, dst));
		}
		else
		{
			// cannot identify the destination element's type
			ASSERT(0);
		}
	}
#pragma endregion Signal2InitialValue
#pragma region ZeroJunction2De
	else if("ZeroJunction2De" == szType)
	{
		char buffer[33];
		string src;
		string dst;
		string thisname;
		string path;
		DML::De deObj;

		deObj = DML::ZeroJunction2De::Cast(oConnectionBase).dstZeroJunction2De_end();
		thisname = deObj.name();
		path = GetParentPath(oConnectionBase);

		DML::ZeroJunction zj;
		zj = DML::ZeroJunction2De::Cast(oConnectionBase).srcZeroJunction2De_end();
		src = zj.name();
		src += "/";

		// Get bond to connect sensing on
		int numbonds;
		for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
		{
			if(jit->p.uniqueId() == zj.uniqueId())
			{
				numbonds = jit->connectioncount;
				_ltoa_s(numbonds+1, buffer, 10);
				src += string(buffer);
				break;
			}
		}

		dst = thisname;
		dst += "/1";	
		conns.push_back(BlockConnection(path, src, dst));
	}
#pragma endregion ZeroJunction2De
#pragma region ZeroJunction2Dp
	else if("ZeroJunction2Dp" == szType)
	{
		char buffer[33];
		string src;
		string dst;
		string thisname;
		string path;
		DML::Dp dpObj;

		dpObj = DML::ZeroJunction2Dp::Cast(oConnectionBase).dstZeroJunction2Dp_end();
		thisname = dpObj.name();
		path = GetParentPath(oConnectionBase);

		DML::ZeroJunction zj;
		zj = DML::ZeroJunction2Dp::Cast(oConnectionBase).srcZeroJunction2Dp_end();
		src = zj.name();
		src += "/";

		// Get bond to connect sensing on
		int numbonds;
		for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
		{
			if(jit->p.uniqueId() == zj.uniqueId())
			{
				numbonds = jit->connectioncount;
				_ltoa_s(numbonds+1, buffer, 10);
				src += string(buffer);
				break;
			}
		}

		dst = thisname;
		dst += "/1";	
		conns.push_back(BlockConnection(path, src, dst));
	}
#pragma endregion ZeroJunction2Dp
#pragma region OneJunction2Df
	else if("OneJunction2Df" == szType)
	{
		char buffer[33];
		string src;
		string dst;
		string thisname;
		string path;
		DML::Df dfObj;

		dfObj = DML::OneJunction2Df::Cast(oConnectionBase).dstOneJunction2Df_end();
		thisname = dfObj.name();
		path = GetParentPath(oConnectionBase);

		DML::OneJunction oj;
		oj = DML::OneJunction2Df::Cast(oConnectionBase).srcOneJunction2Df_end();
		src = oj.name();
		src += "/";

		// Get bond to connect sensing on
		int numbonds;
		for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
		{
			if(jit->p.uniqueId() == oj.uniqueId())
			{
				numbonds = jit->connectioncount;
				_ltoa_s(numbonds+1, buffer, 10);
				src += string(buffer);
				break;
			}
		}

		dst = thisname;
		dst += "/1";
		conns.push_back(BlockConnection(path, src, dst));
	}
#pragma endregion OneJunction2Df
#pragma region OneJunction2Dq
	else if("OneJunction2Dq" == szType)
	{
		char buffer[33];
		string src;
		string dst;
		string thisname;
		string path;
		DML::Dq dqObj;

		dqObj = DML::OneJunction2Dq::Cast(oConnectionBase).dstZeroJunction2Dq_end();
		thisname = dqObj.name();
		path = GetParentPath(oConnectionBase);

		DML::OneJunction oj;
		oj = DML::OneJunction2Dq::Cast(oConnectionBase).srcZeroJunction2Dq_end();
		src = oj.name();
		src += "/";

		// Get bond to connect sensing on
		int numbonds;
		for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
		{
			if(jit->p.uniqueId() == oj.uniqueId())
			{
				numbonds = jit->connectioncount;
				_ltoa_s(numbonds+1, buffer, 10);
				src += string(buffer);
				break;
			}
		}

		dst = thisname;
		dst += "/1";
		conns.push_back(BlockConnection(path, src, dst));
	}
#pragma endregion OneJunction2Dq
#pragma region ZeroJunction2MonitorEffort
	else if("ZeroJunction2MonitorEffort" == szType)
	{
		char buffer[33];
		string src;
		string dst;
		string thisname;
		string path;
		DML::MonitorEffort meObj;

		meObj = DML::ZeroJunction2MonitorEffort::Cast(oConnectionBase).dstZeroJunction2MonitorEffort_end();
		thisname = meObj.name();
		path = GetParentPath(oConnectionBase);

		DML::ZeroJunction zj;
		zj = DML::ZeroJunction2MonitorEffort::Cast(oConnectionBase).srcZeroJunction2MonitorEffort_end();
		src = zj.name();
		src += "/";

		// Get bond to connect sensing on
		int numbonds;
		for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
		{
			if(jit->p.uniqueId() == zj.uniqueId())
			{
				numbonds = jit->connectioncount;
				_ltoa_s(numbonds+1, buffer, 10);
				src += string(buffer);
				break;
			}
		}

		dst = thisname;
		dst += "/1";	
		conns.push_back(BlockConnection(path, src, dst));

		if (bMDAO)
		{
			/*string DS_dst = "DS_";
			DS_dst += thisname;
			MDAODataStorage.insert(DS_dst);
			DS_dst += "/1";
			conns.push_back(BlockConnection(path, src, DS_dst));*/
		}
	}
#pragma endregion ZeroJunction2MonitorEffort
#pragma region ZeroJunction2MonitorMomentum
	else if("ZeroJunction2MonitorMomentum" == szType)
	{
		char buffer[33];
		string src;
		string dst;
		string thisname;
		string path;
		DML::MonitorMomentum mmObj;

		mmObj = DML::ZeroJunction2MonitorMomentum::Cast(oConnectionBase).dstZeroJunction2MonitorMomentum_end();
		thisname = mmObj.name();
		path = GetParentPath(oConnectionBase);

		DML::ZeroJunction zj;
		zj = DML::ZeroJunction2MonitorMomentum::Cast(oConnectionBase).srcZeroJunction2MonitorMomentum_end();
		src = zj.name();
		src += "/";

		// Get bond to connect sensing on
		int numbonds;
		for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
		{
			if(jit->p.uniqueId() == zj.uniqueId())
			{
				numbonds = jit->connectioncount;
				_ltoa_s(numbonds+1, buffer, 10);
				src += string(buffer);
				break;
			}
		}

		dst = thisname;
		dst += "/1";	
		conns.push_back(BlockConnection(path, src, dst));
	}
#pragma endregion ZeroJunction2MonitorMomentum
#pragma region OneJunction2MonitorFlow
	else if("OneJunction2MonitorFlow" == szType)
	{
		char buffer[33];
		string src;
		string dst;
		string thisname;
		string path;
		DML::MonitorFlow mfObj;

		mfObj = DML::OneJunction2MonitorFlow::Cast(oConnectionBase).dstOneJunction2MonitorFlow_end();
		thisname = mfObj.name();
		path = GetParentPath(oConnectionBase);

		DML::OneJunction oj;
		oj = DML::OneJunction2MonitorFlow::Cast(oConnectionBase).srcOneJunction2MonitorFlow_end();
		src = oj.name();
		src += "/";

		// Get bond to connect sensing on
		int numbonds;
		for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
		{
			if(jit->p.uniqueId() == oj.uniqueId())
			{
				numbonds = jit->connectioncount;
				_ltoa_s(numbonds+1, buffer, 10);
				src += string(buffer);
				break;
			}
		}

		dst = thisname;
		dst += "/1";
		conns.push_back(BlockConnection(path, src, dst));
		
		if (bMDAO)
		{
			/*string DS_dst = "DS_";
			DS_dst += thisname;
			MDAODataStorage.insert(DS_dst);
			DS_dst += "/1";
			conns.push_back(BlockConnection(path, src, DS_dst)); */
		}
	}
#pragma endregion OneJunction2MonitorFlow
#pragma region OneJunction2MonitorDisplacement
	else if("OneJunction2MonitorDisplacement" == szType)
	{
		char buffer[33];
		string src;
		string dst;
		string thisname;
		string path;
		DML::MonitorDisplacement mdObj;

		mdObj = DML::OneJunction2MonitorDisplacement::Cast(oConnectionBase).dstOneJunction2MonitorDisplacement_end();
		thisname = mdObj.name();
		path = GetParentPath(oConnectionBase);

		DML::OneJunction oj;
		oj = DML::OneJunction2MonitorDisplacement::Cast(oConnectionBase).srcOneJunction2MonitorDisplacement_end();
		src = oj.name();
		src += "/";

		// Get bond to connect sensing on
		int numbonds;
		for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit != AllJunctions.end(); ++jit)
		{
			if(jit->p.uniqueId() == oj.uniqueId())
			{
				numbonds = jit->connectioncount;
				_ltoa_s(numbonds+1, buffer, 10);
				src += string(buffer);
				break;
			}
		}

		dst = thisname;
		dst += "/1";
		conns.push_back(BlockConnection(path, src, dst));
	}
#pragma endregion OneJunction2MonitorDisplacement
#pragma region Signal2SignalMonitor
	else if("Signal2SignalMonitor" == szType)
	{
		char buffer[33];
		string src;
		string dst;
		string thisname;
		string path;
		DML::SignalMonitor meObj;

		meObj = DML::Signal2SignalMonitor::Cast(oConnectionBase).dstSignal2SignalMonitor_end();
		thisname = meObj.name();
		path = GetParentPath(oConnectionBase);

		vector<PortMapping>::iterator itsrc, itdst;

		string srcName = DML::Signal::Cast(DML::Signal2SignalMonitor::Cast(oConnectionBase).srcSignal2SignalMonitor_end()).name();
		bool ValidSrc = false;

		// Get the two ends of the connection
		for (vector<PortMapping>::iterator it = pmSignal.begin(); it != pmSignal.end(); ++it)
		{
			if (DML::Signal::Cast(DML::Signal2SignalMonitor::Cast(oConnectionBase).srcSignal2SignalMonitor_end()).uniqueId()
				== it->GetUniqueID())
			{
				itsrc = it;
				ValidSrc = true;
			}
		}

		string tmp2 = thisname;
		tmp2 += "/1";

		if ((ValidSrc) == false)
		{
			string msg = "";
			msg += "Connection was skipped: ";
			msg += GMEConsole::Formatter::MakeObjectHyperlink(oConnectionBase.name(), oConnectionBase);
			pqMessages.push(Message(msg,MSG_ERROR));
			return;
		}

		if (itsrc->GetPath() == path)
		{
			string tmp1 = itsrc->GetUniqueName();
			tmp1 += "/1";

			conns.push_back(BlockConnection(path, tmp1, tmp2));
		}
		else
		{
			string tmp1 = itsrc->GetParentName();
			tmp1 += "/";
			_itoa_s(itsrc->GetPortNumber(),buffer,radix);
			tmp1 += string(buffer);
			conns.push_back(BlockConnection(path, tmp1, tmp2));
		}

	}
#pragma endregion Signal2SignalMonitor
#pragma region Constant2Signal
	else if("Constant2Signal" == szType)
	{
		// We do NOT have this kind of connection
		// Processed in "Process_Constant" function
	}
#pragma endregion Constant2Signal
}


void Graph::GetNextPowerPort(DML::MgaObject &pp, vector<vector<DML::MgaObject>> &Vectors)
{
	set<Udm::Object> NextPowerPortSet;
	set<Udm::Object> tempDSTSet;
	
    if (pp.type() == DML::ElectricalPort::meta)
	{
		NextPowerPortSet = pp.getAssociation(DML::ElectricalPort::meta_dstElectricalConnection);
	}
    else if (pp.type() == DML::MechanicalDPort::meta)
	{
		NextPowerPortSet = pp.getAssociation(DML::MechanicalDPort::meta_dstMechanicalDConnection);
	}
    else if (pp.type() == DML::MechanicalRPort::meta)
	{
		NextPowerPortSet = pp.getAssociation(DML::MechanicalRPort::meta_dstMechanicalRConnection);
	}
	else if (string(pp.type().name()) == "ThermalPort")
	{
		NextPowerPortSet = pp.getAssociation(DML::ThermalPort::meta_dstThermalConnection);
	}
	else if (string(pp.type().name()) == "HydraulicPort")
	{
		NextPowerPortSet = pp.getAssociation(DML::HydraulicPort::meta_dstHydraulicConnection);
	}
	
#ifdef COMPONENTPOWERPROPAGATION
	// COMPONENT POWER PROPAGATION
	if (Udm::IsDerivedFrom(pp.type(),DML::BGPowerPort::meta))
	{
		tempDSTSet = pp.getAssociation(DML::BGPowerPort::meta_dstPowerPort2PhysicalPort);
        set<Udm::Object> tempDSTSet2 = pp.getAssociation(DML::BGPowerPort::meta_srcPhysicalPort2PowerPort);
	}
	else if (Udm::IsDerivedFrom(pp.type(),DML::PowerPortType::meta))
	{
		NextPowerPortSet = pp.getAssociation(DML::PowerPortType::meta_dstPowerFlow);
		tempDSTSet = pp.getAssociation(DML::PowerPortType::meta_dstPhysicalPort2PowerPort);
	}
#endif

	for (set<Udm::Object>::iterator tempPPIt = tempDSTSet.begin(); tempPPIt != tempDSTSet.end(); ++tempPPIt)
	{
		NextPowerPortSet.insert(*tempPPIt);
	}

	if (NextPowerPortSet.size() == 1)
	{
		DML::MgaObject NextPowerPort = DML::MgaObject::Cast(*(NextPowerPortSet.begin()));
		
		if (IsNeedToBeProcessed(Udm::Object::Cast(NextPowerPort)))
		{
			(*Vectors.rbegin()).push_back(NextPowerPort);

			GetNextPowerPort(NextPowerPort, Vectors);
		}
		else
		{
			// do not need to be processed
			return;
		}
	}
	else if(NextPowerPortSet.size() == 0)
	{
		return;
	}
	else
	{
		string szName = "";

		DML::MgaObject NextPowerPort = DML::MgaObject::Cast(*(NextPowerPortSet.begin()));
		(*Vectors.rbegin()).push_back(NextPowerPort);
		szName =  NextPowerPort.name();
		GetNextPowerPort(NextPowerPort, Vectors);

		set<Udm::Object>::iterator NextPPIt = NextPowerPortSet.begin();
		NextPPIt++;
		for( ;
			NextPPIt != NextPowerPortSet.end();
			++NextPPIt)
		{
			if (!IsNeedToBeProcessed(Udm::Object::Cast(*NextPPIt)))
			{
				// do not need to be processed
				continue;
			}
			vector<DML::MgaObject> NewPPVector;
			for(vector<DML::MgaObject>::iterator CopyVectorIt = Vectors.rbegin()->begin();
				CopyVectorIt->uniqueId() != pp.uniqueId(); ++CopyVectorIt)
			{
				NewPPVector.push_back(*CopyVectorIt);
			}
			NewPPVector.push_back(pp);

			DML::MgaObject NextPowerPort1 = DML::MgaObject::Cast(*NextPPIt);
			NewPPVector.push_back(NextPowerPort1);

			Vectors.push_back(NewPPVector);
			GetNextPowerPort(DML::MgaObject::Cast(*NextPPIt), Vectors);
		}

		szName = "";
		for (vector<DML::MgaObject>::iterator itPP = (*Vectors.rbegin()).begin();
			itPP != (*Vectors.rbegin()).end();
			++itPP)
		{
			szName+= itPP->name();
		}

	}
}

void Graph::CheckPowerPorts2Junctions()
{
	// store the type of power port, which is connected to a junction
	string szPowerPortTypes;
	string szMsg;

	set<DML::Junction2PowerPort> j2pp;
	set<DML::PowerPort2Junction> pp2j;
	DML::BGPowerPort ppCurrent;

	szPowerPortTypes = "";

	for (vector<TJunction>::iterator itJunction = AllJunctions.begin();
			 itJunction != AllJunctions.end();
			 ++itJunction)
	{
		szPowerPortTypes = "";
		// go through all junctions
		
		j2pp = itJunction->p.dstJunction2PowerPort();

		for (set<DML::Junction2PowerPort>::iterator itj2pp = j2pp.begin();
				 itj2pp != j2pp.end();
				 ++itj2pp)
		{
			// get the current connected power port
			ppCurrent = itj2pp->dstJunction2PowerPort_end();

			if (szPowerPortTypes == "")
			{
				szPowerPortTypes = ppCurrent.type().name();
			}
			else
			{
				if (szPowerPortTypes != string(ppCurrent.type().name()))
				{
					szMsg = "This junction has power port connections to/from multiple domains: ";
					szMsg += GMEConsole::Formatter::MakeObjectHyperlink(itJunction->p.name(), itJunction->p);
					pqMessages.push(Message::Message(szMsg, MSG_ERROR));
				}
			}
		}

		pp2j = itJunction->p.srcPowerPort2Junction();

		for (set<DML::PowerPort2Junction>::iterator itpp2j = pp2j.begin();
				 itpp2j != pp2j.end();
				 ++itpp2j)
		{
			// get the current connected power port
			ppCurrent = itpp2j->srcPowerPort2Junction_end();

			if (szPowerPortTypes == "")
			{
				szPowerPortTypes = ppCurrent.type().name();
			}
			else
			{
				if (szPowerPortTypes != string(ppCurrent.type().name()))
				{
					szMsg = "This junction has power port connections to/from multiple domains: ";
					szMsg += GMEConsole::Formatter::MakeObjectHyperlink(itJunction->p.name(), itJunction->p);
					pqMessages.push(Message::Message(szMsg, MSG_ERROR));
				}
			}
		}
	} // AllJunctions for cycle
}