/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#ifndef _GRAPH_H_
#define _GRAPH_H_

#include "CyPhyML.h"

#define DML CyPhyML // Domain Modeling Language definition

// It this is defined you have power port propagation support between
// components. The interface must contain the specific ports and connections.
#define COMPONENTPOWERPROPAGATION

typedef map<Udm::Object, bool> MapContainer;

////////////////////////////////////////////////////////////////////////////////
/// <summary> Clears all globals. </summary>
///
/// <remarks> Zsolt, 8/18/2011. </remarks>
////////////////////////////////////////////////////////////////////////////////

void ClearAllGlobals();

////////////////////////////////////////////////////////////////////////////////
/// <summary>
///  Contains a collection of functions, which are used to process the GME
///  model.
/// </summary>
///
/// <remarks> Zsolt, 8/16/2011. </remarks>
////////////////////////////////////////////////////////////////////////////////

class Graph
{

public:

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Check Bond Graph constraints. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void CheckBGConstraints();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Process the given data_network / GME Project. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	///
	/// <param name="data_network">
	///  [in,out] If non-null, the data network.
	/// </param>
	////////////////////////////////////////////////////////////////////////////////

	void Process(Udm::DataNetwork* data_network);

#pragma region FOLDERS

	// Process the DesignSpace elements inside the domain model.
	// in: C
	void ProcessDesignSpace(DML::DesignSpace &oDesignSpace);

	void ProcessTesting(DML::Testing &oTesting);

	// Process the components elements inside the domain model.
	// in: Components
	void ProcessComponents(DML::Components &oComponents);
	// Process the ComponentAssemblies elements inside the domain model.
	// in: 
	void ProcessComponentAssemblies(DML::ComponentAssemblies &oCAssemblies);

	void ProcessPEF(DML::ParametricExplorationFolder &oPef);


#pragma endregion FOLDERS

#pragma region MODELS

	// Process the DesignContainer elements inside the domain model.
	void ProcessDesignContainer(DML::DesignContainer &oDesignContainer);

	// Process the component elements inside the domain model.
	// in: Component
	void ProcessComponent(DML::Component &oComponent);

	void ProcessTestComponent(DML::TestComponent &oTestComponent);
	void ProcessTestBench(DML::TestBench &oTestBench);

	void ProcessComponentAssembly(DML::ComponentAssembly &oCAssembly);

	// Process the physicalcomponent elements inside the domain model.
	// in: PhysicalComponent
	// Description: PhysicalComponent is a container of all the extended bond
	//              bond graph elements.
	void ProcessPhyComponent(DML::BondGraph &oPhyComponent);

	void ProcessPE(DML::ParametricExploration &oPe);

#pragma endregion MODELS

#pragma region ATOMS_OTHER_ELEMENTS

	void ProcessExtendedElements(DML::ExtendedElements &oExtElements);
	void ProcessBGNode(DML::BGNode &oBGNode);
	void ProcessBond(DML::Bond &oBond);

	void ProcessControlNode(DML::ControlNode &oControlNode);

	void ProcessIoPorts(DML::TestComponent &oTestComponent);
	void ProcessIoPorts(DML::DesignElement &oDesignElement);
	void ProcessIoPorts(DML::BondGraph &oPhyComponent);
	void ProcessLocalSignal(DML::LocalSignal &oLocalSignal);

	void ProcessValueFlowTarget(DML::ValueFlowTarget &oValueFlowTarget);

#pragma endregion ATOMS_OTHER_ELEMENTS

#pragma region CONNECTIONS

	void ProcessConnection(DML::InformationFlow &oIFlow);
	void ProcessConnection(DML::SignalConnectionType &oSType);
	void ProcessConnection(DML::ValueFlow2SignalPort &oVf2Sp);
	void ProcessConnection(DML::ConnectionBase &oConnectionBase);

#pragma endregion CONNECTIONS

	////////////////////////////////////////////////////////////////////////////////
	/// <summary>
	///  Traverses power ports (recursive function). Usage: Call this function
	///  ONLY for the first power port in the chain, while it is a recursive
	///  function it handles all the propagations.
	/// </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	///
	/// <param name="pp">
	///  [in,out] PowerPort, which is the current power port or the start power
	///  port.
	/// </param>
	/// <param name="Vectors">
	///  [in,out] Vectors contains the sequence 'chain' of the connected power
	///  ports.
	/// </param>
	////////////////////////////////////////////////////////////////////////////////

	void GetNextPowerPort(DML::MgaObject &pp,
												vector<vector<DML::MgaObject>> &Vectors);

	// Check the connected power port types for the junctions.
	// input: global AllJunctions list. (GLOBAL variable)
	// output: save messages into the pqMessage queue. (GLOBAL variable)

	////////////////////////////////////////////////////////////////////////////////
	/// <summary>
	///  Check the connected power port types for the junctions. Notes: Input:
	///  global AllJunctions list. (GLOBAL variable) Output: save messages into
	///  the pqMessage queue. (GLOBAL variable)
	/// </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void CheckPowerPorts2Junctions();
};

#endif // _GRAPH_H_