/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
// this code is kept from a previous project
// the appropriate class names have been renamed

#include "HbgGenerator.h"
#include "Tools.h"

#include "TBGNode.h"
#include "TBond.h"
#include "TJunction.h"

#include <fstream>
#include <algorithm>

extern map<long,TBond> AllBonds;
extern vector<TJunction> AllJunctions;
extern vector<TBGNode> BGNodeList;
set<long> VisitedBGNodes;

using namespace std;

// This method derives a simple vector of Nodes from the edgelist
void DeriveBGNodeList(string modelName)
{
	// This loop creates the nodelist from the available list of bonds
	for(map<long,TBond>::iterator i=AllBonds.begin(); i!=AllBonds.end(); ++i)
	{
		if(VisitedBGNodes.find(i->second.src.uniqueId()) == VisitedBGNodes.end())
		{
			VisitedBGNodes.insert(i->second.src.uniqueId());
			TBGNode nb;
			BGNodeList.push_back(nb);
			BGNodeList[BGNodeList.size()-1].ID = BGNodeList.size();
			BGNodeList[BGNodeList.size()-1].Name = GetModifiedName(i->second.src.name());
			
			string tempPath = GetModifiedName(i->second.src.getPath2("/"));
			size_t start = 0;
			size_t end = tempPath.find_first_of("/");
			tempPath = tempPath.replace(start, end, modelName);
			BGNodeList[BGNodeList.size()-1].Path = tempPath;

			BGNodeList[BGNodeList.size()-1].TypeName = i->second.src.type().name();
			BGNodeList[BGNodeList.size()-1].NumberOfBonds = 0;
			BGNodeList[BGNodeList.size()-1].Ref = &(i->second.src);
		}
		if(VisitedBGNodes.find(i->second.dst.uniqueId()) == VisitedBGNodes.end())
		{
			VisitedBGNodes.insert(i->second.dst.uniqueId());
			TBGNode nb;
			BGNodeList.push_back(nb);
			BGNodeList[BGNodeList.size()-1].ID = BGNodeList.size();
			BGNodeList[BGNodeList.size()-1].Name = GetModifiedName(i->second.dst.name());

			string tempPath = GetModifiedName(i->second.dst.getPath2("/"));
			size_t start = 0;
			size_t end = tempPath.find_first_of("/");
			tempPath = tempPath.replace(start, end, modelName);

			BGNodeList[BGNodeList.size()-1].Path = tempPath;
			BGNodeList[BGNodeList.size()-1].TypeName = i->second.dst.type().name();
			BGNodeList[BGNodeList.size()-1].NumberOfBonds = 0;
			BGNodeList[BGNodeList.size()-1].Ref = &(i->second.dst);
		}
	}

	// This loop creates a map in every BGNode of its incident bonds
	for(vector<TBGNode>::iterator i=BGNodeList.begin(); i!=BGNodeList.end(); ++i)
	{
		int counter=1;
		for(map<long,TBond>::iterator it=AllBonds.begin(); it!= AllBonds.end(); ++it)
		{
			if(i->Ref->uniqueId() == it->second.src.uniqueId())
			{
				int a = i->ID;
				BGNodeList[(i->ID)-1].BondMap.insert(pair<int, pair<TBond*, EndPointType>>(counter, pair<TBond*,EndPointType>(&it->second,SOURCE)));
				BGNodeList[(i->ID)-1].NumberOfBonds++;
				counter++;
			}
			else if(i->Ref->uniqueId() == it->second.dst.uniqueId())
			{
				BGNodeList[(i->ID)-1].BondMap.insert(pair<int, pair<TBond*, EndPointType>>(counter, pair<TBond*, EndPointType>(&it->second,DESTINATION)));
				BGNodeList[(i->ID)-1].NumberOfBonds++;
				counter++;
			}
		}
	}
}

// the input parameter has been modified
// the first three lines have been deleted
void GenerateDataStructureOutput(ofstream &DataStructure)
{
	DataStructure << "% HBG datastructure printer - test version" << endl;
	DataStructure << "clear bg;" << endl;
	DataStructure << "global bg;" << endl;
	DataStructure << "bg = [];" << endl;
	DataStructure << "bg.sampleTime = 0.5;" << endl;
	
	int numberOfHybridJunctions = 0;

	// Print IBD output for every bond-node
	for(vector<TBGNode>::iterator i=BGNodeList.begin(); i!=BGNodeList.end(); ++i)
	{
		// Print header line
		DataStructure << endl;
		DataStructure << "% Printing bond node #" << i->ID << endl;

		// Print the ID of the node
		DataStructure << "bg.node(" << i->ID << ").ID = " << i->ID << ";" << endl;
		
		// Print name of the node
		DataStructure << "bg.node(" << i->ID << ").name = \'";
		DataStructure << i->Path << "\';" << endl;
		
		// Print type of the node
		DataStructure << "bg.node(" << i->ID << ").type = \'";
		DataStructure << i->TypeName << "\';" << endl;
		
		// Print number of adjacent bonds
		DataStructure << "bg.node(" << i->ID << ").numBonds = ";
		DataStructure << i->NumberOfBonds << ";" << endl;
		
		// Print references to other bond-nodes via incident bonds
		for(map<int, pair<TBond*,EndPointType>>::iterator mapit = i->BondMap.begin(); mapit!=i->BondMap.end(); ++mapit)
		{
			string nodename = "";
			long nodeUID = 0;
			if(mapit->second.second == SOURCE)
			{
				nodename = string(mapit->second.first->dst.name());
				nodeUID = mapit->second.first->dst.uniqueId();
			}
			else
			{
				nodename = string(mapit->second.first->src.name());
				nodeUID = mapit->second.first->src.uniqueId();
			}
			
			DataStructure << "bg.node(" << i->ID << ").bond(" << mapit->first << ") = ";
			for(vector<TBGNode>::iterator nodeit = BGNodeList.begin(); nodeit!=BGNodeList.end(); ++nodeit)
			{
				if(nodeUID == nodeit->Ref->uniqueId())
				{
					DataStructure << nodeit->ID << ";";
					DataStructure << " % " << nodeit->Path << endl;
					break;
				}
			}
		}
		

		// Junction-types have additional fields to print
		if (i->TypeName == "OneJunction" || i->TypeName == "ZeroJunction")
		{
			CyPhyML::Junction thisJunction;
			int thisJunctionID = 1;
			
			// Search for BDM instance in AllJunctions vector
			for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit < AllJunctions.end(); ++jit)
			{
				if(jit->p.uniqueId() == i->Ref->uniqueId())
				{
					thisJunction = jit->p;
					break;
				}
				thisJunctionID++;
			}
			
			// Print initial state of the junction
			DataStructure << "bg.node(" << i->ID << ").state = ";
			if(string(thisJunction.InitialState()) == "ON")
			{
				DataStructure << "1;" << endl;
			}
			else
			{
				DataStructure << "0;" << endl;
			}

			// Print index of the junction
			DataStructure << "bg.node(" << i->ID << ").index = " << thisJunctionID << ";" << endl;

			// Print whether the junction is hybrid or not, determined by the presence of guard expressions
			DataStructure << "bg.node(" << i->ID << ").hybrid = ";
			
			//if(string(thisJunction.OffCondition()) != "" || string(thisJunction.OnCondition()) != "")
			
			set<CyPhyML::Switching2Junction> sw2j = thisJunction.srcSwitching2Junction();
			set<CyPhyML::Signal2Junction> sig2j = thisJunction.srcSignal2Junction();
			if(sw2j.size() != 0 || sig2j.size() != 0)
			{
        // check junction's ON/OFF conditions based on the defined names
				DataStructure << "1;" << endl;
				// Also print the guard expressions here
				DataStructure << "bg.node(" << i->ID << ").onGuard = '";
				DataStructure << thisJunction.OnCondition() << "';" << endl;
				DataStructure << "bg.node(" << i->ID << ").offGuard = '";
				DataStructure << thisJunction.OffCondition() << "';" << endl;
				++numberOfHybridJunctions;
			}
			else
			{
				DataStructure << "0;" << endl;
			}
		}
	}

	// Print additional Bond-graph properties
	DataStructure << endl;
	DataStructure << "bg.numNodes = " << BGNodeList.size() << ";" << endl;
	DataStructure << "bg.numJunctions = " << AllJunctions.size() << ";" << endl;
	DataStructure << "bg.numHybridJunctions = " << numberOfHybridJunctions << ";" << endl;

	DataStructure << "bg.db = [ ";
	for(unsigned int i=1; i<=AllJunctions.size(); ++i)
	{
		DataStructure << "-1 ";
	}
	DataStructure << "];" << endl;
	DataStructure << "global determiningBonds;" << endl;
}