/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "MdaoComponent.h"

MdaoComponent::MdaoComponent(string SubDirName) : MdaoBase(SubDirName)
{
	BaseClasses.push_back("Component");
	MdaoBase::InitBaseClasses();
	InitImports();
	CustomFormulas.clear();
	MetricCustomFormula.clear();
	CustomFormulaOrder.clear();
}


MdaoComponent::~MdaoComponent(void)
{
}

void MdaoComponent::InitImports()
{
	MdaoBase::InitImports();
	Imports.insert(make_pair("Component", "openmdao.main.api"));
}

void MdaoComponent::PrintPython()
{
	MdaoBase::PrintPython();

	py.CreateComment("OpenMDAO Component");
	if (Type == MdaoComponent::MatLab)
	{
		py.CreateComment("  Type: Dynamic Simulation (MatLab)");
	}
	else if (Type == MdaoComponent::Block)
	{
		py.CreateComment("  Type: Block");
	}
	else if (Type == MdaoComponent::Excel)
	{
		py.CreateComment("  Type: Excel");
	}


	if (Type == MdaoComponent::MatLab)
	{
		Imports.insert(make_pair("os", ""));
		Imports.insert(make_pair("subprocess", ""));
	}

	// import classes
	PrintImports();

	if (Type == MdaoComponent::Block)
	{
		py.CreateImport("*","math");
		py.CreateImport("os");
	}
	else 	if (Type == MdaoComponent::Excel)
	{
		py.CreateImport("os","");
		py.CreateImport("logging","");
		py.CreateImport("win32com.client","");
	}
	else if (Type == MdaoComponent::MatLab)
	{
		py.CreateImport("os","");
		py.CreateImport("win32com.client","");
	}

	py.CreateEmptyLine();

	PrintClassDef();

	if (Type == MdaoComponent::MatLab)
	{
		py.CreateComment("matlab com handle");
		py.CreateCodeLine("__h = win32com.client.Dispatch('matlab.application')");
	}
	py.CreateComment("set up interface to the framework", 2);

	PrintInputs();

	PrintOutputs();

	if (Type == MdaoComponent::Block)
	{
		py.CreateFunction("__init__");

		py.CreateComment("$Static");
		py.CreateCodeLine("super(" + Name + ", self).__init__()");

		py.CreateComment("$GENERIC - all formulas");
		py.CreateComment("$set the strings based on what was put in for muparser");

		for(map<string, string>::const_iterator it = CustomFormulas.begin();
			it != CustomFormulas.end();
			++it)
		{
			string expression=it->second;
			tr1::regex rx("[\n]");
			string fmt(" ");
			expression = string(tr1::regex_replace(expression, rx, fmt));
			py.CreateCodeLine("self.__mustring_" + it->first + " = '" + expression + "'");
		}

		for(map<string, string>::const_iterator it = CustomFormulas.begin();
			it != CustomFormulas.end();
			++it)
		{
			py.CreateCodeLine("self." + it->first + " = 0");
		}

		py.CloseSection("__init__");

		py.CreateFunction("testComputation");
		
		py.CreateCodeLine("print \"--- Input Defaults ---\"");
		for (map<string, string>::const_iterator it = Inputs.begin();
			it != Inputs.end();
			++it)
		{
			// print "VARNAME:" + str(singleComponent.VARNAME)
			py.CreateCodeLine("print \"" + it->first +
				":\" + str(self." + it->first +")");
		}

		py.CreateCodeLine("print \"--- Collecting default Output Values ---\"");
		py.CreateComment("$Generic - all outputs (metrics)");
		for (map<string, string>::const_iterator it = Outputs.begin();
			it != Outputs.end();
			++it)
		{
			py.CreateCodeLine("local_" + it->first +
				" = self." + it->first);
		}
		py.CreateComment("$Static");
		py.CreateCodeLine("print \"--- Evaluating ---\"");
		py.CreateCodeLine("self.execute()");

		py.CreateComment("$Generic - all outputs (metrics)");
		if (Outputs.size() > 0)
		{
			py.CreateCodeLine("outputList = [");
		}
		for (map<string, string>::const_iterator it = Outputs.begin();
			it != Outputs.end();
			++it)
		{
			py.CreateCodeLine(" (\"" + it->first +
				"\", local_" + it->first +
				", self." + it->first + "),");
		}
		if (Outputs.size() > 0)
		{
			py.CreateCodeLine("]");
		}

		py.CreateComment("$Static");
		py.CreateCodeLine("absRelError = 0");
		py.CreateCodeLine("print \"--- Metric, Output, relative error ---\"");

		py.CreateCodeLine("for name, expected, computed in outputList:", true);
		py.CreateCodeLine("relError = abs(expected-computed)/(abs(computed)+1e-8)");
		py.CreateCodeLine("print name + \", \\t\" + str(computed) + \", \\t\" + str(relError)");
		py.CreateCodeLine("absRelError = max(absRelError, relError)");
		py.CloseSection("for");

		py.CreateCodeLine("print \"Max Relative Error: \"");
		py.CreateCodeLine("print str(absRelError) ");
		py.CloseSection("testComputation");

		py.CreateFunction("execute");
		py.CreateComment("$GENERIC - all formulas and outputs (metrics) in order of dependency");

		py.CreateComment(" TODO: Dependency is not captured yet!");

		for(vector<string>::const_iterator it = CustomFormulaOrder.begin();
			it != CustomFormulaOrder.end();
			++it)
		{
			py.CreateCodeLine("self."+ *it + " = self.muParse(self.__mustring_" + *it + ")");
		}

		for(map<string, string>::const_iterator it = CustomFormulas.begin();
			it != CustomFormulas.end();
			++it)
		{
			for(map<string, string>::const_iterator itMetric = MetricCustomFormula.begin();
				itMetric != MetricCustomFormula.end();
				++itMetric)
			{
				if (itMetric->second == it->first)
				{
					py.CreateCodeLine("self." + itMetric->first + " = self." + it->first);
				}
			}
		}

		py.CloseSection("execute");

		py.CreateFunction("muParse","s");
		py.CreateComment("$GENERIC - all inputs (parameters) formulas and outputs(metrics)");
		
		for (map<string, string>::const_iterator it = Inputs.begin();
			it != Inputs.end();
			++it)
		{
			// VARNAME_IN = self.VARNAME_IN
			py.CreateCodeLine(it->first + " = self." + it->first);
		}

		for(map<string, string>::const_iterator it = CustomFormulas.begin();
			it != CustomFormulas.end();
			++it)
		{
			py.CreateCodeLine(it->first + " = self."+ it->first);
		}

		for (map<string, string>::const_iterator it = Outputs.begin();
			it != Outputs.end();
			++it)
		{
			// VARNAME_OUT = self.VARNAME_OUT
			py.CreateCodeLine(it->first + " = self." + it->first);
		}

		py.CreateCodeLine("return eval(s.replace('^', '**'))");

		py.CloseSection("muParse");

		py.CloseSection(Name + "class");

		py.CreateCodeLine("if __name__ == '__main__':", true);
		py.CreateCodeLine("print \"Instantiating " + Name + " component\"");
		py.CreateCodeLine("singleComponent = " + Name + "()");

		py.CreateCodeLine("singleComponent.testComputation()");
		py.CreateComment("del(singleComponent)");
		py.CreateComment("os._exit(1)");

	}
	else if (Type == MdaoComponent::Excel)
	{
		// TODO: Remove this line
		//TargetFileName = "powertrain_RSEs2.xls";
		// start
		py.CreateComment("$STATIC");
		py.CreateFunction("__init__","");
		py.CreateCodeLine("super(" + Name + ", self).__init__()");
		py.CreateCodeLine("self.excelFile = r\"" + TargetFileName + "\"");
		py.CreateCodeLine("self.xlInstance = None");
		py.CreateCodeLine("self.workbook = None");
		py.CreateCodeLine("self.ExcelConnectionIsValid = True");
		
		py.CreateCodeLine("if not os.path.exists(self.excelFile):", true);
		py.CreateCodeLine("print \"Invalid file given\"");
		py.CreateCodeLine("self.ExcelConnectionIsValid = False");
		py.CloseSection();
		
		py.CreateCodeLine("else:", true);
		py.CreateCodeLine("self.excelFile = os.path.abspath(self.excelFile)");
		py.CreateCodeLine("xl = self.openExcel()");

		py.CreateCodeLine("if xl is None:", true);
		py.CreateCodeLine("print \"Connection to Excel failed.\"");
		py.CreateCodeLine("self.ExcelConnectionIsValid = False");
		py.CloseSection();

		py.CreateCodeLine("else:", true);
		py.CreateCodeLine("self.xlInstance = xl");
		py.CreateCodeLine("self.workbook = xl.Workbooks.Open(self.excelFile)");
		py.CloseSection();

		py.CloseSection();
		py.CloseSection("__init__");

		py.CreateFunction("__del__", "");
		py.CreateCodeLine("if self.workbook is not None:", true);
		py.CreateCodeLine("self.workbook.Close(SaveChanges=False)");
		py.CloseSection();

		py.CreateCodeLine("if self.xlInstance is not None:", true);
		py.CreateCodeLine("del(self.xlInstance)");
		py.CreateCodeLine("self.xlInstance = None");
		py.CloseSection();
		py.CloseSection("__del__");

		py.CreateFunction("openExcel","");
		py.CreateCodeLine("try:", true);
		py.CreateCodeLine("xl = win32com.client.Dispatch(\"Excel.Application\")");
		py.CloseSection();
		py.CreateCodeLine("except:", true);
		py.CreateCodeLine("return None");
		py.CloseSection();
		py.CreateCodeLine("return xl");
		py.CloseSection("openExcel");

		py.CreateComment("$/STATIC");

		py.CreateFunction("testComputation", "");
		py.CreateCodeLine("print \"--- Input Defaults ---\"");
		py.CreateComment("$Generic - all inputs (parameters)");
		for (map<string, string>::iterator it = Inputs.begin();
			it != Inputs.end();
			++it)
		{
			py.CreateCodeLine("print \"" +
				it->first +
				":\" + str(self." +
				it->first +
				")");
		}
		py.CreateComment("$Static");
		py.CreateCodeLine("print \"---   Evaluating   ---\"");
		py.CreateCodeLine("self.execute()");
		py.CreateCodeLine("print \"---    Outputs     ---\"");
		py.CreateComment("$Generic - all outputs (metrics)");
		for (map<string, string>::iterator it = Outputs.begin();
			it != Outputs.end();
			++it)
		{
			py.CreateCodeLine("print \"" +
				it->first +
				":\" + str(self." +
				it->first +
				")");
		}
		py.CloseSection("testComputation");
		// end

		// create execute function
		py.CreateFunction("execute");
		py.CreateComment("$STATIC");
		py.CreateCodeLine("if not self.ExcelConnectionIsValid or \\", true);
		py.CreateCodeLine("self.xlInstance is None or \\");
		py.CreateCodeLine("self.workbook is None:");
		py.CreateCodeLine("print \"Aborted Execution of Bad Excel Component Instance\"");
		py.CreateCodeLine("return");
		py.CloseSection();
		py.CreateCodeLine("wb = self.workbook");
		py.CreateComment("$/STATIC");
		py.CreateEmptyLine();
		py.CreateComment("Set the inputs");
		py.CreateComment("$Generic - all inputs (parameters)");
		for (map<string, string>::iterator it = Inputs.begin();
			it != Inputs.end();
			++it)
		{
			py.CreateCodeLine("self.xlInstance.Range(wb.Names(\"" +
				it->first +
				"\").RefersToLocal).Value = self." +
				it->first);
		}
		py.CreateEmptyLine();
		py.CreateComment("Get the outputs");
		py.CreateComment("$Generic - all outputs (metrics)");
		for (map<string, string>::iterator it = Outputs.begin();
			it != Outputs.end();
			++it)
		{
			py.CreateCodeLine("self." +
				it->first +
				" = self.xlInstance.Range(wb.Names(\"" +
				it->first +
				"\").RefersToLocal).Value");
		}
		py.CloseSection("execute");
		py.CloseSection(Name + " class");

		py.CreateCodeLine("if __name__ == '__main__':", true);
		py.CreateComment("$Static");
		py.CreateCodeLine("print \"Instantiating EXCEL_Cost component\"");
		py.CreateCodeLine("singleComponent = " + Name + "()");
		py.CreateCodeLine("singleComponent.testComputation()");
		py.CreateCodeLine("del(singleComponent)");
		py.CreateCodeLine("os._exit(1)");
	}
	else if (Type == MdaoComponent::MatLab)
	{
		py.CreateFunction("__init__","dir");
		py.CreateCodeLine("super(" + Name + ", self).__init__()");

		py.CreateCodeLine("if dir is None:", true);
		py.CreateCodeLine("self.workingDir = os.path.abspath(r\"" + 
			TargetFileName.substr(0, TargetFileName.length() - 1) + 
			"\")");

		py.CloseSection("if");
		py.CreateCodeLine("else:", true);
		py.CreateCodeLine("self.workingDir = os.path.abspath(dir) + '/'");
		py.CloseSection("else");

		py.CreateCodeLine("self._" + Name + "__h.Execute('cd ' + self.workingDir);");
		py.CreateCodeLine("self._" + Name + "__h.Execute('build_model');");
		py.CreateCodeLine("self._" + Name + "__h.Execute('writeSample');");

		py.CloseSection("__init__");


		// create execute function
		py.CreateFunction("execute");

		// HACK:
		//py.CreateCodeLine("return");

		string inFileName = "testIn_py.txt";
		string outFileName = "testOut_py.txt";
		
		//py.CreateComment("cwdPath = os.getcwd()");
		py.CreateCodeLine("print 'Executing " + Name + " component...'");
		py.CreateComment("Write input file ");
		py.CreateCodeLine("self.write_input(os.path.join(self.workingDir, '" + inFileName + "'))");
		py.CreateCodeLine("self._" + Name + "__h.Execute('cd ' + self.workingDir);");


		py.CreateCodeLine("self._" + Name + "__h.Execute(\"run_model(\\'" + inFileName + "\\',\\'" + outFileName + "\\');\")");

		py.CreateComment("Read the results");
		py.CreateCodeLine("self.read_result(os.path.join(self.workingDir, '" + outFileName + "'))");
		//py.CreateComment("Change dir back to origin");
		py.CloseSection("execute");

				// create the write_input function
		py.CreateFunction("write_input", "inputFile");
		py.CreateCodeLine("f = open(os.path.join(self.workingDir, 'testInputSample.txt'),'r')");
		py.CreateCodeLine("firstLine = f.readline()");
		py.CreateCodeLine("fout = open(inputFile,'w')");
		py.CreateCodeLine("fout.write(firstLine)");

		py.CreateCodeLine("for var in firstLine.split() :", true);
		py.CreateCodeLine("fout.write('%1.10e ' % self.get(var))");
		py.CloseSection("for");

		py.CreateCodeLine("f.close()");
		py.CreateCodeLine("fout.close()");

		py.CloseSection("write_input");

		// create the read_result function
		py.CreateFunction("read_result", "outputFile");
		py.CreateCodeLine("f = open(outputFile, 'r')");
		py.CreateCodeLine("varNames = f.readline().split()");
		py.CreateComment("NOTE: (first set , using only the first line)'");
		py.CreateCodeLine("varValues = f.readline().split()");

		py.CreateCodeLine("for n,v in zip(varNames,varValues) :", true);
		py.CreateCodeLine("self.set(n,float(v))");
		py.CloseSection("for");

		py.CreateCodeLine("f.close()");
		py.CloseSection("read_result");

	}
	py.CloseSection(Name + ".py");
}
