/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "PrintScript.h"
#include "Tools.h"

#include "CyPhyML.h"

#include "TBGNode.h"
#include "TBond.h"
#include "TJunction.h"

#include "PortMapping.h"
#include "PowerPortInfo.h"
#include "PowerChain.h"
#include "ScriptFile.h"
#include "BlockConnection.h"

#include "Graph.h"

#include "Message.h"
#include <queue>

#include <fstream>

// Global variables
extern priority_queue<Message> pqMessages;

extern ScriptFile sfFile; // output script file and datastructure file

extern map<long,TBond>   AllBonds;
extern vector<TJunction> AllJunctions;
extern vector<TBGNode>   BGNodeList;

extern vector<PowerChain>       PowerPortChains;
extern map<int, map<long, PowerPortInfo>> PPInfos;
extern map<long, PortNumbers>   ioPortNumbers;

extern vector<BlockConnection>  conns;

extern bool bAlgLoops;

// Traverse bond-nodes and print their buildfunction-calls
void PrintBondElements(ofstream& outf)
{
	outf << endl;
	outf << "% Print bond-graph element blocks" << endl;
	for(vector<TBGNode>::iterator bondIt = BGNodeList.begin(); bondIt != BGNodeList.end(); bondIt++)
	{
		if(bondIt->TypeName == "ZeroJunction" || bondIt->TypeName == "OneJunction")
		{
			// Search for CyPhyML instance in AllJunctions vector
			CyPhyML::Junction thisJunction;
			for(vector<TJunction>::iterator jit = AllJunctions.begin(); jit < AllJunctions.end(); ++jit)
			{
				if(jit->p.uniqueId() == bondIt->Ref->uniqueId())
				{
					thisJunction = jit->p;
					break;
				}
			}
			
			set<CyPhyML::Switching2Junction> sw2j = thisJunction.srcSwitching2Junction();
			set<CyPhyML::Signal2Junction> sig2j = thisJunction.srcSignal2Junction();
			if(sw2j.size() != 0 || sig2j.size() != 0)
			{
				// Junction is hybrid
				// params: (ssName, sys, pos, numBonds), decVars, index, signArray, onExp, offExp, initState

				// ssName
				outf << "hbg_hybridjunction('" << bondIt->Name << "', ";				

				// sys
				outf << "'" << bondIt->getPathWOName() << "', ";

				// pos
				outf << "[" << GetObjectXPosition(*(bondIt->Ref)) << ",";
				outf << GetObjectYPosition(*(bondIt->Ref)) << ",";
				outf << GetObjectXPosition(*(bondIt->Ref))+36 << ",";
				outf << GetObjectYPosition(*(bondIt->Ref))+36 << "], ";

				// numBonds
				// this argument is calculated from the datasctructure
				outf << "bg.node(" << bondIt->ID << ").numBonds, ";

				// index
				// this argument is calculated from the datasctructure
				outf << "bg.node(" << bondIt->ID << ").index, ";

				// signArray
				outf << "'[";
				bool firstTime = true;
				// Go through the BondMap IN ASCENDING ORDER
				for(unsigned int i=1; i < bondIt->BondMap.size()+1; ++i)
				{
					for(map<int, pair<TBond*,EndPointType>>::iterator jit = bondIt->BondMap.begin();
						jit != bondIt->BondMap.end(); ++jit)
					{
						if(jit->first == i)
						{
							if(jit->second.second == SOURCE ||
								jit->second.first->src.type() == DML::C::meta ||
								jit->second.first->dst.type() == DML::C::meta ||
								jit->second.first->src.type() == DML::I::meta ||
								jit->second.first->dst.type() == DML::I::meta ||
								jit->second.first->src.type() == DML::R::meta ||
								jit->second.first->dst.type() == DML::R::meta )
							{
								if(firstTime)
								{
									outf << "1";
								}
								else
								{
									outf << " 1";
								}
							}
							else if(jit->second.second == DESTINATION)
							{
								if(firstTime)
								{
									outf << "-1";
								}
								else
								{
									outf << " -1";
								}
							}
							firstTime = false;
							break;
						}
					}
				}
				outf << "]', ";

				// decVars
				int decVars = 0;
				string onCondition = string(thisJunction.OnCondition());
				string offCondition = string(thisJunction.OffCondition());

				// Get connected Switching elements
				set<CyPhyML::Switching2Junction> sw2juncSet = thisJunction.srcSwitching2Junction();
				for(set<CyPhyML::Switching2Junction>::iterator sw2juncIterator = sw2juncSet.begin();
					sw2juncIterator != sw2juncSet.end(); ++sw2juncIterator)
				{
					++decVars;

					// Process the guardexpression strings
					string inputStr = "u(";
					char buff[33];
					_itoa_s(decVars+bondIt->BondMap.size(), buff, 10);
					inputStr += string(buff);
					inputStr += ")";

					// Replace controller Switching's names
					string swName = string(CyPhyML::Switching(sw2juncIterator->srcSwitching2Junction_end()).name());
					size_t found = onCondition.find(swName);
					while(found != string::npos)
					{
						onCondition.replace(found, swName.size(), inputStr);
						found = onCondition.find(swName);
					}

					found = offCondition.find(swName);
					while(found != string::npos)
					{
						offCondition.replace(found, swName.size(), inputStr);
						found = offCondition.find(swName);
					}
				}

				// TODO: ?works for ONLY if sig2junc.size() == 1; ???
				// Get connected signals
				set<CyPhyML::Signal2Junction> sig2junc  = thisJunction.srcSignal2Junction();
				if(sig2junc.size() != 0)
				{
					++decVars;
					
					// Process the guardexpression strings
					string inputStr = "u(";
					char buff[33];
					_itoa_s(decVars+bondIt->BondMap.size(), buff, 10);
					inputStr += string(buff);
					inputStr += ")";

					// TODO: for cycle is necessary here sig2junc.begin()
					// Replace controller Switching's names
					string sigName = string(CyPhyML::Signal(sig2junc.begin()->srcSignal2Junction_end()).name());
					size_t found = onCondition.find(sigName);
					while(found != string::npos)
					{
						onCondition.replace(found, sigName.size(), inputStr);
						found = onCondition.find(sigName);
					}

					found = offCondition.find(sigName);
					while(found != string::npos)
					{
						offCondition.replace(found, sigName.size(), inputStr);
						found = offCondition.find(sigName);
					}
				}

				outf << decVars << ", ";
				
				// onExp
				outf << "'" << onCondition << "', ";

				// offExp
				outf << "'" << offCondition << "', ";

				// initState
				if(string(thisJunction.InitialState()) == "ON")
				{
					outf << "1);";
				}
				else
				{
					outf << "0);";
				}
				
				outf << endl;
			}
			else
			{
				// Junction is NOT hybrid
				// params: (ssName, sys, pos, numBonds), index, signArray

				// ssName
				outf << "hbg_junction('" << bondIt->Name << "', ";				

				// sys
				outf << "'" << bondIt->getPathWOName() << "', ";

				// pos
				outf << "[" << GetObjectXPosition(*(bondIt->Ref)) << ",";
				outf << GetObjectYPosition(*(bondIt->Ref)) << ",";
				outf << GetObjectXPosition(*(bondIt->Ref))+36 << ",";
				outf << GetObjectYPosition(*(bondIt->Ref))+36 << "], ";

				// numBonds
				outf << "bg.node(" << bondIt->ID << ").numBonds, ";

				// index
				outf << "bg.node(" << bondIt->ID << ").index, ";

				// signArray
				outf << "'[";
				bool firstTime = true;
				// Go through the BondMap IN ASCENDING ORDER
				for(unsigned int i=1; i<bondIt->BondMap.size()+1; ++i)
				{
					for(map<int, pair<TBond*,EndPointType>>::iterator jit = bondIt->BondMap.begin();
						jit != bondIt->BondMap.end(); ++jit)
					{
						if(jit->first == i)
						{
							if(jit->second.second == SOURCE ||
								jit->second.first->src.type() == DML::C::meta ||
								jit->second.first->dst.type() == DML::C::meta ||
								jit->second.first->src.type() == DML::I::meta ||
								jit->second.first->dst.type() == DML::I::meta ||
								jit->second.first->src.type() == DML::R::meta ||
								jit->second.first->dst.type() == DML::R::meta )
							{
								if(firstTime)
								{
									outf << "1";
								}
								else
								{
									outf << " 1";
								}
							}
							else if(jit->second.second == DESTINATION)
							{
								if(firstTime)
								{
									outf << "-1";
								}
								else
								{
									outf << " -1";
								}
							}
							firstTime = false;
							break;
						}
					}
				}
				outf << "]');";
				outf << endl;
			}
		}
		else if(bondIt->TypeName == "Se" || bondIt->TypeName == "MSe")
		{
			// params: ssName, sys, pos, value, ismod
			
			// ssName
			outf << "hbg_se_element('" << bondIt->Name << "', ";

			// sys
			outf << "'" << bondIt->getPathWOName() << "', ";

			// pos
			outf << "[" << GetObjectXPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectYPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectXPosition(*(bondIt->Ref))+36 << ",";
			outf << GetObjectYPosition(*(bondIt->Ref))+36 << "], ";

			if(bondIt->TypeName == "Se")
			{
				// value
				outf << CyPhyML::Se::Cast(*(bondIt->Ref)).ParameterValue() << ", ";
				outf << "0);";
			}
			else // modulated
			{
				// value contant zero
				outf << "0, ";

				// ismod
				outf << "1);";
			}
			
			outf << endl;
			

		}
		else if(bondIt->TypeName == "Sf" || bondIt->TypeName == "MSf")
		{
			// params: ssName, sys, pos, value, ismod
			
			// ssName
			outf << "hbg_sf_element('" << bondIt->Name << "', ";

			// sys
			outf << "'" << bondIt->getPathWOName() << "', ";

			// pos
			outf << "[" << GetObjectXPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectYPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectXPosition(*(bondIt->Ref))+36 << ",";
			outf << GetObjectYPosition(*(bondIt->Ref))+36 << "], ";

			if(bondIt->TypeName == "Sf")
			{
				// value
				outf << CyPhyML::Sf::Cast(*(bondIt->Ref)).ParameterValue() << ", ";
				
				// ismod
				outf << "0);";
			}
			else // modulated
			{
				// value contant zero
				outf << "0, ";

				// ismod
				outf << "1);";
			}
			
			outf << endl;
		}
		else if(bondIt->TypeName == "R" || bondIt->TypeName == "MR")
		{
			// params: ssName, sys, pos, j, b, juncType, value, ismod
			
			if (bAlgLoops)
			{
				outf << "hbg_resistor_w_memory('";
			}
			else
			{
				outf << "hbg_resistor('";
			}

			// ssName
			outf << bondIt->Name << "', ";

			// sys
			outf << "'" << bondIt->getPathWOName() << "', ";

			// pos
			outf << "[" << GetObjectXPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectYPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectXPosition(*(bondIt->Ref))+36 << ",";
			outf << GetObjectYPosition(*(bondIt->Ref))+36 << "], ";


			// j
			
			// Get neighbourjunction (first CyPhyML object, than TBGNode) using name as primary ID (unique in CyPhyML)
			CyPhyML::Junction CyPhyMLNeighbourJunction;
			if(bondIt->BondMap.find(1)->second.second == SOURCE)
			{
				CyPhyMLNeighbourJunction = CyPhyML::Junction::Cast(bondIt->BondMap.find(1)->second.first->dst);
			}
			else
			{
				CyPhyMLNeighbourJunction = CyPhyML::Junction::Cast(bondIt->BondMap.find(1)->second.first->src);
			}
			
			TBGNode nj;
			for(vector<TBGNode>::iterator bit = BGNodeList.begin(); bit != BGNodeList.end(); ++bit)
			{
				if(CyPhyMLNeighbourJunction.uniqueId() == bit->Ref->uniqueId())
				{
					nj = *bit;
					break;
				}
			}
			
			int b = 0;
			for(map<int, pair<TBond*,EndPointType>>::iterator bit = nj.BondMap.begin();
				bit != nj.BondMap.end(); ++bit)
			{
				if(bit->second.second == SOURCE)
				{
					if(bit->second.first->dst.uniqueId() == bondIt->Ref->uniqueId())
					{
						b = bit->first;
					}
				}
				else
				{
					if(bit->second.first->src.uniqueId() == bondIt->Ref->uniqueId())
					{
						b = bit->first;
					}
				}
			}
			outf << "bg.node(" << nj.ID << ").index, ";
			
			// b
			if(b != 0) // Found R on neighbouring junction's BondMap
			{
				outf << b << ", ";
			}
			else
			{
				outf << "ERROR, ";
			}
			
			// juncType
			if(nj.TypeName == "OneJunction")
			{
				outf << "1, ";
			}
			else
			{
				outf << "0, ";
			}

			// value, ismod
			if(bondIt->TypeName == "R")
			{
				// value
				outf << string(CyPhyML::R::Cast(*bondIt->Ref).ParameterValue()) << ", ";
				
				// ismod
				outf << "0);";

			}
			else // modulated
			{
				// value
				outf << string(CyPhyML::MR::Cast(*bondIt->Ref).ParameterValue()) << ", ";
				
				// ismod
				outf << "1);";

			}

			outf << endl;
		}
		else if(bondIt->TypeName == "C" || bondIt->TypeName == "MC")
		{
			// params: ssName, sys, pos, value, initial, ismod
			DML::Storage s = DML::Storage::Cast(*bondIt->Ref);
			set<DML::Signal2InitialValue> s2iv = s.srcSignal2InitialValue();
			if (s2iv.empty())
			{
				// initial value is a constant
				outf << "hbg_capacitor('";
			}
			else
			{
				// initial value is a signal, it should be a port
				outf << "hbg_capacitor_w_iv_port('";
			}

			// ssName
			outf << bondIt->Name << "', ";

			// sys
			outf << "'" << bondIt->getPathWOName() << "', ";

			// pos
			outf << "[" << GetObjectXPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectYPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectXPosition(*(bondIt->Ref))+36 << ",";
			outf << GetObjectYPosition(*(bondIt->Ref))+36 << "], ";

			// value, initial, ismod
			if(bondIt->TypeName == "C")
			{
				// value
				outf << string(CyPhyML::C::Cast(*bondIt->Ref).ParameterValue()) << ", ";

				// initial
				outf << string(CyPhyML::C::Cast(*bondIt->Ref).InitialValue()) << ", ";

				// ismod
				outf << "0);";
			}
			else // MOD
			{
				// value
				outf << "1" << ", ";

				// initial
				outf << string(CyPhyML::MC::Cast(*bondIt->Ref).InitialValue()) << ", ";

				// ismod
				outf << "1);";
			}			

			outf << endl;
		}
		else if(bondIt->TypeName == "I" || bondIt->TypeName == "MI")
		{
			// params: ssName, sys, pos, value, initial, ismod
			
			DML::Storage s = DML::Storage::Cast(*bondIt->Ref);
			set<DML::Signal2InitialValue> s2iv = s.srcSignal2InitialValue();
			if (s2iv.empty())
			{
				// initial value is a constant
				outf << "hbg_inertia('";
			}
			else
			{
				// initial value is a signal, it should be a port
				outf << "hbg_inertia_w_iv_port('";
			}

			// ssName
			outf << bondIt->Name << "', ";

			// sys
			outf << "'" << bondIt->getPathWOName() << "', ";

			// pos
			outf << "[" << GetObjectXPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectYPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectXPosition(*(bondIt->Ref))+36 << ",";
			outf << GetObjectYPosition(*(bondIt->Ref))+36 << "], ";

			// value, initial, ismod
			if(bondIt->TypeName == "I")
			{
				// value
				outf << string(CyPhyML::I::Cast(*bondIt->Ref).ParameterValue()) << ", ";

				// initial
				outf << string(CyPhyML::I::Cast(*bondIt->Ref).InitialValue()) << ", ";

				// ismod
				outf << "0);";
			}
			else // MOD
			{
				// value
				outf << "1" << ", ";

				// initial
				outf << string(CyPhyML::MI::Cast(*bondIt->Ref).InitialValue()) << ", ";

				// ismod
				outf << "1);";
			}	

			outf << endl;
		}
		else if(bondIt->TypeName == "GY" || bondIt->TypeName == "MGY")
		{
			// params: ssName, sys, pos, GYnode, junc, value, ismod
			
			// ssName
			outf << "hbg_gyrator('" << bondIt->Name << "', ";

			// sys
			outf << "'" << bondIt->getPathWOName() << "', ";

			// pos
			outf << "[" << GetObjectXPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectYPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectXPosition(*(bondIt->Ref))+36 << ",";
			outf << GetObjectYPosition(*(bondIt->Ref))+36 << "], ";

			// GYnode
			outf << bondIt->ID << ", ";

			// junc
			// Get neighbourjunction (first CyPhyML object, than TBGNode) using name as primary ID (unique in CyPhyML)
			CyPhyML::Junction CyPhyMLNeighbourJunction;
			if(bondIt->BondMap.find(1)->second.second == SOURCE)
			{
				CyPhyMLNeighbourJunction = CyPhyML::Junction::Cast(bondIt->BondMap.find(1)->second.first->dst);
			}
			else
			{
				CyPhyMLNeighbourJunction = CyPhyML::Junction::Cast(bondIt->BondMap.find(1)->second.first->src);
			}
			
			TBGNode nj;
			for(vector<TBGNode>::iterator bit = BGNodeList.begin(); bit != BGNodeList.end(); ++bit)
			{
				if(CyPhyMLNeighbourJunction.uniqueId() == bit->Ref->uniqueId())
				{
					nj = *bit;
					break;
				}
			}
			
			outf << "bg.node(" << nj.ID << ").index, ";
			
			
			// value, ismod
			if(bondIt->TypeName == "GY")
			{
				// GY
				// value
				outf << string(CyPhyML::GY::Cast(*bondIt->Ref).ParameterValue()) << ", ";
				
				// ismod
				outf << "0);";
			}
			else
			{
				// MGY
				// value contant zero
				outf << "0, ";

				// ismod
				outf << "1);";
			}
			outf << endl;
		}
		else if(bondIt->TypeName == "TF" || bondIt->TypeName == "MTF")
		{
			// params: ssName, sys, pos, TFnode, junc, value, ismod
			
			// ssName
			outf << "hbg_transformer('" << bondIt->Name << "', ";

			// sys
			outf << "'" << bondIt->getPathWOName() << "', ";

			// pos
			outf << "[" << GetObjectXPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectYPosition(*(bondIt->Ref)) << ",";
			outf << GetObjectXPosition(*(bondIt->Ref))+36 << ",";
			outf << GetObjectYPosition(*(bondIt->Ref))+36 << "], ";
			
			// TFnode
			outf << bondIt->ID << ", ";
			
			// junc
			// Get neighbourjunction (first CyPhyML object, than TBGNode) using name as primary ID (unique in CyPhyML)
			CyPhyML::Junction CyPhyMLNeighbourJunction;
			if(bondIt->BondMap.find(1)->second.second == SOURCE)
			{
				CyPhyMLNeighbourJunction = CyPhyML::Junction::Cast(bondIt->BondMap.find(2)->second.first->src);
			}
			else
			{
				CyPhyMLNeighbourJunction = CyPhyML::Junction::Cast(bondIt->BondMap.find(1)->second.first->src);
			}
			
			TBGNode nj;
			for(vector<TBGNode>::iterator bit = BGNodeList.begin(); bit != BGNodeList.end(); ++bit)
			{
				if(CyPhyMLNeighbourJunction.uniqueId() == bit->Ref->uniqueId())
				{
					nj = *bit;
					break;
				}
			}
			
			outf << "bg.node(" << nj.ID << ").index, ";
			
			
			// value, ismod
			if(bondIt->TypeName == "TF")
			{
				// TF
				// value
				outf << string(CyPhyML::TF::Cast(*bondIt->Ref).ParameterValue()) << ", ";
				
				// ismod
				outf << "0);";
			}
			else
			{
				// MTF
				// value contant zero
				outf << "0, ";

				// ismod
				outf << "1);";
			}
			outf << endl;
		}
	}
	
	// Print connections
	outf << endl << "% Print bond-graph element connections" << endl;
	// Go through all the BGNodes
	for(vector<TBGNode>::iterator nodeIterator = BGNodeList.begin(); nodeIterator != BGNodeList.end(); ++nodeIterator)
	{
		// For each TBGNode go through its bonds
		for(map<int, pair<TBond*,EndPointType>>::iterator bondIterator = nodeIterator->BondMap.begin();
			bondIterator != nodeIterator->BondMap.end(); ++bondIterator)
		{
			// If this node is the destination of bond
			// OR
			// they are in different Component
			if(bondIterator->second.second == DESTINATION
				|| (string(bondIterator->second.first->src.GetParent().getPath2("/")) !=
				string(bondIterator->second.first->dst.GetParent().getPath2("/"))))
			{
				continue;
			}

			// This is the source of the bond
			TBond thisBond = *bondIterator->second.first;
			TBGNode neighbourNode;

			// Search for the TBGNode adjacent on the actual bond
			for(vector<TBGNode>::iterator neighbourNodeIterator = BGNodeList.begin();
				neighbourNodeIterator != BGNodeList.end(); ++neighbourNodeIterator)
			{
				if(thisBond.dst.uniqueId() == neighbourNodeIterator->Ref->uniqueId())
				{
					neighbourNode = *neighbourNodeIterator;
					break;
				}
			}

			// Search the number of bond at neighbours BondMap on which this node is connected
			int bondOnNeighbour = 0;
			for(map<int, pair<TBond*,EndPointType>>::iterator nBondIterator = neighbourNode.BondMap.begin();
				nBondIterator != neighbourNode.BondMap.end(); ++nBondIterator)
			{
				if(nBondIterator->second.first->src.uniqueId() == nodeIterator->Ref->uniqueId())
				{
					bondOnNeighbour = nBondIterator->first;
					break;
				}
			}
			
			// Construct line adding paths
			char buffer[33];
			int radix = 10;
			string source = nodeIterator->Name;
			source += "/";
			_itoa_s(bondIterator->first, buffer, radix);
			source += string(buffer);
			string destination = neighbourNode.Name;
			destination += "/";
			_itoa_s(bondOnNeighbour, buffer, radix);
			destination += string(buffer);
			//sfFile.AddLine(GetModifiedName(string(nodeIterator->Ref->GetParent().getPath2("/"))), source, destination);
			//sfFile.AddLine(GetModifiedName(string(nodeIterator->Ref->GetParent().getPath2("/"))), destination, source);
			sfFile.AddLine(GetObjectPath(nodeIterator->Ref->GetParent()), source, destination);
			sfFile.AddLine(GetObjectPath(nodeIterator->Ref->GetParent()), destination, source);
		}
	}
	outf << endl;


	// Add switchhandler element (buildscript determines whether its actually necessary or not)
	if(!AllJunctions.empty())
	{
		outf << "% Add switch-handler element" << endl;
		outf << "hbg_switchhandler_element('" << sfFile.szModelName << "');";
		outf << endl;
	}
}


void PrintPowerPorts(ofstream& outf)
{
  // check for inconsistancy
	for(vector<PowerChain>::iterator PowerChainIt = PowerPortChains.begin();
    PowerChainIt != PowerPortChains.end();
    ++PowerChainIt)
	{
		for(vector<CyPhyML::MgaObject>::iterator PPIterator = PowerChainIt->PowerPortVector.begin();
				PPIterator != PowerChainIt->PowerPortVector.end();
        ++PPIterator)
		{
      if (Udm::IsDerivedFrom(PPIterator->type(), DML::BGPowerPort::meta))
      {
        set<DML::Junction2PowerPort> j2pp;
        j2pp = DML::BGPowerPort::Cast(*PPIterator).srcJunction2PowerPort();
        set<DML::PhysicalPort2PowerPort> php2pp;
        php2pp = DML::BGPowerPort::Cast(*PPIterator).srcPhysicalPort2PowerPort();
        if (j2pp.size() + php2pp.size() == 0)
        {
          // does not contain any incoming connections
			    string err = "";
			    err += "PowerPort does not contain any incoming connections: ";
			    err += GMEConsole::Formatter::MakeObjectHyperlink(PPIterator->name(), *PPIterator);
          pqMessages.push(Message(err, MSG_INFO));
        }
        set<DML::PowerPort2Junction> pp2j;
        pp2j = DML::BGPowerPort::Cast(*PPIterator).dstPowerPort2Junction();
        set<DML::PowerPort2PhysicalPort> pp2php;
        pp2php = DML::BGPowerPort::Cast(*PPIterator).dstPowerPort2PhysicalPort();
        if (pp2j.size() + pp2php.size() == 0)
        {
          // does not contain any outgoing connections
 			    string err = "";
			    err += "PowerPort does not contain any outgoing connections: ";
			    err += GMEConsole::Formatter::MakeObjectHyperlink(PPIterator->name(), *PPIterator);
			    pqMessages.push(Message(err, MSG_INFO));
        }
      }
      else if (Udm::IsDerivedFrom(PPIterator->type(), DML::PowerPortType::meta))
      {
        set<DML::PowerFlow> j2pp;
        j2pp = DML::PowerPortType::Cast(*PPIterator).srcPowerFlow();
        set<DML::PowerPort2PhysicalPort> php2pp;
        php2pp = DML::PowerPortType::Cast(*PPIterator).srcPowerPort2PhysicalPort();
        if (j2pp.size() + php2pp.size() == 0)
        {
          // does not contain any incoming connections
			    string err = "";
			    err += "PowerPort does not contain any incoming connections: ";
			    err += GMEConsole::Formatter::MakeObjectHyperlink(PPIterator->name(), *PPIterator);
			    pqMessages.push(Message(err, MSG_INFO));
        }
        set<DML::PowerFlow> pp2j;
        pp2j = DML::PowerPortType::Cast(*PPIterator).dstPowerFlow();
        set<DML::PhysicalPort2PowerPort> pp2php;
        pp2php = DML::PowerPortType::Cast(*PPIterator).dstPhysicalPort2PowerPort();
        if (pp2j.size() + pp2php.size() == 0)
        {
          // does not contain any outgoing connections
 			    string err = "";
			    err += "PowerPort does not contain any outgoing connections: ";
			    err += GMEConsole::Formatter::MakeObjectHyperlink(PPIterator->name(), *PPIterator);
			    pqMessages.push(Message(err, MSG_INFO));
        }
      }
      else
      {
        // unrecognized power port

      }
    }
  }
	int ChainID = 1;
	// Iterate over all the PP chains
	for(vector<PowerChain>::iterator PowerChainIt = PowerPortChains.begin(); PowerChainIt != PowerPortChains.end(); ++PowerChainIt)
	{
		// In each chain, traverse the PP vector
		for(vector<CyPhyML::MgaObject>::iterator PPIterator = PowerChainIt->PowerPortVector.begin();
				PPIterator != PowerChainIt->PowerPortVector.end(); ++PPIterator)
		{
			// Assemble the a port-info block
			PowerPortInfo NewPPInfo;
			NewPPInfo.Ref = &(*PPIterator);
			
			long uid = PPIterator->uniqueId();
			//map<long, PowerPortInfo>::iterator itFound = PPInfos.find(PPIterator->uniqueId());
			//if (itFound != PPInfos.end())
			//{
			//	// PP is already in PPInfos
			//	int width = PPInfos[uid].ChainWidth;
			//	PowerChainIt->ChainWidthUpdate();
			//}

			// Width of the chain = #(source juncions) * #(destination junctions)
			NewPPInfo.ChainWidth = PowerChainIt->ChainWidth;
			
			if ((NewPPInfo.ChainWidth == 0) && (PowerChainIt->SourceJunctionSet.size() != 0) && (PowerChainIt->DestinationJunctionSet.size() != 0))
			{
				NewPPInfo.ChainWidth = PowerChainIt->SourceJunctionSet.size() * PowerChainIt->DestinationJunctionSet.size();
				//ASSERT(0);
			}

			// Path and parentpath
			//NewPPInfo.ParentPathWithParentName = GetSimulinkPath(GetModifiedName(PPIterator->GetParent().getPath2("/")));
			NewPPInfo.ParentPathWithParentName = GetSimulinkPath(GetObjectPath(PPIterator->GetParent()));

			//NewPPInfo.PathWithName = GetSimulinkPath(GetModifiedName(PPIterator->getPath2("/")));
			NewPPInfo.PathWithName = GetSimulinkPath(GetObjectPath(*PPIterator));

			// Create portnames for each PP
			// - a single PowerPort will map to #Chainwidth input-output pairs
			// - input-output pairs should have the same number in Simulink
			// > this is ensured by creating PowerPort-mapped Simulink ports first
			//   so SignalPorts will not ruin it

			for(int i=1; i <= NewPPInfo.ChainWidth; ++i)
			{
				char buffer[33];
				//char IDbuffer[33];
				_ltoa_s(i, buffer,10);

				// Naming convention: <PowerPortName> + "_" + *number* + "_" + [IN | OUT]
				// *number = No. in order of DFS traverse of SOURCEJUNCTIONSDESTINATIONJUNCTIONS pairs
				string SimulinkInPortName = PPIterator->name();
				SimulinkInPortName += "_";
				SimulinkInPortName += string(buffer);
				SimulinkInPortName += "_IN";

				string SimulinkOutPortName = PPIterator->name();
				SimulinkOutPortName += "_";
				SimulinkOutPortName += string(buffer);
				SimulinkOutPortName += "_OUT";
				
				// Get the structure which contains the number of already contained ports for each subsystem
				// (PP and SignalPorts merged together)
                if (ioPortNumbers.find(CyPhyML::MgaObject::Cast(PPIterator->GetParent()).uniqueId()) == ioPortNumbers.end())
				{
					// code fix needed, if we select a system
                    string err = "";
                    err += "Subsystem was not found in PowerPortMap ";
                    DML::MgaObject obj = CyPhyML::MgaObject::Cast(PPIterator->GetParent());
                    err += GMEConsole::Formatter::MakeObjectHyperlink(obj.name(), obj);
                    pqMessages.push(Message(err, MSG_ERROR));
                    continue;
				}
				PortNumbers *ContainerPortNumbers = &ioPortNumbers.find(CyPhyML::MgaObject::Cast(PPIterator->GetParent()).uniqueId())->second;
				
				// We process PowerPorts first therefore number of Inports and Outports should be equal
				// because a PP always maps to one input and one output (Effort/Flow)
				if(ContainerPortNumbers->in != ContainerPortNumbers->out)
				{
                    std::string n = PPIterator->name();
                    std::string t = PPIterator->type().name();
                    
                    string err = "";
                    err += "PowerPort: Problem with port numbering ";
                    err += GMEConsole::Formatter::MakeObjectHyperlink(PPIterator->name(), *PPIterator);
                    pqMessages.push(Message(err, MSG_ERROR));
                    return;
					//throw udm_exception("Unexpected problem occured with port-numbering");
				}
				else
				{
					// Assign Simulink portnumber
					// Could have been ContainerPortNumbers->out+1 as well! (see comment above)
					NewPPInfo.SimulinkPortsAndNames.insert(make_pair(ContainerPortNumbers->in + i, pair<string, string>(SimulinkInPortName, SimulinkOutPortName)));
				}
			}
			// uniqueId is already in PPINFOS!!!!!!!!!!!!!!!!!!!!
			// Insert the PPinfo
			typedef map<long,PowerPortInfo> PowerPortInfoMap;
			PowerPortInfoMap oldpair;
			if (PPInfos.size() == ChainID)
			{
				// ChainId is 1 based indexing
				oldpair = PPInfos[ChainID]; // save current data
				PPInfos.erase(ChainID); // delete the existing data
			}
			oldpair.insert(make_pair<long, PowerPortInfo>(PPIterator->uniqueId(), NewPPInfo)); // add new data
			PPInfos.insert(pair<int , PowerPortInfoMap>(ChainID, oldpair));
		}
		ChainID++;
	}
	
	ChainID = 1;
	char IDbuffer[33];

	for(vector<PowerChain>::iterator PowerChainIt = PowerPortChains.begin();
		PowerChainIt != PowerPortChains.end();
		++PowerChainIt, ++ChainID)
	{	
		// Calculate chainID
		_ltoa_s(ChainID, IDbuffer, 10);

		// Print some information
		int NumberOfSourceJunctions = PowerChainIt->SourceJunctionSet.size();
		int NumberOfDestinationJunctions = PowerChainIt->DestinationJunctionSet.size();

		

		if (NumberOfDestinationJunctions == 0)
		{
			// if there is no destination junction in the chain skip this power chain
			// TODO: we have to create some kind of error message

			continue;
		}

		outf << endl;
		outf << "% PowerChain source junctions: " << NumberOfSourceJunctions << endl;
		outf << "% PowerChain destination junctions: " << NumberOfDestinationJunctions << endl;
		
		

		// Traverse the chain again
		// Could have been done once, but the first loop has been moved here from another function
		for(vector<CyPhyML::MgaObject>::iterator PPIterator = PowerChainIt->PowerPortVector.begin();
			PPIterator != PowerChainIt->PowerPortVector.end(); PPIterator++)
		{
			string path = GetSimulinkPath(PPIterator->getPath2("/"));

			outf << "% PowerPort #" << PPIterator->uniqueId() << endl;
			outf << "% PowerPort path: " << path << endl;
			
			// Get PowerPort info object
			PowerPortInfo ThisInfo(PPInfos.find(ChainID)->second.find(PPIterator->uniqueId())->second);
			/*PowerPortInfo ThisInfo;
			vector<PowerPortInfo> sPPInfo;
			for (map<long, PowerPortInfo>::iterator itFound = PPInfos.begin();
				itFound != PPInfos.end();
				++itFound)
			{
				if (itFound->second.Ref->uniqueId() == PPIterator->uniqueId())
				{
					sPPInfo.push_back(itFound->second);
				}
			}*/

			/*for (vector<PowerPortInfo>::iterator itsPPInfo = sPPInfo.begin();
				itsPPInfo != sPPInfo.end();
				++itsPPInfo)
			{*/
				//ThisInfo = *itsPPInfo;
			
			if (ThisInfo.ChainWidth != (PowerChainIt->SourceJunctionSet.size() * PowerChainIt->DestinationJunctionSet.size()))
			{
				long uid = PPIterator->uniqueId();
				//int width = PPInfos[uid].ChainWidth;
				PowerChainIt->ChainWidthUpdate();
			}

            if (ThisInfo.ChainWidth == 0)
            {
                outf << "% Chain width: 0" << endl;
                outf << "% Source PP: " << PowerChainIt->PowerPortVector.begin()->getPath2("/") << endl;
                outf << "% Destination PP: " << (PowerChainIt->PowerPortVector.end()-1)->getPath2("/") << endl;
                outf << "% Chain elements: ";
                for (vector<DML::MgaObject>::iterator itt = PowerChainIt->PowerPortVector.begin();
                    itt != PowerChainIt->PowerPortVector.end();
                    ++itt)
                {
                    outf << itt->name();
                    outf << ", ";
                }
                outf << endl;
                continue;
            }

            if (ThisInfo.SimulinkPortsAndNames.size() == 0)
            {
                outf << "% Chain width: " << ThisInfo.ChainWidth << endl;
                outf << "% SimulinkPortsAndNames.size(): 0" << endl;
                outf << "% Source PP: " << PowerChainIt->PowerPortVector.begin()->getPath2("/") << endl;
                outf << "% Destination PP: " << (PowerChainIt->PowerPortVector.end()-1)->getPath2("/") << endl;
                outf << "% Chain elements: ";
                for (vector<DML::MgaObject>::iterator itt = PowerChainIt->PowerPortVector.begin();
                    itt != PowerChainIt->PowerPortVector.end();
                    ++itt)
                {
                    outf << itt->name();
                    outf << ", ";
                }
                outf << endl;
                continue;
            }

			// Create input and output ports
			for(int i=0; i < ThisInfo.ChainWidth; ++i)
			{
				// Generate input port path
				string InPortPath = ThisInfo.ParentPathWithParentName;
				InPortPath += "/"; 
				InPortPath += ThisInfo.SimulinkPortsAndNames.find(i+1)->second.first;
				InPortPath += "_";
				InPortPath += string(IDbuffer);

				// Generate output port path
				string OutPortPath = ThisInfo.ParentPathWithParentName;
				OutPortPath += "/";
				OutPortPath += ThisInfo.SimulinkPortsAndNames.find(i+1)->second.second;
				OutPortPath += "_";
				OutPortPath += string(IDbuffer);
				
				// Print MatLAB add_block functions and set position
				outf << "add_block('built-in/Inport', '" << GetModifiedName(InPortPath) << "');";
				outf << endl;
				outf << "set_param('" << GetModifiedName(InPortPath) << "', ";
				outf << "'orientation','right', 'position',";
				outf << "[10," << (i+1+ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.in)*30 << ",25,";
				outf << ((i+1+ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.in)*30)+15 << "], ";
				outf << "'Port','" << i+1+ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.in << "');";
				outf << endl;

				outf << "add_block('built-in/Outport', '" << GetModifiedName(OutPortPath) << "');";
				outf << endl;
				outf << "set_param('" << GetModifiedName(OutPortPath) << "', ";
				outf << "'orientation','right', 'position',";
				outf << "[300," << (i+1+ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.out)*30 << ",315,";
				outf << ((i+1+ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.out)*30)+15 << "], ";
				outf << "'Port','" << i+1+ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.out << "');";
				outf << endl;
			}




			// First PP in the chain
			if(PPIterator == PowerChainIt->PowerPortVector.begin())
			{
				// Add connections with source junctions
				int j = 1;
				for(set<CyPhyML::Junction>::iterator SourceJunctionIterator = PowerChainIt->SourceJunctionSet.begin();
					SourceJunctionIterator != PowerChainIt->SourceJunctionSet.end(); ++SourceJunctionIterator)
				{
					for(set<CyPhyML::Junction>::iterator DestinationJunctionIterator = PowerChainIt->DestinationJunctionSet.begin();
					DestinationJunctionIterator != PowerChainIt->DestinationJunctionSet.end(); ++DestinationJunctionIterator)
					{
						// Go through SourceJunction  DestinationJunction pairs
						
						string ConnectionPath = ThisInfo.ParentPathWithParentName;

						// Add connection from junction to PP
						string src1 = SourceJunctionIterator->name();
						src1 += "/";
						
						char buffer[33];

						// Get bond number of junction to connect PP on (get the corresponding junction from the other side)
						for(vector<TBGNode>::iterator BGNodeIt = BGNodeList.begin(); BGNodeIt != BGNodeList.end(); BGNodeIt++)
						{
							if(SourceJunctionIterator->uniqueId() == BGNodeIt->Ref->uniqueId())
							{
								for(map<int, pair<TBond*,EndPointType>>::iterator mapit = BGNodeIt->BondMap.begin();
									mapit != BGNodeIt->BondMap.end(); ++mapit)
								{
									if(mapit->second.second == SOURCE)
									{
										if(mapit->second.first->dst.uniqueId() == DestinationJunctionIterator->uniqueId())
										{
											_ltoa_s(mapit->first,buffer,10);
											break;
										}
									}
									else
									{
										if(mapit->second.first->src.uniqueId() == DestinationJunctionIterator->uniqueId())
										{
											_ltoa_s(mapit->first,buffer,10);
											break;
										}
									}
								}
								break;
							}
						}
						src1 += string(buffer);
					
						string dst1, src2;

						// PP and Junction are in the same depth
						//if(ThisInfo.ParentPathWithParentName == GetSimulinkPath(GetPathWOName(SourceJunctionIterator->getPath2("/"))))
						if(ThisInfo.ParentPathWithParentName == GetSimulinkPath(GetObjectPath(SourceJunctionIterator->GetParent())))
						{
							dst1 = ThisInfo.SimulinkPortsAndNames.find(j)->second.second;
							dst1 += "_";
							dst1 += string(IDbuffer);
							dst1 += "/1";

							src2 = ThisInfo.SimulinkPortsAndNames.find(j)->second.first;
							src2 += "_";
							src2 += string(IDbuffer);
							src2 += "/1";
						}
						else // SourceJunction is one level upper
						{
							char buffer[33];

							dst1 = CyPhyML::MgaObject::Cast(ThisInfo.Ref->GetParent()).name();
							dst1 += "/";
							_itoa_s(ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.in + 1, buffer, 10);
							dst1 += string(buffer);
							
							src2 = CyPhyML::MgaObject::Cast(ThisInfo.Ref->GetParent()).name();
							src2 += "/";
							_itoa_s(ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.out + 1, buffer, 10);
							src2 += string(buffer);

							ConnectionPath = GetPathWOName(ThisInfo.ParentPathWithParentName);
						}

						string dst2 = SourceJunctionIterator->name();
						dst2 += "/";
						dst2 += string(buffer);
						
						// Insert connections to a collection processed later
						conns.push_back(BlockConnection(ConnectionPath, src1, dst1));
						conns.push_back(BlockConnection(ConnectionPath, src2, dst2));
						++j;
					}
				}
			}
			
			// Last PP in the chain
			if(PPIterator == (PowerChainIt->PowerPortVector.end())-1) 
			{
				int j = 1;
				for(set<CyPhyML::Junction>::iterator SourceJunctionIterator = PowerChainIt->SourceJunctionSet.begin();
					SourceJunctionIterator != PowerChainIt->SourceJunctionSet.end(); ++SourceJunctionIterator)
				{
					for(set<CyPhyML::Junction>::iterator DestinationJunctionIterator = PowerChainIt->DestinationJunctionSet.begin();
					DestinationJunctionIterator != PowerChainIt->DestinationJunctionSet.end(); ++DestinationJunctionIterator)
					{
						// Go through SourceJunction  DestinationJunction pairs
						
						string ConnectionPath = ThisInfo.ParentPathWithParentName;

						// Add connection from junction to PP
						string src1 = DestinationJunctionIterator->name();
						src1 += "/";
						
						char buffer[33];

						// Get bond number of junction to connect PP on (get the corresponding junction from the other side)
						for(vector<TBGNode>::iterator BGNodeIt = BGNodeList.begin(); BGNodeIt != BGNodeList.end(); BGNodeIt++)
						{
							if(DestinationJunctionIterator->uniqueId() == BGNodeIt->Ref->uniqueId())
							{
								for(map<int, pair<TBond*,EndPointType>>::iterator mapit = BGNodeIt->BondMap.begin();
									mapit != BGNodeIt->BondMap.end(); ++mapit)
								{
									if(mapit->second.second == SOURCE)
									{
										if(mapit->second.first->dst.uniqueId() == SourceJunctionIterator->uniqueId())
										{
											_ltoa_s(mapit->first,buffer,10);
											break;
										}
									}
									else
									{
										if(mapit->second.first->src.uniqueId() == SourceJunctionIterator->uniqueId())
										{
											_ltoa_s(mapit->first,buffer,10);
											break;
										}
									}
								}
								break;
							}
						}
						src1 += string(buffer);
					
						string dst1, src2;

						// PP and Junction are in the same depth
						//if(ThisInfo.ParentPathWithParentName == GetSimulinkPath(GetPathWOName(DestinationJunctionIterator->getPath2("/"))))
						if(ThisInfo.ParentPathWithParentName == GetSimulinkPath(GetObjectPath(DestinationJunctionIterator->GetParent())))
						{
							dst1 = ThisInfo.SimulinkPortsAndNames.find(j)->second.second;
							dst1 += "_";
							dst1 += string(IDbuffer);
							dst1 += "/1";

							src2 = ThisInfo.SimulinkPortsAndNames.find(j)->second.first;
							src2 += "_";
							src2 += string(IDbuffer);
							src2 += "/1";
						}
						else // DestinationJunction is one level upper
						{
							char buffer[33];

							dst1 = CyPhyML::MgaObject::Cast(ThisInfo.Ref->GetParent()).name();
							dst1 += "/";
							_itoa_s(ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.in + 1, buffer, 10);
							dst1 += string(buffer);
							
							src2 = CyPhyML::MgaObject::Cast(ThisInfo.Ref->GetParent()).name();
							src2 += "/";
							_itoa_s(ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.out + 1, buffer, 10);
							src2 += string(buffer);

							ConnectionPath = GetPathWOName(ThisInfo.ParentPathWithParentName);
						}

						string dst2 = DestinationJunctionIterator->name();
						dst2 += "/";
						dst2 += string(buffer);
						
						// Insert connections to a collection processed later
						conns.push_back(BlockConnection(ConnectionPath, src1, dst1));
						conns.push_back(BlockConnection(ConnectionPath, src2, dst2));
						++j;
					}
				}

				ioPortNumbers.find(PPIterator->GetParent().uniqueId())->second.in += PowerChainIt->ChainWidth;
				ioPortNumbers.find(PPIterator->GetParent().uniqueId())->second.out += PowerChainIt->ChainWidth;
			}

			// Process PP to PP connections
			// if not the last PP in the chain
			if(PPIterator+1 != PowerChainIt->PowerPortVector.end())
			{
				PowerPortInfo NextInfo = PPInfos.find(ChainID)->second.find((PPIterator+1)->uniqueId())->second;
				
				if(ThisInfo.ParentPathWithParentName == NextInfo.ParentPathWithParentName)
				{
					// 1. _________
					//   |         |
					//   | PP---PP |
					//   |_________|
					//

					string ConnectionPath = ThisInfo.ParentPathWithParentName;
					
					for(int i = 1; i<=ThisInfo.ChainWidth; ++i)
					{
						string src1, dst1, src2, dst2;
						src1 = dst1 = src2 = dst2 = "";

						src1 += ThisInfo.SimulinkPortsAndNames.find(i)->second.first;
						src1 += "_";
						src1 += string(IDbuffer);
						src1 += "/1";
						dst1 += NextInfo.SimulinkPortsAndNames.find(i)->second.second;
						dst1 += "_";
						dst1 += string(IDbuffer);
						dst1 += "/1";
						conns.push_back(BlockConnection(ConnectionPath, src1, dst1));

						src2 += NextInfo.SimulinkPortsAndNames.find(i)->second.first;
						src2 += "_";
						src2 += string(IDbuffer);
						src2 += "/1";
						dst2 += ThisInfo.SimulinkPortsAndNames.find(i)->second.second;
						dst2 += "_";
						dst2 += string(IDbuffer);
						dst2 += "/1";
						conns.push_back(BlockConnection(ConnectionPath, src2, dst2));
					}

					ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.in += ThisInfo.ChainWidth;
					ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.out += ThisInfo.ChainWidth;

				}
				else if(GetPathWOName(ThisInfo.ParentPathWithParentName) == NextInfo.ParentPathWithParentName)
				{
					// 2. _________
					//   |         |
					//   |      PP-|--PP
					//   |_________|
					//	

					string ConnectionPath = NextInfo.ParentPathWithParentName;
					
					for(int i = 1; i<=ThisInfo.ChainWidth; ++i)
					{
						string src1, dst1, src2, dst2;
						src1 = dst1 = src2 = dst2 = "";

						src1 += CyPhyML::MgaObject::Cast(ThisInfo.Ref->GetParent()).name();
						src1 += "/";

						char buffer[33];
						_itoa_s(i + ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.in, buffer, 10);
						src1 += string(buffer);

						dst1 += NextInfo.SimulinkPortsAndNames.find(i)->second.second;
						dst1 += "_";
						dst1 += string(IDbuffer);
						dst1 += "/1";
						conns.push_back(BlockConnection(ConnectionPath, src1, dst1));

						src2 += NextInfo.SimulinkPortsAndNames.find(i)->second.first;
						src2 += "_";
						src2 += string(IDbuffer);
						src2 += "/1";
						
						dst2 += CyPhyML::MgaObject::Cast(ThisInfo.Ref->GetParent()).name();
						dst2 += "/";
						dst2 += string(buffer);

						conns.push_back(BlockConnection(ConnectionPath, src2, dst2));
					}
					ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.in += ThisInfo.ChainWidth;
					ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.out += ThisInfo.ChainWidth;
				}
				else if(ThisInfo.ParentPathWithParentName == GetPathWOName(NextInfo.ParentPathWithParentName))
				{
					// 3.   _________
					//     |         |
					// PP--| PP      |
					//     |_________|
					//
					
					string ConnectionPath = ThisInfo.ParentPathWithParentName;
					
					for(int i = 1; i<=ThisInfo.ChainWidth; ++i)
					{
						string src1, dst1, src2, dst2;
						src1 = dst1 = src2 = dst2 = "";

						src1 += CyPhyML::MgaObject::Cast(NextInfo.Ref->GetParent()).name();
						src1 += "/";

						char buffer[33];
						_itoa_s(i + ioPortNumbers.find(NextInfo.Ref->GetParent().uniqueId())->second.in, buffer, 10);
						src1 += string(buffer);

						dst1 += ThisInfo.SimulinkPortsAndNames.find(i)->second.second;
						dst1 += "_";
						dst1 += string(IDbuffer);
						dst1 += "/1";
						conns.push_back(BlockConnection(ConnectionPath, src1, dst1));

						src2 += ThisInfo.SimulinkPortsAndNames.find(i)->second.first;
						src2 += "_";
						src2 += string(IDbuffer);
						src2 += "/1";
						
						dst2 += CyPhyML::MgaObject::Cast(NextInfo.Ref->GetParent()).name();
						dst2 += "/";
						dst2 += string(buffer);

						conns.push_back(BlockConnection(ConnectionPath, src2, dst2));
					}
					ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.in += ThisInfo.ChainWidth;
					ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.out += ThisInfo.ChainWidth;
				}
				else if(GetPathWOName(ThisInfo.ParentPathWithParentName) == GetPathWOName(NextInfo.ParentPathWithParentName))
				{
					// 4. _________      _________
					//   |         |    |         |
					//   |      PP-|----|-PP      |
					//   |_________|    |_________|
					//

					string ConnectionPath = GetPathWOName(ThisInfo.ParentPathWithParentName);
					
					for(int i = 1; i<=ThisInfo.ChainWidth; ++i)
					{
						string src, dst;
						src = dst = "";

						char buffer[33];

						_itoa_s(i + ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.in, buffer, 10);
						src += CyPhyML::MgaObject::Cast(ThisInfo.Ref->GetParent()).name();
						src += "/";						
						src += string(buffer);
						
						_itoa_s(i + ioPortNumbers.find(NextInfo.Ref->GetParent().uniqueId())->second.in, buffer, 10);
						dst += CyPhyML::MgaObject::Cast(NextInfo.Ref->GetParent()).name();
						dst += "/";
						dst += string(buffer);
						
						// Assuming the same number for the generated IN/OUT pair for each PP
						conns.push_back(BlockConnection(ConnectionPath, src, dst));
						conns.push_back(BlockConnection(ConnectionPath, dst, src));
					}
					ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.in += ThisInfo.ChainWidth;
					ioPortNumbers.find(ThisInfo.Ref->GetParent().uniqueId())->second.out += ThisInfo.ChainWidth;
				}
				else
				{
					// ? ERROR OCCURED
				}

			}
			
		}
	}	
}