/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "stdafx.h"

#include <iostream>

#include "stdafx.h"

#include "UdmApp.h"

#include "CyPhyML.h"

#include "TBGNode.h"
#include "TBond.h"

#include "ScriptFile.h"
#include "Tools.h"
#include "PrintScript.h"
#include "Graph.h"
#include "BlockConnection.h"
#include "PortNumbers.h"
#include "PortMapping.h"

#include "HbgGenerator.h"

#include "Message.h"
#include <queue>
#include <fstream>

#define PROJECTNAME "META"

extern bool Automation;

string CUdmApp::szConnString;

extern vector<TBGNode> BGNodeList;
extern map<long,TBond> AllBonds; // contains all bonds (flat BG graph)

extern vector<BlockConnection>  conns;
extern map<long,PortNumbers>    ioPortNumbers;
extern vector<PortMapping>      pmSignal;
extern vector<CyPhyML::MgaObject> vIoPorts;

extern priority_queue<Message> pqMessages;

map<long, int> mLayout; // long = uid, int = # of object wo position

extern string EsMoLMainBlock;
extern string EsMoLBuildScript;

extern bool bUseESMoL;
extern bool bMDAO;
extern bool bSilentMode;
extern map<string, string> MDAOInput;
extern set<string> MDAOParameter;
extern map<string, string> MDAOOutput;
extern set<string> MDAODataStorage;
extern set<string> SimulinkSystemNames; // Referenced simulink block/system names

extern bool bParametricExp;

extern set<Udm::Object> SelectedObj;

extern set<DML::MgaObject> esmolExclude;

string StopTime = "100.0";

string ConfigurationID = "C0000";

set<CyPhyML::Metric> MetricIDs;

ScriptFile::ScriptFile()
{
	OutputDirectory = "";
	this->szModelName = "";
	this->nScope = -1;
	this->InitMatLabConfig();
	this->m_ResizeSubsystems.clear();
}

ScriptFile::ScriptFile(string &oszModelName)
{
	OutputDirectory = "";
	this->szModelName = oszModelName;
	this->nScope = -1;
	this->InitMatLabConfig();
	this->m_ResizeSubsystems.clear();
}

void ScriptFile::InitMatLabConfig()
{
	map< int, string> mConfig;
	
	// CyPhyML::AlternativeRepresentationContainer configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "Magenta"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("AlternativeRepresentationContainer", mConfig));

	// CyPhyML::DeviceConfigurationFolder configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "LightBlue"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("DeviceConfigurationFolder", mConfig));

	// CyPhyML::DeviceConfiguration configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "Blue"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("DeviceConfiguration", mConfig));

	// CyPhyML::Testing configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "LightBlue"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("Testing", mConfig));

	// CyPhyML::TestBench configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "Orange"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("TestBench", mConfig));

	// CyPhyML::TestComponent configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "Orange"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("TestComponent", mConfig));

	// CyPhyML::DesignContainer configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "Yellow"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("DesignContainer", mConfig));

	// CyPhyML::Components configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "Green"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("Components", mConfig));

	// CyPhyML::ComponentAssemblies configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "Orange"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("ComponentAssemblies", mConfig));

	// CyPhyML::Component configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "Green"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("Component", mConfig));

	// CyPhyML::CyberComponent configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "Cyan"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("CyberComponent", mConfig));

	// CyPhyML::ComponentAssembly configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "Orange"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("ComponentAssembly", mConfig));

	// CyPhyML::Alternative configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("Alternative", mConfig));

	// CyPhyML::Mandatory configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("Mandatory", mConfig));

	// CyPhyML::Optional configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("Optional", mConfig));

	// CyPhyML::ElectricalSignalPort configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "20"));
	mConfig.insert(make_pair(M_HEIGHT, "20"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "LightBlue"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "!NOT_IN_USE!"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("ElectricalSignalPort", mConfig));

	// CyPhyML::PhysicalComponent configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "120"));
	mConfig.insert(make_pair(M_HEIGHT, "80"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "Orange"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("PhysicalComponent", mConfig));



    // CyPhyML::BondGraph configuration
    mConfig.clear();
    mConfig.insert(make_pair(M_WIDTH, "120"));
    mConfig.insert(make_pair(M_HEIGHT, "80"));
    mConfig.insert(make_pair(M_FOREGROUND, "Black"));
    mConfig.insert(make_pair(M_BACKGROUND, "Orange"));
    mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Subsystem"));
    mConfig.insert(make_pair(M_AUTORESIZE, "True"));

    this->MatLabConfig.insert(make_pair("BondGraph", mConfig));

	// elements inside the physical component
	// CyPhyML::InSignal configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "20"));
	mConfig.insert(make_pair(M_HEIGHT, "20"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Inport"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("InSignal", mConfig));

	// CyPhyML::OutSignal configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "20"));
	mConfig.insert(make_pair(M_HEIGHT, "20"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/Outport"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("OutSignal", mConfig));

	// CyPhyML::LocalSignal configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "45"));
	mConfig.insert(make_pair(M_HEIGHT, "25"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "emf/AtomicSubsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("LocalSignal", mConfig));

	// CyPhyML::MonitorFlow configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "25"));
	mConfig.insert(make_pair(M_HEIGHT, "30"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "simulink/Sinks/Scope"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("MonitorFlow", mConfig));

	// CyPhyML::SignalMonitor
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "25"));
	mConfig.insert(make_pair(M_HEIGHT, "30"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "simulink/Sinks/Scope"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("SignalMonitor", mConfig));

	// CyPhyML::MonitorEffort configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "25"));
	mConfig.insert(make_pair(M_HEIGHT, "30"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "simulink/Sinks/Scope"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("MonitorEffort", mConfig));

	// CyPhyML::MonitorDisplacement configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "25"));
	mConfig.insert(make_pair(M_HEIGHT, "30"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "emf/MonitorDisplacement"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("MonitorDisplacement", mConfig));

	// CyPhyML::MonitorMomentum configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "25"));
	mConfig.insert(make_pair(M_HEIGHT, "30"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "emf/MonitorMomentum"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("MonitorMomentum", mConfig));

	// CyPhyML::Df configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "45"));
	mConfig.insert(make_pair(M_HEIGHT, "25"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "emf/DetectFlow"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("Df", mConfig));

	// CyPhyML::De configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "45"));
	mConfig.insert(make_pair(M_HEIGHT, "25"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "emf/DetectEffort"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("De", mConfig));

	// CyPhyML::Dq configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "45"));
	mConfig.insert(make_pair(M_HEIGHT, "25"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "emf/DetectDisplacement"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("Dq", mConfig));

	// CyPhyML::Dp configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "45"));
	mConfig.insert(make_pair(M_HEIGHT, "25"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "emf/DetectMomentum"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("Dp", mConfig));

	// CyPhyML::Actuation configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "45"));
	mConfig.insert(make_pair(M_HEIGHT, "25"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "emf/AtomicSubsystem"));
	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("Actuation", mConfig));

	// CyPhyML::ControlFunction configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "80"));
	mConfig.insert(make_pair(M_HEIGHT, "60"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "White"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "NULL")); // we are not using this
 	mConfig.insert(make_pair(M_AUTORESIZE, "False"));

	this->MatLabConfig.insert(make_pair("ControlFunction", mConfig));

	// CyPhyML::SimulinkModel configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "200"));
	mConfig.insert(make_pair(M_HEIGHT, "150"));
	mConfig.insert(make_pair(M_FOREGROUND, "LightBlue"));
	mConfig.insert(make_pair(M_BACKGROUND, "Gray"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "NULL")); // we are not using this
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("SimulinkModel", mConfig));

	// CyPhyML::Parameter configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "150"));
	mConfig.insert(make_pair(M_HEIGHT, "30"));
	mConfig.insert(make_pair(M_FOREGROUND, "LightBlue"));
	mConfig.insert(make_pair(M_BACKGROUND, "Yellow"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/constant")); // we are not using this
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("Parameter", mConfig));

	// CyPhyML::CyberParameter configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "150"));
	mConfig.insert(make_pair(M_HEIGHT, "30"));
	mConfig.insert(make_pair(M_FOREGROUND, "LightBlue"));
	mConfig.insert(make_pair(M_BACKGROUND, "Yellow"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/constant")); // we are not using this
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("CyberParameter", mConfig));

	// CyPhyML::Property configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "150"));
	mConfig.insert(make_pair(M_HEIGHT, "30"));
	mConfig.insert(make_pair(M_FOREGROUND, "LightBlue"));
	mConfig.insert(make_pair(M_BACKGROUND, "Yellow"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/constant")); // we are not using this
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("Property", mConfig));

	// CyPhyML::CADProperty configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "150"));
	mConfig.insert(make_pair(M_HEIGHT, "30"));
	mConfig.insert(make_pair(M_FOREGROUND, "LightBlue"));
	mConfig.insert(make_pair(M_BACKGROUND, "Yellow"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/constant")); // we are not using this
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("CADProperty", mConfig));

	// CyPhyML::Metric configuration
	mConfig.clear();
	mConfig.insert(make_pair(M_WIDTH, "150"));
	mConfig.insert(make_pair(M_HEIGHT, "30"));
	mConfig.insert(make_pair(M_FOREGROUND, "Black"));
	mConfig.insert(make_pair(M_BACKGROUND, "LightBlue"));
	mConfig.insert(make_pair(M_BLOCKTYPE, "built-in/ToWorkspace")); // we are not using this
	mConfig.insert(make_pair(M_AUTORESIZE, "True"));

	this->MatLabConfig.insert(make_pair("Metric", mConfig));
}


void ScriptFile::PrintGraphViz()
{
	if (bParametricExp)
	{
		return;
	}
	// create graphviz file in 'dot' format
	ofstream graphViz;
	string nodePrefix = "node_";
	//string szGraphVizFileName = "";
	szGraphVizFileName = this->szModelName;
	szGraphVizFileName += ".graph";
	szGraphVizFileName = OutputDirectory + szGraphVizFileName;
	remove(szGraphVizFileName.c_str());
	graphViz.open(szGraphVizFileName); // open

	graphViz << "digraph BG {" << endl;

	graphViz << "labelloc=\"t\"";
	graphViz << "label=\"Flat Bond Graph of project: " << this->szModelName << "\"";

	graphViz << "rankdir=LR" << endl;
	// set up the default shape and color of nodes
	graphViz << "node [shape=box, color=blue]" << endl;
	//graphViz << "node1 [style=filled]" << endl;
	//graphViz << "node2 [style=filled, fillcolor=red]" << endl;
	//graphViz << "node0 -> node1 -> node2" << endl;

	Udm::Object parent = Udm::null;
	set<Udm_VS10::Object> parents;
	set<TBGNode> sBGNodeList;
	for (vector<TBGNode>::iterator it = BGNodeList.begin();
		it != BGNodeList.end();
		++it)
	{
		sBGNodeList.insert(*it);
		if (parents.find(it->Ref->GetParent().GetParent()) == parents.end())
		{
			parents.insert(it->Ref->GetParent().GetParent());
		}
	}
	set<Udm_VS10::Object>::iterator itParent = parents.begin();
	//parent = parents.begin();
	while (!sBGNodeList.empty() && itParent != parents.end())
	{
		graphViz << "subgraph cluster_";
		graphViz << itParent->uniqueId();
		graphViz << " {" << endl;
		graphViz << "style=filled;" << endl;
		graphViz << "color=gray90;" << endl;
		// print out all BondGraphNodes (Junctions and Elements)
		for (vector<TBGNode>::iterator it = BGNodeList.begin();
			it != BGNodeList.end();
			++it)
		{
			if (*itParent == it->Ref->GetParent().GetParent())
			{
#ifdef DEBUG
				graphViz << GetModifiedName(it->Ref->getPath2("_"));
#else
				graphViz << nodePrefix << it->Ref->uniqueId();
#endif
				graphViz << " [label=\"";
				graphViz << GetModifiedName(it->Name);
				graphViz << "\\n" << it->Ref->type().name();
				string conditions = "";
				if (Udm::IsDerivedFrom(it->Ref->type(), DML::Junction::meta))
				{
					DML::Junction junction = DML::Junction::Cast(*(it->Ref));
					if (string(junction.OnCondition()) != "" ||
							string(junction.OffCondition()) != "")
					{
						conditions = "\\nInitialState:";
						conditions += string(junction.InitialState());
						conditions += "\\nON:";
						conditions += string(junction.OnCondition());
						conditions += "\\nOFF:";
						conditions += string(junction.OffCondition());
						conditions += "\", color=orange";
					}
				}
				if (conditions != "")
				{
					graphViz << conditions;
				}
				else
				{
					graphViz << "\"";
				}
				graphViz <<"];";
				graphViz << endl;
				sBGNodeList.erase(*it);
			}
		}
		graphViz << "fontcolor=seagreen" << endl;
		graphViz << "label=\"" << GetModifiedName(Udm_VS10::Object::Cast(*itParent)) << "\"" << endl;
		graphViz << "}" << endl;
		itParent++;
	}

	// print out all Connections (Bonds) between the elements
	for (map<long,TBond>::iterator it = AllBonds.begin();
		it != AllBonds.end();
		++it)
	{
#ifdef DEBUG
		graphViz << GetModifiedName(it->second.src.getPath2("_"));
#else
		graphViz << nodePrefix << it->second.src.uniqueId();
#endif
		graphViz << " -> ";

#ifdef DEBUG
		graphViz << GetModifiedName(it->second.dst.getPath2("_"));
#else
		graphViz << nodePrefix << it->second.dst.uniqueId();
#endif
		graphViz << " [arrowhead=\"lvee\", label=\"e" << it->second.bondID << ", f" << it->second.bondID << "\"];";
		graphViz << endl;
	}

	// close the graph
	graphViz << "}";
	graphViz.close();

	ofstream graphVizCmd;
	//string szGraphVizCmdFileName = "";
	szGraphVizCmdFileName = this->szModelName;
	szGraphVizCmdFileName += "CreatePNG.cmd";
	szGraphVizCmdFileName = OutputDirectory + szGraphVizCmdFileName;
	remove(szGraphVizCmdFileName.c_str());
	graphVizCmd.open(szGraphVizCmdFileName); // open

	HKEY hkMeta = 0;
	if (RegOpenKeyExA(HKEY_LOCAL_MACHINE, "Software\\META", 0, KEY_READ, &hkMeta) == ERROR_SUCCESS)
	{
		TCHAR data[MAX_PATH];
		DWORD size = sizeof(data);
		if (RegQueryValueEx(hkMeta, "META_DOT_PATH", NULL, NULL, (LPBYTE)data, &size) == ERROR_SUCCESS)
		{
			data[size] = _T('\0');
			
			// Add escaping if needed
			if (data[0] != '"')
				graphVizCmd << '"';
			
			graphVizCmd << data;
			
			// Add escaping if needed
			if (data[size-1] != '"')
				graphVizCmd << '"';
			graphVizCmd << "\\";
		}
		RegCloseKey(hkMeta);
	}

	graphVizCmd << "dot.exe -Tpng ";
	if (SelectedObj.size() == 1)
	{
		szGraphVizPNGFileName = OutputDirectory + GetModifiedName(Udm::Object::Cast(*(SelectedObj.begin()))) + ".png";
		graphVizCmd << "-o\"" << szGraphVizPNGFileName <<"\"";
	}
	else
	{
		szGraphVizPNGFileName = OutputDirectory + this->szModelName + ".png";
		graphVizCmd << "-o\"" << szGraphVizPNGFileName <<"\"";
	}
	graphVizCmd << " ";
	graphVizCmd << "\"" << szGraphVizFileName << "\""; // this->szModelName << ".graph";
	graphVizCmd.close();
}

void ScriptFile::PrintValueRanges(set<DML::ValueFlowTarget> &oTargets)
{
	if (bParametricExp)
	{
		return;
	}
	ofstream valueRanges;
	string szValueRangesFileName = "";
	szValueRangesFileName += this->szModelName;
	szValueRangesFileName += "_value_ranges.m";
	szValueRangesFileName = OutputDirectory + szValueRangesFileName;
	remove(szValueRangesFileName.c_str());
	valueRanges.open(szValueRangesFileName); // open

	valueRanges << "% ranges config" << endl;
	valueRanges << endl;
	
	unsigned int idx = 0;

	for (set<DML::ValueFlowTarget>::iterator it = oTargets.begin();
		it != oTargets.end();
		++it)
	{
		valueRanges << "%% idx: " << idx++ << endl;
		valueRanges << "% Path: " << GetParentPath(Udm::Object::Cast(*it)) << endl;
		valueRanges << "% Name: " << GetModifiedName(Udm::Object::Cast(*it)) << endl;
		if (it->type() == DML::Parameter::meta)
		{
			valueRanges << "% Value: " << DML::Parameter::Cast(*it).Value() << endl;
			valueRanges << "% Default value: " << DML::Parameter::Cast(*it).DefaultValue() << endl;
			valueRanges << "% Range: " << DML::Parameter::Cast(*it).Range() << endl;
		}
		else if (it->type() == DML::Property::meta)
		{
			valueRanges << "% Value: " << DML::Property::Cast(*it).Value() << endl;
			valueRanges << "% Default value: " << DML::Property::Cast(*it).DefaultValue() << endl;
			//valueRanges << "% Range: " << DML::Property::Cast(*it).Range() << endl;
		}
		//else if (it->type() == DML::CADProperty::meta)
		//{
		//	valueRanges << "% Value: " << DML::CADProperty::Cast(*it).Value() << endl;
		//	valueRanges << "% Default value: " << DML::CADProperty::Cast(*it).DefaultValue() << endl;
		//	//valueRanges << "% Range: " << DML::Property::Cast(*it).Range() << endl;
		//}
		valueRanges << endl;
	}

	valueRanges.close();
}

void ScriptFile::Open()
{
	// check dir or create
	CreateFolder(OutputDirectory);
	if (bParametricExp)
	{
		return;
	}
	if (szModelName != "")
	{

		// szModelName is specified
		string szBuildScript;														// buildscript filename
		szBuildScript = szModelName;										// model name
		szBuildScript += "_buildscript";								// buildscript postfix
		this->szScriptWOExt = szBuildScript;						// without extension
		szBuildScript += ".m";													// extension
		szBuildScript = OutputDirectory + szBuildScript;
		remove(szBuildScript.c_str());									// delete file if exists
		this->script_file_.open(szBuildScript.c_str()); // create the new file

		string szDataStruct;															 // datastruct filename
		szDataStruct = szModelName;												 // model name
		szDataStruct += "_datastruct";										 // datastruct postfix
		this->szDataStructWOExt = szDataStruct;						 // without extension
		szDataStruct += ".m";															 // extension
		szDataStruct = OutputDirectory + szDataStruct;
		remove(szDataStruct.c_str());											 // delete file if exists
		this->datastruct_file_.open(szDataStruct.c_str()); // create the new file

		string szRegisterLibraries;
		szRegisterLibraries = OutputDirectory + "register" PROJECTNAME "Libraries.m";
		ofstream ofsRegisterLibraries;
		ofsRegisterLibraries.open(szRegisterLibraries);
		ofsRegisterLibraries <<
"try\n"
"META_MATLAB_DIR = winqueryreg('HKEY_LOCAL_MACHINE', 'Software\\" PROJECTNAME "', '" PROJECTNAME "_MATLAB_PATH');\n"
"addpath(genpath(strcat(META_MATLAB_DIR, '\\HBG-matlab-library')));\n"
"addpath(genpath(strcat(META_MATLAB_DIR, '\\MDAO-matlab-library')));\n"
"addpath(genpath(strcat(META_MATLAB_DIR, '\\Scenario-matlab-library')));\n"
"catch ME\n"
"disp('Warning: couldn''t locate " PROJECTNAME " installation path');\n"
"rethrow(ME);\n"
"end\n";
		if (!ofsRegisterLibraries.good())
			throw std::runtime_error(string("Error: couldn't open ") + szRegisterLibraries + " for writing");
	}
	else
	{
		// szModelName is not specified
		ASSERT(0);
		string szErrorText;
		szErrorText = "szModelName is not specified in file: ";
		szErrorText += __FILE__;
		exception e(szErrorText.c_str());
		throw e; // throw an exception
	}
}

void ScriptFile::Close()
{
	if (this->script_file_.is_open())
	{
		this->script_file_.close();
	}
	if (this->datastruct_file_.is_open())
	{
		this->datastruct_file_.close();
	}
}

void ScriptFile::PrintHeader()
{
	if (bParametricExp)
	{
		return;
	}
	m_ResizeSubsystems.clear();
	// TODO: reformat this region
	// Matlab script file header start
	this->script_file_ << "% MatLab/Simulink buildscript" << endl;
	this->script_file_ << "% Generated by the META interpreter" << endl;


	// Add user signature if possible
	char acUserName[100];
	DWORD nUserName = sizeof(acUserName);
	if (GetUserName(acUserName, &nUserName))
	{
		this->script_file_ << "% Model was generated by user: " << acUserName << endl;
	}

	// Add timestamp and connection string to header
	time_t t;
	t=time(NULL);
	this->script_file_ << "% Time generated: " << ctime(&t);
	this->script_file_ << "% " << CUdmApp::szConnString;
	this->script_file_ << endl << endl << endl;

	// Clear MatLab variables, clear screen
	this->script_file_ << "% Cleanup MatLab variables and screen" << endl;
	this->script_file_ << "clear all;" << endl;
	
	this->script_file_ << "clc;" << endl << endl;

	this->script_file_ << "disp('Loading libraries...')" << endl;
	this->script_file_ << "% Add matlab libraries to Matlab path" << endl;
	this->script_file_ << "register" PROJECTNAME "Libraries;" << endl;

	this->script_file_ << "disp('Start model building...')" << endl;

	this->script_file_ << "global fixedJuncList;" << endl;


	// Prompt installation warning
	this->script_file_ << "% Prompt installation warning" << endl;
	this->script_file_ << "q = sprintf(['Before you click the Continue button, please make sure that ";
	this->script_file_ << "you have run once the installHBGSimulink.m script\\n', ..." << endl;
	this->script_file_ << "'(The script is usually located at the HBG_MatLab-library directory)\\n', ..." << endl;
	this->script_file_ << "'If the " << this->szModelName << ".mdl system exists, it will be deleted.\\n', ..." << endl;
	this->script_file_ << "'Do you want to continue generating the model?']);" << endl;
	if (bSilentMode)
	{
		this->script_file_ << "result = 1;" << endl;
	}
	else
	{
		this->script_file_ << "result = 1;" << endl;
		//this->script_file_ << "result = menu(q, 'Continue', 'Cancel');" << endl << endl;
	}
	this->script_file_ << "if result == 1" << endl;

	this->script_file_ << "tic; % start timer" << endl;

	this->script_file_ << "disp('Creating new system.')" << endl;
	// Delete if Simulink model is present with same name
	this->script_file_ << "hbg_newsystem('" << this->szModelName << "');" << endl;

	if (bUseESMoL)
	{
		script_file_ << EsMoLBuildScript << endl;

		this->script_file_ << "load_system('";
		this->script_file_ << EsMoLMainBlock.substr(0, EsMoLMainBlock.find_first_of('/'));
		this->script_file_ << "');";


		this->script_file_ << endl;
		this->script_file_ << "% Type: ";
		this->script_file_ << EsMoLMainBlock;
		this->script_file_ << endl;

		this->script_file_ << "add_block('";
		this->script_file_ << EsMoLMainBlock;
		this->script_file_ << "', '";
		this->script_file_ << this->szModelName << "/MainEsMoLBlock";
		this->script_file_ << "');";
		this->script_file_ << endl;
	}

	this->script_file_ << "disp('Load simulink library.')" << endl;
	this->script_file_ << "load_system('simulink');" << endl;
	// Open simulink library
	this->script_file_ << "%simulink('open');" << endl << endl;

	// Invoke datasctructurescript to enter BG structure to MatLAB variables
	this->script_file_ << endl << "% Invoke datastructure-script to store BG structure in MatLab variables" << endl;

	// Check if datascructure exists, and invoke the script and run SCAP if yes
	this->script_file_ << "if(exist('" << this->szDataStructWOExt << "', 'file'))" << endl;
	this->script_file_ << "\t" << this->szDataStructWOExt << ";" << endl;
	this->script_file_ << "\t" << "SCAP(1)" << endl;
	this->script_file_ << "end" << endl;

	// Set up the generated Simulunk system, to clear the bond graph on init
	this->script_file_ << "sys = '" << this->szModelName << "';" << endl;
	this->script_file_ << "% If all junctions are fixed" << endl;
	this->script_file_ << "if isempty(find(fixedJuncList == 0,1))" << endl;
	this->script_file_ << "  set_param(sys, 'InitFcn', char('" << this->szDataStructWOExt << "'));" << endl;
	this->script_file_ << "else" << endl;
	this->script_file_ << "% If any unfixed junction exists" << endl;
	this->script_file_ << "  set_param(sys, 'InitFcn', char('" << this->szDataStructWOExt << "','SCAP'));" << endl;
	this->script_file_ << "end" << endl << endl;
	this->script_file_ << "load_system('emf');" << endl << endl;
	this->script_file_ << "disp('Creating subsystems.')" << endl;
}

void ScriptFile::PrintFooter()
{
	if (bParametricExp)
	{
		return;
	}
	// TODO: reformat this region
	// Matlab footer start
	this->script_file_ << endl;

	this->script_file_ << "set_param('";
	this->script_file_ << this->szModelName;
	this->script_file_ << "', 'StopTime', '";
  this->script_file_ << StopTime << "');" << endl;

	if (bMDAO ||
		MDAOInput.empty() == false ||
		MDAOOutput.empty() == false)
	{
		// TODO: $$$What happens if we have switched junctions??? SCAP and causality update should be compiled as well
		// overwrite the init and stop functions
		this->script_file_ << "set_param('";
		this->script_file_ << this->szModelName;
		this->script_file_ << "', 'InitFcn', sprintf('";
		this->script_file_ << "MDAO_Init\\n";
    // TODO: test SCAP
		this->script_file_ << this->szModelName << "_datastruct\\n";
		this->script_file_ << "SCAP(1)'), 'StopFcn', sprintf('PostProc(0);\\n%SaveSimulationResultsInJSON;'), "
			"'PreLoadFcn', 'register" PROJECTNAME "Libraries;');";
		this->script_file_ << endl;

	}

	this->script_file_ << "% Init parameters" << endl;
	this->script_file_ << "MDAO_Init;" << endl;
	this->PrintMDAO(); // into a separate file

	if (bMDAO)
	{
		this->script_file_ << "%% MDAO region " << endl;
		//this->script_file_ << "% Init MDAO parameters" << endl;
		//this->script_file_ << "MDAO_Init;" << endl;

		this->script_file_ << "% Add annotations" << endl;

		this->script_file_ << "add_block('built-in/Note','";
		this->script_file_ << this->szModelName;
		this->script_file_ << "/Generated for MDAO process', ";
		this->script_file_ << "'Position', [300, 250], ";
		this->script_file_ << "'BackgroundColor', 'yellow', ";
		this->script_file_ << "'FontSize', '20', ";
		this->script_file_ << "'FontWeight', 'bold');";
		this->script_file_ << endl;

		this->script_file_ << "% Add data storage elements" << endl;

		int i = 0;


		// TODO: $$$replace this with a generic one
		this->script_file_ << "% These lines hard coded BEGIN ";
		this->script_file_ << endl;

		this->script_file_ <<"set_param('";
		this->script_file_ << this->szModelName << "', 'Solver', 'ode45', 'MaxStep', '0.1', 'Location', [530 100 1000 400]);";
		this->script_file_ << endl;
		this->script_file_ << "% These lines hard coded END ";
		this->script_file_ << endl;

		this->script_file_ << "PostLoadFcn = get_param('";
		this->script_file_ << this->szModelName;
		this->script_file_ << "','PostLoadFcn');";
		this->script_file_ << endl;
		this->script_file_ << "eval(PostLoadFcn);";
		this->script_file_ << endl;

		// bulid the executable
		if (!bSilentMode)
		{
			this->script_file_ << "% ";
		}
		else
		{
			this->script_file_ << "disp('Start building the executable.')" << endl;
		}
		this->script_file_ << "build";
		this->script_file_ << endl;
	}

	if ((this->sScopes.size() > 0) && (this->nScope != 0))
	{
		this->script_file_ << "% Open scopes" << endl;
	}
	// open the exact number of scopes if nScope is > 0, otherwise open all scopes

	int numScopes = sScopes.size();
	int maxColumn = 4;
	int maxRow = 2;
	int idColumn = 0;
	int idRow = 0;
	int id = 0;

	for (set<CyPhyML::ExtendedElements>::iterator itsScopes = this->sScopes.begin();
		(itsScopes != this->sScopes.end()) && (this->nScope != 0);
			 ++itsScopes, --this->nScope)
	{
		int mod = id % (maxColumn * maxRow);
		idRow = mod / maxColumn;
		idColumn = (mod - idRow * maxColumn);
		id = id + 1;

		if (bSilentMode)
		{
			this->script_file_ << "%";
		}
		this->script_file_ << "set_param('";
		this->script_file_ << GetObjectPath(Udm_VS10::Object::Cast(*itsScopes));
		this->script_file_ << "', 'Location', [";
		this->script_file_ << "10+(350+5)*" << idColumn;
		this->script_file_ << " ";
		this->script_file_ << "55+(300+10)*" << idRow;
		this->script_file_ << " ";
		this->script_file_ << "350+(350+5)*" << idColumn;
		this->script_file_ << " ";
		this->script_file_ << "300+(300+10)*" << idRow;
		this->script_file_ << "]);";
		this->script_file_ << endl;

		// additional scope settings

		this->script_file_ << "set_param('";
		this->script_file_ << GetObjectPath(Udm_VS10::Object::Cast(*itsScopes));
		this->script_file_ << "', 'SaveToWorkspace', 'on');";
		this->script_file_ << endl;

		this->script_file_ << "set_param('";
		this->script_file_ << GetObjectPath(Udm_VS10::Object::Cast(*itsScopes));

		string variableName = UdmGme::UdmId2GmeId(itsScopes->uniqueId());
		tr1::regex rx("-");
		string fmt = "_";
		variableName = string(tr1::regex_replace(variableName, rx, fmt));
		this->script_file_ << "', 'SaveName', '" << variableName << "');";
		this->script_file_ << endl;


		if (bSilentMode)
		{
			this->script_file_ << "%";
		}
		this->script_file_ << "open_system('";
		this->script_file_ << GetObjectPath(Udm_VS10::Object::Cast(*itsScopes));
		this->script_file_ << "');";
		this->script_file_ << endl;
	}
	this->script_file_ << endl;


	this->script_file_ << "disp('Save new system.')" << endl;
	this->script_file_ << endl << "%% Finalize the system" << endl;
	this->script_file_ << "save_system('" << this->szModelName << ".mdl');" << endl << endl;
	this->script_file_ << "close_system('" << this->szModelName << ".mdl');" << endl << endl;

	this->script_file_ << "% Open system" << endl;
	if (bSilentMode)
	{
		this->script_file_ << "%";
	}
	this->script_file_ << this->szModelName << ";" << endl << endl;


	this->script_file_ << "disp('Model has been generated successfully');" << endl << endl;

	this->script_file_ << "toc; % Display elapsed time" << endl;

	this->script_file_ << "% When user clicks cancel on the installation prompt" << endl;

	if (bSilentMode &&
		Automation == false)
	{
		this->script_file_ << "exit" << endl;
	}

	this->script_file_ << "else" << endl; 

	this->script_file_ << "  disp('Model has not been generated.');" << endl;
	this->script_file_ << "end" << endl;
	// Matlab footer end
}

void ScriptFile::PrintMDAO()
{
	if (bParametricExp)
	{
		return;
	}
	ofstream mdaofile;
	string szMDAOInitFileName = "";
	szMDAOInitFileName = "MDAO_Init.m";
	szMDAOInitFileName = OutputDirectory + szMDAOInitFileName;
	remove(szMDAOInitFileName.c_str());
	mdaofile.open(szMDAOInitFileName.c_str()); // open

	mdaofile << "%% META data" << endl;
	mdaofile << "% configuration data" << endl;
	mdaofile << "global ConfigurationID;" << endl;
	mdaofile << "ConfigurationID = '" << ConfigurationID << "';" << endl;

	mdaofile << "% Metrics" << endl;
	string keys = "{";
	string values = "{";
	
	mdaofile << "global MetricIDs;" << endl;
	mdaofile << "MetricIDs = containers.Map( ..." << endl;
	for (set<CyPhyML::Metric>::const_iterator it = MetricIDs.begin();
		it != MetricIDs.end();
		++it)
	{
		keys +=	"'";
		keys += GetModifiedName(Udm::Object::Cast(*it));
		keys += "'";

		values +=	"'";
		values += GetRegNodeValueByName(*it, REGNODE_METRIC_ID);
		values += "'";

		++it;
		if (it != MetricIDs.end())
		{
			keys +=	", ";
			values +=	", ";
		}
		--it;
	}
	if (MetricIDs.size() == 0)
	{
		keys += "'null key'";
		values += "'null value'";
	}

	keys +=	"}";
	values +=	"}";

	mdaofile << keys << ", ..." << endl;
	mdaofile << values << ");" << endl;

	if (MetricIDs.size() == 0)
	{
		mdaofile << "remove(MetricIDs, 'null key');" << endl;
	}

	//MetricIDs
	mdaofile << "%% simulation config" << endl;
	mdaofile << "global StopTime;" << endl;
	mdaofile << "StopTime = 500.0;" << endl;
	mdaofile << "%% Generated code region" << endl;
	mdaofile << "global sysName;" << endl;
	mdaofile << "sysName = '";
	mdaofile << this->szModelName;
	mdaofile << "';" << endl;

	mdaofile << "global PostFunction;" << endl;
	mdaofile << "PostFunction = {'PostProc'};" << endl;
	mdaofile << "global FunctionsToCompile;" << endl;
	mdaofile << "FunctionsToCompile = {'PostProc.m', ..." << endl;
	mdaofile << "                      '" << this->szModelName << "_datastruct.m'};" << endl;


	mdaofile << "% Response names" << endl;
	mdaofile << "global responseNames;" << endl;

	mdaofile << "responseNames = {";
	for (map<string, string>::iterator it = MDAOOutput.begin();
		it != MDAOOutput.end();
		++it)
	{
		if (it != MDAOOutput.begin())
		{
			mdaofile << "                 ";
		}
		mdaofile << "'";
		mdaofile << it->first;
		if (++it != MDAOOutput.end())
		{
			// if not the last element
			mdaofile << "'; ..." << endl;
		}
		else
		{
			mdaofile << "'";
		}
		--it;
	}
	mdaofile << "};" << endl;
	mdaofile << endl;

	mdaofile << "% Unit names" << endl;
	mdaofile << "global unitNames;" << endl;

	mdaofile << "unitNames = {";
	for (map<string, string>::iterator it = MDAOOutput.begin();
		it != MDAOOutput.end();
		++it)
	{
		if (it != MDAOOutput.begin())
		{
			mdaofile << "                 ";
		}
		mdaofile << "'";
		mdaofile << it->second;
		if (++it != MDAOOutput.end())
		{
			// if not the last element
			mdaofile << "'; ..." << endl;
		}
		else
		{
			mdaofile << "'";
		}
		--it;
	}
	mdaofile << "};" << endl;
	mdaofile << endl;


	mdaofile << "% Parameter specs BEGIN" << endl;
	mdaofile << "global Parameters;" << endl;

	mdaofile << "Parameters = {";
	for (map<string, string>::iterator it = MDAOInput.begin();
		it != MDAOInput.end();
		++it)
	{
		if (it != MDAOInput.begin())
		{
			mdaofile << "              ";
		}
		mdaofile << "'";
		mdaofile << it->first;
		if (++it != MDAOInput.end())
		{
			// if not the last element
			mdaofile << "'; ..." << endl;
		}
		else
		{
			mdaofile << "'";
		}
		--it;
	}
	mdaofile << "};" << endl;
	
	int iDefaultValue = 1;

	unsigned int MaxParamLength = 0;

	mdaofile << endl;
	for (map<string, string>::iterator it = MDAOInput.begin();
		it != MDAOInput.end();
		++it)
	{
		// *it parameter name
		mdaofile << "global " << it->first << ";" << endl;
		mdaofile << it->first << " = " << it->second << ";" << endl;
		if ((it->first).length() > MaxParamLength)
		{
			MaxParamLength = (it->first).length();
		}
	}

	mdaofile << "% Parameter specs END" << endl;

	mdaofile.close();

	ofstream mdaofileWrapper;
	szMDAOInitFileName = "";
	szMDAOInitFileName += this->szModelName;
	szMDAOInitFileName += ".fileWrapper";
	szMDAOInitFileName = OutputDirectory + szMDAOInitFileName;
	remove(szMDAOInitFileName.c_str());
	mdaofileWrapper.open(szMDAOInitFileName.c_str()); // open

	mdaofileWrapper << "#" << endl;
	mdaofileWrapper << "# @author: -Generated File- " << endl;
	mdaofileWrapper << "# @version: 0.0" << endl;
	mdaofileWrapper << "# @description: " << this->szModelName << " wrapper" << endl;
	mdaofileWrapper << "#" << endl;
	mdaofileWrapper << endl;
	mdaofileWrapper << "RunCommands" << endl;
	mdaofileWrapper << "{" << endl;
	mdaofileWrapper << "  generate inputFile" << endl;
	mdaofileWrapper << "  run \"" << this->szModelName << "_Wrapper.exe " << this->szModelName <<".in " << this->szModelName << ".out\"" << endl;
	mdaofileWrapper << "  parse outputFile" << endl;
	mdaofileWrapper << "}" << endl;
	mdaofileWrapper << endl;

	mdaofileWrapper << "RowFieldInputFile inputFile" << endl;
	mdaofileWrapper << "{" << endl;
	mdaofileWrapper << "  templateFile:        testInputSample.txt" << endl;
	mdaofileWrapper << "  fileToGenerate:      " << this->szModelName << ".in" << endl;
	mdaofileWrapper << endl;

	int i = 1;
	for (map<string, string>::iterator it = MDAOInput.begin();
		it != MDAOInput.end();
		++it)
	{
		mdaofileWrapper << "  variable: ";
		mdaofileWrapper.width(MaxParamLength);
		mdaofileWrapper << it->first;             // name
		mdaofileWrapper.width();
		mdaofileWrapper << " double";       // type
		mdaofileWrapper << " 2 ";           // row
		mdaofileWrapper << i++;             // parameter column
		mdaofileWrapper << endl;
	}
	mdaofileWrapper << "}" << endl;
	mdaofileWrapper << endl;

	mdaofileWrapper << "RowFieldOutputFile outputFile" << endl;
	mdaofileWrapper << "{" << endl;
	mdaofileWrapper << "  fileToParse:        " << this->szModelName << ".out" << endl;
	mdaofileWrapper << endl;

	i = 1;
	for (map<string, string>::iterator it = MDAOOutput.begin();
		it != MDAOOutput.end();
		++it)
	{
		mdaofileWrapper << "  variable: ";
		mdaofileWrapper.width(MaxParamLength);
		mdaofileWrapper << it->first;       // name
		mdaofileWrapper.width();
		mdaofileWrapper << " double";       // type
		mdaofileWrapper << " 2 ";           // row
		mdaofileWrapper << i++;             // parameter column
		mdaofileWrapper << endl;
	}

	mdaofileWrapper << "}" << endl;
	mdaofileWrapper.close();
}

void ScriptFile::PrintDataStructure()
{
	if (bParametricExp)
	{
		return;
	}
	GenerateDataStructureOutput(this->datastruct_file_);
}

void ScriptFile::PrintBGElements()
{
	if (bParametricExp)
	{
		return;
	}
	this->script_file_ << "disp('Creating Bond Graph elements.')" << endl;
	PrintBondElements(this->script_file_);
}

void ScriptFile::PrintPowerPort()
{
	if (bParametricExp)
	{
		return;
	}
	this->script_file_ << "disp('Creating power ports.')" << endl;
	PrintPowerPorts(this->script_file_);
}

void ScriptFile::PrintIoPort()
{
	if (bParametricExp)
	{
		return;
	}
	string szType;

	for (vector<CyPhyML::MgaObject>::iterator itIoPorts = vIoPorts.begin();
			 itIoPorts != vIoPorts.end();
			 ++itIoPorts)
	{
		if ((itIoPorts->type() == DML::InSignal::meta) ||
			(itIoPorts->type() == DML::OutSignal::meta))
		{
			szType = itIoPorts->type().name();
			this->AddBlock(*itIoPorts, this->MatLabConfig[szType][M_BLOCKTYPE]);
		}
		else if (itIoPorts->type() == DML::ElectricalSignalPort::meta)
		{
			if (IO_INPUT == CheckType(DML::ElectricalSignalPort::Cast(*itIoPorts)))
			{
				this->AddBlock(*itIoPorts, this->MatLabConfig["InSignal"][M_BLOCKTYPE]);
			}
			else if (IO_OUTPUT == CheckType(DML::ElectricalSignalPort::Cast(*itIoPorts)))
			{
				this->AddBlock(*itIoPorts, this->MatLabConfig["OutSignal"][M_BLOCKTYPE]);
			}
			else
			{
				// unrecognized port type
			}

		}
	}
}

void ScriptFile::ResizeSubsystems()
{
	if (bParametricExp)
	{
		return;
	}
	this->script_file_ << "disp('Resizing subsystems.')" << endl;
	// resize subsystems
	for (set<string>::iterator i (this->m_ResizeSubsystems.begin());
			 i != this->m_ResizeSubsystems.end();
			 ++i)
	{
		this->script_file_ << "ResizeSubsystems('";
		this->script_file_ << *i;
		this->script_file_ << "');";
		this->script_file_ << endl;
	}
}

void ScriptFile::PrintConnections()
{
	if (bParametricExp)
	{
		return;
	}
	this->script_file_ << "disp('Creating connections.')" << endl;
	this->script_file_ << endl;
	this->script_file_ << "% Print all connections BEGIN";
	this->script_file_ << endl;

	for (vector<BlockConnection>::iterator itConnection = conns.begin();
			 itConnection != conns.end();
			 ++itConnection)
	{
		this->AddLine( itConnection->path,
									 itConnection->src,
									 itConnection->dst);
	}

	this->script_file_ << "% Print all connections END";
	this->script_file_ << endl;
}


void ScriptFile::AddBlock(Udm::Object &oUdm,
													string szKind)
{
	if (bParametricExp)
	{
		return;
	}
	string szType;
	szType = oUdm.type().name();

	this->script_file_ << endl;
	this->script_file_ << "% Type: ";
	this->script_file_ << szType;
	this->script_file_ << endl;

	this->script_file_ << "add_block('";
	this->script_file_ << szKind;
	this->script_file_ << "', '";
	this->script_file_ << GetObjectPath(oUdm);
	this->script_file_ << "');";
	this->script_file_ << endl;

	if (Udm::IsDerivedFrom(oUdm.type(),CyPhyML::MgaObject::meta))
	{
		this->SetParam(CyPhyML::MgaObject::Cast(oUdm));
	}
	else
	{
		this->SetParam(oUdm);
	}

	if (this->MatLabConfig[oUdm.type().name()][M_AUTORESIZE] == "True")
	{
		// store subsystems to resize
		this->m_ResizeSubsystems.insert(GetObjectPath(oUdm));
	}
}


void ScriptFile::SetParam(CyPhyML::MgaObject &oMga)
{
	if (bParametricExp)
	{
		return;
	}
	int iX;
	int iY;
	int iPortNumber;

	string szType;
	string szParentPath;
	string szParentName;

	string szWidth = "";
	string szHeight = "";
	string szBackColor = "";

	PortNumbers* ioPortNumber;

	szType = oMga.type().name();
	szParentPath = GetParentPath(oMga.GetParent());
	szParentName = GetModifiedName(oMga.GetParent());

	ioPortNumber = &(ioPortNumbers.find(oMga.GetParent().uniqueId())->second);

	if ((szType == "Component") ||
			(szType == "CyberComponent") ||
			(szType == "Alternative") ||
			(szType == "Mandatory") ||
			(szType == "Optional") ||
			(szType == "DeviceConfiguration") ||
			(szType == "ComponentAssembly"))
	{
		// its parent can be a folder
		map<long, int>::iterator itmLayout;

		string szParentType;
		szParentType = oMga.GetParent().type().name();

		iX = 0;
		iY = 100; // default Y offset needed if switchhandelr is generated

		// should be extended for all folder types in general
		if ((szParentType == "Components") ||
				(szParentType == "DesignSpace") ||
				(szParentType == "DeviceConfigurationFolder") ||
				(szParentType == "DeviceConfiguration") ||
				(szParentType == "ComponentAssemblies"))
		//if (GetObjectXPosition(oMga) == -1)
		{
			// this element's parent is a folder
			itmLayout = mLayout.find(oMga.GetParent().uniqueId());
			if (itmLayout == mLayout.end())
			{
				// Insert the parent element into the container
				mLayout.insert(make_pair(oMga.GetParent().uniqueId(), 1));
			}
			else
			{
				string szWidth;
				int iWidth;
				szWidth = this->MatLabConfig[oMga.type().name()][M_WIDTH];
				iWidth = atoi(szWidth.c_str());
				// Increase the # of elements in the container
				iX = itmLayout->second*(iWidth+20)+15;
				itmLayout->second++;
			}
		}
		else if ((szParentType == "Alternative") ||
						 (szParentType == "Mandatory") ||
						 (szParentType == "Optional") ||
						 (szParentType == "ComponentAssembly"))
		{
			if (GetObjectXPosition(oMga) != -1)
			{
				// we have valid coordinates
				// its parent not a folder
				iX = GetObjectXPosition(oMga);
				iY = GetObjectYPosition(oMga);
			}
			else
			{
				itmLayout = mLayout.find(oMga.GetParent().uniqueId());
				if (itmLayout == mLayout.end())
				{
					// Insert the parent element into the container
					mLayout.insert(make_pair(oMga.GetParent().uniqueId(), 1));
				}
				else
				{
					string szWidth;
					int iWidth;
					szWidth = this->MatLabConfig[oMga.type().name()][M_WIDTH];
					iWidth = atoi(szWidth.c_str());
					// Increase the # of elements in the container
					iX = itmLayout->second*(iWidth+20)+15;
					itmLayout->second++;
				}
			}
		}
	}
	else
	{
		// its parent cannot be a folder
		iX = GetObjectXPosition(oMga);
		iY = GetObjectYPosition(oMga);
	}
	
	szWidth = this->MatLabConfig[oMga.type().name()][M_WIDTH];
	szHeight = this->MatLabConfig[oMga.type().name()][M_HEIGHT];
	szBackColor = this->MatLabConfig[oMga.type().name()][M_BACKGROUND];


	if ((szWidth == "") ||
			(szHeight == "") ||
			(szBackColor == ""))
	{
		// this object's configuration parameters is not defined in MatLabConfig
		szType;
		szWidth = "40";
		szHeight = "40";
		szBackColor = "Red";
		//ASSERT(0);
	}

	string additionalParameters = "";
	if (oMga.type() == DML::Parameter::meta )
	{
		additionalParameters = ", ";
		additionalParameters += " 'value' ";
		additionalParameters += ", '";
		if (oMga.GetParent().type() == DML::TestBench::meta)
		{
			additionalParameters += GetModifiedName(oMga);
		}
		else
		{
			if (DML::Parameter::Cast(oMga).Value() == "")
			{
				additionalParameters += "0";
			}
			else
			{
				additionalParameters += DML::Parameter::Cast(oMga).Value();
			}
		}
		additionalParameters += "'";

		set<DML::ValueFlow2SignalPort> s = DML::Parameter::Cast(oMga).dstValueFlow2SignalPort();
		if (DML::Parameter::Cast(oMga).Value() == "" && !s.empty())
		{
			string err = "";
			err += "Parameter Value is missing: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(oMga.name(), oMga);
			pqMessages.push(Message(err, MSG_ERROR));
		}

	}
	else if (oMga.type() == DML::Property::meta)
	{
		additionalParameters = ", ";
		additionalParameters += " 'value' ";
		additionalParameters += ", '";
		if (oMga.GetParent().type() == DML::TestBench::meta)
		{
			additionalParameters += GetModifiedName(oMga);
		}
		else
		{
			if (DML::Property::Cast(oMga).Value() == "")
			{
				additionalParameters += "0";
			}
			else
			{
				additionalParameters += DML::Property::Cast(oMga).Value();
			}
		}
		additionalParameters += "'";

		set<DML::ValueFlow2SignalPort> s = DML::Property::Cast(oMga).dstValueFlow2SignalPort();
		if (DML::Property::Cast(oMga).Value() == "" && !s.empty())
		{
			string err = "";
			err += "Property Value is missing: ";
			err += GMEConsole::Formatter::MakeObjectHyperlink(oMga.name(), oMga);
			pqMessages.push(Message(err, MSG_ERROR));
		}
	}
	//else if (oMga.type() == DML::CADProperty::meta)
	//{
	//	additionalParameters = ", ";
	//	additionalParameters += " 'value' ";
	//	additionalParameters += ", '";
	//	if (oMga.GetParent().type() == DML::TestBench::meta)
	//	{
	//		additionalParameters += GetModifiedName(oMga);
	//	}
	//	else
	//	{
	//		if (DML::CADProperty::Cast(oMga).Value() == "")
	//		{
	//			additionalParameters += "0";
	//		}
	//		else
	//		{
	//			additionalParameters += DML::CADProperty::Cast(oMga).Value();
	//		}
	//	}
	//	additionalParameters += "'";

	//	set<DML::ValueFlow2SignalPort> s = DML::CADProperty::Cast(oMga).dstValueFlow2SignalPort();
	//	if (DML::CADProperty::Cast(oMga).Value() == "" && !s.empty())
	//	{
	//		string err = "";
	//		err += "CAD Property Value is missing: ";
	//		err += GMEConsole::Formatter::MakeObjectHyperlink(oMga.name(), oMga);
	//		pqMessages.push(Message(err, MSG_ERROR));
	//	}
	//}
	else if (oMga.type() == DML::Metric::meta)
	{
		additionalParameters = ", ";
		additionalParameters += " 'VariableName' ";
		additionalParameters += ", '";
		additionalParameters += GetModifiedName(oMga);
		additionalParameters += "'";
		additionalParameters += ", 'MaxDataPoints' ";
		additionalParameters += ", 'inf'";
		additionalParameters += ", 'SaveFormat' ";
		additionalParameters += ", 'Structure With Time'";
	}


	this->script_file_ << "set_param('";
	this->script_file_ << GetObjectPath(oMga);
	this->script_file_ << "', 'orientation', 'right', ";
	this->script_file_ << "'position', [";
	this->script_file_ << iX;
	this->script_file_ << ", ";
	this->script_file_ << iY;
	this->script_file_ << ", ";
	this->script_file_ << iX << "+" << szWidth;
	this->script_file_ << ", ";
	this->script_file_ << iY << "+" << szHeight;
	this->script_file_ << "], 'BackgroundColor', '";
	this->script_file_ << szBackColor;
	
	if (szType == "InSignal")
	{
		this->script_file_ << "', 'Port', '";
		ioPortNumber->signalIN++;
		iPortNumber = ioPortNumbers.find(oMga.GetParent().uniqueId())->second.signalIN;
		iPortNumber += ioPortNumbers.find(oMga.GetParent().uniqueId())->second.in;
		// Add PortMapping
		PortMapping NewPortMapping(oMga.uniqueId(),
															 iPortNumber,
															 szParentPath,
															 szParentName,
															 oMga.name());
		pmSignal.push_back(NewPortMapping);

		this->script_file_ << iPortNumber;
	}
	else if (szType == "OutSignal")
	{
		this->script_file_ << "', 'Port', '";
		ioPortNumber->signalOUT++;
		iPortNumber = ioPortNumbers.find(oMga.GetParent().uniqueId())->second.signalOUT;
		iPortNumber += ioPortNumbers.find(oMga.GetParent().uniqueId())->second.out;
		// Add PortMapping
		PortMapping NewPortMapping(oMga.uniqueId(),
															 iPortNumber,
															 szParentPath,
															 szParentName,
															 oMga.name());
		pmSignal.push_back(NewPortMapping);

		this->script_file_ << iPortNumber;
	}
	else if (oMga.type() == DML::ElectricalSignalPort::meta)
	{
		if (IO_INPUT == CheckType(DML::ElectricalSignalPort::Cast(oMga)))
		{
			this->script_file_ << "', 'Port', '";
			ioPortNumber->signalIN++;
			iPortNumber = ioPortNumbers.find(oMga.GetParent().uniqueId())->second.signalIN;
			iPortNumber += ioPortNumbers.find(oMga.GetParent().uniqueId())->second.in;
			// Add PortMapping
			PortMapping NewPortMapping(oMga.uniqueId(),
																 iPortNumber,
																 szParentPath,
																 szParentName,
																 oMga.name());
			pmSignal.push_back(NewPortMapping);

			this->script_file_ << iPortNumber;
		}
		else if (IO_OUTPUT == CheckType(DML::ElectricalSignalPort::Cast(oMga)))
		{
			this->script_file_ << "', 'Port', '";
			ioPortNumber->signalOUT++;
			iPortNumber = ioPortNumbers.find(oMga.GetParent().uniqueId())->second.signalOUT;
			iPortNumber += ioPortNumbers.find(oMga.GetParent().uniqueId())->second.out;
			// Add PortMapping
			PortMapping NewPortMapping(oMga.uniqueId(),
																 iPortNumber,
																 szParentPath,
																 szParentName,
																 oMga.name());
			pmSignal.push_back(NewPortMapping);

			this->script_file_ << iPortNumber;
		}
	}
	this->script_file_ << "'";
	this->script_file_ << additionalParameters;
	this->script_file_ << ");";
	this->script_file_ << endl;
}


void ScriptFile::SetParam(Udm::Object &oUdm)
{
	if (bParametricExp)
	{
		return;
	}
	int iX;
	int iY;
	map<long, int>::iterator itmLayout;

	// default value
	iX = 0;
	iY = 100;

	itmLayout = mLayout.find(oUdm.GetParent().uniqueId());
	if (itmLayout == mLayout.end())
	{
		// Insert the parent element into the container
		mLayout.insert(make_pair(oUdm.GetParent().uniqueId(), 1));
	}
	else
	{
		string szWidth;
		int iWidth;
		szWidth = this->MatLabConfig[oUdm.type().name()][M_WIDTH];
		iWidth = atoi(szWidth.c_str());
		// Increase the # of elements in the container
		iX = itmLayout->second*(iWidth+20)+15;
		itmLayout->second++;
	}

	string additionalParameters = "";

	this->script_file_ << "set_param('";
	this->script_file_ << GetObjectPath(oUdm);
	this->script_file_ << "', 'orientation', 'right', ";
	this->script_file_ << "'position', [";
	this->script_file_ << iX;
	this->script_file_ << ", ";
	this->script_file_ << iY;
	this->script_file_ << ", ";
	this->script_file_ << iX << "+" << this->MatLabConfig[oUdm.type().name()][M_WIDTH];
	this->script_file_ << ", ";
	this->script_file_ << iY << "+" << this->MatLabConfig[oUdm.type().name()][M_HEIGHT];
	this->script_file_ << "], 'BackgroundColor', '";
	this->script_file_ << this->MatLabConfig[oUdm.type().name()][M_BACKGROUND];
	this->script_file_ << "'";
	this->script_file_ << additionalParameters;
	this->script_file_ << ");";
	this->script_file_ << endl;
}

void ScriptFile::AddLine(const string cszBlock,
												 const string cszSource,
												 const string cszDestination)
{
	if (bParametricExp)
	{
		return;
	}
	this->script_file_ << "add_line('";
	this->script_file_ << cszBlock;
	this->script_file_ << "', '";
	this->script_file_ << GetModifiedName(cszSource);
	this->script_file_ << "', '";
	this->script_file_ << GetModifiedName(cszDestination);
	this->script_file_ << "', 'autorouting', 'on');";
	this->script_file_ << endl;
}

void ScriptFile::operator <<(Udm::Object &oUdm)
{
	if (bParametricExp)
	{
		return;
	}
	if (oUdm.type() != DML::SimpleFormula::meta &&
		oUdm.type() != DML::CADParameter::meta &&
		oUdm.type() != DML::CustomFormula::meta)
	{
		AddBlock(oUdm, this->MatLabConfig[oUdm.type().name()][M_BLOCKTYPE]);
	}
}

void ScriptFile::operator <<(CyPhyML::ExtendedElements &oExtElements)
{
	if (bParametricExp)
	{
		return;
	}
	string szType;
	string szParentPath;
	string szParentName;
	string szObjectPath;

	szType = oExtElements.type().name();
	szParentPath = GetParentPath(oExtElements.GetParent());
	szParentName = GetModifiedName(oExtElements.GetParent());
	szObjectPath = GetParentPath(oExtElements);

	if (szType == "Constant")
	{
		AddBlock(oExtElements, "built-in/Constant"); // TODO: we do not handle this yet.
	}
	else if (szType == "Parameter" || szType == "CyberParameter")
	{
		// TODO: we do not handle this yet.
	}
	else if (szType == "SimulinkModel")
	{
		DML::SimulinkModel smThis;
		smThis = DML::SimulinkModel::Cast(oExtElements);

		this->script_file_ << endl;
		this->script_file_ << "% Load required system";
		this->script_file_ << endl;
		this->script_file_ << "load_system('";
		string smName = smThis.SimulinkModelName();
		this->script_file_ << smName.substr(0,smName.find_last_of('/'));
		this->script_file_ << "');";
		this->script_file_ << endl;

		if (smName == "")
		{
			string _console = "Model name is not defined for simulink model: ";
			_console += GMEConsole::Formatter::MakeObjectHyperlink(oExtElements.name(), oExtElements);
			pqMessages.push(Message(_console,MSG_ERROR));
		}

		AddBlock(oExtElements, smThis.SimulinkModelName());

		this->script_file_ << "ImportFunctionCode('";
		this->script_file_ << smName.substr(0,smName.find_last_of('/'));
		this->script_file_ << "','";
		this->script_file_ << smName.substr(smName.find_last_of('/') + 1, smName.length()-smName.substr(0,smName.find_last_of('/')).length());
		this->script_file_ << "','";
		this->script_file_ << this->szModelName;
		this->script_file_ << "','%%% BEGIN ','%%% END ',' CODE %%%','PostLoadFcn'";
		this->script_file_ << ");";
		this->script_file_ << endl;

		if (SimulinkSystemNames.find(smThis.SimulinkModelName()) == SimulinkSystemNames.end())
		{
			// include each names only once
			SimulinkSystemNames.insert(smThis.SimulinkModelName());
		}

		int i = 1; // port number
		set<DML::InSignal> isig = smThis.InSignal_kind_children();	
		for(set<DML::InSignal>::iterator it = isig.begin(); it != isig.end(); ++it)
		{
			PortMapping pmTmp(it->uniqueId(),i++, GetParentPath(smThis), GetModifiedName(smThis) , it->name());
			pmSignal.push_back(pmTmp);
		}
		i = 1;
		set<DML::OutSignal> osig = smThis.OutSignal_kind_children();	
		for(set<DML::OutSignal>::iterator it = osig.begin(); it != osig.end(); ++it)
		{
			PortMapping pmTmp(it->uniqueId(),i++, GetParentPath(smThis), GetModifiedName(smThis) , it->name());
			pmSignal.push_back(pmTmp);
		}

	}
	else if (szType == "ControlFunction")
	{
		CyPhyML::ControlFunction cfThis;
		cfThis = CyPhyML::ControlFunction::Cast(oExtElements);

		if (string(cfThis.Code()).empty())
		{
			string _console = "Warning: Empty Code field in: ";
			_console += GMEConsole::Formatter::MakeObjectHyperlink(cfThis.name(),cfThis);
			pqMessages.push(Message(_console,MSG_WARNING));
		}
		else
		{
			// Print this control function script
			//this->script_file_ << "load_system('emf');" << endl;
			this->script_file_ << "add_block('emf/ControlFunction', '" << szObjectPath << "/" << GetModifiedName(cfThis.name()) << "');" << endl;

			this->script_file_ << "S = sfroot;" << endl;
			this->script_file_ << "A = S.find('-isa','Simulink.SubSystem');" << endl;
			this->script_file_ << "B = A.find('-isa','Stateflow.EMChart');" << endl;
			string code = cfThis.Code();
			string header = code.substr(0, code.find_first_of(')') + 1);

			code = code.substr(0,code.find("#Modelica"));

			this->script_file_ << "for i=1 : 1 : B.size(1)" << endl;
			this->script_file_ << " if strcmp(B(i).Path, '" << szObjectPath << "/" << GetModifiedName(cfThis.name()) << "')" << endl;
			this->script_file_ << "  B(i).Script = '" << header << "';" << endl;
			this->script_file_ << "  G = sprintf('";
			for (int it = code.find_first_of(')')+1; it != code.size(); it++ )
			{
				if (code.at(it) != '\n')
				{
					if (code.at(it) == '%')
					{
						this->script_file_ << "%";
					}

					if (code.at(it) == '\'')
					{
						this->script_file_ << "'";
					}
					
					this->script_file_ << code.at(it);
				}
				else
				{
					this->script_file_ << "\\" << 'n';
				}
			}
			this->script_file_ << "');" << endl;
			this->script_file_ << "  B(i).Script = G;" << endl;
			this->script_file_ << "  break;" << endl;
			this->script_file_ << " end" << endl;
			this->script_file_ << "end" << endl << endl;

			// Set up the parameters
			this->SetParam(cfThis);
			
			// Add into the port mapping
			vector<string> out;
			vector<string> in;
			vector<unsigned int> out_index;
			vector<unsigned int> in_index;
			GetVariables(out, out_index, in, in_index, cfThis);
			for (unsigned int i = 0; i < out.size(); i++)
			{
				PortMapping pmTmp( oExtElements.uniqueId(),
													 out_index[i],
													 szObjectPath,
													 GetModifiedName(cfThis.name()),
													 out[i]);
				pmSignal.push_back(pmTmp);
			}									
			for (unsigned int i = 0; i < in.size(); i++)
			{
				PortMapping pmTmp( oExtElements.uniqueId(),
													 in_index[i],
													 szObjectPath,
													 GetModifiedName(cfThis.name()),
													 in[i]);
				pmSignal.push_back(pmTmp);
			}
		}
	}
	else if (szType == "Dp")
	{
		this->AddBlock(oExtElements, this->MatLabConfig[szType][M_BLOCKTYPE]);
		PortMapping pmTmp(oExtElements.uniqueId(),
											1,
											szParentPath,
											szParentName,
											GetModifiedName(oExtElements.name()));
		pmSignal.push_back(pmTmp);
	}
	else if (szType == "De")
	{
		this->AddBlock(oExtElements, this->MatLabConfig[szType][M_BLOCKTYPE]);
		PortMapping pmTmp(oExtElements.uniqueId(),
											1,
											szParentPath,
											szParentName,
											GetModifiedName(oExtElements.name()));
		pmSignal.push_back(pmTmp);
	}
	else if (szType == "Df")
	{
		this->AddBlock(oExtElements, this->MatLabConfig[szType][M_BLOCKTYPE]);
		PortMapping pmTmp(oExtElements.uniqueId(),
											1,
											szParentPath,
											szParentName,
											GetModifiedName(oExtElements.name()));
		pmSignal.push_back(pmTmp);
	}
	else if (szType == "Dq")
	{
		this->AddBlock(oExtElements, this->MatLabConfig[szType][M_BLOCKTYPE]);
		PortMapping pmTmp(oExtElements.uniqueId(),
											1,
											szParentPath,
											szParentName,
											GetModifiedName(oExtElements.name()));
		pmSignal.push_back(pmTmp);
	}
	else if (szType == "Actuation")
	{
		this->AddBlock(oExtElements, this->MatLabConfig[szType][M_BLOCKTYPE]);
		PortMapping pmTmp(oExtElements.uniqueId(),
											1,
											szParentPath,
											szParentName,
											GetModifiedName(oExtElements.name()));
		pmSignal.push_back(pmTmp);
	}
	else if (szType == "Modulation")
	{
		CyPhyML::Modulation o;
		o = CyPhyML::Modulation::Cast(oExtElements);
		// Inward connection process order: Junction, Constant, Signal
		this->script_file_ << endl;
		this->script_file_ << "% Insert modulation element " << endl;
		
		// params: ssName, sys, pos, funcSpec, numPorts

		// ssName, sys
		this->script_file_ << "hbg_modulation_element('" << GetModifiedName(o.name()) << "', '" << szObjectPath << "', ";
		
		// pos
		this->script_file_ << "[" << GetObjectXPosition(o) << "," << GetObjectYPosition(o) << "," << GetObjectXPosition(o) << "+35," << GetObjectYPosition(o) << "+35], ";

		// funcSpec
		int numPorts = 0;
		string funcSpec = string(o.Expression());

		set<CyPhyML::Junction2Modulation> junc2modSet = o.srcJunction2Modulation();
		for(set<CyPhyML::Junction2Modulation>::iterator junc2modIterator = junc2modSet.begin();
			junc2modIterator != junc2modSet.end(); ++junc2modIterator)
		{
			++numPorts;
			string neighbourJuncName = CyPhyML::Junction(junc2modIterator->srcJunction2Modulation_end()).name();

			string inputStr = "u(";
			char buff[33];
			_itoa_s(numPorts, buff, 10);
			inputStr += string(buff);
			inputStr += ")";

			// Replace input junctions' names
			size_t found = funcSpec.find(neighbourJuncName);
			while(found != string::npos)
			{
				funcSpec.replace(found, neighbourJuncName.size(), inputStr);
				found = funcSpec.find(neighbourJuncName);
			}
		}

		/*set<CyPhyML::Constant2Modulation> const2modSet = o.srcConstant2Modulation();	
		for(set<CyPhyML::Constant2Modulation>::iterator const2modIterator = const2modSet.begin();
			const2modIterator != const2modSet.end(); ++const2modIterator)
		{
			++numPorts;
			string constantName = CyPhyML::Constant(const2modIterator->srcConstant2Modulation_end()).name();

			string inputStr = "u(";
			char buff[33];
			itoa(numPorts, buff, 10);
			inputStr += string(buff);
			inputStr += ")";

			// Replace input constants' names
			size_t found = funcSpec.find(constantName);
			while(found != string::npos)
			{
				funcSpec.replace(found, constantName.size(), inputStr);
				found = funcSpec.find(constantName);
			}
		}*/

		// Print result
		this->script_file_ << "'" << funcSpec << "', ";

		// numPorts
		this->script_file_ << numPorts << ");" << endl;
		}
	else if (szType == "Switching")
	{
		CyPhyML::Switching o;
		o = CyPhyML::Switching::Cast(oExtElements);
		// TODO: THIS!!!
		// Inward connection process order: Junction, Constant, Signal
		this->script_file_ << endl << "% Insert switching element " << endl;
		
		// params: ssName, sys, pos, funcSpec, numPorts

		// ssName, sys
		this->script_file_ << "hbg_switching_element('" << GetModifiedName(o.name()) << "', '" << szObjectPath << "', ";
		
		// pos
		this->script_file_ << "[" << GetObjectXPosition(o) << "," << GetObjectYPosition(o) << "," << GetObjectXPosition(o) << "+35," << GetObjectYPosition(o) << "+35], ";

		// funcSpec
		int numPorts = 0;
		string funcSpec = string(o.SwExpression());

		set<CyPhyML::Junction2Switching> junc2swSet = o.srcJunction2Switching();
		for(set<CyPhyML::Junction2Switching>::iterator junc2swIterator = junc2swSet.begin();
			junc2swIterator != junc2swSet.end(); ++junc2swIterator)
		{
			++numPorts;
			string neighbourJuncName = CyPhyML::Junction(junc2swIterator->srcJunction2Switching_end()).name();

			string inputStr = "u(";
			char buff[33];
			_itoa_s(numPorts, buff, 10);
			inputStr += string(buff);
			inputStr += ")";

			// Replace input junctions' names
			size_t found = funcSpec.find(neighbourJuncName);
			while(found != string::npos)
			{
				funcSpec.replace(found, neighbourJuncName.size(), inputStr);
				found = funcSpec.find(neighbourJuncName);
			}
		}

		/*set<CyPhyML::Constant2Switching> const2swSet = o.srcConstant2Switching();
		for(set<CyPhyML::Constant2Switching>::iterator const2swIterator = const2swSet.begin();
			const2swIterator != const2swSet.end(); ++const2swIterator)
		{
			++numPorts;
			string constantName = CyPhyML::Constant(const2swIterator->srcConstant2Switching_end()).name();

			string inputStr = "u(";
			char buff[33];
			itoa(numPorts, buff, 10);
			inputStr += string(buff);
			inputStr += ")";

			// Replace input junctions' names
			size_t found = funcSpec.find(constantName);
			while(found != string::npos)
			{
				funcSpec.replace(found, constantName.size(), inputStr);
				found = funcSpec.find(constantName);
			}
		}
		*/
		// Print result
		this->script_file_ << "'" << funcSpec << "', ";

		// numPorts
		this->script_file_ << numPorts << ");" << endl;
	}
	else if (szType == "SignalMonitor")
	{
		this->AddBlock(oExtElements, this->MatLabConfig[szType][M_BLOCKTYPE]);

		this->script_file_ << "set_param('";
		this->script_file_ << GetObjectPath(oExtElements);
		this->script_file_ << "', ";
		this->script_file_ << "'LimitDataPoints', 'off";
		this->script_file_ << "');";
		this->script_file_ << endl;

		this->sScopes.insert(oExtElements);

	}
	else if (szType == "MonitorEffort")
	{
		this->AddBlock(oExtElements, this->MatLabConfig[szType][M_BLOCKTYPE]);

		this->script_file_ << "set_param('";
		this->script_file_ << GetObjectPath(oExtElements);
		this->script_file_ << "', ";
		this->script_file_ << "'LimitDataPoints', 'off";
		this->script_file_ << "');";
		this->script_file_ << endl;

		this->sScopes.insert(oExtElements);

	}
	else if (szType == "MonitorFlow")
	{
		this->AddBlock(oExtElements, this->MatLabConfig[szType][M_BLOCKTYPE]);

		this->script_file_ << "set_param('";
		this->script_file_ << GetObjectPath(oExtElements);
		this->script_file_ << "', ";
		this->script_file_ << "'LimitDataPoints', 'off";
		this->script_file_ << "');";
		this->script_file_ << endl;

		this->sScopes.insert(oExtElements);

	}
	else if (szType == "MonitorDisplacement")
	{
		this->AddBlock(oExtElements, this->MatLabConfig[szType][M_BLOCKTYPE]);

		this->script_file_ << "set_param('";
		this->script_file_ << GetObjectPath(oExtElements);
		this->script_file_ << "/Scope', ";
		this->script_file_ << "'LimitDataPoints', 'off";
		this->script_file_ << "');";
		this->script_file_ << endl;

		//string szScopeName;
		//szScopeName = GetObjectPath(oExtElements);
		//szScopeName += "/";
		//szScopeName += GetModifiedName(oExtElements);
		this->sScopes.insert(oExtElements);

		// Replace scope's name in the new block
		this->script_file_ << "set_param('";
		this->script_file_ << GetObjectPath(oExtElements);
		this->script_file_ << "/Scope', ";
		this->script_file_ << "'name', '";
		this->script_file_ << GetModifiedName(oExtElements);
		this->script_file_ << "');";
		this->script_file_ << endl;
	}
	else if (szType == "MonitorMomentum")
	{
		this->AddBlock(oExtElements, this->MatLabConfig[szType][M_BLOCKTYPE]);

		this->script_file_ << "set_param('";
		this->script_file_ << GetObjectPath(oExtElements);
		this->script_file_ << "/Scope', ";
		this->script_file_ << "'LimitDataPoints', 'off";
		this->script_file_ << "');";
		this->script_file_ << endl;

		//string szScopeName;
		//szScopeName = GetObjectPath(oExtElements);
		//szScopeName += "/";
		//szScopeName += GetModifiedName(oExtElements);
		this->sScopes.insert(oExtElements);

		// Replace scope's name in the new block
		this->script_file_ << "set_param('";
		this->script_file_ << GetObjectPath(oExtElements);
		this->script_file_ << "/Scope', ";
		this->script_file_ << "'name', '";
		this->script_file_ << GetModifiedName(oExtElements);
		this->script_file_ << "');";
		this->script_file_ << endl;
	}

}

// Returns the input/output type of the signal port
// based on its connections. If it cannot be determined
// returns undefined
InOut CheckType(CyPhyML::ElectricalSignalPort oPort)
{
	int input = 0;
	int output = 0;

	set<CyPhyML::InformationFlow> srcInfoFlow_base = oPort.srcInformationFlow();
	set<CyPhyML::InformationFlow> dstInfoFlow_base = oPort.dstInformationFlow();

	set<CyPhyML::SignalPort2PhysicalPort> srcSig2PhyPort_base = oPort.srcSignalPort2PhysicalPort();
	set<CyPhyML::PhysicalPort2SignalPort> dstPhyPort2Sig_base = oPort.dstPhysicalPort2SignalPort();


	set<CyPhyML::MgaObject> srcInfoFlow;
	set<CyPhyML::MgaObject> dstInfoFlow;
	
		for (set<DML::SignalPort2PhysicalPort>::const_iterator it = srcSig2PhyPort_base.begin();
		it != srcSig2PhyPort_base.end();
		++it)
	{
		// exclude esmol connections
		if (esmolExclude.find(
				DML::MgaObject::Cast(
				DML::MgaObject::Cast(it->srcSignalPort2PhysicalPort_end()).GetParent())) == esmolExclude.end() &&
			esmolExclude.find(
				DML::MgaObject::Cast(
				DML::MgaObject::Cast(it->dstSignalPort2PhysicalPort_end()).GetParent())) == esmolExclude.end())
		{
			srcInfoFlow.insert(*it);
		}
	}

	for (set<DML::PhysicalPort2SignalPort>::const_iterator it = dstPhyPort2Sig_base.begin();
		it != dstPhyPort2Sig_base.end();
		++it)
	{
		// exclude esmol connections
		if (esmolExclude.find(
				DML::MgaObject::Cast(
				DML::MgaObject::Cast(it->srcPhysicalPort2SignalPort_end()).GetParent())) == esmolExclude.end() &&
			esmolExclude.find(
				DML::MgaObject::Cast(
				DML::MgaObject::Cast(it->dstPhysicalPort2SignalPort_end()).GetParent())) == esmolExclude.end())
		{
			dstInfoFlow.insert(*it);
		}
	}

	for (set<DML::InformationFlow>::const_iterator it = srcInfoFlow_base.begin();
		it != srcInfoFlow_base.end();
		++it)
	{
		// exclude esmol connections
		if (esmolExclude.find(
				DML::MgaObject::Cast(
				DML::MgaObject::Cast(it->srcInformationFlow_end()).GetParent())) == esmolExclude.end() &&
			esmolExclude.find(
				DML::MgaObject::Cast(
				DML::MgaObject::Cast(it->dstInformationFlow_end()).GetParent())) == esmolExclude.end())
		{
			srcInfoFlow.insert(*it);
		}
	}

	for (set<DML::InformationFlow>::const_iterator it = dstInfoFlow_base.begin();
		it != dstInfoFlow_base.end();
		++it)
	{
		// exclude esmol connections
		if (esmolExclude.find(
				DML::MgaObject::Cast(
				DML::MgaObject::Cast(it->srcInformationFlow_end()).GetParent())) == esmolExclude.end() &&
			esmolExclude.find(
				DML::MgaObject::Cast(
				DML::MgaObject::Cast(it->dstInformationFlow_end()).GetParent())) == esmolExclude.end())
		{
			dstInfoFlow.insert(*it);
		}
	}


	string path;
	path = GetObjectPath(oPort);

	if (srcInfoFlow.size() > 1)
	{
		return IO_UNDEFINED;
	}
	else if (srcInfoFlow.size() == 1)
	{
		if (srcInfoFlow.begin()->GetParent().uniqueId() ==
			oPort.GetParent().uniqueId())
		{
			bool valid = true;

			for (set<CyPhyML::MgaObject>::iterator it = dstInfoFlow.begin();
				it != dstInfoFlow.end();
				++it)
			{
				if (it->GetParent().uniqueId() ==
					oPort.GetParent().uniqueId())
				{
					valid = false;
				}
			}
			if (valid)
			{
				return IO_OUTPUT;
			}
			else
			{
				return IO_UNDEFINED;
			}
		}
		else
		{
			bool valid = true;

 			for (set<CyPhyML::MgaObject>::iterator it = dstInfoFlow.begin();
				it != dstInfoFlow.end();
				++it)
			{
				if (it->GetParent().uniqueId() ==
					oPort.GetParent().GetParent().uniqueId())
				{
					valid = false;
				}
			}

			if (valid)
			{
				return IO_INPUT;
			}
			else
			{
				return IO_UNDEFINED;
			}
		}
	}
	else if ((srcInfoFlow.size() == 0) && 
		(dstInfoFlow.size() != 0))
	{
		bool valid = true;
		bool input = true;
		bool output = true;

		for (set<CyPhyML::MgaObject>::iterator it = dstInfoFlow.begin();
			it != dstInfoFlow.end();
			++it)
		{
			if (it->GetParent().uniqueId() ==
				oPort.GetParent().uniqueId())
			{
				input = input && true;
				output = output && false;
			}
			else
			{
				input = input && false;
				output = output && true;
			}
		}

		if (input && output)
		{
			valid = false;
		}
		else if (!input && !output)
		{
			valid = false;
		}
		else
		{
			valid = true;
		}

		if (valid)
		{
			if (input)
			{
				return IO_INPUT;
			}
			if (output)
			{
				return IO_OUTPUT;
			}
		}
	}
	return IO_UNDEFINED;
}