/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#ifndef _SCRIPTFILE_H_
#define _SCRIPTFILE_H_

#include "CyPhyML.h"
#include "Uml.h"
#include <direct.h>
#include <fstream>
#include <regex>
#include <string>

// TODO: ?is this implementation good enough in this way???
//#define CONSTANT "built-in/Constant"
#define INTEGRATOR "simulink/Continuous/Integrator"
#define TRANSPORT_DELAY "simulink/Continuous/Transport Delay"
#define DERIVATIVE "simulink/Continuous/Derivative"

#define REGNODE_METRIC_ID "DB_Metric_ID"
#define REGNODE_CONFIG_ID "DB_Config_ID"

class ScriptFile
{
public:
	friend class Graph;

	/// <summary> name of the generated model </summary>
	string szModelName; // name of the generated model
	/// <summary> if &lt; 0 popup all scopes otherwise use the exact number </summary>
	int nScope; // if < 0 popup all scopes otherwise use the exact number
	/// <summary> Name of MatLab scopes (with full path) </summary>
	set<CyPhyML::ExtendedElements> sScopes; // MatLab scopes

	ScriptFile();
	ScriptFile(string &szModelName);

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Print value ranges based on the defined ranges in GME. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	///
	/// <param name="oTargets"> [in,out] Value Flow Targets. </param>
	////////////////////////////////////////////////////////////////////////////////

	void PrintValueRanges(set<CyPhyML::ValueFlowTarget> &oTargets);

	////////////////////////////////////////////////////////////////////////////////
	/// <summary>
	///  Creates a graph viz representation of the flat bond graph in 'dot' format.
	/// </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void PrintGraphViz();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Creates and opens the script and the datastruct files. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void Open();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Closes the script and the datastruct files. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void Close();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Prints script header. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void PrintHeader();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Prints script footer. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void PrintFooter();
	// creates the sysName_MDAOInit.m and the Model Center Wrapper sysName.fileWrapper

	////////////////////////////////////////////////////////////////////////////////
	/// <summary>
	///  Print mdao specific files.
	///  <para>1. MDAOInit.m : initializes the input parameters of the model</para>
	///  <para>2. %szModelName%.fileWrapper : a wrapper file for Model
	///  Center</para>
	/// </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void PrintMDAO();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Print data structure for Bond Graphs. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void PrintDataStructure();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Print Bond Graph elements into the script file. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void PrintBGElements();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Print power port elements (input/output port pairs). </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void PrintPowerPort();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Print i/o ports, which are NOT power ports. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void PrintIoPort();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Resize subsystems. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void ResizeSubsystems();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Print connections. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void PrintConnections();

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Adds a block to MatLab. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	///
	/// <param name="oUdm">	  [in,out] The udm object. </param>
	/// <param name="szKind"> MatLab subsystem type. </param>
	////////////////////////////////////////////////////////////////////////////////

	void AddBlock(Udm::Object &oUdm, string szKind);

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Sets the parameter. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	///
	/// <param name="oUdm"> [in,out] The udm object. </param>
	////////////////////////////////////////////////////////////////////////////////

	void SetParam(Udm::Object &oUdm);

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Sets a parameter. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	///
	/// <param name="oMga"> [in,out] The mga object. </param>
	////////////////////////////////////////////////////////////////////////////////

	void SetParam(CyPhyML::MgaObject &oMga);

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Adds a line to the MatLab model. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	///
	/// <param name="cszBlock">			  The name of the subsystem. </param>
	/// <param name="cszSource">		  The source port (name/portnumber). </param>
	/// <param name="cszDestination">
	///  The destination port (name/portnumber)
	/// </param>
	////////////////////////////////////////////////////////////////////////////////

	void AddLine(const string cszBlock,
							 const string cszSource,
							 const string cszDestination);

	////////////////////////////////////////////////////////////////////////////////
	/// <summary>
	///  Prints an object to as a MatLab block into the script file.
	/// </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	///
	/// <param name="oUdm"> [in,out] The udm object. </param>
	////////////////////////////////////////////////////////////////////////////////

	void operator <<(Udm::Object &oUdm);

	////////////////////////////////////////////////////////////////////////////////
	/// <summary>
	///  Prints an object to as a MatLab block into the script file.
	/// </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	///
	/// <param name="oExtElements"> [in,out] The extended element. </param>
	////////////////////////////////////////////////////////////////////////////////

	void operator <<(CyPhyML::ExtendedElements &oExtElements);

	////////////////////////////////////////////////////////////////////////////////
	/// <summary> Values that represent MatLab specific properties. </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	enum eMatLab { M_WIDTH,
								 M_HEIGHT,
								 M_FOREGROUND,
								 M_BACKGROUND,
								 M_BLOCKTYPE,
								 M_AUTORESIZE};

	/// <summary> Pathname of the output directory </summary>
	string OutputDirectory; // Pathname of the output directory
	string szGraphVizCmdFileName;
	string szGraphVizFileName;
	string szGraphVizPNGFileName;
private:
	ofstream script_file_;
	ofstream datastruct_file_;
	string szScriptWOExt;
	string szDataStructWOExt;

	map<string, map<int, string>> MatLabConfig;

	set<string> m_ResizeSubsystems; // subsystems that will be resized

	// main configuration for MatLab visualization

	////////////////////////////////////////////////////////////////////////////////
	/// <summary>
	///  Initializes main configuration for MatLab block's properties.
	/// </summary>
	///
	/// <remarks> Zsolt, 8/16/2011. </remarks>
	////////////////////////////////////////////////////////////////////////////////

	void InitMatLabConfig();
};

typedef enum InOut { IO_INPUT,
						 IO_OUTPUT,
						 IO_UNDEFINED} InOut;

////////////////////////////////////////////////////////////////////////////////
/// <summary> Check the type of the port. </summary>
///
/// <remarks> Zsolt, 8/16/2011. </remarks>
///
/// <param name="oPort"> The port. </param>
///
/// <returns> The type of the port based on its connections. </returns>
////////////////////////////////////////////////////////////////////////////////

InOut CheckType(CyPhyML::ElectricalSignalPort oPort);

#endif _SCRIPTFILE_H_ // _SCRIPTFILE_H_