/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
#include "Tools.h"

#include "Graph.h"
#include "ScriptFile.h"

#include "UdmConsole.h"
#include "UdmApp.h"

#include "CyPhyML.h"

#include "Message.h"

#include <sstream> // for istringstream
#include <regex>
#include <queue>

#include <GMECOM.h>
extern IMgaProject *_project;

extern ScriptFile sfFile;
extern priority_queue<Message> pqMessages;

using namespace std;

bool FolderExists(string file)
{
	DWORD returnvalue;
	returnvalue = GetFileAttributes(file.c_str());
	if(returnvalue == ((DWORD)-1))
	{
		return false;
	}
	else
	{
		return true;
	}
}

void CreateFolder(string FolderName)
{
	if (FolderName != "")
	{
		if (FolderExists(FolderName.c_str()) == false)
		{
			// create folder
			_mkdir(FolderName.c_str());

		}
		//CreateDirectory(FolderName.c_str(), NULL);
	}
	
}

int GetObjectXPosition(const CyPhyML::MgaObject &o)
{
	string AspectName = "";
	/*if (string(Udm::Object::Cast(o.GetParent()).type().name()) == "System")
	{
		AspectName = "BDMSketcher";
	}
	else if (string(Udm::Object::Cast(o.GetParent()).type().name()) == "Component")
	{
		AspectName = "All";
	}
	else if (string(Udm::Object::Cast(o.GetParent()).type().name()) == "PowerLink")
	{
		AspectName = "BondGraph";
	}*/
	return GetObjectXPosition(o.position(), AspectName);
}
int GetObjectYPosition(const CyPhyML::MgaObject &o)
{
	string AspectName = "";
	/*if (string(Udm::Object::Cast(o.GetParent()).type().name()) == "System")
	{
		AspectName = "BDMSketcher";
	}
	else if (string(Udm::Object::Cast(o.GetParent()).type().name()) == "Component")
	{
		AspectName = "All";
	}
	else if (string(Udm::Object::Cast(o.GetParent()).type().name()) == "PowerLink")
	{
		AspectName = "BondGraph";
	}*/
	return GetObjectYPosition(o.position(), AspectName);
}

// Returns the X position of a specific aspect from GME position string
int GetObjectXPosition(const std::string position, const std::string AspectName) {
	std::string __tmp;
	if ((position.find(AspectName) == -1) || (AspectName == "")) {
		__tmp = position.substr(position.find_first_of('(')+1,position.find_first_of(',')-position.find_first_of('(')-1);
	} else {		
		__tmp = position.substr(position.find(AspectName));
		__tmp = __tmp.substr(AspectName.size()+1,__tmp.find_first_of(',')-AspectName.size()-1);	
	}

	return atoi(__tmp.c_str());
}

// Returns the Y position of a specific aspect from GME position string
int GetObjectYPosition(const std::string position, const std::string AspectName) {
	std::string __tmp;
	if ((position.find(AspectName) == -1) || (AspectName == ""))
	{
		__tmp = position.substr(position.find_first_of(',')+1,position.find_first_of(')')-position.find_first_of(',')-1);
	}
	else
	{		
		__tmp = position.substr(position.find(AspectName));
		__tmp = __tmp.substr(__tmp.find_first_of(',')+1,__tmp.find_first_of(')')-__tmp.find_first_of(',')-1);	
	}	

	return atoi(__tmp.c_str());
}

string GetModifiedName(string name)
{
	// Replace whitespaces to underscores
	string ret=name;
	tr1::regex rx("\\s");
	string fmt("_");
	ret = string(tr1::regex_replace(ret, rx, fmt));

	// Replace dots with underscores
	rx = "[.]";
	ret = string(tr1::regex_replace(ret, rx, fmt));

	// Replace hyphens with underscores
	rx = "[-]";
	ret = string(tr1::regex_replace(ret, rx, fmt));

	// Replace ( with underscores
	rx = "[(]";
	ret = string(tr1::regex_replace(ret, rx, fmt));

	// Replace ) with underscores
	rx = "[)]";
	ret = string(tr1::regex_replace(ret, rx, fmt));

	// Replace : with underscores
	rx = "[:]";
	ret = string(tr1::regex_replace(ret, rx, fmt));

	// Replace ; with underscores
	rx = "[;]";
	ret = string(tr1::regex_replace(ret, rx, fmt));

	return ret;
}

string GetPathWOName(string path)
{
	path = GetModifiedName(path);
	size_t found = path.find_last_of("/");
	path = path.substr(0, found);
	return path;
}

string GetModifiedName(Udm::Object &oMga)
{
	string szModifiedName;
	szModifiedName = oMga.getStringAttr(Uml::Attribute::meta_name);
	szModifiedName = GetModifiedName(szModifiedName);
	return szModifiedName;
}

string GetParentPath(Udm::Object &oMga)
{
	string szParentPath;
	szParentPath = sfFile.szModelName;
	szParentPath += "/";
	szParentPath += oMga.GetParent().getPath2("/",false);
	szParentPath = GetModifiedName(szParentPath);
	return szParentPath;
}


string GetObjectPath(Udm::Object &oMga)
{
	string szObjectPath;
	szObjectPath = sfFile.szModelName;
	szObjectPath += "/";
	szObjectPath += oMga.getPath2("/",false);
	szObjectPath = GetModifiedName(szObjectPath);
	return szObjectPath;
}

void InvalidConnection(CyPhyML::MgaObject &o)
{
	string err = "The following connection is invalid: ";
	err += GMEConsole::Formatter::MakeObjectHyperlink(o.name(),o);
	pqMessages.push(Message(err,MSG_ERROR));	
}

int CheckSignalConstraints(set<CyPhyML::SignalConnection> &SignalSet)
{
	int ErrorNum = 0;
	
	for (std::set<CyPhyML::SignalConnection>::iterator itsignalconn = SignalSet.begin(); itsignalconn != SignalSet.end(); ++itsignalconn)
	{
		CyPhyML::Signal srcsig = itsignalconn->srcSignalConnection_end();
		CyPhyML::Signal dstsig = itsignalconn->dstSignalConnection_end();
		
		// If source and destination is in the same container
		if((srcsig.GetParent().uniqueId() == dstsig.GetParent().uniqueId()))
		{
			if(string(srcsig.type().name()) == "InSignal")
			{
				if(string(dstsig.type().name()) == "InSignal")
				{
					InvalidConnection(CyPhyML::MgaObject::Cast(*itsignalconn));
					ErrorNum++;
				}
			}
			else if(string(srcsig.type().name()) == "OutSignal")
			{
				InvalidConnection(CyPhyML::MgaObject::Cast(*itsignalconn));
				ErrorNum++;
			}
			else if(string(srcsig.type().name()) == "LocalSignal")
			{
				if(string(dstsig.type().name()) == "InSignal")
				{
					InvalidConnection(CyPhyML::MgaObject::Cast(*itsignalconn));
					ErrorNum++;
				}
			}
		}
		else if((srcsig.GetParent().GetParent().uniqueId() == dstsig.GetParent().uniqueId()))
		{
			if(string(srcsig.type().name()) == "InSignal")
			{
				InvalidConnection(CyPhyML::MgaObject::Cast(*itsignalconn));
				ErrorNum++;
			}
			else if(string(srcsig.type().name()) == "OutSignal")
			{
				if(string(dstsig.type().name()) == "InSignal")
				{
					InvalidConnection(CyPhyML::MgaObject::Cast(*itsignalconn));
					ErrorNum++;
				}
			}
		}
		else if((srcsig.GetParent().uniqueId() == dstsig.GetParent().GetParent().uniqueId()))
		{
			if(string(srcsig.type().name()) == "InSignal")
			{
				if(string(dstsig.type().name()) == "OutSignal")
				{
					InvalidConnection(CyPhyML::MgaObject::Cast(*itsignalconn));
					ErrorNum++;
				}
			}
			else if(string(srcsig.type().name()) == "OutSignal")
			{
				InvalidConnection(CyPhyML::MgaObject::Cast(*itsignalconn));
				ErrorNum++;
			}
			else if(string(srcsig.type().name()) == "LocalSignal")
			{
				if(string(dstsig.type().name()) == "OutSignal")
				{
					InvalidConnection(CyPhyML::MgaObject::Cast(*itsignalconn));
					ErrorNum++;
				}
			}
		}
		else if((srcsig.GetParent().GetParent().uniqueId() == dstsig.GetParent().GetParent().uniqueId())
			&& (srcsig.GetParent().uniqueId() != dstsig.GetParent().uniqueId()))
		{
			if(string(srcsig.type().name()) == "InSignal")
			{
				InvalidConnection(CyPhyML::MgaObject::Cast(*itsignalconn));
				ErrorNum++;
			}
			else if(string(srcsig.type().name()) == "OutSignal")
			{
				if(string(dstsig.type().name()) == "OutSignal")
				{
					InvalidConnection(CyPhyML::MgaObject::Cast(*itsignalconn));
					ErrorNum++;
				}
			}
		}
		else
		{
			// This cannot happen
		}
	}
	return ErrorNum;
}

void GetVariables(vector<string> &out,vector<unsigned int> &out_index, vector<string> &in,vector<unsigned int> &in_index, CyPhyML::ControlFunction cf)
{
	std::string code = cf.Code();
	std::string _console;
	// check the sytax of first line
	// Embedded MATLAB function function header's regular expression
	const std::tr1::regex EMFFunctionheader_pattern("function\\s*([\\[]?)\\s*\\w+\\s*([,]\\s*\\w+\\s*)*([\\]])?\\s*=\\s*(\\w+)[\\(]\\s*(\\w+)(\\s*[,]\\s*\\w+)*\\s*[\\)]");
	//const std::tr1::regex EMFFunctionheader_pattern("function\\s*([\[]?)\\s*\\w+\\s*([,]\\s*\\w+\\s*)*([\\]])?\\s*=\\s*(\\w+)[\\(]\\s*(\\w+)(\\s*[,]\\s*\\w+)*\\s*[\\)]");

	bool match = false;
	std::tr1::smatch result;
	match = std::tr1::regex_search(code, result, EMFFunctionheader_pattern);
	if (match == false)
	{
		// the code does not fit to the target language
		_console = "The code attribute does not fit to Embedded MATLAB function header. Please correct it in: ";
		_console += GMEConsole::Formatter::MakeObjectHyperlink(cf.name(),Udm::Object::Cast(cf));
		pqMessages.push(Message(_console,MSG_WARNING));
		//cout << "Does not match for the pattern" << endl;
		return;
	}
	// process the code field
	set<CyPhyML::Signal2ControlFunction> cf2s = cf.srcSignal2ControlFunction();
	set<CyPhyML::ControlFunction2Signal> s2cf = cf.dstControlFunction2Signal();
	set<string> inputs;
	set<string> outputs;

	for (set<CyPhyML::Signal2ControlFunction>::iterator i = cf2s.begin(); i != cf2s.end(); ++i )
	{
		inputs.insert(string(CyPhyML::Signal::Cast(i->srcSignal2ControlFunction_end()).name()));
	}
	for (set<CyPhyML::ControlFunction2Signal>::iterator i = s2cf.begin(); i != s2cf.end(); ++i )
	{
		outputs.insert(string(CyPhyML::Signal::Cast(i->dstControlFunction2Signal_end()).name()));
	}
	int first = -1; // first character's position
	int last = -1; // last's character's position
	int i = 1; // fro indexing
	if (code.empty())
	{
		// error empty code field
		_console = "Code attribure is empty in: ";
		_console += GMEConsole::Formatter::MakeObjectHyperlink(cf.name(),Udm::Object::Cast(cf));
		pqMessages.push(Message(_console,MSG_WARNING));	
	}
	else
	{
		if (code.find_first_of('[') > code.find_first_of('='))
		{
			// one output
			for (first = code.find_first_of(' '); code.at(first) == ' '; first++);			
			last = first;
			for (; (code.at(last) != ' ') && (code.at(last) != '='); last++);
			
			if (outputs.find(code.substr(first,last-first)) != outputs.end())
			{
				out.push_back(code.substr(first,last-first));
				cout << code.substr(first,last-first) << ",";
				out_index.push_back(1); // only one output port
				outputs.erase(code.substr(first,last-first));
			}
			else
			{
				_console = "Does not connect signal named as: ";
				_console += code.substr(first,last-first);
				_console += " in ";
				_console += GMEConsole::Formatter::MakeObjectHyperlink(cf.name(),Udm::Object::Cast(cf));
				pqMessages.push(Message(_console,MSG_WARNING));	
			}
			code = code.substr(code.find_first_of('('));
		}
		else
		{
			// more outputs
			i=1;
			first = code.find_first_of('[')+1;
			do
			{
				for (; (code.at(first) == ' ') || (code.at(first) == ','); first++);			
				last = first;
				for (; (code.at(last) != ' ') && (code.at(last) != ',') && (code.at(last) != ']'); last++);
			
				if (outputs.find(code.substr(first,last-first)) != outputs.end())
				{
					out.push_back(code.substr(first,last-first));
					cout << code.substr(first,last-first) << ",";
					out_index.push_back(i);
					outputs.erase(code.substr(first,last-first));
				}
				else
				{
					_console = "Does not connect output signal named as: ";
					_console += code.substr(first,last-first);
					_console += " in ";
					_console += GMEConsole::Formatter::MakeObjectHyperlink(cf.name(),Udm::Object::Cast(cf));
					pqMessages.push(Message(_console,MSG_WARNING));	
				}

				code = code.substr(last);
				first = code.find_first_of(',');
				i++;
			}
			while (code.find_first_of(',') < code.find_first_of(']'));
			code = code.substr(code.find_first_of('('));
		}
		// input parameter(s)		
		i=1;
		first = code.find_first_of('(')+1;
		do
		{
			for (; (code.at(first) == ' ') || (code.at(first) == ','); first++);			
			last = first;
			for (; (code.at(last) != ' ') && (code.at(last) != ',') && (code.at(last) != ')'); last++);
			
			if (inputs.find(code.substr(first,last-first)) != inputs.end())
			{
				in.push_back(code.substr(first,last-first));
				cout << code.substr(first,last-first) << ",";
				in_index.push_back(i);
				inputs.erase(code.substr(first,last-first));
			}
			else
			{
				_console = "Does not connect input signal named as: ";
				_console += code.substr(first,last-first);
				_console += " in ";
				_console += GMEConsole::Formatter::MakeObjectHyperlink(cf.name(),Udm::Object::Cast(cf));
				pqMessages.push(Message(_console,MSG_WARNING));	
			}
			code = code.substr(last);
			first = code.find_first_of(',');
			i++;
		}
		while (code.find_first_of(',') < code.find_first_of(')'));
	}
}

bool isNumeric( const char* pszInput, int nNumberBase )
{
	istringstream iss( pszInput );
 
	if ( nNumberBase == 10 )
	{
		double dTestSink;
		iss >> dTestSink;
	}
	else if ( nNumberBase == 8 || nNumberBase == 16 )
	{
		int nTestSink;
		iss >> ( ( nNumberBase == 8 ) ? oct : hex ) >> nTestSink;
	}
	else
	{
		return false;
	}
 
	// was any input successfully consumed/converted?
	if ( ! iss )
	{
		return false;
	}

	// was all the input successfully consumed/converted?
	return ( iss.rdbuf()->in_avail() == 0 );
}

double diffclock(clock_t clock1,clock_t clock2)
{
	double diffticks=clock1-clock2;
	double diffms=(diffticks*1000)/CLOCKS_PER_SEC;
	return diffms;
}

string GetElapsedTime(clock_t begin, clock_t end)
{
	char buffer[50];
	string returnValue = "";
	double dTime = double(diffclock(end,begin));
	if (dTime/1000 >= 1.0)
	{
		returnValue += _ltoa_s((dTime-fmod(dTime,1000.0))/1000.0, buffer, 10);
		returnValue += "s ";
	}
	_ltoa_s(fmod(dTime,1000.0), buffer, 10);
	returnValue += std::string(buffer);
	returnValue += "ms";
	
	return returnValue;
}

void GetReferences(Udm::Object &connection, Udm::Object &srcRef, Udm::Object &dstRef)
{
	// TODO: check the connection object is connection!
	CComQIPtr<IMgaObject> obj;
	CComBSTR id;
	HRESULT res;
	id.Empty();
	//connection.type().association()	!= Udm::null
	//Udm::DataNetwork *x; x->ObjectById(gmeid)
	id.Append(UdmGme::UdmId2GmeId(connection.uniqueId()).c_str());
	res = _project->GetObjectByID(id, &obj);
	CComBSTR name;
	if (res == S_OK)
	{
		CComQIPtr<IMgaConnection> conn = obj;
		conn->get_Name(&name);
		CComQIPtr<IMgaConnPoints> cps;
		conn->get_ConnPoints(&cps);
		long cnt;
		cps->get_Count(&cnt);
		for (long i = 1; i <= cnt; ++i)
		{
			CComQIPtr<IMgaConnPoint> cp;
			cps->get_Item(i, &cp);
			CComQIPtr<IMgaFCOs> refs;
			cp->get_References(&refs);
			long cntref;
			refs->get_Count(&cntref);
			for (long ii = 1; ii <= cntref; ++ii)
			{
				CComQIPtr<IMgaFCO> ref;
				refs->get_Item(ii, &ref);
				ref->get_Name(&name);
				id.Empty();
				ref->get_ID(&id);
				string szid = _bstr_t(id);
				CComBSTR role;
				cp->get_ConnRole(&role);
				if (role == "src")
				{
					//srcRef = _bstr_t (name);
					srcRef = connection.__impl()->__getdn()->ObjectById(UdmGme::GmeId2UdmId(szid.c_str()));
				}
				else if (role == "dst")
				{
					//dstRef = _bstr_t (name);
					dstRef = connection.__impl()->__getdn()->ObjectById(UdmGme::GmeId2UdmId(szid.c_str()));
				}
			}
		}
	}
}



string GetSrcLabel1(Udm::Object &connection)
{
  return GetConnecitonLabel(connection, "srcLabel1");
}
string GetSrcLabel2(Udm::Object &connection)
{
  return GetConnecitonLabel(connection, "srcLabel2");
}
string GetDstLabel1(Udm::Object &connection)
{
  return GetConnecitonLabel(connection, "dstLabel1");
}
string GetDstLabel2(Udm::Object &connection)
{
  return GetConnecitonLabel(connection, "dstLabel2");
}

////////////////////////////////////////////////////////////////////////////////
/// <summary> Gets a conneciton label. </summary>
///
/// <remarks> Zsolt, 7/28/2011. </remarks>
///
/// <param name="connection"> [in,out] The connection. </param>
/// <param name="LabelName">
///  Name of the label. Supported label names:
///  <para>- srcLabel1</para>
///  <para>- srcLabel2</para>
///  <para>- dstLabel1</para>
///  <para>- dstLabel2</para>
/// </param>
///
/// <returns> The conneciton label. </returns>
////////////////////////////////////////////////////////////////////////////////

string GetConnecitonLabel(Udm::Object &connection, string LabelName)
{
  string result = "<blank>";
  if (connection == Udm::null)
  {
    return result;
  }

  if (LabelName != "srcLabel1" &&
      LabelName != "srcLabel2" &&
      LabelName != "dstLabel1" &&
      LabelName != "dstLabel2")
  {
    return result;
  }

  CComQIPtr<IMgaObject> obj;
	CComBSTR id;
  CComBSTR value;
	HRESULT res;
	id.Empty();
  id.Append(UdmGme::UdmId2GmeId(connection.uniqueId()).c_str());
	res = _project->GetObjectByID(id, &obj);
	CComBSTR name;
	if (res == S_OK)
	{
    long cntref;
    CComQIPtr<IMgaConnection> conn = obj;
		conn->get_Name(&name);
		CComQIPtr<IMgaRegNodes> nodes;

    // get virtual/non-virtual nodes
    conn->get_Registry(VARIANT_TRUE, &nodes);
		nodes->get_Count(&cntref);
		for (long ii = 1; ii <= cntref; ++ii)
		{
      CComQIPtr<IMgaRegNode> node;
			nodes->get_Item(ii, &node);
			node->get_Name(&name);
			value.Empty();
      node->get_Value(&value);
			string szvalue = _bstr_t(value);
      string szname = _bstr_t(name);
      if (LabelName == szname)
      {
        result = szvalue;
        return result;
      }
		}
  }
  return result;
}


Udm::Object GetBaseType(Udm::Object &obj)
{
  if (obj.archetype() != Udm::null)
  {
    return GetBaseType(obj.archetype());
  }
  else
  {
    return obj;
  }
}

string GetGuid(Udm::Object &subject)
{
	string guid = "";
	if (subject != Udm::null)
	{
		CComQIPtr<IMgaObject> obj;
		CComBSTR id;
		HRESULT res;
		id.Empty();

		id.Append(UdmGme::UdmId2GmeId(subject.uniqueId()).c_str());
		res = _project->GetObjectByID(id, &obj);
		if (res == S_OK)
		{
			HRESULT res2;
			CComBSTR _comGuid;
			res2 = obj->GetGuidDisp(&_comGuid);
			if (res2 == S_OK)
			{
				guid = _bstr_t(_comGuid);
			}
		}
	}
	return guid;
}

string GetRegNodeValueByName(
	CyPhyML::MgaObject object,
	string regnodeName)
{
  CComQIPtr<IMgaFCO> tb;
	CComPtr<IUnknown> punk = UdmGme::Udm2Gme(object);
	punk.QueryInterface(&tb);
	CComBSTR name;
	
	if (tb)
	{
		_bstr_t id;
		COMTHROW(tb->get_Name(&name));
		CComBSTR value;
		HRESULT result = tb->get_RegistryValue(CComBSTR(regnodeName.c_str()), id.GetAddress());
		// check bad ptr
		if (result == S_OK)
		{
			if (id.length())
			{
				return static_cast<const char*>(id);
			}
		}
  }
	return "";
}