/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
/*
*******************************************************************************
*                                    META
*******************************************************************************
* Interface for the CUdmApp class and the entry point of the model processing.
*
* Filename:  UdmApp.cpp 
*
* Revision History: available in the META svn
* Link: 
* 
* Version      Date         Author         Remarks (including PR/RCP refs)
* -------      ----         ------         -------------------------------
*    1      01/13/2011  Zsolt Lattmann             Initial issue
*   
*
*******************************************************************************
* Copyright: ...
*******************************************************************************
*/
#include "stdafx.h"
#include <time.h>

#include "UdmApp.h"
#include "UdmConfig.h"
#include "Uml.h"
#include "UdmUtil.h"
#include "UdmConsole.h"

#include "SimulinkDlg.h"
#include "Graph.h"
#include "Tools.h"
#include "ScriptFile.h"

#include "CyPhyToolbox.h"

#include <fstream>

#include "boost/filesystem.hpp"

extern bool bCanceledByUser;
bool bUseESMoL = false;
extern set<Udm::Object> SelectedObj;
// object to process
extern MapContainer ObjToProcess; // <Obj, true = full container process>
extern string EsMoLMainBlock;
extern string EsMoLBuildScript;

extern string ConfigurationID;

//extern set<string> sFullProcessElements;

// global variables for automation
extern bool Automation;
extern bool Expanded;
extern string OutputDir;
extern string ModelName;
extern string DependencyDir;
//extern bool SilentMode;

bool bParametricExp = false;

extern IMgaProject *_project;

/*********************************************************************************/
/* Initialization function. The framework calls it before preparing the backend. */
/* Initialize here the settings in the config global object.					 */
/* Return 0 if successful.														 */
/*********************************************************************************/
int CUdmApp::Initialize()
{
	ClearAllGlobals();
	
	//sFullProcessElements.insert(DML::TestBench::meta_name.name());

	bCanceledByUser = false;
	return 0;
}

/***********************************************/
/* Main entry point for Udm-based Interpreter  */
/***********************************************/

void CUdmApp::UdmMain(
					 Udm::DataNetwork* p_backend,		// Backend pointer(already open!)
					 Udm::Object focusObject,			// Focus object
					 set<Udm::Object> selectedObjects,	// Selected objects
					 long param)						// Parameters
{	
	try
	{
		bParametricExp = false;

		if (Automation == false)
		{
			Expanded = false;
		}

		// new message line buffer
		string szNewLine;

		//string focusObjectGuid;
		//focusObjectGuid = GetGuid(focusObject);

		// stores the expanded component assemblies
		set<DML::ComponentAssembly> caExpanded;
		DML::TestBench test;

		// elaborator helper
		CyPhyElaborate elaborator;

		if (focusObject != Udm::null &&
			focusObject.type() == DML::ParametricExploration::meta)
		{
			bParametricExp = true;
		}

		if (selectedObjects.size() > 1)
		{
			throw exception("The number of selected objects must be 1 or 0.");
		}
		else if (selectedObjects.size() == 0 &&
			focusObject != Udm::null)
		{
			selectedObjects.insert(focusObject);
			if (Expanded == false)
			{
				GMEConsole::Console::Info::writeLine("Expanding graph...");

				if (focusObject.type() == DML::TestBench::meta)
				{
					// save meta data for metrics
					// ASSUMPTION: Metric names are unique
					ofstream f(string(OutputDir + "_MetaData").c_str());
					
					set<DML::Metric> metrics = DML::TestBench::Cast(focusObject).Metric_kind_children();
					for (set<DML::Metric>::iterator it = metrics.begin();
						it != metrics.end();
						++it)
					{
						f << it->name();
						f << " ";
						f << it->uniqueId();
						f << " ";
						f << UdmGme::UdmId2GmeId(it->uniqueId());
						f << " ";
						f << GetGuid(Udm_VS10::Object::Cast(*it));
						f << endl;
					}

					f.close();

					// the focused object is a test bench
					DML::ComponentAssembly assembly;
					test = DML::TestBench::Cast(focusObject);
					elaborator.elaborate(test);
				}
				else if (focusObject.type() == DML::ComponentAssembly::meta)
				{
					DML::ComponentAssembly ca = DML::ComponentAssembly::Cast(focusObject);
					elaborator.elaborate(ca);
				}
			}
		}

		// (1) get all selected object and put those into the ObjToProcess container
		//     and also determine the type of process that is needed FULL or not FULL
		//     FULL process means all children must be processed
		for (set<Udm::Object>::iterator i = selectedObjects.begin();
			i != selectedObjects.end();
			++i)
		{
			if ((*i).type() == DML::ParametricExploration::meta)
			{
				bParametricExp = true;
				set<DML::TestBenchRef> sTestBenchRef = DML::ParametricExploration::Cast(*i).TestBenchRef_kind_children();
				for (set<DML::TestBenchRef>::iterator itTestBenchRef = sTestBenchRef.begin();
					itTestBenchRef != sTestBenchRef.end();
					++itTestBenchRef)
				{
					DML::TestBenchRef ref = DML::TestBenchRef::Cast(*itTestBenchRef);
					// TODO: Replace MATLAB to attribute...
					//if (GetModifiedName(ref.getReferencedObject()).find("MATLAB") != string::npos)
					//{
					//	Udm::Object o = ref.getReferencedObject();
					//	SelectedObj.insert(o);

					//	bool FullProcess = true;
					//	MapContainer::iterator it;
					//	it = ObjToProcess.find(o);
					//	if (it != ObjToProcess.end())
					//	{
					//		it->second = FullProcess;
					//	}
					//	else
					//	{
					//		// Insert the selected object for full process
					//		ObjToProcess.insert(make_pair(o, FullProcess));
					//	}
					//
					//	while (o.GetParent() != Udm::null)
					//	{
					//		string path = o.getPath();
					//		o = o.GetParent();
					//		it = ObjToProcess.find(o);
					//		if (it != ObjToProcess.end())
					//		{
					//			//it->second = true;
					//		}
					//		else
					//		{
					//			FullProcess = false;
					//			// Insert the selected object for full process
					//			ObjToProcess.insert(make_pair(o, FullProcess));
					//		}
					//	}
					//}
				}
			}

			SelectedObj.insert(*i);

			bool FullProcess = true;
			MapContainer::iterator it;
			it = ObjToProcess.find(*i);
			if (it != ObjToProcess.end())
			{
				it->second = FullProcess;
			}
			else
			{
				// Insert the selected object for full process
				ObjToProcess.insert(make_pair(*i, FullProcess));
			}
			

			Udm::Object o = *i;
			while (o.GetParent() != Udm::null)
			{
				string path = o.getPath();
				o = o.GetParent();
				it = ObjToProcess.find(o);
				if (it != ObjToProcess.end())
				{
					//it->second = true;
				}
				else
				{
					FullProcess = false;
					// Insert the selected object for full process
					ObjToProcess.insert(make_pair(o, FullProcess));
				}
			}
		}

		//if (Automation == false)
		//{
			// (2) set up default values for Directories and output model name
			string ProjectDir;
			ProjectDir = CUdmApp::szConnString.
				substr(string("MGA=").length()).
				substr(0,CUdmApp::szConnString.find_last_of('\\') - string("MGA=").length() + 1);

			// set the DependencyDir
			DependencyDir = ProjectDir;
			DependencyDir += "Include\\";

			if (Automation == false)
			{
				// set the OutputDir
				OutputDir = ProjectDir;
				OutputDir += "generated\\";
			}
			// set the name of the model
			Udm_VS10::Object rootFolder;
			rootFolder = DML::RootFolder::Cast(p_backend->GetRootObject());
			ModelName = GetModifiedName(rootFolder);
			//ModelName += "_renamed";
		//}

		if (Automation == false)
		{
			// if not automated show a dialog box with a set of options
			CSimulinkDlg dlg;
			dlg.DoModal();
		}

		// get the start time
		clock_t begin=clock();
		if (!bCanceledByUser)
		{
			if (bUseESMoL)
			{
				CComPtr<IMgaComponentEx> CyPhy2ESMoL;
				CyPhy2ESMoL.CoCreateInstance(L"MGA.Interpreter.CyPhy2ESMoL", NULL, CLSCTX_INPROC);
				if (CyPhy2ESMoL == NULL)
					throw udm_exception("Could not create CyPhy2ESMoL interpreter");
				IUnknown* p = UdmGme::Udm2Gme(focusObject);
				HRESULT hr = p ? S_OK : E_FAIL;
				CComPtr<IMgaFCO> focus;
				if (SUCCEEDED(hr)) {
					hr = p->QueryInterface(&focus);
					p->Release();
				}
				if (SUCCEEDED(hr))
					hr = CyPhy2ESMoL->put_ComponentParameter(_bstr_t(L"create_transaction"), CComVariant(false));
				CComPtr<IMgaProject> project;
				if (SUCCEEDED(hr))
					hr = focus->get_Project(&project);
				if (SUCCEEDED(hr))
					hr = CyPhy2ESMoL->put_ComponentParameter(_bstr_t("output_dir"), CComVariant(CComBSTR(OutputDir.c_str())));
				if (SUCCEEDED(hr))
					hr = CyPhy2ESMoL->put_ComponentParameter(_bstr_t("automation"), CComVariant(CComBSTR(Automation ? "true" : "false")));
				if (SUCCEEDED(hr))
				{
					hr = CyPhy2ESMoL->InvokeEx(project, focus, NULL, 128);
					if (FAILED(hr))
					{
						CComPtr<IErrorInfo> errorInfo;
						// KMS: no need to QI(ISupportErrorInfo)
						GetErrorInfo(NULL, &errorInfo);
						if (errorInfo)
						{
							_bstr_t description;
							errorInfo->GetDescription(description.GetAddress());
							if (description.length())
								throw udm_exception(std::string("CyPhy2ESMoL failed: ") + static_cast<const char*>(description));
						}
					}
				}
				_variant_t mainEsmolPath;
				hr = CyPhy2ESMoL->get_ComponentParameter(_bstr_t(L"OutputBlock"), &mainEsmolPath);
				if (FAILED(hr))
					throw udm_exception("Couldn't get output path from CyPhy2ESMoL");
				EsMoLMainBlock = std::string(_bstr_t(mainEsmolPath));
				_variant_t buildScript;
				hr = CyPhy2ESMoL->get_ComponentParameter(_bstr_t(L"BuildScript"), &buildScript);
				if (FAILED(hr))
					throw udm_exception("Couldn't get buildscript path from CyPhy2ESMoL");
				EsMoLBuildScript = _bstr_t(buildScript);
				if (FAILED(hr))
					throw udm_exception("CyPhy2ESMoL failed");
			}

			// Create a new Graph for the existing domain model
			Graph DomainModel;
			
			if (focusObject != Udm::null)
			{
				ConfigurationID = GetRegNodeValueByName(CyPhyML::MgaObject::Cast(focusObject), REGNODE_CONFIG_ID);
			}

			// Process the data network
			DomainModel.Process(p_backend);
			
			// copy the dependencies to the output directory
			// this code copies only files.
			if (!bParametricExp)
			{
				std::string includeDir = ProjectDir + "\\Include";
				
				if (boost::filesystem::is_directory(includeDir))
				{
					boost::filesystem::directory_iterator includeFiles(includeDir);
					while (includeFiles != boost::filesystem::directory_iterator())
					{
						boost::filesystem::path to_fp(OutputDir + "\\" + boost::filesystem::path(*includeFiles).filename());
						if (boost::filesystem::is_regular_file(*includeFiles))
						{
							if (boost::filesystem::exists(to_fp))
								boost::filesystem::remove(to_fp);
							boost::filesystem::copy_file(*includeFiles, to_fp);
							includeFiles++;
						}
					}
				}
			}
		}
		else
		{
			// if the user hit the cancel button
			string szErrorText;
			szErrorText =  "Operation is cancelled by the user.";
			AfxMessageBox(szErrorText.c_str());
		}

		if (Automation == true &&
			Expanded == false)
		{
			GMEConsole::Console::Info::writeLine("Collapsing graph...");

			for (set<DML::ComponentAssembly>::iterator it = caExpanded.begin();
				it != caExpanded.end();
				++it)
			{
				elaborator.collapse(*it);
			}
		}

		// get the stop time
		clock_t end=clock();

		// get elapsed time and show it in the (GME)Console
		szNewLine = "Time elapsed: ";
		szNewLine += GetElapsedTime(begin, end);
		szNewLine += ".";
		GMEConsole::Console::Out::writeLine(szNewLine);

	}
	catch (const udm_exception &e)
	{
		// this is a udm exception.
		AfxMessageBox(e.what());
	}
	catch (const exception &e)
	{
		// std exception
		AfxMessageBox(e.what());
	}
	catch (...)
	{
		// other type of exceptions
		string szErrorText;
		szErrorText =  "Unexpected error. Please check all constraints and";
		szErrorText += " try to interpret your model again.\nIf the problem";
		szErrorText += " still occur, report to the developer.";
		AfxMessageBox(szErrorText.c_str());
	}
}