/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.Runtime.InteropServices;
using System.Text;
using System.Windows.Forms;
using GME.MGA.Core;
using GME.CSharp;
using GME;
using GME.MGA;
using GME.MGA.Meta;
using System.Linq;
using System.Diagnostics.Contracts;
using System.Diagnostics;
using GME.Util;
using System.Collections;
using JobManager;
using System.Xml.Serialization;
using System.Xml;
using CyPhyMasterInterpreter.Interpreter;
using META;
using Newtonsoft.Json;
using System.Reflection;

using CyPhy = ISIS.GME.Dsml.CyPhyML.Interfaces;

namespace CyPhyMasterInterpreter
{
    class MgaObjectEqualityComparor<T> : EqualityComparer<T> where T : IMgaObject
    {
        public override bool Equals(T x, T y)
        {
            return x.ID.Equals(y.ID);
        }

        public override int GetHashCode(T obj)
        {
            return obj.ID.GetHashCode();
        }
    }

    namespace Dashboard
    {
        public static class Config
        {
            public const string RegNodeConfig = "DB_Config_ID";
            public const string RegNodeMetric = "DB_Metric_ID";
            public const string RegNodeMetricUndefined = "MXXXXXXXX";
        }
    }

    /// <summary>
    /// This class implements the necessary COM interfaces for a GME 
    /// interpreter component.
    /// </summary>
    [Guid(ComponentConfig.guid),
    ProgId(ComponentConfig.progID),
    ClassInterface(ClassInterfaceType.AutoDual)]
    [ComVisible(true)]
    public class CyPhyMasterInterpreter : IMgaComponentEx, IGMEVersionInfo
    {

        /// <summary>
        /// Contains information about the GUI event that initiated the invocation.
        /// </summary>
        [ComVisible(false)]
        public enum ComponentStartMode
        {
            GME_MAIN_START = 0, 		// Not used by GME
            GME_BROWSER_START = 1,      // Right click in the GME Tree Browser window
            GME_CONTEXT_START = 2,		// Using the context menu by right clicking a model element in the GME modeling window
            GME_EMBEDDED_START = 3,		// Not used by GME
            GME_MENU_START = 16,		// Clicking on the toolbar icon, or using the main menu
            GME_BGCONTEXT_START = 18,	// Using the context menu by right clicking the background of the GME modeling window
            GME_ICON_START = 32,		// Not used by GME
            GME_SILENT_MODE = 128 		// Not used by GME, available to testers not using GME
        }

        /// <summary>
        /// This function is called for each interpreter invocation before Main.
        /// Don't perform MGA operations here unless you open a tansaction.
        /// </summary>
        /// <param name="project">The handle of the project opened in GME, for 
        /// which the interpreter was called.</param>
        public void Initialize(MgaProject project)
        {

        }

        [ComVisible(false)]
        public MgaFCO OriginalSubject { get; set; }
        [ComVisible(false)]
        public List<MgaReference> CaRefsToProcess;

        // Valid when Mode == ME_DS
        [ComVisible(false)]
        public MgaFCO designSpace { get; set; }
        [ComVisible(false)]
        public List<MgaReference> testBenchRefs = new List<MgaReference>();
        [ComVisible(false)]
        public List<MgaModel> expandedParametricExplorations = new List<MgaModel>();
        [ComVisible(false)]
        public List<MgaModel> expandedSoTs = new List<MgaModel>();

        [ComVisible(false)]
        public AVM.DDP.MetaAvmProject avmProj = null;

        /// <summary>
        /// The main entry point of the interpreter. A transaction is already open,
        /// GMEConsole is avaliable. A general try-catch block catches all the exceptions
        /// coming from this function, you don't need to add it. For more information,
        /// see InvokeEx.
        /// </summary>
        /// <param name="project">
        /// The handle of the project opened in GME, for which the interpreter was called.
        /// </param>
        /// <param name="currentobj">
        /// The model open in the active tab in GME. Its value is null if no model is open 
        /// (no GME modeling windows open). </param>
        /// <param name="selectedobjs">
        /// A collection for the selected  model elements. It is never null.
        /// If the interpreter is invoked by the context menu of the GME Tree Browser, 
        /// then the selected items in the tree browser. Folders
        /// are never passed (they are not FCOs).
        /// If the interpreter is invoked by clicking on the toolbar icon or the context 
        /// menu of the modeling window, then the selected items 
        /// in the active GME modeling window. If nothing is selected, the collection is 
        /// empty (contains zero elements).
        /// </param>
        /// <param name="startMode">
        /// Contains information about the GUI event that initiated the invocation.
        /// </param>
        public void Main(
            MgaProject project,
            MgaFCO currentobj,
            MgaFCOs selectedobjs,
            ComponentStartMode startMode)
        {
            // start the stopper
            sw = Stopwatch.StartNew();

            OriginalSubject = currentobj;

            // set console for (testbenchgateway) log
            tbg.Console = GMEConsole;


            // handle the different modes, which are supported
            if (Mode == Mode_enum.TestBench_DS)
            {
                // TestBench contains a Design Space ref

                // generate test benches, based on the given template (currentObjInvoked)
                tbg.Extract(currentobj as MgaModel);
                foreach (var item in tbg.fcosToProcess)
                {
                    toProcess.Add(new FCO(item.Key, item.Value), GetWorkFlow(currentobj));
                }

                this.avmProj.UpdateTestBenchJson(OriginalSubject);

            }
            else if (Mode == Mode_enum.TestBench_Regular)
            {
                MgaFolder f = currentobj.ParentFolder;

                if (f == null)
                {
                    throw new Exception("Testbench does not have a TestFolder parent!");
                }

                MgaFolder fNew = f.CreateFolder(f.MetaFolder);

                fNew.Name = currentobj.Name + DateTime.Now.ToString(" (MM-dd-yyyy HH:mm:ss)");

                MgaFCO newTestbench = fNew.CopyFCODisp(currentobj);
                //newTestbench.Name += " TestName";

                if (newTestbench != null)
                {
                    SelectedItems.Add(new FcoItem(newTestbench));
                }

                foreach (MgaFCO item in selectedobjs)
                {
                    SelectedItems.Add(new FcoItem(item));
                }

                if (newTestbench != null)
                {
                    // TestBench contains a component assembly ref
                    if (toProcess.Where(x => x.Key.Fco == newTestbench).Count() == 0)
                    {
                        toProcess.Add(new FCO(newTestbench, newTestbench.Name), GetWorkFlow(newTestbench));
                    }
                }
                else
                {
                    foreach (var item in mf.lbExportedCAs.SelectedItems.Cast<SelectedItem>())
                    {
                        toProcess.Add(new FCO(item.ExportedCa, item.DisplayedName), GetWorkFlow(item.ExportedCa));
                    }
                }
            }
            else if (Mode == Mode_enum.ME_Regular)
            {
                // start point is a ParametricExploration

                toProcess.Add(new FCO(OriginalSubject as MgaFCO, OriginalSubject.Name), GetWorkFlow(OriginalSubject));
            }
            else if (Mode == Mode_enum.ME_DS)
            {
                // The TestBench referenced by TestbenchRef in Parameteric Exploration model contains a Design Space ref

                Dictionary<MgaReference, Dictionary<MgaReference, MgaModel>> allExpandedTestbenches = new Dictionary<MgaReference, Dictionary<MgaReference, MgaModel>>(new MgaObjectEqualityComparor<IMgaReference>());
                // generate test benches, based on the referenced template(s)
                foreach (MgaReference testBench in testBenchRefs)
                {
                    Dictionary<MgaReference, MgaModel> expandedTestbenches = tbg.Extract(testBench.Referred as MgaModel);
                    allExpandedTestbenches[testBench] = expandedTestbenches;
                    this.avmProj.UpdateTestBenchJson(testBench.Referred);
                }

                // string d1 = string.Join(" ", allExpandedTestbenches.Keys.Select(x=>x.AbsPath));

                Main_ME_DS(allExpandedTestbenches);
            }
            else if (Mode == Mode_enum.SoT_DS)
            {
                Dictionary<MgaReference, Dictionary<MgaReference, MgaModel>> allExpandedTestbenches = new Dictionary<MgaReference, Dictionary<MgaReference, MgaModel>>(new MgaObjectEqualityComparor<IMgaReference>());
                // generate test benches, based on the referenced template(s)
                foreach (MgaReference testBench in testBenchRefs)
                {
                    Dictionary<MgaReference, MgaModel> expandedTestbenches = tbg.Extract(testBench.Referred as MgaModel);
                    allExpandedTestbenches[testBench] = expandedTestbenches;
                    this.avmProj.UpdateTestBenchJson(testBench.Referred);
                }

                // string d1 = string.Join(" ", allExpandedTestbenches.Keys.Select(x=>x.AbsPath));

                // create a mode for CyPhySoT similar to ME_DS
                Main_ME_SoT(allExpandedTestbenches);
            }
            else if (Mode == Mode_enum.CA_DS)
            {
                // TODO: handle Mode_enum.CA_DS

            }
            else if (Mode == Mode_enum.CA_Regular)
            {
                // TODO: handle Mode_enum.CA_Regular
                if (currentobj != null)
                {
                    if (toProcess.Where(x => x.Key.Fco == currentobj).Count() == 0)
                    {
                        Queue<ComComponent> wf = new Queue<ComComponent>();
                        wf.Enqueue((mf.lbWorkFlow.SelectedItem as WorkFlowListItem).Interpreter);
                        toProcess.Add(new FCO(currentobj, currentobj.Name), wf);
                    }
                }
                else
                {
                    foreach (var item in mf.lbExportedCAs.SelectedItems.Cast<SelectedItem>())
                    {
                        Queue<ComComponent> wf = new Queue<ComComponent>();
                        wf.Enqueue((mf.lbWorkFlow.SelectedItem as WorkFlowListItem).Interpreter);
                        toProcess.Add(new FCO(item.ExportedCa, item.DisplayedName), wf);
                    }
                }
            }
            else if (Mode == Mode_enum.Undefined)
            {
                // TODO: handle Mode_enum.Undefined
            }
            else
            {
                // if somebody modifies the enum.
                throw new NotSupportedException("Mode is not supported at all. " + Mode);
            }
        }

        private void Main_ME_SoT(Dictionary<MgaReference, Dictionary<MgaReference, MgaModel>> allExpandedTestbenches)
        {
            MgaFolder sotFolder = OriginalSubject.ParentFolder;
            MgaFolder tempFolder = sotFolder.CreateFolder(sotFolder.MetaFolder);
            tempFolder.Name = OriginalSubject.Name + " " + DateTime.Now.ToString();

            foreach (var caRef in CaRefsToProcess)
            {
                MgaModel expandedSot = tempFolder.CopyFCODisp(this.OriginalSubject) as MgaModel;
                expandedSot.Name = caRef.Name;

                expandedSot.RegistryValue["SoTUniqueName"] = this.OriginalSubject.Name;

                var expandedTestBenchRefs = expandedSot
                    .ChildFCOs
                    .Cast<MgaFCO>()
                    .Where(fco => fco.MetaRole.Name == "TestBenchRef");

                foreach (MgaReference expandedTestBenchRef in expandedTestBenchRefs)
                {
                    MgaReference testBenchRef = OriginalSubject.ObjectByPath["/#" + expandedTestBenchRef.RelID] as MgaReference;
                    Dictionary<MgaReference, MgaModel> expandedTestBenchByCARef = allExpandedTestbenches[testBenchRef];

                    ReferenceSwitcher.Switcher.MoveReferenceWithRefportConnections(
                        expandedTestBenchByCARef[caRef],
                        expandedTestBenchRef,
                        WriteLine);
                }

                toProcess.Add(
                    new FCO(expandedSot as MgaFCO, expandedSot.Name),
                    GetWorkFlow(OriginalSubject));

                expandedSoTs.Add(expandedSot);
            }
        }

        public void Main_ME_DS(Dictionary<MgaReference, Dictionary<MgaReference, MgaModel>> allExpandedTestbenches)
        {
            MgaFolder parametricExplorationFolder = OriginalSubject.ParentFolder;
            MgaFolder tempFolder = parametricExplorationFolder.CreateFolder(parametricExplorationFolder.MetaFolder);
            tempFolder.Name = OriginalSubject.Name + " " + DateTime.Now.ToString();

            foreach (var caRef in CaRefsToProcess)
            {
                MgaModel expandedParametricExploration = tempFolder.CopyFCODisp(OriginalSubject) as MgaModel;
                expandedParametricExploration.Name = caRef.Name;

                var expandedTestBenchRefs = expandedParametricExploration
                    .ChildFCOs
                    .Cast<MgaFCO>()
                    .Where(fco => fco.MetaRole.Name == "TestBenchRef");

                foreach (MgaReference expandedTestBenchRef in expandedTestBenchRefs)
                {
                    MgaReference testBenchRef = OriginalSubject.ObjectByPath["/#" + expandedTestBenchRef.RelID] as MgaReference;
                    Dictionary<MgaReference, MgaModel> expandedTestBenchByCARef = allExpandedTestbenches[testBenchRef];

                    ReferenceSwitcher.Switcher.MoveReferenceWithRefportConnections(
                        expandedTestBenchByCARef[caRef],
                        expandedTestBenchRef,
                        WriteLine);
                }

                toProcess.Add(
                    new FCO(expandedParametricExploration as MgaFCO, expandedParametricExploration.Name),
                    GetWorkFlow(OriginalSubject));

                expandedParametricExplorations.Add(expandedParametricExploration);
            }
        }

        Dictionary<string, int> MetricIds = new Dictionary<string, int>();

        private string GetDashboardMetricID(string p)
        {
            string format = "{0:00}";

            if (MetricIds.ContainsKey(p))
            {
                return String.Format(format, MetricIds[p]);
            }
            else
            {
                int max = 0;
                if (MetricIds.Values.Count > 0)
                {
                    max = MetricIds.Values.Max() + 1;
                }
                MetricIds.Add(p, max);
                return String.Format(format, MetricIds[p]);
            }
        }

        #region Additional Properties

        public Uri JobServerConnection = new Uri("tcp://localhost:35010/JobServer");

        /// <summary>
        /// contains the object that needs to be passed to the interpreters
        /// and the workflow of the interpreters
        /// </summary>
        public Dictionary<FCO, Queue<ComComponent>> toProcess =
            new Dictionary<FCO, Queue<ComComponent>>();

        /// <summary>
        /// helps to get the names outside a transaction
        /// </summary>
        public class FCO
        {
            public MgaFCO Fco { get; set; }
            public string Name { get; set; }

            public FCO(MgaFCO fco, string name)
            {
                Fco = fco;
                Name = name;
            }
        }

        /// <summary>
        /// Represents the current mode of the interpreter.
        /// </summary>
        public Mode_enum Mode { get; set; }

        public enum Mode_enum
        {
            /// <summary>
            /// This start point (current object) is invalid.
            /// </summary>
            Undefined,

            /// <summary>
            /// Current object is ONE Component Assembly
            /// </summary>
            CA_Regular,

            /// <summary>
            /// Current object is a Design Container, which represent multiple 
            /// Component assemblies
            /// </summary>
            CA_DS,

            /// <summary>
            /// ONE Test bench
            /// </summary>
            TestBench_Regular,

            /// <summary>
            /// Test bench with a Design Space
            /// </summary>
            TestBench_DS,

            /// <summary>
            /// Current object is a Multi-Experiment, which does not contain 
            /// Design Space references.
            /// </summary>
            ME_Regular,

            /// <summary>
            /// Current object is a Multi-Experiment, which contains 
            /// Design Space references.
            /// </summary>
            ME_DS,

            /// <summary>
            /// Current object is a Set of test benches, which contains 
            /// Design Space references.
            /// </summary>
            SoT_DS,
        }

        /// <summary>
        /// Contains the workflow
        /// </summary>
        internal Queue<ComComponent> Workflow { get; set; }

        /// <summary>
        /// Full path of the output (root) directory. User can specify this.
        /// </summary>
        public string OutputDir { set; get; }

        /// <summary>
        /// Mga project file is in this directory.
        /// </summary>
        public string ProjectPath { set; get; }
        #endregion

        #region Additional Members

        Stopwatch sw;

        /// <summary>
        /// Gateway to the main form
        /// </summary>
        internal MainForm mf { get; set; }

        /// <summary>
        /// Gateway for test bench extraction.
        /// </summary>
        [ComVisible(false)]
        public TestBenchGateway tbg { get; set; }
        internal List<MgaModel> ConfigModels = new List<MgaModel>();
        internal List<DesignConfigItem> DesignConfigs = new List<DesignConfigItem>();
        internal List<FcoItem> SelectedItems = new List<FcoItem>();

        #endregion

        #region Additional Functions

        private void ShowDialog(
                MgaProject project,
                MgaFCO currentobj,
                MgaFCOs selectedobjs,
                ComponentStartMode componentStartMode)
        {
            try
            {
                // validates the acceptable start points
                if (Mode == Mode_enum.Undefined)
                {
                    StringBuilder sb = new StringBuilder();
                    sb.AppendLine("This interpreter can be used in 3 different modes. This one is none of following.");
                    sb.AppendLine("Please start this interpreter from a:");
                    sb.AppendLine(" - Component Assembly (Mode 1R)");
                    sb.AppendLine(" - Component (Mode 1R)");
                    sb.AppendLine(" - Design Container (Mode 1DS)");
                    sb.AppendLine(" - Test Bench, which does not contain Design Space reference. (Mode 2R)");
                    sb.AppendLine(" - Test Bench, which contains Design Space reference. (Mode 2DS)");
                    sb.AppendLine(" - Parametric Exploration, where the test benches do not contain Design Space reference. (Mode 3R)");
                    sb.AppendLine(" - Parametric Exploration, where the test benches contain Design Space reference. (Mode 3DS)");

                    throw new Exception(sb.ToString());
                }

                // initiate start
                GMEConsole.Info.WriteLine("Start");

                SetOutputDir(project, currentobj);

                // get the project path
                string ProjectPath;
                ProjectPath = Path.GetDirectoryName(
                        project.ProjectConnStr.Substring("MGA=".Length));

                // the main form will show the Workflow content
                if (Mode == Mode_enum.TestBench_Regular ||
                    Mode == Mode_enum.TestBench_DS ||
                    Mode == Mode_enum.ME_Regular ||
                    Mode == Mode_enum.ME_DS ||
                    Mode == Mode_enum.SoT_DS)
                {
                    if (currentobjInvoked != null)
                    {
                        Workflow = GetWorkFlow(currentobjInvoked);
                    }
                }
                else if (Mode == Mode_enum.CA_Regular ||
                    Mode == Mode_enum.CA_DS)
                {
                    Workflow = GetValidInterpreters();
                }

                // init the main form
                mf.CyPhy = this;
                mf.ProjectPath = ProjectPath;
                mf.OutputDir = OutputDir;
                mf.InitForm();
                mf.ShowDialog();

                // set up variables based on user choice on the main form
                OutputDir = mf.OutputDir;
                saveTestBenches = mf.chbSaveTestBenches.Checked;
            }
            catch (Exception ex)
            {
                MessageBox.Show(
                    ex.Message,
                    "Exception",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
            }
        }

        bool saveTestBenches { get; set; }

        [ComVisible(false)]
        public void SetOutputDir(MgaProject project, MgaFCO currentobj)
        {
            this.ProjectPath = Path.GetDirectoryName(project.ProjectConnStr.Substring("MGA=".Length));

            string subDir = "";

            if (currentobj != null)
            {
                subDir = currentobj.Name;
            }
            else
            {
                subDir = project.Name;
            }

            // set the default output directory
            //OutputDir = Path.Combine(ProjectPath, "Output", subDir);
            // TODO: get this directory from the project's manifest file.
            ResultsDir = Path.Combine(this.ProjectPath, "results");
            //OutputDir = Path.Combine(ResultsDir, subDir);
            OutputDir = Path.GetFullPath(Path.Combine(ResultsDir));
        }

        private Queue<ComComponent> GetValidInterpreters(string paradigm = "CyPhyML")
        {
            Queue<ComComponent> ComComponents = new Queue<ComComponent>();

            MgaRegistrar registrar = new MgaRegistrar();
            regaccessmode_enum r = regaccessmode_enum.REGACCESS_BOTH;
            IEnumerable a = (IEnumerable)(object)registrar.GetComponentsDisp(r);

            foreach (string comProgId in a)
            {
                bool isAssociated;
                bool canAssociate;

                registrar.IsAssociated(comProgId, paradigm, out isAssociated, out canAssociate, r);
                string DllPath = string.Empty;

                try
                {
                    DllPath = registrar.LocalDllPath[comProgId];
                }
                catch (System.Runtime.InteropServices.COMException ex)
                {
                    Trace.TraceError("DllPath is empty or not found for ProgID: {0}", comProgId);
                    Trace.TraceError(ex.ToString());
                }

                if (string.IsNullOrWhiteSpace(DllPath) |
                    File.Exists(DllPath) == false)
                {
                    Trace.TraceError("DllPath is empty or the path does not exist for ProgID: {0}, path {1}",
                        comProgId,
                        DllPath);
                    continue;
                }

                componenttype_enum Type;
                string desc;
                registrar.QueryComponent(
                    comProgId,
                    out Type,
                    out desc,
                    regaccessmode_enum.REGACCESS_BOTH);

                bool isInterpreter = false;
                isInterpreter = (Type == componenttype_enum.COMPONENTTYPE_INTERPRETER);

                if (canAssociate &&
                    File.Exists(DllPath) &&
                    isInterpreter)
                {
                    ComComponent component = new ComComponent(comProgId);
                    if (component.isValid)
                    {
                        ComComponents.Enqueue(component);
                    }
                }
            }

            return ComComponents;
        }

        bool validTestBench = true;

        /// <summary>
        /// Determines the actual mode, and gets the possible configrations 
        /// for the specified design space if ther is any.
        /// </summary>
        /// <param name="project"></param>
        /// <param name="currentobj"></param>
        /// <param name="selectedobjs"></param>
        /// <param name="componentStartMode"></param>
        private void GetConfigModels(
                MgaProject project,
                MgaFCO currentobj,
                MgaFCOs selectedobjs,
                ComponentStartMode componentStartMode)
        {
            if (currentobj == null)
            {
                if (selectedobjs.Cast<MgaFCO>().
                    All(x =>
                        x.MetaBase.Name == "ComponentAssembly" ||
                        x.MetaBase.Name == "Component"))
                {
                    Mode = Mode_enum.CA_Regular;
                    foreach (MgaFCO item in selectedobjs)
                    {
                        mf.AddExportedCAItem(new SelectedItem(item));
                    }
                }
                else if (selectedobjs.Cast<MgaFCO>().
                     All(x => x.MetaBase.Name == "TestBench" || x.MetaBase.Name == "CADTestBench"))
                {
                    Mode = Mode_enum.TestBench_Regular;
                    foreach (MgaFCO item in selectedobjs)
                    {
                        mf.AddExportedCAItem(new SelectedItem(item));
                    }
                }
                return;
            }

            Trace.TraceInformation("Current object: {0}", currentobj.AbsPath);

            tbg.Console = GMEConsole;
            ConfigModels.Clear();
            ConfigModels = tbg.GetConfigurationModels(currentobj);
            DesignConfigs.Clear();
            ConfigModels.ForEach(x => DesignConfigs.Add(new DesignConfigItem(x)));

            if (currentobj.MetaBase.Name == "TestBench" ||
                currentobj.MetaBase.Name == "CADTestBench")
            {
                IEnumerable<MgaReference> tlsuts = currentobj.ChildObjects.
                    Cast<MgaObject>().
                    OfType<MgaReference>().
                    Where(x => x.MetaBase.Name == "TopLevelSystemUnderTest");

                MgaReference tlsut = tlsuts.FirstOrDefault();

                // - Put all detailed messages and hyperlinks on the console. 
                // Criteria for valid Test Benches:

                // 1. One and only one SUT (Compound* DesignContainer) 
                if (tlsuts.Count() != 1)
                {
                    validTestBench = false;
                    StringBuilder sb = new StringBuilder();
                    sb.AppendFormat("Test Bench must contain one and only one TopLevelSystemUnderTest. Found:");
                    foreach (var item in tlsuts)
                    {
                        string formatStr = " {0}";
                        if (item.Referred == null)
                        {
                            formatStr += " [NULL reference]";
                        }
                        sb.AppendFormat(formatStr, GMEConsole.MakeObjectHyperlink(item, item.Name));
                        if (item != tlsuts.LastOrDefault())
                        {
                            sb.AppendFormat(",");
                        }
                    }
                    GMEConsole.Error.WriteLine(sb.ToString());
                }

                // 2. SUT is not a null ref 
                if (tlsuts.Count() == 1 && tlsut.Referred == null)
                {
                    validTestBench = false;

                    GMEConsole.Error.WriteLine(
                        "TopLevelSystemUnderTest {0} is a NULL reference. It must point to a valid DesignContainer or ComponentAssembly.",
                        GMEConsole.MakeObjectHyperlink(tlsut, tlsut.Name));
                }

                // 3. TIPs cannot be null refs 
                IEnumerable<MgaReference> tips = currentobj.ChildObjects.
                    Cast<MgaObject>().
                    OfType<MgaReference>().
                    Where(x => x.MetaBase.Name == "TestInjectionPoint");

                if (tlsuts.Count() == 1 && tlsut.Referred != null)
                {
                    foreach (var item in tips)
                    {
                        if (item.Referred == null)
                        {
                            validTestBench = false;

                            string formatStr = "TestInjectionPoint {0} is a NULL reference. It must point to a descendant of the TopLevelSystemUnderTest {1}.";
                            GMEConsole.Error.WriteLine(
                                formatStr,
                                GMEConsole.MakeObjectHyperlink(item, item.Name),
                                GMEConsole.MakeObjectHyperlink(tlsut.Referred, tlsut.Referred.Name));
                        }
                    }
                }

                // 4. All TIP must point to a descendant of the SUT 
                if (tlsuts.Count() == 1)
                {
                    if (tlsut.Referred != null)
                    {
                        foreach (var item in tips)
                        {
                            if (item.Referred != null)
                            {
                                MgaModel parent = item.Referred.ParentModel;
                                while (parent != null &&
                                    parent.RootFCO != parent &&
                                    parent != tlsut.Referred)
                                {
                                    parent = parent.ParentModel;
                                }
                                if (parent != tlsut.Referred ||
                                    item.Referred.RootFCO != tlsut.Referred.RootFCO)
                                {
                                    validTestBench = false;
                                    string formatStr = "TestInjectionPoint {0} must point to a descendant of the TopLevelSystemUnderTest {1}. Currently it points to a descendant of {2}.";
                                    GMEConsole.Error.WriteLine(
                                        formatStr,
                                        GMEConsole.MakeObjectHyperlink(item, item.Name),
                                        GMEConsole.MakeObjectHyperlink(tlsut.Referred, tlsut.Referred.Name),
                                        GMEConsole.MakeObjectHyperlink(item.Referred.ExGetParent(), item.Referred.ExGetParent().Name));
                                }
                            }
                        }
                    }
                }

                // - The main form should not be displayed. 

                // - Dialog box should tell to the user that they need to look at the console for detailed information. 
                if (validTestBench == false)
                {
                    System.Windows.Forms.Application.DoEvents();

                    MessageBox.Show(
                        "Invalid TestBench structure. See console for details.",
                        "Error",
                        MessageBoxButtons.OK,
                        MessageBoxIcon.Error);
                    return;
                }



                if (tlsut.Referred.MetaBase.Name == "ComponentAssembly")
                {
                    Mode = Mode_enum.TestBench_Regular;
                    mf.AddExportedCAItem(new ExportedCaItem(tlsut));
                }
                if (tlsut.Referred.MetaBase.Name == "DesignContainer")
                {
                    Mode = Mode_enum.TestBench_DS;
                }
            }
            else if (currentobj.MetaBase.Name == "Component" ||
                currentobj.MetaBase.Name == "ComponentAssembly")
            {
                Mode = Mode_enum.CA_Regular;
            }
            else if (currentobj.MetaBase.Name == "DesignContainer")
            {
                Mode = Mode_enum.CA_DS;
            }
            else if (currentobj.MetaBase.Name == "ParametricExploration")
            {
                //throw new NotSupportedException("This feature is not supported yet.");
                IEnumerable<MgaReference> testBenchRefsLocal = currentobj.ChildObjects.
                    Cast<MgaFCO>().
                    OfType<MgaReference>().
                    Where(x => x.MetaBase.Name == "TestBenchRef");

                int NoSut = 0;
                bool SutPointsToDS = false;
                int nSutPointsToDS = 0;

                designSpace = null;

                foreach (MgaReference item in testBenchRefsLocal)
                {
                    if (item.Referred == null)
                    {
                        throw new NullReferenceException("TestBenchRef is referred to NULL.");
                    }

                    this.testBenchRefs.Add(item as MgaReference);

                    IEnumerable<MgaReference> refs = item.Referred.ChildObjects.
                        Cast<MgaFCO>().
                        OfType<MgaReference>().
                        Where(x => x.MetaBase.Name == "TopLevelSystemUnderTest");

                    int count = refs.Count();
                    if (count == 0)
                    {
                        NoSut++;
                    }
                    else if (count == 1)
                    {
                        // one sut
                        if (refs.FirstOrDefault().Referred == null)
                        {
                            throw new NullReferenceException(
                                "TestBench has a TopLevelSystemUnderTest null reference.");
                        }
                        if (refs.FirstOrDefault().Referred.MetaBase.Name == "ComponentAssembly")
                        {
                            if (designSpace != null)
                            {
                                throw new NullReferenceException("TestBenches are not consistent.");
                            }
                            SutPointsToDS = false;
                        }
                        else if (refs.FirstOrDefault().Referred.MetaBase.Name == "DesignContainer")
                        {
                            SutPointsToDS = true;
                            nSutPointsToDS++;
                            MgaFCO currentDS = GetDesignSpace(refs.FirstOrDefault().Referred);
                            if (designSpace != null)
                            {
                                if (currentDS != designSpace)
                                {
                                    throw new Exception(
                                        "TestBenches are referred to different design spaces.");
                                }
                            }
                            else
                            {
                                designSpace = currentDS;
                            }
                        }
                    }
                    else
                    {
                        throw new Exception(
                            "TestBench could not contain more than one TopLevelSystemUnderTest.");
                    }
                }

                if (NoSut == testBenchRefsLocal.Count())
                {
                    // there is no sut in the referred test benches.
                    Mode = Mode_enum.ME_Regular;
                }
                else if (SutPointsToDS == false)
                {
                    // there is suts but those point into a Component Assembly not a DS.
                    Mode = Mode_enum.ME_Regular;
                }
                else if (SutPointsToDS == true &&
                    (nSutPointsToDS + NoSut == testBenchRefsLocal.Count()))
                {
                    Mode = Mode_enum.ME_DS;
                }
                else
                {
                    throw new Exception(
                        "Invalid usage: Some test benches refer to DesignSpaces some refer to ComponentAssemblies.");
                }
            }
            else if (currentobj.MetaBase.Name == "TestBenchSuite")
            {
                // no TB null ref
                var tbRefs = (currentobj as MgaModel)
                    .ChildObjects
                    .OfType<MgaReference>()
                    .Where(x => x.Meta.Name == "TestBenchRef");

                var tbs = tbRefs.Where(x => x.Referred != null).Select(x => x.Referred);

                if (tbs.Count() != tbRefs.Count())
                {
                    GMEConsole.Error.WriteLine("Some test benches are null references.");
                    throw new Exception("Invalid usage: TestBenchSuite.");
                }

                this.testBenchRefs.AddRange(tbRefs.Where(x => x.Referred != null));

                List<MgaFCO> uniqueTBs = new List<MgaFCO>();
                // one test bench can be contained only once.
                foreach (var item in tbs)
                {
                    if (uniqueTBs.FirstOrDefault(x => x.ID == item.ID) == null)
                    {
                        uniqueTBs.Add(item);
                    }
                    else
                    {
                        GMEConsole.Error.WriteLine("One test bench can be contained only once. {0}", item.ToMgaHyperLink());
                        throw new Exception("Invalid usage: TestBenchSuite.");
                    }
                }

                MgaFCO referredRootDesignNode = null;

                foreach (var tb in tbs)
                {
                    var suts = tb.ChildObjects.Cast<IMgaFCO>().Where(x => x.MetaBase.Name == "TopLevelSystemUnderTest");

                    if (suts.Count() != 1)
                    {
                        // referenced test benches have one system under test object
                        string msg = string.Format("Test bench must have one system under test object. {0}",
                            tb.ToMgaHyperLink());

                        GMEConsole.Error.WriteLine(msg);
                        Trace.TraceError(msg);
                    }
                    else
                    {
                        if ((suts.FirstOrDefault() as MgaReference).Referred.RootFCO.MetaBase.Name != "DesignContainer")
                        {
                            // cannot point to a design space
                            GMEConsole.Error.WriteLine(
                                "Test bench's system under test object: {0} must point to a design space {1}.",
                                tb.ToMgaHyperLink(),
                                (suts.FirstOrDefault() as MgaReference).Referred.RootFCO.ToMgaHyperLink());
                            throw new Exception("Invalid usage: TestBenchSuite.");
                        }

                        // referenced test bechnes have system under test object referred to the same design
                        if (referredRootDesignNode == null)
                        {
                            referredRootDesignNode = (suts.FirstOrDefault() as MgaReference).Referred.RootFCO;
                        }
                        else if (referredRootDesignNode.ID != (suts.FirstOrDefault() as MgaReference).Referred.RootFCO.ID)
                        {
                            GMEConsole.Error.WriteLine(
                                "Test bench: {0} refers to {1} design, while other test benches refer to: {2}",
                                tb.ToMgaHyperLink(),
                                (suts.FirstOrDefault() as MgaReference).Referred.RootFCO.ToMgaHyperLink(),
                                referredRootDesignNode.ToMgaHyperLink());
                        }
                    }
                }

                Mode = Mode_enum.SoT_DS;
            }
        }

        /// <summary>
        /// Gets the defined workflow based on the contained workflow Ref
        /// in the current object container
        /// </summary>
        /// <param name="currentobj"></param>
        /// <returns></returns>
        [ComVisible(false)]
        public Queue<ComComponent> GetWorkFlow(MgaFCO currentobj, Mode_enum? mode=null)
        {
            if (mode == null)
                mode = this.Mode;
            Queue<ComComponent> workflow = new Queue<ComComponent>();

            if (mode == Mode_enum.ME_DS || mode == Mode_enum.ME_Regular)
            {
                workflow.Enqueue(new ComComponent("MGA.Interpreter.CyPhyPET", true));
                return workflow;
            }

            if (mode == Mode_enum.SoT_DS)
            {
                workflow.Enqueue(new ComComponent("MGA.Interpreter.CyPhySoT", true));
                return workflow;
            }

            List<MgaReference> WorkflowRefs = currentobj.
                ExKindChildren("WorkflowRef").
                Cast<MgaReference>().
                ToList();

            if (WorkflowRefs.Count > 1)
            {
                // not supported
                throw new NotImplementedException("WorkflowRef is greater than 1.");
            }

            if (WorkflowRefs.Count == 1)
            {
                workflow.Clear();
                MgaReference wfRef = WorkflowRefs[0] as MgaReference;
                Queue<string> items = new Queue<string>();
                List<MgaAtom> tasks = new List<MgaAtom>();
                List<MgaFCO> processed = new List<MgaFCO>();

                if (wfRef.Referred != null)
                {
                    tasks.AddRange(wfRef.Referred.ChildObjects.OfType<MgaAtom>());

                    MgaAtom StartTask = null;

                    StartTask = tasks.
                            Where(x => x.ExSrcFcos().Count() == 0).
                            FirstOrDefault();

                    if (StartTask != null)
                    {
                        MgaFCO NextTask = null;

                        var component = new ComComponent(StartTask.StrAttrByName["COMName"]);
                        string Parameters = StartTask.StrAttrByName["Parameters"];
                        try
                        {
                            component.Parameters = (Dictionary<string, string>)JsonConvert.DeserializeObject(Parameters, typeof(Dictionary<string, string>));
                        }
                        catch (JsonReaderException)
                        {
                        }
                        workflow.Enqueue(component);

                        processed.Add(StartTask as MgaFCO);

                        NextTask = StartTask.ExDstFcos().FirstOrDefault();

                        // avoid loops
                        while (NextTask != null &&
                                processed.Contains(NextTask) == false)
                        {
                            component = new ComComponent(NextTask.StrAttrByName["COMName"]);
                            Parameters = NextTask.StrAttrByName["Parameters"];
                            try
                            {
                                component.Parameters = (Dictionary<string, string>)JsonConvert.DeserializeObject(Parameters, typeof(Dictionary<string, string>));
                            }
                            catch (JsonReaderException)
                            {
                            }
                            workflow.Enqueue(component);

                            NextTask = NextTask.ExDstFcos().FirstOrDefault();
                        }
                    }
                }
            }
            else if (WorkflowRefs.Count == 0)
            {
                workflow.Clear();

                if (currentobj.Meta.Name == "CADTestBench")
                {
                    // use CyPhy2CAD by default
                    workflow.Enqueue(new ComComponent("MGA.Interpreter.CyPhy2CAD", true));
                }
                else
                {
                    // use NOTHING by default see JIRA
                }
            }
            return workflow;
        }

        /// <summary>
        /// Gets the Root element of a design space, where the parent is not a model.
        /// </summary>
        /// <param name="mgaFCO"></param>
        /// <returns></returns>
        private MgaFCO GetDesignSpace(MgaFCO mgaFCO)
        {
            if (mgaFCO.ParentModel == null)
            {
                return mgaFCO;
            }
            else
            {
                return GetDesignSpace(mgaFCO.ParentModel as MgaFCO);
            }
        }


        #endregion

        #region IMgaComponentEx Members

        public MgaGateway MgaGateway { get; set; }
        internal GMEConsole GMEConsole { get; set; }
        [ComVisible(false)]
        public System.IO.TextWriter InfoWriter { get; set; }
        [ComVisible(false)]
        public System.IO.TextWriter ErrorWriter { get; set; }

        public MgaProject projectInvoked = null;
        internal MgaFCO currentobjInvoked = null;
        internal MgaFCOs selectedobjsInvoked = null;
        internal int paramInvoked = 0;

        public void InvokeEx(
                MgaProject project,
                MgaFCO currentobj,
                MgaFCOs selectedobjs,
                int param)
        {
            string currentWorkDir = System.IO.Directory.GetCurrentDirectory();
            try
            {
                System.IO.Directory.SetCurrentDirectory(Path.GetDirectoryName(project.ProjectConnStr.Substring("MGA=".Length)));
                StartTime = DateTime.Now;
                projectInvoked = project;
                currentobjInvoked = currentobj;
                selectedobjsInvoked = selectedobjs;
                paramInvoked = param;

                isProcessStaticMetrics = false;

                MethodDelegateCopy dashboardCopy = new MethodDelegateCopy(DirectoryCopy);
                IAsyncResult ar = dashboardCopy.BeginInvoke(VersionInfo.DashboardPath, "dashboard", true, null, null);

                InvokeEx2(project, currentobj, selectedobjs, param);

                dashboardCopy.EndInvoke(ar);

                projectInvoked = null;
                currentobjInvoked = null;
                selectedobjsInvoked = null;
                //Marshal.FinalReleaseComObject(this.GMEConsole.gme);

                this.GMEConsole = null;
                this.MgaGateway = null;
                this.tbg = null;

                GC.Collect();
                GC.WaitForPendingFinalizers();
                GC.Collect();
                GC.WaitForPendingFinalizers();
            }
            finally
            {
                System.IO.Directory.SetCurrentDirectory(currentWorkDir);
            }
            //finally
            // chdir(cwd)
        }

        private delegate void MethodDelegateCopy(
            string sourceDirName,
            string destDirName,
            bool copySubDirs);

        private static void DirectoryCopy(
            string sourceDirName,
            string destDirName,
            bool copySubDirs)
        {
            DirectoryInfo dir = new DirectoryInfo(sourceDirName);
            DirectoryInfo[] dirs = dir.GetDirectories();

            if (!dir.Exists)
            {
                throw new DirectoryNotFoundException(
                    "Source directory does not exist or could not be found: "
                    + sourceDirName);
            }

            if (!Directory.Exists(destDirName))
            {
                Directory.CreateDirectory(destDirName);
            }

            FileInfo[] files = dir.GetFiles();
            foreach (FileInfo file in files)
            {
                string temppath = Path.Combine(destDirName, file.Name);
                file.CopyTo(temppath, true);
            }

            if (copySubDirs)
            {
                foreach (DirectoryInfo subdir in dirs)
                {
                    string temppath = Path.Combine(destDirName, subdir.Name);
                    DirectoryCopy(subdir.FullName, temppath, copySubDirs);
                }
            }
        }

        [ComVisible(false)]
        public void InvokeEx2(
                MgaProject project,
                MgaFCO currentobj,
                MgaFCOs selectedobjs,
                int param)
        {
            Contract.Requires(project != null);
            if (!enabled)
            {
                return;
            }

            int numCores = Environment.ProcessorCount;
            string ProjectPath;
            ProjectPath = Path.GetDirectoryName(project.ProjectConnStr.Substring("MGA=".Length));
            loggingFileName = Path.Combine(
                ProjectPath,
                "log",
                this.ComponentName + "." + Process.GetCurrentProcess().Id + ".log");

            if (Directory.Exists(Path.GetDirectoryName(loggingFileName)) == false)
            {
                Directory.CreateDirectory(Path.GetDirectoryName(loggingFileName));
            }

            META.Logger.AddFileListener(loggingFileName, this.ComponentName);

            try
            {
                GMEConsole = GMEConsole.CreateFromProject(project);
                InfoWriter = GMEConsole.Info;
                ErrorWriter = GMEConsole.Error;
                MgaGateway = new MgaGateway(project);
                project.CreateTerritoryWithoutSink(out MgaGateway.territory);
                mf = new MainForm();
                tbg = new TestBenchGateway();
                tbg.CyPhy = this;
                //tbg.Extract(null);
                GMEConsole.Clear();
                System.Windows.Forms.Application.DoEvents();

                // create workflow
                Workflow = new Queue<ComComponent>();

                Mode = Mode_enum.Undefined;

                // get configurations and determine mode
                MgaGateway.PerformInTransaction(delegate
                {
                    Trace.TraceInformation("Model file: {0}", project.ProjectConnStr);
                    Trace.TraceInformation("Paradigm file: {0}", project.ParadigmConnStr);

                    SetOutputDir(project, currentobj);
                    GetConfigModels(
                            project,
                            currentobj,
                            selectedobjs,
                            Convert(param));
                },
                transactiontype_enum.TRANSACTION_NON_NESTED);

                if (validTestBench == false)
                {
                    return;
                }

                // pre-actions
                MgaGateway.PerformInTransaction(delegate
                {
                    ShowDialog(
                            project,
                            currentobj,
                            selectedobjs,
                            Convert(param));
                },
                transactiontype_enum.TRANSACTION_NON_NESTED);

                // pre-actions
                if (mf.DialogResult != DialogResult.OK)
                {
                    GMEConsole.Warning.WriteLine("Operation was cancelled by the user");
                    return;
                }

                string projectName = "";

                string avmProjDir = Path.Combine(Path.GetDirectoryName(project.ProjectConnStr.Substring("MGA=".Length)));

                MgaGateway.PerformInTransaction(delegate
                {
                    MgaModel rootDS = null;
                    if (DesignConfigs != null)
                    {
                        var first = DesignConfigs.FirstOrDefault();
                        if (first != null)
                        {
                            rootDS = first.DesignConfig.ParentModel;
                        }
                    }

                    // create avmproj
                    this.avmProj = AVM.DDP.MetaAvmProject.Create(avmProjDir, project, rootDS, this.GMEConsole);
                },
                transactiontype_enum.TRANSACTION_NON_NESTED);

                GMEConsole.Info.WriteLine("Generating test benches. Please wait...");

                string currentObjectName = string.Empty;
                string testBenchUniqueName = string.Empty;
                // main action
                MgaGateway.PerformInTransaction(delegate
                {
                    Main(project, currentobj, selectedobjs, Convert(param));
                    currentObjectName = currentobj.Name;
                    testBenchUniqueName = currentobj.RegistryValue["TestBenchUniqueName"];
                    // overwrite TestBenchUniqueName
                    if (currentobj.Meta.Name == "ParametricExploration")
                    {
                        var pe = ISIS.GME.Dsml.CyPhyML.Classes.ParametricExploration.Cast(currentobj);
                        var tbRef = pe.Children.TestBenchRefCollection.FirstOrDefault();
                        if (tbRef != null)
                        {
                            if (tbRef.Referred.TestBench != null)
                            {
                                testBenchUniqueName = (tbRef.Referred.TestBench.Impl as MgaFCO).RegistryValue["TestBenchUniqueName"];
                            }
                        }
                    }
                },
                transactiontype_enum.TRANSACTION_NON_NESTED);

                // format string for elements
                string Format = "> [{0:d" + toProcess.Count.ToString().Count() + "}/{1}] {2}";
                int idx = 0;

                bool isCyPhySoT = false;
                List<string> cyphySoTDirectories = new List<string>();

                // call the interpreters
                foreach (var item in toProcess)
                {
                    GMEConsole.Info.WriteLine(
                            Format,
                            idx + 1,
                            toProcess.Count,
                            item.Key.Name);
                    idx++;

                    System.Windows.Forms.Application.DoEvents();

                    foreach (var interpreter in item.Value)
                    {
                        if (isCyPhySoT == false &&
                            interpreter.Name == "MGA.Interpreter.CyPhySoT")
                        {
                            // save a flag it is CyPhySoT
                            isCyPhySoT = true;
                        }
                        // 
                        if (interpreter.Parameters == null)
                        {
                            interpreter.Parameters = new Dictionary<string, string>();
                        }

                        interpreter.Parameters["TestBenchUniqueName"] = testBenchUniqueName;
                    }

                    var automation = true;
                    bool needConfig = item.Key == toProcess.FirstOrDefault().Key;

                    if (isCyPhySoT)
                    {
                        // if interpreter is CyPhySoT
                        // run CyPhySoT
                        // set post to job manager false

                        if (needConfig)
                        {
                            MgaGateway.PerformInTransaction(delegate
                            {
                                Dictionary<string, ComComponent> components = new Dictionary<string, ComComponent>();
                                foreach (MgaFCO testbench in OriginalSubject.ChildObjects.Cast<MgaFCO>().Where(c => c.MetaRole.Name == "TestBenchRef").Cast<MgaReference>().Select(c => c.Referred))
                                {
                                    foreach (ComComponent component in GetWorkFlow(testbench, Mode_enum.TestBench_Regular))
                                    {
                                        components[component.ProgId] = component;
                                    }
                                }
                                foreach (ComComponent component in components.Values)
                                {
                                    component.MgaComponent.ComponentParameter["do_config_now"] = OriginalSubject.Project;
                                }
                            });
                        }

                        var outputDirs = CallInterpreters(item, false, automation, needConfig);

                        // collect output directories
                        // save a list of output directories

                        var success = EnqueueSoT(outputDirs);

                        if (success)
                        {
                            InfoWriter.WriteLine("=> Sot was posted: {0}", item.Key.Name);
                            System.Windows.Forms.Application.DoEvents();
                        }
                        else
                        {
                            ErrorWriter.WriteLine("=> Sot cannot be posted: {0}", item.Key.Name);
                            System.Windows.Forms.Application.DoEvents();
                        }
                    }
                    else
                    {
                        CallInterpreters(item, mf.chbPostJobs.Checked, automation, needConfig);
                    }

                    MgaGateway.PerformInTransaction(delegate
                    {
                        string interpreterOutputDir = OutputDir;

                        if (item.Value.FirstOrDefault() != null)
                        {
                            interpreterOutputDir = item.Value.FirstOrDefault().Parameters["output_dir"];
                        }

                        this.avmProj.SaveSummaryReportJson(interpreterOutputDir, item.Key.Fco);

                        var saved = this.avmProj.SaveDesign(item.Key.Fco);
                        if (saved)
                        {
                            GMEConsole.Info.WriteLine("{0} was saved.", item.Key.Fco.Name);
                        }
                        else
                        {
                            GMEConsole.Error.WriteLine("{0} was not saved.", item.Key.Fco.Name);
                        }
                    });


                    if (isCyPhySoT)
                    {
                        AddSoTs();
                    }
                    else
                    {
                        AddJobs();
                    }
                }

                RemoveTestBenches();

                this.avmProj.Serialize();

                // Generate python scripts if not already there
                string export_for_dashboard_scoring = Path.GetFullPath(
                    Path.Combine(this.ProjectPath, "export_for_dashboard_scoring.py"));

                if (File.Exists(export_for_dashboard_scoring) == false)
                {
                    using (StreamWriter writer = new StreamWriter(export_for_dashboard_scoring))
                    {
                        writer.WriteLine(Properties.Resources.export_for_dashboard_scoring);
                    }
                }

                // Check if the "stats" directory exists; if not, make it
                string stat_dir_path = Path.GetFullPath(
                    Path.Combine(this.ProjectPath, "stats"));

                if (Directory.Exists(stat_dir_path) == false)
                {
                    Directory.CreateDirectory(stat_dir_path);
                }

                string gather_stat_json = Path.GetFullPath(
                    Path.Combine(stat_dir_path, "gather_stat_json.py"));

                if (File.Exists(gather_stat_json))
                {
                    File.Delete(gather_stat_json);
                }
                using (StreamWriter writer = new StreamWriter(gather_stat_json))
                {
                    writer.WriteLine(Properties.Resources.gather_stat_json);
                }

                // Check if the "log" folder exists; if not, create it
                string log_folder_path = Path.GetFullPath(
                    Path.Combine(this.ProjectPath, "log"));

                if (Directory.Exists(log_folder_path) == false)
                {
                    Directory.CreateDirectory(log_folder_path);
                }

                string gather_all_logfiles = Path.GetFullPath(
                    Path.Combine(log_folder_path, "gather_all_logfiles.py"));

                if (File.Exists(gather_all_logfiles))
                {
                    File.Delete(gather_all_logfiles);
                }
                using (StreamWriter writer = new StreamWriter(gather_all_logfiles))
                {
                    writer.WriteLine(Properties.Resources.gather_all_logfiles);
                }
                
                /*
                string clean_up = Path.GetFullPath(
                    Path.Combine(this.ProjectPath, "clean_up.py"));
                if (File.Exists(clean_up) == false)
                {
                    using (StreamWriter writer = new StreamWriter(clean_up))
                    {
                        writer.WriteLine(Properties.Resources.clean_up);
                    }
                }
                 */

                index_html index = new index_html();
                index.ProjectName = projectName;
                index.AvmProjectFileName = Path.GetFileName(this.avmProj.m_filename);

                string indexFileName = Path.GetFullPath(Path.Combine(this.ProjectPath, "index.html"));

                using (StreamWriter writer = new StreamWriter(indexFileName))
                {
                    string index_content = index.TransformText();
                    writer.WriteLine(index_content);

                    if (mf.chbOpenDashboard.Checked)
                    {
                        OpenWithChrome(indexFileName);
                    }
                }

                // statistics
                GMEConsole.Info.WriteLine(
                        "Duration: {0}.",
                        sw.Elapsed.ToString("c"));

                Trace.TraceInformation("{0} has finished.", this.ComponentName);
            }
            catch (Exception ex)
            {
                Trace.TraceError(ex.ToString());

                GMEConsole.Error.WriteLine(
                    "Log file is here: <a href=\"file:///{1}\" target=\"_blank\">{1}</a>\\{0}",
                    Path.GetFileName(loggingFileName),
                    Path.GetDirectoryName(loggingFileName));
            }
            finally
            {
                if (mf != null)
                    mf.Dispose();
                // clean up
                project = null;
                currentobj = null;
                selectedobjs = null;
                componentParameters = null;
                if (mf != null)
                {
                    mf.CyPhy = null;
                    mf.Dispose();
                }
                mf = null;
                if (tbg != null)
                {
                    tbg.CyPhy = null;
                }
                tbg = null;
                Workflow = null;

                ConfigModels = null;
                DesignConfigs = null;
                SelectedItems = null;

                GC.Collect();
                GC.WaitForPendingFinalizers();
                GC.Collect();
                META.Logger.RemoveFileListener(this.ComponentName);
            }
        }

        private void OpenWithChrome(string indexFileName)
        {
            // try to get this value better HKLM + 32 vs 64bit...

            // chrome is installed for all users
            // http://msdn.microsoft.com/en-us/library/windows/desktop/ee872121(v=vs.85).aspx#app_exe
            string keyName = @"HKEY_LOCAL_MACHINE\SOFTWARE\Microsoft\Windows\CurrentVersion\App Paths\chrome.exe";

            string chromePath = (string)Microsoft.Win32.Registry.GetValue(
                keyName,
                "Path",
                "ERROR: " + keyName + " InstallLocation does not exist!");

            if (chromePath.StartsWith("ERROR"))
            {
                // Installed only for this user
                keyName = @"HKEY_CURRENT_USER\SOFTWARE\Microsoft\Windows\CurrentVersion\App Paths\chrome.exe";

                chromePath = (string)Microsoft.Win32.Registry.GetValue(
                    keyName,
                    "Path",
                    "ERROR: " + keyName + " InstallLocation does not exist!");
            }

            if (chromePath.StartsWith("ERROR"))
            {
                GMEConsole.Warning.WriteLine("Local Dashboard requires Chrome. Chrome was not found.");
                Trace.TraceWarning("Local Dashboard requires Chrome. Chrome was not found.");
            }
            else
            {
                string chromeExe = Path.Combine(chromePath, "chrome.exe");

                Process p = new Process();
                p.StartInfo = new ProcessStartInfo()
                {
                    Arguments = " --allow-file-access-from-files --new-window \"" + Path.GetFullPath(indexFileName) + "\"",
                    FileName = chromeExe,
                    UseShellExecute = false
                };

                // open dashboard
                p.Start();
            }
        }

        private void RemoveTestBenches()
        {
            if (saveTestBenches == false)
            {
                MgaGateway.PerformInTransaction(delegate
                {
                    // remove all test bench folders that got generated
                    foreach (var testBenchFolder in tbg.NewTestBenchFolders)
                    {
                        testBenchFolder.DestroyObject();
                    }

                    MgaModel model = this.expandedParametricExplorations.FirstOrDefault();
                    if (model != null)
                    {
                        model.ParentFolder.DestroyObject();
                    }

                    // remove CyPhySoTs
                    model = this.expandedSoTs.FirstOrDefault();
                    if (model != null)
                    {
                        model.ParentFolder.DestroyObject();
                    }
                });
            }
        }


        /// <summary>
        /// Calls all interpreters in the queue and passes the given fco 
        /// as a current obj to the interpreters.
        /// </summary>
        /// <param name="kvp"></param>
        [ComVisible(false)]
        public List<string> CallInterpreters(
            KeyValuePair<FCO, Queue<ComComponent>> kvp,
            bool queueToJobManager,
            bool automation = false,
            bool needConfig = false)
        {
            Contract.Requires(kvp.Key.Fco != null);
            Contract.Requires(string.IsNullOrEmpty(kvp.Key.Name) == false);

            List<string> outputDirectories = new List<string>();

            MgaFCO currentObj = kvp.Key.Fco;

            // define common interpreter parameters
            ComComponent.CommonParameters.Clear();
            if (automation)
            {
                ComComponent.CommonParameters.Add("automation", "true");
            }
            ComComponent.CommonParameters.Add("do_config", needConfig ? "true" : "false");
            ComComponent.CommonParameters.Add("console_messages", "off");


            string OutputSubDir = "";

            string Format = ">> [{0:d" + kvp.Value.Count.ToString().Count() + "}/{1}] {2}";
            int idx = 0;

            if (kvp.Value.Count == 0)
            {
                ErrorWriter.WriteLine("No interpreters were defined.");
                System.Windows.Forms.Application.DoEvents();
            }

            // call the interpreters
            foreach (var interpreter in kvp.Value)
            {
                InfoWriter.WriteLine(
                            Format,
                            idx + 1,
                            kvp.Value.Count,
                            interpreter.Name);

                idx++;

                if (interpreter.Parameters == null)
                {
                    interpreter.Parameters = new Dictionary<string, string>();
                }

                if (kvp.Key.Fco != currentobjInvoked)
                {
                    // output dir %root%/%currentobjInvoked%/%this object%/%interpreter name%/
                    //OutputSubDir = Path.Combine(OutputDir, kvp.Key.Name, item.Name);

                    // We need shorter path due windows' path limitations

                    string randomFolderName = Path.GetFileNameWithoutExtension(Path.GetRandomFileName());
                    List<char> illegalStartChars = new List<char>()
                    {
                        '0','1','2','3','4','5','6','7','8','9'
                    };

                    OutputSubDir = Path.Combine(
                        OutputDir,
                        randomFolderName);

                    int maxFolders = 0;

                    while (illegalStartChars.Contains(randomFolderName.FirstOrDefault()) ||
                        File.Exists(OutputSubDir) ||
                        Directory.Exists(OutputSubDir))
                    {
                        if (maxFolders++ > 2000000)
                        {
                            throw new Exception(
                                string.Format("Number of tries ({0}) to create an output folder exceeded in {0}. ",
                                    maxFolders,
                                    OutputDir));
                        }

                        randomFolderName = Path.GetFileNameWithoutExtension(Path.GetRandomFileName());

                        OutputSubDir = Path.Combine(
                            OutputDir,
                            randomFolderName);
                    }

                    outputDirectories.Add(OutputSubDir);

                    MgaGateway.PerformInTransaction(delegate
                    {
                        interpreter.Parameters["TestBench"] = @".\test-benches\" +
                            OriginalSubject.Name + ".testbench.json";

                        this.avmProj.UpdateResultsJson(kvp.Key.Fco, OutputSubDir);
                    });
                }
                else
                {
                    // output dir %root%/%currentobjInvoked%/%interpreter name%/
                    OutputSubDir = Path.Combine(OutputDir, interpreter.Name);
                }

                // set up base directory
                if (Directory.Exists(OutputSubDir) == false)
                {
                    Directory.CreateDirectory(OutputSubDir);
                }

                // item.Parameters.Clear();
                interpreter.Parameters["output_dir"] = OutputSubDir;

                // set up the interpreter specific parameters
                if (interpreter.ProgId == "MGA.Decorator.CyPhy2CAD")
                {
                    interpreter.Parameters["generateCAD"] = "true";
                }

                interpreter.Initialize(currentObj.Project);
                interpreter.InvokeEx(
                    currentObj.Project,
                    currentObj,
                    (MgaFCOs)Activator.CreateInstance(Type.GetTypeFromProgID("Mga.MgaFCOs")), //selectedobjsInvoked,
                    paramInvoked);


                string runCommandTEMP = "";

                runCommandTEMP = (string)interpreter.MgaComponent.ComponentParameter["runCommand"];
                // enqueue job
                Job.TypeEnum type = Job.TypeEnum.Command;
                string runCommand = "";
                if (queueToJobManager &&
                    (String.IsNullOrWhiteSpace(runCommandTEMP) ||
                    runCommandTEMP.StartsWith("ERROR:")))
                {
                    // some interpreters (e.g. FormulaEvaluator) are not supposed to have runCommands?
                    // ErrorWriter.WriteLine("runCommand parameter was not defined for component {0}", item.ProgId);
                    Trace.TraceError("runCommand parameter was not defined '{0}' for component {0}", runCommandTEMP, interpreter.MgaComponent);
                    ErrorWriter.WriteLine("Job will not be posted to the JobManager because runCommand is not defined: '{0}'", runCommandTEMP);
                    queueToJobManager = false;
                }
                else
                {
                    runCommand = runCommandTEMP;
                }

                if (interpreter.ProgId == "MGA.Interpreter.CyPhyDynamics")
                {
                    type = Job.TypeEnum.Matlab;
                }
                else if (interpreter.ProgId == "MGA.Decorator.CyPhy2CAD")
                {
                    type = Job.TypeEnum.CAD;
                }

                string interpreterName = interpreter.Name.StartsWith("MGA.Interpreter.") ?
                    interpreter.Name.Substring("MGA.Interpreter.".Length) :
                    interpreter.Name;

                string title = String.Format("{0}_{1}", interpreterName, kvp.Key.Name);
                title = title.Replace(" ", "_");
                string workingDirectory = OutputSubDir;

                if (queueToJobManager)
                {
                    bool success = EnqueueJob(runCommand, title, workingDirectory, interpreter, type);

                    if (success)
                    {
                        InfoWriter.WriteLine("=> Job was posted: {0}", title);
                        System.Windows.Forms.Application.DoEvents();
                    }
                    else
                    {
                        ErrorWriter.WriteLine("=> Job cannot be posted: {0}", title);
                        System.Windows.Forms.Application.DoEvents();
                    }
                }
            }

            return outputDirectories;
        }

        public Queue<KeyValuePair<JobServer, Job>> jobsToAdd = new Queue<KeyValuePair<JobServer, Job>>();
        public void AddJobs()
        {
            try
            {
                foreach (var j in jobsToAdd)
                {
                    j.Value.Status = Job.StatusEnum.Ready;
                    j.Key.AddJob(j.Value);
                }
            }
            catch (System.Net.Sockets.SocketException ex)
            {
                throw new Exception("JobManager is not running. Please start it.", ex);
            }
            catch (Exception e)
            {
                this.ErrorWriter.WriteLine("=> Job cannot be posted: {0}", e.ToString());
            }
            finally
            {
                jobsToAdd.Clear();
            }
        }


        public Queue<KeyValuePair<JobServer, SoT>> sotsToAdd = new Queue<KeyValuePair<JobServer, SoT>>();
        public void AddSoTs()
        {
            try
            {
                foreach (var sot in sotsToAdd)
                {
                    sot.Key.AddSoT(sot.Value);
                }
            }
            catch (System.Net.Sockets.SocketException ex)
            {
                throw new Exception("JobManager is not running. Please start it.", ex);
            }
            catch (Exception e)
            {
                this.ErrorWriter.WriteLine("=> Job cannot be posted: {0}", e.ToString());
            }
            finally
            {
                sotsToAdd.Clear();
            }
        }


        /// <summary>
        /// Posts a new job on the job server.
        /// </summary>
        /// <param name="runCommand"></param>
        /// <param name="title"></param>
        /// <param name="workingDirectory"></param>
        /// <returns>True if successfully enqueued, otherwise false</returns>
        public bool EnqueueJob(
            string runCommand,
            string title,
            string workingDirectory,
            ComComponent interpreter,
            Job.TypeEnum type = Job.TypeEnum.Command)
        {
            if (mf != null &&
                mf.chbPostJobs != null &&
                mf.chbPostJobs.Checked == false)
            {
                return false;
            }

            try
            {
                JobServer manager;
                Job j = CreateJob(out manager);

                j.RunCommand = runCommand;
                j.Title = title;
                j.WorkingDirectory = workingDirectory;
                j.Type = type;

                j.Labels = String.IsNullOrWhiteSpace(interpreter.MgaComponent.ComponentParameter["labels"] as string) ?
                    Job.DefaultLabels :
                    interpreter.MgaComponent.ComponentParameter["labels"] as string;

                j.BuildQuery = String.IsNullOrWhiteSpace(interpreter.MgaComponent.ComponentParameter["build_query"] as string) ?
                    Job.DefaultBuildQuery :
                    interpreter.MgaComponent.ComponentParameter["build_query"] as string;

                if (String.IsNullOrWhiteSpace(interpreter.MgaComponent.ComponentParameter["results_zip_py"] as string) == false)
                {
                    j.ResultsZip = interpreter.MgaComponent.ComponentParameter["results_zip_py"] as string;
                }

                jobsToAdd.Enqueue(new KeyValuePair<JobServer, Job>(manager, j));

                return true;
            }
            catch (System.Net.Sockets.SocketException ex)
            {
                throw new Exception("JobManager is not running. Please start it.", ex);
            }
            catch (Exception e)
            {
                this.ErrorWriter.WriteLine("=> Job cannot be posted: {0}", e.ToString());
                return false;
            }
        }

        public bool EnqueueSoT(List<string> workingDirs)
        {
            if (mf != null &&
                mf.chbPostJobs != null &&
                mf.chbPostJobs.Checked == false)
            {
                return false;
            }

            try
            {
                JobServer manager;
                foreach (var workingDir in workingDirs)
                {
                    SoT sot = CreateSoT(out manager);
                    sot.WorkingDirectory = workingDir;

                    sotsToAdd.Enqueue(new KeyValuePair<JobServer, SoT>(manager, sot));
                }

                return true;
            }
            catch (System.Net.Sockets.SocketException ex)
            {
                throw new Exception("JobManager is not running. Please start it.", ex);
            }
            catch (Exception e)
            {
                this.ErrorWriter.WriteLine("=> Job cannot be posted: {0}", e.ToString());
                return false;
            }
        }

        private SoT CreateSoT(out JobServer manager)
        {
            SoT sot;
            try
            {
                manager = (JobServer)Activator.GetObject(typeof(JobServer), JobServerConnection.OriginalString);
                sot = manager.CreateSoT();
            }
            catch (System.Net.Sockets.SocketException)
            {
                string exe = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "JobManager.exe");
                if (!File.Exists(exe))
                    exe = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "..\\..\\..\\JobManager\\JobManager\\bin\\Release\\JobManager.exe");
                if (!File.Exists(exe))
                    exe = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "..\\..\\..\\JobManager\\JobManager\\bin\\Debug\\JobManager.exe");
                if (File.Exists(exe))
                {
                    Process proc = new Process();
                    proc.StartInfo.UseShellExecute = false;
                    proc.StartInfo.FileName = exe;
                    proc.StartInfo.RedirectStandardOutput = true;
                    proc.Start();
                    proc.WaitForInputIdle(10 * 1000);
                    proc.StandardOutput.ReadLine(); // matches Console.Out.WriteLine("JobManager has started"); in JobManager
                    //System.Threading.Thread.Sleep(3 * 1000);
                }
                manager = (JobServer)Activator.GetObject(typeof(JobServer), JobServerConnection.OriginalString);
                sot = manager.CreateSoT();
            }
            return sot;
        }

        public Job CreateJob(out JobServer manager)
        {
            Job j;
            try
            {
                manager = (JobServer)Activator.GetObject(typeof(JobServer), JobServerConnection.OriginalString);
                j = manager.CreateJob();
            }
            catch (System.Net.Sockets.SocketException)
            {
                string exe = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "JobManager.exe");
                if (!File.Exists(exe))
                    exe = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "..\\..\\..\\JobManager\\JobManager\\bin\\Release\\JobManager.exe");
                if (!File.Exists(exe))
                    exe = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "..\\..\\..\\JobManager\\JobManager\\bin\\Debug\\JobManager.exe");
                if (File.Exists(exe))
                {
                    Process proc = new Process();
                    proc.StartInfo.UseShellExecute = false;
                    proc.StartInfo.FileName = exe;
                    proc.StartInfo.RedirectStandardOutput = true;
                    proc.Start();
                    proc.WaitForInputIdle(10 * 1000);
                    proc.StandardOutput.ReadLine(); // matches Console.Out.WriteLine("JobManager has started"); in JobManager
                    //System.Threading.Thread.Sleep(3 * 1000);
                }
                manager = (JobServer)Activator.GetObject(typeof(JobServer), JobServerConnection.OriginalString);
                j = manager.CreateJob();
            }
            return j;
        }

        private ComponentStartMode Convert(int param)
        {
            switch (param)
            {
                case (int)ComponentStartMode.GME_BGCONTEXT_START:
                    return ComponentStartMode.GME_BGCONTEXT_START;
                case (int)ComponentStartMode.GME_BROWSER_START:
                    return ComponentStartMode.GME_BROWSER_START;

                case (int)ComponentStartMode.GME_CONTEXT_START:
                    return ComponentStartMode.GME_CONTEXT_START;

                case (int)ComponentStartMode.GME_EMBEDDED_START:
                    return ComponentStartMode.GME_EMBEDDED_START;

                case (int)ComponentStartMode.GME_ICON_START:
                    return ComponentStartMode.GME_ICON_START;

                case (int)ComponentStartMode.GME_MAIN_START:
                    return ComponentStartMode.GME_MAIN_START;

                case (int)ComponentStartMode.GME_MENU_START:
                    return ComponentStartMode.GME_MENU_START;
                case (int)ComponentStartMode.GME_SILENT_MODE:
                    return ComponentStartMode.GME_SILENT_MODE;
            }

            return ComponentStartMode.GME_SILENT_MODE;
        }

        #region Component Information
        public string ComponentName
        {
            get { return GetType().Name; }
        }

        public string ComponentProgID
        {
            get
            {
                return ComponentConfig.progID;
            }
        }

        public componenttype_enum ComponentType
        {
            get { return ComponentConfig.componentType; }
        }
        public string Paradigm
        {
            get { return ComponentConfig.paradigmName; }
        }
        #endregion

        #region Enabling
        bool enabled = true;
        public void Enable(bool newval)
        {
            enabled = newval;
        }
        #endregion

        #region Interactive Mode
        protected bool interactiveMode = true;
        public bool InteractiveMode
        {
            get
            {
                return interactiveMode;
            }
            set
            {
                interactiveMode = value;
            }
        }
        #endregion

        #region Custom Parameters
        SortedDictionary<string, object> componentParameters = null;

        public object get_ComponentParameter(string Name)
        {
            if (Name == "type")
            {
                return "csharp";
            }

            if (Name == "path")
            {
                return GetType().Assembly.Location;
            }

            if (Name == "fullname")
            {
                return GetType().FullName;
            }

            object value;
            if (componentParameters != null &&
                    componentParameters.TryGetValue(Name, out value))
            {
                return value;
            }

            return null;
        }

        public void set_ComponentParameter(string Name, object pVal)
        {
            if (componentParameters == null)
            {
                componentParameters = new SortedDictionary<string, object>();
            }

            componentParameters[Name] = pVal;
        }
        #endregion

        #region Unused Methods
        // Old interface, it is never called for MgaComponentEx interfaces
        public void Invoke(MgaProject Project, MgaFCOs selectedobjs, int param)
        {
            throw new NotImplementedException();
        }

        // Not used by GME
        public void ObjectsInvokeEx(
                MgaProject Project,
                MgaObject currentobj,
                MgaObjects selectedobjs,
                int param)
        {
            throw new NotImplementedException();
        }

        #endregion

        #endregion

        #region IMgaVersionInfo Members

        public GMEInterfaceVersion_enum version
        {
            get { return GMEInterfaceVersion_enum.GMEInterfaceVersion_Current; }
        }

        #endregion

        #region Registration Helpers

        [ComRegisterFunctionAttribute]
        public static void GMERegister(Type t)
        {
            Registrar.RegisterComponentsInGMERegistry();

        }

        [ComUnregisterFunctionAttribute]
        public static void GMEUnRegister(Type t)
        {
            Registrar.UnregisterComponentsInGMERegistry();
        }

        #endregion

        [ComVisible(false)]
        public bool isProcessStaticMetrics { get; set; }
        [ComVisible(false)]
        public static DateTime StartTime { get; set; }
        [ComVisible(false)]
        public string ResultsDir { get; set; }

        public void WriteLine(Func<string, string, string> f, IMgaFCO a, IMgaFCO b)
        {
            if (GMEConsole != null)
            {
                GMEConsole.Out.WriteLine(f(GMEConsole.MakeObjectHyperlink(a, a.Name), GMEConsole.MakeObjectHyperlink(b, b.Name)));
            }
            else
            {
                Console.Out.WriteLine(f(a.ExGetPath(), b.ExGetPath()));
            }
        }

        public string loggingFileName { get; set; }
    }


}
