/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Xml.Serialization;
using System.Xml;
using GME.MGA;
using Microsoft.Win32;
using System.Diagnostics;
using GME.Util;
using System.Collections;
using JobManager;
using CyPhyMasterInterpreter.Interpreter;
using META;
using System.Runtime.InteropServices;

namespace CyPhyMasterInterpreter
{
    public partial class MainForm : Form
    {
        public MainForm()
        {
            InitializeComponent();

            chbPostJobs.Checked = Properties.Settings.Default.bPostTojobManager;
            chbSaveTestBenches.Checked = Properties.Settings.Default.bSaveTestBenches;
            chbOpenDashboard.Checked = Properties.Settings.Default.bOpenDashboard;
        }

        public void InitForm()
        {
            HideControls();

            //bool success = CyPhy.EnqueueJob(
            //    "cmd.exe",
            //    String.Format("Main form was opened {0}", DateTime.Now.ToShortTimeString()),
            //    "c:\\");

            //for (int i = 0; i < 15; ++i)
            //{
            //    CyPhy.EnqueueJob(
            //        "exit",
            //        String.Format("Test job {1} {0}", DateTime.Now.ToShortTimeString(), i),
            //        "c:\\");
            //}

            if (CyPhy.Mode == CyPhyMasterInterpreter.Mode_enum.CA_Regular ||
                CyPhy.Mode == CyPhyMasterInterpreter.Mode_enum.ME_Regular ||
                CyPhy.Mode == CyPhyMasterInterpreter.Mode_enum.TestBench_Regular)
            {
                lbConfigModels.Enabled = false;
                lbConfigModels.BackColor = Color.LightGray;
                chbShowDirty.Enabled = false;
            }
            else
            {
                lbConfigModels.Enabled = true;
                chbShowDirty.Enabled = true;
            }


            txtOutputDir.Text = Path.Combine(ProjectPath, OutputDir);

            chbElaboratorExpand.Text = "Expand (Pre-Action)";

            chbDynamics.Text = "CyPhyDynamics";

            chbElaboratorCollapse.Text = "Collapse (Post-Action)";

            tsslStatus.Text = "Mode: " + CyPhy.Mode;

            foreach (DesignConfigItem item in CyPhy.DesignConfigs)
            {
                if (item.isDirty == false)
                {
                    lbConfigModels.Items.Add(item);
                }
            }

            if (lbConfigModels.Items.Count > 0)
            {
                lbConfigModels.SelectedItem = lbConfigModels.Items[0];
            }
            else if (lbExportedCAs.Items.Count > 0)
            {
                lbExportedCAs.SelectedItem = lbExportedCAs.Items.Cast<SelectedItem>().FirstOrDefault();

            }

            foreach (FcoItem item in CyPhy.SelectedItems)
            {
                lbSelectedElements.Items.Add(item);
            }

            foreach (ComComponent component in CyPhy.Workflow)
            {
                WorkFlowListItem item = new WorkFlowListItem(
                    component,
                    lbWorkFlow.Items.Count);
                lbWorkFlow.Items.Add(item);
            }

            LoadInterpreters();

            lbExportedCAs.Refresh();
            lbConfigModels.Refresh();

            chbShowDirty.Checked = Properties.Settings.Default.bShowDirty;
        }

        private void HideControls()
        {
//#if DEBUG
//            Height = 800;
//            Width = 600;
//#else
            lblExported.Visible = false;
            chbSelectedElements.Visible = false;
            btnExpand.Visible = false;
            chbElaboratorExpand.Visible = false;
            chbDynamics.Visible = false;
            chbElaboratorCollapse.Visible = false;
            chbWorkFlow.Visible = false;
            btnInterpreters.Visible = false;
            nupMaxConfig.Visible = false;
            lbInterpreters.Visible = false;
            rtbInspector.Visible = false;
            btnAddToWorkFlow.Visible = false;
            btnUpInWorkFlow.Visible = false;
            btnDownInWorkFlow.Visible = false;
            btnRemoveFromWorkFlow.Visible = false;
//#endif
        }

        public string ProjectPath { get; set; }

        public string OutputDir { get; set; }

        public CyPhyMasterInterpreter CyPhy { set; get; }

        private void btnBrowseOutputDir_Click(object sender, EventArgs e)
        {
            using (META.FolderBrowserDialog fbd = new META.FolderBrowserDialog())
            {
                fbd.Description = "Please select an output directory for the generated files.";
                if (Directory.Exists(OutputDir) == false)
                {
                    Directory.CreateDirectory(OutputDir);
                }

                fbd.SelectedPath = txtOutputDir.Text;
                // TODO: implement this in META.FolderBrowserDialog
                //fbd.RootFolder = Environment.SpecialFolder.MyComputer;
                //fbd.ShowNewFolderButton = true;
                DialogResult dr = fbd.ShowDialog();
                if (dr == DialogResult.OK)
                {
                    txtOutputDir.Text = fbd.SelectedPath;
                    OutputDir = txtOutputDir.Text;
                }
            }
        }

        private void btnOK_Click(object sender, EventArgs e)
        {

            Properties.Settings.Default.bPostTojobManager = chbPostJobs.Checked;
            Properties.Settings.Default.bSaveTestBenches = chbSaveTestBenches.Checked;
            Properties.Settings.Default.bOpenDashboard = chbOpenDashboard.Checked;
            Properties.Settings.Default.bShowDirty = chbShowDirty.Checked;
            Properties.Settings.Default.Save();

            // if this is false the form will not be closed
            bool CanClose = true;

            // create specified directory if it does not exist
            if (Directory.Exists(txtOutputDir.Text) == false)
            {
                try
                {
                    Directory.CreateDirectory(txtOutputDir.Text);
                }
                catch (Exception ex)
                {
                    throw ex;
                }
            }

            if (CyPhy.Mode == CyPhyMasterInterpreter.Mode_enum.CA_Regular)
            {
                if (lbWorkFlow.SelectedItem == null)
                {
                    // please select an interpreter
                    MessageBox.Show(
                        "No interpreter was selected.",
                        "Warning",
                        MessageBoxButtons.OK,
                        MessageBoxIcon.Warning);
                    CanClose = false;
                }
            }
            else if (CyPhy.Mode == CyPhyMasterInterpreter.Mode_enum.CA_DS)
            {
                if (lbWorkFlow.SelectedItem == null)
                {
                    // please select an interpreter
                    MessageBox.Show(
                        "No interpreter was selected.",
                        "Warning",
                        MessageBoxButtons.OK,
                        MessageBoxIcon.Warning);
                    CanClose = false;
                }
                else
                {
                    MessageBox.Show(
                        "Test version.",
                        "Warning",
                        MessageBoxButtons.OK,
                        MessageBoxIcon.Information);

                    if (lbExportedCAs.SelectedItems.Count == 0)
                    {
                        MessageBox.Show(
                            "No configuration was selected.",
                            "Warning",
                            MessageBoxButtons.OK,
                            MessageBoxIcon.Warning);
                        CanClose = false;
                    }
                    else
                    {
                        foreach (var selected in lbExportedCAs.SelectedItems.Cast<SelectedItem>())
                        {
                            CyPhyMasterInterpreter.FCO item = null;
                            ExportedCaItem exCa = selected as ExportedCaItem;
                            if (exCa != null)
                            {
                                item = new CyPhyMasterInterpreter.FCO(
                                    (exCa.ExportedCa as MgaReference).Referred,
                                    (exCa.ExportedCa as MgaReference).Referred.Name);
                            }
                            else
                            {
                                item = new CyPhyMasterInterpreter.FCO(
                                    selected.ExportedCa, selected.ExportedCa.Name);
                            }
                            Queue<ComComponent> interpreters = new Queue<ComComponent>();
                            interpreters.Enqueue((lbWorkFlow.SelectedItem as WorkFlowListItem).Interpreter);

                            if (item != null)
                            {
                                CyPhy.toProcess.Add(item, interpreters);
                            }
                        }
                    }
                }
            }
            else if (CyPhy.Mode == CyPhyMasterInterpreter.Mode_enum.TestBench_DS)
            {
                if (lbExportedCAs.SelectedItems.Count == 0)
                {
                    // user must select configuration items
                    MessageBox.Show(
                        "No configuration was selected.",
                        "Warning",
                        MessageBoxButtons.OK,
                        MessageBoxIcon.Warning);
                    CanClose = false;
                }
            }



            //if (CyPhy.Mode != CyPhyMasterInterpreter.Mode_enum.Undefined)
            if (CanClose)
            {
                DialogResult = System.Windows.Forms.DialogResult.OK;
                OutputDir = txtOutputDir.Text;
                Close();
            }
        }

        private void btnInterpreters_Click(object sender, EventArgs e)
        {
            LoadInterpreters();
        }

        private void LoadInterpreters()
        {
            List<ComComponent> ComComponents = new List<ComComponent>();

            MgaRegistrar registrar = new MgaRegistrar();
            regaccessmode_enum r = regaccessmode_enum.REGACCESS_BOTH;
            IEnumerable a = (IEnumerable)(object)registrar.GetComponentsDisp(r);

            string paradigm = "CyPhyML";

            foreach (string comProgId in a)
            {
                bool isAssociated;
                bool canAssociate;

                registrar.IsAssociated(comProgId, paradigm, out isAssociated, out canAssociate, r);
                string DllPath = string.Empty;
                
                try
                {
                    DllPath = registrar.LocalDllPath[comProgId];
                }
                catch (System.Runtime.InteropServices.COMException ex)
                {
                    Trace.TraceError("DllPath is empty or not found for ProgID: {0}", comProgId);
                    Trace.TraceError(ex.ToString());
                }

                if (string.IsNullOrWhiteSpace(DllPath) |
                    File.Exists(DllPath) == false)
                {
                    Trace.TraceError("DllPath is empty or the path does not exist for ProgID: {0}, path {1}",
                        comProgId,
                        DllPath);
                    continue;
                }

                componenttype_enum Type;
                string desc;
                registrar.QueryComponent(
                    comProgId,
                    out Type,
                    out desc,
                    regaccessmode_enum.REGACCESS_BOTH);

                bool isInterpreter = false;
                isInterpreter = (Type == componenttype_enum.COMPONENTTYPE_INTERPRETER);

                if (isAssociated &&
                    File.Exists(DllPath) &&
                    isInterpreter)
                {
                    ComComponent component = new ComComponent(comProgId);
                    if (component.isValid)
                    {
                        ComComponents.Add(component);
                    }
                }
            }

            lbInterpreters.Items.Clear();
            foreach (ComComponent c in ComComponents)
            {
                lbInterpreters.Items.Add(c);
            }
        }

        private void lbInterpreters_SelectedIndexChanged(object sender, EventArgs e)
        {
            rtbInspector.Clear();

            string type = "";
            string path = "";
            ComComponent selected = lbInterpreters.SelectedItem as ComComponent;
            if (selected != null &&
                selected.isValid)
            {
                type = (string)selected.MgaComponent.ComponentParameter["type"];
                path = (string)selected.MgaComponent.ComponentParameter["path"];

                StringBuilder sb = new StringBuilder();
                sb.AppendLine(selected.Name);
                sb.AppendLine(selected.ProgId);
                sb.AppendLine(type);
                sb.AppendLine(path);

                sb.AppendLine(selected.Help);
                sb.AppendLine(selected.Description);
                if (selected.ParameterList != null)
                {
                    sb.AppendLine(String.Join("\n", selected.ParameterList.ToArray()));
                }

                rtbInspector.Text = sb.ToString();
                if (selected.ProgId == Definitions.COM[Definitions.Interpreter_enum.Dynamics])
                {
                    selected.MgaComponent.ComponentParameter["console_messages"] = "off";
                }
            }
            else
            {
                rtbInspector.Text = string.Empty;
            }
        }

        private void btnAddToWorkFlow_Click(object sender, EventArgs e)
        {
            if (lbInterpreters.SelectedItem != null)
            {
                WorkFlowListItem item = new WorkFlowListItem(
                    lbInterpreters.SelectedItem as ComComponent,
                    lbWorkFlow.Items.Count);
                lbWorkFlow.Items.Add(item);
            }
        }

        private void btnRemoveFromWorkFlow_Click(object sender, EventArgs e)
        {
            if (lbWorkFlow.SelectedItem != null)
            {
                WorkFlowListItem selected = lbWorkFlow.SelectedItem as WorkFlowListItem;
                List<WorkFlowListItem> items = new List<WorkFlowListItem>();
                items.AddRange(lbWorkFlow.Items.Cast<WorkFlowListItem>());
                items.Remove(selected);
                items.Where(x => x.Id > selected.Id).ToList().ForEach(x => x.Id--);
                lbWorkFlow.Items.Clear();
                lbWorkFlow.Items.AddRange(items.ToArray());
            }
        }

        private void btnUpInWorkFlow_Click(object sender, EventArgs e)
        {
            if (lbWorkFlow.SelectedItem != null)
            {
                WorkFlowListItem selected = lbWorkFlow.SelectedItem as WorkFlowListItem;
                List<WorkFlowListItem> items = new List<WorkFlowListItem>();
                items.AddRange(lbWorkFlow.Items.Cast<WorkFlowListItem>());
                WorkFlowListItem itemBefore = null;
                itemBefore = items.Where(x => x.Id == selected.Id - 1).FirstOrDefault();
                if (itemBefore != null)
                {
                    itemBefore.Id++;
                    selected.Id--;
                    lbWorkFlow.Items.Clear();
                    items.Sort(WorkFlowListItem.Compare);
                    lbWorkFlow.Items.AddRange(items.ToArray());
                    lbWorkFlow.SelectedItem = selected;
                }
            }
        }


        private void btnDownInWorkFlow_Click(object sender, EventArgs e)
        {
            if (lbWorkFlow.SelectedItem != null)
            {
                WorkFlowListItem selected = lbWorkFlow.SelectedItem as WorkFlowListItem;
                List<WorkFlowListItem> items = new List<WorkFlowListItem>();
                items.AddRange(lbWorkFlow.Items.Cast<WorkFlowListItem>());
                WorkFlowListItem itemAfter = null;
                itemAfter = items.Where(x => x.Id == selected.Id + 1).FirstOrDefault();
                if (itemAfter != null)
                {
                    itemAfter.Id--;
                    selected.Id++;
                    lbWorkFlow.Items.Clear();
                    items.Sort(WorkFlowListItem.Compare);
                    lbWorkFlow.Items.AddRange(items.ToArray());
                    lbWorkFlow.SelectedItem = selected;
                }
            }
        }

        private void chbWorkFlow_CheckedChanged(object sender, EventArgs e)
        {
            if (chbWorkFlow.Checked)
            {
                chbDynamics.Enabled = false;
            }
            else
            {
                chbDynamics.Enabled = true;
            }
        }

        [DllImport("shlwapi.dll", CharSet = CharSet.Unicode, ExactSpelling = true)]
        static extern int StrCmpLogicalW(String x, String y);

        internal void AddExportedCAItem(SelectedItem item)
        {
            int i = 0;
            for (; i < lbExportedCAs.Items.Count; i++)
            {
                if (StrCmpLogicalW(((SelectedItem)lbExportedCAs.Items[i]).DisplayedName, item.DisplayedName) != -1)
                {
                    break;
                }
            }
            lbExportedCAs.Items.Insert(i, item);
        }

        private void lbConfigModels_SelectedIndexChanged(object sender, EventArgs e)
        {
            //DesignConfigItem item = lbConfigModels.SelectedItem as DesignConfigItem;
            lbExportedCAs.Items.Clear();
            foreach (DesignConfigItem item in lbConfigModels.SelectedItems.Cast<DesignConfigItem>())
            {
                item.ExportedCAs.ForEach(x => AddExportedCAItem(x));
            }
            lbExportedCAs.Refresh();
            lblCASelected.Text = lbExportedCAs.SelectedItems.Count + " / " + lbExportedCAs.Items.Count;
        }

        int LastTipIdx { get; set; }
        private void lbExportedCAs_MouseMove(object sender, MouseEventArgs e)
        {
            string TipNew = "";
            int nIdx = lbExportedCAs.IndexFromPoint(e.Location);
            if (nIdx >= 0 &&
                nIdx < lbExportedCAs.Items.Count)
            {
                TipNew = (lbExportedCAs.Items[nIdx] as SelectedItem).ToolTip;
            }
            if (LastTipIdx != nIdx)
            {
                // refresh only if the mouse is over a new element
                toolTipCA.SetToolTip(lbExportedCAs, TipNew);
                LastTipIdx = nIdx;
            }
        }

        private void btnClearAll_Click(object sender, EventArgs e)
        {
            lbExportedCAs.SelectedItems.Clear();
        }

        private void btnSelectAll_Click(object sender, EventArgs e)
        {
            SelectAll(lbExportedCAs);
        }

        private void SelectAll(ListBox listBox)
        {
            for (int i = 0; i < listBox.Items.Count; ++i)
            {
                listBox.SetSelected(i, true);
            }
        }

        private void lbExportedCAs_KeyDown(object sender, KeyEventArgs e)
        {
            bool ctrlA; //will be true if Ctrl + Shift + M is pressed, false otherwise

            ctrlA = ((e.KeyCode == Keys.A) &&               // test for M pressed
                                        ((e.Modifiers & Keys.Control) != 0));  // test for Ctrl modifier

            if (ctrlA)
            {
                SelectAll(lbExportedCAs);
            }
        }

        private void btnExpand_Click(object sender, EventArgs e)
        {
            try
            {
                Elaborate(
                        CyPhy.projectInvoked,
                        CyPhy.currentobjInvoked,
                        CyPhy.selectedobjsInvoked,
                        CyPhy.paramInvoked);
            }
            catch (Exception ex)
            {
                MessageBox.Show(
                    ex.Message,
                    "Exception",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Error);
            }
        }

        private void Elaborate(
            MgaProject mgaProject,
            MgaFCO currentobj,
            MgaFCOs mgaFCOs,
            int p)
        {
            Stopwatch watch = Stopwatch.StartNew();
            CyPhy.tbg.time.Reset();
            CyPhy.tbg.Expand(currentobj as MgaModel, false);
            CyPhy.GMEConsole.Info.WriteLine("Elapsed: {0}, time: {1}", watch.Elapsed, CyPhy.tbg.time.Elapsed);
        }

        private void btnCancel_Click(object sender, EventArgs e)
        {
            DialogResult = System.Windows.Forms.DialogResult.Cancel;
            Close();
        }

        private void chbShowDirty_CheckedChanged(object sender, EventArgs e)
        {
            lbConfigModels.Items.Clear();
            if (chbShowDirty.Checked)
            {
                foreach (DesignConfigItem item in CyPhy.DesignConfigs)
                {
                    lbConfigModels.Items.Add(item);
                }
            }
            else
            {
                foreach (DesignConfigItem item in CyPhy.DesignConfigs)
                {
                    if (item.isDirty == false)
                    {
                        lbConfigModels.Items.Add(item);
                    }
                }
            }

            if (CyPhy.Mode == CyPhyMasterInterpreter.Mode_enum.CA_DS ||
                CyPhy.Mode == CyPhyMasterInterpreter.Mode_enum.ME_DS ||
                CyPhy.Mode == CyPhyMasterInterpreter.Mode_enum.SoT_DS ||
                CyPhy.Mode == CyPhyMasterInterpreter.Mode_enum.TestBench_DS)
            {
                lbExportedCAs.Items.Clear();
                lblCASelected.Text = "0 / 0";
            }

            if (lbConfigModels.Items.Count == 1)
            {
                lbConfigModels.SelectedItem = lbConfigModels.Items[0];
            }
        }

        private void lbExportedCAs_SelectedIndexChanged(object sender, EventArgs e)
        {
            lblCASelected.Text = lbExportedCAs.SelectedItems.Count + " / " + lbExportedCAs.Items.Count;
        }
    }
}
