/*
Copyright (C) 2011-2013 Vanderbilt University

Permission is hereby granted, free of charge, to any person obtaining a
copy of this data, including any software or models in source or binary
form, as well as any drawings, specifications, and documentation
(collectively "the Data"), to deal in the Data without restriction,
including without limitation the rights to use, copy, modify, merge,
publish, distribute, sublicense, and/or sell copies of the Data, and to
permit persons to whom the Data is furnished to do so, subject to the
following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE DATA OR THE USE OR OTHER DEALINGS IN THE DATA.  
*/
﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using GME.MGA;
using GME.CSharp;
using GME.MGA.Meta;
using System.Reflection;
using System.IO;
using System.Threading;
using GME;
using System.Diagnostics;
using System.Diagnostics.Contracts;
using CyPhyMasterInterpreter.Interpreter;
using META;
using System.Threading.Tasks;

namespace CyPhyMasterInterpreter
{
    ////////////////////////////////////////////////////////////////////////////////
    /// <summary> Test bench gateway.  </summary>
    ///
    /// <remarks> Zsolt, 8/18/2011. </remarks>
    ///
    /// <seealso cref="T:System.Object"/>
    ////////////////////////////////////////////////////////////////////////////////

    public class TestBenchGateway
    {
        #region Public Properties

        /// <summary>
        /// These objects will be passed to the interpreters
        /// </summary>
        public Dictionary<MgaFCO, string> fcosToProcess = new Dictionary<MgaFCO, string>();

        public GMEConsole Console { set; get; }

        public CyPhyMasterInterpreter CyPhy { get; set; }

        private string _CyPhyMasterExeProjectConnStr;
        private string CyPhyMasterExeProjectConnStr
        {
            get
            {
                if (_CyPhyMasterExeProjectConnStr == null)
                {
                    if (CyPhy.projectInvoked.ProjectConnStr.StartsWith("MGA="))
                    {
                        string path = CyPhy.projectInvoked.ProjectConnStr.Substring("MGA=".Length);
                        _CyPhyMasterExeProjectConnStr = "MGA=" + Path.Combine(Path.GetDirectoryName(path), Path.GetFileNameWithoutExtension(path) + "_CyPhyMaster.mga");
                        if ((CyPhy.projectInvoked.ProjectStatus & 8) != 0)
                        {
                            CyPhy.projectInvoked.CommitTransaction(); // FIXME: bit of a kludge
                            CyPhy.projectInvoked.Save(_CyPhyMasterExeProjectConnStr, true);
                            CyPhy.projectInvoked.BeginTransactionInNewTerr();
                        }
                        else
                            CyPhy.projectInvoked.Save(_CyPhyMasterExeProjectConnStr, true);
                    }
                    else
                    {
                        // TODO: can fix this with xme export and import into MGA
                        throw new Exception("Multiuser backend is unsupported");
                    }
                }
                return _CyPhyMasterExeProjectConnStr;
            }
        }

        #endregion

        #region Private Properties

        // new connections to add
        Dictionary<int, MgaMetaRole> roles = new Dictionary<int, MgaMetaRole>();
        Dictionary<int, MgaFCO> srcFCOs = new Dictionary<int, MgaFCO>();
        Dictionary<int, MgaFCO> dstFCOs = new Dictionary<int, MgaFCO>();
        Dictionary<int, MgaFCO> srcFCOrefs = new Dictionary<int, MgaFCO>();
        Dictionary<int, MgaFCO> dstFCOrefs = new Dictionary<int, MgaFCO>();

        #endregion

        #region Public Functions

        public Stopwatch time = new Stopwatch();

        public bool Expand(
            MgaModel subject,
            bool ShowLog = false)
        {
            if (subject == null)
            {
                return false;
            }

            if (subject.Meta.Name == "TestBench" || subject.Meta.Name == "CADTestBench")
            {
                MgaFCO m = PrepareTestBench(subject);
                if (m == null)
                {
                    return false;
                }
                // ASSUMPTION one system under test reference
                subject = m as MgaModel;
            }

            if (ShowLog)
            {
                CyPhy.GMEConsole.Info.WriteLine(
                    "Visited subsystem: {0}",
                    subject.Name
                    );
            }

            // call the recursively
            foreach (MgaModel m in subject.
                ExKindChildren("ComponentAssembly").
                Cast<MgaModel>())
            {
                Expand(m, ShowLog);
            }

            // check results
            int NumberOfObjectsBefore = subject.ChildObjects.Count;

            IEnumerable<MgaReference> references = null;

            references = subject.
                    ExKindChildren("ComponentRef").
                    Cast<MgaReference>();

            foreach (MgaReference r in references)
            {
                if (ShowLog)
                {
                    CyPhy.GMEConsole.Info.WriteLine(
                        "Visited reference: {0}",
                        r.Name
                        );
                }
                if (r.Referred == null)
                {
                    continue;
                }

                // create an instance of the target component/componentasm
                MgaFCO copied = null;

                string roleName = "";
                if (subject.Meta.Name == "TestBench" || subject.Meta.Name == "CADTestBench")
                {
                    if (r.Referred.Meta.Name == "ComponentAssembly")
                    {
                        roleName = "ComponentAssemblyTopLevelSystemUnderTest";
                    }
                    if (r.Referred.Meta.Name == "Component")
                    {
                        roleName = "ComponentTopLevelSystemUnderTest";
                    }
                }
                else if (subject.Meta.Name == "ComponentAssembly")
                {
                    roleName = r.Referred.Meta.Name;
                }

                MgaModel parent = subject as MgaModel;
                MgaMetaRole role = (parent.Meta as MgaMetaModel).RoleByName[roleName];
                MgaFCO to_move = r.Referred;

                copied = parent.CopyFCODisp(to_move, role);

                if (r.Referred.Meta.Name == "ComponentAssembly")
                {
                    Expand(copied as MgaModel, ShowLog);
                }

                // save connections

                Dictionary<MgaConnection, string> cps = new Dictionary<MgaConnection, string>();

                foreach (MgaObject obj in subject.ChildObjects)
                {
                    if (obj is MgaConnection)
                    {
                        foreach (MgaConnPoint connPoint in (obj as MgaConnection).ConnPoints)
                        {
                            foreach (MgaFCO refPort in connPoint.References)
                            {
                                if (refPort == r)
                                {
                                    cps.Add(obj as MgaConnection, connPoint.ConnRole);
                                }
                            }
                        }
                    }
                }

                // create connections
                int NewConnID = 0;
                // add connections

                foreach (KeyValuePair<MgaConnection, string> kvp in cps)
                {
                    // get src and dst connPoints
                    MgaConnPoint src = kvp.Key.ConnPoints.ExSrcPoint();
                    MgaConnPoint dst = kvp.Key.ConnPoints.ExDstPoint();
                    MgaFCO srcOriginalRef = (kvp.Key as MgaSimpleConnection).SrcReferences.Cast<MgaFCO>().FirstOrDefault();
                    MgaFCO dstOriginalRef = (kvp.Key as MgaSimpleConnection).DstReferences.Cast<MgaFCO>().FirstOrDefault();

                    MgaFCO newFCOtarget = null;
                    if (kvp.Value == "src")
                    {
                        newFCOtarget = copied.
                            ExKindChildren(src.Target.Meta.Name).
                            Where(x => x.Name == src.Target.Name).
                            FirstOrDefault() as MgaFCO;

                        if (newFCOtarget == null)
                        {
                            if (ShowLog)
                            {
                                CyPhy.GMEConsole.Error.WriteLine(
                                    "Connection could not be created in {0}",
                                    copied.Name
                                    );
                            }
                            continue;
                        }

                        if (src.References.OfType<MgaFCO>().FirstOrDefault() != r)
                        {
                            srcFCOrefs.Add(NewConnID, src.References.OfType<MgaFCO>().FirstOrDefault());
                        }
                        else
                        {
                            srcFCOrefs.Add(NewConnID, srcOriginalRef);
                        }

                        if (dst.References.OfType<MgaFCO>().FirstOrDefault() != r)
                        {
                            dstFCOrefs.Add(NewConnID, dst.References.OfType<MgaFCO>().FirstOrDefault());
                        }
                        else
                        {
                            dstFCOrefs.Add(NewConnID, dstOriginalRef);
                        }

                        roles.Add(NewConnID, kvp.Key.MetaRole);
                        srcFCOs.Add(NewConnID, newFCOtarget);
                        dstFCOs.Add(NewConnID, dst.Target);
                        NewConnID++;
                    }
                    else if (kvp.Value == "dst")
                    {
                        newFCOtarget = copied.
                            ExKindChildren(dst.Target.Meta.Name).
                            Where(x => x.Name == dst.Target.Name).
                            FirstOrDefault() as MgaFCO;

                        if (newFCOtarget == null)
                        {
                            if (ShowLog)
                            {
                                CyPhy.GMEConsole.Error.WriteLine(
                                    "Connection could not be created in {0}",
                                    copied.Name
                                    );
                            }
                            continue;
                        }

                        if (src.References.OfType<MgaFCO>().FirstOrDefault() != r)
                        {
                            srcFCOrefs.Add(NewConnID, src.References.OfType<MgaFCO>().FirstOrDefault());
                        }
                        else
                        {
                            srcFCOrefs.Add(NewConnID, null);
                        }

                        if (dst.References.OfType<MgaFCO>().FirstOrDefault() != r)
                        {
                            dstFCOrefs.Add(NewConnID, dst.References.OfType<MgaFCO>().FirstOrDefault());
                        }
                        else
                        {
                            dstFCOrefs.Add(NewConnID, null);
                        }

                        roles.Add(NewConnID, kvp.Key.MetaRole);
                        srcFCOs.Add(NewConnID, src.Target);
                        dstFCOs.Add(NewConnID, newFCOtarget);
                        NewConnID++;
                    }
                }


                // get original reference
                MgaFCO OriginalRef = r as MgaFCO;

                int xPosition = 0;
                int yPosition = 0;
                string icon = "";
                // copy all position info to the new object
                foreach (MgaPart part in OriginalRef.Parts)
                {
                    part.GetGmeAttrs(out icon, out xPosition, out yPosition);

                    //MgaPart p = copied.Parts.Cast<MgaPart>().
                    //  Where(x => x.Meta.MetaRef == part.Meta.MetaRef).
                    //  FirstOrDefault();

                    MgaPart p = copied.Part[part.MetaAspect];
                    if (p != null)
                    {
                        p.SetGmeAttrs(icon, xPosition, yPosition);
                    }
                }

                // delete original
                OriginalRef.DestroyObject();
                time.Start();
                // make new connections
                foreach (KeyValuePair<int, MgaMetaRole> kvp in roles)
                {
                    subject.CreateSimpleConnDisp(
                        kvp.Value,
                        srcFCOs[kvp.Key],
                        dstFCOs[kvp.Key],
                        srcFCOrefs[kvp.Key],
                        dstFCOrefs[kvp.Key]);
                }
                time.Stop();
                // init the Dictionaries for next iteration use
                roles.Clear();
                srcFCOs.Clear();
                dstFCOs.Clear();
                srcFCOrefs.Clear();
                dstFCOrefs.Clear();

                // check results
                int NumberOfObjectsAfter = subject.ChildObjects.Count;
                // check whether the process was successful
                if (NumberOfObjectsBefore != NumberOfObjectsAfter)
                {
                    //if (ShowLog)
                    {
                        CyPhy.GMEConsole.Error.WriteLine(
                            "{0} connections are missing in {1}",
                            NumberOfObjectsBefore - NumberOfObjectsAfter,
                            copied.Name
                            );
                    }
                }

            }
            return false;
        }

        private MgaFCO PrepareTestBench(MgaModel subject, bool ShowLog = false)
        {
            //throw new NotImplementedException("This feature is not supported for test benches yet.");
            //return;
            if (subject == null)
            {
                return null;
            }

            // ASSUMPTION: only ONE TLSUT
            MgaReference r = subject.
                ExKindChildren("TopLevelSystemUnderTest").
                Cast<MgaReference>().
                FirstOrDefault();

            if (r == null ||
                 r.Referred == null ||
                 r.Referred.Meta.Name != "ComponentAssembly")
            {
                return null;
            }

            // save connections
            int NumberOfObjectsBefore = subject.ChildObjects.Count;

            MgaFCO copied = null;

            MgaMetaRole role = (subject.Meta as MgaMetaModel).
                        RoleByName["ComponentAssemblyTopLevelSystemUnderTest"];

            MgaFCOs rs = (MgaFCOs)Activator.CreateInstance(Type.GetTypeFromProgID("Mga.MgaFCOs"));
            rs.Append(r.Referred);
            MgaMetaRoles rolesxx = (MgaMetaRoles)Activator.CreateInstance(Type.GetTypeFromProgID("Mga.MgaMetaRoles"));
            rolesxx.Append(role);

            MgaFCOs copieds;
            (subject as MgaModel).CopyFCOs(rs, rolesxx, out copieds);
            copied = copieds[1];
            copied.Name = r.Name;



            Dictionary<MgaConnection, string> cps = new Dictionary<MgaConnection, string>();

            foreach (MgaObject obj in subject.ChildObjects)
            {
                if (obj is MgaConnection)
                {
                    foreach (MgaConnPoint connPoint in (obj as MgaConnection).ConnPoints)
                    {
                        foreach (MgaFCO refPort in connPoint.References)
                        {
                            if (refPort == r)
                            {
                                cps.Add(obj as MgaConnection, connPoint.ConnRole);
                            }
                        }
                    }
                }
            }

            // create connections
            int NewConnID = 0;
            // add connections

            foreach (KeyValuePair<MgaConnection, string> kvp in cps)
            {
                // get src and dst connPoints
                MgaConnPoint src = kvp.Key.ConnPoints.ExSrcPoint();
                MgaConnPoint dst = kvp.Key.ConnPoints.ExDstPoint();

                MgaFCO newFCOtarget = null;
                if (kvp.Value == "src")
                {
                    newFCOtarget = copied.
                        ExKindChildren(src.Target.Meta.Name).
                        Where(x => x.Name == src.Target.Name).
                        FirstOrDefault() as MgaFCO;

                    if (newFCOtarget == null)
                    {
                        if (ShowLog)
                        {
                            CyPhy.GMEConsole.Error.WriteLine(
                                "Connection could not be created in {0}",
                                copied.Name
                                );
                        }
                        continue;
                    }

                    if (src.References.OfType<MgaFCO>().FirstOrDefault() != r)
                    {
                        srcFCOrefs.Add(NewConnID, src.References.OfType<MgaFCO>().FirstOrDefault());
                    }
                    else
                    {
                        srcFCOrefs.Add(NewConnID, null);
                    }

                    if (dst.References.OfType<MgaFCO>().FirstOrDefault() != r)
                    {
                        dstFCOrefs.Add(NewConnID, dst.References.OfType<MgaFCO>().FirstOrDefault());
                    }
                    else
                    {
                        dstFCOrefs.Add(NewConnID, null);
                    }

                    roles.Add(NewConnID, kvp.Key.MetaRole);
                    srcFCOs.Add(NewConnID, newFCOtarget);
                    dstFCOs.Add(NewConnID, dst.Target);
                    NewConnID++;
                }
                else if (kvp.Value == "dst")
                {
                    newFCOtarget = copied.
                        ExKindChildren(dst.Target.Meta.Name).
                        Where(x => x.Name == dst.Target.Name).
                        FirstOrDefault() as MgaFCO;

                    if (newFCOtarget == null)
                    {
                        if (ShowLog)
                        {
                            CyPhy.GMEConsole.Error.WriteLine(
                                "Connection could not be created in {0}",
                                copied.Name
                                );
                        }
                        continue;
                    }

                    if (src.References.OfType<MgaFCO>().FirstOrDefault() != r)
                    {
                        srcFCOrefs.Add(NewConnID, src.References.OfType<MgaFCO>().FirstOrDefault());
                    }
                    else
                    {
                        srcFCOrefs.Add(NewConnID, null);
                    }

                    if (dst.References.OfType<MgaFCO>().FirstOrDefault() != r)
                    {
                        dstFCOrefs.Add(NewConnID, dst.References.OfType<MgaFCO>().FirstOrDefault());
                    }
                    else
                    {
                        dstFCOrefs.Add(NewConnID, null);
                    }

                    roles.Add(NewConnID, kvp.Key.MetaRole);
                    srcFCOs.Add(NewConnID, src.Target);
                    dstFCOs.Add(NewConnID, newFCOtarget);
                    NewConnID++;
                }
            }


            // get original reference
            MgaFCO OriginalRef = r as MgaFCO;

            int xPosition = 0;
            int yPosition = 0;
            string icon = "";
            // copy all position info to the new object
            foreach (MgaPart part in OriginalRef.Parts)
            {
                part.GetGmeAttrs(out icon, out xPosition, out yPosition);

                MgaPart p = copied.Parts.Cast<MgaPart>().
                    Where(x => x.Meta.MetaRef == part.Meta.MetaRef).
                    FirstOrDefault();

                if (p != null)
                {
                    p.SetGmeAttrs(icon, xPosition, yPosition);
                }
            }

            // delete original
            OriginalRef.DestroyObject();
            time.Start();
            // make new connections
            foreach (KeyValuePair<int, MgaMetaRole> kvp in roles)
            {
                subject.CreateSimpleConnDisp(
                    kvp.Value,
                    srcFCOs[kvp.Key],
                    dstFCOs[kvp.Key],
                    srcFCOrefs[kvp.Key],
                    dstFCOrefs[kvp.Key]);
            }
            time.Stop();
            // init the Dictionaries for next iteration use
            roles.Clear();
            srcFCOs.Clear();
            dstFCOs.Clear();
            srcFCOrefs.Clear();
            dstFCOrefs.Clear();

            // check results
            int NumberOfObjectsAfter = subject.ChildObjects.Count;
            // check whether the process was successful
            if (NumberOfObjectsBefore != NumberOfObjectsAfter)
            {
                //if (ShowLog)
                {
                    CyPhy.GMEConsole.Error.WriteLine(
                        "{0} connections are missing in {1}",
                        NumberOfObjectsBefore - NumberOfObjectsAfter,
                        copied.Name
                        );
                }
            }
            return copied;
        }



        /// <summary>
        /// Extracts the given test bench, which contains a DesignSpace
        ///  TestInjectionPoint object.
        /// </summary>
        /// <param name="testBench"></param>
        /// <returns>A map from CARef to expanded TestBench</returns>
        public Dictionary<MgaReference, MgaModel> Extract(MgaModel testBench)
        {
            if (testBench == null)
            {
                return null;
            }

            // TODO: test this
            // set dashboard metric ids
            foreach (MgaObject item in testBench.ExKindChildren("Metric"))
            {
                MgaReference metric = item as MgaReference;

                bool isConnected = false;
                foreach (MgaConnPoint cpvf in metric.PartOfConns)
                {
                    MgaSimpleConnection valueflow = cpvf.Owner as MgaSimpleConnection;
                    if (valueflow.MetaBase.Name == "ValueFlow")
                    {
                        if (valueflow.Src == metric)
                        {
                            metric.RegistryValue[Dashboard.Config.RegNodeMetric] = valueflow.Dst.RegistryValue[Dashboard.Config.RegNodeMetric];
                            isConnected = true;
                        }
                    }
                }

                if (isConnected == false)
                {
                    metric.RegistryValue[Dashboard.Config.RegNodeMetric] = Dashboard.Config.RegNodeMetricUndefined;
                }
            }

            MgaReference TLSUT = GetTLSUT(testBench);
            if (TLSUT == null)
                return null;

            // check type (compound alternative optional)
            MgaModel DesignContainer = TLSUT.Referred.RootFCO as MgaModel;

            if (DesignContainer == null)
            {
                return null;
            }
            //Configurations
            //CWCReference
            List<MgaModel> Configurations = new List<MgaModel>();
            List<MgaReference> CWCReferences = new List<MgaReference>();

            CWCReferences.AddRange(
                DesignContainer.ExKindChildren("CWCReference").
                Cast<MgaReference>());

            // get config models
            if (CWCReferences.Count == 0)
            {
                Configurations.AddRange(
                    DesignContainer.ExKindChildren("Configurations").
                    Cast<MgaModel>());
            }
            else
            {
                CWCReferences.Where(x => x.Referred != null).
ToList().ForEach(x => Configurations.Add(x.Referred as MgaModel));
            }

            if (Configurations.Count == 0)
            {
                // no config models were found
                return null;
            }

            if (CyPhy.mf.lbConfigModels.SelectedItem == null)
            {
                // there is no selected item
                MessageBox.Show(
                    "There is no selected item in the configuration list." +
                    Environment.NewLine,
                    "Warning",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Warning);
                return null;
            }

            MgaModel selectedConfig = (CyPhy.mf.lbConfigModels.SelectedItem as DesignConfigItem).DesignConfig;
            if (selectedConfig == null)
            {
                // there is no selected item
                MessageBox.Show(
                    "There is no selected item in the configuration list." +
                    Environment.NewLine,
                    "Warning",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Warning);
                return null;
            }

            MgaModel Config = Configurations.Where(x => x.ID == selectedConfig.ID).FirstOrDefault();

            if (Config == null)
            {
                // config not found
                return null;
            }
            // select the config model based on the selected elements on the GUI
            List<string> items = CyPhy.mf.lbExportedCAs.SelectedItems.
                Cast<SelectedItem>().
                ToList().Select(x => x.ExportedCa.ID).ToList();

            if (items.Count == 0)
            {
                // there is no selected item
                MessageBox.Show(
                    "There is no selected item in the configuration list." +
                    Environment.NewLine,
                    "Warning",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Warning);
                return null;
            }


            IEnumerable<DesignConfigItem> ConfigurationItems = CyPhy.mf.lbConfigModels.SelectedItems.Cast<DesignConfigItem>();

            MgaFolder TestFolder = testBench.ParentFolder;
            if (TestFolder == null)
            {
                return null;
            }
            List<MgaReference> CaRefsToProcess = CyPhy.CaRefsToProcess = new List<MgaReference>();
            // get CA refs all
            // filter CA refs based on the selected element on the GUI
            foreach (DesignConfigItem item in ConfigurationItems)
            {
                // Adding selected componentAssemblyRefs
                CaRefsToProcess.AddRange(
                    item.DesignConfig.ExKindChildren("ComponentAssemblyRef").
                    Where(x => items.Contains(x.ID)).
                    Cast<MgaReference>());

                // Adding selected CWCs if they were exported as part of this process
                foreach (var cwc in item.DesignConfig.ExKindChildren("CWC").Where(x => items.Contains(x.ID)).Cast<MgaFCO>())
                {
                    foreach (MgaConnPoint cp in cwc.PartOfConns)
                    {
                        MgaSimpleConnection Config2CA = cp.Owner as MgaSimpleConnection;
                        CaRefsToProcess.Add(Config2CA.Dst as MgaReference);
                    }
                }
            }

            foreach (DesignConfigItem item in ConfigurationItems)
            {
                // Get those which do not have exported configurations
                var cwcs = item.DesignConfig.ExKindChildren("CWC").
                            Where(x => items.Contains(x.ID) &&
                                (x as MgaFCO).PartOfConns.Count == 0).ToList();

                if (false &&
                    CyPhy.mf != null &&
                    CyPhy.mf.chbPostJobs != null &&
                    CyPhy.mf.chbPostJobs.Checked == true)
                {
                    foreach (MgaFCO cwc in cwcs)
                    {
                        string args = "\"" + CyPhyMasterExeProjectConnStr + "\" " + CyPhy.OriginalSubject.ID + " " + cwc.ID;
                        //this.Console.Out.WriteLine(args);
                        
                        /*string WorkingDir = Path.Combine(CyPhy.OutputDir, cwc.Name + "_" + cwc.ID);
                        if (Directory.Exists(WorkingDir) == false)
                        {
                            Directory.CreateDirectory(WorkingDir);
                        }

                        string exe = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "CyPhyMasterExe.exe");
                        if (!File.Exists(exe))
                            exe = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "..\\..\\..\\CyPhyMasterExe\\bin\\Release\\CyPhyMasterExe.exe");
                        if (!File.Exists(exe))
                            exe = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "..\\..\\..\\CyPhyMasterExe\\bin\\Debug\\CyPhyMasterExe.exe");
                        string args = "\"" + CyPhyMasterExeProjectConnStr + "\" " + CyPhy.OriginalSubject.ID + " " + cwc.ID;

                        EnqueueExpand(exe, args);
                         */ 
                    }
                }
                else
                {
                    if (cwcs != null)
                    {
                        RunCAExporter(CaRefsToProcess, cwcs.Cast<MgaFCO>().ToList());
                    }
                }
            }


            CaRefsToProcess.RemoveAll(x => x.Referred == null);
            var returnVal = InnerExtract(testBench, TLSUT, DesignContainer, TestFolder, CaRefsToProcess);
            if (ExpandTasks.Count > 0)
            {
                Semaphore sem = new Semaphore(0, 1);
                ExpandTaskFactory.ContinueWhenAll(ExpandTasks.ToArray(), completedTasks => { sem.Release(); });
                sem.WaitOne(); // FIXME: pop up a dialog with a cancel button instead
            }
            return returnVal;
        }

        public List<MgaFolder> NewTestBenchFolders = new List<MgaFolder>();

        public Dictionary<MgaReference, MgaModel> InnerExtract(MgaModel testBench, MgaReference TLSUT, MgaModel DesignContainer, MgaFolder TestFolder, List<MgaReference> CaRefsToProcess)
        {
            var NewTestBenchFolder = TestFolder.CreateFolder(TestFolder.MetaFolder);
            NewTestBenchFolder.Name = testBench.Name + DateTime.Now.ToString(" (MM-dd-yyyy hh:mm:ss)");

            NewTestBenchFolders.Add(NewTestBenchFolder);

            Dictionary<MgaReference, MgaModel> ret = CaRefsToProcess.ToDictionary(key => key,
                x => ProcessCaRef(
                    NewTestBenchFolder,
                    testBench,
                    x.Referred,
                    DesignContainer,
                    TLSUT,
                    x.ParentModel.ParentModel),
                    new MgaObjectEqualityComparor<MgaReference>());

            return ret;
        }

        public void RunCAExporter(List<MgaReference> CaRefsToProcess, IEnumerable<MgaFCO> cwcs)
        {
            if (cwcs.Count() == 0)
            {
                return;
            }

            IMgaComponentEx elab = (IMgaComponentEx)Activator.CreateInstance(Type.GetTypeFromProgID("MGA.Interpreter.CyPhyCAExporter"));
            MgaFCOs selected = (MgaFCOs)Activator.CreateInstance(Type.GetTypeFromProgID("Mga.MgaFCOs"));
            foreach (MgaFCO cwc in cwcs)
            {
                selected.Append(cwc);
            }
            elab.InvokeEx(cwcs.First().Project, cwcs.First().ParentModel as MgaFCO, selected, 128);
            foreach (MgaFCO cwc in cwcs)
            {
                foreach (MgaConnPoint cp in cwc.PartOfConns)
                {
                    MgaSimpleConnection Config2CA = cp.Owner as MgaSimpleConnection;
                    CaRefsToProcess.Add(Config2CA.Dst as MgaReference);
                }
            }
        }

        public MgaReference GetTLSUT(MgaModel testBench)
        {
            MgaReference TLSUT;
            // get designspace references
            List<MgaReference> TopLevelSystemUnderTests = new List<MgaReference>();

            TopLevelSystemUnderTests.AddRange(
                testBench.ExKindChildren("TopLevelSystemUnderTest").
                Cast<MgaReference>());

            List<MgaReference> TestInjectionPoints = new List<MgaReference>();

            TestInjectionPoints.AddRange(
                testBench.ExKindChildren("TestInjectionPoint").
                Cast<MgaReference>());

            // check there is only one
            if (TopLevelSystemUnderTests.Count > 1)
            {
                // error there must be only one!
                CyPhy.GMEConsole.Error.WriteLine("# of Top Level System Under test is more than one.");
                return null;
            }

            if (TopLevelSystemUnderTests.Count > 0)
            {
                TLSUT = TopLevelSystemUnderTests[0];
            }
            else
            {
                return null;
            }
            return TLSUT;
        }

        bool Valid = false;

        /// <summary>
        /// 
        /// </summary>
        /// <param name="NewTestBenchFolder"></param>
        /// <param name="testBench"></param>
        /// <param name="ComponentAssembly"></param>
        /// <param name="DesignContainer"></param>
        /// <param name="SystemUnderTest"></param>
        /// <returns>The copied TestBench</returns>
        private MgaModel ProcessCaRef(
            MgaFolder NewTestBenchFolder,
            MgaModel testBench,
            MgaFCO ComponentAssembly,
            MgaModel DesignContainer,
            MgaReference SystemUnderTest,
            MgaModel ConfigurationModels)
        {
            string ContainerType = "";
            ContainerType = SystemUnderTest.Referred.ExAttrString("ContainerType");

            bool isRoot = false;
            if (SystemUnderTest.Referred.ExGetParent().MetaBase.ObjType == objtype_enum.OBJTYPE_FOLDER)
            {
                isRoot = true;
            }


            MgaFCO TargetFco = null;
            if (isRoot)
            {
                TargetFco = ComponentAssembly;
            }
            else
            {
                string Delimiter = "/";
                string CaPath = ConfigurationModels.ExGetPath(Delimiter);
                string ConfigPath = SystemUnderTest.Referred.ExGetParent().ExGetPath(Delimiter);
                string RelPath = "";
                RelPath = ConfigPath.Substring(CaPath.Length);
                if (SystemUnderTest.Referred.ExAttrString("ContainerType") == "Compound")
                {

                    // type
                    TargetFco = ComponentAssembly.
                        get_ObjectByPath(RelPath + Delimiter + SystemUnderTest.Referred.Name) as MgaFCO;
                }
                else if (SystemUnderTest.Referred.ExAttrString("ContainerType") == "Alternative")
                {
                    // we need to find the 'target object'
                    MgaFCO TargetFcoParent = ComponentAssembly.
                        get_ObjectByPath(RelPath) as MgaFCO;

                    IEnumerable<MgaFCO> possibleObjects = null;
                    possibleObjects = SystemUnderTest.Referred.
                        ExKindChildren("Component").Cast<MgaFCO>();

                    IEnumerable<MgaObject> possibleTargets = null;
                    possibleTargets = TargetFcoParent.
                        ExKindChildren("ComponentRef");

                    // select only possible objects.
                    IEnumerable<MgaObject> Targets = null;
                    Targets = possibleTargets.
                        Where(x => possibleObjects.Any(y => (x as MgaReference).Referred == y));

                    MgaReference TargetFcoRef = Targets.Cast<MgaFCO>().FirstOrDefault() as MgaReference;
                    if (TargetFcoRef != null)
                    {
                        TargetFco = TargetFcoRef.Referred;
                    }
                }
                else
                {
                    throw new Exception("Only Compound and Alternative Design Space References are supported.");
                }
                if (TargetFco == null)
                {
                    MessageBox.Show(
                        "The SUT reference could not be found/generated for configuration." +
                        Environment.NewLine +
                        "Please make sure this is a hierarchical configuration." +
                        Environment.NewLine +
                        ComponentAssembly.ExGetPath("/"));
                    return null;
                }
            }

            // copy test bench
            MgaModel NewTestBench = NewTestBenchFolder.
                CopyFCODisp(testBench as MgaFCO) as MgaModel;

            NewTestBench.Name = GetNewName(ComponentAssembly.Name);

            // dashboard config id
            NewTestBench.RegistryValue[Dashboard.Config.RegNodeConfig] = "C" + ComponentAssembly.StrAttrByName["ConfigurationUniqueID"];
            NewTestBench.RegistryValue[Dashboard.Config.RegNodeConfig + "_Path"] = ComponentAssembly.ExGetPath("/");

            // DASHBOARD: book keeping of test bench unique name / id
            NewTestBench.RegistryValue["TestBenchUniqueName"] = testBench.Name;
            NewTestBench.RegistryValue["TestBenchGMEID"] = testBench.ID;
            NewTestBench.RegistryValue["TestBenchGMEGUID"] = testBench.GetGuidDisp();

            int NumberOfObjectsBefore = NewTestBench.ChildObjects.Count;
#if DEBUG
            List<string> objectsBefore = NewTestBench.ChildObjects.Cast<IMgaObject>().Select(x => x.MetaBase.Name + " : " + x.Name).ToList();
#endif

            MgaReference copiedSUT = NewTestBench.
                ExKindChildren(SystemUnderTest.MetaRole.Name).
                FirstOrDefault() as MgaReference;

            // create a reference to the target component/componentasm
            MgaReference NewSUT = null;

            NewSUT = NewTestBench.CreateReference(
                SystemUnderTest.MetaRole,
                TargetFco) as MgaReference;

            // 2012-08-08 ZL: Post processing might refer to this container
            // keep the original name.
            NewSUT.Name = SystemUnderTest.Name; // +"_" + ComponentAssembly.Name;

            // save connections
            Dictionary<MgaConnection, string> cps = new Dictionary<MgaConnection, string>();
            foreach (MgaObject obj in NewTestBench.ChildObjects)
            {
                if (obj is MgaConnection)
                {
                    foreach (MgaConnPoint connPoint in (obj as MgaConnection).ConnPoints)
                    {
                        foreach (MgaFCO refPort in connPoint.References)
                        {
                            if (refPort == copiedSUT)
                            {
                                cps.Add(obj as MgaConnection, connPoint.ConnRole);
                            }
                        }
                    }
                }
            }

            // create connections
            int NewConnID = 0;
            // add connections

            foreach (KeyValuePair<MgaConnection, string> kvp in cps)
            {
                // get src and dst connPoints
                MgaConnPoint src = kvp.Key.ConnPoints.ExSrcPoint();
                MgaConnPoint dst = kvp.Key.ConnPoints.ExDstPoint();
                MgaFCO srcOriginalRef = (kvp.Key as MgaSimpleConnection).SrcReferences.Cast<MgaFCO>().FirstOrDefault();
                MgaFCO dstOriginalRef = (kvp.Key as MgaSimpleConnection).DstReferences.Cast<MgaFCO>().FirstOrDefault();

                MgaFCO newFCOtarget = null;
                if (kvp.Value == "src")
                {
                    newFCOtarget = (NewSUT.
                        Referred as MgaModel).
                        get_ObjectByPath(src.Target.Name) as MgaFCO;
                    if (newFCOtarget == null)
                    {
                        CyPhy.GMEConsole.Warning.WriteLine("Element was not found: " + src.Target.Name + " in " + "<A HREF=\"mga:" + NewSUT.Referred.ID + "\">" + NewSUT.Referred.Name + "</A>");
                        continue;
                    }
                    srcFCOrefs.Add(NewConnID, NewSUT as MgaFCO);
                    dstFCOrefs.Add(NewConnID, dstOriginalRef);
                    roles.Add(NewConnID, kvp.Key.MetaRole);
                    srcFCOs.Add(NewConnID, newFCOtarget);
                    dstFCOs.Add(NewConnID, dst.Target);
                    if (newFCOtarget == null)
                    {
                        CyPhy.GMEConsole.Error.WriteLine("<A HREF=\"mga:" + kvp.Key.ID + "\">ERROR</A>");
                    }
                    if (dst.Target == null)
                    {
                        CyPhy.GMEConsole.Error.WriteLine("<A HREF=\"mga:" + kvp.Key.ID + "\">ERROR</A>");
                    }
                    NewConnID++;
                }
                else if (kvp.Value == "dst")
                {
                    newFCOtarget = (NewSUT.
                        Referred as MgaModel).
                        get_ObjectByPath(dst.Target.Name) as MgaFCO;
                    if (newFCOtarget == null)
                    {
                        CyPhy.GMEConsole.Warning.WriteLine("Element was not found: " + dst.Target.Name + " in " + "<A HREF=\"mga:" + NewSUT.Referred.ID + "\">" + NewSUT.Referred.Name + "</A>");
                        continue;
                    }
                    srcFCOrefs.Add(NewConnID, srcOriginalRef);
                    dstFCOrefs.Add(NewConnID, NewSUT as MgaFCO);

                    roles.Add(NewConnID, kvp.Key.MetaRole);
                    srcFCOs.Add(NewConnID, src.Target);
                    dstFCOs.Add(NewConnID, newFCOtarget);
                    if (newFCOtarget == null)
                    {
                        CyPhy.GMEConsole.Error.WriteLine("<A HREF=\"mga:" + kvp.Key.ID + "\">ERROR</A>");
                    }
                    if (src.Target == null)
                    {
                        CyPhy.GMEConsole.Error.WriteLine("<A HREF=\"mga:" + kvp.Key.ID + "\">ERROR</A>");
                    }
                    NewConnID++;
                }
            }

            // get original system under test
            // ASSUMPTION we have only 1 original SUT
            MgaFCO OriginalSUT = NewTestBench.
                ExKindChildren(SystemUnderTest.MetaRole.Name).
                Where(x => x.ID != NewSUT.ID).
                FirstOrDefault() as MgaFCO;

            int xPosition = 0;
            int yPosition = 0;
            string icon = "";
            // copy all position info to the new object
            foreach (MgaPart part in OriginalSUT.Parts)
            {
                part.GetGmeAttrs(out icon, out xPosition, out yPosition);

                MgaPart p = NewSUT.Parts.Cast<MgaPart>().
                    Where(x => x.Meta.MetaRef == part.Meta.MetaRef).
                    FirstOrDefault();

                if (p != null)
                {
                    p.SetGmeAttrs(icon, xPosition, yPosition);
                }
            }
            MgaModel sutDesignContainer = (OriginalSUT as MgaReference).Referred as MgaModel;
            // delete original
            OriginalSUT.DestroyObject();

            // make new connections
            foreach (KeyValuePair<int, MgaMetaRole> kvp in roles)
            {
                NewTestBench.CreateSimpleConnDisp(
                    kvp.Value,
                    srcFCOs[kvp.Key],
                    dstFCOs[kvp.Key],
                    srcFCOrefs[kvp.Key],
                    dstFCOrefs[kvp.Key]);
            }

            // init the Dictionaries for next iteration use
            roles.Clear();
            srcFCOs.Clear();
            dstFCOs.Clear();
            srcFCOrefs.Clear();
            dstFCOrefs.Clear();

            // check results
            int NumberOfObjectsAfter = NewTestBench.ChildObjects.Count;
            // check whether the process was successful
#if DEBUG
            List<string> objectsAfter = NewTestBench.ChildObjects.Cast<IMgaObject>().Select(x => x.MetaBase.Name + " : " + x.Name).ToList();
#endif
            if (NumberOfObjectsBefore != NumberOfObjectsAfter)
            {
#if DEBUG
                this.Console.Out.WriteLine( string.Join("<br>\n", objectsBefore.ToList()));
                this.Console.Out.WriteLine("");
                this.Console.Out.WriteLine( string.Join("<br>\n", objectsAfter.ToList()));
#endif
                MessageBox.Show(
                    "Test Bench Error: One or more connections are missing. Probably some port names are the same in the System Under Test object. Please make sure you use unique names for ports.",
                    "Warning",
                    MessageBoxButtons.OK,
                    MessageBoxIcon.Warning);
                Valid = false;
                return null;
            }

            // replace TestInjectionPoints
            ReplaceTestInjectionPoints(NewTestBench, NewSUT, sutDesignContainer);

            // insert the newtestbench into the fcotoprocess list.
            fcosToProcess.Add(NewTestBench as MgaFCO, NewTestBench.Name);

            string extracted_message = String.Format("{0} test bench was extracted for {1}.", NewTestBench.RegistryValue["TestBenchUniqueName"],  NewTestBench.Name);
            CyPhy.InfoWriter.WriteLine(extracted_message);
            System.Windows.Forms.Application.DoEvents();

            Valid = false;
            return NewTestBench;
        }

        private void ReplaceTestInjectionPoints(
            MgaModel testBench,
            MgaReference systemUnderTest,
            MgaModel sutDesignContainer)
        {
            Contract.Requires(systemUnderTest != null);
            Contract.Requires(systemUnderTest.Referred != null);
            Contract.Requires(testBench != null);
            Contract.Requires(sutDesignContainer != null);

            //MgaModel DesignConfig = systemUnderTest.Referred.RootFCO as MgaModel;
            MgaModel DesignConfig = sutDesignContainer;

            foreach (var item in testBench.ExKindChildren("TestInjectionPoint").Cast<MgaReference>())
            {
                // TODO: what if 2 TIP are connected together on the test bench level?

                MgaFCO DesignContainer = (item as MgaReference).Referred;

                if (DesignContainer == null)
                {
                    CyPhy.GMEConsole.Warning.WriteLine("TestInjectionPoint is a null reference. {0}", item.Name);
                    continue;
                }

                #region CODE REVIEW
                bool isDesignContainerType = (DesignContainer.MetaBase.Name == "DesignContainer");          // 2/27/12 DY: Added for Alternative + Component Ref support
                string ContainerType = "";
                ContainerType = DesignContainer.ExAttrString("ContainerType");
                bool isRoot = false;
                if (DesignContainer.ExGetParent().MetaBase.ObjType == objtype_enum.OBJTYPE_FOLDER)
                {
                    isRoot = true;
                }

                MgaFCO ComponentAssembly = systemUnderTest.Referred; //item.Referred;

                string DCName = "";

                MgaFCO TargetFco = null;
                if (isRoot)
                {
                    TargetFco = item.Referred;
                }
                else
                {
                    string Delimiter = "/";
                    //string CaPath = DesignConfig.ExGetParent().ExGetPath(Delimiter);
                    string CaPath = sutDesignContainer.ExGetPath(Delimiter);
                    string ConfigPath = item.Referred.ExGetParent().ExGetPath(Delimiter);
                    string RelPath = "";
                    RelPath = ConfigPath.Substring(CaPath.Length);
                    if (item.Referred.ExAttrString("ContainerType") == "Compound" || (!isDesignContainerType))
                    //if (item.Referred.ExAttrString("ContainerType") == "Compound")
                    {
                        List<string> objectNames = new List<string>();
                        List<MgaObject> objsss = ComponentAssembly.ChildObjects.Cast<MgaObject>().ToList();
                        objsss.ForEach(x => objectNames.Add(x.ExGetPath(Delimiter)));
                        // type     
                        TargetFco = ComponentAssembly.
                            get_ObjectByPath(RelPath + Delimiter + item.Referred.Name) as MgaFCO;

                        DCName = DesignContainer.Name;
                    }
                    else if (item.Referred.ExAttrString("ContainerType") == "Alternative")
                    {
                        //throw new Exception("Alternative Design Space References are not supported.");
                        // we need to find the 'target object'

                        string tbPath = testBench.ExGetPath(Delimiter);
                        string sutPath = systemUnderTest.ExGetPath(Delimiter);
                        string sutDCPath = sutDesignContainer.ExGetPath(Delimiter);
                        string caPath = ComponentAssembly.ExGetPath(Delimiter);

                        MgaFCO TargetFcoParent = null;

                        if (RelPath == "")
                        {
                            TargetFcoParent = ComponentAssembly;
                        }
                        else
                        {
                            TargetFcoParent = ComponentAssembly.
                             get_ObjectByPath(RelPath + Delimiter + item.Referred.Name) as MgaFCO;
                        }

                        IEnumerable<MgaFCO> possibleObjects = null;
                        possibleObjects = item.Referred.
                            ExKindChildren("Component").Cast<MgaFCO>();

                        IEnumerable<MgaObject> possibleTargets = null;
                        possibleTargets = TargetFcoParent.
                            ExKindChildren("ComponentRef");

                        // select only possible objects.
                        IEnumerable<MgaObject> Targets = null;
                        Targets = possibleTargets.
                            Where(x => possibleObjects.Any(y => (x as MgaReference).Referred == y));

                        MgaReference TargetFcoRef = Targets.Cast<MgaFCO>().FirstOrDefault() as MgaReference;
                        if (TargetFcoRef != null)
                        {
                            TargetFco = TargetFcoRef as MgaFCO;
                            DCName = TargetFco.Name;
                        }
                    }
                    else
                    {
                        throw new Exception("Only Compound Design Space References are supported.");
                    }

                    if (TargetFco == null)
                    {
                        MessageBox.Show(
                            "The SUT reference could not be found/generated for configuration." +
                            Environment.NewLine +
                            "Please make sure this is a hierarchical configuration." +
                            Environment.NewLine +
                            ComponentAssembly.ExGetPath("/"));
                        return;
                    }
                }


                int NumberOfObjectsBefore = testBench.ChildObjects.Count;

                MgaReference copiedTIP = item;

                // create a reference to the target component/componentasm
                MgaReference NewTIP = null;

                NewTIP = testBench.CreateReference(
                    copiedTIP.MetaRole,         // GetRoleByName()
                    TargetFco) as MgaReference;

                NewTIP.Name = DCName + "_" + ComponentAssembly.RootFCO.Name;

                // save connections
                Dictionary<MgaConnection, string> cps = new Dictionary<MgaConnection, string>();
                foreach (MgaObject obj in testBench.ChildObjects)
                {
                    if (obj is MgaConnection)
                    {
                        foreach (MgaConnPoint connPoint in (obj as MgaConnection).ConnPoints)
                        {
                            if ((connPoint.References.Count > 0 && connPoint.References[1] == copiedTIP)
                                || (connPoint.References.Count == 0 && connPoint.Target == copiedTIP))
                            {
                                cps.Add(obj as MgaConnection, connPoint.ConnRole);
                                break;
                            }

                        }
                    }
                }

                // create connections
                int NewConnID = 0;
                // add connections

                foreach (KeyValuePair<MgaConnection, string> kvp in cps)
                {
                    // get src and dst connPoints
                    MgaConnPoint src = kvp.Key.ConnPoints.ExSrcPoint();
                    MgaConnPoint dst = kvp.Key.ConnPoints.ExDstPoint();
                    MgaFCO srcOriginalRef = (kvp.Key as MgaSimpleConnection).SrcReferences.Cast<MgaFCO>().FirstOrDefault();
                    MgaFCO dstOriginalRef = (kvp.Key as MgaSimpleConnection).DstReferences.Cast<MgaFCO>().FirstOrDefault();

                    MgaFCO newFCOtarget = null;
                    MgaModel tipRefModel = FindTIPRefInstance(NewTIP);
                    if (kvp.Value == "src")
                    {
                        if (src.References.Count == 0)
                        {
                            newFCOtarget = NewTIP as MgaFCO;
                            srcFCOrefs.Add(NewConnID, null);
                        }
                        else
                        {
                            newFCOtarget = tipRefModel.
                                get_ObjectByPath(src.Target.Name) as MgaFCO;
                            srcFCOrefs.Add(NewConnID, NewTIP as MgaFCO);
                        }
                        //newFCOtarget = (NewTIP.
                        //Referred as MgaModel).
                        //get_ObjectByPath(src.Target.Name) as MgaFCO;
                        if (newFCOtarget == null)
                        {
                            CyPhy.GMEConsole.Warning.WriteLine("Element was not found: " + src.Target.Name + " in " + "<A HREF=\"mga:" + NewTIP.Referred.ID + "\">" + NewTIP.Referred.Name + "</A>");
                            continue;
                        }
                        dstFCOrefs.Add(NewConnID, dstOriginalRef);

                        roles.Add(NewConnID, kvp.Key.MetaRole);
                        srcFCOs.Add(NewConnID, newFCOtarget);
                        dstFCOs.Add(NewConnID, dst.Target);
                        if (newFCOtarget == null)
                        {
                            CyPhy.GMEConsole.Error.WriteLine("<A HREF=\"mga:" + kvp.Key.ID + "\">ERROR</A>");
                        }
                        if (dst.Target == null)
                        {
                            CyPhy.GMEConsole.Error.WriteLine("<A HREF=\"mga:" + kvp.Key.ID + "\">ERROR</A>");
                        }
                        NewConnID++;
                    }
                    else if (kvp.Value == "dst")
                    {
                        if (dst.References.Count == 0)
                        {
                            newFCOtarget = NewTIP as MgaFCO;
                            dstFCOrefs.Add(NewConnID, null);
                        }
                        else
                        {
                            newFCOtarget = tipRefModel.
                                get_ObjectByPath(dst.Target.Name) as MgaFCO;
                            dstFCOrefs.Add(NewConnID, NewTIP as MgaFCO);
                        }
                        //newFCOtarget = (NewTIP.
                        //Referred as MgaModel).
                        //get_ObjectByPath(dst.Target.Name) as MgaFCO;
                        if (newFCOtarget == null)
                        {
                            CyPhy.GMEConsole.Warning.WriteLine("Element was not found: " + dst.Target.Name + " in " + "<A HREF=\"mga:" + NewTIP.Referred.ID + "\">" + NewTIP.Referred.Name + "</A>");
                            continue;
                        }
                        srcFCOrefs.Add(NewConnID, srcOriginalRef);

                        roles.Add(NewConnID, kvp.Key.MetaRole);
                        srcFCOs.Add(NewConnID, src.Target);
                        dstFCOs.Add(NewConnID, newFCOtarget);
                        if (newFCOtarget == null)
                        {
                            CyPhy.GMEConsole.Error.WriteLine("<A HREF=\"mga:" + kvp.Key.ID + "\">ERROR</A>");
                        }
                        if (src.Target == null)
                        {
                            CyPhy.GMEConsole.Error.WriteLine("<A HREF=\"mga:" + kvp.Key.ID + "\">ERROR</A>");
                        }
                        NewConnID++;
                    }
                }

                int xPosition = 0;
                int yPosition = 0;
                string icon = "";
                // copy all position info to the new object

                foreach (MgaPart part in item.Parts)
                {
                    part.GetGmeAttrs(out icon, out xPosition, out yPosition);

                    MgaPart p = NewTIP.Part[part.MetaAspect];
                    if (p != null)
                    {
                        p.SetGmeAttrs(icon, xPosition, yPosition);
                    }

                }

                // delete original
                item.DestroyObject();

                // make new connections
                foreach (KeyValuePair<int, MgaMetaRole> kvp in roles)
                {
                    StringBuilder sb = new StringBuilder();
                    sb.Append(" src " + srcFCOs[kvp.Key].Name);
                    sb.Append(" dst " + dstFCOs[kvp.Key].Name);
                    if (srcFCOrefs[kvp.Key] != null)
                    {
                        sb.Append(" srcref " + srcFCOrefs[kvp.Key].Name);
                    }
                    else
                    {
                        sb.Append(" null");
                    }
                    if (dstFCOrefs[kvp.Key] != null)
                    {
                        sb.Append(" dstref " + dstFCOrefs[kvp.Key].Name);
                    }
                    else
                    {
                        sb.Append(" null");
                    }
                    System.Diagnostics.Debug.WriteLine(sb.ToString());



                    testBench.CreateSimpleConn(
                        kvp.Value,
                        srcFCOs[kvp.Key],
                        dstFCOs[kvp.Key],
                        GetReferenceChain(srcFCOrefs[kvp.Key]),
                        GetReferenceChain(dstFCOrefs[kvp.Key]));
                }

                // init the Dictionaries for next iteration use
                roles.Clear();
                srcFCOs.Clear();
                dstFCOs.Clear();
                srcFCOrefs.Clear();
                dstFCOrefs.Clear();

                // check results
                int NumberOfObjectsAfter = testBench.ChildObjects.Count;
                // check whether the process was successful
                if (NumberOfObjectsBefore != NumberOfObjectsAfter)
                {
                    MessageBox.Show(
                        "Test Bench Error: One or more connections are missing.  Probably some port names are the same in the System Under Test object. Please make sure you use unique names for ports.",
                        "Warning",
                        MessageBoxButtons.OK,
                        MessageBoxIcon.Warning);
                    Valid = false;
                    return;
                }

                #endregion
            }
        }

        private static MgaFCOs GetReferenceChain(MgaFCO srcRef)
        {
            if (srcRef == null)
                return null;
            MgaFCOs srcRefs = Activator.CreateInstance(Type.GetTypeFromProgID("MGA.MgaFCOs")) as MgaFCOs;
            while (true)
            {
                srcRefs.Append(srcRef);
                srcRef = (srcRef as MgaReference).Referred;
                if (srcRef == null || srcRef.ObjType != objtype_enum.OBJTYPE_REFERENCE)
                    break;
            }
            return srcRefs;
        }

        private MgaModel FindTIPRefInstance(MgaReference tip)
        {
            MgaFCO tipReferred = tip.Referred;
            while (tipReferred.MetaBase.ObjType == objtype_enum.OBJTYPE_REFERENCE)
            {
                tipReferred = (tipReferred as MgaReference).Referred;
            }

            return tipReferred as MgaModel;
        }

        private string GetNewName(string name)
        {
            return name;
            //string Name = name;
            //int i = 0;
            //while (fcosToProcess.Values.Contains(Name))
            //{
            //    Name = String.Format("{0}-{1}", name, i);
            //    i++;
            //    if (i > 1000)
            //    {
            //        // avoid infinite loop
            //        throw new Exception("Maximum number of the same configuration is 1000.");
            //    }
            //}
            //return Name;
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="currentObject"></param>
        /// <returns></returns>
        public List<MgaModel> GetConfigurationModels(MgaFCO currentObject)
        {
            List<MgaModel> ConfigModels = new List<MgaModel>();
            // check the given object
            if (CheckObject(currentObject) == false)
            {
                return ConfigModels;
            }

            if (currentObject.Meta.Name == "ParametricExploration")
            {
                MgaReference tbref = currentObject.ExKindChildren("TestBenchRef").FirstOrDefault() as MgaReference;
                if (tbref == null || tbref.Referred == null)
                {
                    throw new Exception("Multi-experiment TestBenchRef is missing or a null references");
                }
                currentObject = tbref.Referred;
            }
            else if (currentObject.Meta.Name == "TestBenchSuite")
            {
                MgaReference tbref = currentObject.ExKindChildren("TestBenchRef").FirstOrDefault() as MgaReference;
                if (tbref == null || tbref.Referred == null)
                {
                    throw new Exception("Multi-experiment TestBenchRef is missing or a null references");
                }
                currentObject = tbref.Referred;
            }

            // there is only one test injection point object
            MgaReference topLevelSystemUnderTest = currentObject.
                ExKindChildren("TopLevelSystemUnderTest").
                FirstOrDefault() as MgaReference;

            //string roleNameRef = "TestInjectionPoint";
            if (currentObject.ExKindChildren("TopLevelSystemUnderTest").Count() > 1)
            {
                //MessageBox.Show(
                //  "This TestBench must contain exactly one TopLevelSystemUnderTest reference.",
                //  "Error",
                //  MessageBoxButtons.OK,
                //  MessageBoxIcon.Error);
                return ConfigModels;
            }

            if (topLevelSystemUnderTest == null &&
                currentObject.Meta.Name != "DesignContainer")
            {
                // test injection object not found
                //MessageBox.Show(
                //  "This TestBench must contain exactly one testinjection reference.",
                //  "Error",
                //  MessageBoxButtons.OK,
                //  MessageBoxIcon.Error);
                return ConfigModels;
            }
            else
            {
                // test injection object found
                //List<MgaFCO> DesignSpaceConfigs = new List<MgaFCO>();

                // get the DesignSpaceConfigs in the top level design space
                // FIRST CWC config models ONLY
                MgaModel Config = null;
                if (topLevelSystemUnderTest != null)
                {
                    Config = topLevelSystemUnderTest.Referred as MgaModel;
                }
                else
                {
                    Config = currentObject as MgaModel;
                }
                if (Config != null)
                {
                    //foreach (MgaObject child in testInjection.Referred.ChildObjects)
                    foreach (MgaObject child in Config.ChildObjects)
                    {
                        if (child is MgaReference)
                        {
                            if (child.MetaBase.Name == "CWCReference")
                            {
                                // internal DS container in the design space
                                if ((child as MgaReference).Referred != null)
                                {
                                    //DesignSpaceConfigs.Add((child as MgaReference).Referred);
                                    ConfigModels.Add((child as MgaReference).Referred as MgaModel);
                                }
                            }
                        }
                        else if (child is MgaModel)
                        {
                            if (child.MetaBase.Name == "Configurations")
                            {
                                // DesignSpace Root
                                //DesignSpaceConfigs.Add(child as MgaFCO);
                                ConfigModels.Add(child as MgaModel);
                            }
                        }
                    }
                }
                return ConfigModels;
            }
        }

        /// <summary>
        ///  Calls all interpreter in the queue based on their order.
        /// </summary>
        /// <param name="interpreters">
        /// Interpreter components
        /// </param>
        /// <param name="OverwriteOutputDir">
        /// If this is true the output directory will be set by this function.
        /// </param>
        /// <param name="OutputDirKey">
        /// Key value to access to output directory parameter of the interpreter.
        /// This key must be defined in CommonParameters!
        /// </param>
        public void CallInterpreters(
            Queue<ComComponent> interpreters,
            bool OverwriteOutputDir = true,
            string OutputDirKey = "output_dir")
        {
            if (Console == null)
            {
                throw new Exception("Please set the Console object for logging.");
            }

            int NumConfig = fcosToProcess.Count;
            int NumInterpreter = interpreters.Count;
            int CurrentConfig = 0;
            int CurrentInterpreter = 0;

            StringBuilder ConfigZeros = new StringBuilder();
            StringBuilder InterpreterZeros = new StringBuilder();

            for (int i = 0; i < NumConfig.ToString().Count(); ++i)
            {
                ConfigZeros.Append("0");
            }

            for (int i = 0; i < NumInterpreter.ToString().Count(); ++i)
            {
                InterpreterZeros.Append("0");
            }

            StringBuilder sb = new StringBuilder();
            sb.Append("Iterating through models. ");
            sb.AppendFormat("[# of models: {0} | ", NumConfig);
            sb.AppendFormat("# of interpreters: {0}]", NumInterpreter);

            Console.Info.Write(sb.ToString());

            foreach (var kvp in fcosToProcess)
            {
                CurrentConfig++;
                CurrentInterpreter = 0;

                Console.Info.WriteLine(
                    "> [{0:" + ConfigZeros + ".}/{1}] {2}",
                    CurrentConfig,
                    NumConfig,
                    kvp.Value);
                foreach (var component in interpreters)
                {
                    CurrentInterpreter++;

                    Console.Info.WriteLine(
                        ">> [{0:" + InterpreterZeros + ".}/{1}] {2}",
                        CurrentInterpreter,
                        NumInterpreter,
                        component.Name);

                    string DirValue = string.Empty;
                    string OriginalCommonDir = string.Empty;
                    string OriginalDir = string.Empty;
                    //if (component.Parameters.TryGetValue(OutputDirKey, out OriginalDir))
                    //{
                    //  // if output dir defined
                    //  if (OverwriteOutputDir)
                    //  {
                    //    DirValue = OriginalDir + "\\" + kvp.Value + "\\";
                    //    component.Parameters[OutputDirKey] = DirValue;
                    //  }
                    //  if (Directory.Exists(DirValue) == false)
                    //  {
                    //    Directory.CreateDirectory(DirValue);
                    //  }
                    //}
                    if (ComComponent.CommonParameters.TryGetValue(OutputDirKey, out OriginalCommonDir))
                    {
                        // if output dir defined
                        if (OverwriteOutputDir)
                        {
                            DirValue = Path.Combine(OriginalCommonDir, kvp.Value);
                            // TODO: replace this --> but we need PATH\ format.
                            DirValue += "\\";
                            ComComponent.CommonParameters[OutputDirKey] = DirValue;
                        }
                        if (Directory.Exists(DirValue) == false)
                        {
                            Directory.CreateDirectory(DirValue);
                        }
                    }
                    component.Initialize(kvp.Key.Project);
                    // pass the current test bench for the interpreter
                    component.InvokeEx(
                        kvp.Key.Project,
                        kvp.Key,
                        ComComponent.Options.SelectedObjects,
                        ComComponent.Options.Param);

                    //if (component.Parameters.TryGetValue(OutputDirKey, out DirValue))
                    //{
                    //  component.Parameters[OutputDirKey] = OriginalDir;
                    //}
                    if (ComComponent.CommonParameters.TryGetValue(OutputDirKey, out DirValue))
                    {
                        ComComponent.CommonParameters[OutputDirKey] = OriginalCommonDir;
                    }
                }
                //break;
            }
        }

        private void ProcessNode(
            KeyValuePair<MgaFCO, string> kvp,
            Queue<ComComponent> interpreters,
            bool OverwriteOutputDir = true,
            string OutputDirKey = "output_dir")
        {
            foreach (var component in interpreters)
            {
                //CurrentInterpreter++;

                //Console.Info.WriteLine(
                //  ">> [{0:" + InterpreterZeros + ".}/{1}] {2}",
                //  CurrentInterpreter,
                //  NumInterpreter,
                //  component.ProgId);

                string DirValue = string.Empty;
                string OriginalCommonDir = string.Empty;
                string OriginalDir = string.Empty;

                if (ComComponent.CommonParameters.TryGetValue(OutputDirKey, out OriginalCommonDir))
                {
                    // if output dir defined
                    if (OverwriteOutputDir)
                    {
                        DirValue = Path.Combine(OriginalCommonDir, kvp.Value);
                        // TODO: replace this --> but we need PATH\ format.
                        DirValue += "\\";
                        ComComponent.CommonParameters[OutputDirKey] = DirValue;
                    }
                    if (Directory.Exists(DirValue) == false)
                    {
                        Directory.CreateDirectory(DirValue);
                    }
                }
                component.Initialize(kvp.Key.Project);
                // pass the current test bench for the interpreter
                component.InvokeEx(
                    kvp.Key.Project,
                    kvp.Key,
                    (MgaFCOs)Activator.CreateInstance(Type.GetTypeFromProgID("Mga.MgaFCOs")), //ComComponent.Options.SelectedObjects,
                    ComComponent.Options.Param);

                //if (component.Parameters.TryGetValue(OutputDirKey, out DirValue))
                //{
                //  component.Parameters[OutputDirKey] = OriginalDir;
                //}
                if (ComComponent.CommonParameters.TryGetValue(OutputDirKey, out DirValue))
                {
                    ComComponent.CommonParameters[OutputDirKey] = OriginalCommonDir;
                }
            }
        }

        #endregion

        #region Private Functions

        private bool CheckObject(MgaFCO currentobj)
        {
            // TODO: we should be able to handle if the user starts from the DS
            if (currentobj == null ||
                (currentobj.Meta.Name != "TestBench" &&
                currentobj.Meta.Name != "CADTestBench" &&
                currentobj.Meta.Name != "DesignContainer" &&
                currentobj.Meta.Name != "ParametricExploration" &&
                currentobj.Meta.Name != "TestBenchSuite"))
            {
                //MessageBox.Show(
                //  "Please start this interpreter from a TestBench.",
                //  "Error",
                //  MessageBoxButtons.OK,
                //  MessageBoxIcon.Error);
                return false;
            }
            return true;
        }

        //internal void SaveSelectedList()
        //{
        //  List<ExportedCaItem> ExportedCas = new List<ExportedCaItem>();
        //  ExportedCas.AddRange(CyPhy.mf.lbExportedCAs.SelectedItems.Cast<ExportedCaItem>());

        //  fcosToProcess.Clear();

        //  ExportedCas.Where(x => x.ExportedCa.Referred != null).
        //    ToList().
        //    ForEach(x => fcosToProcess.Add(
        //      x.ExportedCa.Referred as MgaFCO,
        //      x.ExportedCa.Referred.Name));
        //}
        #endregion

        TaskFactory ExpandTaskFactory { get; set; }
        CancellationTokenSource ExpandCancellationTokenSource { get; set; }
        List<Task> ExpandTasks { get; set; }

        public Task EnqueueExpand(string command, string args)
        {
            JobManager.JobServer jobManager;
            CyPhy.CreateJob(out jobManager); // make sure the JobManager is running
            Task<int> t = ExpandTaskFactory.StartNew(() => ExternalExpand(command, args));
            ExpandTasks.Add(t);
            return t;
        }

        private int ExternalExpand(string command, string args)
        {
            System.Diagnostics.Process proc0 = new System.Diagnostics.Process();
            System.Diagnostics.ProcessStartInfo psi = new System.Diagnostics.ProcessStartInfo();

            psi.Arguments = args;
            psi.CreateNoWindow = true;
            psi.FileName = command;
            psi.UseShellExecute = false;
            psi.WindowStyle = System.Diagnostics.ProcessWindowStyle.Minimized;
            // psi.WorkingDirectory

            psi.RedirectStandardOutput = true;
            psi.RedirectStandardError = true;
            proc0.StartInfo = psi;

            StringBuilder stderr = new StringBuilder();
            StringBuilder stdout = new StringBuilder();

            proc0.OutputDataReceived += ((sender, e) =>
            {
                stdout.Append(e.Data);
            });
            proc0.ErrorDataReceived += ((sender, e) =>
            {
                stderr.Append(e.Data);
            });
            proc0.Start();
            if (psi.RedirectStandardOutput)
            {
                proc0.BeginOutputReadLine();
            }
            if (psi.RedirectStandardError)
            {
                proc0.BeginErrorReadLine();
            }
            proc0.WaitForExit();

            int exitCode = proc0.ExitCode;
            proc0.Close();
            if (exitCode != 0)
            {
                lock (this.CyPhy.GMEConsole)
                {
                    this.CyPhy.GMEConsole.Error.WriteLine(stderr.ToString());
                }
            }
            return exitCode;
        }

        #region Constructors

        /// <summary>
        /// Constructor.
        /// </summary>
        public TestBenchGateway()
        {

            ExpandCancellationTokenSource = new CancellationTokenSource();

            ExpandTaskFactory = new TaskFactory(
                ExpandCancellationTokenSource.Token,
                TaskCreationOptions.LongRunning,
                TaskContinuationOptions.None,
                null);

            ExpandTasks = new List<Task>();
        }

        public TestBenchGateway(CyPhyMasterInterpreter cyphy)
            : this()
        {
            CyPhy = cyphy;
        }

        #endregion

    }
}
